/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <stdlib.h>
#include <string.h>

#include "acache.h"
#include "articlelist.h"
#include "globals.h"
#include "gui.h"
#include "gui-paned.h"
#include "nntp.h"
#include "prefs.h"
#include "queue.h"
#include "queue-item-headers.h"
#include "text.h"
#include "util.h"

#define DISPLAY_PREFS_PREVIEW_FONT_SIZE 12


/* server profile */
typedef struct
{
	GtkWidget *profile_name_entry;
	GtkWidget *server_address;
	GtkWidget *server_port;
	GtkWidget *auth_cbutton;
	GtkWidget *server_username_label;
	GtkWidget *server_username;
	GtkWidget *server_password_label;
	GtkWidget *server_password;
	GtkWidget *msgid_cbutton;
	GtkWidget *max_connections;
	server_data *sdata;
} server_profile;

typedef struct
{
	GnomePropertyBox *box;
	GtkWidget *server_clist;
	GtkWidget *edit_profile_dialog;

/* smtp server*/
	GtkWidget *smtp_address;
	GtkWidget *smtp_port;

/* cache */
	GtkWidget *cache_kbytes_entry;
	GtkWidget *flush_cache_on_exit_check;

/* general */
	GtkWidget *online_cbutton;
	GtkWidget *one_click_preview_cbutton;
	GtkWidget *hide_mpart_child_nodes_cbutton;
	GtkWidget *subject_as_filename_cbutton;
	GtkWidget *attribution_entry;
	GtkWidget *external_mailer_entry;
	GtkWidget *external_editor_entry;

/* user */
	GtkWidget *user_email;
	GtkWidget *user_organization;
	GtkWidget *user_fullname;
	GtkWidget *user_signature_file;

/* directories */
	GtkWidget *dir_download;
	GtkWidget *dir_temp;
	GtkWidget *dir_data;

/* display */
	GtkWidget *grouplist_gfp;
	GtkWidget *articlelist_gfp;
	GtkWidget *message_body_gfp;

	GtkWidget *read_color_picker;
	GtkWidget *unread_color_picker;
	GtkWidget *new_color_picker;
	GtkWidget *killfile_color_picker;
	GtkWidget *watched_color_picker;

/* article headers */
	GtkWidget *article_headers_clist;
} PrefsWindow;

typedef struct
{
	server_data *old;
	server_data *new;
} prefs_sdata;


static GtkWidget* prefs_general_page ( void );
static void edit_profile_changed (GnomeDialog *d);
static void prefs_apply_cb (GtkWidget *box, gint page_num);
static GtkWidget* prefs_message_page (void);

static void prefs_cancel_cb (void);
static void new_server_cb (void);
static void edit_server_cb (void);
static void delete_server_cb (void);
/*
 * static void online_cbutton_toggled (GtkWidget *widget, gpointer data);
 */
static void one_click_preview_cbutton_toggled (GtkWidget *widget, gpointer data);
static GtkWidget* prefs_smtp_page (void);
static GtkWidget* prefs_cache_page (void);
static GtkWidget* prefs_server_page (void);
static GtkWidget* prefs_display_page (void);
static GtkWidget* prefs_general_page (void);
static GtkWidget* prefs_directories_page (void);
static GtkWidget* prefs_userinfo_page (void);
static void prefs_create_clist (void);
static void edit_profile_dialog_clicked (GnomeDialog *dialog, int button, prefs_sdata *data);
static void edit_profile_dialog (prefs_sdata *data);
static void server_clist_button_press (GtkWidget *widget, GdkEventButton *bevent);
static gulong get_header_flags (void);
static void prefs_servers_changed (void);


static PrefsWindow *prefs_window;

/* Items that we will keep in memory, since
 * they might be looked up frequently */
gchar *temp_dir = NULL;
gchar *data_dir = NULL;
gchar *download_dir = NULL;

gchar *grouplist_font = NULL;
gchar *articlelist_font = NULL;
gchar *message_body_font = NULL;
gchar *attribution_line = NULL;
gchar *external_mailer = NULL;
gchar *external_editor = NULL;

gboolean use_subject_as_filename = FALSE;

static gboolean articlelist_changed = FALSE;
static gboolean articlelist_font_changed = FALSE;
static gboolean grouplist_changed = FALSE;
static gboolean message_body_changed = FALSE;
static gboolean servers_changed = FALSE;

static GSList *sdata_deletes = NULL;

static GtkWidget *sections_clist;

/* Ok or Apply pressed in the Preferences dialog, save all information */
static void
prefs_apply_cb (GtkWidget *box, gint page_num)
{
	gushort r, g, b, a;
	gulong new_header_flags = 0;
	gulong old_header_flags = 0;
	char *p;
	const gchar *key;
	gchar *pch;
	int cache_old;
	int cache_new;

	/* -1 means all pages, ignore anything else */
	if (page_num == -1)
	{
		/* Commit all of the data to the config file */

		/* No entry for these yet
		 * gnome_config_set_int("/Pan/Articles/Expiration", 30);
		 */

		gnome_config_set_string ("/Pan/User/Full_Name",
					 gtk_entry_get_text(GTK_ENTRY (prefs_window->user_fullname)));
		gnome_config_set_string ("/Pan/User/Email",
					 gtk_entry_get_text(GTK_ENTRY (prefs_window->user_email)));
		gnome_config_set_string ("/Pan/User/Organization",
					 gtk_entry_get_text(GTK_ENTRY (prefs_window->user_organization)));
		gnome_config_set_string ("/Pan/User/Signature_File",
					gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(prefs_window->user_signature_file)))));
		gnome_config_set_string ("/Pan/Paths/download_dir",
					gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(prefs_window->dir_download)))));
		gnome_config_set_string ("/Pan/Paths/temp_dir",
					gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(prefs_window->dir_temp)))));
		gnome_config_set_string ("/Pan/Paths/data_dir",
					gtk_entry_get_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(prefs_window->dir_data)))));

		/* update the mail server */
		key = "/Pan/Mail/smtp_address";
		pch = g_strstrip(gtk_editable_get_chars (GTK_EDITABLE(prefs_window->smtp_address),0,-1));
		if (!*pch)
			gnome_config_clean_key (key);
		else
			gnome_config_set_string (key, pch);
		g_free (pch);

		/* update the mail server port */
		key = "/Pan/Mail/smtp_port";
		pch = g_strstrip(gtk_editable_get_chars (GTK_EDITABLE(prefs_window->smtp_port),0,-1));
		if (!*pch)
			gnome_config_clean_key (key);
		else
			gnome_config_set_int (key, atoi(pch));
		g_free (pch);

		/* Handle color pickers */

		gnome_config_push_prefix ("/Pan/Display/");
		gnome_color_picker_get_i16(
			GNOME_COLOR_PICKER(prefs_window->unread_color_picker),
			&r, &g, &b, &a);
		if ((unread_color.red != r)
			|| (unread_color.green != g)
			|| (unread_color.blue != b))
		{
			unread_color.red = r;
			unread_color.green = g;
			unread_color.blue = b;
			gnome_config_set_int ("unread_color_r", r);
			gnome_config_set_int ("unread_color_g", g);
			gnome_config_set_int ("unread_color_b", b);
			gnome_config_set_int ("unread_color_a", a);
		}

		gnome_color_picker_get_i16(
			GNOME_COLOR_PICKER(prefs_window->read_color_picker),
			&r, &g, &b, &a);
		if ((read_color.red != r)
		    || (read_color.green != g)
		    || (read_color.blue != b))
		{
			read_color.red = r;
			read_color.green = g;
			read_color.blue = b;
			gnome_config_set_int ("read_color_r", r);
			gnome_config_set_int ("read_color_g", g);
			gnome_config_set_int ("read_color_b", b);
			gnome_config_set_int ("read_color_a", a);
		}

		gnome_color_picker_get_i16(
			GNOME_COLOR_PICKER(prefs_window->new_color_picker),
			&r, &g, &b, &a);
		if ((new_color.red != r)
		    || (new_color.green != g)
		    || (new_color.blue != b))
		{
			new_color.red = r;
			new_color.green = g;
			new_color.blue = b;
			gnome_config_set_int ("new_color_r", r);
			gnome_config_set_int ("new_color_g", g);
			gnome_config_set_int ("new_color_b", b);
			gnome_config_set_int ("new_color_a", a);
		}

		gnome_color_picker_get_i16(
			GNOME_COLOR_PICKER(prefs_window->killfile_color_picker),
			&r, &g, &b, &a);
		if ((killfile_color.red != r)
		    || (killfile_color.green != g)
		    || (killfile_color.blue != b))
		{
			killfile_color.red = r;
			killfile_color.green = g;
			killfile_color.blue = b;
			gnome_config_set_int ("killfile_color_r", r);
			gnome_config_set_int ("killfile_color_g", g);
			gnome_config_set_int ("killfile_color_b", b);
			gnome_config_set_int ("killfile_color_a", a);
		}

		gnome_color_picker_get_i16(
			GNOME_COLOR_PICKER(prefs_window->watched_color_picker),
			&r, &g, &b, &a);
		if ((watched_color.red != r)
		    || (watched_color.green != g)
		    || (watched_color.blue != b))
		{
			watched_color.red = r;
			watched_color.green = g;
			watched_color.blue = b;
			gnome_config_set_int ("watched_color_r", r);
			gnome_config_set_int ("watched_color_g", g);
			gnome_config_set_int ("watched_color_b", b);
			gnome_config_set_int ("watched_color_a", a);
		}

		gnome_config_pop_prefix ();				 
		
		if (articlelist_changed)
			articlelist_queue_reload();

		/* Handle font pickers */
		if (articlelist_font_changed)
		{
			p = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(
				prefs_window->articlelist_gfp));
			gnome_config_set_string (
				"/Pan/Display/articlelist_font", p);
			g_free (articlelist_font);
			articlelist_font = g_strdup (p);
			articlelist_font_changed = FALSE;

			killfile_style = NULL;
			unread_style = NULL;
			read_style = NULL;
			new_style = NULL;

			/* Make the color changes take effect */
			cmap = gdk_colormap_get_system ();
			if (!gdk_color_alloc (cmap, &unread_color))
				g_warning ("couldn't allocate color");
			if (!gdk_color_alloc (cmap, &read_color))
				g_warning ("couldn't allocate color");
			if (!gdk_color_alloc (cmap, &new_color))
				g_warning ("couldn't allocate color");
			if (!gdk_color_alloc (cmap, &killfile_color))
				g_warning ("couldn't allocate color");
			if (!gdk_color_alloc (cmap, &watched_color))
				g_warning ("couldn't allocate color");
	
			articlelist_queue_reload();
		}

		if (grouplist_changed)
		{
			p = gnome_font_picker_get_font_name (GNOME_FONT_PICKER(
					prefs_window->grouplist_gfp));
			gnome_config_set_string (
				"/Pan/Display/grouplist_font", p);
			g_free (grouplist_font);
			grouplist_font = g_strdup (p);
			grouplist_changed = FALSE;

			/* make the new settings take effect */
			widget_set_font (
				GTK_WIDGET(Pan.group_clist), grouplist_font);
		}

		if (message_body_changed)
		{
			p = gnome_font_picker_get_font_name (GNOME_FONT_PICKER(
					prefs_window->message_body_gfp));
			gnome_config_set_string (
				"/Pan/Display/message_body_font", p);
			g_free (message_body_font);
			message_body_font = g_strdup (p);
			message_body_changed = FALSE;

			/* make the new settings take effect */
			widget_set_font (
				GTK_WIDGET(Pan.text), message_body_font);
		}

		/* cache */
		cache_old = gnome_config_get_int_with_default(
			"/Pan/Cache/MaxSize=1242880", NULL);
		cache_new = atoi(gtk_entry_get_text(
				GTK_ENTRY(prefs_window->cache_kbytes_entry)));
		if (cache_old != cache_new) {
			gnome_config_set_int ("/Pan/Cache/MaxSize", cache_new);
			acache_expire ();
		}
		gnome_config_set_bool ("/Pan/Cache/FlushOnExit",
				gtk_toggle_button_get_active (
					GTK_TOGGLE_BUTTON( prefs_window->flush_cache_on_exit_check)));

		/* article headers */
		old_header_flags = (gulong) gnome_config_get_int_with_default("/Pan/State/Headers=243", NULL);
		new_header_flags = get_header_flags ( );
		if (old_header_flags != new_header_flags) {
			gnome_config_set_int ("/Pan/State/Headers", new_header_flags);
			gui_set_headers ( NULL, current_article, new_header_flags );
		}
		
		if (servers_changed) {
			/* Handle the server mess */
			prefs_servers_changed ();
		}

		/* attribution line */
		if (1)
		{
			GtkEntry *e = GTK_ENTRY(
				prefs_window->attribution_entry);
			const gchar *s = gtk_entry_get_text (e);
			if (strcmp(s, attribution_line))
			{
				g_free (attribution_line);
				attribution_line = g_strdup (s);
				gnome_config_set_string (
					"/Pan/User/attribution_line", s);
			}
		}

		/* external mailer line */
		if (1)
		{
			GtkEntry *e = GTK_ENTRY(prefs_window->external_mailer_entry);
			const gchar *s = gtk_entry_get_text (e);
			if (strcmp(s, external_mailer))
			{
				g_free (external_mailer);
				external_mailer = g_strdup (s);
				gnome_config_set_string ("/Pan/User/external_mailer", s);
			}
		}

		/* external editor line */
		if (1)
		{
			GtkEntry *e = GTK_ENTRY(prefs_window->external_editor_entry);
			const gchar *s = gtk_entry_get_text (e);
			if (strcmp(s, external_editor))
			{
				g_free (external_editor);
				external_editor = g_strdup (s);
				gnome_config_set_string ("/Pan/User/external_editor", s);
			}
		}

		/* use subject as file */
		if (1)
		{
			GtkToggleButton* tb = GTK_TOGGLE_BUTTON(
				prefs_window->subject_as_filename_cbutton);
			gboolean b = gtk_toggle_button_get_active(tb);
			use_subject_as_filename = b;
			gnome_config_set_bool (
				"/Pan/General/use_subject_as_filename", b);
		}

		gnome_config_sync();

		prefs_init ();
	}
}

static void
prefs_servers_changed (void)
{
	gchar *p;
	gint r = 0;
	prefs_sdata *p_sdata;
	gboolean menu_needs_update = FALSE;

	while ((p_sdata = gtk_clist_get_row_data (GTK_CLIST(prefs_window->server_clist), r))) {
		if (p_sdata->new) {
			if (p_sdata->old) {
				/* Modified */
				server_data *new = p_sdata->new;
				server_data *old = p_sdata->old;
				
				if (strcmp (old->name, new->name))
					menu_needs_update = TRUE;
				p = old->name;
				old->name = g_strdup (new->name);
				g_free (p);
				p = old->address;
				old->address = g_strdup (new->address);
				old->port = new->port;
				old->need_auth = new->need_auth;
				old->max_connections = new->max_connections;
				old->gen_msgid = new->gen_msgid;
				p = old->username;
				old->username = g_strdup (new->username);
				g_free (p);
				p = old->password;
				old->password = g_strdup (new->password);
				g_free (p);
			} else {
				/* New addition */
				server_data *new = p_sdata->new;
				gchar *path = g_strdup_printf ("/%s/%s/", data_dir, new->name);
				directory_check (path);
				Pan.serverlist = g_slist_append (Pan.serverlist, new);
				p_sdata->new = NULL;
				menu_needs_update = TRUE;
				g_free (path);
			}
		}
		r++;
	}
	if (sdata_deletes) {
		GSList *l;
		for (l = sdata_deletes; l; l = l->next) {
			prefs_sdata *p_sdata = l->data;
			Pan.serverlist = g_slist_remove (Pan.serverlist, p_sdata->old);
			server_free (p_sdata->new);
			g_free (p_sdata);
		}
		g_slist_free (sdata_deletes);
		menu_needs_update = TRUE;
	}

	if (menu_needs_update)
		server_menu_update ();
}

#define prefs_get_dir(var, pandir, confstr)				\
	g_free (var);							\
	var = gnome_config_get_string ("/Pan/Paths/" confstr);		\
	if (var == NULL)						\
		var = gnome_util_prepend_user_home (".pan/" pandir);	\
	directory_check (var);

/**
 * prefs_init:
 *
 * Called from pan_init() before gtk_main(), this is where all the
 * user preferences get loaded in.
 **/
void
prefs_init (void)
{
	gchar *pch = NULL;

	/* directories */

	prefs_get_dir (temp_dir, "tmp", "temp_dir");
	prefs_get_dir (data_dir, "data", "data_dir");
	prefs_get_dir (download_dir, "download", "download_dir");

	/* (Set TMPDIR env var, so that uulib uses the right tmpdir) */

	pch = g_strdup_printf ("TMPDIR=%s", temp_dir);
        putenv (pch);
	pch = NULL; /* Purify says not to free pch */

	/* general settings */

	online_prompt = gnome_config_get_bool (
		"/Pan/General/online_prompt");
	one_click_preview = gnome_config_get_bool (
		"/Pan/General/one_click_preview=FALSE");
	use_subject_as_filename = gnome_config_get_bool (
		"/Pan/General/use_subject_as_filename=FALSE");
	hide_mpart_child_nodes = gnome_config_get_bool (
		"/pan/General/hide_mpart_child_nodes=TRUE");
	attribution_line = gnome_config_get_string (
		"/Pan/User/attribution_line");
	if (!attribution_line)
		attribution_line = g_strdup (_("In article %i, %a wrote:")),
	external_mailer = gnome_config_get_string
		("/Pan/User/external_mailer=xterm -e mutt -s '%s' -i '%t' '%r'");
	external_editor = gnome_config_get_string (
		"/Pan/User/external_editor=xterm -e vi '%t'");

	/* display settings */

	grouplist_font = gnome_config_get_string (
		"/Pan/Display/grouplist_font");
	articlelist_font = gnome_config_get_string (
		"/Pan/Display/articlelist_font");
	message_body_font = gnome_config_get_string (
		"/Pan/Display/message_body_font");

	unread_color.red = gnome_config_get_int (
		"/Pan/Display/unread_color_r=0");
	unread_color.green = gnome_config_get_int (
		"/Pan/Display/unread_color_g=0");
	unread_color.blue = gnome_config_get_int (
		"/Pan/Display/unread_color_b=47165");

	read_color.red = gnome_config_get_int (
		"/Pan/Display/read_color_r=9000");
	read_color.green = gnome_config_get_int (
		"/Pan/Display/read_color_g=9000");
	read_color.blue = gnome_config_get_int (
		"/Pan/Display/read_color_b=9000");
	
	new_color.red = gnome_config_get_int (
		"/Pan/Display/new_color_r=65535");
	new_color.green = gnome_config_get_int (
		"/Pan/Display/new_color_g=9000");
	new_color.blue = gnome_config_get_int (
		"/Pan/Display/new_color_b=9000");
	
	killfile_color.red = gnome_config_get_int (
		"/Pan/Display/killfile_color_r=35723");
	killfile_color.green = gnome_config_get_int (
		"/Pan/Display/killfile_color_g=17733");
	killfile_color.blue = gnome_config_get_int (
		"/Pan/Display/killfile_color_b=4883");

	watched_color.red = gnome_config_get_int (
		"/Pan/Display/watched_color_r=0");
	watched_color.green = gnome_config_get_int (
		"/Pan/Display/watched_color_g=52428");
	watched_color.blue = gnome_config_get_int (
		"/Pan/Display/watched_color_b=0");
}

static void
prefs_select_row_cb (GtkCList *clist,
		     gint row,
		     gint column,
		     GdkEventButton *event,
		     gpointer user_data)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (prefs_window->box->notebook), row);
}

static void
prefs_add_section_page (gchar     *name,
			GtkWidget *widget)
{
	gchar *text[2] = { name, NULL };

	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
				  widget,
				  gtk_label_new (name));

	gtk_clist_append (GTK_CLIST (sections_clist), text);

}

static void
prefs_create_clist (void)
{
	gchar *text[2];
	GSList *l;
	server_data *sdata;
	prefs_sdata *p_sdata;
	gint row;

	gtk_clist_freeze (GTK_CLIST(prefs_window->server_clist));
	gtk_clist_clear (GTK_CLIST(prefs_window->server_clist));

	for (l = Pan.serverlist; l; l = l->next) {
		sdata = (server_data *) l->data;
		p_sdata = g_malloc0 (sizeof (prefs_sdata));
		p_sdata->old = sdata;
		text[0] = (sdata->name) ? sdata->name : _("Not Named");
		text[1] = (sdata->address) ? sdata->address : _("No Address");
		row = gtk_clist_prepend (GTK_CLIST(prefs_window->server_clist), text);
		gtk_clist_set_row_data (GTK_CLIST(prefs_window->server_clist), row, p_sdata);
	}

	gtk_clist_thaw (GTK_CLIST(prefs_window->server_clist));
}


static void
edit_profile_dialog_clicked (GnomeDialog *dialog, int button, prefs_sdata *p_sdata)
{
	server_profile *sprof = gtk_object_get_data (GTK_OBJECT(dialog), "sprof");

	g_assert (sprof);

	if (button == 0) {
		gint row;
		server_data *sdata;

		/* OK button */
		sdata = g_malloc0(sizeof(server_data));
		sdata->name = gtk_editable_get_chars (GTK_EDITABLE (sprof->profile_name_entry), 0, -1);
		sdata->address = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_address), 0, -1);
		sdata->port = atoi((gtk_entry_get_text (GTK_ENTRY(sprof->server_port))));
		sdata->need_auth = (!GTK_TOGGLE_BUTTON (sprof->auth_cbutton)->active) ? 0 : 1;
		sdata->gen_msgid = (!GTK_TOGGLE_BUTTON (sprof->msgid_cbutton)->active) ? 0 : 1;
		sdata->username = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_username), 0, -1);
		sdata->password = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_password), 0, -1);
		sdata->max_connections = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (sprof->max_connections));

		if (p_sdata == NULL) {
			gchar *text[2];

			p_sdata = g_malloc0 (sizeof(prefs_sdata));
			p_sdata->new = sdata;
			text[0] = (sdata->name) ? sdata->name : _("Not Named");
			text[1] = (sdata->address) ? sdata->address : _("No Address");
			row = gtk_clist_append  (GTK_CLIST(prefs_window->server_clist), text);
			gtk_clist_set_row_data (GTK_CLIST(prefs_window->server_clist), row, p_sdata);
		} else {
			server_free (p_sdata->new);
			p_sdata->new = sdata;
			row = gtk_clist_find_row_from_data (GTK_CLIST(prefs_window->server_clist), p_sdata);
			gtk_clist_set_text (GTK_CLIST(prefs_window->server_clist), row, 1, sdata->address);
			gtk_clist_set_text (GTK_CLIST(prefs_window->server_clist), row, 0, sdata->name);
		}
	}

	gtk_widget_destroy (GTK_WIDGET(dialog));
}


static void
edit_profile_changed (GnomeDialog *d)
{
	servers_changed = TRUE;
	gnome_dialog_set_sensitive (d, 0, TRUE);
	gnome_property_box_changed (prefs_window->box);
}


static void
auth_cbutton_set (GtkWidget* widget, server_profile *sprof)
{
	servers_changed = TRUE;
	if (!GTK_TOGGLE_BUTTON (widget)->active) {
		gtk_widget_set_sensitive (sprof->server_username_label,0);
		gtk_widget_set_sensitive (sprof->server_username,0);
		gtk_widget_set_sensitive (sprof->server_password_label,0);
		gtk_widget_set_sensitive (sprof->server_password,0);
	} else {
		/* disable everything */
		gtk_widget_set_sensitive (sprof->server_username_label,1);
		gtk_widget_set_sensitive (sprof->server_username,1);
		gtk_widget_set_sensitive (sprof->server_password_label,1);
		gtk_widget_set_sensitive (sprof->server_password,1);
	}
}


/*---[ edit_profile_dialog ]------------------------------------------
 * The dialog for editing a server profile.
 *
 * FIXME: changing the 'profile' entry for any server segfaults Pan
 *--------------------------------------------------------------------*/
static void
edit_profile_dialog (prefs_sdata *p_sdata)
{
	GtkWidget *dialog;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkAdjustment *adj;
	server_data *sdata;
	server_profile *sprof;

	if (p_sdata)
		if (p_sdata->new)
			sdata = p_sdata->new;
		else
			sdata = p_sdata->old;
	else 
		sdata = NULL;

	sprof = g_malloc0 (sizeof(server_profile));
	
	dialog = gnome_dialog_new (_("New/Edit Server"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL, NULL);
	
	gtk_object_set_data_full (GTK_OBJECT(dialog), "sprof", sprof, g_free);

	gnome_dialog_set_sensitive (GNOME_DIALOG(dialog), 0, FALSE);

	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (edit_profile_dialog_clicked),
			    p_sdata);

	table = gtk_table_new (4, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), table);

/* FRAME: Server Information */
	frame = gtk_frame_new (_("Server Information"));
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  /* Profile Name */
	label = gtk_label_new (_("Profile Name"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	sprof->profile_name_entry = gtk_entry_new_with_max_length(32);
	if (sdata && sdata->name)
		gtk_entry_set_text (GTK_ENTRY(sprof->profile_name_entry),
				    sdata->name);
	gtk_signal_connect_object (GTK_OBJECT(sprof->profile_name_entry),
				   "changed",
				   GTK_SIGNAL_FUNC(edit_profile_changed),
				   GTK_OBJECT(dialog));
	gtk_box_pack_start (GTK_BOX (hbox), sprof->profile_name_entry, FALSE, FALSE, 0);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  /* Server Address */
	label = gtk_label_new (_("Server Address"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	sprof->server_address = gtk_entry_new_with_max_length (32);
	if (sdata && sdata->address)
		gtk_entry_set_text (GTK_ENTRY(sprof->server_address),
				    sdata->address);
	gtk_signal_connect_object (GTK_OBJECT(sprof->server_address),
				   "changed",
				   GTK_SIGNAL_FUNC(edit_profile_changed),
				   GTK_OBJECT(dialog));
	gtk_box_pack_start (GTK_BOX (hbox), sprof->server_address, FALSE, FALSE, 0);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  /* Server Port */
	label = gtk_label_new (_("Server Port"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	sprof->server_port = gtk_entry_new_with_max_length (32);
	if (sdata && (sdata->port != 0)) {
		gchar *port_str = g_strdup_printf ("%d", sdata->port);
		gtk_entry_set_text (GTK_ENTRY(sprof->server_port), port_str);
		g_free (port_str);
	}
	else
		gtk_entry_set_text (GTK_ENTRY(sprof->server_port), "119");

	gtk_signal_connect_object (GTK_OBJECT(sprof->server_port),
				   "changed",
				   GTK_SIGNAL_FUNC(edit_profile_changed),
				   GTK_OBJECT(dialog));
	gtk_box_pack_start (GTK_BOX (hbox), sprof->server_port, FALSE, FALSE, 0);

	gtk_table_attach (GTK_TABLE(table), frame, 
			  0, 2, 0, 3,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);

/* FRAME: Authorization */
	frame = gtk_frame_new (_("Authorization"));
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	
  /* needs autorization information checkbox */

	sprof->auth_cbutton = gtk_check_button_new_with_label (_("My server requires my username and password"));
	if (sdata && sdata->need_auth)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sprof->auth_cbutton), TRUE);
	else {
	}

	gtk_signal_connect (GTK_OBJECT (sprof->auth_cbutton), "toggled",
			    GTK_SIGNAL_FUNC (auth_cbutton_set), sprof);
	gtk_signal_connect_object (GTK_OBJECT (sprof->auth_cbutton), "toggled",
				   GTK_SIGNAL_FUNC (edit_profile_changed),
				   GTK_OBJECT (dialog));
	gtk_box_pack_start (GTK_BOX (vbox), sprof->auth_cbutton, FALSE, FALSE, 0);


	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  /* Server Username */
	sprof->server_username_label = gtk_label_new (_("Username"));
	gtk_box_pack_start (GTK_BOX (hbox), sprof->server_username_label, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC (sprof->server_username_label), 1.0, 0.5);

	sprof->server_username = gtk_entry_new_with_max_length (32);
	if (sdata && sdata->username)
		gtk_entry_set_text (GTK_ENTRY(sprof->server_username),
				    sdata->username);
	if (sdata && sdata->need_auth)
		gtk_widget_set_sensitive (sprof->server_username, 1);
	else {
		gtk_widget_set_sensitive (GTK_WIDGET (sprof->server_username_label), 0);
		gtk_widget_set_sensitive (sprof->server_username, 0);
	}

	gtk_signal_connect_object (GTK_OBJECT(sprof->server_username),
				   "changed",
				   GTK_SIGNAL_FUNC(edit_profile_changed),
				   GTK_OBJECT(dialog));
	gtk_box_pack_start (GTK_BOX (hbox), sprof->server_username, FALSE, FALSE, 0);


  /* Server Password */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	sprof->server_password_label = gtk_label_new (_("Password"));
	gtk_box_pack_start (GTK_BOX (hbox), sprof->server_password_label, TRUE, TRUE, 0);
//	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (sprof->server_password_label), 1.0, 0.5);

	sprof->server_password = gtk_entry_new_with_max_length (32);
	gtk_entry_set_visibility (GTK_ENTRY(sprof->server_password),
				  FALSE);
	if (sdata && sdata->password)
		gtk_entry_set_text (GTK_ENTRY(sprof->server_password),
				    sdata->password);
	if (sdata && sdata->need_auth)
		gtk_widget_set_sensitive (sprof->server_password, 1);
	else {
		gtk_widget_set_sensitive (GTK_WIDGET (sprof->server_password_label), 0);
		gtk_widget_set_sensitive (sprof->server_password, 0);
	}

	gtk_signal_connect_object (GTK_OBJECT(sprof->server_password),
				   "changed",
				   GTK_SIGNAL_FUNC(edit_profile_changed),
				   GTK_OBJECT(dialog));

	gtk_box_pack_start (GTK_BOX (hbox), sprof->server_password, FALSE, FALSE, 0);

	gtk_table_attach (GTK_TABLE(table), frame, 
			  0, 2, 4, 6,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);
	
/* FRAME: Misc */
        frame = gtk_frame_new (_("Misc"));
        vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
        hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
        gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);
        gtk_container_add (GTK_CONTAINER (frame), vbox);

	/* SPIN: Max Connections */
	label = gtk_label_new (_("Max Connections:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	adj = (GtkAdjustment *) gtk_adjustment_new (4.0, 1.0, 4.0, 1.0, 1.0, 0.0);
	sprof->max_connections = gtk_spin_button_new (adj, 0, 0);

	if (sdata && sdata->max_connections)
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (sprof->max_connections), sdata->max_connections);

	gtk_signal_connect_object (GTK_OBJECT (adj), "value_changed",
				   GTK_SIGNAL_FUNC (edit_profile_changed),
				   GTK_OBJECT (dialog));

	gtk_box_pack_start (GTK_BOX (hbox), sprof->max_connections, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

        /* TOGGLE: Generate Message-ID */
        sprof->msgid_cbutton = gtk_check_button_new_with_label (_("Generate Message-ID"));
        if (sdata && sdata->gen_msgid) {
                gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sprof->msgid_cbutton), TRUE);
	}
        else {
        }
        
        gtk_signal_connect_object (GTK_OBJECT (sprof->msgid_cbutton), "toggled",
                                   GTK_SIGNAL_FUNC (edit_profile_changed),
                                   GTK_OBJECT (dialog));
        
        gtk_box_pack_start (GTK_BOX (vbox), sprof->msgid_cbutton, FALSE, FALSE, 0);

        gtk_table_attach (GTK_TABLE(table), frame,
                          0, 2, 7, 9,
                          GTK_EXPAND | GTK_FILL,
                          GTK_EXPAND | GTK_FILL,
                          0, 0);

	gtk_widget_show_all (dialog);
}


static void
server_clist_button_press (GtkWidget *widget, GdkEventButton *bevent)
{
	int row;
	int column;
	prefs_sdata *p_sdata = NULL;

	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					      "button_press_event");

	if ((bevent->button) == 1 && (bevent->type == GDK_2BUTTON_PRESS)) {
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					      "button_press_event");
		gtk_clist_get_selection_info (GTK_CLIST(widget), bevent->x,
					      bevent->y, &row, &column);
		p_sdata = gtk_clist_get_row_data (GTK_CLIST(widget), row);
		edit_profile_dialog (p_sdata);
	}
}


static void
new_server_cb (void)
{
	edit_profile_dialog (NULL);
}


static void
edit_server_cb (void)
{
	GList *list;
	prefs_sdata *p_sdata = NULL;

	if ((list = GTK_CLIST(prefs_window->server_clist)->selection) == NULL)
		return;
	p_sdata = gtk_clist_get_row_data(GTK_CLIST(prefs_window->server_clist),
						      GPOINTER_TO_INT(list->data));
	edit_profile_dialog (p_sdata);
}


static void
delete_server_cb (void)
{
	GList *l;
	prefs_sdata *p_sdata;

	if ((l = GTK_CLIST(prefs_window->server_clist)->selection) == NULL)
		return;
	p_sdata = gtk_clist_get_row_data (GTK_CLIST(prefs_window->server_clist), GPOINTER_TO_INT(l->data));
	sdata_deletes = g_slist_prepend (sdata_deletes, p_sdata);
	gtk_clist_remove (GTK_CLIST(prefs_window->server_clist), GPOINTER_TO_INT(l->data));
	gnome_property_box_changed (prefs_window->box);
	servers_changed = TRUE;
}


/* Cancel button pressed callback */
static void
prefs_cancel_cb (void)
{
	gint r = 0;
	prefs_sdata *p_sdata = NULL;

	while ((p_sdata = gtk_clist_get_row_data (GTK_CLIST(prefs_window->server_clist), r))) {
		server_free (p_sdata->new);
		g_free (p_sdata);
		r++;
	}
	
	gtk_widget_destroy (GTK_WIDGET(prefs_window->box));
	g_free (prefs_window);
	prefs_window = NULL;
}

/*
static void
online_cbutton_toggled (GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		online_prompt = TRUE;
		gnome_config_set_bool ("/Pan/General/online_prompt", TRUE);
	}
	else {
		online_prompt = FALSE;
		gnome_config_set_bool ("/Pan/General/online_prompt", FALSE);
	}
}
*/


static void
hide_mpart_child_nodes_cbutton_toggled (GtkWidget *widget, gpointer data)
{
	gnome_property_box_changed (prefs_window->box);
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		hide_mpart_child_nodes = 1;
		gnome_config_set_bool ("/Pan/General/hide_mpart_child_nodes", TRUE);
	}
	else {
		hide_mpart_child_nodes = 0;
		gnome_config_set_bool ("/Pan/General/hide_mpart_child_nodes", FALSE);
	}

	/* signal a re-loading of the current articlelist */
	articlelist_changed = TRUE;
}

static void
one_click_preview_cbutton_toggled (GtkWidget *widget, gpointer data)
{
	gnome_property_box_changed (prefs_window->box);
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		one_click_preview = 1;
		gnome_config_set_bool ("/Pan/General/one_click_preview", TRUE);
	}
	else {
		one_click_preview = 0;
		gnome_config_set_bool ("/Pan/General/one_click_preview", FALSE);
	}
}


/*--------------------------------------------------------------------
 * 'general' properties: toggles and such that relate to general
 * functionality and operation of Pan
 *--------------------------------------------------------------------*/
static GtkWidget*
prefs_general_page (void)
{
	GtkWidget *table;
	table = gtk_table_new (10, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	/* go online dialog toggle */
	/* FIXME: Not used 
	prefs_window->online_cbutton = gtk_check_button_new_with_label (_("Ask to go online if a connection is needed"));
	if (online_prompt)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (prefs_window->online_cbutton), TRUE);
		
	gtk_signal_connect (GTK_OBJECT (prefs_window->online_cbutton), "toggled",
			    GTK_SIGNAL_FUNC (online_cbutton_toggled), NULL);

	gtk_table_attach (GTK_TABLE (table), prefs_window->online_cbutton, 0, 1, 0,
			  1, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);
	*/

	/* single click previewing toggle */
	prefs_window->one_click_preview_cbutton =
		gtk_check_button_new_with_label (
			_("Use single click message "
			  "previewing (in paned layout)"));
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (prefs_window->one_click_preview_cbutton),
		one_click_preview);
	gtk_signal_connect (
		GTK_OBJECT (prefs_window->one_click_preview_cbutton),
		"toggled",
		GTK_SIGNAL_FUNC (one_click_preview_cbutton_toggled),
		NULL);

	/* hide 2..n of multipart nodes */
	prefs_window->hide_mpart_child_nodes_cbutton =
		gtk_check_button_new_with_label (
			_("Show only the first part of "
			  "complete multipart binary posts"));
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(prefs_window->hide_mpart_child_nodes_cbutton),
		hide_mpart_child_nodes);
	gtk_signal_connect (
		GTK_OBJECT (prefs_window->hide_mpart_child_nodes_cbutton),
		"toggled",
		GTK_SIGNAL_FUNC (hide_mpart_child_nodes_cbutton_toggled),
		NULL);

	/* use subject as filename for attachments */
	prefs_window->subject_as_filename_cbutton =
		gtk_check_button_new_with_label (
			_("Use article's subject as the filename for "
			  "binary attachments"));
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(prefs_window->subject_as_filename_cbutton),
		use_subject_as_filename);
	gtk_signal_connect_object (
		GTK_OBJECT (prefs_window->subject_as_filename_cbutton),
		"toggled",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));


	gtk_table_attach (
		GTK_TABLE(table),
		prefs_window->one_click_preview_cbutton,
		0, 1, 0, 1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		4, 0);

	gtk_table_attach (
		GTK_TABLE(table),
		prefs_window->hide_mpart_child_nodes_cbutton,
		0, 1, 1, 2,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		4, 0);

	gtk_table_attach (
		GTK_TABLE(table),
		prefs_window->subject_as_filename_cbutton,
		0, 1, 2, 3,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		4, 0);

	gtk_table_attach (
		GTK_TABLE(table),
		gtk_label_new(_("Attribution Line (%i for message-id, %a for author, %d for date)")),
		0, 1, 3, 4,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	prefs_window->attribution_entry = gtk_entry_new();
	gtk_entry_set_text (
		GTK_ENTRY(prefs_window->attribution_entry), attribution_line);
        gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->attribution_entry), "changed",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));
	gtk_table_attach (
		GTK_TABLE(table),
		prefs_window->attribution_entry,
		0, 1, 4, 5,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	gtk_table_attach (
		GTK_TABLE(table),
		gtk_label_new(_("External Mail Program (%r for recipient, %s for subject, %t for body filename)")),
		0, 1, 5, 6,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	prefs_window->external_mailer_entry = gtk_entry_new();
	gtk_entry_set_text (GTK_ENTRY(prefs_window->external_mailer_entry), external_mailer);
        gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->external_mailer_entry), "changed",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));
	gtk_table_attach (
		GTK_TABLE(table),
		prefs_window->external_mailer_entry,
		0, 1, 6, 7,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	gtk_table_attach (
		GTK_TABLE(table),
		gtk_label_new(_("External Editor (%t for temporary filename)")),
		0, 1, 7, 8,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	prefs_window->external_editor_entry = gtk_entry_new();
	gtk_entry_set_text (
		GTK_ENTRY(prefs_window->external_editor_entry), external_editor);
        gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->external_editor_entry), "changed",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));
	gtk_table_attach (
		GTK_TABLE(table),
		prefs_window->external_editor_entry,
		0, 1, 8, 9,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

#if 0
	hbox = gtk_hbox_new (k
	label = gtk_label_new (_("Max Connections:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	adj = (GtkAdjustment *) gtk_adjustment_new (4.0, 1.0, 4.0, 1.0, 1.0, 0.0);
	sprof->max_connections = gtk_spin_button_new (adj, 0, 0);

	/** The maximum number of sockets allowed across all servers */
	//extern int max_socket_connections;
#endif


	return table;
}

static void
set_grouplist_changed (void)
{
	grouplist_changed = TRUE;
}

static void
set_articlelist_font_changed (void)
{
	articlelist_font_changed = TRUE;
}

static void
set_message_body_changed (void)
{
	message_body_changed = TRUE;
}

/* Build and return the 'Display' preferences page, lets users
 * change fonts and colors
 */
static GtkWidget *
prefs_display_page (void)
{
	GtkWidget *table;
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *w;
	gchar *p;

	/* initialize the variables */
	prefs_window->grouplist_gfp = gnome_font_picker_new ();
	prefs_window->articlelist_gfp = gnome_font_picker_new ();
	prefs_window->message_body_gfp = gnome_font_picker_new ();

	prefs_window->unread_color_picker = gnome_color_picker_new ();
	prefs_window->read_color_picker = gnome_color_picker_new ();
	prefs_window->new_color_picker = gnome_color_picker_new ();
	prefs_window->killfile_color_picker = gnome_color_picker_new ();
	prefs_window->watched_color_picker = gnome_color_picker_new ();

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);


/**
 * fonts frame
 **/
	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	frame = gtk_frame_new (_("Fonts"));

   /* grouplist font */
	if ((p = gnome_config_get_string ("/Pan/Display/grouplist_font")) != NULL) {
		gnome_font_picker_set_font_name (GNOME_FONT_PICKER (prefs_window->grouplist_gfp), p);
		g_free (p);
	}

	gnome_font_picker_set_mode (GNOME_FONT_PICKER (prefs_window->grouplist_gfp),
				    GNOME_FONT_PICKER_MODE_FONT_INFO);
	gnome_font_picker_fi_set_use_font_in_label (GNOME_FONT_PICKER (prefs_window->grouplist_gfp),
						    TRUE,
						    DISPLAY_PREFS_PREVIEW_FONT_SIZE);

	gtk_signal_connect (GTK_OBJECT(prefs_window->grouplist_gfp), "font_set",
			    GTK_SIGNAL_FUNC(set_grouplist_changed), NULL);

	gtk_signal_connect_object (GTK_OBJECT (prefs_window->grouplist_gfp), "font_set",
				   GTK_SIGNAL_FUNC (gnome_property_box_changed),
				   GTK_OBJECT (prefs_window->box));

	w = gtk_label_new (_("Group List"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, 0);
	gtk_table_attach (GTK_TABLE (table), prefs_window->grouplist_gfp, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD_SMALL, 0);

	/* articlelist font */
	if ((p = gnome_config_get_string ("/Pan/Display/articlelist_font")) != NULL) {
		gnome_font_picker_set_font_name (GNOME_FONT_PICKER (prefs_window->articlelist_gfp), p);
		g_free (p);
	}
//	else
//		gnome_font_picker_set_font_name (GNOME_FONT_PICKER (prefs_window->articlelist_gfp), "-adobe-helvetica-medium-r-normal-*-*-100-*-*-p-*-iso8859-1");

	gnome_font_picker_set_mode (GNOME_FONT_PICKER (prefs_window->articlelist_gfp),
				    GNOME_FONT_PICKER_MODE_FONT_INFO);
	gnome_font_picker_fi_set_use_font_in_label (GNOME_FONT_PICKER (prefs_window->articlelist_gfp),
						    TRUE,
						    DISPLAY_PREFS_PREVIEW_FONT_SIZE);

	gtk_signal_connect (GTK_OBJECT(prefs_window->articlelist_gfp), "font_set",
			    GTK_SIGNAL_FUNC(set_articlelist_font_changed), NULL);

	gtk_signal_connect_object (GTK_OBJECT(prefs_window->articlelist_gfp), "font_set",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));

	w = gtk_label_new (_("Article List"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, 0);
	gtk_table_attach (GTK_TABLE (table), prefs_window->articlelist_gfp, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD_SMALL, 0);

	/* message body font */
	if ((p = gnome_config_get_string ("/Pan/Display/message_body_font")) != NULL) {
		gnome_font_picker_set_font_name (GNOME_FONT_PICKER (prefs_window->message_body_gfp), p);
		g_free (p);
	}

	gnome_font_picker_set_mode (GNOME_FONT_PICKER (prefs_window->message_body_gfp),
				    GNOME_FONT_PICKER_MODE_FONT_INFO);
	gnome_font_picker_fi_set_use_font_in_label (GNOME_FONT_PICKER(prefs_window->message_body_gfp),
						    TRUE,
						    DISPLAY_PREFS_PREVIEW_FONT_SIZE);

	gtk_signal_connect (GTK_OBJECT(prefs_window->message_body_gfp), "font_set",
			    GTK_SIGNAL_FUNC(set_message_body_changed), NULL);

	gtk_signal_connect_object (GTK_OBJECT(prefs_window->message_body_gfp), "font_set",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));

	w = gtk_label_new (_("Message Body"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, 0);
	gtk_table_attach (GTK_TABLE (table), prefs_window->message_body_gfp, 1, 2, 2, 3,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD_SMALL, 0);

	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	*** colors frame
	**/

	frame = gtk_frame_new (_("Colors"));
	table = gtk_table_new (2, 6, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	/* unread article */

	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (prefs_window->unread_color_picker),
		unread_color.red,
		unread_color.green,
		unread_color.blue,
		gnome_config_get_int ("/Pan/Display/unread_color_a=65534"));
	w = gtk_label_new (_("Unread Message"));
	gtk_misc_set_alignment (
		GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (
		GTK_TABLE (table), w,
		0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, 0);
	gtk_table_attach (
		GTK_TABLE (table), prefs_window->unread_color_picker,
		1, 2, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, 0);
	gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->unread_color_picker), "color_set",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));

	/* read article */
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (prefs_window->read_color_picker),
		read_color.red,
		read_color.green,
		read_color.blue,
		gnome_config_get_int ("/Pan/Display/read_color_a=65534"));
	w = gtk_label_new (_("Read Message"));
	gtk_misc_set_alignment (
		GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (
		GTK_TABLE(table), w,
		0, 1, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (
		GTK_TABLE (table), prefs_window->read_color_picker,
		1, 2, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->read_color_picker), "color_set",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));

	/* new article */
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (prefs_window->new_color_picker),
		new_color.red,
		new_color.green,
		new_color.blue,
		gnome_config_get_int ("/Pan/Display/new_color_a=65534"));
	w = gtk_label_new (_("New Message"));
	gtk_misc_set_alignment (
		GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (
		GTK_TABLE(table), w,
		0, 1, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (
		GTK_TABLE(table), prefs_window->new_color_picker,
		1, 2, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->new_color_picker), "color_set",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));

	/* killfile article */
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (prefs_window->killfile_color_picker),
		killfile_color.red,
		killfile_color.green,
		killfile_color.blue,
		gnome_config_get_int ("/Pan/Display/killfile_color_a=65534"));
	w = gtk_label_new (_("Killfile Message"));
	gtk_misc_set_alignment (
		GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (
		GTK_TABLE (table), w,
		0, 1, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (
		GTK_TABLE (table),
		prefs_window->killfile_color_picker,
		1, 2, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->killfile_color_picker), "color_set",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));

	/* watched thread */
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (prefs_window->watched_color_picker),
		watched_color.red,
		watched_color.green,
		watched_color.blue,
		gnome_config_get_int ("/Pan/Display/watched_color_a=65534"));
	w = gtk_label_new (_("Tracked Thread"));
	gtk_misc_set_alignment (
		GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (
		GTK_TABLE (table), w,
		0, 1, 4, 5,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (
		GTK_TABLE (table),
		prefs_window->watched_color_picker,
		1, 2, 4, 5,
		GTK_FILL, GTK_FILL, GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect_object (
		GTK_OBJECT(prefs_window->watched_color_picker), "color_set",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(prefs_window->box));
	gtk_container_add (GTK_CONTAINER (frame), table);

	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	return vbox;
}


/*---[ prefs_directories_page ]---------------------------------------
 * Directory preferences page for setting the data, temp, and download
 * dirs in Pan.
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_directories_page (void)
{
	GtkWidget *table;
	GtkWidget *label;
	gchar *p;

	table = gtk_table_new (3, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("Download Directory:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->dir_download = gnome_file_entry_new ("Prefs", _("Download Directory"));
	if ((p = gnome_config_get_string ("/Pan/Paths/download_dir")) == NULL)
		p = gnome_util_prepend_user_home (".pan/download");
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->dir_download))), p);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (prefs_window->dir_download), TRUE);

	g_free (p);
	gtk_signal_connect_object (GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->dir_download))),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT (prefs_window->box));
	gtk_table_attach (GTK_TABLE(table), prefs_window->dir_download, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Temporary Directory:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->dir_temp = gnome_file_entry_new ("Prefs", _("Temporary Directory"));
	if ((p = gnome_config_get_string("/Pan/Paths/temp_dir")) == NULL) {
		p = gnome_util_prepend_user_home(".pan/tmp");
	}
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->dir_temp))), p);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (prefs_window->dir_temp), TRUE);
	g_free (p);
	gtk_signal_connect_object (GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->dir_temp))),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT (prefs_window->box));
	gtk_table_attach (GTK_TABLE(table), prefs_window->dir_temp, 1, 2, 1,
			  2, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

/* Data Directory: */
	label = gtk_label_new (_("Data Directory:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->dir_data = gnome_file_entry_new ("Prefs", _("Data Directory"));
	if ((p = gnome_config_get_string("/Pan/Paths/data_dir")) == NULL) {
		p = gnome_util_prepend_user_home(".pan/data");
	}
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->dir_data))), p);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (prefs_window->dir_data), TRUE);
	g_free (p);
	gtk_signal_connect_object (GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->dir_data))),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT (prefs_window->box));
	gtk_table_attach (GTK_TABLE(table), prefs_window->dir_data, 1, 2, 2,
			  3, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	return table;
}


/*---[ prefs_userinfo_page ]------------------------------------------
 * page that holds the user information such as full name,
 * e-mail, organization, and signature file (this page is optional).
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_userinfo_page (void)
{
	GtkWidget *table;
	GtkWidget *label;
	//GtkWidget *frame;
	char *p;

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	//frame = gtk_frame_new (_("Personal Information"));

	label = gtk_label_new (_("Full Name:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->user_fullname = gtk_entry_new ();
	if ((p = gnome_config_get_string("/Pan/User/Full_Name"))) {
		gtk_entry_set_text (GTK_ENTRY(prefs_window->user_fullname), p);
		g_free(p);
	}
	gtk_signal_connect_object (GTK_OBJECT(prefs_window->user_fullname),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));
	gtk_widget_show (prefs_window->user_fullname);
	gtk_table_attach (GTK_TABLE(table), prefs_window->user_fullname, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Email Address:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->user_email = gtk_entry_new ();
	if ((p = gnome_config_get_string("/Pan/User/Email"))) {
		gtk_entry_set_text (GTK_ENTRY(prefs_window->user_email), p);
		g_free(p);
	}
	gtk_signal_connect_object (GTK_OBJECT(prefs_window->user_email),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));
	gtk_widget_show (prefs_window->user_email);
	gtk_table_attach (GTK_TABLE(table), prefs_window->user_email, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->user_organization = gtk_entry_new ();
	if ((p = gnome_config_get_string("/Pan/User/Organization"))) {
		gtk_entry_set_text (GTK_ENTRY(prefs_window->user_organization), p);
		g_free(p);
	}
	gtk_signal_connect_object (GTK_OBJECT(prefs_window->user_organization),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));
	gtk_widget_show (prefs_window->user_organization);
	gtk_table_attach (GTK_TABLE(table), prefs_window->user_organization, 1,
			  2, 2, 3, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Signature File:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, 0, 0);

	prefs_window->user_signature_file = gnome_file_entry_new ("Prefs", _("Signature File:"));
	if ((p = gnome_config_get_string("/Pan/User/Signature_File"))) {
		gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->user_signature_file))), p);
		g_free(p);
	}
	gtk_signal_connect_object (GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (prefs_window->user_signature_file))),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));
	gtk_widget_show (GTK_WIDGET (prefs_window->user_signature_file));
	gtk_table_attach (GTK_TABLE(table), GTK_WIDGET (prefs_window->user_signature_file), 1, 2, 3, 4,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);
	return table;
}

static void
header_selection_changed (GtkCList* clist,
			  gint row,
			  gint column,
			  GdkEventButton *event,
			  gpointer user_data)
{
	gnome_property_box_changed (prefs_window->box);
}

static gulong
get_header_flags (void)
{
	GtkCList *clist = GTK_CLIST(prefs_window->article_headers_clist);
	GList* sel = clist->selection;
	gulong flags = 0;

	while ( sel != NULL ) {
		int row_index = GPOINTER_TO_INT ( sel->data );
		gulong* pulong = gtk_clist_get_row_data ( clist, row_index );
		flags |= *pulong;
		sel = sel->next;
	}

	return flags;
}

typedef struct
{
	gchar* text;
	gulong flag;
} MessageListThrowawayStruct;

static GtkWidget *
prefs_message_page (void)
{
	int i;
	GtkWidget *label;
	GtkWidget *vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	GtkCList* list = GTK_CLIST ( gtk_clist_new ( 1 ) );
	const ulong current_flags = (ulong) gnome_config_get_int_with_default("/Pan/State/Headers=243", NULL);
	MessageListThrowawayStruct rows [] = {
		{ _("Author"), HEADER_AUTHOR },
		{ _("Date"), HEADER_DATE },
		{ _("Followup-To (if different from Newsgroups)"), HEADER_FOLLOWUP_TO },
		{ _("Message-Id"), HEADER_MESSAGE_ID },
		{ _("Newsgroups (if more than one group)"), HEADER_NEWSGROUPS },
		{ _("References (if article is a reply)"), HEADER_REFERENCES },
		{ _("Reply-To (if different from Author)"), HEADER_REPLY_TO },
		{ _("Subject"), HEADER_SUBJECT }
	};
	const int row_qty = sizeof(rows) / sizeof(rows[0]);

	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);

	prefs_window->article_headers_clist = GTK_WIDGET(list);
	gtk_clist_set_selection_mode ( list, GTK_SELECTION_MULTIPLE );

	for ( i=0; i!=row_qty; ++i )
	{
		gint row = gtk_clist_insert ( list, -1, &rows[i].text );

		gulong* plong = g_new0 ( gulong, 1 );
		*plong = rows[i].flag;
		gtk_clist_set_row_data_full ( list, row, plong, g_free );

		if ( rows[i].flag & current_flags )
			gtk_clist_select_row ( list, row, -1 );
	}

	gtk_signal_connect ( GTK_OBJECT(list), "select-row",
		GTK_SIGNAL_FUNC(header_selection_changed), NULL );
	gtk_signal_connect ( GTK_OBJECT(list), "unselect-row",
		GTK_SIGNAL_FUNC(header_selection_changed), NULL );

	label = gtk_label_new (_("Select the fields you would like to see at the top of each message that is displayed in the message pane.") );
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);

	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET(list), FALSE, FALSE, GNOME_PAD_SMALL);

	return vbox;
}

static void
clear_acache_clicked_cb (GtkButton *button, gpointer data)
{
	const int i = acache_expire_all ();
	if (i != 0)
	{
		gchar* buf = g_strdup_printf ("%d files deleted.", i);
		gnome_ok_dialog (buf);
		g_free (buf);
	}
}

/* Build and return the 'Cache' page */
static GtkWidget*
prefs_cache_page (void)
{
	const int cache_max = gnome_config_get_int_with_default("/Pan/Cache/MaxSize=1242880", NULL);
	const gboolean flush_on_exit = gnome_config_get_bool("/Pan/Cache/FlushOnExit=true");
	GtkWidget *label;
	GtkWidget *vbox = gtk_vbox_new (FALSE, GNOME_PAD);
	GtkWidget *table;
	GtkWidget *hbox;
	GtkWidget *button;
	GtkWidget *tb;
	char buf[16];

	label = gtk_label_new (_("The cache is used to keep local copies of articles\nand thus reduce the time connected to the network."));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);

	table = gtk_table_new (3, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	label = gtk_label_new (_("Disk Cache:"));
	gtk_widget_show (label);
	gtk_table_attach_defaults (GTK_TABLE(table), label, 0, 1, 0, 1);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	prefs_window->cache_kbytes_entry = gtk_entry_new();
	gtk_widget_show (prefs_window->cache_kbytes_entry);
	sprintf (buf, "%d", cache_max);
	gtk_entry_set_text (GTK_ENTRY(prefs_window->cache_kbytes_entry), buf);
	gtk_signal_connect_object (GTK_OBJECT(prefs_window->cache_kbytes_entry),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT (prefs_window->box));
	gtk_box_pack_start (GTK_BOX(hbox), prefs_window->cache_kbytes_entry, FALSE, FALSE, GNOME_PAD_SMALL);
	label = gtk_label_new (_("bytes"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_widget_show (hbox);
	gtk_table_attach_defaults (GTK_TABLE(table), hbox, 1, 2, 0, 1);

	button = gtk_button_new_with_label (_("Clear Disk Cache"));
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (clear_acache_clicked_cb),
			    NULL);
	gtk_widget_show (button);
	gtk_table_attach_defaults (GTK_TABLE(table), button, 2, 3, 0, 1);
	gtk_widget_show (table);

	tb = prefs_window->flush_cache_on_exit_check =
		gtk_check_button_new_with_label (_("Flush Cache when exiting Pan"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(tb), flush_on_exit);
	gtk_signal_connect_object (GTK_OBJECT(tb), "toggled",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT (prefs_window->box));
	gtk_table_attach_defaults (GTK_TABLE(table), tb, 0, 3, 1, 2);


	gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_widget_show (vbox);
	return vbox;
}

/* Build and return the 'SMTP Server' page */
static GtkWidget *
prefs_smtp_page (void)
{
	GtkWidget *table;
	GtkWidget *label;
	gchar *tmp;

	table = gtk_table_new (2, 2, FALSE);
        gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
        gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);


	label = gtk_label_new (_("Outgoing Mail (SMTP) Server:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 4, 0);

	prefs_window->smtp_address = gtk_entry_new ();
	if ((tmp = gnome_config_get_string ("/Pan/Mail/smtp_address"))) {
		gtk_entry_set_text (GTK_ENTRY (prefs_window->smtp_address), tmp);
		g_free (tmp);
	}
	gtk_table_attach (GTK_TABLE (table), prefs_window->smtp_address, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	gtk_signal_connect_object (GTK_OBJECT(prefs_window->smtp_address),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));

	label = gtk_label_new (_("SMTP Port:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 4, 0);

	prefs_window->smtp_port = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (prefs_window->smtp_port), "25");
	gtk_table_attach (GTK_TABLE (table), prefs_window->smtp_port, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	gtk_signal_connect_object (GTK_OBJECT(prefs_window->smtp_port),
				   "changed",
				   GTK_SIGNAL_FUNC(gnome_property_box_changed),
				   GTK_OBJECT(prefs_window->box));

	return table;
}

/*---[ prefs_server_page ]--------------------------------------------
 * This is the Server tab of the preferences.  It has a list of the
 * server's and buttons that allow users to make a new server entry,
 * edit a previous one, or delete one.  This doesn't actually let
 * them switch servers, that's done from the main menu of Pan.
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_server_page (void)
{
	GtkWidget *vbox, *hbox, *button;

	char *titles[] = {
		_("Name"),
		_("Address"),
	};

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);

	prefs_window->server_clist = gtk_clist_new_with_titles (2, titles);
	gtk_clist_column_titles_passive (GTK_CLIST(prefs_window->server_clist));
	gtk_clist_set_selection_mode (GTK_CLIST(prefs_window->server_clist),
				      GTK_SELECTION_SINGLE);

//	gtk_clist_set_column_width (GTK_CLIST(prefs_window->server_clist), 0, 200);
//	gtk_clist_set_column_width (GTK_CLIST(prefs_window->server_clist), 1, 40);
	gtk_clist_set_column_width (GTK_CLIST(prefs_window->server_clist), 0, 100);
	gtk_clist_set_column_auto_resize (GTK_CLIST(prefs_window->server_clist), 1, TRUE);

	gtk_widget_set_usize (prefs_window->server_clist, 280, 200);
	gtk_box_pack_start (GTK_BOX(hbox), prefs_window->server_clist, TRUE,
			    TRUE, 5);
	gtk_signal_connect (GTK_OBJECT(prefs_window->server_clist),
			    "button_press_event",
			    GTK_SIGNAL_FUNC(server_clist_button_press), NULL);

	prefs_create_clist ();

	/* one button vbox */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), vbox, FALSE, FALSE, 5);

	/* new server button */
	button = gtk_button_new_with_label (_("New..."));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GNOME_PAD);
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (new_server_cb), NULL);

	/* edit server button */
	button = gtk_button_new_with_label (_("Edit..."));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GNOME_PAD);
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (edit_server_cb), NULL);

	/* delete server button */
	button = gtk_button_new_with_label(_("Delete"));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GNOME_PAD);
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (delete_server_cb), NULL);

/* make default button */
//	button = gtk_button_new_with_label ("Make Default");
//	gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, GNOME_PAD);
//	gtk_signal_connect (GTK_OBJECT (button), "clicked",
//			    GTK_SIGNAL_FUNC (default_server_cb), NULL);

	return hbox;
}

/**
 * prefs_spawn:
 *
 * Build and draw the main Preferences dialog.  Its a GnomePropertyBox
 * with a notebook in it.
 **/
void
prefs_spawn (void)
{
	GtkWidget *hbox;

	/* There can be only one! (properties window) */
	if (prefs_window) {
		gdk_window_raise (GTK_WIDGET(prefs_window->box)->window);
		return;
	}

	prefs_window = (PrefsWindow *) g_malloc0 (sizeof(PrefsWindow));
	prefs_window->box = GNOME_PROPERTY_BOX (gnome_property_box_new());

	gtk_window_set_title (GTK_WINDOW (prefs_window->box), _("Pan: Preferences"));

	hbox = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

	sections_clist = gtk_clist_new (1);
	gtk_clist_set_selection_mode (GTK_CLIST (sections_clist),
				      GTK_SELECTION_BROWSE);
	gtk_signal_connect (GTK_OBJECT (sections_clist), "select_row",
			    GTK_SIGNAL_FUNC (prefs_select_row_cb), NULL);


//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_server_page(),
//				  gtk_label_new (_("News Servers")));

	prefs_add_section_page (_("Cache"),
				prefs_cache_page());

	prefs_add_section_page (_("General"),
				prefs_general_page());

	prefs_add_section_page (_("Headers"),
				prefs_message_page());

	prefs_add_section_page (_("Directories"),
				prefs_directories_page());

	prefs_add_section_page (_("Display"),
				prefs_display_page());

	prefs_add_section_page (_("Mail Server"),
				prefs_smtp_page());

	prefs_add_section_page (_("News Servers"),
				prefs_server_page());

	prefs_add_section_page (_("User"),
				prefs_userinfo_page());

	gtk_notebook_set_show_tabs (GTK_NOTEBOOK(prefs_window->box->notebook),
				    FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK(prefs_window->box->notebook),
				      FALSE);

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_smtp_page(),
//				  gtk_label_new (_("Mail Server")));

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_cache_page(),
//				  gtk_label_new (_("Cache")));

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_general_page(),
//				  gtk_label_new (_("General")));

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_userinfo_page(),
//				  gtk_label_new (_("User")));

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_directories_page(),
//				  gtk_label_new (_("Directories")));

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_message_page(),
//				  gtk_label_new (_("Headers")));

//	gtk_notebook_append_page (GTK_NOTEBOOK(prefs_window->box->notebook),
//				  prefs_display_page(),
//				  gtk_label_new (_("Display")));

	gtk_signal_connect (GTK_OBJECT (prefs_window->box), "apply",
			    GTK_SIGNAL_FUNC (prefs_apply_cb), NULL);
	gtk_signal_connect (GTK_OBJECT (prefs_window->box), "destroy",
			    GTK_SIGNAL_FUNC (prefs_cancel_cb), NULL);


	gtk_widget_ref (GNOME_PROPERTY_BOX (prefs_window->box)->notebook);
	
	gtk_container_remove (GTK_CONTAINER (GNOME_DIALOG (prefs_window->box)->vbox),
			      GNOME_PROPERTY_BOX (prefs_window->box)->notebook);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG(prefs_window->box)->vbox),
			    hbox, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox),
			    sections_clist,
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox),
			    GNOME_PROPERTY_BOX (prefs_window->box)->notebook,
			    TRUE, TRUE, 0);

	gtk_widget_unref (GNOME_PROPERTY_BOX (prefs_window->box)->notebook);
			    
	gtk_widget_set_usize (GTK_WIDGET (sections_clist), 120, 0);

	gui_popup_draw (GTK_WIDGET (prefs_window->box), Pan.window);
}

#if 0
void
default_server_cb (void)
{
	GList *l;
	server_data *data;

	l = GTK_CLIST(prefs_window->server_clist)->selection;

	data = (server_data *) gtk_clist_get_row_data (GTK_CLIST (GTK_CLIST(prefs_window->server_clist)), GPOINTER_TO_INT (l->data));

/* FIXME if this code is ever made live again, remember to update the active server db here.
   better yet, just call set_active_server() from here */
	Pan.active_server = data;
	prefs_update_clist ();
	gnome_property_box_changed (prefs_window->box);
}
#endif
