/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __DB_H__
#define __DB_H__

typedef struct PanDB * pan_db;

/**
 * Creates a new instance of a database accessor for the specified file.
 * This accessor is not shared.  For a shared accessor, one must call pan_db_ref().
 * @param filename
 * @return db_accessor
 */
extern pan_db pan_db_open ( const char* filename );

/**
 * Closes a database accessor opened by pan_db_open.
 * @param db_accessor db to close
 */
extern void pan_db_close ( pan_db );

/**
 * Returns a shared database accessor for the specified file,
 * creating a new one if no shared accessor already exists.
 * @param filename
 * @return db_accessor
 */
extern pan_db pan_db_ref ( const char* filename );

/**
 * Unreferences a shared database accessor for the specified file.
 * When the last client unreferences the accessor, the accesor is
 * internally closed via pan_db_close().
 * @param db_accessor
 */
extern void pan_db_unref ( pan_db );

typedef void (*DBFunc)( const char* key, const char* val, gpointer user_data);

/**
 * Calls func for each key/value pair in the database.
 * @param db_accessor
 * @param func DBFunc to be called for each key/value pair
 * @param user_data to be passed as the user_data parameter to func
 */
extern void pan_db_foreach ( pan_db, DBFunc func, gpointer user_data );

/**
 * Erases a record from the database.
 * @param db_accessor
 * @param key 
 */
extern void pan_db_erase ( pan_db, const char* key );

/**
 * Erases all the records from the database.
 * @param db_accessor
 */
extern void pan_db_erase_all ( pan_db );

/**
 * Retrieves a value from the database.
 * @param db_accessor
 * @param key
 * @return value, or NULL if none.  This value must be g_free()d by the client.
 */
extern gchar* pan_db_get_value_str ( pan_db, const char* key );

/**
 * Writes a value to the database, overwriting any previous value at that key.
 * @param db_accessor
 * @param key 
 * @param value
 */
extern void pan_db_put_value_str ( pan_db, const char* key, const char* value );

/**
 * Convenience function that converts the value of pan_db_get_value_str to an int.
 * @param db_accessor
 * @param key
 */
extern int pan_db_get_value_i ( pan_db, const char* key );

/**
 * Convenience function that converts the int to a string and calls pan_db_put_value_str
 * @param db_accessor
 * @param key
 * @param int_value
 */
extern void pan_db_put_value_i ( pan_db, const char* key, int i );

/**
 * Convenience function that interprets the current database value associated with
 * key as an int, adds i to it, and writes it back to the database.
 * @param db_accessor
 * @param key
 * @param increment
 */
extern void pan_db_add_value_i ( pan_db, const char* key, int i );

/**
 * Flushes all database changes to the database file.
 * This should be called after all changes.  Note that it's much more efficient
 * to call this after batching a group of changes, so it's not automatically called
 * by the database mutator functions.
 * @param db_accessor
 */
extern void pan_db_sync ( pan_db );

#endif /* __DB_H__ */
