/*
fetchnews -- post articles to and get news from upstream server(s)

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998, 1999.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"
#include <sys/types.h>
#include <ctype.h>
#include <dirent.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <syslog.h>
#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <utime.h>

int verbose = 0;
int debug = 0;

struct serverlist * current_server;

int artno;

time_t now;

struct stringlist *posted = NULL;

/* Variables set by command-line options which are specific for fetch */
int extraarticles = 0;
int usesupplement = 0;
int postonly = 0;	/* if 1, don't read files from upstream server */
int noexpire = 0;	/* if 1, don't automatically unsubscribe newsgroups */
int forceactive = 0;	/* if 1, reread complete active file */
int headerbody = 0;	/* if 1, get headers only; if 2, get bodies only */

static jmp_buf jmpbuffer ;

int isgrouponserver( char * newsgroups );
int ismsgidonserver( char * msgid );
void delposted( void );
int age( const char * date );
void supersede( const char * msgid );
int getgroup ( struct newsgroup *g , int server );
int postarticles( void );
void fixxover( void );


static void tabstospaces(char *s) {
while (*s)
    {
	if (*s=='\t')
		*s=' ';
	s++;
    }

}


#ifdef NOTYET
static void _ignore_answer( FILE * f ) {
    char * l;
    l = NULL;
    while ( ((l=getaline(f)) != NULL) && strcmp(l, ".") )
	;
}
#endif

static void sig_int( int signo ) {
    if ( signo == SIGINT || signo == SIGTERM )
	longjmp( jmpbuffer, 1 );
}

#ifdef NOTYET
static int testheaderbody( char optchar ) {
    if ( !delaybody ) {
	printf( "Option -%c can only be used in conjunction with delaybody -"
		"ignored\n", optchar );
	return 0;
    }
    if ( headerbody == 1 ) {	/* already seen -H */
	printf( "Option -%c and -H cannot be used together - ignored\n",
		optchar );
	return 0;
    }
    if ( headerbody == 2 ) {	/* already seen -B */
	printf( "Option -%c and -B cannot be used together - ignored\n",
		optchar );
	return 0;
    }
    return 1;
}
#endif

static void usage( void ) {
   fprintf( stderr, "Usage: fetchnews [-v] [-x #] [-l] [-n] [-f] [-P]\n"
                    "  -v: more verbose (may be repeated)\n"
                    "  -x: check for # extra articles in each group\n"
                    "  -l: do not use supplementary servers\n"
                    "  -n: do not automatically expire unread groups\n"
		    "  -f: force reload of groupinfo file\n"
		    "  -P: only post outgoing articles, don't fetch any\n"
		    );
}

/*
 * check whether any of the newsgroups is on server
 * return TRUE if yes, FALSE otherwise
 */
int isgrouponserver( char * newsgroups ) {
    char * p, *q;
    int retval;

    if ( !newsgroups )
	return FALSE;

    retval = FALSE;
    p = newsgroups;
    do {
	q = strchr( p, ',' );
	if ( q )
	    *q++ = '\0';
	sprintf( lineout, "GROUP %s\r\n", p );
	putaline();
	if ( nntpreply() == 211 )
	    retval = TRUE;
	p = q;
	while ( p && *p && isspace((unsigned char)*p) )
	    p++;
    } while ( q && !retval );

    return retval;
}

/*
 * check whether message-id is on server
 * return TRUE if yes, FALSE otherwise
 */
int ismsgidonserver( char * msgid ) {
    if ( !msgid )
        return FALSE;
    sprintf( lineout, "STAT %s\r\n", msgid );
    putaline();
    if ( nntpreply() == 223 )
	return TRUE;
    else
	return FALSE;
}

/*
 * delete posted articles. If anything was not posted correctly, there
 * is still a copy in /var/spool/news/failed.postings/
 */
void delposted( void ) {
    struct stringlist *a, *b;
    struct stat st;
    struct dirent * de;
    DIR * d;

    if ( chdir( spooldir ) || chdir ( "out.going" ) ) {
	printf( "Unable to cd to outgoing directory\n" );
        syslog( LOG_ERR, "Unable to cd to outgoing directory: %m" );
        return;
    }

    a = posted;
    while ( a ) {
	b = a;
	unlink( a->string );
	a = a->next ;
	free( b );
    }

    d = opendir( "." );
    if ( !d ) {
	syslog( LOG_ERR, "Unable to opendir out.going: %m" );
	printf( "Unable to opendir %s/out.going\n", spooldir );
	return;
    }

    while( ( de = readdir( d ) ) != NULL ) {
	if ( ( stat( de->d_name, &st ) == 0 ) && S_ISREG( st.st_mode ) ) {
	    if ( st.st_nlink == 1 )
		;       /* possibly new posting since fetch started */
	    else if ( unlink( de->d_name ) )
		/* delete stuff that has been moved to failed.postings */
		syslog( LOG_ERR, "Failed to unlink %s: %m", de->d_name );
	}
    }
    closedir( d );
}

int age( const char * date ) {
    char monthname[4];
    int month;
    int year;
    int day;
    const char * d;
    time_t tmp;
    struct tm time_struct;

    if ( !date )
	return 1000; /* large number: OLD */
    d = date;
    if(!(strncasecmp(d, "date:", 5)))
	d += 5;
    while( isspace((unsigned char)*d) )
	d++;

    if ( isalpha((unsigned char)*d) ) {
	while ( !isspace((unsigned char)*d) )	/* skip "Mon" or "Tuesday," */
	    d++;
    }

    /* RFC 822 says we have 1*LWSP-char between tokens */
    while ( isspace( (unsigned char)*d ) )
	d++;

    /* parsing with sscanf leads to crashes */
    day = strtol( d, NULL, 10 );
    while ( isdigit((unsigned char)*d) || isspace((unsigned char)*d) )
	d++;
    if ( !isalpha((unsigned char)*d) ) {
	syslog( LOG_INFO, "Unable to parse %s", date );
	return 1003;
    }
    monthname[0] = *d++;
    monthname[1] = *d++;
    monthname[2] = *d++;
    monthname[3] = '\0';
    if ( strlen(monthname) != 3 ) {
	syslog( LOG_INFO, "Unable to parse month in %s", date );
	return 1004;
    }
    while ( isalpha((unsigned char)*d) )
	d++;
    while ( isspace((unsigned char)*d) )
	d++;
    year = strtol( d, NULL, 10 );

    if ( ( year < 1970 ) && ( year > 99 ) ) {
	syslog( LOG_INFO, "Unable to parse year in %s", date );
	return 1005;
    } else if ( !(day > 0 && day < 32) ) {
	syslog( LOG_INFO, "Unable to parse day in %s", date );
	return 1006;
    } else {
	if ( !strcasecmp( monthname, "jan" ) )
	    month = 0;
	else if ( !strcasecmp( monthname, "feb" ) )
	    month = 1;
	else if ( !strcasecmp( monthname, "mar" ) )
	    month = 2;
	else if ( !strcasecmp( monthname, "apr" ) )
	    month = 3;
	else if ( !strcasecmp( monthname, "may" ) )
	    month = 4;
	else if ( !strcasecmp( monthname, "jun" ) )
	    month = 5;
	else if ( !strcasecmp( monthname, "jul" ) )
	    month = 6;
	else if ( !strcasecmp( monthname, "aug" ) )
	    month = 7;
	else if ( !strcasecmp( monthname, "sep" ) )
	    month = 8;
	else if ( !strcasecmp( monthname, "oct" ) )
	    month = 9;
	else if ( !strcasecmp( monthname, "nov" ) )
	    month = 10;
	else if ( !strcasecmp( monthname, "dec" ) )
	    month = 11;
	else {
	    syslog( LOG_INFO, "Unable to parse %s", date );
	    return 1001;
	}
	if ( year < 70 )        /* years 2000-2069 in two-digit form */
	    year += 100;
	else if ( year > 1970 ) /* years > 1970 in four-digit form */
	    year -= 1900;

	memset( &time_struct, 0, sizeof(time_struct) );
        time_struct.tm_sec = 0;
	time_struct.tm_min = 0;
	time_struct.tm_hour = 0;
	time_struct.tm_mday = day;
	time_struct.tm_mon = month;
	time_struct.tm_year = year;
	time_struct.tm_isdst = 0;

	tmp = mktime( &time_struct );

	if ( tmp == -1 )
	    return 1002;
	
	return( ( now - tmp ) / SECONDS_PER_DAY );
    }
}		       

/*
 * delete a message
 */
void supersede( const char *msgid ) {
    const char * filename;
    char *p, *q, *r, *hdr;
    struct stat st;

    filename = lookup( msgid );
    if ( !filename )
	return;

    p = hdr = getheader( filename, "Xref:" );
    if ( !hdr )
	return;

    /* jump hostname entry */
    while ( p && !isspace((unsigned char)*p) )
	p++;
    while ( p && isspace((unsigned char)*p) )
	p++;
    /* now p points to the first newsgroup */

    /* unlink all the hardlinks in the various newsgroups directories */
    while ( p && ( ( q = strchr( p, ':' ) ) != NULL ) ) {
	*q++ = '\0';
	if ( chdirgroup( p, FALSE ) ) {
	    r = p;
	    p = q;
	    while ( q && !isspace((unsigned char)*q) )
		q++;
	    *q = '\0';
#ifdef NOTYET
	    if ( unlink( p ) )
		syslog( LOG_ERR, "Failed to unlink %s: %s: %m", r, p );
	    else {
#endif
		if ( verbose > 2 )
		    printf( "Superseded %s in %s\n", p, r );
		syslog( LOG_INFO, "Superseded %s in %s", p, r );
#ifdef NOTYET
	    }
#endif
	}
	while ( p && !isspace((unsigned char)*p) )
	    p++;
	while ( p && isspace((unsigned char)*p) )
	    p++;
    }

    /* unlink the message-id hardlink */
    if ( stat( filename, &st ) )
	syslog( LOG_ERR, "%s does not exist: %m", filename );
    else if ( st.st_nlink > 1 )
	syslog( LOG_ERR, "%s: link count is %d", filename, st.st_nlink );
#ifdef NOTYET
    else if ( unlink( filename ) )
	syslog( LOG_ERR, "Failed to unlink %s: %m", filename );
#endif
    else {
	if ( verbose > 2 )
	    printf( "Superseded %s\n", filename );
	syslog( LOG_INFO, "Superseded %s", filename );
    }
    free( hdr );
}
    
/*
 * Get body of a single message of which the header has already been
 * downloaded and append it to the file with the header.
 * Returns 0 if file could not be retrieved, 1 otherwise.
 */
static int getbody( struct newsgroup* group, int id ) {
    const char *c ;
    char *l;
    char *p, *q;
    char messageid[ 1024 ];
    char newsgroups[ 1024 ];	/* I hope this is enough */
    char xref[ 4096 ];
    FILE *f, *g;

    (void) chdirgroup( group->name, TRUE );

    /* extract message-id: and xref: headers */
    sprintf( s, "%d", id );
    if ( ! ( f = fopen( s, "r" ) ) ) {
        syslog( LOG_INFO, "%s: cannot open %s for reading -- possibly expired",
		group->name, s );
        return 1;	/* pretend to have read file successfully so that
			   it is purged from the list */
    }
    strcpy( messageid, "" );
    strcpy( xref, "" );
    strcpy( newsgroups, "" );
    debug = 0;
    while ( ( l = getaline( f ) ) != NULL ) {
	if ( ! strncmp( l, "Newsgroups: ", 12 ) )
            strcpy( newsgroups, l + 12 );
        if ( ! strncmp( l, "Message-ID: ", 12 ) )
            strcpy( messageid, l + 12 );
	if ( ! strncmp( l, "Xref: ", 6 ) )
	    strcpy( xref, l + 6 );
    }
    debug = debugmode;
    fclose( f );

    /* check whether we can retrieve the body */
    if ( verbose > 2 )
        printf( "%s: BODY %s\n", group->name, messageid );
    sprintf( lineout, "BODY %s\r\n", messageid );
    putaline();

    if ( nntpreply() != 222 ) {
        syslog( LOG_ERR, "%s: Retrieving body %s failed. No response",
                group->name, messageid );
        return 0;
    }

    sprintf( s, "%d", id );
    c = lookup( messageid );
    unlink( c );	/* make space for new file */
    
    if ( ! ( f = fopen( c, "w" ) ) ) {
	syslog( LOG_ERR, "%s: cannot open %s for writing", group->name, l );
	link( s, c );	/* if we can't open new file restore old one */
	return 0;
    }

    /* copy all headers except Xref: into new file */
    g = fopen( s, "r" );
    debug = 0;
    while ( ( l = getaline( g ) ) != NULL ) {
	if ( strncmp( l, "Xref: ", 6 ) ) 
	    fprintf( f, "%s\n", l );
    }
    debug = debugmode;
    fclose( g );

    /* create a whole bunch of new hardlinks */
    store( c, f, 0, newsgroups, "", "", "", messageid, "", "" );

    /* retrieve body */
    fprintf( f, "\n" ); /* Empty line between header and body. */
    debug = 0;
    while ( ( ( l = getaline( nntpin ) ) != NULL ) && strcmp( l, "." ) ) {
        if ( *l == '.' )
            ++l;
        fprintf( f, "%s\n", l );
    }
    debug = debugmode;
    if ( ! l ) {
        syslog( LOG_ERR, "%s: Retrieving body %s failed. "
                "Transmission interrupted.",
                group->name, messageid );
        fprintf( f, "\n\t[ Leafnode: ]\n"
                    "\t[ An error occured while "
                    "retrieving this message. ]\n" );
	fclose( f );
	return 0;
    }
    fclose( f );

    /* Remove old article files since we don't need them anymore.
       This is done by evaluating the old Xref: header.
     */
    if ( xref[0] == '\0' )	/* no Xref: header --> make a fake one */
	sprintf( xref, "%s %s:%d", fqdn, group->name, id );

    if ( debugmode )
        syslog( LOG_DEBUG, "xref: %s", xref );
    c = strchr( xref, ' ' );
    while ( ( c++ ) && ( *c ) && ( q = strchr( c, ':' ) ) != NULL ) {
	*q++ = '\0';
	if ( ( p = strchr( q, ' ' ) ) != NULL )
	    *p = '\0';
	
	/* c points to the newsgroup, q to the article number */
	(void) chdirgroup( c, TRUE );
	if ( unlink( q ) )
	    syslog( LOG_NOTICE, "unlinking %s/%s failed", c, q );
	else if ( debugmode )
	    syslog( LOG_DEBUG, "unlinking %s/%s", c, q );

	c = strchr( q, ' ' );
    }
    return 1;
}

/*
 * Get bodies of messages that have marked for download.
 * The group must already be selected at the remote server and
 * the current directory must be the one of the group.
 */
static void getmarked( struct newsgroup* group ) {
    int n, i;
    FILE *f;
    char filename[ 1024 ];
    int id[ BODY_DOWNLOAD_LIMIT ];

    if ( verbose )
        printf( "Getting bodies of marked messages for group %s ...\n",
                group->name );
    n = 0;
    sprintf( filename, "%s/interesting.groups/%s", spooldir, group->name );
    if ( ! ( f = fopen( filename, "r+" ) ) )
        syslog( LOG_ERR, "Cannot open %s for reading", filename );
    else {
        while ( fgets( s, 1024, f ) ) {
            if ( sscanf( s, "%d", &id[ n ] ) )
                ++n;
        }
        fclose( f );
        if ( n )
            truncate( filename, 0 );
    }
    syslog( LOG_INFO, "%s: marked bodies %d", group->name, n );
    if ( verbose > 1 )
	printf( "%s: marked bodies %d\n", group->name, n );
    for ( i = 0; i < n; ++i )
        if ( getbody( group, id[ i ] ) )
	    id[i] = 0;

    /* write back ids of all articles which could not be retrieved */
    if ( ! ( f = fopen( filename, "w" ) ) )
	syslog( LOG_ERR, "Cannot open %s for writing", filename );
    else {
	for ( i = 0; i < n; ++i )
	    if ( id[i] != 0 )
		fprintf( f, "%d\n", id[i] );
	fclose( f );
    }
    if ( verbose )
        printf( "Done.\n" );
}

/*
 * get newsgroup from a server. "server" is the last article that
 * was previously read from this group on that server
 */
int getgroup ( struct newsgroup * g, int server ) {
    static char *hd[10];
    char *hnames[10] = {"Path: ", "Message-ID: ", "From: ", "Newsgroups: ",
			"Subject: ", "Date: ", "References: ", "Lines: ", 
			"Xref: ", ""};
    int fetched, killed;
    int h;
    int n;
    int last;
    int window; /* last ARTICLE n command sent */
    char *l;
    FILE *f;
    const char *c;
    struct stat st;
    int outstanding = 0, i;
    static int * stufftoget;

    if ( !g )
	return server;

    if (g->first > g->last)
	g->last = g->first;
    (void) chdirgroup( g->name, TRUE );

    sprintf( lineout, "GROUP %s\r\n", g->name );
    putaline();

    l = getaline( nntpin );
    if ( !l )
	return server;

    if ( sscanf( l, "%3d", &n ) && n == 480 ) {
	if ( authenticate() ) {
	    sprintf( lineout, "GROUP %s\r\n", g->name );
	    putaline();
	} else
	    return server;
    }
    if ( n == 411 )	/* group not available on server */
	return 0;

    if (sscanf(l, "%3d %d %d %d ", &n, &h, &window, &last) < 4
	|| n != 211 )
	return server;

    if ( extraarticles ) {
	i = server - extraarticles;
	if ( i < window )
	    i = window;
	if ( verbose > 1 )
	    printf( "backing up from %d to %d\n",
		    server, i );
	server = i;
    }

    if ( server > last+1 ) {
	syslog( LOG_INFO, "%s: last seen article was %d, server now has %d-%d",
		g->name, server, window, last );
	if ( server > last+5 ) {
	    if ( verbose )
		printf( "switched upstream servers? %d > %d\n",
			server-1, last );
	    server = window; /* insane - recover thoroughly */
	} else {
	    if ( verbose )
		printf( "rampant spam cancel? %d > %d\n",
			server-1, last );
	    server = last - 5; /* a little bit too much */
	}
    }

    if ( initiallimit && server == 1 && 
	 last-server > initiallimit ) {
	if ( verbose > 1 )
	    printf( "skipping articles %d-%d inclusive (initial limit)\n", 
		    server, last-initiallimit );
	syslog( LOG_INFO, "skipping articles %d-%d inclusive (initial limit)",
		server, last-initiallimit );
	server = last-initiallimit+1;
    }

    if ( artlimit && last-server > artlimit ) {
	if ( verbose > 1 )
	    printf( "skipping articles %d-%d inclusive (article limit)\n",
		    server, last-artlimit-1 );
	syslog( LOG_INFO, "skipping articles %d-%d inclusive (article limit)",
		server, last-artlimit-1 );
	server = last-artlimit;
    }

    if ( ( noexpire == 0 ) && ( server <= last ) ) {
	sprintf( s, "%s/interesting.groups/%s", spooldir, g->name );
	if ( stat( s, &st ) < 0 )
	    return last;
	/* reading a newsgroup changes the ctime; if the newsgroup is
	   newly created, the mtime is changed as well */
	if ( ( ( st.st_mtime == st.st_ctime ) && 
	       ( now-st.st_ctime > (timeout_short * SECONDS_PER_DAY) ) ) ||
	     ( now-st.st_ctime > (timeout_long * SECONDS_PER_DAY) ) ) {
	    if ( verbose > 1 )
		printf("skipping %s from now on\n", g->name);
	    syslog( LOG_INFO, "skip %s (%d): %d, %d", g->name, now,
		    now-st.st_ctime, now-st.st_mtime );
	    unlink(s);
	    g->first = 1;
	    g->last = 1;
	    return server ;
       }
    }

    if ( window < server )
	window = server;
    if ( window < 1 )
	window = 1;
    server = window;

    if ( server > last ) {
	if ( verbose > 1 )
	    printf( "%s: no new articles\n", g->name );
	return server;
    }

    if ( verbose > 1 )
	printf( "%s: considering articles %d-%d\n", g->name, server, last );
    sprintf( lineout, "XHDR Message-ID %d-%d\r\n", server, last );
    putaline();
    if ( nntpreply() == 221 ) {
	stufftoget = realloc(stufftoget, sizeof(int)*(last + 1 - server) );
	if ( stufftoget )
	    memset( stufftoget, 0, last + 1 - server );
	else {
	    printf( "Not enough memory for XHDRs.\r\n" );
	    syslog( LOG_ERR, "not enough memory for XHDRs" );
	    return server;
	}
	debug = 0;
	while ( (l = getaline( nntpin )) && strcmp( l, "." ) ) {
	    int art;
	    char * t;
	    art = strtol(l, &t, 10);
	    if (t && isspace((unsigned char)*t) ) {
		while ( isspace((unsigned char)*t) )
		    t++;
		if ( art >= server && art <= last &&
		     stat( lookup( t ), &st ) != 0 ) {
		    stufftoget[outstanding] = art;
		    outstanding++;
		    if ( verbose > 2 )
			printf( "%s: will fetch %d (%s)\n",
			        g->name, art, t );
		}
	    }
	}
	debug = debugmode;
	if ( !l )
	    return server;
    }
    else {
	return server;
    }

    if ( outstanding == 0 ) {
	return server;
    }

    syslog( LOG_INFO, "%s: will fetch %d articles", g->name, outstanding );

    fetched = 0; killed = 0;

    i = 0;
    if ( maxbytes ) {
	sprintf( lineout, "XHDR Bytes %d-%d\r\n", server, last );
	putaline();
	if ( nntpreply() == 221 ) {
	    while ( (l = getaline( nntpin )) && strcmp( l, "." ) ) {
		int art, bytes = 0;
		char * t;
		art = strtol(l, &t, 10);
		if ( t )
		    bytes = strtol( t, NULL, 10 );
		if ( ( art == stufftoget[i] ) && ( bytes > maxbytes ) ) {
		    stufftoget[i] = 0;
		    syslog( LOG_INFO, "Killed article %d (%d > maxbytes)",
			    art, bytes );
		    if ( verbose > 2 )
			printf( "Killed article %d (%d > maxbytes).\n",
				art, bytes );
		    killed++;
		}
		i++;
	    }
	    if ( !l )		/* timeout */
		return server;
	}
    }
    i = 0;
    if ( maxage ) {
	sprintf( lineout, "XHDR Date %d-%d\r\n", server, last );
	putaline();
	if ( nntpreply() == 221 ) {
	    debug = 0;
	    while ( (l = getaline( nntpin )) && strcmp( l, "." ) ) {
		int art, aage = 0;
		char * t;
		art = strtol(l, &t, 10);
		if ( t )
		    aage = age( t );
		if ( ( art == stufftoget[i] ) && ( aage > maxage ) ) {
		    stufftoget[i] = 0;
		    syslog( LOG_INFO, "Killed article %d (%d > %d = maxage)",
			    art, aage, maxage );
		    if ( verbose > 2 )
			printf( "Killed article %d (%d > %d = maxage).\n",
				art, aage, maxage);
		    killed++; 
		    i++;
		}
	    }
	    debug = debugmode;
	}
    }
    

    /* now we have a list of articles in stufftoget[] */
    /* let's get the header and possibly bodies of these */
    while ( outstanding > 0 ) {
        i = outstanding-1;
	outstanding--;
	if ( !stufftoget[i] )
	    continue;

	if ( stufftoget[i] < server ) {
	    if ( verbose > 2 )
		printf( "%s: skipping %d - not available or too old\n",
			g->name, stufftoget[i] );
	    syslog( LOG_INFO, "%s: skipping %d - not available or too old",
	    	    g->name, stufftoget[i] );
	    continue;
	}

	sprintf(lineout, "HEAD %d\r\n", stufftoget[i]);
	putaline();
	l = getaline( nntpin );
	if ( !l )	/* timeout */
	    return server;
	if (sscanf(l, "%3d %d", &n, &h) < 2 || ( (n/10) != 22 ) ) {
	    if ( verbose > 2 )
		printf( "%s %d: reply %s (%d more up in the air)\n",
			g->name, stufftoget[i], l, outstanding );
	    syslog( LOG_INFO, "%s %d: reply %s (%d more up in the air)",
		    g->name, stufftoget[i], l, outstanding );
	    continue;
	}

	debug = 0;
	if ( verbose > 2 )
	    printf( "%s: receiving article %d (%d more up in the air)\n",
		    g->name, stufftoget[i], outstanding );

	for ( h=0; h<10; h++ ) {
	    if (hd[h])
		free(hd[h]);
	    hd[h] = strdup("");
	}
	c = NULL;
	n = 9;			/* "other" header */
	while ( (l = getaline( nntpin )) && strcmp( l, "." ) ) {
	    /* Quick && dirty fix */
	    tabstospaces(l);
	    /* allow for multiline headers */
	    if ( !isspace((unsigned char)l[0]) ) {
		n = 0;
		while ( strncasecmp( l, hnames[n], strlen(hnames[n]) ) )
		    n++;
		if ( n<9 && hd[n] && *(hd[n]) )
		    /* second occurance is "other header" */
		    n = 9;
	    } 
	    hd[n] = critrealloc( hd[n], strlen(hd[n])+strlen(l)+2,
		    "Fetching article header" );
	    if ( strlen(hd[n]) )
		strcat( hd[n], "\n" );
	    strcat( hd[n], l );
	    if ( verbose > 3 && hnames[n] && *hnames[n] )
		printf( "...saw header %s\n", hnames[n] );
	}
	debug = debugmode;
	if ( !l )		/* timeout */
	    return server;

	/* check headers */
	for ( h = 0; h < 4; h++ ) {
	    if ( !hd[h] || ! *(hd[h]) ) {
		if ( verbose )
		    printf( "Discarding article %d - no %s found\n",
			    stufftoget[i], hnames[h] );
		syslog( LOG_INFO,
			"Discarding article %d - no %s found",
			stufftoget[i], hnames[h] );
		killed++;
		continue;
	    }
	}

	if ( age( hd[5] ) > maxage ) {
	    if ( verbose > 2 )
		printf( "Discarding article %d - older than %d days\n",
			stufftoget[i], maxage );
	    syslog( LOG_INFO, "Discarding article %d %s - older than %d days",
		    stufftoget[i], hd[4], maxage );
	    killed++;
	    continue;
	}
	if ( minlines || maxlines ) {
	    char * t;
	    t = strchr( hd[7], ' ' );
	    if ( t ) {
		n = strtol( t, NULL, 10 );
		if ( minlines && n < minlines ) {
		    if ( verbose > 2 )
			printf("Discarding article %d - %d < minlines\n",
				stufftoget[i], n );
		    syslog( LOG_INFO,
			    "Discarding article %d %s -- %d < minlines",
			    stufftoget[i], hd[4], n );
		    killed++;
		    continue;
		}
		if ( maxlines && n > maxlines ) {
		    if ( verbose > 2 )
			printf("Discarding article %d - %d > maxlines\n",
				stufftoget[i], n );
		    syslog( LOG_INFO,
			    "Discarding article %d %s -- %d > maxlines",
			    stufftoget[i], hd[4], n );
		    killed++;
		    continue;
		}
	    }
	}
	if ( crosspostlimit ) {
	    char * t;
	    t = hd[3];
	    n = 1;	/* number of groups the article is posted to */
	    while ( ( t = strchr( t, ',' ) ) != NULL ) {
		t++ ;
		n++ ;
	    }
	    if ( crosspostlimit < n ) {
		if ( verbose > 2 )
		    printf( "Discarding article %d - posted to %d groups "
			    "(max. %d)\n", stufftoget[i], n, crosspostlimit );
		syslog( LOG_INFO,
			"Discarding article %d %s - posted to %d groups "
			"(max. %d)", stufftoget[i], hd[4], n, crosspostlimit );
		killed++;
		continue;
	    }
	}
	/* regexp pattern matching */
	if ( filterfile ) {
	    int error;
	    h = 0;
	    error = 0;
	    while ( ( h < 10 ) && !error ) {
		error = dofilter( hd[h] );
		if ( error ) {
		    killed++;
		    syslog( LOG_INFO, "Killed article %d %s", stufftoget[i],
				hd[4] );
		}
		h++;
	    }
	    if ( error )
		continue;
	}

	/* store articles */
	f = NULL;
	c = lookup( strchr( hd[1], '<' ) ) ;	/* also replaces / with @ */
	if ( !c ) {
	    syslog( LOG_ERR, "lookup of %s failed", hd[1] );
	    continue;
	}
	if ( !stat( c, &st ) ) {
	    syslog( LOG_INFO, "article %s already stored", c );
	    continue;	/* for some reasons, article is already there */
	} else if ( errno == ENOENT ) {
	    if ( (f = fopen( c, "w" )) == NULL) {
		/* check whether directory exists and possibly create new one */
		char * newc, *slp ;
		newc = strdup(c);
		slp = strrchr(newc, '/');
		if ( slp )
		    *slp = '\0';
		if ( stat( newc, &st ) ) {
		    if ( mkdir( newc, 0775 ) < 0 ) {
			syslog( LOG_ERR, "Cannot create directory %s: %m",
				newc );
			if ( verbose )
		            printf( "Cannot create directory %s\n", newc );
		    }
		}
		free( newc );
		f = fopen( c, "w" ); /* Try opening file again */
	    }
	    if ( !f ) {
		syslog( LOG_ERR, "unable to create article %s: %m", c );
		if ( verbose )
		    printf( "unable to create article %s\n", c );
		continue;
	    }
	} else {
	    syslog( LOG_ERR, "unable to store article %s: %m", c );
	    if ( verbose )
		printf( "unable to store article %s\n", c );
	    continue;
	}
	for ( h=0; h<10; h++ )
	    if ( h!=8 && hd[h] && *(hd[h]))
		 fprintf( f, "%s\n", hd[h] );
	h = 0;
	/* replace tabs and other odd signs with spaces */
	while ( h<8 ) {
	    char * p1;
	    char * p2;
	    p1 = p2 = hd[h];
	    while (p1 && *p1) {
		if ( isspace((unsigned char)*p1) ) {
		    *p2 = ' ';
		    do {
			p1++;
		    } while ( isspace((unsigned char)*p1) );
		} else {
		    *p2 = *p1++;
		}
		p2++;
	    }
	    *p2='\0'; 	
	    h++;
	}
	/* generate hardlinks; this procedure also increments g->last */
	store( c, f, st.st_size, *hd[3] ? hd[3]+strlen(hnames[3]) : "", 
				 *hd[4] ? hd[4]+strlen(hnames[4]) : "", 
				 *hd[2] ? hd[2]+strlen(hnames[2]) : "", 
				 *hd[5] ? hd[5]+strlen(hnames[5]) : "", 
				 *hd[1] ? hd[1]+strlen(hnames[1]) : "", 
				 *hd[6] ? hd[6]+strlen(hnames[6]) : "", 
				 *hd[7] ? hd[7]+strlen(hnames[7]) : "");

	if ( delaybody ) {
	    fclose( f );
	    fetched++;
	    continue;
	}

	sprintf(lineout, "BODY %d\r\n", stufftoget[i]);
	putaline();
	l = getaline(nntpin);
	if ( !l )		/* timeout */
	    return server;
	if ( sscanf( l, "%3d", &n ) != 1 || (n/10 != 22) ) {
	    syslog( LOG_INFO, "%d: reply %s", stufftoget[i], l );
	    continue;
	}
	debug = 0;
	fprintf( f, "\n" ); /* empty line between header and body */
	while ( (( l = getaline(nntpin) ) != NULL ) && strcmp( l, "." ) ) {
	    if ( *l && *l == '.' )
		fprintf( f, "%s\n", l+1 );
	    else
		fprintf( f, "%s\n", l );
	}
	debug = debugmode;
	fetched++;
	fclose( f );
	if ( !l ) {	/* article didn't terminate with a .: timeout */
	    unlink( c );
	    return server;
	}
	else if ( hd[9] ) {
#ifdef NOTYET
	    /* Handle Supersedes: header */
	    c = hd[9] ;
	    while ( c && *c && strncasecmp( c, "Supersedes:", 11 ) ) {
		c = strchr( hd[9], '\n' );
		if ( c )
		    c++;
	    }
	    /* Now c either points to the start of the Supersedes: line
	       or is NULL
	     */
	    if ( c ) {
		while ( ( c = strchr( c, '<' ) ) != NULL )
		    supersede( c ) ;
	    }
#endif
	}
    }

    syslog( LOG_INFO, "%s: %d articles fetched (to %ld), %d killed",
	    g->name, fetched, g->last, killed );
    if ( verbose > 1 )
	printf( "%s: %d articles fetched, %d killed\n",
		g->name, fetched, killed );
    return last+1;
}

/*
 * get active file from current_server
 */
static void nntpactive( time_t update ) {
    struct stat st;
    char * l, * p;
    struct stringlist * a, * b, *c = NULL;
    char timestr[64];
    int reply = 0, error;

    sprintf( s, "%s/leaf.node/%s", spooldir, current_server->name );
    if ( active && !forceactive && ( stat( s, &st ) == 0 ) ) {
	if ( verbose )
	    printf( "Getting new newsgroups from %s\n", current_server->name );
	strftime( timestr, 64, "%y%m%d %H%M00", gmtime( &update ) );
	sprintf( lineout, "NEWGROUPS %s GMT\r\n", timestr );
	putaline();
	if ( nntpreply() != 231 ) {
	    printf( "Reading new newsgroups failed.\n" );
	    syslog( LOG_ERR, "Reading new newsgroups failed" );
	    return;
	}
	a = NULL;
	while ( (l=getaline(nntpin)) && ( *l != '.' ) ) {
	    p = l;
	    while (!isspace((unsigned char)*p) )
		p++;
	    if (*p)
		*p = '\0';
	    b = (struct stringlist *)critmalloc( sizeof(struct stringlist)
		 + strlen(l), "allocating space for newsgroup name");
	    strcpy( b->string, l );
	    b->next = NULL;
	    if ( a == NULL )
		a = b;
	    else
		c->next = b;
	    c = b;
	}
	if ( !l )		/* timeout */
	    return;
	b = a;
	while ( b != NULL ) {
	    if ( current_server->descriptions ) {
		error = 0;
		sprintf( lineout, "XGTITLE %s\r\n", b->string );
		putaline();
		reply = nntpreply();
		if ( reply != 282 ) {
		    sprintf( lineout, "LIST NEWSGROUPS %s\r\n", b->string );
		    putaline();
		    reply = nntpreply();
		    if ( reply && ( reply != 215 ) )
			error = 1;
		}
		if ( !error ) {
		    l = getaline(nntpin);
		    if ( l && *l && *l != '.' ) {
			p = l;
			while ( !isspace((unsigned char)*p) )
			    p++;
			while ( isspace((unsigned char)*p) ) {
		            *p = '\0';
		    	    p++;
			}
			if ( reply == 215 || reply == 282 )
			    insertgroup( l, 0, 0, *p ? p : NULL , update );
			do {
			    l = getaline(nntpin);
			    error++;
			} while ( l && *l && *l != '.' );
			if ( error > 1 ) {
			    syslog( LOG_NOTICE, "warning: %s does not process "
				   "LIST NEWSGROUPS %s correctly: use nodesc\n",
				   current_server->name, b->string );
			    printf("warning: %s does not process LIST "
				   "NEWSGROUPS %s correctly: use nodesc\n",
				   current_server->name, b->string );
			}
		    }
		    else
			insertgroup( b->string, 0, 0, NULL, update );
		}
	    } /* if ( current_server->descriptions ) */
	    else
		insertgroup( b->string, 0, 0, NULL, update );
	    b = b->next;
	}
	b = a;
	while ( b != NULL ) {
	   c = b->next;
	   free(b);
	   b = c;
	}
    } else {
	if ( verbose )
	    printf( "Getting all newsgroups from %s\n", current_server->name );
	sprintf(lineout, "LIST ACTIVE\r\n" );
	putaline();
	if ( nntpreply() != 215 ) {
	    printf( "Reading all newsgroups failed.\n" );
	    syslog( LOG_ERR, "Reading all newsgroups failed" );
	    return;
	}
	debug = 0;
	while ( ( l = getaline( nntpin ) ) && ( strcmp( l, "." ) ) ) {
            p = l;
            while (!isspace((unsigned char)*p) )
                p++;
            while ( isspace((unsigned char)*p) ) {
                *p = '\0';
                p++;
            }
            insertgroup( l, 0, 0, NULL, update );
        }
	debug = debugmode;
	if ( !l )		/* timeout */
	    return;
	sprintf(lineout, "LIST NEWSGROUPS\r\n" );
	putaline();
	l = getaline( nntpin );
	/* correct reply starts with "215". However, INN 1.5.1 is broken
	   and immediately returns the list of groups */
	if ( ( l == NULL ) || ( strlen(l) < 4 ) ) {
	    printf( "Reading newsgroups descriptions failed.\n" );
	    syslog( LOG_ERR, "Reading newsgroups descriptions failed" );
	    return;
	}
	debug = 0;
	if ( l[3] == ' ' ) {
	    /* assume that a correct reply has the format "215 blablabla" */
	    reply = strtol( l, NULL, 10 );
	    if ( reply != 215 ) {
		printf( "Reading newsgroups descriptions failed.\n" );
		syslog( LOG_ERR, "Reading newsgroups descriptions failed: %d",
			reply );
		return;
	    }
	    l = getaline( nntpin );	/* get first description */
	}
	while ( l && ( strcmp( l, "." ) ) ) {
	    p = l;
	    while (!isspace((unsigned char)*p) )
		p++;
	    while ( isspace((unsigned char)*p) ) {
                *p = '\0';
                p++;
            }
	    insertgroup( l, 0, 0, *p ? p : NULL, update );
	    l = getaline(nntpin);
	}
	debug = debugmode;
	if ( !l )
	    return;		/* timeout */
    }
}


/*
 * post all spooled articles
 *
 * if all postings succeed, returns 1
 * if there are no postings to post, returns 1
 * if a posting is strange for some reason, returns 0
 */
int postarticles( void ) {
    struct stringlist *a, *b;
    struct stat st;
    char * line;
    DIR * d;
    struct dirent * de;
    FILE * f;
    int r, haveid, n;

    a = posted;
    n = 0;

    if ( chdir( spooldir ) || chdir ( "out.going" ) ) {
	syslog( LOG_ERR, "Unable to cd to outgoing directory: %m" );
	printf( "Unable to cd to %s/out.going\n", spooldir );
	return 1;
    }

    d = opendir( "." );
    if ( !d ) {
	syslog( LOG_ERR, "Unable to opendir out.going: %m" );
	printf( "Unable to opendir %s/out.going\n", spooldir );
	return 1;
    }

    while ( (de=readdir( d )) != NULL ) {
	haveid = 0;
	if(!strcmp(".", de->d_name) || !strcmp("..", de->d_name)) {
	     continue;
	}
	if ( ( stat( de->d_name, &st ) != 0 ) ) {
	  int err;

	  err = errno;
	  syslog( LOG_ERR, "Cannot stat %s: %s", de->d_name, strerror(err) );
	  printf( "Unable to stat %s: %s\n", de->d_name, strerror(err) );
	} else if (S_ISREG( st.st_mode ) &&
	     ( ( f = fopen( de->d_name, "r" ) ) != NULL ) &&
	     isgrouponserver( fgetheader( f, "Newsgroups:" ) ) &&
	     ( ( haveid = 
		 ismsgidonserver( fgetheader( f, "Message-ID:" ) ) ) != 1 ) ) {
	    if ( verbose > 2 )
		printf( "Posting %s...\n", de->d_name );
	    syslog( LOG_INFO, "Posting %s...",de->d_name );
	    sprintf( lineout, "POST\r\n" );
	    putaline();
	    r = nntpreply();
	    if ( r != 340 ) {
		if ( verbose )
		    printf( "Posting %s failed: %03d reply to POST\n",
			     de->d_name, r );
		sprintf( s, "%s/failed.postings/%s", spooldir, de->d_name );
		syslog( LOG_ERR, "unable to post (%d), moving to %s", r, s );
		if ( link( de->d_name, s ) )
		    syslog( LOG_ERR, "unable to move failed posting to %s: %m",
				      s );
	    } else {
	        debug = 0;
		while ( ( line = getaline( f ) ) != NULL ) {
		    /* can't use putaline() here because
		       line length of lineout is restricted */
		    if ( strlen(line) && line[0] == '.' )
			fprintf( nntpout, ".%s\r\n", line );
		    else
			fprintf( nntpout, "%s\r\n", line );
		};
		fflush( nntpout );
		debug = debugmode;
		sprintf( lineout, ".\r\n" );
		putaline();
		line = getaline( nntpin );
		if ( !line )		/* timeout: posting failed */
		    return 0;
		if ( strncmp( line, "240", 3 ) == 0 ) {
		    if ( verbose > 2 )
			printf( " - OK\n" );
		    n++;
		    b = (struct stringlist *)critmalloc
		        ( sizeof(struct stringlist) + strlen(de->d_name),
			"allocating space for posted article");
		    strcpy( b->string, de->d_name );
		    b->next = NULL;
		    if ( !posted )
			posted = b;
		    else
			a->next = b;
		    a = b;
		} else {
		    if ( verbose )
			printf( "Article %s rejected: %s\n",
				de->d_name, line );
		    sprintf( s, "%s/failed.postings/%s", 
			     spooldir, de->d_name );
		    syslog( LOG_ERR, 
			    "Article %s rejected (%s), moving to %s",
			    de->d_name, line, s );
		    if ( link( de->d_name, s ) )
			syslog( LOG_ERR,
				"unable to link failed posting to %s: %m", s );
		} 
	    }
	    fclose( f );
	    haveid = 0;
	} else if ( haveid ) {
	    syslog( LOG_INFO, "Message-ID of %s already in use upstream -- "
			"article discarded\n", de->d_name );
	    if ( verbose > 2 )
		printf( "%s already available upstream\n", de->d_name );
	    unlink( de->d_name ); 
	}
    }
    closedir( d );
    if ( verbose && n )
	syslog( LOG_INFO, "%s: %d articles posted", current_server->name, n );
    return 1;
}

static void processupstream( const char * server ) {
    FILE * f;
    DIR * d;
    struct dirent * de;
    struct newsgroup * g;
    int havefile;
    int newserver = 0;
    char * l;
    char * oldfile ;

    struct stringlist * ngs = NULL, * a, *b = NULL;

    /* read server info */
    sprintf( s, "%s/leaf.node/%s", spooldir, server );
    oldfile = strdup( s );
    havefile = 0;
    if ( ( f = fopen( s, "r" ) ) != NULL ) {
	/* a sorted array or a tree would be better than a list */
	ngs = NULL;
	debug = 0;
        if ( verbose > 1 )
            printf( "Read server info from %s\n", s );
	syslog( LOG_INFO, "Read server info from %s", s );
	while ( ( ( l = getaline( f ) ) != NULL ) && ( strlen(l) ) ) {
	    a = (struct stringlist *)critmalloc( sizeof(struct stringlist)
		 + strlen(l), "Reading server info" );
	    strcpy( a->string, l );
	    a->next = NULL;
	    if ( ngs == NULL )
		ngs = a;
	    else
		b->next = a;
	    b = a;
	}
	havefile = 1;
	debug = debugmode;
	fclose( f );
    }

    sprintf( s, "%s/interesting.groups", 
	     spooldir );
    d = opendir( s );
    if ( !d ) {
	syslog( LOG_ERR, "opendir %s: %m", s );
	return;
    }

    sprintf( s, "%s/leaf.node/%s~", spooldir, server );
    f = fopen( s, "w" );
    if ( f == NULL ) {
	syslog( LOG_ERR, "Could not open %s for writing.", s );
        printf( "Could not open %s for writing.\n", s );
    }
    while ( (de=readdir(d)) ) {
	if ( isalnum((unsigned char)*(de->d_name)) ) {
	    g = findgroup( de->d_name );
	    if ( g != NULL ) {
		sprintf( s, "%s ", g->name );
		l = havefile ? findinlist( ngs, s ) : NULL;
		if ( delaybody && ( headerbody == 2 ) )
		    ;	/* get only bodies, so skip this */
		else if ( l && *l ) {
		    l = strchr( l, ' ' );
		    newserver = getgroup( g, strtol( l, NULL, 10 ) );
		}
		else
		    newserver = getgroup( g, 1 );
		if ( delaybody && ( headerbody != 1 ) )
		    getmarked( g );
		if ( f != NULL )
                    fprintf( f, "%s %d\n", g->name, newserver );
            } else {
		printf( "%s not found in groupinfo file\n", de->d_name );
		syslog( LOG_INFO, "%s not found in groupinfo file",
			de->d_name );
	    }
	}
    }
    closedir( d );
    if ( f != NULL ) {
        fclose(f);
	sprintf( s, "%s/leaf.node/%s~", spooldir, server );
	rename( s, oldfile );
    }

    free( oldfile );
    freelist(ngs);
}

/*
 * checks whether all newsgroups have to be retrieved anew
 * returns 0 if yes, time of last update if not
 */
static int checkactive( void ) {
    struct stat st;

    sprintf( s, "%s/active.read", spooldir );
    if ( stat( s, &st ) )
	return 0;
    if ( ( now - st.st_mtime ) < ( timeout_active * SECONDS_PER_DAY ) ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    "Last LIST ACTIVE done %d seconds ago: NEWGROUPS\n",
		    now - st.st_mtime );
	return st.st_atime ;
    } else {
	if ( debugmode )
	    syslog( LOG_DEBUG, "Last LIST ACTIVE done %d seconds ago: LIST\n",
		    now - st.st_mtime );
	return 0;
    }
}

static void updateactive( void ) {
    struct stat st;
    struct utimbuf buf;
    FILE * f;

    sprintf( s, "%s/active.read", spooldir );
    if ( stat( s, &st ) ) {
	/* active.read doesn't exist */
	if ( ( f = fopen( s, "w" ) ) != NULL )
	    fclose( f );
    } else {
	buf.actime = ( now < st.st_atime ) ? st.st_atime : now;
			/* now < update may happen through HW failures */
	if ( (now - st.st_mtime) < (timeout_active*SECONDS_PER_DAY) )
	    buf.modtime = st.st_mtime;
	else
	    buf.modtime = now;
	if ( utime( s, &buf ) ) {
	    if ( ( f = fopen( s, "w" ) ) != NULL )
		fclose( f );
	    else {
		printf("Unable to access %s/active.read\n", spooldir );
		syslog( LOG_NOTICE,
			"Unable to access %s/active.read", spooldir );
	    }
	}
    }
}

void fixxover( void ) {
    DIR * d;
    struct dirent * de;

    sprintf( s, "%s/interesting.groups",
             spooldir );
    d = opendir( s );
    if ( !d ) {
        syslog( LOG_ERR, "opendir %s: %m", s );
        return;
    }

    while ( (de=readdir(d)) ) {
        if ( isalnum((unsigned char)*(de->d_name) ) && findgroup( de->d_name ) ) {
            if ( chdirgroup( de->d_name, FALSE ) )
        	getxover();
        }
    }
    closedir( d );
}

int main( int argc, char ** argv ) {
    int option, reply;
    volatile time_t lastrun;

    verbose = 0;
    usesupplement = 0;
    postonly = 0;

    if ( !initvars( argv[0] ) )
	exit( 1 );

    openlog( "fetchnews", LOG_PID|LOG_CONS, LOG_NEWS );

    while ( (option=getopt( argc, argv, "Pflnvx:" )) != -1 ) {
	if ( option == 'v' ) {
	    verbose++;
	} else if ( option == 'x' ) {
	    char *nptr, *endptr;
	    nptr = optarg;
	    endptr = NULL;
	    extraarticles = strtol( nptr, &endptr, 0 );
	    if ( !nptr || !*nptr || !endptr || *endptr || !extraarticles ) {
	        usage();
		exit( 1 );
	    }
	} else if ( option == 'l' ) {
	    usesupplement = -1; /* don't use supplementary servers */
        } else if ( option == 'n' ) {
            noexpire = 1;
	} else if ( option == 'f' ) {
	    forceactive = 1;
	} else if ( option == 'P' ) {
	    postonly = 1;
	} else {
	    usage();
	    exit( 1 );
	}
    }

    if ( !postonly && lockfile_exists() )
        exit( 1 );

    syslog( LOG_DEBUG, "%s: verbosity level is %d", version, verbose );
    if ( verbose ) {
	printf( "%s: verbosity level is %d\n", version, verbose );
	if ( verbose > 1 && noexpire ) {
	    printf("Don't automatically unsubscribe unread newsgroups.\n");
        }
    }

    whoami();

    now = time(NULL);

    umask(2);

    if ( !readconfig() ) {
	printf("Reading configuration failed, exiting "
	       "(see syslog for more information).\n");
	if ( !postonly )
	    unlink( lockfile );
	exit( 2 );
    }

    if ( !postonly ) {
	if ( forceactive )
	    fakeactive();
	else
	    readactive();
	readfilter( filterfile );
    }

    lastrun = 0;
    if ( !forceactive ) {
	lastrun = checkactive();
	if ( !lastrun )
	    forceactive = 1;
    }

    if ( signal( SIGINT, sig_int ) == SIG_ERR )
	fprintf( stderr, "Can't catch SIGINT.\n" );
    else if ( setjmp( jmpbuffer ) != 0 ) {
	servers = NULL;		/* in this case, jump the while ... loop */
    }

    while ( servers ) {
	current_server = servers;
	if ( verbose )
	    printf( "Trying to connect to %s ... ", servers->name);
	fflush( stdout );
	reply = nntpconnect( current_server );
	if ( reply ) {
	    if ( verbose )
	        printf("connected.\n");
	    if ( current_server->username )
		( void ) authenticate();
	    sprintf( lineout, "MODE READER\r\n" );
	    putaline();
	    reply = nntpreply();
	    if ( reply == 498 )
		syslog( LOG_INFO, "Connection refused by %s",
			current_server->name );
	    else {
		if ( reply == 200 )
		    ( void ) postarticles();
		if ( !postonly ) {
		    nntpactive( lastrun );
				/* get list of newsgroups or new newsgroups */
		    processupstream( current_server->name );
		}
	    }
	    sprintf( lineout, "QUIT\r\n"); /* say it, then just exit :) */
	    putaline();
	    nntpdisconnect();
	    if ( verbose )
	        printf( "Disconnected from %s.\n", current_server->name );
	}
	else {
	    if ( verbose )
	        printf("failed.\n");
	}
	if ( usesupplement < 0 )
	    servers = NULL;	/* we don't clean up the list */
	else
	    servers = servers->next;
    }

    fflush( stdout );	/* to avoid double logging of stuff */

    switch ( fork() ) {
	case -1: {
	    syslog( LOG_NOTICE, "fork: %m" );
	    break;
	}
	case  0: {
	    setsid();
	    if ( debugmode )
		syslog( LOG_DEBUG, "Process forked." );
	    if ( !postonly ) {
		updateactive();
		writeactive( forceactive );
		fixxover();
		unlink( lockfile );
	    }
	    delposted();
	}
	default: break;
    }

    exit(0);
}
