/*
** oidentd - ojnk ident daemon
** Copyright (C) 1998,1999,2000 Ryan McCabe <odin@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
**
** $Id: linux.c,v 1.11 2000/01/16 17:32:52 odin Exp $
*/

#define _GNU_SOURCE
#include <config.h>
#include <stdlib.h>
#include <signal.h>
#include <oidentd.h>
#include <setjmp.h>

#ifdef MASQ_SUPPORT
extern u_long proxy;
extern int fwdport;
#endif
extern u_char *charset;
extern u_char *ret_os;
extern u_int flags;


static sigjmp_buf timebuf;
static int find_entry(u_long host, u_char *user, size_t ursz, u_char *os,
					size_t osz);
static int fwd_request(int sock, int rlport, int lport, int fport,
				const struct in_addr *remote);
static void fwdalrm(int sig);


/*
** Determine the owner of the connection.
*/

int get_user(int lport, int fport, const struct in_addr *laddr, const struct in_addr *faddr) {
	int uid, portl, portf;
	FILE *fp = NULL;
	u_char buf[1024];
	struct in_addr remote, local;

	fp = fopen(CFILE, "r");

	if (fp == NULL) {
		if (flags & DEBUG)
			syslog(DPRI, "fopen: %s: %s", CFILE, strerror(errno));

		return (-1);
	}

	fgets(buf, sizeof(buf) - 1, fp);

	/*
	** The line should never be longer than 1024 chars, so fgets should be OK.
	*/

	while (fgets(buf, sizeof(buf) - 1, fp)) {
		if (sscanf(buf, "%*d: %lX:%x %lX:%x %*x %*X:%*X %*x:%*X %*x %d %*d %*d",
			(long *) &local, &portl, (long *) &remote, &portf, &uid) == 5) {
#ifdef MASQ_SUPPORT
			if (flags & PROXY) {
				if (faddr->s_addr == proxy && remote.s_addr != proxy
					&& lport == portl && fport == portf) {
					fclose(fp);
					return (uid);
				} else if (local.s_addr == laddr->s_addr && portl == lport
					&& remote.s_addr == faddr->s_addr && portf == fport) {
						fclose(fp);

						return (uid);
				}
			} else if (local.s_addr == laddr->s_addr && portl == lport
				&& remote.s_addr == faddr->s_addr && portf == fport) {
				fclose(fp);

				return (uid);
			}
#else
			if (local.s_addr == laddr->s_addr && portl == lport
				&& remote.s_addr == faddr->s_addr && portf == fport) {
					fclose(fp);

					return (uid);
			}
#endif
		}
	}

	fclose(fp);
	return (-1);
}

#ifdef MASQ_SUPPORT
/*
** Forward an ident request to another machine, return the response to the
** client that has connected to us and requested it.
*/

static int fwd_request(int sock, int rlport, int lport, int fport,
				const struct in_addr *remote)
{
	extern int fsock;
	u_char buf[128], user[128];
	struct sockaddr_in fr_sin;

	fsock = socket(PF_INET, SOCK_STREAM, 0);

	if (fsock == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: socket: %s", strerror(errno));

		return (-1);
	}

	memset(&fr_sin, 0, sizeof(fr_sin));
	fr_sin.sin_family = AF_INET;
	fr_sin.sin_port = htons(fwdport);
	fr_sin.sin_addr.s_addr = remote->s_addr;

	if (sigsetjmp(timebuf, 1) != 0)
		return (-1);

	signal(SIGALRM, fwdalrm);

	/*
	** Five seconds should be plenty, seeing as we're forwarding to a machine
	** on a local network.
	*/
	alarm(5);
	
	if (connect(fsock, (struct sockaddr *) &fr_sin, sizeof(fr_sin)) == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: connect: %s", strerror(errno));

		close(fsock);
		return (-1);
	}

	if (dprintf(fsock, "%d , %d\r\n", lport, fport) < 1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: write: %s", strerror(errno));

		close(fsock);
		return (-1);
	}

	if (sockread(fsock, buf, sizeof(buf)) < 1) {
		close(fsock);
		return (-1);
	}

	close(fsock);

	if (sscanf(buf, "%*d , %*d : USERID :%*[^:]:%127s", user) != 1) {
		if (flags & DEBUG)
			syslog(DPRI, "[%s] Bad response: %s", inet_ntoa(*remote), buf);

		return (-1);
	}

	dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n",
		rlport, fport, OS(ret_os), (charset != NULL ? " , " : ""),
		(charset != NULL ? charset : (u_char *) ""), user);

	if (!(flags & QUIET)) {
		syslog(PRIORITY,
			"[%s] (Forwarded) Successful lookup: %d , %d : USERID : %s : %s",
			inet_ntoa(*remote), rlport, fport, OS(ret_os), user);
	}

	free(user);
	return (0);
}

/*
** Handle the timeout of a forward request.
*/

static void fwdalrm(int sig) {
	extern int fsock;

	if (flags & MASQ) {
		if (!(flags & QUIET))
		    syslog(PRIORITY, "Forward timed out.  Falling back to oidentd.users");
    	close(fsock);
	} else {
		if (!(flags & QUIET))
			syslog(PRIORITY, "Forward timed out.");

		exit(0);
	}

	siglongjmp(timebuf, sig);
}

/*
** Handle a request to a host that's IP masquerading through us.
*/

int masq(int sock, int lport, int fport, const struct in_addr *remote) {
	FILE *fp = NULL;
	u_char buf[1024], proto[12];
	u_char os[24], user[16];
	int mport, lportm, fportm;
	int netfilter = 0, l1, l2, l3, l4, r1, r2, r3, r4;
	struct in_addr localm, remotem;

	fp = fopen(MASQFILE, "r");

	if (fp == NULL) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: %s: %s", MASQFILE, strerror(errno));

		fp = fopen(CONNTRACK, "r");

		if (fp == NULL) {
			if (flags & DEBUG)
				syslog(DPRI, "Error: %s: %s", CONNTRACK, strerror(errno));

			return (-1);
		}

		netfilter = 1;
	}

	/* chomp */
	fgets(buf, sizeof(buf) - 1, fp);
	
	while (fgets(buf, sizeof(buf) - 1, fp)) {
		if (netfilter == 0) {
			if (sscanf(buf, "%11s %lX:%X %lX:%X %X %*X %*d %*d %*u", proto,
							(long *) &localm, &lportm, (long *) &remotem,
							&fportm, &mport) != 6)
				continue;
		} else {
			if (sscanf(buf, "%s %*d %*d ESTABLISHED src=%d.%d.%d.%d dst=%d.%d.%d.%d sport=%d dport=%d %*s %*s %*s dport=%d", proto, &l1, &l2, &l3, &l4,
					&r1, &r2, &r3, &r4, &lportm, &fportm, &mport) != 12)
                continue;

			localm.s_addr = l1 << 24 | l2 << 16 | l3 << 8 | l4;
			remotem.s_addr = r1 << 24 | r2 << 16 | r3 << 8 | r4;
		}

		if (!strcasecmp(proto, "TCP") && mport == lport && fportm == fport
			&& remotem.s_addr == ntohl(remote->s_addr)) {
			if (flags & FWD) {
				struct in_addr fhost;

				fhost.s_addr = htonl(localm.s_addr);

				if (!fwd_request(sock, lport, lportm, fportm, &fhost))
					return (0);
				else {
					if (flags & DEBUG) {
						syslog(DPRI, "Forward to %s (%d %d) failed.",
							inet_ntoa(fhost), lportm, fportm);
					}
				}
			}
			if (!find_entry(htonl(localm.s_addr), user, sizeof(user) - 1,
				os, sizeof(os) - 1)) {
					dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n",
						lport, fport, OS(os),
						(charset != NULL ? " , " : ""),
						(charset != NULL ? charset : (u_char *) ""), user);

					if (!(flags & QUIET)) {
						syslog(PRIORITY,
							"[%s] (Masqueraded) Successful lookup: %d , %d : USERID : %s : %s",
							inet_ntoa(*remote), lport, fport, OS(os), user);
					}

					fclose(fp);
					return (0);
			}
		}
	}
	
	fclose(fp);
	return (-1);
}

/*
** Parse the masquerading map.
*/

static int find_entry(u_long host, u_char *user, size_t ursz, u_char *os,
					size_t osz)
{
	int c = 0;
	u_int i = 0;
	u_long addr = 0, mask = 0, mask2 = 0;
	FILE *fp = NULL;
	u_char buf[1024], *temp;
	
	fp = fopen(MAP, "r");

	if (fp == NULL) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: fopen: %s: %s", MAP, strerror(errno));

		return (-1);
	}

	for (;; i = 0) {
		while (i < sizeof(buf) - 1) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n') {
				i = 0;
				continue;
			}
			if (c == '#' && i == 0)
				goto newline;
			if (isblank(c)) {
				if (i > 0)
					break;
				continue;
			}
			buf[i++] = c;
		}

		buf[i] = '\0';

		if (strlen(buf) == sizeof(buf)) {
			if (flags & DEBUG)
				syslog(DPRI, "Long line in %s.\n", MAP);
			while (!isspace(getc(fp)))
				;
		}
		
		temp = strchr(buf, '/');
		if (temp != NULL) {
			*temp++ = '\0';
			if (strchr(temp, '.') || !isdigit(*temp)) {
				if (get_addr(temp, &mask2) == -1) {
					if (flags & DEBUG)
						syslog(DPRI,
							"Error: %s: Invalid mask: %s/%s", MAP, buf, temp);
					goto newline;
				}
			} else {
				mask = strtoul(temp, NULL, 10);
				if (mask < 1 || mask > 32) {
					if (flags & DEBUG)
						syslog(DPRI,
							"Error: %s: Invalid mask: %s/%s", MAP, buf, temp);
					goto newline;
				}
			}
		}

		if (get_addr(buf, &addr) == -1)
			goto newline;

		if (mask)
			mask2 = htonl(~((1 << (32 - mask)) - 1));

		if (mask2) {
			addr &= mask2;
			host &= mask2;
		}

		if (host != addr)
			goto newline;

		for (i = 0 ; i < ursz ;) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n') {
				i = 0;
				goto newline;
			}
			if (isblank(c)) {
				if (i > 0)
					break;
				continue;
			}
			user[i++] = c;
		}

		user[i] = '\0';

		if (strlen(user) == ursz) {
			if (flags & DEBUG)
				syslog(DPRI, "Long line in %s.\n", MAP);
			while (!isspace(getc(fp)))
				;
		}

		for (i = 0 ; i < osz ;) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n') {
				if (i)
					break;
				goto newline;
			}
			if (isblank(c)) {
				if (i)
					break;
				continue;
			}
			os[i++] = c;
		}

		os[i] = '\0';
		return (0);

newline:
		for (;;) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n')
				break;
		}
	}

eof:
	fclose(fp);
	return (-1);
}

#endif /* masq */
