/*
 *  nwfsinfo.c
 *
 *  Print the info strings of a server, maybe sometime more.
 *
 *  Copyright (C) 1996 by Volker Lendecke
 *
 *  Non-IPX addresses listing by Petr Vandrovec, 1999.
 *
 */

#include "config.h"

#include <stdio.h>
#include <unistd.h>
#include <ncp/nwcalls.h>

#include <libintl.h>
#define _(X) gettext(X)

static char *progname;

static void
usage(void)
{
	fprintf(stderr, _("usage: %s [pattern]\n"), progname);
}

static void
str_trim_right(char *s, char c)
{
	int len = strlen(s) - 1;

	while ((len > 0) && (s[len] == c))
	{
		s[len] = '\0';
		len -= 1;
	}
}

static void
help(void)
{
	printf(_("\n"
		 "usage: %s [options]\n"), progname);
	printf(_("\n"
	       "-h             Print this help text\n"
	       "-S server      Server name to be used\n"
	       "-a             Print Station's addr\n"
	       "\n"));
}

int
main(int argc, char **argv)
{
	struct ncp_conn *conn;
	int opt;
	long err;
	struct ncp_file_server_info info;
	struct ncp_bindery_object user;
	time_t login_time;
	int i;
	int print_addr = 0;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	
	progname = argv[0];

	if ((conn = ncp_initialize(&argc, argv, 1, &err)) == NULL)
	{
		com_err(argv[0], err, _("when initializing"));
		goto finished;
	}
	while ((opt = getopt(argc, argv, "h?a")) != EOF)
	{
		switch (opt)
		{
		case 'h':
		case '?':
			help();
			goto finished;
		case 'a':
			print_addr = 1;
			break;
		default:
			usage();
			goto finished;
		}
	}

	if (ncp_get_file_server_information(conn, &info) != 0)
	{
		perror(_("Could not get server information"));
		ncp_close(conn);
		return 1;
	}
	
	if (isatty(1))
	{
		if (print_addr == 0)
		{
			printf(_("\n%-6s%-21s%-12s\n"
			       "---------------------------------------------"
			       "------\n"),
			       _("Conn"),
			       _("User name"),
			       _("Login time"));
		} else
		{
			printf(_("\n%-6s%-21s%-27s%-12s\n"
			       "---------------------------------------------"
			       "---------------------------------\n"),
			       _("Conn"),
			       _("User name"),
			       _("Station Address"),
			       _("Login time"));
		}
	}
	for (i = 1; i <= info.MaximumServiceConnections; i++)
	{
		char name[49];
		name[48] = '\0';
		if (ncp_get_stations_logged_info(conn, i, &user,
						 &login_time) != 0)
		{
			continue;
		}
		memcpy(name, user.object_name, 48);
		str_trim_right(name, ' ');
		printf(_("%4d: %-20s "), i, name);

		if (print_addr != 0)
		{
			union ncp_sockaddr addr;
			u_int8_t conn_type;

			memset(&addr, 0, sizeof(addr));
			if (ncp_get_internet_address(conn, i, &addr.any, &conn_type)) {
				printf("XXXXXXXX:YYZZYYXXTTXX:QQQQ");
			} else switch (addr.any.sa_family) {
#ifdef NCP_IPX_SUPPORT
				case AF_IPX:	ipx_print_saddr(&addr.ipx);
						break;
#endif
#ifdef NCP_IN_SUPPORT
				case AF_INET:	{
							char q[30];
							u_int32_t sa = ntohl(addr.inet.sin_addr.s_addr);
							sprintf(q, "%d.%d.%d.%d/%d", (u_int8_t)(sa >> 24), (u_int8_t)(sa >> 16),
										     (u_int8_t)(sa >> 8), (u_int8_t)(sa), 
										     ntohs(addr.inet.sin_port));
     							printf("%-26s", q);
     						}
     						break;
#endif
				default:	printf("%-26s", _("Unknown format"));
						break;
			}
			printf(" ");
		}
		{
			char text_login_time[200];
			struct tm* tm;
			
			tm = localtime(&login_time);
			strftime(text_login_time, sizeof(text_login_time), "%c", tm);
			printf("%s\n", text_login_time);
		}
	}

finished:
	ncp_close(conn);
	return 0;
}
