/*
 log.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *logs;

gboolean log_file_open(LOG_REC *log)
{
    gchar *str;

    g_return_val_if_fail(log != NULL, FALSE);

    /* Append/create log file */
    str = convert_home(log->fname);
    log->handle = open(str, O_WRONLY | O_APPEND | O_CREAT, LOG_FILE_CREATE_MODE);
    g_free(str);

    if (log->handle == -1) return FALSE;
    lseek(log->handle, 0, SEEK_END);

    log->last = time(NULL);
    str = g_strdup_printf("--- Log opened at %s", asctime(localtime(&log->last)));
    write(log->handle, str, strlen(str));
    g_free(str);

    signal_emit("log opened", 1, log);
    return TRUE;
}

void log_file_close(LOG_REC *log)
{
    gchar *str;
    time_t t;

    g_return_if_fail(log != NULL);

    signal_emit("log closed", 1, log);

    t = time(NULL);
    str = g_strdup_printf("--- Log closed at %s", asctime(localtime(&t)));
    write(log->handle, str, strlen(str));
    g_free(str);

    close(log->handle);
    log->handle = -1;
}

static void write_log(LOG_REC *log, gchar *str)
{
    gchar *logline;
    struct tm *tm;
    time_t t;
    gint day;

    g_return_if_fail(log != NULL);
    g_return_if_fail(str != NULL);
    g_return_if_fail(log->handle != -1);

    t = time(NULL);

    tm = localtime(&t);
    day = tm->tm_mday;
    logline = g_strdup_printf("%02d:%02d %s\n", tm->tm_hour, tm->tm_min, str);

    tm = localtime(&log->last);
    if (tm->tm_mday != day)
    {
        /* day changed */
        gchar *daystr;

        daystr = g_strdup_printf("--- %s", asctime(localtime(&t)));
        write(log->handle, daystr, strlen(daystr));
        g_free(daystr);
    }

    log->last = t;
    write(log->handle, logline, strlen(logline));
    g_free(logline);

    signal_emit("log written", 2, log, str);
}

static void log_file_write(SERVER_REC *server, gchar *channel, gint level, gchar *str)
{
    gboolean found;
    GList *tmp, *fallbacks;

    g_return_if_fail(str != NULL);
    if (channel == NULL)
        channel = channel_find_closest(server, channel, level)->name;

    fallbacks = NULL; found = FALSE;
    for (tmp = logs; tmp != NULL; tmp = tmp->next)
    {
        LOG_REC *rec = tmp->data;
        GList *sub;

        if (rec->handle == -1) continue; /* log not opened yet */

        if (level & rec->level)
            fallbacks = g_list_append(fallbacks, rec);

        for (sub = rec->items; sub != NULL; sub = sub->next)
        {
            LOG_ITEM_REC *subrec = sub->data;

            if (g_strcasecmp(subrec->name, channel) == 0 && level & subrec->level)
            {
                found = TRUE;
                write_log(rec, str);
            }
        }
    }

    if (!found)
    {
        gchar *tmp = str;

        /* not found from subrecords so write it to all main logs.. */
        if (level & MSGLEVEL_PUBLIC)
        {
            /* public message in some channel */
            tmp = g_strconcat(channel, ": ", str, NULL);
        }
        g_list_foreach(fallbacks, (GFunc) write_log, tmp);
        if (level & MSGLEVEL_PUBLIC) g_free(tmp);
        g_list_free(fallbacks);
    }
}

static gboolean sig_log(SERVER_REC *server, gchar *channel, gpointer level, gchar *str)
{
    gint loglevel;

    g_return_val_if_fail(str != NULL, FALSE);

    loglevel = GPOINTER_TO_INT(level);
    if (loglevel == MSGLEVEL_NEVER || logs == NULL) return TRUE;

    /* Check if line should be saved in logs */
    log_file_write(server, channel, loglevel, str);

    return TRUE;
}

LOG_REC *log_file_find(gchar *fname)
{
    GList *tmp;

    for (tmp = g_list_first(logs); tmp != NULL; tmp = tmp->next)
    {
        LOG_REC *rec = tmp->data;

        if (strcmp(rec->fname, fname) == 0)
            break;
    }

    return tmp == NULL ? NULL : tmp->data;
}

static LOG_ITEM_REC *log_item_find(LOG_REC *log, gchar *item)
{
    GList *tmp;

    for (tmp = g_list_first(log->items); tmp != NULL; tmp = tmp->next)
    {
        LOG_ITEM_REC *rec = tmp->data;

        if (g_strcasecmp(rec->name, item) == 0)
            return rec;
    }
    return NULL;
}

LOG_REC *log_create(gchar *fname, gchar *data)
{
    LOG_REC *rec;
    gchar *dup, *ptr;
    gint *level;

    g_return_val_if_fail(fname != NULL, NULL);
    g_return_val_if_fail(data != NULL, NULL);

    rec = log_file_find(fname);
    if (rec == NULL)
    {
        rec = g_new0(LOG_REC, 1);
        logs = g_list_append(logs, rec);

        rec->fname = g_strdup(fname);
        rec->handle = -1;
        rec->level = MSGLEVEL_ALL;
    }
 
    dup = data = g_strdup(data);
    level = &rec->level;
    for (;;)
    {
        ptr = cmd_get_param(&data);
        if (*ptr == '\0') break;

        if (*ptr == '+' || *ptr == '-')
        {
            /* level */
            gint tmplevel;

            tmplevel = level2bits(ptr+1);
            if (*ptr == '+')
                *level |= tmplevel;
            else
                *level &= ~tmplevel;
        }
        else
        {
            /* channel/nick name */
            LOG_ITEM_REC *item;

            item = log_item_find(rec, ptr);
            if (item == NULL)
            {
                item = g_new(LOG_ITEM_REC, 1);
                rec->items = g_list_append(rec->items, item);

                item->name = g_strdup(ptr);
                item->level = rec->level;
            }
            level = &item->level;
        }
    }
    g_free(dup);

    signal_emit("log created", 1, rec);
    return rec;
}

LOG_REC *log_create_with_level(gchar *fname, gint level)
{
    LOG_REC *rec;

    g_return_val_if_fail(fname != NULL, NULL);

    rec = log_file_find(fname);
    if (rec == NULL)
    {
        rec = g_new0(LOG_REC, 1);
        logs = g_list_append(logs, rec);

        rec->fname = g_strdup(fname);
        rec->handle = -1;
    }
    rec->level = level;

    signal_emit("log created", 1, rec);
    return rec;
}

void log_append_item(LOG_REC *log, gchar *name, gint level)
{
    LOG_ITEM_REC *item;

    g_return_if_fail(log != NULL);
    g_return_if_fail(name != NULL);

    item = log_item_find(log, name);
    if (item == NULL)
    {
        item = g_new0(LOG_ITEM_REC, 1);
        item->name = g_strdup(name);
        log->items = g_list_append(log->items, item);
    }
    item->level = level;

    signal_emit("log item created", 2, log, item);
}

void log_remove_item(LOG_REC *log, gchar *name)
{
    LOG_ITEM_REC *item;

    g_return_if_fail(log != NULL);
    g_return_if_fail(name != NULL);

    item = log_item_find(log, name);
    if (item != NULL)
    {
        log->items = g_list_remove(log->items, item);
        signal_emit("log item destroyed", 2, log, item);

        g_free(item->name);
        g_free(item);
    }
}

void log_file_destroy(LOG_REC *log)
{
    g_return_if_fail(log != NULL);

    if (log->handle != -1)
        log_file_close(log);

    logs = g_list_remove(logs, log);
    signal_emit("log destroyed", 1, log);

    while (log->items != NULL)
    {
        LOG_ITEM_REC *rec = log->items->data;

        log_remove_item(log, rec->name);
    }
    g_free(log->fname);
    g_free(log);
}

static gboolean event_away(gchar *data, SERVER_REC *server)
{
    LOG_REC *log;
    gchar *fname, *level;

    fname = setup_get_str("awaylog_file");
    level = setup_get_str("awaylog_level");
    if (*fname == '\0' || *level == '\0') return TRUE;

    log = log_file_find(fname);
    if (log != NULL)
    {
        /* awaylog already created */
        if (log->handle == -1)
        {
            /* ..but not open, open it. */
            log_file_open(log);
        }
        return TRUE;
    }

    log = log_create(fname, level);
    if (log != NULL) log_file_open(log);
    return TRUE;
}

static gboolean event_unaway(gchar *data, SERVER_REC *server)
{
    LOG_REC *rec;
    gchar *fname;

    fname = setup_get_str("awaylog_file");
    if (*fname == '\0') return TRUE;

    rec = log_file_find(fname);
    if (rec == NULL || rec->handle == -1)
    {
        /* awaylog not open */
        return TRUE;
    }

    log_file_destroy(rec);
    return TRUE;
}

void log_init(void)
{
    logs = NULL;

    signal_add("print text stripped", (SIGNAL_FUNC) sig_log);
    signal_add("event 306", (SIGNAL_FUNC) event_away);
    signal_add("event 305", (SIGNAL_FUNC) event_unaway);
}

void log_deinit(void)
{
    while (logs != NULL)
        log_file_destroy(logs->data);

    signal_remove("print text stripped", (SIGNAL_FUNC) sig_log);
    signal_remove("event 306", (SIGNAL_FUNC) event_away);
    signal_remove("event 305", (SIGNAL_FUNC) event_unaway);
}
