/*
 server.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *servers;

static GList *lookup_servers;
static gint cmd_tag;

static void server_init(SERVER_REC *server)
{
    GString *str;

    g_return_if_fail(server != NULL);

    str = g_string_new(NULL);

    if (setup_get_bool("toggle_use_ircproxy"))
    {
        g_string_sprintf(str, setup_get_str("proxy_string"), server->address, server->port);
        irc_send_cmd(server, str->str);
    }

    if (server->password != NULL && *server->password != '\0')
    {
        server->cmdcount = 0; /* no slowups needed here :) */
        g_string_sprintf(str, "PASS %s", server->password);
        irc_send_cmd(server, str->str);
    }

    server->cmdcount = 0; /* no slowups needed here :) */
    g_string_sprintf(str, "NICK %s", server->wanted_nick);
    irc_send_cmd(server, str->str);

    server->cmdcount = 0; /* no slowups needed here :) */
    /* if real_name is empty send "??" instead, if nothing is sent, the irc
       server complains */
    g_string_sprintf(str, "USER %s - - :%s", server->username, server->realname);
    irc_send_cmd(server, str->str);

    g_string_free(str, TRUE);
    server->cmdcount = 0;
}

static gboolean server_remove_channels(SERVER_REC *server)
{
    CHANNEL_REC *channel;
    GList *tmp, *next;
    gboolean found;

    g_return_val_if_fail(server != NULL, FALSE);

    found = FALSE;
    for (tmp = g_list_first(channels); tmp != NULL; tmp = next)
    {
        channel = tmp->data;
        next = tmp->next;

        if (channel->server == server)
        {
            channel->server = NULL;
	    if (channel->type != CHANNEL_TYPE_CHANNEL)
		signal_emit("channel server changed", 1, channel);
	    else
	    {
		channel_destroy(channel);
		found = TRUE;
	    }
        }
    }

    return found;
}

static void server_cant_connect(SERVER_REC *server, gchar *msg)
{
    g_return_if_fail(server != NULL);

    lookup_servers = g_list_remove(lookup_servers, server);

    server->connection_lost = TRUE;

    signal_emit("server connect failed", 2, server, msg);
    server_remove_channels(server);

    if (server->connect_tag != -1)
        gui_input_remove(server->connect_tag);

    if (server->connect_pipe[0] != -1)
    {
        close(server->connect_pipe[0]);
        close(server->connect_pipe[1]);
    }
    if (server->ircnet != NULL) g_free(server->ircnet);
    if (server->password != NULL) g_free(server->password);
    g_free(server->tag);
    g_free(server->address);
    g_free(server->nick);
    g_free(server);
}

static gchar *server_create_tag(SERVER_REC *server)
{
    GString *str;
    gchar *start, *end, *tag;
    gint num;

    /* irc.blah.org -> blah */
    start = strchr(server->address, '.');
    end = strrchr(server->address, '.');
    if (start == end) start = server->address; else start++;
    if (end == NULL) end = server->address + strlen(server->address);

    tag = g_new(gchar, end-start+1);
    tag[end-start] = '\0';
    memcpy(tag, start, end-start);

    /* then just append numbers after blah until unused is found.. */
    str = g_string_new(tag);
    for (num = 2; server_find_tag(str->str) != NULL; num++)
        g_string_sprintf(str, "%s%d", tag, num);
    g_free(tag);

    tag = g_strdup(str->str);
    g_string_free(str, TRUE);

    return tag;
}

static gint server_cmd_timeout(void)
{
    struct timeval tv;
    glong secs, usecs;
    GList *tmp;
    gchar *cmd;
    gint len, ret;

    if (gettimeofday(&tv, NULL) != 0)
	g_warning("server_cmd_timeout() : gettimeofday() failed\n");

    for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
    {
	SERVER_REC *server = tmp->data;

	if (server->cmdcount == 0)
	    continue; /* no commands in queue */

	/* get the time difference */
        secs = tv.tv_sec - server->last_cmd.tv_sec;
        usecs = tv.tv_usec - server->last_cmd.tv_usec;
        if (usecs < 0)
        {
            usecs += 1000000;
            secs--;
	}
	usecs = usecs/1000 + secs * 1000;

	if (usecs < server->cmd_queue_speed || --server->cmdcount == 0)
	    continue;

	memcpy(&server->last_cmd, &tv, sizeof(struct timeval));

	/* send command */
	cmd = server->cmdqueue->data;
	len = strlen(cmd);

	ret = net_transmit(server->handle, cmd, len);
	if (ret != len)
	{
	    /* this really shouldn't happen unless connection has broken
	       somehow weirdly. anyway, try again later.. */
	    gchar *p;

	    p = g_strdup(cmd + (ret < 0 ? 0 : ret));
	    g_free(cmd);
            server->cmdqueue->data = p;
	    return 1;
	}

	/* add to rawlog without CR+LF */
	cmd[strlen(cmd)-2] = '\0';
	rawlog_output(server, cmd);

	/* remove from queue */
	g_free(cmd);
	server->cmdqueue = g_list_remove(server->cmdqueue, cmd);
    }

    return 1;
}

static void server_connect_callback_init(SERVER_REC *server, gint handle)
{
    IPADDR ip;
    gint error;

    error = net_geterror(handle);
    if (error != 0)
    {
	server_cant_connect(server, strerror(error));
	return;
    }

    net_getsockname(handle, &ip, NULL);

    gui_input_remove(server->connect_tag);
    server->connect_tag = -1;
    server->connect_time = time(NULL);

    lookup_servers = g_list_remove(lookup_servers, server);
    servers = g_list_append(servers, server);

    server->eventtable = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
    server->eventgrouptable = g_hash_table_new((GHashFunc) g_direct_hash, (GCompareFunc) g_direct_equal);
    server->cmdtable = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
    server->splits = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);

    server->buffer = g_string_new(NULL);
    server_init(server);

    signal_emit("server connected", 1, server);
}

static void get_source_host_ip(void)
{
    IPADDR ip;

    /* FIXME: This will block! */
    source_host_ok = *setup_get_str("source_host") != '\0' &&
	net_gethostname(setup_get_str("source_host"), &ip) != 0;
}

void server_connect_callback_readpipe(SERVER_REC *server, gint handle)
{
    GList *tmp;
    IPADDR ip;
    gchar *errormsg;
    gint error, len;
    gboolean readok;

    for (tmp = g_list_first(lookup_servers); tmp != NULL; tmp = tmp->next)
        if (tmp->data == server) break;

    if (tmp == NULL)
    {
        /* Already closed .. */
        return;
    }

    gui_input_remove(server->connect_tag);
    server->connect_tag = -1;

    errormsg = NULL;
    len = read(handle, &ip, sizeof(ip));
    len += read(handle, &error, sizeof(error));
    readok = len == sizeof(ip) + sizeof(error);

    if (error == -1 && readok)
    {
	/* read the error message */
	if (read(handle, &len, sizeof(len)) == sizeof(len))
	{
	    errormsg = g_malloc(len+1);
	    if (read(handle, errormsg, len) != len)
	    {
		/* failed */
		g_free(errormsg);
		errormsg = NULL;
	    }
	}
    }

    close(server->connect_pipe[0]);
    close(server->connect_pipe[1]);

    server->connect_pipe[0] = -1;
    server->connect_pipe[1] = -1;

    if (!readok)
    {
	server_cant_connect(server, "server_connect_callback_readpipe() : internal error (theory != reality), "
			    "this should never happen, so I don't bother to fix it..");
        return;
    }

    get_source_host_ip();
    server->handle = error == -1 ? -1 :
	net_connect_ip(&ip, setup_get_bool("toggle_use_ircproxy") ? setup_get_int("proxy_port") : server->port,
		       source_host_ok ? &source_host_ip : NULL);
    if (server->handle == -1)
    {
        /* failed */
	server_cant_connect(server,
			    error != -1 ? strerror(errno) : /* connect() failed */
			    (errormsg != NULL ? errormsg : "Host lookup failed")); /* gethostbyname() failed */
	if (errormsg != NULL) g_free(errormsg);
        return;
    }

    server->connect_tag = gui_input_add(server->handle, GUI_INPUT_WRITE/* | GUI_INPUT_EXCEPTION*/,
                                        (GUIInputFunction) server_connect_callback_init, server);
    signal_emit("server connecting", 2, server, &ip);
}

static IRCNET_REC *ircnet_find(gchar *name)
{
    GList *tmp;

    g_return_val_if_fail(name != NULL, NULL);

    for (tmp = ircnets; tmp != NULL; tmp = tmp->next)
    {
	IRCNET_REC *rec = tmp->data;

	if (g_strcasecmp(rec->name, name) == 0)
	    return rec;
    }

    return NULL;
}

SERVER_REC *server_connect(gchar *address, gint port, gchar *password, gchar *nick)
{
    SETUP_SERVER_REC *srec;
    SERVER_REC *rec;
    IRCNET_REC *ircnet;

    g_return_val_if_fail(address != NULL, NULL);
    if (*address == '\0') return NULL;

    rec = g_new0(SERVER_REC, 1);

    if (pipe(rec->connect_pipe) != 0)
    {
        g_warning("server_connect(): pipe() failed.");
        g_free(rec);
        return NULL;
    }

    lookup_servers = g_list_append(lookup_servers, rec);

    /* get ircnet/password from setup */
    srec = server_setup_find(address, -1);
    if (srec != NULL)
    {
	rec->ircnet = srec->ircnet == NULL ? NULL : g_strdup(srec->ircnet);
        rec->password = srec->password == NULL ? NULL : g_strdup(srec->password);
	rec->cmd_queue_speed = srec->cmd_queue_speed > 0 ?
	    srec->cmd_queue_speed : setup_get_int("cmd_queue_speed");
	srec->last_connect = time(NULL);
    }
    if (password != NULL && *password != '\0')
    {
	if (rec->password != NULL) g_free(rec->password);
	rec->password = g_strdup(password);
    }

    rec->max_kicks_in_cmd = 4;
    rec->max_msgs_in_cmd = 3;
    rec->max_modes_in_cmd = 3;

    ircnet = rec->ircnet == NULL ? NULL : ircnet_find(rec->ircnet);
    if (ircnet != NULL)
    {
        if (ircnet->nick != NULL) rec->wanted_nick = g_strdup(ircnet->nick);;
        if (ircnet->username != NULL) rec->username = g_strdup(ircnet->username);;
        if (ircnet->realname != NULL) rec->realname = g_strdup(ircnet->realname);;
        if (ircnet->max_kicks > 0) rec->max_kicks_in_cmd = ircnet->max_kicks;
        if (ircnet->max_msgs > 0) rec->max_msgs_in_cmd = ircnet->max_msgs;
        if (ircnet->max_modes > 0) rec->max_modes_in_cmd = ircnet->max_modes;
    }

    if (rec->wanted_nick == NULL)
	rec->wanted_nick = g_strdup(nick != NULL && *nick != '\0' ? nick : setup_get_str("default_nick"));
    if (rec->username == NULL)
    {
	rec->username = setup_get_str("user_name");
	if (*rec->username == '\0') rec->username = g_get_user_name();
	if (*rec->username == '\0') rec->username = "-";
	rec->username = g_strdup(rec->username);
    }
    if (rec->realname == NULL)
    {
	rec->realname = setup_get_str("real_name");
	if (*rec->realname == '\0') rec->realname = g_get_real_name();
	if (*rec->realname == '\0') rec->realname = "-";
	rec->realname = g_strdup(rec->realname);
    }

    rec->nick = g_strdup(rec->wanted_nick);
    rec->address = g_strdup(address);
    rec->port = port == 0 ? 6667 : port;
    rec->tag = server_create_tag(rec);
    rec->handle = -1;

    rec->connect_pid =
        net_gethostname_nonblock(setup_get_bool("toggle_use_ircproxy") ? setup_get_str("proxy_address") : rec->address,
                                 rec->connect_pipe[1]);

    rec->connect_tag = gui_input_add(rec->connect_pipe[0], GUI_INPUT_READ/* | GUI_INPUT_EXCEPTION*/,
                                     (GUIInputFunction) server_connect_callback_readpipe, rec);

    signal_emit("server looking", 1, rec);

    return rec;
}

void server_disconnect(SERVER_REC *server)
{
    gboolean chans;

    g_return_if_fail(server != NULL);

    if (server->connect_tag != -1)
    {
        /* still connecting to server.. */
        if (server->connect_pid != -1)
            net_disconnect_nonblock(server->connect_pid);
        server_cant_connect(server, NULL);
        return;
    }

    servers = g_list_remove(servers, server);

    signal_emit("server disconnected", 1, server);

    /* close all channels */
    chans = server_remove_channels(server);

    g_list_foreach(server->cmdqueue, (GFunc) g_free, NULL);
    g_list_free(server->cmdqueue);

    if (server->handle != -1)
    {
	if (!chans)
	    net_disconnect(server->handle);
	else
	{
	    /* we were on some channels, try to let the server disconnect so
	       that our quit message is guaranteed to get displayed */
	    net_disconnect_later(server->handle);
	}
    }
    g_string_free(server->buffer, TRUE);
    if (server->ircnet != NULL) g_free(server->ircnet);
    if (server->password != NULL) g_free(server->password);
    if (server->usermode != NULL) g_free(server->usermode);
    if (server->real_address != NULL) g_free(server->real_address);
    g_free(server->tag);
    g_free(server->nick);
    g_free(server->address);
    g_free(server);
}

SETUP_SERVER_REC *server_setup_find(gchar *address, gint port)
{
    GList *tmp;

    g_return_val_if_fail(address != NULL, NULL);

    for (tmp = g_list_first(setupservers); tmp != NULL; tmp = tmp->next)
    {
        SETUP_SERVER_REC *rec = tmp->data;

        if (g_strcasecmp(rec->server, address) == 0 &&
            (port == -1 || rec->port == port)) return rec;
    }

    return NULL;
}

SERVER_REC *server_find_tag(gchar *tag)
{
    SERVER_REC *server;
    GList *servlist;

    g_return_val_if_fail(tag != NULL, NULL);
    if (*tag == '\0') return NULL;

    for (servlist = g_list_first(servers); servlist != NULL; servlist = servlist->next)
    {
        server = (SERVER_REC *) servlist->data;
        if (strcmp(server->tag, tag) == 0) return server;
    }

    return NULL;
}

SERVER_REC *server_find_ircnet(gchar *ircnet)
{
    SERVER_REC *server;
    GList *servlist;

    g_return_val_if_fail(ircnet != NULL, NULL);
    if (*ircnet == '\0') return NULL;

    for (servlist = g_list_first(servers); servlist != NULL; servlist = servlist->next)
    {
        server = (SERVER_REC *) servlist->data;
        if (server->ircnet != NULL && strcmp(server->ircnet, ircnet) == 0) return server;
    }

    return NULL;
}

static gboolean event_connected(gchar *data, SERVER_REC *server, gchar *from)
{
    gchar *params, *ircd_version, *usermodes, *chanmodes, *str;

    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 5, NULL, NULL, &ircd_version, &usermodes, &chanmodes);

    if (strncmp(ircd_version, "pircd", 5) == 0)
    {
	/* yet another stupid irc server that can't understand even the most
	   simple command: PING. Forget about lag detection with this server. */
        server->no_lag_check = TRUE;
    }

    if (server->real_address == NULL)
    {
	/* set the server address */
	server->real_address = g_strdup(from);
    }

    /* check if server understands I and e channel modes */
    if (strchr(chanmodes, 'I') == NULL || strchr(chanmodes, 'e') == NULL)
	server->emode_not_known = TRUE;

    /* last welcome message found - commands can be sent to server now. */
    server->connected = 1;

    /* send default user mode */
    /* FIXME: check what modes are already set, don't send them */
    str = g_strdup_printf("MODE %s %s", server->nick, setup_get_str("default_user_mode"));
    irc_send_cmd(server, str);
    g_free(str);

    signal_emit("event connected", 1, server);
    g_free(params);
    return TRUE;
}

static gboolean event_ping(gchar *data, SERVER_REC *server)
{
    gchar *str;

    g_return_val_if_fail(data != NULL, FALSE);

    str = g_strdup_printf("PONG %s", data);
    irc_send_cmd(server, str);
    g_free(str);

    return TRUE;
}

static gboolean sig_config_read(void)
{
    cmd_tag = gui_timeout_add(setup_get_int("cmd_queue_speed") > 500 ? 500 : setup_get_int("cmd_queue_speed"), (GUITimeoutFunction) server_cmd_timeout, NULL);
    return TRUE;
}

static gboolean event_empty(void)
{
    return TRUE;
}

void servers_init(void)
{
    lookup_servers = servers = NULL;

    signal_add("startup settings read", (SIGNAL_FUNC) sig_config_read);
    signal_add("event 004", (SIGNAL_FUNC) event_connected);
    signal_add("event ping", (SIGNAL_FUNC) event_ping);
    signal_add("event empty", (SIGNAL_FUNC) event_empty);
}

void servers_deinit(void)
{
    gui_timeout_remove(cmd_tag);

    while (servers != NULL)
        server_disconnect(servers->data);
    while (lookup_servers != NULL)
        server_cant_connect(lookup_servers->data, NULL);
    g_list_free(lookup_servers);

    signal_remove("startup settings read", (SIGNAL_FUNC) sig_config_read);
    signal_remove("event 004", (SIGNAL_FUNC) event_connected);
    signal_remove("event ping", (SIGNAL_FUNC) event_ping);
    signal_remove("event empty", (SIGNAL_FUNC) event_empty);
}
