/*
 ui-commands.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static GString *tmpstr;
GList *commands;

void command_bind(gchar *cmd, gchar *category, SIGNAL_FUNC func)
{
    COMMAND_REC *rec;

    g_return_if_fail(cmd != NULL);
    g_return_if_fail(func != NULL);

    rec = g_new0(COMMAND_REC, 1);
    rec->cmd = g_strdup(cmd);
    rec->category = category == NULL ? NULL : g_strdup(category);
    commands = g_list_append(commands, rec);

    cmd = g_strconcat("command ", cmd, NULL);
    signal_add(cmd, func);
    g_free(cmd);

    signal_emit("commandlist new", 1, rec);
}

void command_free(COMMAND_REC *rec)
{
    commands = g_list_remove(commands, rec);
    signal_emit("commandlist remove", 1, rec);

    if (rec->category != NULL) g_free(rec->category);
    g_free(rec->cmd);
    g_free(rec);
}

void command_unbind(gchar *cmd, SIGNAL_FUNC func)
{
    GList *tmp;

    g_return_if_fail(cmd != NULL);
    g_return_if_fail(func != NULL);

    for (tmp = commands; tmp != NULL; tmp = tmp->next)
    {
	COMMAND_REC *rec = tmp->data;

	if (g_strcasecmp(rec->cmd, cmd) == 0)
	{
	    command_free(rec);
	    break;
	}
    }

    cmd = g_strconcat("command ", cmd, NULL);
    signal_remove(cmd, func);
    g_free(cmd);
}

void command_runsub(gchar *cmd, gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *subcmd, *args;

    g_return_if_fail(data != NULL);

    /* get command.. */
    subcmd = g_strdup_printf("command %s %s", cmd, data);
    args = strchr(subcmd+9+strlen(cmd), ' ');
    if (args != NULL) *args++ = '\0'; else args = "";
    while (*args == ' ') args++;

    g_strdown(subcmd);
    if (!signal_emit(subcmd, 3, args, server, channel))
    {
	g_free(subcmd);
	subcmd = g_strdup_printf("default command %s", cmd);
	signal_emit(subcmd, 3, data, server, channel);
    }
    g_free(subcmd);
}

void command_split(SERVER_REC *server, gchar *cmd, gint arg, gint max_nicks)
{
    gchar *pre, *post, *nicks, *p;
    GList *nicklist, *tmp;
    GString *nickstr;
    gint count;

    g_return_if_fail(server != NULL);
    g_return_if_fail(cmd != NULL);

    /* split command into pre, nicks and post pieces */
    pre = g_strdup(cmd); post = nicks = NULL;
    for (p = pre; *p != '\0'; p++)
    {
	if (!isspace(*p)) continue;

	if (arg == 1)
	{
	    /* text after nicks */
	    *p++ = '\0';
	    while (isspace(*p)) p++;
            post = p+1;
	    break;
	}
	else
	{
	    /* find nicks */
	    while (isspace(p[1])) p++;
	    if (--arg == 1)
	    {
                *p = '\0'; nicks = p+1;
	    }
	}
    }

    /* split the nicks */
    nickstr = g_string_new(NULL);
    nicklist = str2list(nicks, ','); count = 0;
    for (tmp = nicklist; ; tmp = tmp->next)
    {
	if (tmp != NULL)
	    g_string_sprintfa(nickstr, nickstr->len == 0 ? "%s" : ",%s", (gchar *) tmp->data);

	if (tmp == NULL || ++count == max_nicks)
	{
	    count = 0;
	    g_string_sprintf(tmpstr, "%s %s :%s", pre, nickstr->str, post);
	    irc_send_cmd(server, tmpstr->str);
	    if (tmp == NULL) break;

	    g_string_truncate(nickstr, 0);
	}
    }
    g_free(nicklist->data);
    g_list_free(nicklist);
    g_string_free(nickstr, TRUE);

    g_free(pre);
}

static gboolean cmd_connect(gchar *data)
{
    SERVER_REC *server;
    gchar *params, *addr, *portstr, *password, *nick;
    gint port;

    g_return_val_if_fail(data != NULL, FALSE);
    params = cmd_get_params(data, 4, &addr, &portstr, &password, &nick);
    if (*addr == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (strcmp(password, "-") == 0)
	*password = '\0';

    port = 6667;
    if (*portstr != '\0')
	sscanf(portstr, "%d", &port);

    /* connect to server */
    server = server_connect(addr, port, password, nick);

    if (cur_channel->server == NULL || cur_channel->type == CHANNEL_TYPE_EMPTY)
    {
	cur_channel->server = server;
	signal_emit("channel server changed", 1, cur_channel);
    }

    g_free(params);
    return TRUE;
}

static gboolean cmd_disconnect(gchar *data, SERVER_REC *server)
{
    gchar *params, *tag, *msg;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &tag, &msg);
    if (*tag != '\0' && strcmp(tag, "*") != 0) server = server_find_tag(tag);
    if (server == NULL) cmd_param_error(CMDERR_NOT_CONNECTED);

    if (server->handle != -1 && server->buffer != NULL)
    {
        /* flush transmit queue */
        g_list_foreach(server->cmdqueue, (GFunc) g_free, NULL);
        g_list_free(server->cmdqueue);
        server->cmdqueue = NULL;
        server->cmdcount = 0;

	/* then send quit message */
        g_string_sprintf(tmpstr, "QUIT :%s", *msg != '\0' ? msg : setup_get_str("default_quit_message"));
        irc_send_cmd(server, tmpstr->str);
    }
    g_free(params);

    server_disconnect(server);
    return TRUE;
}

static gboolean cmd_server(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);
    if (*data == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*data != '+' && server != NULL)
        cmd_disconnect("* Changing server", server);

    cmd_connect(data + (*data == '+' ? 1 : 0));

    return TRUE;
}

static gboolean cmd_msg(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *channel;
    gchar *params, *target, *msg;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (*target == '\0' || *msg == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*target == '=')
    {
        /* dcc msg - don't even try to handle here.. */
        g_free(params);
        return TRUE;
    }

    if (server == NULL || !server->connected) cmd_param_error(CMDERR_NOT_CONNECTED);
    channel = channel_find(server, target);

    g_string_sprintf(tmpstr, "PRIVMSG %s :%s", target, msg);
    command_split(server, tmpstr->str, 2, server->max_msgs_in_cmd);

    g_free(params);
    return TRUE;
}

static gboolean cmd_notice(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *msg;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (*target == '\0' || *msg == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    g_string_sprintf(tmpstr, "NOTICE %s :%s", target, msg);
    command_split(server, tmpstr->str, 2, server->max_msgs_in_cmd);

    g_free(params);
    return TRUE;
}

static gboolean cmd_ctcp(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *ctcpcmd, *ctcpdata;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &target, &ctcpcmd, &ctcpdata);
    if (*target == '\0' || *ctcpcmd == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    g_strup(ctcpcmd);
    if (*ctcpdata == '\0')
        g_string_sprintf(tmpstr, "PRIVMSG %s :\001%s\001", target, ctcpcmd);
    else
        g_string_sprintf(tmpstr, "PRIVMSG %s :\001%s %s\001", target, ctcpcmd, ctcpdata);
    command_split(server, tmpstr->str, 2, server->max_msgs_in_cmd);

    g_free(params);
    return TRUE;
}

static gboolean cmd_nctcp(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *ctcpcmd, *ctcpdata;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &target, &ctcpcmd, &ctcpdata);
    if (*target == '\0' || *ctcpcmd == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    g_strup(ctcpcmd);
    g_string_sprintf(tmpstr, "NOTICE %s :\001%s %s\001", target, ctcpcmd, ctcpdata);
    command_split(server, tmpstr->str, 2, server->max_msgs_in_cmd);

    g_free(params);
    return TRUE;
}

static gboolean cmd_quit(gchar *data)
{
    GList *tmp, *next;
    gchar *str;

    g_return_val_if_fail(data != NULL, FALSE);

    /* disconnect from every server */
    for (tmp = g_list_first(servers); tmp != NULL; tmp = next)
    {
        next = tmp->next;

        str = g_strdup_printf("* %s", *data != '\0' ? data : setup_get_str("default_quit_message"));
        cmd_disconnect(str, tmp->data);
        g_free(str);
    }

    signal_emit("gui exit", 0);
    return TRUE;
}

static gboolean cmd_join(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *channel;
    gchar *next, *key, *chan;
    GString *str;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    if (*data == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    str = g_string_new("JOIN ");
    data = chan = g_strdup(data);

    for (next = chan; next != NULL; chan = next)
    {
        next = strchr(chan, ',');
        if (next != NULL) *next++ = '\0';

        while (*chan == ' ') chan++;
        if (*chan == '\0') break;

        key = strchr(chan, ' ');
        if (key != NULL)
        {
            /* channel key */
            *key++ = '\0';
        }

	if (ischannel(*chan))
	    chan = g_strdup(chan);
	else
        {
	    /* No #, & or ! at start of channel so add # there.. */
	    chan = g_strdup_printf("#%s", chan);
        }

	channel = channel_find(server, chan);
        if (channel != NULL)
        {
	    /* already joined this channel.. */
            signal_emit("gui channel open", 1, channel);
	    g_free(chan);
            continue;
        }

        /* Add channel to /JOIN */
	g_string_sprintfa(str, str->len == 5 ? "%s" : ",%s", chan);
        if (key != NULL) g_string_sprintfa(str, " %s", key);

	channel = channel_create(server, chan + ((chan[0] == '!' && chan[1] == '!') ? 1 : 0), CHANNEL_TYPE_CHANNEL);

	if (key != NULL)
	{
	    /* key specified, can't add any channels to /JOIN list after this */
	    channel->key = g_strdup(key);
	    irc_send_cmd(server, str->str);
	    g_string_assign(str, "JOIN ");
	}
	g_free(chan);
    }

    /* Send the /JOIN command to server */
    if (str->len > 5)
	irc_send_cmd(server, str->str);
    g_string_free(str, TRUE);

    g_free(data);
    return TRUE;
}

static gboolean cmd_part(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channame, *msg;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTCHAN, channel, &channame, &msg);
    if (*channame == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    chanrec = channel_find(server, channame);
    if (chanrec == NULL || chanrec->type != CHANNEL_TYPE_CHANNEL)
        cmd_param_error(CMDERR_CHAN_NOT_FOUND);

    g_string_sprintf(tmpstr, "PART %s", channame);
    if (*msg != '\0') g_string_sprintfa(tmpstr, " :%s", msg);
    irc_send_cmd(server, tmpstr->str);

    g_free(params);
    return TRUE;
}

static gboolean cmd_kick(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *channame, *nicks, *reason;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_OPTCHAN | PARAM_FLAG_GETREST,
                            channel, &channame, &nicks, &reason);

    if (*channame == '\0' || *nicks == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);
    if (!ischannel(*channame)) cmd_param_error(CMDERR_NOT_JOINED);

    g_string_sprintf(tmpstr, "KICK %s %s :%s", channame, nicks, reason);
    command_split(server, tmpstr->str, 3, server->max_kicks_in_cmd);

#if 0
    /* split the kicks to max. n in one cmd */
    nickstr = g_string_new(NULL);
    nicklist = str2list(nicks, ','); count = 0;
    for (tmp = nicklist; ; tmp = tmp->next)
    {
	if (tmp != NULL)
	    g_string_sprintfa(nickstr, nickstr->len == 0 ? "%s" : ",%s", (gchar *) tmp->data);

	if (tmp == NULL || ++count == server->max_kicks_in_cmd)
	{
	    count = 0;
	    g_string_sprintf(tmpstr, "KICK %s %s :%s", channame, nickstr->str, reason);
	    irc_send_cmd(server, tmpstr->str);
	    if (tmp == NULL) break;

	    g_string_truncate(nickstr, 0);
	}
    }
    g_free(nicklist->data);
    g_list_free(nicklist);
    g_string_free(nickstr, TRUE);
#endif
    g_free(params);
    return TRUE;
}

static gboolean cmd_topic(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *channame, *topic;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTCHAN | PARAM_FLAG_GETREST, channel, &channame, &topic);

    if (*topic != '\0')
    {
        /* Change topic */
        g_string_sprintf(tmpstr, "TOPIC %s :%s", channame, topic);
        irc_send_cmd(server, tmpstr->str);
    }

    g_free(params);
    return TRUE;
}

static gboolean cmd_invite(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *nick, *channame;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2, &nick, &channame);
    if (*nick == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);
    if (*channame == '\0' || strcmp(channame, "*") == 0) channame = channel->name;

    g_string_sprintf(tmpstr, "INVITE %s %s", nick, channame);
    irc_send_cmd(server, tmpstr->str);
    g_free(params);
    return TRUE;
}

static gboolean cmd_list(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    gchar *params, *args, *str;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTARGS | PARAM_FLAG_GETREST, &args, &str);

    if (*str == '\0' && stristr(args, "-yes") == NULL)
	cmd_param_error(CMDERR_NOT_GOOD_IDEA);

    g_string_sprintf(tmpstr, "LIST %s", str);
    irc_send_cmd(server, tmpstr->str);
    g_free(params);

    /* add default redirection */
    server_redirect_default(server, "bogus command list");
    return TRUE;
}

static gboolean cmd_who(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    gchar *params, *channel, *args, *rest;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(cur_channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_OPTARGS | PARAM_FLAG_GETREST, &args, &channel, &rest);

    if (strcmp(channel, "*") == 0 || *channel == '\0')
    {
	if (cur_channel->type != CHANNEL_TYPE_CHANNEL)
	    cmd_param_error(CMDERR_NOT_JOINED);

	channel = cur_channel->name;
    }
    if (strcmp(channel, "**") == 0)
    {
	/* ** displays all nicks.. */
	*channel = '\0';
    }

    g_string_sprintf(tmpstr, *rest == '\0' ? "WHO %s" : "WHO %s %s",
		     channel, rest);
    irc_send_cmd(server, tmpstr->str);
    g_free(params);

    /* add default redirection */
    server_redirect_default(server, "bogus command who");
    return TRUE;
}

static gboolean cmd_names(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(cur_channel != NULL, FALSE);

    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);
    if (*data == '\0') cmd_return_error(CMDERR_NOT_GOOD_IDEA);

    if (strcmp(data, "*") == 0)
	data = cur_channel->name;

    if (g_strcasecmp(data, "-YES") == 0)
	g_string_assign(tmpstr, "NAMES");
    else
	g_string_sprintf(tmpstr, "NAMES %s", data);
    irc_send_cmd(server, tmpstr->str);

    return TRUE;
}

static gboolean cmd_whois(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    if (*data == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    g_string_sprintf(tmpstr, "WHOIS %s", data);
    irc_send_cmd(server, tmpstr->str);
    return TRUE;
}

static gboolean cmd_ping(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    struct timeval tv;
    GString *str;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    if (gettimeofday(&tv, NULL) != 0)
    {
        g_warning("cmd_ping() : gettimeofday() failed\n");
        return TRUE;
    }

    str = g_string_new(NULL);
    g_string_sprintf(str, "%s PING %ld %ld", data, tv.tv_sec, tv.tv_usec);
    signal_emit("command ctcp", 3, str->str, server, channel);
    g_string_free(str, TRUE);

    return TRUE;
}

static gboolean cmd_kill(gchar *data, SERVER_REC *server)
{
    gchar *params, *dest, *reason;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &dest, &reason);
    g_string_sprintf(tmpstr, "KILL %s :%s", dest, reason);
    irc_send_cmd(server, tmpstr->str);

    g_free(params);
    return TRUE;
}

static gboolean cmd_away(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    g_string_sprintf(tmpstr, "AWAY :%s", data);
    irc_send_cmd(server, tmpstr->str);

    return TRUE;
}

static gboolean cmd_awayall(gchar *data)
{
    GList *list;

    g_return_val_if_fail(data != NULL, FALSE);

    g_string_sprintf(tmpstr, "AWAY :%s", data);
    for (list = g_list_first(servers); list != NULL; list = list->next)
        irc_send_cmd(list->data, tmpstr->str);

    return TRUE;
}

static gboolean cmd_wallops(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel, *msg;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2, &channel, &msg);
    g_string_sprintf(tmpstr, "WALLOPS %s :%s", channel, msg);
    irc_send_cmd(server, tmpstr->str);

    g_free(params);
    return TRUE;
}

static gboolean cmd_ison(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);
    if (*data == '\0') cmd_return_error(CMDERR_NOT_ENOUGH_PARAMS);

    g_string_sprintf(tmpstr, "ISON :%s", data);
    irc_send_cmd(server, tmpstr->str);

    return TRUE;
}

static gboolean cmd_quote(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL) cmd_return_error(CMDERR_NOT_CONNECTED);

    irc_send_cmd(server, data);
    return TRUE;
}

static void cmd_wall_hash(gpointer key, NICK_REC *nick, GList **nicks)
{
    if (nick->op) *nicks = g_list_append(*nicks, nick);
}

static gboolean cmd_wall(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channame, *msg;
    GList *tmp, *nicks;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTCHAN | PARAM_FLAG_GETREST, channel, &channame, &msg);
    if (*msg == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    chanrec = channel_find(server, channame);
    if (chanrec == NULL || chanrec->type != CHANNEL_TYPE_CHANNEL)
        cmd_param_error(CMDERR_CHAN_NOT_FOUND);

    /* send notice to all ops */
    nicks = NULL;
    g_hash_table_foreach(chanrec->nicks, (GHFunc) cmd_wall_hash, &nicks);

    for (tmp = nicks; tmp != NULL; tmp = tmp->next)
    {
        NICK_REC *rec = tmp->data;

	g_string_sprintf(tmpstr, "NOTICE %s :%s", rec->nick, msg);
	irc_send_cmd(server, tmpstr->str);
    }
    g_list_free(nicks);

    g_free(params);
    return TRUE;
}

static gboolean default_cmd(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL) cmd_return_error(CMDERR_NOT_CONNECTED);

    irc_send_cmd(server, data);
    return FALSE;
}

static gboolean sig_connected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server_redirect_init(server, "", 2, "event 318", "event 402", "event 401",
                         "event 301", "event 311", "event 312", "event 313",
			 "event 317", "event 319", NULL);

    /* gui-gnome can use server_redirect_event() in who/list commands so
       we can't use "command who" or list here.. */
    server_redirect_init(server, "bogus command who", 2, "event 401", "event 315", "event 352", NULL);
    server_redirect_init(server, "bogus command list", 1, "event 321", "event 322", "event 323", NULL);
    return TRUE;
}

void commands_init(void)
{
    tmpstr = g_string_new(NULL);

    signal_add("server connected", (SIGNAL_FUNC) sig_connected);
    command_bind("server", NULL, (SIGNAL_FUNC) cmd_server);
    command_bind("connect", NULL, (SIGNAL_FUNC) cmd_connect);
    command_bind("disconnect", NULL, (SIGNAL_FUNC) cmd_disconnect);
    command_bind("msg", NULL, (SIGNAL_FUNC) cmd_msg);
    command_bind("notice", NULL, (SIGNAL_FUNC) cmd_notice);
    command_bind("ctcp", NULL, (SIGNAL_FUNC) cmd_ctcp);
    command_bind("nctcp", NULL, (SIGNAL_FUNC) cmd_nctcp);
    command_bind("quit", NULL, (SIGNAL_FUNC) cmd_quit);
    command_bind("join", NULL, (SIGNAL_FUNC) cmd_join);
    command_bind("part", NULL, (SIGNAL_FUNC) cmd_part);
    command_bind("kick", NULL, (SIGNAL_FUNC) cmd_kick);
    command_bind("topic", NULL, (SIGNAL_FUNC) cmd_topic);
    command_bind("invite", NULL, (SIGNAL_FUNC) cmd_invite);
    command_bind("list", NULL, (SIGNAL_FUNC) cmd_list);
    command_bind("who", NULL, (SIGNAL_FUNC) cmd_who);
    command_bind("names", NULL, (SIGNAL_FUNC) cmd_names);
    command_bind("whois", NULL, (SIGNAL_FUNC) cmd_whois);
    command_bind("ping", NULL, (SIGNAL_FUNC) cmd_ping);
    command_bind("kill", NULL, (SIGNAL_FUNC) cmd_kill);
    command_bind("away", NULL, (SIGNAL_FUNC) cmd_away);
    command_bind("awayall", NULL, (SIGNAL_FUNC) cmd_awayall);
    command_bind("wallops", NULL, (SIGNAL_FUNC) cmd_wallops);
    command_bind("ison", NULL, (SIGNAL_FUNC) cmd_ison);
    command_bind("quote", NULL, (SIGNAL_FUNC) cmd_quote);
    command_bind("wall", NULL, (SIGNAL_FUNC) cmd_wall);
    signal_add("default command", (SIGNAL_FUNC) default_cmd);
}

void commands_deinit(void)
{
    signal_remove("server connected", (SIGNAL_FUNC) sig_connected);
    command_unbind("server", (SIGNAL_FUNC) cmd_server);
    command_unbind("connect", (SIGNAL_FUNC) cmd_connect);
    command_unbind("disconnect", (SIGNAL_FUNC) cmd_disconnect);
    command_unbind("msg", (SIGNAL_FUNC) cmd_msg);
    command_unbind("notice", (SIGNAL_FUNC) cmd_notice);
    command_unbind("ctcp", (SIGNAL_FUNC) cmd_ctcp);
    command_unbind("nctcp", (SIGNAL_FUNC) cmd_nctcp);
    command_unbind("quit", (SIGNAL_FUNC) cmd_quit);
    command_unbind("join", (SIGNAL_FUNC) cmd_join);
    command_unbind("part", (SIGNAL_FUNC) cmd_part);
    command_unbind("kick", (SIGNAL_FUNC) cmd_kick);
    command_unbind("topic", (SIGNAL_FUNC) cmd_topic);
    command_unbind("invite", (SIGNAL_FUNC) cmd_invite);
    command_unbind("list", (SIGNAL_FUNC) cmd_list);
    command_unbind("who", (SIGNAL_FUNC) cmd_who);
    command_unbind("names", (SIGNAL_FUNC) cmd_names);
    command_unbind("whois", (SIGNAL_FUNC) cmd_whois);
    command_unbind("ping", (SIGNAL_FUNC) cmd_ping);
    command_unbind("kill", (SIGNAL_FUNC) cmd_kill);
    command_unbind("away", (SIGNAL_FUNC) cmd_away);
    command_unbind("awayall", (SIGNAL_FUNC) cmd_awayall);
    command_unbind("wallops", (SIGNAL_FUNC) cmd_wallops);
    command_unbind("ison", (SIGNAL_FUNC) cmd_ison);
    command_unbind("quote", (SIGNAL_FUNC) cmd_quote);
    command_unbind("wall", (SIGNAL_FUNC) cmd_wall);
    signal_remove("default command", (SIGNAL_FUNC) default_cmd);

    g_string_free(tmpstr, TRUE);
}
