/*
 gui-printtext.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint mirc_colors[] = { 15, 0, 1, 2, 4, 6, 5, 4, 14, 10, 3, 11, 9, 13, 8, 7, 15 };
static gint max_textwidget_lines;

static void newline(WINDOW_REC *window, gboolean active, gboolean wordsplit)
{
    GUI_WINDOW_REC *gui = WINDOW_GUI(window);

    if (gui->ypos < LINES-3)
        gui->ypos++;
    else
    {
        gui->starty++;
        if (active)
        {
            scroll_up(0, LINES-3);
            move(LINES-3, 0); clrtoeol();
        }
    }

    gui->cur_line = g_new0(LINE_REC, 1);
    gui->cur_line->time = time(NULL);
    if (!wordsplit || COLS <= 10)
        gui->xpos = 0;
    else
    {
        gint n;

        gui->xpos = 10;
        gui->cur_line->line = g_new(gushort, 10*2);
        gui->cur_line->linelen = 10;

        for (n = 0; n < 10*2; n += 2)
        {
            gui->cur_line->line[n] = current_theme->default_color;
            gui->cur_line->line[n+1] = ' ';
        }

    }
    gui->lines = g_slist_append(gui->lines, gui->cur_line);
}

static void redraw_line(WINDOW_REC *window, gint line)
{
    GSList *lines;
    LINE_REC *rec;
    gushort *l;
    gint n;

    lines = g_slist_nth(WINDOW_GUI(window)->lines, line);
    if (lines == NULL) return;

    rec = lines->data;
    l = rec->line;
    for (n = 0; n < COLS && n < rec->linelen; n++)
    {
	set_color(*l); l++;
	addch(*l); l++;
    }
}

static gboolean gui_printtext(CHANNEL_REC *channel, gpointer fgcolor, gpointer bgcolor, gpointer flags, gchar *str, gpointer level)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;
    gint fg, bg, iflags;
    gboolean active, wordsplit;
    gchar *ptr, *p;
    gushort *line;
    guchar c;
    gint n, len;

    g_return_val_if_fail(str != NULL, FALSE);

    iflags = GPOINTER_TO_INT(flags);

    window = CHANNEL_PARENT(channel);
    gui = WINDOW_GUI(window);
    active = gui->parent->active == window;
    if (active && window->lines-gui->starty > LINES-1)
        active = FALSE;

    if (gui->cur_line == NULL)
    {
        gui->cur_line = g_new0(LINE_REC, 1);
        gui->cur_line->time = time(NULL);
        gui->lines = g_slist_append(gui->lines, gui->cur_line);
    }

    if (gui->cur_line->loglevel <= 0)
        gui->cur_line->loglevel = GPOINTER_TO_INT(level);

    if (max_textwidget_lines > 0 && max_textwidget_lines <= window->lines)
    {
        /* need to cut the first line */
        LINE_REC *line;

        line = gui->lines->data;
        g_free(line->line);
        g_free(line);
        gui->lines = g_slist_remove(gui->lines, line);
	window->lines--;
	gui->starty--;
	if (gui->starty < 0 && gui->parent->active == window)
	{
            scroll_up(0, LINES-3);
	    move(LINES-3, 0);
	    redraw_line(window, LINES-3);
	    clrtoeol();
	}
    }

    if (iflags & PRINTFLAG_MIRC_COLOR)
    {
	/* mirc colors */
	fg = GPOINTER_TO_INT(fgcolor) < 0 || GPOINTER_TO_INT(fgcolor) > 16 ?
	    current_theme->default_color : mirc_colors[GPOINTER_TO_INT(fgcolor)];
	bg = GPOINTER_TO_INT(bgcolor) < 0 || GPOINTER_TO_INT(bgcolor) > 16 ? 0 :
	    mirc_colors[GPOINTER_TO_INT(bgcolor)];
    }
    else
    {
	/* default colors */
	fg = GPOINTER_TO_INT(fgcolor) < 0 || GPOINTER_TO_INT(fgcolor) > 15 ?
	    current_theme->default_color : GPOINTER_TO_INT(fgcolor);
	bg = GPOINTER_TO_INT(bgcolor) < 0 || GPOINTER_TO_INT(bgcolor) > 15 ? 0 :
	    GPOINTER_TO_INT(bgcolor);

	if (fg > 8) fg -= 8;
    }

    if (iflags & PRINTFLAG_REVERSE)
    {
        gint tmp;

        tmp = fg; fg = bg; bg = tmp;
    }

    if (iflags & PRINTFLAG_BOLD) fg |= 8;
    if (fg == 8) fg |= ATTR_COLOR8;
    if (iflags & PRINTFLAG_UNDERLINE) fg |= ATTR_UNDERLINE;

    if (active)
        set_color(fg | (bg << 4));

    /* \n can be only at the start of the line.. */
    if (*str == '\n')
    {
        newline(window, active, FALSE);
        str++;
    }

    wordsplit = FALSE;
    for (ptr = str; *ptr != '\0';)
    {
        if (wordsplit || gui->xpos == COLS)
        {
            gui->cur_line->next_joined = TRUE;
            window->lines++;
            newline(window, active, wordsplit);
        }

        /* check how much text fits to screen.. */
        len = strlen(ptr);
        if (len+gui->xpos > COLS)
        {
            len = COLS-gui->xpos;

            /* try to split at space */
            if (gui->xpos > 0)
            {
                for (p = ptr+len-1; p > ptr; p--)
                {
                    if (isspace(*p))
                    {
                        len = (gint) (p-ptr)+1;
                        wordsplit = TRUE;
                        break;
                    }
                }

                if (p == ptr && gui->xpos > COLS/5)
                {
                    /* FIXME: we should check here if the end of the last line
                       should be split to this line too .. */
                    wordsplit = TRUE;
                    continue;
                }
            }
        }

        /* get memory for line.. */
        gui->cur_line->line =
            gui->cur_line->line == NULL ? g_new(gushort, len*2) :
            g_realloc(gui->cur_line->line, sizeof(gushort) * 2*(len + gui->cur_line->linelen));
        gui->cur_line->linelen += len;

        /* write text.. */
        if (active) move(gui->ypos, gui->xpos);
        line = gui->cur_line->line + gui->xpos*2;

        for (n = 0; n < len; n++, ptr++)
        {
	    c = (guchar) (*ptr == '\t' ? ' ' : *ptr);
	    /*FIXME: c = transtable[(guchar) *ptr] */

	    if (c < 32)
	    {
		/* low-ascii */
		c = c+'A'-1;

		*line++ = 7 << 4;
		*line++ = c;
		if (active)
		{
		    set_color(7 << 4);
		    addch(c);
		    set_color(fg | (bg << 4));
		}
	    }
	    else
	    {
		*line++ = fg | (bg << 4);
		*line++ = c;
		if (active) addch(c);
	    }
        }
        gui->xpos += len;
    }

    return TRUE;
}

static gboolean cmd_clear(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;
    GSList *tmp;
    gint n;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    gui = WINDOW_GUI(window);

    /* clear screen */
    if (WINDOW_GUI(CHANNEL_PARENT(cur_channel))->parent->active == window)
    {
        for (n = 0; n < LINES-2; n++)
        {
            move(n, 0);
            clrtoeol();
        }
        screen_refresh();
    }

    /* clear window buffer */
    for (tmp = gui->lines; tmp != NULL; tmp = tmp->next)
    {
        LINE_REC *rec = tmp->data;

        g_free(rec->line);
        g_free(rec);
    }
    g_slist_free(gui->lines);
    gui->lines = NULL;
    gui->cur_line = NULL;
    gui->xpos = gui->ypos = gui->starty = 0;
    window->lines = 0;

    return TRUE;
}

static gboolean sig_printtext_finished(CHANNEL_REC *channel)
{
    if (WINDOW_GUI(CHANNEL_PARENT(channel))->parent->active == CHANNEL_PARENT(channel))
	screen_refresh();
    return TRUE;
}

static gboolean sig_setup(void)
{
    max_textwidget_lines = setup_get_int("max_textwidget_lines");
    return TRUE;
}

void gui_printtext_init(void)
{
    signal_add("gui print text", (SIGNAL_FUNC) gui_printtext);
    command_bind("clear", NULL, (SIGNAL_FUNC) cmd_clear);
    signal_add("print text finished", (SIGNAL_FUNC) sig_printtext_finished);
    signal_add("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_add("setup changed", (SIGNAL_FUNC) sig_setup);
}

void gui_printtext_deinit(void)
{
    signal_remove("gui print text", (SIGNAL_FUNC) gui_printtext);
    command_unbind("clear", (SIGNAL_FUNC) cmd_clear);
    signal_remove("print text finished", (SIGNAL_FUNC) sig_printtext_finished);
    signal_remove("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_remove("setup changed", (SIGNAL_FUNC) sig_setup);
}
