/*
 setup-keyboard.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "setup-int.h"

static GtkWidget *setup_dialog;
static GList *temp_keys;

static void clist_add_key(GtkCList *clist, KEY_REC *key, gint pos)
{
    gchar *titles[4];

    g_return_if_fail(clist != NULL);
    g_return_if_fail(key != NULL);

    titles[0] = key->key;
    titles[1] = key->info->id;
    titles[2] = key->data == NULL ? NULL : key->data;
    titles[3] = key->info->description;

    pos = pos < 0 ? gtk_clist_append(GTK_CLIST(clist), titles) :
	gtk_clist_insert(GTK_CLIST(clist), pos, titles);
    gtk_clist_set_row_data(GTK_CLIST(clist), pos, key);
}

static void setup_key_remove(GtkCList *clist, KEY_REC *key)
{
    gint pos;

    g_return_if_fail(key != NULL);

    temp_keys = g_list_remove(temp_keys, key);
    if (key->data != NULL) g_free(key->data);
    g_free(key->key);
    g_free(key);

    pos = gtk_clist_find_row_from_data(clist, key);
    if (pos != -1) gtk_clist_remove(clist, pos);
}

static void key_fill_data(GtkWidget *dialog, KEY_REC *key)
{
    GtkEntry *eaction, *ekey, *edata;

    eaction = gtk_object_get_data(GTK_OBJECT(dialog), "action_entry");
    ekey = gtk_object_get_data(GTK_OBJECT(dialog), "key_entry");
    edata = gtk_object_get_data(GTK_OBJECT(dialog), "data_entry");

    key->info = key_info_find(gtk_entry_get_text(eaction));
    key->key = g_strdup(gtk_entry_get_text(ekey));
    key->data = gtk_entry_get_text(edata);
    key->data = *((gchar *) key->data) == '\0' ? NULL : g_strdup(key->data);
}

static void sig_add_ok(GtkWidget *dialog)
{
    GtkCList *clist;
    KEY_REC *key;

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");

    key = g_new0(KEY_REC, 1);
    temp_keys = g_list_append(temp_keys, key);

    key_fill_data(dialog, key);
    clist_add_key(clist, key, -1);

    gtk_widget_destroy(dialog);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_edit_ok(GtkWidget *dialog)
{
    GtkCList *clist;
    KEY_REC *key;
    gint pos;

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    key = gtk_object_get_data(GTK_OBJECT(dialog), "key");
    pos = gtk_clist_find_row_from_data(clist, key);

    if (key->data != NULL) g_free(key->data);
    g_free(key->key);
    key_fill_data(dialog, key);

    gtk_clist_remove(clist, pos);
    clist_add_key(clist, key, pos);

    gtk_widget_destroy(dialog);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

/* signal: key pressed (in entry field) */
static gint sig_keypress(GtkWidget *widget, GdkEventKey *event, GtkEntry *entry)
{
    gchar *key;

    g_return_val_if_fail(event != NULL, 0);
    g_return_val_if_fail(entry != NULL, 0);

    key = gui_key2str(event);
    gtk_entry_set_text(entry, key);
    g_free(key);

    gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
    return 1;
}

static void setup_key_configure(GtkCList *clist, KEY_REC *key)
{
    GtkWidget *dialog, *table, *entry, *label, *combo;
    GList *tmp, *strings;
    gint actionpos;

    g_return_if_fail(clist != NULL);

    /* key key infos */
    strings = NULL; actionpos = 0;
    for (tmp = keyinfos; tmp != NULL; tmp = tmp->next)
    {
	KEYINFO_REC *rec = tmp->data;

	if (key != NULL && key->info == rec) actionpos = g_list_length(strings);
        strings = g_list_append(strings, rec->id);
    }

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL, NULL);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
		       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gui_widget_depends(setup_dialog, dialog);
    if (key != NULL) gui_widget_depends_data(dialog, "keyinfo destroyed", key->info);
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);
    gtk_object_set_data(GTK_OBJECT(dialog), "key", key);

    table = gtk_table_new(3, 3, FALSE);
    gtk_table_set_col_spacing(GTK_TABLE(table), 0, 5);
    gtk_container_border_width(GTK_CONTAINER(table), 5);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);

    /* action */
    label = gtk_label_new(_("Action"));
    gtk_misc_set_alignment(GTK_MISC(label), 1, .5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);

    combo = gtk_combo_new();
    gtk_object_set_data(GTK_OBJECT(dialog), "action_entry", GTK_COMBO(combo)->entry);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
    gtk_combo_set_value_in_list(GTK_COMBO(combo), TRUE, FALSE);
    gtk_combo_set_popdown_strings(GTK_COMBO(combo), strings);
    if (key != NULL) gtk_list_select_item(GTK_LIST(GTK_COMBO(combo)->list), actionpos);
    gtk_table_attach(GTK_TABLE(table), combo, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND, 0, 0, 0);

    /* key */
    entry = gui_create_tablelabelentry(table, 0, 1, _("Key"), key == NULL ? NULL : key->key);
    gtk_object_set_data(GTK_OBJECT(dialog), "key_entry", entry);
    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
    gtk_signal_connect(GTK_OBJECT(entry), "key_press_event",
                       GTK_SIGNAL_FUNC(sig_keypress), entry);

    /* data */
    entry = gui_create_tablelabelentry(table, 0, 2, _("Data"), key == NULL ? NULL : key->data);
    gtk_object_set_data(GTK_OBJECT(dialog), "data_entry", entry);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(key == NULL ? sig_add_ok : sig_edit_ok), GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_widget_show_all(dialog);
}

static void sig_add_key(GtkCList *clist)
{
    setup_key_configure(clist, NULL);
}

static void sig_edit_key(GtkCList *clist)
{
    KEY_REC *key;

    g_return_if_fail(clist != NULL);

    if (clist->selection == NULL)
	return; /* no selection */

    key = gtk_clist_get_row_data(clist, GPOINTER_TO_INT(clist->selection->data));
    if (key != NULL) setup_key_configure(clist, key);
}

static void sig_delete_key(GtkCList *clist)
{
    KEY_REC *key;

    g_return_if_fail(clist != NULL);

    while (clist->selection != NULL)
    {
	key = gtk_clist_get_row_data(clist, GPOINTER_TO_INT(clist->selection->data));
	if (key != NULL)
	    setup_key_remove(clist, key);
	else
            gtk_clist_unselect_row(clist, GPOINTER_TO_INT(clist->selection->data), 0);
    }

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static gboolean sig_keyinfo_destroyed(KEYINFO_REC *info)
{
    GtkCList *clist;
    GList *tmp, *next;

    g_return_val_if_fail(info != NULL, FALSE);

    if (setup_dialog == NULL) return TRUE;
    clist = gtk_object_get_data(GTK_OBJECT(setup_dialog), "clist");

    for (tmp = temp_keys; tmp != NULL; tmp = next)
    {
	KEY_REC *rec = tmp->data;

	next = tmp->next;
	if (rec->info == info)
	    setup_key_remove(clist, rec);
    }

    return TRUE;
}

static gboolean sig_keyboard_apply(GtkWidget *dialog)
{
    GList *tmp;

    /* first remove all keys */
    for (tmp = keyinfos; tmp != NULL; tmp = tmp->next)
    {
	KEYINFO_REC *rec = tmp->data;

	while (rec->keys != NULL)
	    key_configure_remove(((KEY_REC *) rec->keys->data)->key);
    }

    /* add the new keys */
    for (tmp = temp_keys; tmp != NULL; tmp = tmp->next)
    {
	KEY_REC *rec = tmp->data;

        key_configure_add(rec->info->id, rec->data, rec->key);
    }

    keyboard_save();
    return TRUE;
}

static gboolean sig_keyboard_destroyed(void)
{
    setup_dialog = NULL;

    while (temp_keys != NULL)
    {
	KEY_REC *rec = temp_keys->data;

	if (rec->data != NULL) g_free(rec->data);
	g_free(rec->key);
	g_free(rec);

	temp_keys = g_list_remove(temp_keys, rec);
    }

    return TRUE;
}

static gboolean setup_keyboard(GtkWidget *dialog)
{
    GtkWidget *hbox, *vbox, *scrollwin, *clist, *buttonbox, *button;
    gchar *titles[4];
    GList *tmp, *tmp2;

    titles[0] = _("Key"); titles[1] = _("Action");
    titles[2] = _("Data"); titles[3] = _("Description");

    setup_dialog = dialog;
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_set_usize(vbox, 500, 350);

    hbox = gtk_hbox_new(FALSE, 7);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

    scrollwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(hbox), scrollwin, TRUE, TRUE, 0);

    clist = gtk_clist_new_with_titles(4, titles);
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollwin), clist);
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_EXTENDED);

    /* add button box */
    buttonbox = gtk_vbutton_box_new();
    gtk_box_pack_start(GTK_BOX(hbox), buttonbox, FALSE, TRUE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 7);

    button = gtk_button_new_with_label(_("Add..."));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_add_key), GTK_OBJECT(clist));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Edit..."));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_edit_key), GTK_OBJECT(clist));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Delete"));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_delete_key), GTK_OBJECT(clist));
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    /* copy keys to temp_keys list */
    gtk_clist_freeze(GTK_CLIST(clist));
    temp_keys = NULL;
    for (tmp = keyinfos; tmp != NULL; tmp = tmp->next)
    {
	KEYINFO_REC *rec = tmp->data;

	for (tmp2 = rec->keys; tmp2 != NULL; tmp2 = tmp2->next)
	{
	    KEY_REC *rec = tmp2->data;

	    if (rec->key != NULL)
	    {
		KEY_REC *rec2;

		/* duplicate the key record */
		rec2 = g_new0(KEY_REC, 1);
		rec2->info = rec->info;
		rec2->key = g_strdup(rec->key);
		rec2->data = rec->data == NULL ? NULL : g_strdup(rec->data);

		temp_keys = g_list_append(temp_keys, rec2);
		clist_add_key(GTK_CLIST(clist), rec2, -1);
	    }
	}
    }
    gtk_clist_thaw(GTK_CLIST(clist));

    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 3, TRUE);

    gnome_property_box_append_page(GNOME_PROPERTY_BOX(dialog),
				   vbox, gtk_label_new("Keyboard"));

    /* hide tab */
    gtk_widget_show_all(dialog); /* removes Gtk-CRITICAL message that hiding tabs creates. */
#ifdef HAVE_GNOME
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(GNOME_PROPERTY_BOX(dialog)->notebook), FALSE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK(GNOME_PROPERTY_BOX(dialog)->notebook), FALSE);
#else
    {
	GnomePropertyBox *box;

	box = gtk_object_get_data(GTK_OBJECT(dialog), "propertybox");
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(box->notebook), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(box->notebook), FALSE);
    }
#endif
    return TRUE;
}

void init_keyboard(void)
{
    setup_dialog = NULL;

    signal_add("setup keyboard apply", (SIGNAL_FUNC) sig_keyboard_apply);
    signal_add("setup keyboard destroyed", (SIGNAL_FUNC) sig_keyboard_destroyed);
    signal_add("setup keyboard", (SIGNAL_FUNC) setup_keyboard);

    signal_add("keyinfo destroyed", (SIGNAL_FUNC) sig_keyinfo_destroyed);
}

void deinit_keyboard(void)
{
    if (setup_dialog != NULL)
        gtk_widget_destroy(setup_dialog);

    signal_remove("setup keyboard apply", (SIGNAL_FUNC) sig_keyboard_apply);
    signal_remove("setup keyboard destroyed", (SIGNAL_FUNC) sig_keyboard_destroyed);
    signal_remove("setup keyboard", (SIGNAL_FUNC) setup_keyboard);

    signal_remove("keyinfo destroyed", (SIGNAL_FUNC) sig_keyinfo_destroyed);
}
