/*
 setup-ircnets.c : IRC networks notebook entry for preferences

    Copyright (C) 1999 Timo Sirainen
                       Kari Lavikka (tuner@bdb.fi)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "setup-int.h"

GList *ircnets; /* list of irc networks */
GList *temp_ircnets; /* to make cancel work.. */

static GtkWidget *setup_dialog;

static void redraw_ircnets(GtkCList *clist)
{
    GList *tmp;
    gint row;

    g_return_if_fail(clist != NULL);

    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);

    for (tmp = temp_ircnets; tmp != NULL; tmp = tmp->next)
    {
	IRCNET_REC *rec = tmp->data;

        row = gtk_clist_append(clist, (gchar **) &rec->name);
        gtk_clist_set_row_data(clist, row, rec);
    }

    gtk_clist_thaw(clist);

    gtk_object_set_data(GTK_OBJECT(clist), "data", NULL);
    gtk_clist_select_row(clist, 0, 0);
}

IRCNET_REC *ircnet_find(gchar *net)
{
    GList *tmp;

    for (tmp = temp_ircnets; tmp != NULL; tmp = tmp->next)
    {
	IRCNET_REC *rec = tmp->data;

	if (g_strcasecmp(rec->name, net) == 0)
	    return rec;
    }

    return NULL;
}

void ircnet_add(IRCNET_REC *ircnet)
{
    GtkCList *clist;

    temp_ircnets = g_list_append(temp_ircnets, ircnet);

    if (setup_dialog != NULL)
    {
        clist = gtk_object_get_data(GTK_OBJECT(setup_dialog), "ircnet_clist");
        redraw_ircnets(clist);
    }
}

static void ircnet_free(IRCNET_REC *ircnet)
{
    if (ircnet->nick != NULL) g_free(ircnet->nick);
    if (ircnet->username != NULL) g_free(ircnet->username);
    if (ircnet->realname != NULL) g_free(ircnet->realname);
    g_free(ircnet->name);
    g_free(ircnet);
}

static void ircnets_copy(GList *from, GList **to)
{
    IRCNET_REC *ircnet;
    GList *tmp;

    for (tmp = from; tmp != NULL; tmp = tmp->next)
    {
	IRCNET_REC *rec = tmp->data;

	ircnet = g_new0(IRCNET_REC, 1);
	memcpy(ircnet, rec, sizeof(IRCNET_REC));
	ircnet->name = g_strdup(rec->name);
	if (rec->nick != NULL) ircnet->nick = g_strdup(rec->nick);
	if (rec->username != NULL) ircnet->username = g_strdup(rec->username);
	if (rec->realname != NULL) ircnet->realname = g_strdup(rec->realname);

	*to = g_list_append(*to, ircnet);
    }
}

static gboolean sig_setup_apply(void)
{
    proplist_t lprop, iprop, pkey;
    GList *tmp;

    g_list_foreach(ircnets, (GFunc) ircnet_free, NULL);
    g_list_free(ircnets);
    ircnets = NULL;

    ircnets_copy(temp_ircnets, &ircnets);

    /* First clear ircnets list.. */
    pkey = PLMakeString("ircnets");
    PLRemoveDictionaryEntry(cprop, pkey);

    lprop = NULL;
    if (g_list_length(ircnets) > 0)
    {
	/* Create an array of ircnets */
	lprop = PLMakeArrayFromElements(NULL);
	cprop = PLInsertDictionaryEntry(cprop, pkey, lprop);
	PLRelease(pkey);
    }

    for (tmp = ircnets; tmp != NULL; tmp = tmp->next)
    {
	IRCNET_REC *rec = tmp->data;

	/* create dictionary for ircnet and add it to array */
	iprop = PLMakeDictionaryFromEntries(NULL, NULL);
	lprop = PLAppendArrayElement(lprop, iprop);

	config_set_str(iprop, "name", rec->name);
	if (rec->nick != NULL) config_set_str(iprop, "nick", rec->nick);
	if (rec->username != NULL) config_set_str(iprop, "username", rec->username);
	if (rec->realname != NULL) config_set_str(iprop, "realname", rec->realname);
	if (rec->max_kicks > 0) config_set_int(iprop, "max_kicks", rec->max_kicks);
	if (rec->max_modes > 0) config_set_int(iprop, "max_modes", rec->max_modes);
	if (rec->max_msgs > 0) config_set_int(iprop, "max_msgs", rec->max_msgs);
    }

    return TRUE;
}

static gboolean sig_setup_destroyed(void)
{
    g_list_foreach(temp_ircnets, (GFunc) ircnet_free, NULL);
    g_list_free(temp_ircnets);
    temp_ircnets = NULL;

    return TRUE;
}

static GtkWidget *ircnet_dialog(IRCNET_REC *ircnet, GtkSignalFunc func)
{
    GtkWidget *dialog, *table, *entry, *button, *sepa;
    gchar str[100];

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK,
			      GNOME_STOCK_BUTTON_CANCEL, NULL);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
		       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gui_widget_depends(setup_dialog, dialog);

    table = gtk_table_new(10, 2, FALSE);
    gtk_container_border_width(GTK_CONTAINER(table), 7);
    gtk_table_set_row_spacings(GTK_TABLE(table), 3);
    gtk_table_set_col_spacings(GTK_TABLE(table), 7);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);

    entry = gui_create_tablelabelentry(table, 0, 0, _("IRC Network"), ircnet == NULL ? NULL : ircnet->name);
    gtk_object_set_data(GTK_OBJECT(dialog), "ircnet", entry);

    if (ircnet != NULL)
    {
	button = gtk_check_button_new_with_label(_("Change ircnet in server and channel lists also\nWARNING: changing channel list can't be undone!"));
	gtk_object_set_data(GTK_OBJECT(dialog), "button", button);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	gtk_table_attach(GTK_TABLE(table), button, 0, 2, 1, 2,
			 (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 3);
    }

    sepa = gtk_hseparator_new();
    gtk_table_attach(GTK_TABLE(table), sepa, 0, 2, 2, 3,
		     (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 3);

    entry = gui_create_tablelabelentry(table, 0, 3, _("Nick"), ircnet == NULL ? NULL : ircnet->nick);
    gtk_object_set_data(GTK_OBJECT(dialog), "nick", entry);
    entry = gui_create_tablelabelentry(table, 0, 4, _("Username"), ircnet == NULL ? NULL : ircnet->username);
    gtk_object_set_data(GTK_OBJECT(dialog), "username", entry);
    entry = gui_create_tablelabelentry(table, 0, 5, _("Realname"), ircnet == NULL ? NULL : ircnet->realname);
    gtk_object_set_data(GTK_OBJECT(dialog), "realname", entry);

    sepa = gtk_hseparator_new();
    gtk_table_attach(GTK_TABLE(table), sepa, 0, 2, 6, 7,
		     (GtkAttachOptions) GTK_FILL, (GtkAttachOptions) GTK_FILL, 0, 3);

    g_snprintf(str, sizeof(str), "%d", ircnet == NULL ? 4 : ircnet->max_kicks);
    entry = gui_create_tablelabelentry(table, 0, 8, _("Max kicks per cmd"), str);
    gtk_object_set_data(GTK_OBJECT(dialog), "kicks", entry);

    g_snprintf(str, sizeof(str), "%d", ircnet == NULL ? 3 : ircnet->max_modes);
    entry = gui_create_tablelabelentry(table, 0, 9, _("Max modes per cmd"), str);
    gtk_object_set_data(GTK_OBJECT(dialog), "modes", entry);

    g_snprintf(str, sizeof(str), "%d", ircnet == NULL ? 3 : ircnet->max_msgs);
    entry = gui_create_tablelabelentry(table, 0, 10, _("Max msgs per cmd"), str);
    gtk_object_set_data(GTK_OBJECT(dialog), "msgs", entry);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, func, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_widget_show_all(dialog);
    return dialog;
}

static void sig_ircnet_add_ok(GtkWidget *dialog)
{
    GtkToggleButton *button;
    GtkCList *clist;
    IRCNET_REC *ircnet;
    gchar *name, *nick, *username, *realname, *kicks, *modes, *msgs;

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    name = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "ircnet")));
    button = gtk_object_get_data(GTK_OBJECT(dialog), "button");
    nick = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "nick")));
    username = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "username")));
    realname = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "realname")));
    kicks = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "kicks")));
    modes = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "modes")));
    msgs = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "msgs")));

    if (*name == '\0')
    {
	gtk_widget_destroy(dialog);
	return;
    }

    ircnet = g_new0(IRCNET_REC, 1);
    ircnet->name = g_strdup(name);
    ircnet->nick = *nick == '\0' ? NULL : g_strdup(nick);
    ircnet->username = *username == '\0' ? NULL : g_strdup(username);
    ircnet->realname = *realname == '\0' ? NULL : g_strdup(realname);
    sscanf(kicks, "%d", &ircnet->max_kicks);
    sscanf(modes, "%d", &ircnet->max_modes);
    sscanf(msgs, "%d", &ircnet->max_msgs);

    ircnet_add(ircnet);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
    redraw_ircnets(clist);

    gtk_widget_destroy(dialog);
}

static void sig_ircnet_add(GtkWidget *widget, GtkCList *clist)
{
    GtkWidget *dialog;

    dialog = ircnet_dialog(NULL, GTK_SIGNAL_FUNC(sig_ircnet_add_ok));
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);
}

static void sig_ircnet_edit_ok(GtkWidget *dialog)
{
    GtkToggleButton *button;
    GtkCList *clist;
    IRCNET_REC *ircnet;
    gchar *name, *nick, *username, *realname, *kicks, *modes, *msgs;

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "clist");
    name = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "ircnet")));
    button = gtk_object_get_data(GTK_OBJECT(dialog), "button");
    nick = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "nick")));
    username = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "username")));
    realname = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "realname")));
    kicks = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "kicks")));
    modes = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "modes")));
    msgs = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "msgs")));

    if (*name == '\0')
    {
	gtk_widget_destroy(dialog);
	return;
    }

    ircnet = gtk_object_get_data(GTK_OBJECT(dialog), "rec");

    if (strcmp(name, ircnet->name) != 0)
    {
	/* irc net name changed */
	if (gtk_toggle_button_get_active(button))
	{
            /* replace irc network in all server and channel lists... */
            signal_emit("setup ircnet changed", 2, ircnet->name, name);
        }

	g_free(ircnet->name);
	ircnet->name = g_strdup(name);
    }

    if (ircnet->nick) g_free(ircnet->nick);
    if (ircnet->username) g_free(ircnet->username);
    if (ircnet->realname) g_free(ircnet->realname);

    ircnet->nick = *nick == '\0' ? NULL : g_strdup(nick);
    ircnet->username = *username == '\0' ? NULL : g_strdup(username);
    ircnet->realname = *realname == '\0' ? NULL : g_strdup(realname);
    sscanf(kicks, "%d", &ircnet->max_kicks);
    sscanf(modes, "%d", &ircnet->max_modes);
    sscanf(msgs, "%d", &ircnet->max_msgs);

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
    redraw_ircnets(clist);

    gtk_widget_destroy(dialog);
}

static void sig_ircnet_edit(GtkWidget *widget, GtkCList *clist)
{
    GtkWidget *dialog;
    IRCNET_REC *rec;

    rec = gtk_object_get_data(GTK_OBJECT(clist), "data");
    if (rec == NULL) return;

    dialog = ircnet_dialog(rec, GTK_SIGNAL_FUNC(sig_ircnet_edit_ok));

    gtk_object_set_data(GTK_OBJECT(dialog), "rec", rec);
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);
}

static void sig_ircnet_remove(GtkWidget *widget, GtkCList *clist)
{
    IRCNET_REC *rec;

    rec = gtk_object_get_data(GTK_OBJECT(clist), "data");
    if (rec == NULL) return;

    ircnet_free(rec);
    temp_ircnets = g_list_remove(temp_ircnets, rec);

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
    redraw_ircnets(clist);
}

static void sig_select_row(GtkCList *clist, gint row)
{
    gtk_object_set_data(GTK_OBJECT(clist), "data",
                        gtk_clist_get_row_data(clist, row));
}

static void sig_unselect_row(GtkCList *clist, gint row)
{
    gtk_object_set_data(GTK_OBJECT(clist), "data", NULL);
}

static gboolean setup_ircnets(GtkWidget *dialog)
{
    GtkWidget *hbox, *clist, *buttonbox, *button, *scrollbox;

    setup_dialog = dialog;

    temp_ircnets = NULL;
    ircnets_copy(ircnets, &temp_ircnets);

    hbox = gtk_hbox_new(FALSE, 7);
    gtk_container_border_width(GTK_CONTAINER(hbox), 7);

    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(hbox), scrollbox, TRUE, TRUE, 0);

    clist = gtk_clist_new(1);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_object_set_data(GTK_OBJECT(dialog), "ircnet_clist", clist);

    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);

    buttonbox = gtk_vbutton_box_new();
    gtk_box_pack_start(GTK_BOX(hbox), buttonbox, FALSE, TRUE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 7);

    button = gtk_button_new_with_label(_("Add"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_ircnet_add), clist);
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Edit"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_ircnet_edit), clist);
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    button = gtk_button_new_with_label(_("Remove"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_ircnet_remove), clist);
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    redraw_ircnets(GTK_CLIST(clist));

    gnome_property_box_append_page(GNOME_PROPERTY_BOX(dialog),
                                   hbox, gtk_label_new(_("IRC networks")));
    return TRUE;
}

void init_ircnets(void)
{
    signal_add("setup servers apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_add("setup servers destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_add("setup servers", (SIGNAL_FUNC) setup_ircnets);
}

void deinit_ircnets(void)
{
    signal_remove("setup servers apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_remove("setup servers destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_remove("setup servers", (SIGNAL_FUNC) setup_ircnets);
}
