/*
 panel.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "channels-newdata.h"
#include "snapshot.h"

#include <applet-widget.h>

#ifdef SIZE_TINY
/* New gnome panel can change it's size on the fly */
#  define GNOME_PANEL_NEW
#endif

#include "dialog-about.h"

#include "pixmaps/msgs.xpm"
#include "pixmaps/msgs_none.xpm"
#include "pixmaps/status.xpm"
#include "pixmaps/status_none.xpm"

enum
{
    PIXMAP_STATUS,
    PIXMAP_STATUS_NONE,
    PIXMAP_MSGS,
    PIXMAP_MSGS_NONE
};

static GtkWidget *paneltable;
static gint panelchannels;
static gboolean panel_vertical;
static gint panel_size, panel_rows;
static gboolean panel_pixmaps;

static gboolean panel_started = FALSE;

#define CHANNEL_SHOW_IN_PANEL(a) \
        ((a)->type == CHANNEL_TYPE_CHANNEL || \
         (a)->type == CHANNEL_TYPE_QUERY || \
         (a)->type == CHANNEL_TYPE_DCC_CHAT)

static GtkWidget *create_label(CHANNEL_REC *channel)
{
    GtkWidget *label, *event;
    GtkStyle *style;

    g_return_val_if_fail(channel != NULL, NULL);

    label = gtk_label_new(channel->name);
    if (panel_vertical)
        gtk_widget_set_usize(label, panel_size-4, 0);

    style = gtk_style_copy(label->style);
    gtk_style_ref(style);
    style->font = gdk_font_load("-adobe-helvetica-medium-r-normal--*-80-*-*-*-*-*-*");
    gtk_widget_set_style(label, style);
    gtk_style_unref(style);

    newdata_label_colorify(label, channel->new_data);

    event = gtk_event_box_new();
    gtk_signal_connect (GTK_OBJECT (event), "enter_notify_event",
                        GTK_SIGNAL_FUNC(snapshot_sig_show_channel), channel);
    gtk_signal_connect (GTK_OBJECT (event), "leave_notify_event",
                        GTK_SIGNAL_FUNC(snapshot_sig_hide_channel), channel);
    gtk_signal_connect (GTK_OBJECT (event), "button_press_event",
                        GTK_SIGNAL_FUNC(snapshot_sig_open_channel), channel);
    gtk_container_add(GTK_CONTAINER(event), label);

    gtk_widget_show_all(event);

    return event;
}

static void get_table_pos(gint position, gint *x1, gint *x2, gint *y)
{
    g_return_if_fail(x1 != NULL);
    g_return_if_fail(x2 != NULL);
    g_return_if_fail(y != NULL);

    if (panel_vertical)
    {
        *x1 = 0; *x2 = 2;
        *y = position;
        return;
    }

    if (panel_size <= 24)
    {
        position--;
        *x1 = 1+position/2; *x2 = *x1+1;
        *y = position & 1;
    }
    else
    {
        if (position <= panel_rows-(panel_pixmaps ? 1 : 0))
        {
            /* below the buttons */
            *x1 = 0; *x2 = panel_pixmaps ? 2 : 1;
            *y = position-(panel_pixmaps ? 0 : 1);
        }
        else
        {
            /* next column */
            position -= panel_rows+(panel_pixmaps ? 0 : 1);
            *x1 = position/panel_rows+2; *x2 = *x1+1;
            *y = position%panel_rows;
        }
    }
}

static gboolean panel_add_channel(CHANNEL_REC *channel)
{
    GUI_CHANNEL_REC *gui;
    gint x1, x2, y, panelpos, panel_max_channels;
    GList *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    panel_max_channels = setup_get_int("panel_max_channels");
    if (panel_max_channels == 0 || !CHANNEL_SHOW_IN_PANEL(channel))
        return TRUE;

    gui = CHANNEL_GUI(channel);
    if (gui->panellabel != NULL)
        gtk_widget_destroy(gui->panellabel);
    gui->panellabel = create_label(channel);

    panelpos = gui->panelpos != 0 ? gui->panelpos : panel_max_channels;
    gui->panelpos = 0;

    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if (CHANNEL_GUI(rec)->panelpos > 0 && (panelpos < 0 || panelpos < 0 || CHANNEL_GUI(rec)->panelpos <= panelpos))
        {
            gtk_widget_destroy(CHANNEL_GUI(rec)->panellabel);
            if (++CHANNEL_GUI(rec)->panelpos > panel_max_channels && panel_max_channels > -1)
            {
                CHANNEL_GUI(rec)->panelpos = 0;
                CHANNEL_GUI(rec)->panellabel = NULL;
            }
            else
            {
                CHANNEL_GUI(rec)->panellabel = create_label(rec);
                get_table_pos(CHANNEL_GUI(rec)->panelpos, &x1, &x2, &y);
                gtk_table_attach(GTK_TABLE(paneltable), CHANNEL_GUI(rec)->panellabel, x1, x2, y, y+1, 0, 0, 3, 0);
            }
        }
    }
    if (panel_max_channels < 0 || panelchannels < panel_max_channels) panelchannels++;
    gui->panelpos = 1;

    get_table_pos(1, &x1, &x2, &y);
    gtk_table_attach(GTK_TABLE(paneltable), gui->panellabel, x1, x2, y, y+1, 0, 0, 3, 0);

    return TRUE;
}

static gboolean panel_remove_channel(CHANNEL_REC *channel)
{
    GUI_CHANNEL_REC *gui;
    GList *tmp;
    gint pos, x1, x2, y;

    g_return_val_if_fail(channel != NULL, FALSE);

    gui = CHANNEL_GUI(channel);
    if (gui->panelframe != NULL)
    {
        gtk_widget_destroy(gui->panelframe);
        gui->panelframe = NULL;
        return TRUE;
    }
    if (gui->panellabel == NULL) return TRUE;

    pos = gui->panelpos;

    gtk_widget_destroy(gui->panellabel);
    gui->panellabel = NULL;
    gui->panelpos = 0;

    /* redraw the panel table */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if (CHANNEL_GUI(rec)->panelpos > pos)
        {
            gtk_widget_destroy(CHANNEL_GUI(rec)->panellabel);
            CHANNEL_GUI(rec)->panellabel = create_label(rec);
            CHANNEL_GUI(rec)->panelpos--;
            get_table_pos(CHANNEL_GUI(rec)->panelpos, &x1, &x2, &y);
            gtk_table_attach(GTK_TABLE(paneltable), CHANNEL_GUI(rec)->panellabel, x1, x2, y, y+1, 0, 0, 3, 0);
        }
        else if (panelchannels < setup_get_int("panel_max_channels") && CHANNEL_GUI(rec)->panelpos == 0 && rec != channel && CHANNEL_SHOW_IN_PANEL(rec))
        {
            CHANNEL_GUI(rec)->panellabel = create_label(rec);
            CHANNEL_GUI(rec)->panelpos = panelchannels;
            get_table_pos(CHANNEL_GUI(rec)->panelpos, &x1, &x2, &y);
            gtk_table_attach(GTK_TABLE(paneltable), CHANNEL_GUI(rec)->panellabel, x1, x2, y, y+1, 0, 0, 3, 0);
            panelchannels++;
        }
    }

    gtk_widget_queue_resize(paneltable);
    return TRUE;
}

static void panel_hilight_channel(CHANNEL_REC *channel)
{
    GUI_CHANNEL_REC *gui;
    GtkWidget *label;

    g_return_if_fail(channel != NULL);

    gui = CHANNEL_GUI(channel);
    if (gui->panelframe != NULL)
    {
        GtkWidget *pixwidget;
        GdkPixmap *pixmap;
        GdkBitmap *mask;
        gchar **xpm;

        /* remove old pixmap */
        pixwidget = gtk_container_children(GTK_CONTAINER(gui->panelframe))->data;
        gtk_widget_destroy(pixwidget);

        xpm = NULL;
        if (channel->level & MSGLEVEL_MSGS)
        {
            /* msgs channel */
            xpm = channel->new_data ? msgs_xpm : msgs_none_xpm;
        }
        if (channel->level & (MSGLEVEL_ALL ^ MSGLEVEL_MSGS))
        {
            /* status channel */
            xpm = channel->new_data ? status_xpm : status_none_xpm;
        }

        /* create new pixmap */
        pixmap = gdk_pixmap_create_from_xpm_d(gui->panelframe->window, &mask, NULL, xpm);
        pixwidget = gtk_pixmap_new(pixmap, mask);
        gtk_widget_show(pixwidget);
        gtk_container_add(GTK_CONTAINER(gui->panelframe), pixwidget);
    }
    else if (gui->panellabel != NULL)
    {
        label = gtk_container_children(GTK_CONTAINER(gui->panellabel))->data;
        newdata_label_colorify(label, channel->new_data);
    }
}

static void panel_add_pixmap(CHANNEL_REC *channel, gint pos, gint pixmap, GtkWidget *table)
{
    GtkWidget *frame, *pixwidget, *event;
    gchar **xpm;

    GdkPixmap *pix;
    GdkBitmap *mask;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(table != NULL);

    event = gtk_event_box_new();
    if (channel != NULL)
    {
        gtk_signal_connect (GTK_OBJECT (event), "enter_notify_event",
                            GTK_SIGNAL_FUNC(snapshot_sig_show_channel), channel);
        gtk_signal_connect (GTK_OBJECT (event), "leave_notify_event",
                            GTK_SIGNAL_FUNC(snapshot_sig_hide_channel), channel);
        gtk_signal_connect (GTK_OBJECT (event), "button_press_event",
                            GTK_SIGNAL_FUNC(snapshot_sig_open_channel), channel);
    }
    if (panel_size <= 24 && !panel_vertical)
        gtk_table_attach(GTK_TABLE(table), event, 0, 1, pos, pos+1, 0, 0, 0, 0);
    else
        gtk_table_attach(GTK_TABLE(table), event, pos, pos+1, 0, 1, 0, 0, 0, 0);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
    gtk_container_add(GTK_CONTAINER(event), frame);

    switch (pixmap)
    {
        case PIXMAP_STATUS:
            xpm = status_xpm;
            break;
        case PIXMAP_STATUS_NONE:
            xpm = status_none_xpm;
            break;
        case PIXMAP_MSGS:
            xpm = msgs_xpm;
            break;
        case PIXMAP_MSGS_NONE:
            xpm = msgs_none_xpm;
            break;
        default:
            xpm = NULL;
            g_warning("panel_add_pixmap() called with unknown pixmap");
            break;
    }

    gtk_widget_realize(event);
    pix = gdk_pixmap_create_from_xpm_d(event->window, &mask, NULL, xpm);
    pixwidget = gtk_pixmap_new(pix, mask);
    gtk_container_add(GTK_CONTAINER(frame), pixwidget);

    if (channel != NULL) CHANNEL_GUI(channel)->panelframe = frame;
    gtk_widget_show_all(event);
}

static void panel_redraw(GtkWidget *applet)
{
    GList *tmp;

    if (panel_vertical)
        gtk_widget_set_usize(applet, panel_size, 0);
    else
        gtk_widget_set_usize(applet, -1, panel_size);

    /* first check for status/msgs windows */
    panel_pixmaps = FALSE;
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if (rec->level & (MSGLEVEL_ALL ^ MSGLEVEL_MSGS))
        {
            if (CHANNEL_GUI(rec)->panelframe != NULL) gtk_widget_destroy(CHANNEL_GUI(rec)->panelframe);
            panel_add_pixmap(rec, 1, PIXMAP_STATUS_NONE, paneltable);
            panel_pixmaps = TRUE;
        }
        else if (rec->level & MSGLEVEL_MSGS)
        {
            if (CHANNEL_GUI(rec)->panelframe != NULL)
                gtk_widget_destroy(CHANNEL_GUI(rec)->panelframe);
            panel_add_pixmap(rec, 0, PIXMAP_MSGS_NONE, paneltable);
            panel_pixmaps = TRUE;
        }
    }

    /* then the channels.. */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if (CHANNEL_GUI(rec)->panellabel != NULL)
        {
            gtk_widget_destroy(CHANNEL_GUI(rec)->panellabel);
            CHANNEL_GUI(rec)->panellabel = NULL;
            CHANNEL_GUI(rec)->panelpos = 0;
            panel_add_channel(rec);
        }
    }
}

static gint applet_destroy(GtkWidget *widget, gpointer data)
{
    signal_emit("command quit", 1, "");
    return FALSE;
}

static void applet_change_orient(GtkWidget *applet, PanelOrientType orient)
{
    panel_vertical = orient == ORIENT_LEFT || orient == ORIENT_RIGHT;

    panel_redraw(applet);
}

#ifdef GNOME_PANEL_NEW
static void applet_change_size(GtkWidget *applet, PanelSizeType size)
{
    switch (size)
    {
        case SIZE_TINY:
            panel_size = 24;
            panel_rows = 2;
            break;
        case SIZE_STANDARD:
            panel_size = 48;
            panel_rows = 4;
            break;
        case SIZE_LARGE:
            panel_size = 64;
            panel_rows = 5;
            break;
        case SIZE_HUGE:
            panel_size = 80;
            panel_rows = 6;
            break;
    }

    panel_redraw(applet);
}
#endif

static void panel_applet_create(void)
{
    GtkWidget *applet, *frame;

    panelchannels = 0;
    applet = applet_widget_new(PACKAGE);
    if (applet == NULL)
        g_error("Can't create applet, try --no-applet option.\n");

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    applet_widget_add(APPLET_WIDGET(applet), frame);

    gtk_signal_connect(GTK_OBJECT(applet), "destroy",
                       GTK_SIGNAL_FUNC(applet_destroy), NULL);
    gtk_signal_connect(GTK_OBJECT(applet), "change_orient",
		       GTK_SIGNAL_FUNC(applet_change_orient), NULL);
#ifdef GNOME_PANEL_NEW
    gtk_signal_connect(GTK_OBJECT(applet), "change_size",
		       GTK_SIGNAL_FUNC(applet_change_size), NULL);
#endif

    paneltable = gtk_table_new(4, 2, FALSE);
    gtk_container_add(GTK_CONTAINER(frame), paneltable);

    applet_widget_register_stock_callback(APPLET_WIDGET(applet),
                                          "about",
                                          GNOME_STOCK_MENU_ABOUT,
                                          _("About"),
                                          (AppletCallbackFunc) dialog_about,
                                          NULL);

    gtk_widget_show_all(applet);
}

static void panel_select_channel(CHANNEL_REC *channel)
{
    if (setup_get_int("panel_max_channels") > 0 && panelchannels > setup_get_int("panel_max_channels") &&
        channel->type != CHANNEL_TYPE_EMPTY)
    {
        /* move to first position */
        if (CHANNEL_GUI(channel)->panelpos > 0) panel_remove_channel(channel);
        panel_add_channel(channel);
    }
}

static gboolean sig_channel_color(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    panel_hilight_channel(channel);
    if (channel->new_data == 0)
        panel_select_channel(channel);
    return TRUE;
}

void panel_init(void)
{
    panel_size = 48;
    panel_rows = 4;

    panel_started = TRUE;
    signal_add("gui hilight channel", (SIGNAL_FUNC) sig_channel_color);
    signal_add("gui channel created", (SIGNAL_FUNC) panel_add_channel);
    signal_add("channel destroyed", (SIGNAL_FUNC) panel_remove_channel);

    panel_applet_create();
}

void panel_deinit(void)
{
    if (panel_started)
    {
        signal_remove("gui hilight channel", (SIGNAL_FUNC) sig_channel_color);
        signal_remove("gui channel created", (SIGNAL_FUNC) panel_add_channel);
        signal_remove("channel destroyed", (SIGNAL_FUNC) panel_remove_channel);
    }
}
