/*
 gui-events-whois.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *wi_dialogs;

static void gui_whois_show(SERVER_REC *server)
{
    GtkWidget *box, *list, *scrollbox, *label, *sep;
    GUI_SERVER_REC *gui;

    g_return_if_fail(server != NULL);

    gui = SERVER_GUI(server);
    if (gui->listdialog == NULL) return;
    box = gtk_object_get_data(GTK_OBJECT(gui->listdialog), "box");

    /* separator */
    sep = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(box), sep, FALSE, FALSE, 5);

    /* "Channels" text */
    label = gtk_label_new(_("Channels"));
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    /* display the channel list */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(box), scrollbox, TRUE, TRUE, 0);

    list = gtk_object_get_data(GTK_OBJECT(gui->listdialog), "chanlist");
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollbox), list);

    /* show dialog */
    gtk_widget_show_all(gui->listdialog);
    gui->listdialog = NULL;
}

static void sig_refresh(GtkWidget *dialog)
{
    gchar *servertag, *nick;
    SERVER_REC *server;

    servertag = gtk_object_get_data(GTK_OBJECT(dialog), "server");
    nick = gtk_object_get_data(GTK_OBJECT(dialog), "nick");

    server = server_find_tag(servertag);
    if (server != NULL)
        signal_emit("command gwhois", 3, nick, server, cur_channel);
}

static void sig_destroy(GtkWidget *dialog)
{
    wi_dialogs = g_list_remove(wi_dialogs, dialog);
}

static gboolean event_whois(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *user, *host, *realname, *str;
    GtkWidget *dialog, *box, *label, *sep;
    GUI_SERVER_REC *gui;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 6, NULL, &nick, &user, &host, NULL, &realname);

    gui = SERVER_GUI(server);
    if (gui->listdialog != NULL)
    {
	/* whois with multiple nicks - show the last one. */
	gui_whois_show(server);
    }

    /* check if there's already dialog open for this nick */
    dialog = NULL;
    for (tmp = wi_dialogs; tmp != NULL; tmp = tmp->next)
    {
        GtkObject *object = tmp->data;

        if (strcmp(gtk_object_get_data(object, "server"), server->tag) == 0 &&
            strcmp(gtk_object_get_data(object, "nick"), nick) == 0)
        {
            dialog = GTK_WIDGET(object);
	    gdk_window_raise(dialog->window);
            break;
        }
    }

    if (dialog != NULL)
    {
        box = gtk_object_get_data(GTK_OBJECT(dialog), "box");
        gtk_widget_destroy(box);
        gui->listdialog = dialog;
        gtk_object_set_data(GTK_OBJECT(gui->listdialog), "chanlist", NULL);
    }
    else
    {
        /* create dialog */
        nick = g_strdup(nick);

        gui->listdialog = gnome_dialog_new(PACKAGE, _("Refresh"), GNOME_STOCK_BUTTON_CLOSE, NULL);
	gtk_window_set_policy(GTK_WINDOW(gui->listdialog), FALSE, TRUE, FALSE);
        wi_dialogs = g_list_append(wi_dialogs, gui->listdialog);
        gtk_signal_connect(GTK_OBJECT(gui->listdialog), "destroy",
                           GTK_SIGNAL_FUNC(sig_destroy), NULL);
        gtk_signal_connect_object(GTK_OBJECT(gui->listdialog), "destroy",
                                  GTK_SIGNAL_FUNC(g_free), (GtkObject *) nick);
        gtk_signal_connect_object(GTK_OBJECT(gui->listdialog), "delete_event",
                                  GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(gui->listdialog));
        gui_widget_depends_data(gui->listdialog, "server disconnected", server);
        gnome_dialog_button_connect_object(GNOME_DIALOG(gui->listdialog), 0, GTK_SIGNAL_FUNC(sig_refresh), GTK_OBJECT(gui->listdialog));
        gnome_dialog_button_connect_object(GNOME_DIALOG(gui->listdialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(gui->listdialog));

        gtk_object_set_data(GTK_OBJECT(gui->listdialog), "server", server->tag);
        gtk_object_set_data(GTK_OBJECT(gui->listdialog), "nick", nick);
    }

    box = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(gui->listdialog)->vbox), box, TRUE, TRUE, 0);
    gtk_object_set_data(GTK_OBJECT(gui->listdialog), "box", box);

    /* Display realname */
    str = g_strdup_printf(_("Name: %s (%s)"), nick, realname);
    label = gtk_label_new(str); g_free(str);
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    /* Display host */
    str = g_strdup_printf(_("Host: %s@%s"), user, host);
    label = gtk_label_new(str); g_free(str);
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    /* separator */
    sep = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(box), sep, FALSE, FALSE, 5);

    g_free(params);
    return TRUE;
}

static gboolean event_whois_idle(gchar *data, SERVER_REC *server)
{
    GtkWidget *label, *box;
    gchar *params, *nick, *secstr, *signon, *rest;
    GString *str;
    glong secs;
    gint h, m, s;
    time_t t;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 5 | PARAM_FLAG_GETREST, NULL, &nick, &secstr, &signon, &rest);
    if (sscanf(secstr, "%ld", &secs) == 0) secs = 0;
    t = 0;
    if (strstr(rest, ", signon time") != NULL)
        sscanf(signon, "%ld", (glong *) &t);

    h = secs/3600; m = (secs%3600)/60; s = secs%60;

    str = g_string_new(NULL);
    g_string_sprintf(str, _("Idling: %d hours %02d:%02d"), h, m, s);

    if (t != 0)
    {
        gchar *timestr;
        struct tm *tim;

        tim = localtime(&t);
        timestr = g_strdup(asctime(tim));
        if (timestr[strlen(timestr)-1] == '\n') timestr[strlen(timestr)-1] = '\0';

        g_string_sprintfa(str, _("\nSignon: %s"), timestr);
        g_free(timestr);
    }

    label = gtk_label_new(str->str);
    box = gtk_object_get_data(GTK_OBJECT(SERVER_GUI(server)->listdialog), "box");
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    g_free(params);
    return TRUE;
}

static gboolean event_whois_server(gchar *data, SERVER_REC *server)
{
    GtkWidget *label, *box;
    gchar *params, *nick, *whoserver, *desc, *str;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 4, NULL, &nick, &whoserver, &desc);

    str = g_strdup_printf(_("Server: %s (%s)"), whoserver, desc);
    label = gtk_label_new(str); g_free(str);
    box = gtk_object_get_data(GTK_OBJECT(SERVER_GUI(server)->listdialog), "box");
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    g_free(params);
    return TRUE;
}

static gboolean event_whois_oper(gchar *data, SERVER_REC *server)
{
    GtkWidget *label, *box;
    gchar *params, *nick;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &nick);

    label = gtk_label_new(_("[IRC operator]"));
    box = gtk_object_get_data(GTK_OBJECT(SERVER_GUI(server)->listdialog), "box");
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    g_free(params);
    return TRUE;
}

static gboolean event_whois_channels(gchar *data, SERVER_REC *server)
{
    GtkWidget *list, *li;
    gchar *params, *nick, *chans;
    GList *chlist, *tmp, *listitems;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, NULL, &nick, &chans);

    list = gtk_object_get_data(GTK_OBJECT(SERVER_GUI(server)->listdialog), "chanlist");
    if (list == NULL)
    {
	/* create the channel list widget */
	list = gtk_list_new();
	gtk_object_set_data(GTK_OBJECT(SERVER_GUI(server)->listdialog), "chanlist", list);
    }

    /* Add channels to GtkList */
    chlist = str2list(chans, ' ');
    listitems = NULL;
    for (tmp = chlist; tmp != NULL; tmp = tmp->next)
    {
	li = gtk_list_item_new_with_label(tmp->data);
	listitems = g_list_append(listitems, li);
    }
    gtk_list_append_items(GTK_LIST(list), listitems);

    g_free(chlist->data);
    g_list_free(chlist);

    g_free(params);
    return TRUE;
}

static gboolean event_whois_away(gchar *data, SERVER_REC *server)
{
    GtkWidget *label, *box;
    gchar *params, *nick, *awaymsg, *str;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, NULL, &nick, &awaymsg);

    str = g_strdup_printf(_("Away: %s"), awaymsg);
    label = gtk_label_new(str); g_free(str);
    box = gtk_object_get_data(GTK_OBJECT(SERVER_GUI(server)->listdialog), "box");
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_whois(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    gui_whois_show(server);
    return TRUE;
}

static gboolean cmd_whois(gchar *data, SERVER_REC *server)
{
    gchar *params, *qserver, *query;
    gchar *p, *str;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2, &qserver, &query);
    if (*qserver == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*query == '\0')
    {
	/* just query was given.. */
	query = qserver;
	str = g_strdup_printf("WHOIS %s", query);
    }
    else
    {
	/* server+query */
	str = g_strdup_printf("WHOIS %s %s", qserver, query);
    }

    irc_send_cmd(server, str);
    g_free(str);

    /* convert dots to spaces */
    for (p = query; *p != '\0'; p++)
	if (*p == ',') *p = ' ';

    server_redirect_event(server, query, 2,
			  "event 318", "gui event 318", -1,
			  "event 402", "gui event 402", -1,
			  "event 401", "gui event 401", 1,
			  "event 311", "gui event 311", 1,
			  "event 301", "gui event 301", 1,
			  "event 312", "gui event 312", 1,
			  "event 313", "gui event 313", 1,
			  "event 317", "gui event 317", 1,
			  "event 319", "gui event 319", 1, NULL);
    g_free(params);
    return TRUE;
}

void gui_event_whois_init(void)
{
    wi_dialogs = NULL;
    command_bind("gwhois", "GUI commands", (SIGNAL_FUNC) cmd_whois);

    signal_add("gui event 311", (SIGNAL_FUNC) event_whois);
    signal_add("gui event 301", (SIGNAL_FUNC) event_whois_away);
    signal_add("gui event 312", (SIGNAL_FUNC) event_whois_server);
    signal_add("gui event 313", (SIGNAL_FUNC) event_whois_oper);
    signal_add("gui event 317", (SIGNAL_FUNC) event_whois_idle);
    signal_add("gui event 318", (SIGNAL_FUNC) event_end_of_whois);
    signal_add("gui event 319", (SIGNAL_FUNC) event_whois_channels);
}

void gui_event_whois_deinit(void)
{
    g_list_free(wi_dialogs);
    command_unbind("gwhois", (SIGNAL_FUNC) cmd_whois);

    signal_remove("gui event 311", (SIGNAL_FUNC) event_whois);
    signal_remove("gui event 301", (SIGNAL_FUNC) event_whois_away);
    signal_remove("gui event 312", (SIGNAL_FUNC) event_whois_server);
    signal_remove("gui event 313", (SIGNAL_FUNC) event_whois_oper);
    signal_remove("gui event 317", (SIGNAL_FUNC) event_whois_idle);
    signal_remove("gui event 318", (SIGNAL_FUNC) event_end_of_whois);
    signal_remove("gui event 319", (SIGNAL_FUNC) event_whois_channels);
}
