/*
 dialog-rawlog.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static GtkWidget *rawlog_text, *rawlog_menu;
static SERVER_REC *rawlog_server;

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    gui_servermenu_destroy(rawlog_menu);
    *dialog = NULL;
    rawlog_text = NULL;
    rawlog_menu = NULL;
}

static void redraw_rawlog(void)
{
    GtkAdjustment *adj;
    GdkColor *fg;
    GList *tmp;

    if (g_list_find(servers, rawlog_server) == NULL)
    {
	/* this server is disconnected.. */
	return;
    }

    gtk_text_freeze(GTK_TEXT(rawlog_text));

    /* Clear old text */
    gtk_text_set_point(GTK_TEXT(rawlog_text), 0);
    gtk_text_forward_delete(GTK_TEXT(rawlog_text), gtk_text_get_length(GTK_TEXT(rawlog_text)));

    /* Draw new */
    for (tmp = rawlog_server->rawlog; tmp != NULL; tmp = tmp->next)
    {
	gchar *str = tmp->data;

	fg =
	    *str == '>' ? NULL :
	    *str == '-' ? &THEME_GUI(current_theme)->colors[1] :
	    &THEME_GUI(current_theme)->colors[4];
	gtk_text_insert(GTK_TEXT(rawlog_text), NULL, fg, NULL, str, -1);
	gtk_text_insert(GTK_TEXT(rawlog_text), NULL, NULL, NULL, "\n", 1);
    }
    gtk_text_thaw(GTK_TEXT(rawlog_text));

    adj = GTK_TEXT(rawlog_text)->vadj;
    gtk_adjustment_set_value(adj, adj->upper - adj->lower - adj->page_size);
}

static void sig_save_ok(GtkWidget *dialog)
{
    gchar *fname;

    if (g_list_find(servers, rawlog_server) == NULL)
    {
	/* this server is disconnected.. */
	gui_dialog(DIALOG_ERROR, _("Can't save the rawlog,\nthe server has been disconnected"));
	return;
    }

    fname = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "entry")));
    signal_emit("command rawlog", 3, fname, rawlog_server, cur_channel);
    gtk_widget_destroy(dialog);
}

/* Save lastlog buffer to file */
static void sig_save_rawlog(void)
{
    GtkWidget *dialog, *hbox, *label, *entry;

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK,
			      GNOME_STOCK_BUTTON_CANCEL, NULL);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    label = gtk_label_new(_("Save rawlog to: "));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gnome_file_entry_new("rawlog", _("Save"));
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, sig_save_ok, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_object_set_data(GTK_OBJECT(dialog), "entry", gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry)));

    gtk_widget_show_all(dialog);
}

void dialog_rawlog(void)
{
    static GtkWidget *dialog = NULL;
    GtkWidget *hbox, *scrollbox, *button;

    if (dialog != NULL)
    {
        /* dialog already open */
	gdk_window_raise(dialog->window);
	return;
    }

    if (cur_channel->server == NULL)
    {
        gui_dialog(DIALOG_ERROR, _("Not connected to server"));
        return;
    }

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, 500, 350);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    /* Display rawlog in GtkText */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    rawlog_text = gtk_text_new(NULL, NULL);
    gtk_container_add(GTK_CONTAINER(scrollbox), rawlog_text);

    /* Draw all rawlog texts.. */
    rawlog_server = cur_channel->server;
    redraw_rawlog();

    /* Create server selection menu */
    hbox = gtk_hbox_new(FALSE, 10);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, FALSE, TRUE, 0);

    rawlog_menu = gtk_option_menu_new();
    gtk_widget_set_usize(rawlog_menu, 100, -1);
    gtk_box_pack_start(GTK_BOX(hbox), rawlog_menu, FALSE, FALSE, 0);

    gui_servermenu_create(rawlog_menu, NULL);
    gtk_option_menu_set_history(GTK_OPTION_MENU(rawlog_menu), g_list_index(servers, rawlog_server));

    /* Save rawlog button */
    button = gtk_button_new_with_label(_("Save to file..."));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_save_rawlog), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_widget_show_all(dialog);
}

static gboolean sig_rawlog(SERVER_REC *server, gchar *str)
{
    GdkColor *fg;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(str != NULL, FALSE);

    if (rawlog_text != NULL && server == rawlog_server)
    {
	fg =
	    *str == '>' ? NULL :
	    *str == '-' ? &THEME_GUI(current_theme)->colors[1] :
	    &THEME_GUI(current_theme)->colors[4];
	gtk_text_insert(GTK_TEXT(rawlog_text), NULL, fg, NULL, str, -1);
	gtk_text_insert(GTK_TEXT(rawlog_text), NULL, NULL, NULL, "\n", 1);
    }

    return TRUE;
}

static gboolean sig_servermenu_changed(GtkWidget *menu, SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);
    if (menu != rawlog_menu) return TRUE;

    rawlog_server = server;
    redraw_rawlog();
    return TRUE;
}

void dialog_rawlog_init(void)
{
    rawlog_text = NULL;
    signal_add("rawlog", (SIGNAL_FUNC) sig_rawlog);
    signal_add("servermenu changed", (SIGNAL_FUNC) sig_servermenu_changed);
}

void dialog_rawlog_deinit(void)
{
    signal_remove("rawlog", (SIGNAL_FUNC) sig_rawlog);
    signal_remove("servermenu changed", (SIGNAL_FUNC) sig_servermenu_changed);
}
