/*
 *  Copyright (C) 1999 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "download-info.h"
#include "main-window.h"
#include "wget-log.h"

/* Function that returns the state name */
gchar *
file_data_state_name (FileData *file_data)
{
    switch (file_data->state) {
	case DL_NOT_STARTED:
	    return _ ("Not Started");
	    break;
	case DL_NOT_RUNNING:
	    return _ ("Not Running");
	    break;
	case DL_NOT_CONNECTED:
	    return _ ("Not Connected");
	    break;
	case DL_CONNECTED:
	    return _ ("Connected");
	    break;
	case DL_RETRIEVING:
	    return _ ("Retrieving");
	    break;
	case DL_COMPLETED:
	    return _ ("Completed");
	    break;
    }
    return "";
}

void
file_data_free (FileData *file_data)
{
    FileData *aux_file_data;
    gchar *path, **files;
    gint i;

    if (file_data != NULL) {
        if (file_data->file_list != NULL) {
            files = g_new0 (gchar *, GTK_CLIST (file_data->file_list)->rows);
            for (i = 0; i < GTK_CLIST (file_data->file_list)->rows; i++) {
                aux_file_data =
                    gtk_clist_get_row_data (GTK_CLIST (file_data->file_list),
                                            i);
                files[i] = g_strdup_printf ("%d", aux_file_data->id);
            }
            gnome_config_set_vector ("/gtm/FileList/file_ids",
                                     GTK_CLIST (file_data->file_list)->rows,
                                     (const gchar **) files);
            for (i = 0; i < GTK_CLIST (file_data->file_list)->rows; i++)
                g_free (files[i]);
            g_free (files);
        }
        path = g_strdup_printf ("/gtm/File%d/", file_data->id);
        gnome_config_clean_section (path);
        gnome_config_sync ();
        g_free (path);
	g_free (file_data->url);
	g_free (file_data->dir);
	g_free (file_data->filename);
        g_free (file_data->local_filename);
	g_free (file_data->line);
	g_free (file_data);
    }
}

void
file_data_set_state (FileData *file_data, DlState state)
{
    gint row;
    gchar *path;

    file_data->state = state;
    if (state == DL_NOT_STARTED || state == DL_NOT_RUNNING
        || state == DL_COMPLETED) {
        path = g_strdup_printf ("/gtm/File%d/state", file_data->id);
        gnome_config_set_int (path, file_data->state);
        gnome_config_sync ();
        g_free (path);
    }
    if (state == DL_COMPLETED)
        report_event ("complete");
    if (file_data->file_list != NULL) {
        row = gtk_clist_find_row_from_data (GTK_CLIST (file_data->file_list),
                                            file_data);
        file_list_set_row_state (GTK_CLIST (file_data->file_list), row, state);
        menu_toolbar_set_state (GTK_CLIST (file_data->file_list));
        file_list_menu_popup_set_state (GTK_CLIST (file_data->file_list));
        if (row == GTK_CLIST (file_data->file_list)->focus_row
            && g_list_find (GTK_CLIST (file_data->file_list)->selection,
                            (gpointer) row) != NULL)
            download_info_update_state (file_data);
    }
}

void
file_data_set_total_size (FileData *file_data, guint32 total_size)
{
    gint row;
    gchar *path;

    file_data->total_size = total_size;
    path = g_strdup_printf ("/gtm/File%d/size", file_data->id);
    gnome_config_set_int (path, file_data->total_size);
    gnome_config_sync ();
    g_free (path);
    if (file_data->file_list != NULL) {
        row = gtk_clist_find_row_from_data (GTK_CLIST (file_data->file_list),
                                            file_data);
        if (row == GTK_CLIST (file_data->file_list)->focus_row
            && g_list_find (GTK_CLIST (file_data->file_list)->selection,
                            (gpointer) row) != NULL)
            download_info_update_total_size (file_data);
    }
}

void file_data_set_url (FileData *file_data, gchar *url)
{
    gchar *path;

    /* Before changing the URL we must stop the download */
    file_data_stop_download (file_data);
    g_free (file_data->url);
    file_data->url = g_strdup (url);
    path = g_strdup_printf ("/gtm/File%d/url",
                            file_data->id);
    gnome_config_set_string (path, file_data->url);
    gnome_config_sync ();
    g_free (path);
}

void file_data_set_disable_proxy (FileData *file_data,
                                  gboolean disable_proxy)
{
    gchar *path;

    file_data->disable_proxy = disable_proxy;
    path = g_strdup_printf ("/gtm/File%d/disable_proxy",
                            file_data->id);
    gnome_config_set_bool (path, file_data->disable_proxy);
    gnome_config_sync ();
    g_free (path);
}

void file_data_set_disable_auto_dl (FileData *file_data,
                                    gboolean disable_auto_dl)
{
    gchar *path;

    file_data->disable_auto_dl = disable_auto_dl;
    path = g_strdup_printf ("/gtm/File%d/disable_auto_Dl",
                            file_data->id);
    gnome_config_set_bool (path, file_data->disable_auto_dl);
    gnome_config_sync ();
    g_free (path);
}

void
file_data_update_statistics (FileData *file_data)
{
    gint row;
    guint32 cur_size;
    time_t cur_time, elapsed_time;
    struct stat file_stat;
    gchar *path;

    /* Get time and size of the file being retrieved */
    if (stat (file_data->local_filename, &file_stat) != -1) {
	cur_size = file_stat.st_size;
	cur_time = file_stat.st_ctime;
    } else {
	cur_size = 0;
	cur_time = 0;
    }
    file_data->cur_size = cur_size;

    /* Set the total and session time */
    if (file_data->state != DL_NOT_STARTED 
	&& file_data->state != DL_NOT_RUNNING 
	&& file_data->state != DL_COMPLETED) {
	elapsed_time = cur_time - file_data->session_start_time;
	file_data->total_time += elapsed_time - file_data->session_elapsed;
	file_data->session_elapsed = elapsed_time;
    } else
	file_data->session_elapsed = 0;
    path = g_strdup_printf ("/gtm/File%d/time", file_data->id);
    gnome_config_set_int (path, file_data->total_time);
    gnome_config_sync ();
    g_free (path);

    if (file_data->file_list != NULL) {
        row = gtk_clist_find_row_from_data (GTK_CLIST (file_data->file_list), 
                                            file_data);
        file_list_set_row_statistics (GTK_CLIST (file_data->file_list), row, 
                                      file_data);
        if (row == GTK_CLIST (file_data->file_list)->focus_row
            && g_list_find (GTK_CLIST (file_data->file_list)->selection,
                            (gpointer) row) != NULL)
            download_info_update_statistics (file_data);
    }
}

static gint
file_data_process_information (FileData *file_data)
{
    pid_t child_pid;
    gint status;

    /* Process wget_log messages */
    wget_log_process (file_data);

    /* Check that wget process is still running */
    child_pid = waitpid (file_data->wget_pid, &status, WNOHANG | WUNTRACED);
    if (child_pid == file_data->wget_pid) {
	/* Wget process stopped so we must register it's exit */
	close (file_data->log_fd);
	g_free (file_data->line);
	file_data->line = NULL;

        /* Set the appropriate state after stopping */
	if (WIFEXITED (status)) {
            if (WEXITSTATUS (status) == 0)
                file_data_set_state (file_data, DL_COMPLETED);
            else if (WEXITSTATUS (status) == 255) {
                /* Only reaches here if wget is not found */
                file_data_set_state (file_data, DL_NOT_RUNNING);
                g_warning ("couldn't find program wget to exec\n");
            } else
                file_data_set_state (file_data, DL_NOT_RUNNING);
	} else
	    file_data_set_state (file_data, DL_NOT_RUNNING);
        file_data_update_statistics (file_data);

        /* Decrease the number of current downloads */
        if (num_of_download > 0)
            num_of_download--;

        /* All done this download can be started again */
        file_data->log_tag = -1;

	return FALSE;
    }
    return TRUE;
}

void
file_data_start_download (FileData *file_data)
{
    pid_t pid;
    gint pipe_fd[2];
    
    /* First check if we are not starting an already started download */
    if (!file_data_run (file_data) && file_data->state != DL_COMPLETED) {
        if (pipe (pipe_fd) < 0) {
            g_warning ("couldn't create wget pipe");
            return;
        }
        pid = fork ();
        if (pid == 0) {
            /* Child process */
            gint arg;
            gchar *argv[20];
            gchar *opt;

            /* Set stderr of child process to one end of the pipe. The father
             * process reads child output throught the pipe */
            close (pipe_fd[0]);
            dup2 (pipe_fd[1], 2);

            /* Set common arguments */
            argv[0] = "wget";
            argv[1] = "-v";                   /* Verbose */
            argv[2] = "-P";                   /* Use directory prefix */
            argv[3] = file_data->dir;         /* Directory prefix */
            argv[4] = file_data->url;         /* Url to download */
            argv[5] = "-c";                   /* Continue download */
            argv[6] = "-t";                   /* Number of retries */
            argv[7] = g_strdup_printf ("%d", gtm_pref.num_retries);
            arg = 8;

            /* Proxy arguments */
            argv[arg++] = "-Y";
            if (gtm_pref.use_proxy && !file_data->disable_proxy) {
                argv[arg++] = "on";

                /* Proxy user and password */
                if (gtm_pref.proxy_user[0] != '\0')
                    argv[arg++] = g_strconcat ("--proxy-user=",
                                               gtm_pref.proxy_user, 
                                               NULL);
                if (gtm_pref.proxy_pass[0] != '\0')
                    argv[arg++] = g_strconcat ("--proxy-pass=",
                                               gtm_pref.proxy_pass, 
                                               NULL);

                /* Proxy location defined with environment variables */
                if (gtm_pref.proxy_http[0] != '\0') {
                    opt = g_strconcat ("http_proxy=", gtm_pref.proxy_http,
                                       NULL);
                    putenv (opt);
                    /* opt is not freed because it is incorporated on the
                     * environment */
                }
                if (gtm_pref.proxy_http[0] != '\0') {
                    opt = g_strconcat ("ftp_proxy=", gtm_pref.proxy_http,
                                       NULL);
                    putenv (opt);
                    /* opt is not freed because it is incorporated on the
                     * environment */
                } 
            } else
                argv[arg++] = "off";

            /* FTP preferences */
            if (gtm_pref.use_passive_ftp)
                argv[arg++] = "--passive-ftp";  

            /* We must terminate the arguments with a NULL entry */
            argv[arg] = NULL;

            /* Set Language to en_US. This must be done or we will not be able
             * to parse the wget output */
            putenv ("LC_ALL=en_US");

            /* Everything ready run wget */
            execvp ("wget", argv);

            /* If we get here wget was not found so we return an error 255 */
            _exit (255);
        }
        if (pid < 0) {
            g_warning ("couldn't fork wget");
            return;
        }
        close (pipe_fd[1]);
        file_data->wget_pid = pid;
        file_data->log_fd = pipe_fd[0];
        fcntl (file_data->log_fd, F_SETFL, O_NONBLOCK);
        file_data->line = g_new (gchar, MAX_WGET_LINE_SIZE + 1);
        file_data->line_pos = 0;
        file_data_set_state (file_data, DL_NOT_CONNECTED);
        file_data->log_tag = 
            gtk_timeout_add (gtm_pref.file_list_refresh, 
                             (GtkFunction) file_data_process_information,
                             file_data);
        report_event ("start");

        /* Increase the number of downloads */
        num_of_download++;
    }
}

void
file_data_stop_download (FileData *file_data)
{
    pid_t child_pid;
    gint status;

    /* First check if we are not stopping an already stopped download */
    if (file_data_run (file_data)) {
        /* Kill wget process */
        kill (file_data->wget_pid, SIGINT);

        /* Remove callback that communicates with wget */
        gtk_timeout_remove (file_data->log_tag);

        /* Wait the finish of wget */
        child_pid = waitpid (file_data->wget_pid, &status, WUNTRACED);
        if (child_pid == file_data->wget_pid) {
            /* Process the rest of the information that wget process has */
            file_data_process_information (file_data);

            /* Register wget exit */
            close (file_data->log_fd);
            g_free (file_data->line);
            file_data->line = NULL;

            /* Set the appropriate state after stopping */
            if (WIFEXITED (status) && (WEXITSTATUS (status) == 0))
                file_data_set_state (file_data, DL_COMPLETED);
            else
                file_data_set_state (file_data, DL_NOT_RUNNING);
            file_data_update_statistics (file_data);

            /* Decrease the number of current downloads */
            if (num_of_download > 0)
                num_of_download--;

            /* Check if stop action disables auto download */
            if (gtm_pref.manual_stop_disables_auto_dl)
                file_data->disable_auto_dl = TRUE;

            /* All done this download can be started again */
            file_data->log_tag = -1;
        }
    }
}

void
file_data_restart_download (FileData *file_data)
{
    /* First stop the download, clear download info and remove downloaded 
     * file */
    file_data_stop_download (file_data);
    file_data_set_state (file_data, DL_NOT_STARTED);
    file_data->total_time = 0;
    unlink (file_data->local_filename);

    /* Now start again the download */
    file_data_start_download (file_data);
}

FileData *
file_data_create (gchar *url, gchar *dir, gboolean disable_proxy,
                  gboolean disable_auto_dl)
{
    FileData *file_data;
    gchar *filename, *prefix, *file_ids, *new_file_ids;
    gint length;
    
    file_data = g_new (FileData, 1);
    file_data->id = next_file_id++;
    file_data->log_tag = -1;
    file_data->url = g_strdup (url);

    /* If the download dir is empty use the default and if it is not null
     * make sure there is a '/' in the end of dir name */
    length = strlen (dir);
    if (length == 0)
        file_data->dir = g_strdup (gtm_pref.def_download_dir);
    else if (dir[length - 1] != '/')
        file_data->dir = g_strconcat (dir, "/", NULL);
    else
        file_data->dir = g_strdup (dir);

    /* Get the filename from the URL */
    filename = &file_data->url[strlen (file_data->url)];
    while (*filename != '/' && filename != file_data->url)
        filename--;
    filename++;
    file_data->filename = g_strdup (filename);

    file_data->local_filename = g_strconcat (file_data->dir,
                                             file_data->filename,
                                             NULL);
    file_data->disable_proxy = disable_proxy;
    file_data->disable_auto_dl = disable_auto_dl;
    file_data->line = NULL;
    file_data->session_start_time = 0;
    file_data->session_start_size = 0;
    file_data->session_elapsed = 0;
    file_data->cur_size = 0;
    file_data->state = DL_NOT_STARTED;
    file_data->total_size = 0;
    file_data->total_time = 0;
    file_data->file_list = NULL;

    /* Save the new data */
    file_ids = gnome_config_get_string ("/gtm/FileList/file_ids");
    new_file_ids = g_strdup_printf ("%s%d ", file_ids, file_data->id);
    gnome_config_set_string ("/gtm/FileList/file_ids", new_file_ids);
    gnome_config_set_int ("/gtm/FileList/next_file_id", next_file_id);
    prefix = g_strdup_printf ("/gtm/File%d/", file_data->id);
    gnome_config_push_prefix (prefix);
    gnome_config_set_string ("url", file_data->url);
    gnome_config_set_string ("dir", file_data->dir);
    gnome_config_set_bool ("disable_proxy", file_data->disable_proxy);
    gnome_config_set_bool ("disable_auto_dl",
                               file_data->disable_auto_dl);
    gnome_config_pop_prefix ();
    gnome_config_sync ();
    g_free (prefix);
    g_free (file_ids);
    g_free (new_file_ids);

    return file_data;
}
