#include <darxite.h>

/* malloc a buffer, and zero out the contents */
void *malloc0(long size)
{
	void *ptr;

	ptr = malloc(size);
	if(!ptr) return NULL;

	memset(ptr, 0, size);
	return ptr;
}

/* return a pointer to a (statically-allocated) buffer containing
   the string representation of n */
char *itoa(int n)
{
	static char buf[1024];

	sprintf(buf, "%i", n);

	return buf;
}

void stripcrlf(char *s)
{
	while(*s && index("\n\r", s[strlen(s)-1]))
		s[strlen(s)-1] = 0;
}

void stripl(char *s)
{
	while(s && *s && index(" \t", *s)) memmove(s, s+1, strlen(s));
}

void stripr(char *s)
{
	while(*s && index(" \t", s[strlen(s)-1]))
		s[strlen(s)-1] = 0;
}

void strip(char *s)
{
	stripcrlf(s);
	stripl(s);
	stripr(s);
}

/* insert s at the beginning of buf */
void strinss(char *buf, const char *s)
{
	memmove(buf+strlen(s), buf, strlen(buf)+1);
	memcpy(buf, s, strlen(s));
}

/* delete a character from buf */
void strdelch(char *buf)
{
	if(!strlen(buf)) return;

	memmove(buf, buf+1, strlen(buf));
}

char *strupr(char *s)
{
	int i;

	for(i=0; s[i]; i++) s[i] = toupper(s[i]);

	return s;
}


char *strlwr(char *s)
{
	int i;

	for(i=0; s[i]; i++) s[i] = tolower(s[i]);

	return s;
}

/* strcpy; return number of bytes copied, not including null terminator */
unsigned int smov(char *dst, const char *src)
{
	int len;

	len = strlen(src);
	memmove(dst, src, len+1);
	
	return len;
}

/* given an array of words, squish it into a space-separated string.
 * argv must be NULL-terminated. */
char *squish_argv(char **argv)
{
	return squish_argvcs(argv, str_count(&argv), 0, str_count(&argv));
}

/* given an array of words, squish it into a space-separated string.
 * argv must be NULL-terminated. */
char *squish_argvs(char **argv, int skip, int num)
{
	return squish_argvcs(argv, str_count(&argv), skip, num);
}

/* given an array of words, squish it into a space-separated string.
 * argv has argc elements. */
char *squish_argvc(char **argv, int argc)
{
	int len=0, c;
	static char *buf=0;
	return squish_argvcs(argv, argc, 0, argc);
	
	for(c=0; c < argc; c++)
		len+=strlen(argv[c])+1;

	if(buf) free(buf);
	buf=(char *)malloc0(len+1);
	
	for(c=0; c < argc; c++) {
		if(strlen(buf)) strcat(buf, " ");
		strcat(buf, argv[c]);
	}
	return buf;
}

/* given an array of words, squish it into a space-separated string.
 * argv has argc elements.  skip the first skip elements; output
 * num elements.  if num is negative, output up to element #-num.
 * ie:
 * skip = 0, num = 5: output elements 0-4.
 * skip = 3, num = 5: output elements 3-7
 * skip = 3, num = -5: output elements 3-5.
 */
char *squish_argvcs(char **argv, int argc, int skip, int num)
{
	int len=0, c, top;
	static char *buf=0;

	top = num >= 0? skip+num+1: -num;
	if(top > argc) top = argc;
	
	for(c=0; c < argc; c++)
		len+=strlen(argv[c])+1;

	if(buf) free(buf);
	buf=(char *)malloc0(len+1);
	
	for(c=skip; c < top; c++) {
		if(strlen(buf)) strcat(buf, " ");
		strcat(buf, argv[c]);
	}
	return buf;
}

/* return the number of characters matching c in buf */
int char_count(const char *buf, char c)
{
	int i=0;

	while(*buf) {
		if(*buf == c) i++;
		buf++;
	}

	return i;
}

/* Break down buf into a null-terminated array of words.  Since this is a
 * statically-allocated array, duplicate it if you need to keep it. */
char **line_break(const char *buf, char del)
{
	static char *sptr = NULL, **rptr = NULL;
	static char *s, **r;
	int n; /* number of words */

	/* give one for the last word, and one for a null */
	n = char_count(buf, del) + 2;

	/* Free the old allocs */
	if(sptr) free(sptr);
	if(rptr) free(rptr);

	/* Duplicate the string, and make an array of pointers to char *'s big enough
	 * to point at each word. */
	sptr = s = strdup(buf);
	rptr = r = (char **) malloc0(sizeof(char **) * n);

	/* Now, point each value at a word, and change the spaces between words to
	 * null terminators. */
	while(s) {
		*(r++) = s;

		s = index(s, del);
		if(s) *(s++) = 0;
	}

	*r = 0;
	return rptr;
}

/* do a quick ASCIIization: separate the low and high 4 bits of each
 * byte (0-15), and send it as 'a'-'p'. we could easily pack this more,
 * but lets keep it simple */
char *asciize(char *str, int n)
{
	static char *buf = NULL;
	int i, ii;
	if(buf) free(buf);
	buf = malloc0(n * 2 + 1);
	for(i=0, ii=0; i < n; i++, ii+=2) {
		buf[ii] = (str[i] & 0x0F) + 'a';
		buf[ii+1] = ((str[i] & 0xF0) >> 4)+ 'a';
	}
	return buf;
}

/* given an asciize-d string, decode into char, which is at least
 * n bytes long.  return 0 on success, or -1 on failure, with
 * errno set appropriately */
int deasciize(char *dest, const char *str, int n)
{
	int i, ii;
	for(i=0, ii=0; i < n && ii < strlen(str); i++, ii+=2) {
		if(!isalpha(str[ii]) || !isalpha(str[ii+1])) {
			errno = EINVAL;
			return -1;
		}
		
		dest[i] = (str[ii] - 'a') | (str[ii+1] - 'a') << 4;
	}

	return 0;
}

char	*get_token(char **src, const char *token_sep)
/*
 * Just a little more convenient than strtok()
 * This function returns a pointer to the first token
 * in src, and makes src point to the "new string".
 */
{
	char	*tok;

	/* FIXME: we really should whine if !src (rest
	 * are ok) */
	if(!(src && *src && **src))
		return NULL;

	/* first, removes leading token_sep's */
	while(**src && strchr(token_sep, **src))
		(*src)++;
	
	/* first non token_sep */
	if(**src)
		tok = *src;
	else
		return NULL;

	/* Make *src point after token */
	*src = strpbrk(*src, token_sep);
	if(*src)
	{
		**src = '\0';
		(*src)++;
		while(**src && strchr(token_sep, **src))
			(*src)++;
	}
	else
		/* *src = NULL; */
		*src = "";
	return tok;
}

char 	*strcasestr(const char *s1, const char *s2)
{
	char n1[256], n2[256];
	int i;

	for (i = 0; s1[i]; i++) n1[i] = toupper(s1[i]);
	n1[i] = '\0';

	for (i = 0; s2[i]; i++) n2[i] = toupper(s2[i]);
	n2[i] = '\0';
	return(strstr(n1,n2));
}

int array_size(void **array)
{
	int c=0;
	if(!array) return 0;
	while(array[c]) c++;
	return c;
}

void *grow_array(void ***array, size_t size)
{
	int num;
	void **work;

	work = *array;
	num = array_size(work);

	/* increase it by 1 */
	work = (void **) realloc(work, (num + 2) * sizeof (void *));
	work[num] = malloc0(size);
	work[num+1] = NULL;

	memset(work[num], 0, size);

	*array = work;
	return work[num];
}

void shrink_array(void ***array, int n)
{
	void **work;
	int num;

	work = *array;
	num = array_size(work);
	
	free(work[n]);

	/* move all pointers ahead of it down */
	memmove(work+n, work+n+1, sizeof(void *) * (num - n));
	num--;
	work = (void **) realloc(work, sizeof(void *) * (num + 1));
	*array = work;
}

/* stuff for dealing with arrays of strings (char **).  treat the array as
 * null-terminated. */

/* since the array we're using is a char **, and we may want to change this
 * pointer, we have to pass it a (ugh) char ***.  we'll do this for all of
 * these functions, so as to not be confusing to the caller: *always* pass
 * it "&arr". */

/* this stuff should layer on grow_array */

/* i don't like dealing with char ***s. work in this variable, then
 * copy it over when we're done. */

static char **work;

/* return the number of entries in the array, not including the null
 * terminator. */
int str_count(char ***array)
{
	int c = 0;

	if(!array || !*array) return 0;
	work = *array;
	while(work[c]) c++;
	return c;
}

int str_add(char ***array, const char *string)
{
	int num;

	work = array? *array:NULL;

	num = str_count(array);
	/* 2 is 1 for the new element, 1 for the null terminator, which is
	 * not included in str_count */
	if((work = (char **) realloc(work, sizeof(char *) * (num+2))) == NULL) return -1;

	work[num] = strdup(string);
	work[num+1] = NULL;
	
	*array = work;
	
	return 0;
}

/* delete element n from array, freeing memory.  n == 0 is the first
 * element */
int str_del(char ***array, int n)
{
	if(n >= str_count(array)) {
		errno = EINVAL;
		return -1;
	}
	work = *array;

	shrink_array((void ***)&work, n);

	*array = work;
	return 0;
}

int str_clear(char ***array)
{
	int c;
	
	if(!array || !*array) return 0;

	work = *array;
	for(c=0; work[c]; c++)
		free(work[c]);

	free(work);
	*array = NULL;

	return 0;
}

/* search the array for pattern using fnmatch; return the index
 * if found, -1 otherwise. */
int str_search(char ***array, const char *pattern)
{
	int n;

	if(!array || !*array) return -1;
	work = *array;

	for(n=0; work[n]; n++) {
		if(!fnmatch(pattern, work[n], 0)) return n;
	}

	return -1;
}

int str_match(char ***array, const char *pattern)
{
	int n;

	if(!array || !*array) return -1;
	work = *array;

	for(n=0; work[n]; n++) {
		if(!strcasecmp(pattern, work[n])) return n;
	}

	return -1;
}

void str_swap(char ***array, int a, int b)
{
	char *x;
	
	if(!array || !*array) return;
	work = *array;

	x = work[a];
	work[a] = work[b];
	work[b] = x;
}

void str_shuffle(char ***array)
{
	int n, total;

	if(!array || !*array) return;
	work = *array;
	
	total = str_count(&work);

	for(n=0; work[n]; n++) {
		str_swap(&work, n, rand() % total); 		
	}
}


void str_init(char ***array)
{
	*array = 0;
}

// like strdup(), but if the current string isn't null, it frees it first
char *strredup(char *var, const char *str)
{
    if (var != NULL)
        free(var);
    var = strdup(str);
    return var;
}

/* return the higher of a or b */
int max(int a, int b)
{
	return a>b? a:b;
}

/* return the lower of a or b */
int min(int a, int b)
{
	return a<b? a:b;
}

/* return a simple hash of a string */
int smallhash(const char *buf)
{
	int x = 0;
	int y = 1, z = 0;

	while(*buf) {
		x += (*buf ^ y);
		if(!z) {
			y <<= 1;
			if(y == 128) z = 1;
		} else {
			y >>= 1;
			if(y == 1) z = 0;
		}

		buf++;
	}

	return x;
}

/* This stuff depends on md5, which I havn't put in yet. */
#if 0

#define CYCLE_SPEED    750 

#define bin_to_ascii(c) ((c)>=38?((c)-38+'a'):(c)>=12?((c)-12+'A'):(c)+'.')

/*
 * crypt only: convert from 64 bit to 11 bit ASCII 
 * prefixing with the salt
 */
static char *output_conversion(unsigned long v1, unsigned long v2, const char *salt)
{
	static char outbuf[14];
	int  i,
	     s;

	outbuf[0] = salt[0];
	outbuf[1] = salt[1] ? salt[1] : salt[0];

	for (i = 0; i < 5; i++)
		outbuf[i + 2] = bin_to_ascii((v1 >> (26 - 6 * i)) & 0x3f);

	s = (v2 & 0xf) << 2;
	v2 = (v2 >> 2) | ((v1 & 0x3) << 30);

	for (i = 5; i < 10; i++)
		outbuf[i + 2] = bin_to_ascii((v2 >> (56 - 6 * i)) & 0x3f);

	outbuf[12] = bin_to_ascii(s);
	outbuf[13] = 0;

	return outbuf;
}

static int up_and_addem(const char *string)
{
  unsigned long checksum = 0;
  unsigned int i;

  for (i = 0; i < strlen(string); i++)
        checksum += string[i];

  return (checksum % CYCLE_SPEED) + 1;
}

char *mycrypt(const char *key, const char *salt)
{
	int  i, j, size = 0;

	MD5_CTX context;
	unsigned char digest[16];

	static unsigned char string[14];
	char *fullkey,
	    *hostname;

	size += 66;

	if ((hostname = calloc(size, sizeof(char))) == NULL) 
	{
		printf("Insufficient memory for crypt()\n");
		exit(EXIT_FAILURE);
	}

	/*
	 * I've asked serveral ppl which they prefer, a slow, and bloated 
	 * crypt() (takes a long time to crypt and compare) are a lean 
	 * mean crypt, (less time for login). What they wanted was slow.
	 * So this is to make it slow. Also this makes it trickey to use
	 * a UFC that trys to get around our stringcats by only doing them 
         * once and then using the same salt over and over.
	 */
	{
		char *tmpstring;
		if ((tmpstring = (char *)calloc(strlen(key) + 3, sizeof(char))) == NULL)
		{
			printf("Insufficient memory for crypt()\n");
			exit(EXIT_FAILURE);
		}	
		strncat(tmpstring, salt, 2);
		strcat(tmpstring, key);
		j = up_and_addem(tmpstring);
		free(tmpstring);
	}


	strncat(hostname, salt, 2);

	if ((fullkey = (char *) calloc(j * strlen(hostname) + strlen(key) + 3, sizeof(char))) == NULL)
	{
		printf("Insufficient memory for crypt()\n");
		exit(EXIT_FAILURE);
	}

	strcpy(fullkey, key);

	for (i = 0; i < j; i++)
		strcat(fullkey, hostname);

	{
		unsigned long result[2];
		MD5Init(&context);
		MD5Update(&context, fullkey, strlen(fullkey));
		MD5Final(digest, &context);
		
		memcpy(result, digest, 8);

                strcpy(string, output_conversion(result[0], result[1], salt));
	}

	free(hostname);
	free(fullkey);
	string[13] = 0;

	return string;
}

#define bin_to_ascii(c) ((c)>=38?((c)-38+'a'):(c)>=12?((c)-12+'A'):(c)+'.')
char *crypt_passwd(const char *pass)
{
	char salt[2];
	time_t tm;
	static char *c;

	/* from passwd.c: */
	time(&tm); tm ^= getpid(); tm ^= (rand());

	salt[0] = bin_to_ascii(tm & 0x3f);
	salt[1] = bin_to_ascii((tm >> 6) & 0x3f);

	c = mycrypt(pass, salt);

	return c;
}

#endif

int isnumeric(const char *num)
{
	while(*num) if(!isdigit(*(num++))) return 0;

	return 1;
}

int issignednum(const char *num)
{
	if(*num == '-') num++;

	return isnumeric(num);
}

/* *s*printf's that concat instead of copy */
int vscnprintf(char *buf, size_t size, const char *fmt, va_list ap)
{
	int cursize = 200, ret;
	char *buffer = (char *) malloc(cursize);
	while(1) {
		ret = vsnprintf(buffer, cursize, fmt, ap);
	
		/* WORKAROUND: vsnprintf is documented as returning -1 if
		 * the buffer required was less then what was given. 
		 * however, in glibc 2.1.1, it returns ret > cursize;
		 * interpret it as 'not enough'. */
		if(ret != -1 && ret < cursize) break;

		/* resize and retry */
		cursize *= 2;
		buffer = realloc (buffer, cursize);
	}

	/* if it's too big, give up */
	if(size && ret > size) {
		free(buffer);
		return -1;
	}

	strcat(buf, buffer);

	free(buffer);

	return strlen(buf);
}

int scnprintf(char *buf, size_t size, const char *fmt, ...)
{
	int ret;
	va_list msg;

	va_start(msg, fmt);
	ret = vscnprintf(buf, size, fmt, msg);
	va_end(msg);

	return ret;
}

int scprintf(char *buf, const char *fmt, ...)
{
	int ret;
	va_list msg;

	va_start(msg, fmt);
	ret = vscnprintf(buf, 0, fmt, msg);
	va_end(msg);

	return ret;
}

#ifndef HAVE_ASPRINTF
int vasprintf(char **strp, const char *fmt, va_list ap)
{
	int cursize = 200, ret;
	char *buffer = (char *) malloc(cursize);
	while(1) {
		ret = vsnprintf(buffer, cursize, fmt, ap);
	
		/* WORKAROUND: vsnprintf is documented as returning -1 if
		 * the buffer required was less then what was given. 
		 * however, in glibc 2.1.1, it returns ret > cursize;
		 * interpret it as 'not enough'. */
		if(ret != -1 && ret < cursize) break;

		/* resize and retry */
		cursize *= 2;
		buffer = realloc (buffer, cursize);
	}

	/* make an exact copy */
	*strp = strdup(buffer);
	free(buffer);

	return strlen(*strp);
}

int vaprintf(char **strp, const char *fmt, ...)
{
	int ret;
	va_list msg;

	va_start(msg, fmt);
	ret = vasprintf(strp, fmt, msg);
	va_end(msg);

	return ret;
}
#endif

#ifndef HAVE_DPRINTF
int dprintf(int fd, const char *fmt, ...)
{
	int ret;
	va_list msg;
	char *r;

	va_start(msg, fmt);
	ret = vasprintf(&r, fmt, msg);
	va_end(msg);

	write(fd, r, strlen(r));
	free(r);
	
	return ret;
}
#endif

/* Not sure if basename() is portable ... NON-REENTRANT*/
char *mybasename(const char *s)
{
	static char *buf = NULL;
	char *c;

	if(!s) return NULL;

	if(buf) free(buf);
	
	buf = strdup(s);
	c = strrchr(buf, '/');
	if(!c) return buf;

	return c+1;
}

static int real_copy(const char *source, const char *dest)
{
	struct stat dest_stats,source_stats;
	int ifd, ofd, len;
	char buf[1024 * 8];

	if (lstat (source, &source_stats) != 0) return 0;

	if (lstat (dest, &dest_stats) == 0) {
		if (source_stats.st_dev == dest_stats.st_dev
		    && source_stats.st_ino == dest_stats.st_ino) {
			errno = EPERM;
			return -1;
		}

		if (S_ISDIR (dest_stats.st_mode)) {
			errno = EISDIR;
			return -1;
		}
	}
  	else if (errno != ENOENT) return 0; 

	if (!strcmp(source, dest)) {
		errno = EPERM;
		return -1;
	}
	
	if (!S_ISREG (source_stats.st_mode)) {
		errno = EPERM;
		return -1;
	}

	ifd = open (source, O_RDONLY, 0);
	if (ifd < 0) return 0;
	ofd = open (dest, O_WRONLY | O_CREAT | O_TRUNC, 0600);
	if (ofd < 0) {
		close (ifd);
		return 0;
	}

	while ((len = read (ifd, buf, sizeof (buf))) > 0) {
		if (write (ofd, buf, len) < 0) {
			close (ifd);
			close (ofd);
			return 0;
		}
	}
	if (len < 0) {
		close (ifd);
		close (ofd);
		return 0;
	}

	if (close (ifd) < 0) {
		close (ofd);
		return 0;
	}
	if (close (ofd) < 0) return 0;

  /* chown turns off set[ug]id bits for non-root,
     so do the chmod last.  */

  /* Try to copy the old file's modtime and access time.  */
	{
		struct utimbuf tv;

		tv.actime = source_stats.st_atime;
		tv.modtime = source_stats.st_mtime;
		if (utime (dest, &tv)) return 0;
	}

  /* Try to preserve ownership.  For non-root it might fail, but that's ok.
     But root probably wants to know, e.g. if NFS disallows it.  */
	chown (dest, source_stats.st_uid, source_stats.st_gid);
	chmod (dest, source_stats.st_mode & 07777);

	return 1;
}

/* Move file SOURCE onto DEST.  Handles the case when DEST is a directory.
 * Return 0 if successful, -1 if an error occurred.  */

int copyfile (const char *source, const char *dest)
{
	char *base, *new_dest;
	int ret;
	if (fexist(dest) && fisdir(dest)) {
		/* Target is a directory; build full target filename. */

		base = (char *)mybasename(source);
		new_dest = (char *)malloc(strlen(dest)+strlen(base)+2);
		sprintf(new_dest,"%s/%s",dest,base);
		ret = real_copy(source, new_dest);
		free(new_dest);
	} else ret = real_copy(source, dest);

	return ret;
}

int fexist(const char *filename)
{
	struct stat statbuf;

	return lstat(filename,&statbuf) != -1;
}


int fisdir(const char *filename)
{
	struct stat statbuf;

	if(lstat(filename,&statbuf)==-1) {
		fprintf(stderr,"couldn't stat %s: %s\n",filename, strerror(errno));
		return 0;
	}

	return statbuf.st_mode & S_IFDIR;
}

/* clean fgets: strip s */
char *fgetsc(char *s, int size, FILE *stream)
{
	if(!fgets(s, size, stream)) return NULL;

	strip(s);

	return s;
}

char *strregerr(int err, regex_t *trig_reg)
{
	static char *buf = NULL;
	int n;

	n = regerror(err, trig_reg, 0, 0);
	if(buf) free(buf);
	buf = (char *) malloc0(n);
	regerror(err, trig_reg, buf, n);

	return buf;
}

int iswild(char c)
{
	return (c == '?' || c == '*');
}

/* strip all but the last n bytes of fn */
int ftrim(const char *fn, int n)
{
	int fd, newfd, len;
	char c, newfn[1024];
	char buf[1024 * 8];

	if((fd = open(fn, O_RDWR)) == -1) return -1;

	if(lseek(fd, -n, SEEK_END) == -1) return 0; /* OK */

	/* read until a \n */
	while(1) {
		if(read(fd, &c, 1) == -1) {
			/* EOF? The file must be garbage; seek back to where we
			 * were and just chop there */
			lseek(fd, -n, SEEK_END);
			break;
		}
		
		if(c == '\n') break;
	}

	/* make a new temp file */
	snprintf(newfn, 1000, "%s.XXXXXX", fn);
	mktemp(newfn);
	
	if((newfd = open(newfn, O_WRONLY | O_CREAT | O_EXCL, 0600)) == -1) return -1;

	/* copy */

	while ((len = read (fd, buf, sizeof (buf))) > 0) {
		int xerrno;
		if (write (newfd, buf, len) < 0) {
			xerrno = errno;
			close (fd);
			close (newfd);
			unlink(newfn);
			errno = xerrno;
			
			return -1;
		}
	}	

	close (fd);
	close (newfd);

	/* move the new file over the old; unlink the old just to be safe */
	unlink(fn);
	rename(newfn, fn); 

	return 0;
}

/* Read a delimited database line.  buf is the line; fmt is the format
 * of that line; delim is the field delimeter.  Assumptions: backslash-escaped.
 * ie:
 *
 * Hello, this is one field:Hello, this is two fields
 *
 * Hello, this\: is one field:and this is another
 *
 * C:\\This is one field:/usr/doc/this/is/another
 *
 * The format is a list of characters indicating type; only "s" (string)
 * and "i" are implemented.  ie: "ssis" reads "hello:this:3:is a line".
 *
 * The arguments following are pointers to store the values.  If a pointer
 * is NULL, its value is discarded.  If a numeric value contains non-digit
 * characters (or is empty), it will be set to -1.
 *
 * No escaping is done in numeric values; "1\5" is invalid and will
 * return -1.
 *
 * If the character is a capitalized "S", memory is allocated as needed and
 * put in the pointer.
 * 
 * Returns the number of fields processed (including invalid integers, etc).
 * Ex:
 * char s1[1000], *s2;
 * int i;
 * 
 * read_db("hello:there:3", "ssi", ':', s1, NULL, &i);
 * read_db("hello:there:3", "Ssi", ':', &s2, NULL, &i);
 *
 */
int read_db(const char *buf, char delim, const char *fmt, ...)
{
	va_list ap;
	int num = 0, end = 0;
	va_start(ap, fmt);

	while(*fmt && !end) {
		if(*fmt == 's' || *fmt == 'S') {
			/* search for unescaped delim, or EOL */
			const char *p;
			char *ret; /* ... value */
					
			p = buf;
			while(*p) {
				if(*p == '\\') {
					/* skip the escaped char */
					p++;
					if(*p) p++;
					continue;
				}
				if(*p == delim) break;
				p++;
			}
	
			if(*fmt == 'S') {
				char **retp;
				retp = (char **) va_arg(ap, char **);
				ret = (char *) malloc(p - buf + 1);
				*retp = ret;
			} else
				ret = (char *) va_arg(ap, char *);

			/* copy from buf (to p) into ret, unescaping as
			 * we go */
			while(*buf && buf < p) {
				if(*buf == '\\') {
					buf++;
					if(!*buf) break;
				}
				if(ret) *(ret++) = *buf;
				buf++;
			}
			
			if(ret) *ret = 0;
			
			/* skip the delimiter */
			if(*buf) buf++;
			/* no delimiter? end of string */
			else end = 1;

			num++;
		} else if(*fmt == 'i') {
			/* search for delim or EOL */
			const char *p;
			int *ret; /* ... value */
			int invalid = 0;
			int negative = 0;
			
			ret = (int *) va_arg(ap, int *);
		
			p = index(buf, delim);
			/* copy from buf to ret; if we hit a bad
			 * char, reset to -1 and abort */
			if(ret) *ret = 0;
			
			if(*buf == '-') {
				negative = 1;
				buf++;
			}
			
			while(*buf && buf < p) {
				if(!isdigit(*buf)) {
					/* set it invalid, but traverse
					 * the whole string */
					invalid = 1;
					break;
				}
				
				*ret *= 10;
				*ret += *buf - '0';

				buf++;
			}

			if(negative && ret) *ret *= -1;
			if(invalid && ret) *ret = -1;
			/* skip the delimiter */
			if(*buf) buf++;
			/* no delimiter? end of string */
			else end = 1;

			num++;
		}
		
		fmt++;
	}
	
	
	/* if there are values remaining, reset them to blank/NULL/-1 */
	while(*fmt) {
		if(*fmt == 's') {
			char *ret;
			ret = (char *) va_arg(ap, char *);
			if(ret) *ret = 0;
		} else if(*fmt == 'S') {
			char **ret;
			ret = (char **) va_arg(ap, char **);
			*ret = NULL;
		} else if(*fmt == 'i') {
			int *ret;
			ret = (int *) va_arg(ap, int *);
			if(ret) *ret = -1;
		}
		fmt++;
	}
	va_end(ap);

	return num;
}

