// File dealing with client communications

#include <includes.h>

#include "clients.h"
#include "files.h"
#include "network.h"
#include "sockets.h"
#include "ftp.h"

// attempt to help libc5 boxes
#ifndef O_ASYNC
#  define O_ASYNC 020000
#endif
#ifndef O_NONBLOCK
#  define O_NONBLOCK 04000
#endif

void CheckClientSockets(void)
{
    fd_set client_socks;
    int size, rc, flags;
    struct sockaddr_un addr;
    struct sockaddr_in net_addr;
    struct timeval timeout;
    ClientInfo *client = FirstClient;
    BOOL from_top = FALSE;
    char resp_buf[256];

    error(E_TRACE, "Checking client sockets...");
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    FD_ZERO(&client_socks);
    FD_SET(DefaultClientSocket, &client_socks);
    if (NetClientSocket > 0)
        FD_SET(NetClientSocket, &client_socks);
    while (client)
    {
        // check to see if a socket's dodgy
        flags = fcntl(client->Socket, F_GETFL);
        if (flags == -1)
        {
            error(E_TRACE, "%s looks dodgy, I'm removing it", client->Name);
            DisconnectClient(client);
            client = FirstClient;
            if (client == NULL)
                return;
        }
        else
        {
            FD_SET(client->Socket, &client_socks);
        }
        client = client->Next;
    }
    
    /* Block until input arrives on one or more active sockets. */
    rc = select(FD_SETSIZE, &client_socks, NULL, NULL, &timeout);
    //error(E_TRACE, "Selection of client sockets complete");
    if (rc < 0)
    {
        error(E_WARN, "Couldn't select client input: %s", strerror(errno));
        return;
    }
    else if (rc == 0)
    {
        // we didn't really get any input...
        //error(E_TRACE, "There was no input");
        return;
    }
    
    if (FD_ISSET(DefaultClientSocket, &client_socks))
    {
        if (ClientCount >= MAX_CLIENTS)
        {
            error(E_WARN, "Couldn't add client: maximum (%d) reached",
                  MAX_CLIENTS);
            return;
        }
        //error(E_TRACE, "Got connection on default socket");
        // Add the client to the list of clients
        if (ClientCount == 0)
        {
            FirstClient = dxmalloc(sizeof(ClientInfo));
            FirstClient->Prev = NULL;
            FirstClient->Next = NULL;
            ClearClient(FirstClient);
            size = sizeof(addr);
            FirstClient->Socket = accept(DefaultClientSocket,
                                         (struct sockaddr *)&addr, &size);
            if (SetUpClientSocket(FirstClient))
                ClientCount++;
        }
        else
        {
            client = FirstClient;
            while (client)
            {
                if (!client->Next)
                {
                    client->Next = dxmalloc(sizeof(ClientInfo));
                    client->Next->Prev = client;
                    client = client->Next;
                    client->Next = NULL;
                    ClearClient(client);
                    size = sizeof(addr);
                    client->Socket = accept(DefaultClientSocket,
                                            (struct sockaddr *)&addr, &size);
                    if (SetUpClientSocket(client))
                        ClientCount++;
                    break;
                }
                client = client->Next;
            }
        }
    }
    else if ((NetClientSocket > 0) && FD_ISSET(NetClientSocket, &client_socks))
    {
        if (ClientCount >= MAX_CLIENTS)
        {
            error(E_WARN, "Couldn't add client: maximum (%d) reached",
                  MAX_CLIENTS);
            return;
        }
        error(E_TRACE, "Net client socket is set");
        sprintf(resp_buf, "%d Welcome to Darxite %s. Enter your password.\n",
                DX_OK, RELEASE_VER);
        // Add the client to the list of clients
        if (ClientCount == 0)
        {
            FirstClient = dxmalloc(sizeof(ClientInfo));
            FirstClient->Next = NULL;
            FirstClient->Prev = NULL;
            ClearClient(FirstClient);
            size = sizeof(net_addr);
            FirstClient->NeedPassword = TRUE;
            FirstClient->Socket = accept(NetClientSocket,
                                         (struct sockaddr *)&net_addr,
                                         &size);
            if (SetUpClientSocket(FirstClient))
            {
                ClientCount++;
                write(FirstClient->Socket, resp_buf, strlen(resp_buf) + 1);
            }
            else
            {
                dxfree(FirstClient);
            }
        }
        else
        {
            client = FirstClient;
            while (client)
            {
                if (!client->Next)
                {
                    client->Next = dxmalloc(sizeof(ClientInfo));
                    client->Next->Prev = client;
                    client = client->Next;
                    client->Next = NULL;
                    ClearClient(client);
                    size = sizeof(net_addr);
                    client->NeedPassword = TRUE;
                    client->Socket = accept(NetClientSocket,
                                            (struct sockaddr *)&net_addr,
                                            &size);
                    if (SetUpClientSocket(client))
                    {
                        ClientCount++;
                        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                    }
                    else
                    {
                        dxfree(client);
                    }
                    break;
                }
                client = client->Next;
            }
        }
    }
    
    // if we get here, we haven't had a new connection but some input on
    // a current one
    client = FirstClient;
    while (client)
    {
        //error(E_TRACE, "Checking client at %d...", client);
        if (FD_ISSET(client->Socket, &client_socks))
        {
            // if the client disconnected, take it from the top
            if (ReadClientCommands(client))
                from_top = TRUE;
        }
        if (from_top)
        {
            //error(E_TRACE, "Taking it from the top: %d", FirstClient);
            from_top = FALSE;
            client = FirstClient;
        }
        else
        {
            client = client->Next;
        }
    }
}

// resets all a client's attributes to scratch
void ClearClient(ClientInfo *client)
{
    memset(client->Name, 0, sizeof(client->Name));
    client->Socket = -1;
    client->Connected = FALSE;
    client->NeedPassword = FALSE;
    client->ConnectionTime = 0;
    client->EventMask = NO_EVENT;
    memset(client->LastCmd, 0, sizeof(client->LastCmd));
    memset(client->Activity, 0, sizeof(client->Activity));
    client->Server = NULL;
    client->ControlSocket = NULL;
    client->DataSocket = NULL;
}

BOOL SetUpClientSocket(ClientInfo *client)
{
    int fd_flags;

    if (client == NULL)
    {
        error(E_TRACE, "SetUpClientSocket() was passed a NULL");
        return FALSE;
    }
    if (client->Socket < 0)
    {
        error(E_WARN, "Couldn't accept client connection: %s",
              strerror(errno));
        return FALSE;
    }
    
    fd_flags = fcntl(client->Socket, F_GETFL, 0);
    if (fd_flags == -1)
    {
        error(E_WARN, "Couldn't read socket flags: %s", strerror(errno));
        return FALSE;
    }
    fd_flags |= (O_ASYNC | O_NONBLOCK);
    if (fcntl(client->Socket, F_SETFL, fd_flags) < 0)
    {
        error(E_WARN, "Couldn't set socket flags: %s", strerror(errno));
        return FALSE;
    }
    if (fcntl(client->Socket, F_SETOWN, getpid()) < 0)
    {
        error(E_WARN, "Couldn't set IO owner: %s", strerror(errno));
        return FALSE;
    }
    
    time(&client->ConnectionTime);
    client->Connected = TRUE;
    ReadClientCommands(client);
    return TRUE;
}

// Sends an event to any clients that want to know.
// Returns the number of clients it sent a message to.
int SendEvent(int event_id, const char *event_string)
{
    ClientInfo *client;
    int count = 0;
    
    client = FirstClient;
    while (client)
    {
        if (client->EventMask & event_id)
        {
            error(E_TRACE, "Notifying \"%s\": \"%s\"", client->Name,
                  event_string);
            write(client->Socket, event_string, strlen(event_string) + 1);
            count++;
        }
        client = client->Next;
    }
    return count;
}

// Reads any waiting commands from the specified client.
// Returns TRUE if the client disconnected.
BOOL ReadClientCommands(ClientInfo *client)
{
    int nbytes, total_read = 0;
    char buffer[1024], c;
    BOOL read_complete = FALSE;

    memset(buffer, 0, sizeof(buffer));
    while (!read_complete)
    {
        nbytes = read(client->Socket, &c, 1);
        //error(E_TRACE, "sock %d finished read of '%d' was %d bytes",
        //      client->Socket, c, nbytes);
        if (nbytes < 0)
        {
            if (errno == EAGAIN)	// there was no data to read
            {
                if (total_read == 0)
                    return FALSE;
                else
                    read_complete = TRUE;
            }
            else
            {
                error(E_WARN, "Read error from client \"%s\": %s",
                      client->Name, strerror(errno));
                return FALSE;
            }
        }
        else if (nbytes > 0)
        {
            if (c == '\0')
                continue;
            
            // end of line - we've read a full command
            if (c == '\n')
            {
                buffer[total_read] = '\0';
                if (lastchr(buffer) == '\r')
                    lastchr(buffer) = '\0';
                strncpy(client->LastCmd, buffer, sizeof(client->LastCmd));
                // The first command we receive should be the name
                // of the client, unless it's remote, in which case the
                // password should be sent first.
                if (client->NeedPassword)
                {
                    error(E_TRACE, "Client needs to send password of \"%s\"",
                          RemotePassword);
                    if (!strcmp(RemotePassword, buffer))
                    {
                        error(E_TRACE, "Password OK");
                        // sent tbe password OK
                        client->NeedPassword = FALSE;
                        sprintf(buffer, "%d Password OK; tell me your name.\n",
                                DX_OK);
                        write(client->Socket, buffer, strlen(buffer) + 1);
                    }
                    else
                    {
                        error(E_TRACE, "Password \"%s\" invalid", buffer);
                        sprintf(buffer, "%d Invalid password.\n",
                                DX_E_PASSWORD);
                        write(client->Socket, buffer, strlen(buffer) + 1);
                        DisconnectClient(client);
                        return TRUE;
                    }
                }
                else
                {
                    if (!strcmp(client->Name, ""))
                    {
                        strcpy(client->Name, buffer);
                        sprintf(buffer, "%d \"%s\" connected OK.\n", DX_OK,
                                client->Name);
                        write(client->Socket, buffer, strlen(buffer) + 1);
                    }
                    else
                    {
                        if (!ParseClientCommand(client))
                            return TRUE;
                    }
                }
                strcpy(buffer, "");
                total_read = 0;
            }
            else
            {
                buffer[total_read++] = c;
            }
        }
        else	// client has closed connection
        {
            error(E_TRACE, "read() returned 0 - unexpected client death?");
            DisconnectClient(client);
            return TRUE;
        }
    }
    buffer[total_read] = '\0';
    error(E_WARN, "Read incomplete command from client \"%s\": %s",
          client->Name, buffer);
    return FALSE;
}

void DisconnectClient(ClientInfo *client)
{
    close(client->Socket);
    if (strcmp(client->Name, ""))
        error(E_TRACE, "Client \"%s\" has closed connection", client->Name);
    else
        error(E_TRACE, "Unnamed client has closed connection");
    DataDisconnect(client->DataSocket);
    FreeSocket(&client->DataSocket);
    Disconnect(client->ControlSocket, FALSE);
    FreeSocket(&client->ControlSocket);
    
    if (client->Prev)
        client->Prev->Next = client->Next;
    else
        FirstClient = client->Next;
    if (client->Next)
        client->Next->Prev = client->Prev;
    dxfree(client);
    ClientCount--;
}

// NB: A response to a command can be multi-line.
// It is terminated by a NULL (0).

// Returns whether to try to read another or not, ie. whether the client
// is still connected.
BOOL ParseClientCommand(ClientInfo *client)
{
    char *cmd = client->LastCmd;
    char resp_buf[8192], info_buf[1024];
    char serv_buf[256], path[256], protocol[10];
    char num_buf[30], url[256], *field_ptr;
    struct tm *conn_time;
    BOOL verbose = FALSE;
    int errn, rc;
    int total_size = 0, size_got = 0;
    int file_count = 0, total_count = 0;
    ClientInfo *client_ptr;
    ServerInfo *old_server, *server;
    FileInfo *file, *next_file;
    
    // hide passwords
    if (strncasecmp(cmd, "ftp pass", 8))
    {
        error(E_TRACE, "Read command \"%s\" from client \"%s\"",
              cmd, client->Name);
    }
    else
    {
        //error(E_TRACE, "Read command \"FTP PASS ???\" from client \"%s\"",
        //      client->Name);
    }
    memset(resp_buf, 0, sizeof(resp_buf));
    if (!strncasecmp(cmd, "Disconnect", 10) || !strncasecmp(cmd, "Bye", 3) ||
        !strncasecmp(cmd, "Quit", 4) || !strncasecmp(cmd, "Exit", 4))
    {
        DisconnectClient(client);
        return FALSE;
    }
    else if (!strncasecmp(cmd, "Version", 7))
    {
        sprintf(resp_buf, "%d %s daemon v%s, release %s (%s) by %s\n",
                DX_OK, PROG_NAME, VERSION, RELEASE_VER, RELEASE_NAME, AUTHORS);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "Kill", 4))
    {
        exit(0);
    }
    else if (!strncasecmp(cmd, "Open", 4))
    {
        if (strlen(cmd) < 6)
        {
            sprintf(resp_buf, "%d You must specify a server to open!\n",
                    DX_E_BAD_COMMAND);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        }
        else
        {
            cmd += 5;
            client->Server = FindServer(cmd, DX_EnableWritingServerDB);
            if (client->Server)
            {
                errn = Connect(client->Server->Name, DefaultFtpPort,
                               client->Activity, &(client->ControlSocket));
            }
            else
            {
                errn = DX_E_BAD_COMMAND;
            }
            if (errn == DX_OK)
            {
                client->ControlSocket->CmdResponse =
                    (char *)dxmalloc(RESPONSE_MAX);
                errn = GetResponse(client->ControlSocket,
                                   client->ControlSocket->CmdResponse,
                                   RESPONSE_MAX);
                if (errn == DX_OK)
                {
                    snprintf(resp_buf, sizeof(resp_buf), "%s\n",
                             client->ControlSocket->CmdResponse);
                }
                else
                {
                    sprintf(resp_buf, "%d %s\n", errn,
                            DX_ErrorText[errn - DX_OK]);
                }
            }
            else
            {
                sprintf(resp_buf, "%d %s\n", errn, DX_ErrorText[errn - DX_OK]);
            }
            error(E_TRACE, "Responding \"%s\"", resp_buf);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        }
    }
    else if (!strncasecmp(cmd, "Close", 5))
    {
        if (client->ControlSocket)
        {
            Disconnect(client->ControlSocket, FALSE);
            snprintf(resp_buf, sizeof(resp_buf), "%s\n",
                     client->ControlSocket->CmdResponse);
            FreeSocket(&client->ControlSocket);
            client->ControlSocket = NULL;
        }
        else
        {
            sprintf(resp_buf, "%d %s\n", DX_E_DISCONNECTED,
                   DX_ErrorText[DX_E_DISCONNECTED - DX_OK]);
        }
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "FTP", 3))
    {
        if (!client->ControlSocket)
        {
            sprintf(resp_buf, "%d You're not connected to an FTP server\n",
                    DX_E_DISCONNECTED);
        }
        else if (strlen(cmd) < 5)
        {
            sprintf(resp_buf, "%d Usage: FTP <command>\n", DX_E_BAD_COMMAND);
        }
        else
        {
            if (NetWrite(client->ControlSocket, cmd + 4) > -1)
            {
                snprintf(resp_buf, sizeof(resp_buf) - 1, "%s\n",
                         client->ControlSocket->CmdResponse);
            }
            else
            {
                sprintf(resp_buf, "Couldn't write to the network\n");
            }
            // now we've logged in, change directory if possible
            if (!strncasecmp(cmd + 4, "PASS", 4) &&
                strcmp(client->Server->DefaultPath, "/"))
            {
                NetWrite(client->ControlSocket, "CWD %s",
                         client->Server->DefaultPath);
            }
        }
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "List", 4) || !strncasecmp(cmd, "Nlst", 4))
    {
        if (strlen(cmd) > 5)
            strcpy(info_buf, cmd + 5);
        else
            strcpy(info_buf, ".");
        if (!strncasecmp(cmd, "List", 4))
            verbose = TRUE;
                
        if (!client->ControlSocket)
        {
            sprintf(resp_buf, "%d You're not connected to an FTP server\n",
                    DX_E_DISCONNECTED);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            return TRUE;
        }
        if (!SetType(client->ControlSocket, 'A'))
        {
            snprintf(resp_buf, sizeof(resp_buf), "%s\n",
                     client->ControlSocket->CmdResponse);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1); 
            return TRUE;
        }
        client->DataSocket = DataConnect(client->ControlSocket);
        if (!client->DataSocket)
        {
            snprintf(resp_buf, sizeof(resp_buf), "%s\n",
                     client->ControlSocket->CmdResponse);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            return TRUE;
        }
        
        if (!List(client->ControlSocket, info_buf, verbose))
        {
            DataDisconnect(client->DataSocket);
            FreeSocket(&client->DataSocket);
            snprintf(resp_buf, sizeof(resp_buf), "%s\n",
                     client->ControlSocket->CmdResponse);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1); 
            return TRUE;
        }
        
        client->DataSocket->Eof = FALSE;
        while ((TransferData(&(client->DataSocket), 1, NULL, 0, 20) > 0) &&
               !client->DataSocket->Eof)
        {
            rc = buf_write(client->Socket, client->DataSocket->DataBuf,
                           client->DataSocket->DataLength);
            if (rc == -1)
            {
                error(E_TRACE, "Error writing data to client socket!");
                break;
            }
        }
        GetResponse(client->ControlSocket, NULL, 0);
        write(client->Socket, "", 1);
    }
    else if (!strncasecmp(cmd, "AnonLogin", 9))
    {
        if (!client->ControlSocket)
        {
            sprintf(resp_buf, "%d You're not connected to an FTP server\n",
                    DX_E_DISCONNECTED);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            return TRUE;
        }
        if (!LogIn(client->ControlSocket, "anonymous",
                   DX_EMailAddress, client->Activity))
        {
            sprintf(resp_buf, "%d Couldn't log in anonymously\n", DX_E_FTP);
        }
        else
        {
            sprintf(resp_buf, "%d Anonymous login OK\n", DX_OK);
        }
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "FileStatus", 10))
    {
        file = Batch.FirstFile;
        while (file)
        {
            if (file->TotalSize >= 0)
                sprintf(num_buf, "%d", file->TotalSize - file->StartOffset);
            else
                strcpy(num_buf, "?");
            sprintf(resp_buf, "%d%c\"%s://%s%s\" | \"%s\" | %d | %s | %d | "
                    "%d | \"%s\" | %s\n", DX_FILE_STATUS,
                    // if it's not the last file in the response, put
                    // a dash as the 4th character to make it like FTP
                    (file->Next == NULL) ? ' ' : '-', file->Protocol,
                    file->ActualServer->Name,
                    file->Path, file->LocalPath,
                    file->CurrentSize - file->StartOffset, num_buf,
                    (int)(time(NULL) - file->StartTime),
                    (int)(file->RxTxOverall /
                          (time(NULL) - file->StartTime + 1)),
                    file->Activity, file->Id);
            //error(E_TRACE, "rb: \"%s\"", resp_buf);
            write(client->Socket, resp_buf, strlen(resp_buf));
            file = file->Next;
        }
        write(client->Socket, "", 1);
    }
    else if (!strncasecmp(cmd, "FastFileStatus", 14))
    {
        file = Batch.FirstFile;
        while (file)
        {
//Modified by Manuel Clos:
            // we send: id, current size, total size, time, rate, activity
            sprintf(resp_buf, "%d%c%s | %d | %d | %d | %d | %s\n",
                    DX_FAST_FILE_STATUS,
                    // if it's not the last file in the response, put
                    // a dash as the 4th character to make it like FTP
                    (file->Next == NULL) ? ' ' : '-', file->Id,
                    file->CurrentSize - file->StartOffset,
//Added by Manuel Clos: TotalSize was missing.
                    file->TotalSize - file->StartOffset,
                    (int)(time(NULL) - file->StartTime),
                    (int)(file->RxTxOverall /
                          (time(NULL) - file->StartTime + 1)),
                    DX_CodeFromActivity(file->Activity));
            //error(E_TRACE, "rb: \"%s\"", resp_buf);
            write(client->Socket, resp_buf, strlen(resp_buf));
            file = file->Next;
        }
        write(client->Socket, "", 1);
    }
    else if (!strcasecmp(cmd, "FileCount"))
    {
        file = Batch.FirstFile;
        while (file)
        {
            // if the file is being transferred
            if ((file->DataSocket) && (file->DataSocket->Connected))
            {
                file_count++;
                if (file->TotalSize > 0)
                {
                    total_size += file->TotalSize;
                    size_got += file->CurrentSize;
                }
            }
            total_count++;
            file = file->Next;
        }
        sprintf(resp_buf, "%d %d | %d | %d | %d\n", DX_FILE_COUNT, size_got,
                total_size, file_count, total_count);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    // FileStatus is meant to be easily machine-parsable, Status is meant
    // to be easily human-readable.
    else if (!strncasecmp(cmd, "Status", 6))
    {
        memset(resp_buf, 0, sizeof(resp_buf));
        sprintf(resp_buf, "Connected clients:\n");
        client_ptr = FirstClient;
        while (client_ptr)
        {
            conn_time = localtime(&client_ptr->ConnectionTime);
            sprintf(info_buf, "%s has been connected since "
                    "%d:%02d:%02d on the %d/%d/%d\n",
                    client_ptr->Name, conn_time->tm_hour,
                    conn_time->tm_min, conn_time->tm_sec,
                    conn_time->tm_mday, conn_time->tm_mon + 1,
                    conn_time->tm_year + 1900);
            strcat(resp_buf, info_buf);
            client_ptr = client_ptr->Next;
        }
        strcat(resp_buf, "\nFiles in batch:\n");
        file = Batch.FirstFile;
        while (file)
        {
            if (file->TotalSize >= 0)
                sprintf(num_buf, "%d", file->TotalSize);
            else
                strcpy(num_buf, "?");
            sprintf(info_buf, "%s://%s%s: %dB of %sB at %dB/s: %s\n",
                    file->Protocol,
                    file->ActualServer->Name,
                    file->Path, file->CurrentSize,
                    num_buf, (int)(file->RxTxOverall /
                                   (time(NULL) - file->StartTime + 1)),
                    file->Activity);
            strcat(resp_buf, info_buf);
            file = file->Next;
        }
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "Pause", 5) && (strlen(cmd) > 6))
    {
        cmd += strlen("Pause ");
        file = Batch.FirstFile;
        while (file)
        {
            if (file->Complete || file->Paused)
            {
                file = file->Next;
                continue;
            }
            sprintf(info_buf, "%s://%s%s", file->Protocol,
                    file->ActualServer->Name, file->Path);
            if (!strcasecmp(cmd, "all") || !strcasecmp(info_buf, cmd))
            {
                if (file->Starting)
                {
                    error(E_TRACE, "Cancelling file's thread");
                    CANCEL_THREAD(file->ThreadId);
                    // wait for thread to terminate... yuck
                    Sleep(5);
                }
                file->Paused = TRUE;
                file->Starting = FALSE;
                Batch.FileCount--;
                if (Batch.FileCount == 0)
                    Batch.Complete = TRUE;
                CancelTransfer(file->ControlSocket, file->Protocol);
                DisconnectFile(file, FALSE);
                strcpy(file->Activity, "Paused");
                if (strcasecmp(cmd, "all"))
                {
                    sprintf(resp_buf,
                            "%d File %s paused\n", DX_OK, info_buf);
                    write(client->Socket,resp_buf,strlen(resp_buf)+1);
                    return TRUE;
                }
            }
            file = file->Next;
        }
        if (!strcasecmp(cmd, "all"))
        {
            sprintf(resp_buf, "%d All files paused\n", DX_OK);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            return TRUE;
        }
        sprintf(resp_buf, "%d Couldn't pause file\n", DX_ERROR);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "Continue", 8) && (strlen(cmd) > 8))
    {
        cmd += strlen("Continue ");
        file = Batch.FirstFile;
        while (file)
        {
            if (file->Complete || !file->Paused)
            {
                file = file->Next;
                continue;
            }
            sprintf(info_buf, "%s://%s%s", file->Protocol,
                    file->ActualServer->Name, file->Path);                
            if (!strcasecmp(cmd, "all") || !strcasecmp(info_buf, cmd))
            {
                file->Paused = FALSE;
                file->Started = FALSE;
                strcpy(file->Activity, "Transferring file");
                // use the last mirror
                file->NextMirror--;
                if (file->NextMirror < -1)
                    file->NextMirror = -1;
                Batch.Complete = FALSE;
                Batch.FileCount++;
                WakeUpNow = TRUE;
                if (strcasecmp(cmd, "all"))
                {
                    sprintf(resp_buf, "%d File %s unpaused\n", DX_OK, cmd);
                    write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                    return TRUE;
                }
            }
            file = file->Next;
        }
        if (!strcasecmp(cmd, "all"))
        {
            sprintf(resp_buf, "%d All files unpaused\n", DX_OK);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        }
        else
        {
            sprintf(resp_buf, "%d Couldn't unpause file\n", DX_ERROR);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        }
    }
    else if (!strncasecmp(cmd, "Cancel ", 7) && (strlen(cmd) > 7))
    {
        cmd += 7;
        file = Batch.FirstFile;
        while (file)
        {
            sprintf(info_buf, "%s://%s%s", file->Protocol,
                    file->ActualServer->Name, file->Path);
            next_file = file->Next;
            if (!strcasecmp(cmd, "all") || !strcasecmp(info_buf, cmd))
            {
                // if we try to cancel a file that's a massive d/l child,
                // the master will be cancelled instead - because it has the
                // same URL and is the first file with that URL in the batch
                /*if (strchr(file->Flags, 'v'))
                {
                    if (strcasecmp(cmd, "all"))
                    {
                        sprintf(resp_buf, "%d Couldn't cancel file %s",
                                DX_ERROR, info_buf);
                        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                    }
                }
                else
                {*/
                    error(E_TRACE, "Removing %s from batch", info_buf);
                    if (!file->Complete)
                    {
                        if (file->Starting)
                        {
                            error(E_TRACE, "Cancelling file's thread");
                            CANCEL_THREAD(file->ThreadId);
                            // wait for thread to terminate... yuck
                            Sleep(5);
                        }
                        CancelTransfer(file->ControlSocket, file->Protocol);
                        DisconnectFile(file, FALSE);
                        file->Complete = TRUE;
                        FileComplete(file, "File cancelled");
                    }
                    else
                    {
                        RemoveFromBatch(file);
                    }
                    if (strcasecmp(cmd, "all"))
                    {
                        sprintf(resp_buf,
                                "%d File %s cancelled\n", DX_OK, info_buf);
                        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                        //break;
                        siglongjmp(MainJump, 1);
                        // JUMP to the top of the main loop
                    }
                    //}
            }
            file = next_file;
        }
        if (!strcasecmp(cmd, "all"))
        {
            sprintf(resp_buf, "%d All files cancelled\n", DX_OK);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            siglongjmp(MainJump, 1);
            // JUMP to the top of the main loop
        }
        else
        {
            sprintf(resp_buf, "%d Couldn't cancel file\n", DX_ERROR);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        }
    }
    else if (!strncasecmp(cmd, "RemoveCompleteFile ", 19) &&
             (strlen(cmd) > 19))
    {
        cmd += 19;
        file = Batch.FirstFile;
        while (file)
        {
            sprintf(info_buf, "%s://%s%s", file->Protocol,
                    file->ActualServer->Name, file->Path);
            if (file->Complete && !strcasecmp(info_buf, cmd))
            {
                error(E_TRACE, "Removing %s from batch", info_buf);
                RemoveFromBatch(file);
                sprintf(resp_buf, "%d File %s removed from batch\n", DX_OK,
                        info_buf);
                write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                return TRUE;
            }
            file = file->Next;
        }
        sprintf(resp_buf, "%d Couldn't remove complete file\n", DX_ERROR);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strcasecmp(cmd, "ReloadConfig"))
    {
        error(E_TRACE, "Reloading config files...");
        DX_ReadConfigFiles();
        sprintf(resp_buf, "%d Config reloaded\n", DX_CONFIG_RELOADED);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        SendEvent(CONFIG_CHANGE_EVENT, resp_buf);
    }
    else if (!strcasecmp(cmd, "ReloadBookmarks"))
    {
        server = FirstServer;
        while (server)
        {
            old_server = server;
            server = old_server->Next;
            dxfree(old_server);
        }
        sprintf(resp_buf, "%s/servers", DX_ProgDir);
        sprintf(info_buf, "%d server(s) read",  ReadServersFile(resp_buf));
        error(E_TRACE, info_buf);
        strcat(info_buf, "\n");
        write(client->Socket, info_buf, strlen(info_buf) + 1);
    }
    else if (!strncasecmp(cmd, "NotifyOnEvent ", 14) && (strlen(cmd) > 14))
    {
        cmd += 14;
        if (!strcasecmp(cmd, "FileCompletion"))
        {
            client->EventMask |= FILE_COMPLETION_EVENT;
            sprintf(resp_buf, "%d Will notify on file completions\n", DX_OK);
        }
        else if (!strcasecmp(cmd, "ConfigChange"))
        {
            client->EventMask |= CONFIG_CHANGE_EVENT;
            sprintf(resp_buf, "%d Will notify on config changes\n", DX_OK);
        }
        else if (!strcasecmp(cmd, "Redirection"))
        {
            client->EventMask |= REDIRECT_EVENT;
            sprintf(resp_buf, "%d Will notify on redirection\n", DX_OK);
        }
        else if (!strcasecmp(cmd, "Error"))
        {
            client->EventMask |= ERROR_EVENT;
            sprintf(resp_buf, "%d Will notify on errors\n", DX_OK);
        }
        else if (!strcasecmp(cmd, "Exit"))
        {
            client->EventMask |= EXIT_EVENT;
            sprintf(resp_buf, "%d Will notify on exit\n", DX_OK);
        }
        else if (!strcasecmp(cmd, "NewFile"))
        {
            client->EventMask |= NEW_FILE_EVENT;
            sprintf(resp_buf, "%d Will notify on new files\n", DX_OK);
        }
        else
        {
            sprintf(resp_buf, "%d Invalid event\n", DX_E_BAD_COMMAND);
        }
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strcasecmp(cmd, "DeleteFileLog"))
    {
        if (strcmp(DX_FileLogName, ""))
        {
            if (remove(DX_FileLogName) == 0)
            {
                sprintf(resp_buf, "%d File log deleted\n", DX_OK);
            }
            else
            {
                sprintf(resp_buf, "%d Couldn't delete file log: %s\n",
                        DX_ERROR, strerror(errno));
            }
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
        }
    }
    else if (!strncasecmp(cmd, "ServerForAlias ", 15) && (strlen(cmd) > 15))
    {
        cmd += 15;
        server = FirstServer;
        while (server)
        {
            if (!strcasecmp(cmd, server->Name) ||
                !strcasecmp(cmd, server->Alias))
            {
                sprintf(resp_buf, "%d %s\n", DX_OK, server->Name);
                write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                return TRUE;
            }
            server = server->Next;
        }
        sprintf(resp_buf, "%d Alias \"%s\" not found\n", DX_ERROR, cmd);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "Reget", 5))
    {
        if ((strlen(cmd) < 7) || (*(cmd + 5) != ' '))
        {
            sprintf(resp_buf, "%d Usage: Reget <URL>\n", DX_E_BAD_COMMAND);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            return TRUE;
        }
        cmd += 6;
        strcpy(info_buf, cmd);
        field_ptr = DX_GetNextField(info_buf);
        if (*field_ptr)
        {
            if (*field_ptr == '"')
                field_ptr++;
            strcpy(url, field_ptr);
            if (lastchr(url) == '"')
                lastchr(url) = '\0';
        }
        // find the file to cancel
        DX_ParseUrl(url, protocol, serv_buf, sizeof(serv_buf), path,
                    sizeof(path));
        file = Batch.FirstFile;
        while (file)
        {
            error(E_TRACE, "does \"%s\"=\"%s\", \"%s\"=\"%s\", \"%s\"=\"%s\"",
                  path, file->Path, protocol, file->Protocol, serv_buf,
                  file->ActualServer->Name);
            if (!strcasecmp(file->Path, path) &&
                !strcasecmp(file->Protocol, protocol) &&
                !strcasecmp(file->ActualServer->Name, serv_buf))
            {
                error(E_TRACE, "Reget local path: \"%s\"", file->LocalPath);
                CancelTransfer(file->ControlSocket, file->Protocol);
                DisconnectFile(file, FALSE);
                // if it was completed, it's not now
                if (file->Complete)
                    Batch.FileCount++;
                file->Paused = FALSE;
                file->Started = FALSE;
                file->Starting = FALSE;
                file->Complete = FALSE;
                file->GotHeader = FALSE;
                file->StartTime = time(NULL);
                file->TotalSize = SIZE_UNKNOWN;
                strcpy(file->Activity, "Transferring file");
                // use the last mirror
                file->NextMirror--;
                if (file->NextMirror < -1)
                    file->NextMirror = -1;
                Batch.Complete = FALSE;
                // make sure we do a reget from the start
                if (!strchr(file->Flags, 'b'))
                    strcat(file->Flags, "b");
                sprintf(resp_buf, "%d File regot OK\n", DX_OK);
                write(client->Socket, resp_buf, strlen(resp_buf) + 1);
                WakeUpNow = TRUE;
                return TRUE;
            }
            file = file->Next;
        }
        if (AddFileToBatch(TRUE, cmd, client))
            sprintf(resp_buf, "%d File regot OK\n", DX_OK);
        else
            sprintf(resp_buf, "%d Couldn't reget file\n", DX_ERROR);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else if (!strncasecmp(cmd, "Get", 3))
    {
        if ((strlen(cmd) < 5) || (*(cmd + 3) != ' '))
        {
            sprintf(resp_buf, "%d Usage: Get <URL>\n", DX_E_BAD_COMMAND);
            write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            return TRUE;
        }
        if (AddFileToBatch(TRUE, cmd + 4, client))
            sprintf(resp_buf, "%d File added to batch OK\n", DX_OK);
        else
            sprintf(resp_buf, "%d Couldn't add file to batch\n", DX_ERROR);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    else
    {
        //error(E_WARN, "Invalid command \"%s\" from client \"%s\"", cmd,
        //      client->Name);
        sprintf(resp_buf, "%d Invalid command \"%s\"\n",
                DX_E_BAD_COMMAND, cmd);
        write(client->Socket, resp_buf, strlen(resp_buf) + 1);
    }
    return TRUE;
}

void GetClientInfo(char *info_buffer, int buffer_length)
{

}
