// GTK-based Darxite FTP client

#include <sys/stat.h>

#include "global.h"
#include "commands.h"
#include "connect.h"
#include "dialogues.h"
#include "main.h"
#include "xpm.h"

#define OPTION_STRING "vhc:"

GdkColor Blue = {0, 0x0000, 0x0000, 0x8000};
GdkColor Red = {0, 0xFFFF, 0x0000, 0x0000};
GdkColor Cyan = {0, 0x0000, 0x8000, 0x8000};
GdkColor Green = {0, 0x0000, 0x8000, 0x0000};

// A structure containing the DnD information and supporting enum

enum _dndTypes {
   TARGET_URI_LIST = 0,
   TARGET_NETSCAPE_URL
};
static GtkTargetEntry DnDTypes [] = {
    { "text/uri-list", 0, TARGET_URI_LIST },
    { "x-url/http",    0, TARGET_NETSCAPE_URL },
    { "x-url/ftp",     0, TARGET_NETSCAPE_URL },
    { "_NETSCAPE_URL", 0, TARGET_NETSCAPE_URL },
};
static gint DnDTypeCount = sizeof(DnDTypes) / sizeof(DnDTypes[0]);

static char LastDnDUrl[256];
static char *DsgtkPath, *DxprefPath;
static GList *NextSelection;
static BOOL GetPaused;

char *ListTitles[] = {"", "Name", "Size", "Date", "Permissions"};
int IdleTimeout;

void CreateMenus(void);
void CreateFrames(void);

/* "Hook" functions called by GTK */

void lister_clicked(GtkWidget *widget, gint row, gint column,
                    GdkEventButton *event)
{
    int lister_id;
    char *file, buffer[256];
    GList *selection = GTK_CLIST(widget)->selection;

    //printf("%d %d\n", (int)selection, (int)event);
    // only do stuff on double clicks
    if (widget == Lister[REMOTE])
        SelectedRow[REMOTE] = row;
    else
        SelectedRow[LOCAL] = row;
    if (!event || (event->type != GDK_2BUTTON_PRESS))
        return;

    if (widget == Lister[REMOTE])
    {
        lister_id = REMOTE;
        if (LinkId[row] > -1)
        {
            file = LinksTo[LinkId[row]];
            RemoteCwd(File[REMOTE][row]);
        }
        else
        {
            file = File[REMOTE][row];
            //printf("file: \"%s\"\n", file);
            if ((file[strlen(file) - 1] == '/') || !strcmp(file, "..")
                || !strcmp(file, "."))
            {
                RemoteCwd(file);
            }
            else // it's a file, not a directory
            {
                while (selection)
                {
                    file = File[REMOTE][(int)(selection->data)];
                    GetFile(file, FALSE, FALSE, FALSE);
                    selection = selection->next;
                }
            }
        }
    }
    else
    {
        lister_id = LOCAL;
        file = File[LOCAL][row];
        if ((file[strlen(file) - 1] == '/') || !strcmp(file, "..")
            || !strcmp(file, "."))
        {
            LocalCwd(file);
        }
        else
        {
            while (selection)
            {
                file = File[LOCAL][(int)(selection->data)];
                sprintf(buffer, "%s%s", CurrentPath[LOCAL], file);
                if (strchr(FilePerms[LOCAL][row], 'x'))
                {
                    strcat(buffer, " &");
                    system(buffer);
                }
                else
                {
                    ViewFile(buffer);
                }
                selection = selection->next;
            }
        }
    }
}

// called on exit
void close_window(void)
{
    int i;
    char buffer[2048];
    static BOOL response_wait = FALSE;

    if (!response_wait && Connected)
    {
        sprintf(buffer, "Close %s\n", HostName);
        response_wait = TRUE;
        SetResponse(RESP_FUNC(close_window));
        write(DaemonFd, buffer, strlen(buffer));
    }
    else
    {
        if (Connected)
            DX_GetResponse(DaemonFd, buffer, sizeof(buffer), RESP_TIMEOUT);
        DX_DisconnectClient(DaemonFd);
        for (i = 0; i < ServerCount; i++)
        {
            free(ServHost[i]);
            free(ServPath[i]);
            free(ServLogin[i]);
            free(ServPassword[i]);
            free(ServAlias[i]);
            free(ServComment[i]);
        }
        gtk_main_quit();
    }
}

void change_cwd(GtkWidget *widget, GtkWidget *entry)
{
    gchar *entry_text;
  
    entry_text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (entry == Pwd[REMOTE])
        RemoteCwd(entry_text);
    else
        LocalCwd(entry_text);
}

static void open_connect_dialogue(GtkWidget *widget,
                                  GdkEvent *event, gpointer data)
{
    gtk_widget_show(ConnectDialogue);
    gtk_widget_grab_focus(ConnectEntry);
}

// "data" is whether to add it paused or not
static void get_file_ok(int button_clicked)
{
    char file_buf[256], buffer[256];
    BOOL overwrite = FALSE;
    struct stat stat_buf;
    int rc;
    
    if (button_clicked == 1)
        overwrite = TRUE;
    if (button_clicked != 2)
    {
        if (LinkId[(int)(NextSelection->data)] > -1)
        {
            GetFile(LinksTo[LinkId[(int)(NextSelection->data)]], FALSE,
                    GetPaused, overwrite);
        }
        else
        {
            if (*FilePerms[REMOTE][(int)(NextSelection->data)] == 'd')
            {
                GetFile(File[REMOTE][(int)(NextSelection->data)], TRUE,
                        GetPaused, overwrite);
            }
            else
            {
                GetFile(File[REMOTE][(int)(NextSelection->data)], FALSE,
                        GetPaused, overwrite);
            }
        }
    }
    
    // get the next file in the selection
    NextSelection = NextSelection->next;
    if (NextSelection)
    {
        sprintf(file_buf, "%s%s", CurrentPath[LOCAL],
                File[REMOTE][(int)NextSelection->data]);
        rc = stat(file_buf, &stat_buf);
        if ((!DX_EnableSpooling) && (rc != -1))
        {
            sprintf(buffer, "File \"%s\" already exists. Do you want to "
                    "overwrite it?", file_buf);
            YesNoCancelShow(get_file_ok, buffer, "Append", "Overwrite");
        }
        else
        {
            get_file_ok(0);
        }
    }
}

static void get_file(GtkWidget *widget, gpointer data)
{
    char file_buf[256], msg_buf[256];
    struct stat stat_buf;
    int rc;
    
    GetPaused = (int)data;
    
    NextSelection = GTK_CLIST(Lister[REMOTE])->selection;
    if (NextSelection == NULL)
        return;
    
    sprintf(file_buf, "%s%s", CurrentPath[LOCAL],
            File[REMOTE][(int)NextSelection->data]);
    rc = stat(file_buf, &stat_buf);
    if ((!DX_EnableSpooling) && (rc != -1))
    {
        sprintf(msg_buf, "File \"%s\" already exists. Do you want to "
                "overwrite it?", file_buf);
        YesNoCancelShow(get_file_ok, msg_buf, "Append", "Overwrite");
    }
    else
    {
        get_file_ok(0);
    }
}

static void put_file_ok(int button_clicked)
{
    char buffer[256];
    BOOL overwrite = FALSE;
    int i;
    
    if (button_clicked == 1)
        overwrite = TRUE;
    if (button_clicked != 2)
    {
        if (*FilePerms[LOCAL][(int)(NextSelection->data)] == 'd')
        {
            PutFile(File[LOCAL][(int)(NextSelection->data)], TRUE,
                    FALSE, overwrite);
        }
        else
        {
            PutFile(File[LOCAL][(int)(NextSelection->data)], FALSE,
                    FALSE, overwrite);
        }
    }
    
    // put the next file in the selection
    NextSelection = NextSelection->next;
    if (NextSelection)
    {
        for (i = 0; i < FileCount[REMOTE]; i++)
        {
            if (!strcmp(File[LOCAL][(int)NextSelection->data],
                        File[REMOTE][i]))
            {
                sprintf(buffer, "File \"%s\" already exists. Do you want to "
                        "overwrite it?", File[REMOTE][i]);
                YesNoCancelShow(put_file_ok, buffer, "Append", "Overwrite");
                return;
            }
        }
        put_file_ok(1);
    }
}

// "data" is whether to add it paused or not
static void put_file(GtkWidget *widget, gpointer data)
{
    char buffer[256];
    int i;
    
    GetPaused = (int)data;
    
    NextSelection = GTK_CLIST(Lister[LOCAL])->selection;
    if (NextSelection == NULL)
        return;

    // does the file already exist?
    for (i = 0; i < FileCount[REMOTE]; i++)
    {
        if (!strcmp(File[LOCAL][(int)NextSelection->data],
                    File[REMOTE][i]))
        {
            sprintf(buffer, "File \"%s\" already exists. Do you want to "
                    "overwrite it?", File[REMOTE][i]);
            YesNoCancelShow(put_file_ok, buffer, "Append", "Overwrite");
            return;
        }
    }
    // it doesn't exist, so assume overwrite
    put_file_ok(1);
}

void local_mkdir_ok(const char *name)
{
    if (name == NULL)
        return;

    LocalMkDir(name);
}

void remote_mkdir_ok(const char *name)
{
    if (name == NULL)
        return;

    RemoteMkDir(name);
}

void mk_dir(GtkWidget *widget, gpointer data)
{
    if (data == Lister[LOCAL])
        GetInput(local_mkdir_ok, "Enter directory name", "");
    else
        GetInput(remote_mkdir_ok, "Enter directory name", "");
}

void ls(GtkWidget *widget, gpointer data)
{
    if (data == Lister[LOCAL])
        LocalLs();
    else
        RemoteLs();
}

void click_col(GtkWidget *widget, gint column, gpointer data)
{
    if (column == 0)
        ls(widget, data);
}

void cwd(GtkWidget *widget, gpointer data)
{
    if (data == Lister[LOCAL])
    {
        if (SelectedRow[LOCAL] > -1)
            LocalCwd(File[LOCAL][SelectedRow[LOCAL]]);
    }
    else
    {
        if (SelectedRow[REMOTE] > -1)
        {
            if (LinkId[SelectedRow[REMOTE]] > -1)
                RemoteCwd(LinksTo[LinkId[SelectedRow[REMOTE]]]);
            else
                RemoteCwd(File[REMOTE][SelectedRow[REMOTE]]);
        }
    }
}

void cdup(GtkWidget *widget, gpointer data)
{
    if (data == Lister[LOCAL])
        LocalCwd("..");
    else
        RemoteCwd("..");
}

void local_rename_ok(const char *name)
{
    if (name == NULL)
        return;
    LocalRename(File[LOCAL][SelectedRow[LOCAL]], name);
}

void remote_rename_ok(const char *name)
{
    if (name == NULL)
        return;
    RemoteRename(File[REMOTE][SelectedRow[REMOTE]], name);
}

void file_rename(GtkWidget *widget, gpointer data)
{
    char buffer[256];
    
    if ((data == Lister[LOCAL]) && (SelectedRow[LOCAL] > -1))
    {
        sprintf(buffer, "Rename %s%s to:", CurrentPath[LOCAL],
                File[LOCAL][SelectedRow[LOCAL]]);
        GetInput(local_rename_ok, buffer, File[LOCAL][SelectedRow[LOCAL]]);
    }
    else if (SelectedRow[REMOTE] > -1)
    {
        sprintf(buffer, "Rename %s%s to:", CurrentPath[REMOTE],
                File[REMOTE][SelectedRow[REMOTE]]);
        GetInput(remote_rename_ok, buffer, File[REMOTE][SelectedRow[REMOTE]]);
    }
}

void hidden_local_toggled(void)
{
    ShowHiddenFiles[LOCAL] = !ShowHiddenFiles[LOCAL];
    LocalLs();
}

void hidden_remote_toggled(void)
{
    ShowHiddenFiles[REMOTE] = !ShowHiddenFiles[REMOTE];
    RemoteLs();
}

void ftp_log_toggled(void)
{
    FtpLogVisible = !FtpLogVisible;
    if (InfoText)
    {
        if (FtpLogVisible)
        {
            gtk_widget_show(InfoBox);
        }
        else
        {
            gtk_paned_set_position(GTK_PANED(MainFrame),
                                   LeftRightFrame->allocation.height +
                                   InfoBox->allocation.height);
            gtk_widget_hide(InfoBox);
        }
    }
}

void date_toggled(void)
{
    ShowDate = !ShowDate;
#ifdef GTK_HAVE_FEATURES_1_1_4
    gtk_clist_set_column_visibility(GTK_CLIST(Lister[LOCAL]), 3, ShowDate);
    gtk_clist_set_column_visibility(GTK_CLIST(Lister[REMOTE]), 3, ShowDate);
#endif
}

void permissions_toggled(void)
{
    ShowPermissions = !ShowPermissions;
#ifdef GTK_HAVE_FEATURES_1_1_4
    gtk_clist_set_column_visibility(GTK_CLIST(Lister[LOCAL]), 4,
                                    ShowPermissions);
    gtk_clist_set_column_visibility(GTK_CLIST(Lister[REMOTE]), 4,
                                    ShowPermissions);
#endif
}

void delete(GtkWidget *widget, gpointer data)
{
    char buffer[256];
    
    if (data == Lister[REMOTE])
    {
        if (SelectedRow[REMOTE] > -1)
        {
            sprintf(buffer, "Are you sure you want to delete \"%s\"?",
                    File[REMOTE][SelectedRow[REMOTE]]);
            ConfirmAction(RemoteDelete, buffer);
        }
    }
    else
    {
        if (SelectedRow[LOCAL] > -1)
        {
            sprintf(buffer, "Are you sure you want to delete \"%s\"?",
                    File[LOCAL][SelectedRow[LOCAL]]);
            ConfirmAction(LocalDelete, buffer);
        }
    }
}

void ftp_quote_ok(const char *cmd)
{
    char buffer[2048];
    static BOOL response_wait = FALSE;

    if ((!response_wait) && (cmd == NULL))
        return;
    if (response_wait)
    {
        DX_GetResponse(DaemonFd, buffer, sizeof(buffer), RESP_TIMEOUT);
        AddFtpString(buffer);
        SetResponse(RESP_FUNC(DummyResponse));
        response_wait = FALSE;
    }
    else
    {
        sprintf(buffer, "FTP %s\n", cmd);
        response_wait = TRUE;
        SetResponse(RESP_FUNC(ftp_quote_ok));
        write(DaemonFd, buffer, strlen(buffer));
    }
}

void ftp_quote(void)
{
    GetInput(ftp_quote_ok, "Enter FTP command to send", "");
}

void anti_idle(void)
{
    char buffer[256];
    static BOOL wait_response = FALSE;

    if (!wait_response)
    {
        if (!Connected)
            return;
        wait_response = TRUE;
        SetResponse(RESP_FUNC(anti_idle));
        strcpy(buffer, "FTP NOOP\n");
        write(DaemonFd, buffer, strlen(buffer));
    }
    else
    {
        DX_GetResponse(DaemonFd, buffer, sizeof(buffer), RESP_TIMEOUT);
        AddFtpString(buffer);
        wait_response = FALSE;
    }
}

void anti_idle_toggled(void)
{
    AntiIdle = !AntiIdle;
    if (AntiIdle)
    {
        // send a NOOP every 15 minutes
        IdleTimeout = gtk_timeout_add(1000 * 60 * 4, (GtkFunction) anti_idle,
                                      NULL);
    }
    else
    {
        gtk_timeout_remove(IdleTimeout);
    }
}

void view_file(void)
{
    char buffer[256];
    
    if (SelectedRow[LOCAL] > -1)
    {
        sprintf(buffer, "%s%s", CurrentPath[LOCAL],
                File[LOCAL][SelectedRow[LOCAL]]);
        ViewFile(buffer);
    }
}

void open_file_ok(const char *program)
{
    char buffer[256];
    
    if (program == NULL)
        return;

    if (strlen(program) > 0)
        sprintf(buffer, "%s %s &", program, File[LOCAL][SelectedRow[LOCAL]]);
    else
        sprintf(buffer, "%s &", File[LOCAL][SelectedRow[LOCAL]]);
    system(buffer);
}

void open_file(void)
{
    char *filename, buffer[256];
    int i;

    filename = File[LOCAL][SelectedRow[LOCAL]];
    sprintf(buffer, "Open \"%s\" with:", filename);
    for (i = 0; i < DX_FileTypeCount; i++)
    {
        if (!strcmp(filename + strlen(filename) - strlen(DX_FileType[0][i]),
                   DX_FileType[0][i]))
        {
            GetInput(open_file_ok, buffer, DX_FileType[1][i]);
            return;
        }
    }
    GetInput(open_file_ok, buffer, "");
}

void run_dsgtk(void)
{
    char buffer[256];

    if (DsgtkPath != NULL)
    {
        sprintf(buffer, "%s &", DsgtkPath);
        if (system(buffer) != 0)
            AddFtpString(strerror(errno));
    }
}

void run_dxpref(void)
{
    char buffer[256];

    if (DxprefPath != NULL)
    {
        sprintf(buffer, "%s &", DxprefPath);
        if (system(buffer) != 0)
            AddFtpString(strerror(errno));
    }
}

void drag_data_done(int action)
{
    char buffer[256];
    
    switch (action)
    {
    case 0:
        sprintf(buffer, "darxget \"%s\" &", LastDnDUrl);
        if (system(buffer) != 0)
            AddFtpString(strerror(errno));
        break;
        
    case 1:
        AddFtpString("Opening path\n");
        // if there's no path
        if (strrchr(LastDnDUrl, '/') <= (LastDnDUrl + 5))
        {
            OpenUrl(LastDnDUrl, "anonymous", DX_EMailAddress);
        }
        else
        {
            memset(buffer, 0, sizeof(buffer));
            strncpy(buffer, LastDnDUrl,
                    (int)(strrchr(LastDnDUrl, '/') - LastDnDUrl));
            OpenUrl(buffer, "anonymous", DX_EMailAddress);
        }
        break;
        
    case 2:
        AddFtpString("Cancelled!\n");
        break;
    }
}

void drag_upload_done(void)
{
    PutFile(LastDnDUrl, FALSE, FALSE, TRUE);
}

void drag_data_received(GtkWidget *widget, GdkDragContext *context,
                         gint x, gint y, GtkSelectionData *selection_data,
                         guint drag_type, guint time, gpointer data)
{
    char *url;
    char url_buf[256], buffer[256];
    
    switch (drag_type)
    {
        // I think we can treat these both the same way...
    case TARGET_URI_LIST:
    case TARGET_NETSCAPE_URL:
        url = selection_data->data;
        // go through all URLs we were sent
        do
        {
            memset(url_buf, 0, sizeof(url_buf));
            if (strchr(url, '\n') != NULL)
                strncpy(url_buf, url, strchr(url, '\n') - url);
            else
                strcpy(url_buf, url);
            // we may get blank URLs due to gmc sticking a newline at the end
            if (strcmp(url_buf, ""))
            {
                // compensate for gmc weirdnesses
                if (!strncmp(url_buf, "file:/#ftp:", 11))
                {
                    sprintf(buffer, "ftp://%s", url_buf + 11);
                    // gmc for some reason sticks a space at the end
                    buffer[strlen(buffer) - 1] = '\0';
                    strcpy(url_buf, buffer);
                }
                else if (!strncmp(url_buf, "file:/#http:", 12))
                {
                    sprintf(buffer, "http://%s", url_buf + 12);
                    buffer[strlen(buffer) - 1] = '\0';
                    strcpy(url_buf, buffer);
                }
                // check for a gmc-style "file:/" url
                else if (!strncmp(url_buf, "file:/", 6) && (url_buf[6] != '/'))
                {
                    sprintf(buffer, "file:///%s", url_buf + 6);
                    buffer[strlen(buffer) - 1] = '\0';
                    strcpy(url_buf, buffer);
                }
                DX_StripControlChars(url_buf);
                //sprintf(buffer, "URL: \"%s\"\n", url_buf);
                //AddFtpString(buffer);
                if (widget == Lister[LOCAL])
                {
                    // if it's a directory, then go to it immediately
                    if (url_buf[strlen(url_buf) - 1] == '/')
                    {
                        OpenUrl(url_buf, "anonymous", DX_EMailAddress);
                    }
                    else
                    {
                        sprintf(buffer, "Would you like to download \"%s\"\n"
                                "immediately or to open its parent directory?",
                                url_buf);
                        strcpy(LastDnDUrl, url_buf);
                        YesNoCancelShow(drag_data_done, buffer, "Download",
                                        "Open parent");
                    }
                }
                else
                {
                    if (!Connected)
                    {
                        AddFtpString("You must be connected to upload a "
                                     "file.\n");
                    }
                    else if (!strncmp(url_buf, "file://", 7))
                    {
                        strcpy(buffer, url_buf + 7);
                        strcpy(url_buf, buffer);
                        sprintf(buffer,
                                "Are you sure you want to upload \"%s\"?",
                                url_buf);
                        strcpy(LastDnDUrl, url_buf);
                        ConfirmAction(drag_upload_done, buffer);
                    }
                    else
                    {
                        AddFtpString("To download files, drop them on the "
                                     "local directory list box.\n");
                    }
                }
            }
            url = strchr(url, '\n');
            if (url != NULL)
                url++;
        } while (url != NULL);
        break;

    default:
        sprintf(buffer, "Received unsupported DnD type %d", drag_type);
        AddFtpString(buffer);
        break;
    }
}

/* User-callable functions */

void DummyResponse(void)
{
    char buffer[2048];

    DX_GetResponse(DaemonFd, buffer, sizeof(buffer), RESP_TIMEOUT);
    //printf("Response: %s", buffer);
}

void SetResponse(GdkInputFunction function)
{
    if (ResponseId > 0)
        gdk_input_remove(ResponseId);
    ResponseId = gdk_input_add(DaemonFd, GDK_INPUT_READ, function, NULL);
}

void AddFtpString(const char *string)
{
    char *buf_ptr, buffer[2048], line_buf[256];
    BOOL first_time = TRUE;

    strcpy(buffer, string);
    buf_ptr = strtok(buffer, "\n");
    while (buf_ptr)
    {
        if ((atoi(buf_ptr) >= 100) && (atoi(buf_ptr) < 1000))
        {
            snprintf(line_buf, 5, "%s ", buffer);
            if (((atoi(line_buf) >= 400) && (atoi(line_buf) < 900)) ||
                (atoi(line_buf) >= DX_ERROR))
            {
                gtk_text_insert(GTK_TEXT(InfoText),
                                NULL, &Red, NULL, line_buf,
                                strlen(line_buf) + 1);
            }
            else
            {
                gtk_text_insert(GTK_TEXT(InfoText),
                                NULL, &Blue, NULL, line_buf,
                                strlen(line_buf) + 1);
            }
            buf_ptr += 4;
        }
        if (first_time && (string[strlen(string) - 1] != '\n'))
            sprintf(line_buf, "%s", buf_ptr);
        else
            sprintf(line_buf, "%s\n", buf_ptr);
        gtk_text_insert(GTK_TEXT(InfoText), NULL, NULL, NULL, line_buf,
                        strlen(line_buf) + 1);
        buf_ptr = strtok(NULL, "\n");
        first_time = FALSE;
    }
}

void AddSeparator(GtkWidget *menu)
{
    GtkWidget *separator;
    
    separator = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(menu), separator);
    gtk_widget_set_sensitive(separator, FALSE);
    gtk_widget_show(separator);
}

void CreateMenus(void)
{
    DX_ClientInfo *found_clients;
    int rc;
    
    // Create the actual menus
    MainMenu = gtk_menu_new();
    FtpMenu = gtk_menu_new();
    ViewMenu = gtk_menu_new();
    
    ConnectMenuItem = gtk_menu_item_new_with_label("Connect...");
    gtk_menu_append(GTK_MENU(MainMenu), ConnectMenuItem);
    gtk_signal_connect(GTK_OBJECT(ConnectMenuItem), "activate",
			GTK_SIGNAL_FUNC(open_connect_dialogue), NULL);
    gtk_widget_show(ConnectMenuItem);
    
    DisconnectMenuItem = gtk_menu_item_new_with_label("Disconnect");
    gtk_menu_append(GTK_MENU(MainMenu), DisconnectMenuItem);
    gtk_signal_connect(GTK_OBJECT(DisconnectMenuItem), "activate",
			GTK_SIGNAL_FUNC(CloseHost), NULL);
    gtk_widget_show(DisconnectMenuItem);
    
    AddSeparator(MainMenu);
    
    // figure out where the darxstat-GTK executable is
    rc = DX_FindClients("darxstat-GTK", "gtk", ClientList, &found_clients);
    if (rc < 1)
        printf("Couldn't find Darxstat-GTK\n");
    else
        DsgtkPath = strdup(found_clients->Path);
    DX_FreeClientList(found_clients);
    
    StatusMenuItem = gtk_menu_item_new_with_label("Batch status...");
    gtk_menu_append(GTK_MENU(MainMenu), StatusMenuItem);
    gtk_signal_connect(GTK_OBJECT(StatusMenuItem), "activate",
			GTK_SIGNAL_FUNC(run_dsgtk), NULL);
    gtk_widget_show(StatusMenuItem);
    
    // Find the dxpref executable
    rc = DX_FindClients("Control Centre", "gtk", ClientList, &found_clients);
    if (rc < 1)
        printf("Couldn't find Control Centre\n");
    else
        DxprefPath = strdup(found_clients->Path);
    DX_FreeClientList(found_clients);
    
    PrefsMenuItem = gtk_menu_item_new_with_label("Preferences...");
    gtk_menu_append(GTK_MENU(MainMenu), PrefsMenuItem);
    gtk_signal_connect(GTK_OBJECT(PrefsMenuItem), "activate",
			GTK_SIGNAL_FUNC(run_dxpref), NULL);
    gtk_widget_show(PrefsMenuItem);
    
    AddSeparator(MainMenu);
    
    QuitMenuItem = gtk_menu_item_new_with_label("Quit");
    gtk_menu_append(GTK_MENU(MainMenu), QuitMenuItem);
    gtk_signal_connect(GTK_OBJECT(QuitMenuItem), "activate",
			GTK_SIGNAL_FUNC(close_window), NULL);
    gtk_widget_show(QuitMenuItem);

    MainMenuItem = gtk_menu_item_new_with_label("Main");
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(MainMenuItem), MainMenu);
    gtk_widget_show(MainMenuItem);

    QuoteMenuItem = gtk_menu_item_new_with_label("Quote...");
    gtk_menu_append(GTK_MENU(FtpMenu), QuoteMenuItem);
    gtk_signal_connect(GTK_OBJECT(QuoteMenuItem), "activate",
			GTK_SIGNAL_FUNC(ftp_quote), NULL);
    gtk_widget_show(QuoteMenuItem);

    AntiIdleMenuItem = gtk_check_menu_item_new_with_label("Anti-Idle");
    gtk_menu_append(GTK_MENU(FtpMenu), AntiIdleMenuItem);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(AntiIdleMenuItem),
                                   TRUE);
    gtk_signal_connect(GTK_OBJECT(AntiIdleMenuItem), "activate",
			GTK_SIGNAL_FUNC(anti_idle_toggled), NULL);
    gtk_widget_show(AntiIdleMenuItem);
    anti_idle_toggled();

    FtpMenuItem = gtk_menu_item_new_with_label("FTP");
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(FtpMenuItem), FtpMenu);
    gtk_widget_show(FtpMenuItem);

#ifdef GTK_HAVE_FEATURES_1_1_4
    DateMenuItem = gtk_check_menu_item_new_with_label("File date");
    gtk_menu_append(GTK_MENU(ViewMenu), DateMenuItem);
    gtk_signal_connect(GTK_OBJECT(DateMenuItem), "toggled",
                       GTK_SIGNAL_FUNC(date_toggled), NULL);
    gtk_widget_show(DateMenuItem);

    PermissionsMenuItem =
        gtk_check_menu_item_new_with_label("File permissions");
    gtk_menu_append(GTK_MENU(ViewMenu), PermissionsMenuItem);
    gtk_signal_connect(GTK_OBJECT(PermissionsMenuItem), "toggled",
                       GTK_SIGNAL_FUNC(permissions_toggled), NULL);
    gtk_widget_show(PermissionsMenuItem);

    AddSeparator(ViewMenu);
#endif
    
    HiddenLocalMenuItem =
        gtk_check_menu_item_new_with_label("Hidden local files");
    gtk_menu_append(GTK_MENU(ViewMenu), HiddenLocalMenuItem);
    gtk_signal_connect(GTK_OBJECT(HiddenLocalMenuItem), "toggled",
                       GTK_SIGNAL_FUNC(hidden_local_toggled), NULL);
    gtk_widget_show(HiddenLocalMenuItem);

    HiddenRemoteMenuItem =
        gtk_check_menu_item_new_with_label("Hidden remote files");
    gtk_menu_append(GTK_MENU(ViewMenu), HiddenRemoteMenuItem);
    gtk_signal_connect(GTK_OBJECT(HiddenRemoteMenuItem), "toggled",
                       GTK_SIGNAL_FUNC(hidden_remote_toggled), NULL);
    gtk_widget_show(HiddenRemoteMenuItem);

    AddSeparator(ViewMenu);
    
    FtpLogMenuItem =
        gtk_check_menu_item_new_with_label("FTP log");
    gtk_menu_append(GTK_MENU(ViewMenu), FtpLogMenuItem);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(FtpLogMenuItem), TRUE);
    gtk_signal_connect(GTK_OBJECT(FtpLogMenuItem), "toggled",
                       GTK_SIGNAL_FUNC(ftp_log_toggled), NULL);
                       gtk_widget_show(FtpLogMenuItem);
    
    ViewMenuItem = gtk_menu_item_new_with_label("View");
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(ViewMenuItem), ViewMenu);
    gtk_widget_show(ViewMenuItem);
    
    // Finally, create the menu bar and attach the menus to it
    MenuBar = gtk_menu_bar_new();
    gtk_box_pack_start(GTK_BOX(MenuFrame), MenuBar, FALSE, FALSE, 0);
    gtk_menu_bar_append(GTK_MENU_BAR(MenuBar), MainMenuItem);
    gtk_menu_bar_append(GTK_MENU_BAR(MenuBar), FtpMenuItem);
    gtk_menu_bar_append(GTK_MENU_BAR(MenuBar), ViewMenuItem);
    gtk_widget_show(MenuBar);
}

void CreateFrames(void)
{
    int i, j, k;

    for (i = 0; i < 2; i++)
    {
        Frame[i] = gtk_vbox_new(FALSE, 0);
        if (i == 0)
            gtk_paned_add1(GTK_PANED(LeftRightFrame), Frame[i]);
        else
            gtk_paned_add2(GTK_PANED(LeftRightFrame), Frame[i]);
        gtk_widget_show(Frame[i]);
        
        PwdBox[i] = gtk_hbox_new(FALSE, 5);
        gtk_box_pack_start(GTK_BOX(Frame[i]), PwdBox[i], FALSE, FALSE, 2);
        gtk_widget_show(PwdBox[i]);

        if (i == REMOTE)
            PwdLabel[i] = gtk_label_new("Remote dir:");
        else
            PwdLabel[i] = gtk_label_new("Local dir:");
        gtk_box_pack_start(GTK_BOX(PwdBox[i]), PwdLabel[i], FALSE, FALSE, 0);
        gtk_widget_show(PwdLabel[i]);
    
        Pwd[i] = gtk_entry_new();
        gtk_box_pack_start(GTK_BOX(PwdBox[i]), Pwd[i], TRUE, TRUE, 0);
        gtk_signal_connect(GTK_OBJECT(Pwd[i]), "activate",
                           GTK_SIGNAL_FUNC(change_cwd), Pwd[i]);
        gtk_widget_show(Pwd[i]);
        
        // necessary to be able to scroll in the listers
        ScrolledWindow[i] = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrolledWindow[i]),
                                       GTK_POLICY_AUTOMATIC,
                                       GTK_POLICY_AUTOMATIC);
        gtk_widget_set_usize(ScrolledWindow[i], 250, 100);
        gtk_box_pack_start(GTK_BOX(Frame[i]), ScrolledWindow[i],
                           TRUE, TRUE, 5);
        gtk_widget_show(ScrolledWindow[i]);
        
        Lister[i] = gtk_clist_new_with_titles(5, ListTitles);
        gtk_drag_dest_set(Lister[i], GTK_DEST_DEFAULT_ALL,
                          DnDTypes, DnDTypeCount, GDK_ACTION_COPY);
        gtk_signal_connect(GTK_OBJECT(Lister[i]), "drag_data_received",
                           GTK_SIGNAL_FUNC(drag_data_received), NULL);
        gtk_clist_set_selection_mode(GTK_CLIST(Lister[i]),
                                     GTK_SELECTION_EXTENDED);
        gtk_clist_set_column_width(GTK_CLIST(Lister[i]), 0, 16);
        gtk_clist_set_column_width(GTK_CLIST(Lister[i]), 1, 150);
        gtk_clist_set_column_width(GTK_CLIST(Lister[i]), 2, 50);
        gtk_clist_set_column_width(GTK_CLIST(Lister[i]), 3, 75);
        gtk_clist_set_column_width(GTK_CLIST(Lister[i]), 4, 75);
        
#ifdef GTK_HAVE_FEATURES_1_1_4
        gtk_clist_set_column_visibility(GTK_CLIST(Lister[i]), 3, FALSE);
        gtk_clist_set_column_visibility(GTK_CLIST(Lister[i]), 4, FALSE);
#endif
        gtk_container_add(GTK_CONTAINER(ScrolledWindow[i]), Lister[i]);
        gtk_signal_connect(GTK_OBJECT(Lister[i]), "select_row",
                           GTK_SIGNAL_FUNC(lister_clicked), NULL);
        gtk_signal_connect(GTK_OBJECT(Lister[i]), "click_column",
                           GTK_SIGNAL_FUNC(click_col), (gpointer) Lister[i]);
        if (i == REMOTE)
            gtk_widget_set_sensitive(Lister[i], FALSE);
        gtk_widget_show(Lister[i]);
        SelectedRow[i] = -1;
        
        if (i == REMOTE)
        {
            CommandButton[i][0][0] = gtk_button_new_with_label("Get");
            gtk_signal_connect(GTK_OBJECT(CommandButton[i][0][0]), "clicked",
                               GTK_SIGNAL_FUNC(get_file), (gpointer)FALSE);
            
            CommandButton[i][0][1] = gtk_button_new_with_label("Queue");
            gtk_signal_connect(GTK_OBJECT(CommandButton[i][0][1]), "clicked",
                               GTK_SIGNAL_FUNC(get_file), (gpointer)TRUE);
        }
        else
        {
            
            CommandButton[i][0][0] = gtk_button_new_with_label("Put");
            gtk_signal_connect(GTK_OBJECT(CommandButton[i][0][0]), "clicked",
                               GTK_SIGNAL_FUNC(put_file), NULL);
            
            CommandButton[i][0][1] = gtk_button_new_with_label("Open...");
            gtk_signal_connect(GTK_OBJECT(CommandButton[i][0][1]), "clicked",
                               GTK_SIGNAL_FUNC(open_file), NULL);
        }
        CommandButton[i][0][2] = gtk_button_new_with_label("ChDir");
        gtk_signal_connect(GTK_OBJECT(CommandButton[i][0][2]), "clicked",
                           GTK_SIGNAL_FUNC(cwd), Lister[i]);
        CommandButton[i][0][3] = gtk_button_new_with_label("Parent");
        gtk_signal_connect(GTK_OBJECT(CommandButton[i][0][3]), "clicked",
                           GTK_SIGNAL_FUNC(cdup), Lister[i]);
        if (i == REMOTE)
        {
            CommandButton[i][1][0] = gtk_button_new_with_label("List");
            gtk_signal_connect(GTK_OBJECT(CommandButton[i][1][0]), "clicked",
                               GTK_SIGNAL_FUNC(ls), Lister[i]);
        }
        else
        {
            CommandButton[i][1][0] = gtk_button_new_with_label("View");
            gtk_signal_connect(GTK_OBJECT(CommandButton[i][1][0]), "clicked",
                               GTK_SIGNAL_FUNC(view_file), NULL);
        }
        CommandButton[i][1][1] = gtk_button_new_with_label("Rename");
        gtk_signal_connect(GTK_OBJECT(CommandButton[i][1][1]), "clicked",
                           GTK_SIGNAL_FUNC(file_rename), Lister[i]);
        CommandButton[i][1][2] = gtk_button_new_with_label("Delete");
        gtk_signal_connect(GTK_OBJECT(CommandButton[i][1][2]), "clicked",
                           GTK_SIGNAL_FUNC(delete), Lister[i]);
        CommandButton[i][1][3] = gtk_button_new_with_label("MkDir");
        gtk_signal_connect(GTK_OBJECT(CommandButton[i][1][3]), "clicked",
                           GTK_SIGNAL_FUNC(mk_dir), Lister[i]);
        
        for (j = 0; j < 2; j++)
        {
            ButtonBox[i][j] = gtk_hbox_new(TRUE, 0);
            gtk_box_pack_start(GTK_BOX(Frame[i]), ButtonBox[i][j],
                               FALSE, FALSE, 0);
            gtk_widget_show(ButtonBox[i][j]);
            for (k = 0; k < 4; k++)
            {
                gtk_box_pack_start(GTK_BOX(ButtonBox[i][j]),
                                   CommandButton[i][j][k], TRUE, TRUE, 0);
                gtk_widget_show(CommandButton[i][j][k]);
            }
        }
    }
    gtk_widget_set_sensitive(ButtonBox[REMOTE][0], FALSE);
    gtk_widget_set_sensitive(ButtonBox[REMOTE][1], FALSE);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "connect",	1, 0, 'c' },
        { 0, 			0, 0,  0  }
    };

    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'v':
            printf("NoctFTP version %s, release %s (%s) by Ashley Montanaro\n",
                   VERSION, RELEASE_VER, RELEASE_NAME);
            exit(0);
            break;

        case 'h':
            printf("Usage: %s [OPTIONS] [URL]\n"
                   "where options are:\n"
                   "-v, --version\t\t\tShow version and exit\n"
                   "-h, --help\t\t\tShow some usage information\n"
                   "-c<host>:<port> --connect<host>\tConnect to the daemon on "
                   "<host>, port <port>\n",
                   argv[0]);
            exit(0);
            break;

        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(DaemonHost, 0, sizeof(DaemonHost));
                strncpy(DaemonHost, optarg, strchr(optarg, ':') - optarg);
                DaemonPort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(DaemonHost) == 0) || (DaemonPort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(1);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(1);
            }
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

int main(int argc, char *argv[])
{
    GdkColormap *colourmap;
    GtkStyle *style;
    char *url, buffer[256];
    int i;

    gtk_init(&argc, &argv);
    url = ReadCmdLine(argc, argv);
    
    if (strcmp(DaemonHost, "") && (DaemonPort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", DaemonHost);
        DaemonFd = DX_ConnectRemoteClient(DaemonHost, DaemonPort,
                                          getpass(buffer), "NoctFTP");
    }
    else
    {
        DaemonFd = DX_ConnectClient("NoctFTP");
    }
    
    if (DaemonFd < 0)
    {
        fprintf(stderr, "Couldn't connect to the Darxite daemon. "
                "Maybe it isn't running?\n");
        return 1;
    }
    // use async input for responses from the daemon
    SetResponse(RESP_FUNC(DummyResponse));
    strcpy(HostName, "");
    DX_ReadConfigFiles();
    DX_ReadClientDB(&ClientList);
    
    Window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(Window), "NoctFTP");
    gtk_window_position(GTK_WINDOW(Window), GTK_WIN_POS_CENTER);
    gtk_window_set_policy(GTK_WINDOW(Window), TRUE, TRUE, FALSE);
    gtk_widget_set_usize(Window, 500, 500);
    gtk_signal_connect(GTK_OBJECT (Window), "delete_event",
                       GTK_SIGNAL_FUNC(close_window), NULL);
    
    colourmap = gtk_widget_get_colormap(Window);
    style = gtk_widget_get_style(Window);
    gdk_color_alloc(colourmap, &Blue);
    gdk_color_alloc(colourmap, &Red);
    gdk_color_alloc(colourmap, &Cyan);
    gdk_color_alloc(colourmap, &Green);

    MenuFrame = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(Window), MenuFrame);
    gtk_widget_show(MenuFrame);

    CreateMenus();
    
    MainFrame = gtk_vpaned_new();
    gtk_box_pack_start(GTK_BOX(MenuFrame), MainFrame, TRUE, TRUE, 0);
    gtk_widget_show(MainFrame);
    
    LeftRightFrame = gtk_hpaned_new();
    gtk_widget_set_usize(LeftRightFrame, 50, 350);
    gtk_paned_add1(GTK_PANED(MainFrame), LeftRightFrame);
    gtk_widget_show(LeftRightFrame);

    /* Create the info text at the bottom */
    FtpLogVisible = TRUE;
    InfoBox = gtk_hbox_new(FALSE, 0);
    gtk_paned_add2(GTK_PANED(MainFrame), InfoBox);
    gtk_widget_show(InfoBox);
    
    InfoText = gtk_text_new(NULL, NULL);
    gtk_box_pack_start(GTK_BOX(InfoBox), InfoText, TRUE, TRUE, 0);
    gtk_widget_show(InfoText);

    InfoScrollBar = gtk_vscrollbar_new(GTK_TEXT(InfoText)->vadj);
    gtk_box_pack_start(GTK_BOX(InfoBox), InfoScrollBar, FALSE, FALSE, 0);
    gtk_widget_show(InfoScrollBar);

    CreateFrames();

    CreateConfirmationDialogue();
    CreateYesNoCancelDialogue();
    CreateConnectDialogue();
    CreateInputDialogue();
    CreateViewDialogue();

    gtk_widget_show(Window);

    AddFtpString("Welcome to NoctFTP version " VERSION
                 ", built for Darxite " RELEASE_VER
                 ".\nBy Ashley Montanaro.\n");

    // this all needs to be done after the window's been shown
    CreatePixmaps();
    gdk_window_set_icon(Window->window, Window->window, MainIcon, NULL);
    for (i = 0; i < 2; i++)
    {
        TitlePixmap[i] = gtk_pixmap_new(SpinIcon, NULL);
        gtk_clist_set_column_widget(GTK_CLIST(Lister[i]), 0, TitlePixmap[i]);
        gtk_widget_show(TitlePixmap[i]);
    }
    
    TopPixmap = gtk_pixmap_new(PenguinIcon, NULL);
    gtk_box_pack_start(GTK_BOX(PwdBox[REMOTE]), TopPixmap, FALSE, FALSE, 0);
    gtk_widget_show(TopPixmap);
    if (LocalCwd(DX_OutputDir) != 0)
        LocalCwd(DX_HomeDir);
    StopOperation();
    if (url)
    {
        OpenUrl(url, "anonymous", DX_EMailAddress);
    }
    else
    {
        gtk_widget_show(ConnectDialogue);
        gtk_widget_grab_focus(ConnectEntry);
    }
    gtk_main();
    
    return 0;
}
