// GTK-based Darxite status window

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <signal.h>

#include <darxite.h>
#include "stop.xpm"
#include "pause.xpm"
#include "play.xpm"
#include "reget.xpm"
#include "remove.xpm"

#define MAX_FILES 	100
#define VERSION 	"0.4"
#define OPTION_STRING	"vhc:"
#define AUTHORS		"Ashley Montanaro"

void play_clicked(GtkWidget *widget, gpointer data);
char *ReadCmdLine(int argc, char **argv);
void PrintVersion(void);
void Usage(char *prog_name);

enum states {
    STATE_RUNNING = 0,
    STATE_PAUSED,
    STATE_DESPOOLING,
    STATE_COMPLETE
};

typedef struct _fileInfo {
    int State;
    char *RemotePath, *LocalPath, *LabelString;
    BOOL StillThere;
    GtkWidget *Label, *Progress;
    GtkWidget *Frame, *SizeLabel;
    GtkWidget *ProgressFrame, *PauseButton, *CancelButton;
    GtkWidget *StopPixmap, *PausePixmap;
    struct _fileInfo *Next;
    struct _fileInfo *Prev;
} FileInfo;

int DaemonFd, RemotePort, FileCount;
char RemoteHost[256];
guint TimeoutId;
BOOL WaitResponse;
GtkWidget *MainFrame, *BgLabel, *Window;
GdkPixmap *StopIcon, *PauseIcon, *PlayIcon, *RegetIcon, *RemoveIcon;
FileInfo *FirstFile, *LastFile;

void PrintVersion(void)
{
    printf("DarxStat-GTK v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTIONS] command\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-c<host>:<port> --connect<host>\tConnect to <host> on <port>\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "connect",	1, 0, 'c' },
        { 0, 			0, 0,  0  }
    };

    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'v':
            PrintVersion();
            exit(0);
            break;

        case 'h':
            Usage(argv[0]);
            break;

        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(RemoteHost, 0, sizeof(RemoteHost));
                strncpy(RemoteHost, optarg, strchr(optarg, ':') - optarg);
                RemotePort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(RemoteHost) == 0) || (RemotePort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(0);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(0);
            }
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

void TruncatePath(char *path)
{
    char buffer[256];
    int last_pos;

    memset(buffer, 0, sizeof(buffer));
    if (strlen(path) > 60)
    {
        // make sure we pick up the filename
        last_pos = strrchr(path, '/') - path;
        if ((last_pos <= 0) || (last_pos >= strlen(path) - 30))
        {
            last_pos = strlen(path) - 30;
            strncpy(buffer, path, 30);
            strcat(buffer, "...");
            strncat(buffer, path + strlen(path) - 30, 30);
        }
        else
        {
            if (strlen(path) - last_pos < 60)
                strncpy(buffer, path, 30 - (strlen(path) - last_pos - 30));
            else
                strncpy(buffer, path, 10);
            strcat(buffer, "...");
            strncat(buffer, path + last_pos, strlen(path) - last_pos);
        }
    }
    else
    {
        strcpy(buffer, path);
    }
    strcpy(path, buffer);
}

void GetStatus(void)
{
    if (!WaitResponse)
    {
        if (write(DaemonFd, "FileStatus\n", 11) < 0)
        {
            if (errno != EPIPE)
                fprintf(stderr, "darxstat-GTK: write: %s", strerror(errno));
            exit(0);
        }
        WaitResponse = TRUE;
    }
}

void cancel_clicked(GtkWidget *widget, gpointer data)
{
    FileInfo *file = (FileInfo *)data;
    char buffer[256];

    switch (file->State)
    {
    case STATE_RUNNING:
    case STATE_PAUSED:
        gtk_window_set_title(GTK_WINDOW(Window),
                             "Darxite Status - cancelling...");
        sprintf(buffer, "Cancel %s\n", file->RemotePath);
        break;
        
    case STATE_COMPLETE:
        gtk_window_set_title(GTK_WINDOW(Window),
                             "Darxite Status - removing...");
        sprintf(buffer, "RemoveCompleteFile %s\n", file->RemotePath);
        break;

    case STATE_DESPOOLING:
        return;
    }
    write(DaemonFd, buffer, strlen(buffer) + 1);
    WaitResponse = TRUE;
}

void pause_clicked(GtkWidget *widget, gpointer data)
{
    FileInfo *file = (FileInfo *)data;
    char buffer[256];

    switch (file->State)
    {
    case STATE_RUNNING:
        gtk_window_set_title(GTK_WINDOW(Window),
                             "Darxite Status - pausing...");
        //gtk_pixmap_set(GTK_PIXMAP(file->PausePixmap), PlayIcon, NULL);
        sprintf(buffer, "Pause %s\n", file->RemotePath);
        break;

    case STATE_PAUSED:
        gtk_window_set_title(GTK_WINDOW(Window),
                             "Darxite Status - resuming...");
        //gtk_pixmap_set(GTK_PIXMAP(file->PausePixmap), PauseIcon, NULL);
        sprintf(buffer, "Continue %s\n", file->RemotePath);
        break;

    case STATE_COMPLETE:
        gtk_window_set_title(GTK_WINDOW(Window),
                             "Darxite Status - regetting...");
        //gtk_pixmap_set(GTK_PIXMAP(file->PausePixmap), RegetIcon, NULL);
        sprintf(buffer, "Reget %s\n", file->RemotePath);
        break;

    case STATE_DESPOOLING:
        return;
    }
    WaitResponse = TRUE;
    write(DaemonFd, buffer, strlen(buffer) + 1);
}

void got_data(gpointer data, gint source, GdkInputCondition condition)
{
    char buffer[512], full_buffer[4096];
    char *line, buffer2[256], path_buffer[256], buffer3[256];
    char local_path[256], id[4];
    int current_size, total_size, total_time, rc;
    FileInfo *file, *last_file;
    BOOL found_it;
    double percentage;

    char protocol[5], server[256], path[256], activity[256];
    int rate_overall;

    full_buffer[sizeof(full_buffer) - 1] = '\0';
    rc = DX_GetResponse(DaemonFd, full_buffer, sizeof(full_buffer) - 1, 10);
    // clear out all the response
    while (rc == DX_LIB_MORE_DATA)
        rc = DX_GetResponse(DaemonFd, buffer, sizeof(buffer), 0);
    if (rc == DX_LIB_ERROR)
        fprintf(stderr, "There was an error getting the daemon's response!\n");
    
    WaitResponse = FALSE;
    // if it was a response to a command, not a status response
    if ((atoi(full_buffer) != 0) && (atoi(full_buffer) != DX_FILE_STATUS))
    {
        gtk_window_set_title(GTK_WINDOW(Window), "Darxite Status");
        GetStatus();
        return;
    }
    file = FirstFile;
    while (file)
    {
        file->StillThere = FALSE;
        file = file->Next;
    }
    FileCount = 0;
    line = strtok(full_buffer, "\n");
    while (line)
    {
        //printf("%s\n", line);
        // skip new error code where necessary
        if (atoi(line) == DX_FILE_STATUS)
            line += 4;
        DX_ParseFileStatus(line, protocol, server, sizeof(server),
                           path, sizeof(path), local_path, sizeof(local_path),
                           &current_size, &total_size, &total_time,
                           &rate_overall, activity, sizeof(activity), id);
        sprintf(path_buffer, "%s://%s%s", protocol, server, path);
        //printf("Read URL %s\n", path_buffer);
        if ((total_size > -1) && (!strcasecmp(activity, "Transferring file")))
        {
            if (total_size > 10 * 1024 * 1024)
            {
                sprintf(buffer2, "%.1f of %.1fMB",
                        (float)current_size / (1024.0 * 1024.0),
                        (float)total_size / (1024.0 * 1024.0));
            }
            else if (total_size > 10 * 1024)
            {
                sprintf(buffer2, "%d of %dK", current_size / 1024,
                        total_size / 1024);
            }
            else
            {
                sprintf(buffer2, "%d of %d bytes", current_size, total_size);
            }
            if (rate_overall > 1024)
            {
                sprintf(buffer3, "at %.1fK/s", (float)rate_overall / 1024.0);
            }
            else
            {
                sprintf(buffer3, "at %d bytes/s", rate_overall);
            }
            strcpy(buffer, path_buffer);
            TruncatePath(buffer);
            strcat(buffer, ": ");
            strcat(buffer, buffer2);
            strcat(buffer, " (");
            strcat(buffer, buffer3);
            strcat(buffer, ")");
        }
        else
        {
            strcpy(buffer, path_buffer);
            TruncatePath(buffer);
            strcat(buffer, ": ");
            strcat(buffer, activity);
        }
        found_it = FALSE;
        file = FirstFile;
        while (file)
        {
            if (!strcmp(local_path, file->LocalPath))
            {
                //printf("Found file %s\n", path_buffer);
                file->StillThere = TRUE;
                if ((current_size > 0) && (total_size > 0))
                {
                    percentage = ((double)current_size /
                                  (double)total_size);
                    if (percentage > 1.0)
                        percentage = 1.0;
                    gtk_progress_bar_update(GTK_PROGRESS_BAR(file->Progress),
                                            percentage);
                }
                else
                {
                    gtk_progress_bar_update(GTK_PROGRESS_BAR(file->Progress),
                                            0);
                }
                
                if (!strcasecmp(activity,"Complete") &&
                    (file->State != STATE_COMPLETE))
                {
                    file->State = STATE_COMPLETE;
                    gtk_pixmap_set(GTK_PIXMAP(file->PausePixmap), RegetIcon,
                                   NULL);
                    gtk_pixmap_set(GTK_PIXMAP(file->StopPixmap), RemoveIcon,
                                   NULL);
                }
                else if (!strcasecmp(activity, "Paused") &&
                    (file->State != STATE_PAUSED))
                {
                    file->State = STATE_PAUSED;
                    gtk_pixmap_set(GTK_PIXMAP(file->PausePixmap), PlayIcon,
                                   NULL);
                    gtk_pixmap_set(GTK_PIXMAP(file->StopPixmap), StopIcon,
                                   NULL);
                }
                else if (!strcasecmp(activity, "Despooling file..."))
                {
                    file->State = STATE_DESPOOLING;
                }
                else if (strcasecmp(activity, "Paused") &&
                         strcasecmp(activity, "Complete") &&
                         (file->State != STATE_RUNNING))
                {
                    file->State = STATE_RUNNING;
                    gtk_pixmap_set(GTK_PIXMAP(file->PausePixmap), PauseIcon,
                                   NULL);
                    gtk_pixmap_set(GTK_PIXMAP(file->StopPixmap), StopIcon,
                                   NULL);
                }
                // do we need to update the label?
                if (strcmp(file->LabelString, buffer))
                {
                    gtk_label_set(GTK_LABEL(file->Label), buffer);
                    free(file->LabelString);
                    file->LabelString = strdup(buffer);
                }
                // has the file's remote path changed?
                if (strcmp(file->RemotePath, path_buffer))
                {
                    free(file->RemotePath);
                    file->RemotePath = strdup(path_buffer);
                }
                found_it = TRUE;
                break;
            }
            file = file->Next;
        }

        // if the file wasn't in the list then add it
        if (!found_it)
        {
            //printf("Adding %s\n", path_buffer);
            if (LastFile == NULL)
            {
                FirstFile = LastFile = malloc(sizeof(FileInfo));
                LastFile->Prev = NULL;
            }
            else
            {
                LastFile->Next = malloc(sizeof(FileInfo));
                LastFile->Next->Prev = LastFile;
                LastFile = LastFile->Next;
            }
            LastFile->Next = NULL;
            // if the background label's there, get rid of it
            if (BgLabel != NULL)
            {
                gtk_widget_destroy(BgLabel);
                BgLabel = NULL;
            }
            LastFile->LabelString = strdup(buffer);
            LastFile->RemotePath = strdup(path_buffer);
            LastFile->LocalPath = strdup(local_path);
            LastFile->Frame = gtk_vbox_new(FALSE, 0);
            gtk_box_pack_start(GTK_BOX(MainFrame), LastFile->Frame,
                               FALSE, FALSE, 0);
                
            LastFile->Label = gtk_label_new(buffer);
            gtk_box_pack_start(GTK_BOX(LastFile->Frame), LastFile->Label,
                               FALSE, FALSE, 0);
            gtk_widget_show(LastFile->Label);
            
            LastFile->ProgressFrame = gtk_hbox_new(FALSE, 0);
            gtk_box_pack_start(GTK_BOX(LastFile->Frame),
                               LastFile->ProgressFrame, FALSE, FALSE, 0);
            gtk_widget_show(LastFile->ProgressFrame);
            
            LastFile->Progress = gtk_progress_bar_new();
            gtk_box_pack_start(GTK_BOX(LastFile->ProgressFrame),
                               LastFile->Progress, TRUE, TRUE, 5);
            if ((current_size > 0) && (total_size > 0))
            {
                percentage = ((double)current_size / (double)total_size);
                if (percentage > 1.0)
                    percentage = 1.0;
                gtk_progress_bar_update(GTK_PROGRESS_BAR(LastFile->Progress),
                                        percentage);
            }
            gtk_widget_show(LastFile->Progress);

            // set the starting pixmap appropriately
            if (!strcasecmp(activity, "Paused"))
            {
                LastFile->State = STATE_PAUSED;
                LastFile->PausePixmap = gtk_pixmap_new(PlayIcon, NULL);
                LastFile->StopPixmap = gtk_pixmap_new(StopIcon, NULL);
            }
            else if (!strcasecmp(activity, "Complete"))
            {
                LastFile->State = STATE_COMPLETE;
                LastFile->PausePixmap = gtk_pixmap_new(RegetIcon, NULL);
                LastFile->StopPixmap = gtk_pixmap_new(RemoveIcon, NULL);
            }
            else
            {
                LastFile->State = STATE_RUNNING;
                LastFile->PausePixmap = gtk_pixmap_new(PauseIcon, NULL);
                LastFile->StopPixmap = gtk_pixmap_new(StopIcon, NULL);
            }
            LastFile->PauseButton = gtk_button_new();
            gtk_container_add(GTK_CONTAINER(LastFile->PauseButton),
                              LastFile->PausePixmap);
            gtk_box_pack_start(GTK_BOX(LastFile->ProgressFrame),
                               LastFile->PauseButton, FALSE, FALSE, 0);
            gtk_signal_connect(GTK_OBJECT(LastFile->PauseButton), "clicked",
                               GTK_SIGNAL_FUNC(pause_clicked),
                               (gpointer)LastFile);
            gtk_widget_show(LastFile->PausePixmap);
            gtk_widget_show(LastFile->PauseButton);
            
            LastFile->CancelButton = gtk_button_new();
            gtk_container_add(GTK_CONTAINER(LastFile->CancelButton),
                              LastFile->StopPixmap);
            gtk_signal_connect(GTK_OBJECT(LastFile->CancelButton), "clicked",
                               GTK_SIGNAL_FUNC(cancel_clicked),
                               (gpointer)LastFile);
            gtk_box_pack_start(GTK_BOX(LastFile->ProgressFrame),
                               LastFile->CancelButton, FALSE, FALSE, 0);
            gtk_widget_show(LastFile->StopPixmap);
            gtk_widget_show(LastFile->CancelButton);
            
            gtk_widget_show(LastFile->Frame);
            LastFile->StillThere = TRUE;
        }
        FileCount++;
        line = strtok(NULL, "\n");
    }
    
    file = FirstFile;
    while (file)
    {
        if (!file->StillThere)
        {
            gtk_widget_destroy(file->Label);
            gtk_widget_destroy(file->Progress);
            gtk_widget_destroy(file->PausePixmap);
            gtk_widget_destroy(file->PauseButton);
            gtk_widget_destroy(file->StopPixmap);
            gtk_widget_destroy(file->CancelButton);
            gtk_widget_destroy(file->ProgressFrame);
            gtk_widget_destroy(file->Frame);
            free(file->RemotePath);
            free(file->LocalPath);
            free(file->LabelString);
            if (file->Next)
                file->Next->Prev = file->Prev;
            else
                LastFile = file->Prev;
            if (file->Prev)
                file->Prev->Next = file->Next;
            else
                FirstFile = file->Next;
            last_file = file;
            file = file->Next;
            free(last_file);
            continue;
        }
        file = file->Next;
    }
    
    if ((FileCount == 0) && (BgLabel == NULL))
    {
        BgLabel = gtk_label_new("There are no files being transferred.");
        gtk_box_pack_start(GTK_BOX(MainFrame), BgLabel, TRUE, TRUE, 0);
        gtk_widget_show(BgLabel);
    }
    // ensure that debug messages are printed
    fflush(stdout);
}

void close_window(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    gtk_timeout_remove(TimeoutId);
    DX_DisconnectClient(DaemonFd);
    gtk_main_quit();
}

gint update(gpointer data)
{
    GetStatus();
    return TRUE;
}

int main(int argc, char *argv[])
{
    GdkBitmap *bitmap;
    GtkStyle *style;
    char buffer[256];
    
    gtk_init(&argc, &argv);
    ReadCmdLine(argc, argv);
    
    signal(SIGPIPE, SIG_IGN);
    if (strcmp(RemoteHost, "") && (RemotePort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", RemoteHost);
        DaemonFd = DX_ConnectRemoteClient(RemoteHost, RemotePort,
                                          getpass(buffer), "darxstat-GTK");
    }
    else
    {
        DaemonFd = DX_ConnectClient("darxstat-GTK");
    }
    
    if (DaemonFd < 0)
    {
        fprintf(stderr, "Couldn't connect to daemon: %s\n",
                strerror(DX_errno));
        return 1;
    }
    
    Window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(Window), "Darxite Status");
    // allow the window to grow or shrink automatically
    gtk_window_set_policy(GTK_WINDOW(Window), FALSE, FALSE, TRUE);
    gtk_window_position(GTK_WINDOW(Window), GTK_WIN_POS_CENTER);
    gtk_container_border_width(GTK_CONTAINER(Window), 10);
    
    /* It's a good idea to do this for all windows. */
    gtk_signal_connect (GTK_OBJECT (Window), "delete_event",
                        GTK_SIGNAL_FUNC(close_window), NULL);
    
    MainFrame = gtk_vbox_new(TRUE, 5);
    gtk_container_add(GTK_CONTAINER(Window), MainFrame);
    
    // use async input for responses from the daemon
    gdk_input_add(DaemonFd, GDK_INPUT_READ, got_data, NULL);
    
    // poll it for status every second
    TimeoutId = gtk_timeout_add(1000, update, NULL);
    GetStatus();
    
    gtk_widget_show(MainFrame);
    gtk_widget_show(Window);
    
    style = gtk_widget_get_style(Window);
    StopIcon = gdk_pixmap_create_from_xpm_d(Window->window, &bitmap,
                                              &style->bg[GTK_STATE_NORMAL],
                                              (gchar **)stop_xpm);
    PauseIcon = gdk_pixmap_create_from_xpm_d(Window->window, &bitmap,
                                              &style->bg[GTK_STATE_NORMAL],
                                              (gchar **)pause_xpm);
    PlayIcon = gdk_pixmap_create_from_xpm_d(Window->window, &bitmap,
                                              &style->bg[GTK_STATE_NORMAL],
                                              (gchar **)play_xpm);
    RegetIcon = gdk_pixmap_create_from_xpm_d(Window->window, &bitmap,
                                              &style->bg[GTK_STATE_NORMAL],
                                              (gchar **)reget_xpm);
    RemoveIcon = gdk_pixmap_create_from_xpm_d(Window->window, &bitmap,
                                              &style->bg[GTK_STATE_NORMAL],
                                              (gchar **)remove_xpm);
    gtk_main();
    
    return 0;
}
