/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

/* AIX requires this to be the first thing in the file.  */
#include <config.h>
#if __GNUC__
#define alloca __builtin_alloca
#else
#if HAVE_ALLOCA_H
#include <alloca.h>
#else
#ifdef _AIX
#pragma alloca
#else
#ifndef alloca			/* predefined by HP cc +Olibcalls */
char *alloca ();
#endif
#endif
#endif
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include "avl.h"
#include "error.h"
#include "getline.h"
#include "settings.h"
#include "lexer.h"
#include "lexerP.h"
#include "som.h"
#include "tab.h"
#include "str.h"
#include "var.h"
#include "common.h"
#include "command.h"
#include "vfm.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

/* Command table. */

typedef struct command
  {
    char cmd[22];		/* command name */
    int transition[4];		/* transitions to make from each state */
    int (*func) (void);		/* function to call */

    char *cmd1;			/* first command word */
    char *cmd2, *cmd3;		/* second, third command words */
    int ncmd;			/* number of command words */

    struct command *next;	/* next command in the chain */
  }
command;

#define INIT STATE_INIT
#define INPU STATE_INPUT
#define TRAN STATE_TRANS
#define PROC STATE_PROC
#define ERRO STATE_ERROR

/* Prototype all the command functions. */
#define DEFCMD(NAME, T1, T2, T3, T4, FUNC) \
	int FUNC (void);
#define UNIMPL(NAME, T1, T2, T3, T4)
#include "command.def"
#undef DEFCMD
#undef UNIMPL

/* Kluge to get a non-constant null string. */
static char empty_string[] = "";

/* Define the command array. */
#define DEFCMD(NAME, T1, T2, T3, T4, FUNC) \
	{NAME, {T1, T2, T3, T4}, FUNC, 0, 0, 0, 0, 0},
#define UNIMPL(NAME, T1, T2, T3, T4) \
	{NAME, {T1, T2, T3, T4}, NULL, 0, 0, 0, 0, 0},
static command cmd_table[] = 
  {
#include "command.def"
    {"", {ERRO, ERRO, ERRO, ERRO}, NULL, empty_string, 0, 0, 0, 0},
  };
#undef DEFCMD
#undef UNIMPL

/* All the entries in cmd_table[].  All the entries with the same
   initial word are linked together in a node in this tree.  The tree
   is index by initial word. */
static avl_tree *cmdtab;

/* A STATE_* constant giving the current program state. */
int pgm_state;

/* The name of the procedure currently executing, if any. */
char *cur_proc;

/* Command parser. */

static int
cmp_command (const void *_a, const void *_b, unused void *foo)
{
  command *a = (command *) _a;
  command *b = (command *) _b;

  if (id_match (b->cmd1, a->cmd1))
    return 0;
  return strcmp (a->cmd1, b->cmd1);
}

static void
split_words (command * c)
{
  char *cp;

  c->cmd2 = c->cmd3 = NULL;
  cp = strpbrk (c->cmd, " -");
  if (cp)
    {
      c->cmd1 = xstrdup (c->cmd);
      cp = (cp - c->cmd) + c->cmd1;
      *cp = 0;
      c->cmd2 = &cp[1];
      cp = strpbrk (c->cmd2, " -");
      if (cp)
	{
	  *cp = 0;
	  c->ncmd = 3;
	  c->cmd3 = &cp[1];
	}
      else
	c->ncmd = 2;
    }
  else
    {
      c->ncmd = 1;
      c->cmd1 = c->cmd;
    }
}

void
init_cmd_parser (void)
{
  command *c;

  cmdtab = avl_create (NULL, cmp_command, NULL);
  for (c = cmd_table; c->cmd[0]; c++)
    split_words (c);
  for (c = cmd_table; c->cmd[0]; )
    {
      command *first;

      first = c;
      while (streq (c[0].cmd1, c[1].cmd1))
	{
	  c->next = &c[1];
	  c++;
	}
      c->next = NULL;
      c++;

      avl_force_insert (cmdtab, first);
    }
}

/* Finds the command with name NAME. */
static command *
find_command (const char *name)
{
  command *cp;

  for (cp = cmd_table; *cp->cmd; cp++)
    if (streq (cp->cmd, name))
      return cp;
  assert (0);
#if __GNUC__ || __BORLANDC__
  return 0;
#endif
}

/* Determines whether command C is appropriate to call in this
   part of a FILE TYPE structure. */
static int
FILE_TYPE_okay (command * c)
{
  static command *RECORD_TYPE_cmd;
  static command *DATA_LIST_cmd;
  static command *REPEATING_DATA_cmd;
  static command *END_FILE_TYPE_cmd;

  if (!RECORD_TYPE_cmd)
    {
      RECORD_TYPE_cmd = find_command ("RECORD TYPE");
      DATA_LIST_cmd = find_command ("DATA LIST");
      REPEATING_DATA_cmd = find_command ("REPEATING DATA");
      END_FILE_TYPE_cmd = find_command ("END FILE TYPE");
    }

  if (c != RECORD_TYPE_cmd && c != DATA_LIST_cmd && c != REPEATING_DATA_cmd
      && c != END_FILE_TYPE_cmd)
    return msg (SE, _("%s is not allowed inside the FILE TYPE/"
		"END FILE TYPE structure."), c->cmd);
  /* FIXME */
#if 0
  if (c == REPEATING_DATA_cmd && fty.type == FTY_GROUPED)
    return msg (SE, _("%s is not allowed inside FILE "
		"TYPE GROUPED/END FILE TYPE."), c->cmd);
  if (!fty.had_rec_type && c != RECORD_TYPE_cmd)
    return msg (SE, _("RECORD TYPE must be the first command inside a "
		"FILE TYPE structure."));
  if (c == RECORD_TYPE_cmd)
    fty.had_rec_type = 1;
#endif
  return 1;
}

/* Parses an entire PSPP command.  This includes everything from the
   command name to the terminating dot.  Does most of its work by
   passing it off to the respective command dispatchers.  Only called by
   main.c:parse(). */
int
parse_cmd (void)
{
  /* A few common error messages. */
  static const char *unk =
    "The identifier(s) specified do not form a valid command name:";
  static const char *inc = 
    "The identifier(s) specified do not form a complete command name:";

  int result;		/* Result returned by command dispatcher. */
  command *cp;		/* Iterator used to find the proper command. */
  char *prev_proc;	/* Used to save & restore cur_proc. */

#if C_ALLOCA
  /* The generic alloca package performs garbage collection when it is
     called with an argument of zero. */
  alloca (0);
#endif /* C_ALLOCA */

  /* Null commands can result from extra empty lines. */
  if (token == '.')
    return 1;

  /* Parse comments. */
  if ((token == ID && id_match ("COMMENT", tokstr))
      || token == EXP || token == '*' || token == '[')
    {
      skip_comment ();
      return 1;
    }

  /* Otherwise the line must begin with a command name, which is
     always an ID token. */
  if (token != ID)
    return msg (SE, _("This line does not begin with a valid command name."));

  /* Parse the INCLUDE short form.  Note that `@' is a valid character
     in identifiers. */
  if (tokstr[0] == '@')
    cp = &cmd_table[0];
  else
    {
      command c;

      c.cmd1 = tokstr;
      cp = avl_find (cmdtab, &c);
    }
  if (!cp)
    return msg (SE, "%s %s.", _(unk), tokstr);

  /* FIXME: I'd eventually like to revise this ugly structure to be
     more elegant, or elegant at all, actually. */
  if (cp->next)
    {
      if (!isalpha ((unsigned char) (lookahead ())))
	{
	  if (cp->cmd2)
	    return msg (SE, "%s %s.", _(inc), tokstr);
	}
      else
	{
	  command *ocp;

	  get_token ();
	  ocp = cp;
	  if (token == ID)
	    for (; cp; cp = cp->next)
	      {
#if 0
		printf (_("trying %s %s.\n"), cp->cmd1, cp->cmd2);
#endif
		if (cp->cmd2 && id_match (cp->cmd2, tokstr))
		  break;
	      }
	  else
	    cp = NULL;

	  if (!cp)
	    {
	      if (ocp->cmd2)
		return msg (SE, "%s %s %s.", _(unk), ocp->cmd1, tokstr);
	      else
		cp = ocp;
	    }
	  else if (!isalpha ((unsigned char) (lookahead ())))
	    {
	      if (cp->cmd3)
		return msg (SE, "%s %s %s.", _(inc), ocp->cmd1, ocp->cmd2);
	    }
	  else if (cp->cmd3
		   || (cp->next && cp->next->cmd3
		       && streq (cp->cmd2, cp->next->cmd2)))
	    {
	      get_token ();

	      ocp = cp;
	      if (token == ID)
		{
		  for (; cp; cp = cp->next)
		    if (cp->cmd3 && streq (cp->cmd2, ocp->cmd2)
			&& id_match (cp->cmd3, tokstr))
		      break;
		}
	      else
		cp = NULL;

	      if (!cp)
		{
		  if (ocp->cmd3)
		    return msg (SE, "%s %s %s %s.", _(unk), ocp->cmd1,
				ocp->cmd2, tokstr);
		  else
		    cp = ocp;
		}
	    }
	  else
	    {
	      /* - Found a match in the id table.
		 - Lookahead is a identifier.
		 - This command can't have a third word.
		 Therefore there is nothing to do. */
	    }
	}
    }
  
  if (cp->func == NULL)
    {
      msg (SE, _("%s is not yet implemented."), cp->cmd);
      while (token && token != '.')
	get_token ();
      return 1;
    }

  /* If we're in a FILE TYPE structure, only certain commands can be
     allowed. */
  if (pgm_state == STATE_INPUT && vfm_source == &file_type_source
      && !FILE_TYPE_okay (cp))
    return 0;

  /* Certain state transitions may not be made.  Check for these. */
  assert (pgm_state >= 0 && pgm_state < STATE_ERROR);
  if (cp->transition[pgm_state] == STATE_ERROR)
    {
      static const char *state_name[4] =
      {
	N_("%s is not allowed (1) before a command to specify the "
	"input program, such as DATA LIST, (2) between FILE TYPE "
	"and END FILE TYPE, (3) between INPUT PROGRAM and END "
	"INPUT PROGRAM."),
	N_("%s is not allowed within an input program."),
	N_("%s is only allowed within an input program."),
	N_("%s is only allowed within an input program."),
      };
      return msg (SE, gettext (state_name[pgm_state]), cp->cmd);
    }

#if DEBUGGING
  if (cp->func != cmd_remark)
    printf (_("%s command beginning\n"), cp->cmd);
#endif

  /* The structured output manager numbers all its tables.  Increment
     the major table number for each separate procedure. */
  som_new_series ();

  /* Call the command dispatcher.  Save and restore the name of the
     current command around this call. */
  prev_proc = cur_proc;
  cur_proc = cp->cmd;
  result = cp->func ();
  cur_proc = prev_proc;

  /* Perform the state transition if the command completed
     successfully (at least in part). */
  if (result != 0)
    {
      pgm_state = cp->transition[pgm_state];

      if (pgm_state == STATE_ERROR)
	{
	  discard_variables ();
	  pgm_state = STATE_INIT;
	}
    }
  

#if DEBUGGING
  if (cp->func != cmd_remark)
    printf (_("%s command completed\n\n"), cp->cmd);
#endif

  /* Pass the command's success value up to the caller. */
  return result;
}

/* EXIT/QUIT/FINISH commands. */

int
cmd_exit (void)
{
  if (am_reading_script)
    {
      msg (SE, _("This command is not accepted in a syntax file.  "
	   "Instead, use FINISH to terminate a syntax file."));
      get_token ();
    }
  else
    finished = 1;
  return 1;
}

int
cmd_finish (void)
{
  /* do not check for `.'; do not fetch any extra tokens */
  if (getl_interactive)
    {
      msg (SM, _("This command is not executed "
	   "in interactive mode.  Instead, PSPP drops "
	   "down to the command prompt.  Use EXIT if you really want "
	   "to quit."));
      getl_close_all ();
    }
  else
    finished = 1;
  return 1;
}

/* Extracts a null-terminated 8-or-fewer-character PREFIX from STRING.
   PREFIX is converted to lowercase.  Removes trailing spaces from
   STRING as a side effect.  */
static void
extract_prefix (char *string, char *prefix)
{
  /* Length of STRING. */
  int len;

  /* Points to the null terminator in STRING (`end pointer'). */
  char *ep;

  /* Strip spaces from end of STRING. */
  len = strlen (string);
  while (len && isspace ((unsigned char) string[len - 1]))
    string[--len] = 0;

  /* Find null terminator. */
  ep = memchr (string, '\0', 8);
  if (!ep)
    ep = &string[8];

  /* Copy prefix, converting to lowercase. */
  while (string < ep)
    *prefix++ = tolower ((unsigned char) (*string++));
  *prefix = 0;
}

/* Prints STRING on the console and to the listing file, replacing
   \n with newlines. */
static void
output_line (char *string)
{
  /* Location of \n in line read in. */
  char *cp;

  cp = strstr (string, "\\n");
  while (cp)
    {
      *cp = 0;
      tab_output_text (TAB_LEFT | TAT_NOWRAP, string);
      string = &cp[2];
      cp = strstr (string, "\\n");
    }
  tab_output_text (TAB_LEFT | TAT_NOWRAP, string);
}

int
cmd_remark ()
{
  extern int get_plain_line (void);
  extern int get_line (void);

  /* Points to the line read in. */
  char *s;

  /* Index into s. */
  char *cp;

  /* 8-character sentinel used to terminate remark. */
  char sentinel[9];

  /* Beginning of line used to compare with SENTINEL. */
  char prefix[9];

  som_blank_line();
  
  s = get_rest_of_line ();
  if (*s == '-')
    {
      output_line (&s[1]);
      return 1;
    }

  /* Read in SENTINEL from end of current line. */
  cp = s;
  while (isspace ((unsigned char) *cp))
    cp++;
  extract_prefix (cp, sentinel);
  if (sentinel[0] == 0)
    return msg (SE, _("The sentinel may not be the empty string."));

  /* Read in other lines until we encounter the sentinel. */
  while (getl_read_line ())
    {
      extract_prefix (getl_buf, prefix);
      if (streq (sentinel, prefix))
	break;

      /* Output the line. */
      output_line (getl_buf);
    }

  /* Calling get_entire_line() forces the sentinel line to be
     discarded. */
  getl_prompt = GETL_PRPT_STANDARD;
  get_entire_line ();
  return 1;
}

/* Parses the N command. */
int
cmd_n_of_cases (void)
{
  /* Value for N. */
  int x;

  match_id (N);
  match_id (OF);
  match_id (CASES);
  force_int ();
  x = tokint;
  get_token ();
  if (!match_id (ESTIMATED))
    default_dict.N = x;
  if (token != '.')
    return syntax_error (_("expecting end of command"));
  return 1;
}

/* Parses, performs the EXECUTE procedure. */
int
cmd_execute (void)
{
  int code = 1;

  match_id (EXECUTE);

  if (token != '.')
    {
      syntax_error (_("expecting end of command"));
      code = -2;
    }

  procedure (NULL, NULL, NULL);
  return code;
}

/* Parses, performs the ERASE command. */
int
cmd_erase (void)
{
  if (set_safer)
    {
      msg (SE, _("This command not allowed when the SAFER option is set."));
      return 0;
    }
  
  match_id (ERASE);
  force_match_id (FILE);
  match_tok ('=');
  force_string ();
  if (remove (tokstr) == -1)
    {
      msg (SW, _("Error removing `%s': %s."), tokstr, strerror (errno));
      return 0;
    }
  return 1;
}

#if unix
static int
shell (void)
{
  int pid;
  
  pid = fork ();
  switch (pid)
    {
    case 0:
      {
	const char *shell_fn;
	char *shell_process;
	
	{
	  int i;
	  
	  for (i = 3; i < 20; i++)
	    close (i);
	}

	shell_fn = getenv ("SHELL");
	if (shell_fn == NULL)
	  shell_fn = "/bin/sh";
	
	{
	  const char *cp = strrchr (shell_fn, '/');
	  cp = cp ? &cp[1] : shell_fn;
	  shell_process = local_alloc (strlen (cp) + 8);
	  strcpy (shell_process, "-");
	  strcat (shell_process, cp);
	  if (!streq (cp, "sh"))
	    shell_process[0] = '+';
	}
	
	execl (shell_fn, shell_process, NULL);

	hcf (1);
      }

    case -1:
      msg (SE, _("Couldn't fork: %s."), strerror (errno));
      return 0;

    default:
      assert (pid > 0);
      while (wait (NULL) != pid)
	;
      return 1;
    }
}
#endif /* unix */

static int
run_command (void)
{
  int c = lookahead ();
  char *cmd;
  int string;
      
  if (c == '\'' || c == '"')
    {
      get_token ();
      force_string ();
      cmd = tokstr;
      string = 1;
    }
  else
    {
      cmd = get_rest_of_line ();
      string = 0;
    }
      
  if (system (cmd) == -1)
    msg (SE, _("Error executing command: %s."), strerror (errno));
	  
  if (string)
    {
      get_token ();
      if (token != '.')
	{
	  syntax_error (_("expecting end of command"));
	  return 0;
	}
    }
  else token = '.';

  return 1;
}

/* Parses, performs the HOST command. */
int
cmd_host (void)
{
  int success;

  if (set_safer)
    {
      msg (SE, _("This command not allowed when the SAFER option is set."));
      return 0;
    }
  
#if unix
  if (lookahead () == '.')
    {
      get_token ();
      success = shell ();
    }
  else
    success = run_command ();
#else /* !unix */
  match_id (HOST);
  
  if (system (NULL) != 0)
    success = run_command ();
  else
    {
      msg (SE, _("No operating system support for this command."));
      success = 0;
    }
#endif /* !unix */

  return success;
}

/* Parses, performs the NEW FILE command. */
int
cmd_new_file (void)
{
  int success = 1;
  
  match_id (NEW);
  match_id (FILE);
  
  if (token != '.')
    {
      syntax_error (_("expecting end of command"));
      success = -3;
    }

  discard_variables ();

  return success;
}

/* Parses, performs the CLEAR TRANSFORMATIONS command. */
int
cmd_clear_transformations (void)
{
  match_id (CLEAR);
  match_id (TRANSFORMATIONS);

  if (am_reading_script)
    return msg (SW, _("This command is not valid in a syntax file."));

  cancel_transformations ();

  return 1;
}
