;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <99/06/28 17:15:25 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(provide 'wl-util)
(eval-when-compile
  (provide 'elmo-util)) ;; ???

(condition-case ()
    (require 'tm-edit)
  (error))
(eval-when-compile
  (mapcar
   (function
    (lambda (symbol)
      (unless (boundp symbol)
	(make-local-variable symbol)
	(eval (list 'setq symbol nil)))))
   '(mule-version 
     nemacs-version 
     emacs-beta-version
     xemacs-codename
     mime-edit-insert-user-agent-field
     mime-edit-user-agent-value
     mime-editor/version
     mime-editor/codename
     ))
  (or (fboundp 'read-event)
      (defun read-event ()))
  (or (fboundp 'next-command-event)
      (defun next-command-event ()))
  (or (fboundp 'event-to-character) 
      (defun event-to-character (a)))
  (or (fboundp 'key-press-event-p)
      (defun key-press-event-p (a)))
  (or (fboundp 'button-press-event-p)
      (defun button-press-event-p (a)))
  (or (fboundp 'set-process-kanji-code)
      (defun set-process-kanji-code (a b)))
  (or (fboundp 'set-process-coding-system)
      (defun set-process-coding-system (a b c)))
  (or (fboundp 'dispatch-event)
      (defun dispatch-event (a))))

(defalias 'wl-set-work-buf 'elmo-set-work-buf)
(defalias 'wl-eword-decode-string 'elmo-eword-decode-string)

(defmacro wl-append (val func)
  (list 'if val
      (list 'nconc val func)
    (list 'setq val func)))

(defun wl-parse (string regexp &optional matchn)
  (or matchn (setq matchn 1))
  (let (list)
    (store-match-data nil)
    (while (string-match regexp string (match-end 0))
      (setq list (cons (substring string (match-beginning matchn)
                                  (match-end matchn)) list)))
    (nreverse list)))

(defun wl-delete-duplicates (list &optional all hack-addresses)
  "Delete duplicate equivalent strings from the list.
If ALL is t, then if there is more than one occurrence of a string in the list,
 then all occurrences of it are removed instead of just the subsequent ones.
If HACK-ADDRESSES is t, then the strings are considered to be mail addresses,
 and only the address part is compared (so that \"Name <foo>\" and \"foo\"
 would be considered to be equivalent.)"
  (let ((hashtable (make-vector 29 0))
	(new-list nil)
	sym-string sym)
    (fillarray hashtable 0)
    (while list
      (setq sym-string
	    (if hack-addresses
		(wl-address-header-extract-address (car list))
	      (car list))
	    sym-string (or sym-string "-unparseable-garbage-")
	    sym (intern sym-string hashtable))
      (if (boundp sym)
	  (and all (setcar (symbol-value sym) nil))
	(setq new-list (cons (car list) new-list))
	(set sym new-list))
      (setq list (cdr list)))
    (delq nil (nreverse new-list))))

(defun wl-uniq-list (lst)
  "Distractively uniqfy elements of LST."
  (let ((tmp lst))
    (while tmp (setq tmp (setcdr tmp (delete (car tmp) (cdr tmp))))))
  lst)

(defun wl-string-member (string slist)
  "string is member of from the slist."
  (catch 'found
    (while slist
      (if (and (stringp (car slist))
	       (string= string (car slist)))
	  (throw 'found t))
      (setq slist (cdr slist)))))

(defun wl-string-match-member (str list &optional case-ignore)
  (let ((case-fold-search case-ignore))
    (catch 'member
      (while list
	(if (string-match (car list) str)
	    (throw 'member (car list)))
	(setq list (cdr list))))))

(defsubst wl-string-delete-match (string pos)
  (concat (substring string
		     0 (match-beginning pos))
	  (substring string
		     (match-end pos)
		     (length string))))

(defun wl-string-assoc (key alist)
  (let (a)
    (catch 'loop
      (while alist
	(setq a (car alist))
	(if (and (consp a)
		 (stringp (car a))
		 (string= key (car a)))
	    (throw 'loop a))
	(setq alist (cdr alist))))))

(defun wl-string-rassoc (key alist)
  (let (a)
    (catch 'loop
      (while alist
	(setq a (car alist))
	(if (and (consp a)
		 (stringp (cdr a))
		 (string= key (cdr a)))
	    (throw 'loop a))
	(setq alist (cdr alist))))))

(defun wl-parse-addresses (string)
  (if (null string)
      ()
    (wl-set-work-buf
     ;;(unwind-protect
     (let (list start s char)
       (insert string)
       (goto-char (point-min))
       (skip-chars-forward "\t\f\n\r ")
       (setq start (point))
       (while (not (eobp))
	 (skip-chars-forward "^\"\\,(")
	 (setq char (following-char))
	 (cond ((= char ?\\)
		(forward-char 1)
		(if (not (eobp))
		    (forward-char 1)))
	       ((= char ?,)
		(setq s (buffer-substring start (point)))
		(if (or (null (string-match "^[\t\f\n\r ]+$" s))
			(not (string= s "")))
		    (setq list (cons s list)))
		(skip-chars-forward ",\t\f\n\r ")
		(setq start (point)))
	       ((= char ?\")
		(re-search-forward "[^\\]\"" nil 0))
	       ((= char ?\()
		(let ((parens 1))
		  (forward-char 1)
		  (while (and (not (eobp)) (not (zerop parens)))
		    (re-search-forward "[()]" nil 0)
		    (cond ((or (eobp)
			       (= (char-after (- (point) 2)) ?\\)))
			  ((= (preceding-char) ?\()
			   (setq parens (1+ parens)))
			  (t
			   (setq parens (1- parens)))))))))
       (setq s (buffer-substring start (point)))
       (if (and (null (string-match "^[\t\f\n\r ]+$" s))
		(not (string= s "")))
	   (setq list (cons s list)))
       (nreverse list)) ; jwz: fixed order
     )))

(defun wl-version (&optional with-codename)
  (format "%s %s%s" wl-appname wl-version 
	  (if with-codename 
	      (format " - \"%s\"" wl-codename) "")))

(defun wl-version-show ()
  (interactive)
  (message "%s" (wl-version t)))

;; from gnus
(defun wl-extended-emacs-version (&optional with-codename)
  "Stringified Emacs version"
  (interactive)
  (cond
   ((string-match "^\\([0-9]+\\.[0-9]+\\)\\.[.0-9]+$" emacs-version)
    (concat "Emacs " (wl-match-string 1 emacs-version)
	    (and (boundp 'mule-version)(concat "/Mule " mule-version))))
   ((string-match "\\([A-Z]*[Mm][Aa][Cc][Ss]\\)[^(]*\\(\\((beta.*)\\|'\\)\\)?"
		  emacs-version)
    (concat (wl-match-string 1 emacs-version)
	    (format " %d.%d" emacs-major-version emacs-minor-version)
	    (if (and (boundp 'emacs-beta-version)
		     emacs-beta-version)
		(format "b%d" emacs-beta-version))
	    (if with-codename
		(if (boundp 'xemacs-codename)
		    (concat " - \"" xemacs-codename "\"")))))
   (t emacs-version)))

(defun wl-extended-emacs-version2 (&optional delimiter with-codename)
  "Stringified Emacs version"
  (interactive)
  (cond
   ((and (boundp 'mule-version)
	 mule-version
	 (string-match "\\([0-9]+\.[0-9]+\\)\\(.*$\\)" mule-version))
    (format "Mule%s%s@%d.%d%s" 
	    (or delimiter " ")
	    (wl-match-string 1 mule-version)
	    emacs-major-version
	    emacs-minor-version
	    (if with-codename
		(wl-match-string 2 mule-version)
	      "")
	    ))
   ((string-match "^\\([0-9]+\\.[0-9]+\\)\\.[.0-9]+$" emacs-version)
    (if (boundp 'nemacs-version)
	(concat "Nemacs" (or delimiter " ") 
		nemacs-version
		"@"
		(substring emacs-version
			   (match-beginning 1)
			   (match-end 1)))
      (concat "Emacs" (or delimiter " ")
	      (wl-match-string 1 emacs-version))))
   ((string-match "\\([A-Z]*[Mm][Aa][Cc][Ss]\\)[^(]*\\(\\((beta.*)\\|'\\)\\)?"
		  emacs-version)
    (concat (wl-match-string 1 emacs-version)
	    (or delimiter " ")
	    (format "%d.%d" emacs-major-version emacs-minor-version)
	    (if (and (boundp 'emacs-beta-version)
		     emacs-beta-version)
		(format "b%d" emacs-beta-version))
	    (if (and with-codename
		     (boundp 'xemacs-codename)
		     xemacs-codename)
		(format " (%s)" xemacs-codename))))
   (t emacs-version)))

(defun wl-extended-emacs-version3 (&optional delimiter with-codename)
  "Stringified Emacs version"
  (interactive)
  (cond
   ((and (boundp 'mule-version)
	 mule-version
	 (string-match "\\([0-9]+\.[0-9]+\\)\\(.*$\\)" mule-version))
    (format "Emacs%s%d.%d Mule%s%s%s" 
	    (or delimiter " ")
	    emacs-major-version
	    emacs-minor-version
	    (or delimiter " ")
	    (wl-match-string 1 mule-version)
	    (if with-codename
		(wl-match-string 2 mule-version)
	      "")
	    ))
   ((string-match "^\\([0-9]+\\.[0-9]+\\)\\.[.0-9]+$" emacs-version)
    (if (boundp 'nemacs-version)
	(let ((nemacs-codename-assoc '(("3.3.2" . " (FUJIMUSUME)")
				       ("3.3.1" . " (HINAMATSURI)")
				       ("3.2.3" . " (YUMENO-AWAYUKI)"))))
	  (format "Emacs%s%s Nemacs%s%s%s"
		  (or delimiter " ") 		  
		  (wl-match-string 1 emacs-version)
		  (or delimiter " ") 		  
		  nemacs-version
		  (or (and with-codename
			   (cdr (assoc nemacs-version
				       nemacs-codename-assoc)))
		      "")))
      (concat "Emacs" (or delimiter " ")
	      (wl-match-string 1 emacs-version))))
   ((string-match "\\([A-Z]*[Mm][Aa][Cc][Ss]\\)[^(]*\\(\\((beta.*)\\|'\\)\\)?"
		  emacs-version)
    (concat (wl-match-string 1 emacs-version)
	    (or delimiter " ")
	    (format "%d.%d" emacs-major-version emacs-minor-version)
	    (if (and (boundp 'emacs-beta-version)
		     emacs-beta-version)
		(format "b%d" emacs-beta-version))
	    (if (and with-codename
		     (boundp 'xemacs-codename)
		     xemacs-codename)
		(format " (%s)" xemacs-codename))))
   (t emacs-version)))

(defun wl-append-element (list element)
  (if element
      (append list (list element))
    list))

(defun wl-read-event-char ()
  "Get the next event."
  (let ((event (read-event)))
    ;; should be gnus-characterp, but this can't be called in XEmacs anyway
    (cons (and (numberp event) event) event)))

(defun wl-xmas-read-event-char ()
  "Get the next event."
  (let ((event (next-command-event)))
    (sit-for 0)
    ;; We junk all non-key events.  Is this naughty?
    (while (not (or (key-press-event-p event)
		    (button-press-event-p event)))
      (dispatch-event event)
      (setq event (next-command-event)))
    (cons (and (key-press-event-p event)
	       (event-to-character event))
	  event)))

(if running-xemacs
    (fset 'wl-read-event-char 'wl-xmas-read-event-char))

(defmacro wl-push (v l)
  (list 'setq l (list 'cons v l)))

(defmacro wl-pop (l)
  (list 'car (list 'prog1 l (list 'setq l (list 'cdr l)))))

(defun wl-ask-folder (func mes-string)
  (let* (key keve
	     (cmd (if (featurep 'xemacs)
		      (event-to-character last-command-event)
		    (string-to-char (format "%s" (this-command-keys)))))
	     )
    (message mes-string)
    (setq key (car (setq keve (wl-read-event-char))))
    (if (or (equal key ?\ )
	    (and cmd
		 (equal key cmd)))
	(progn
	  (message "")
	  (funcall func))
      (wl-push (cdr keve) unread-command-events))))

(defalias 'wl-make-hash 'elmo-make-hash)

(defalias 'wl-get-hash-val 'elmo-get-hash-val)
  
(defalias 'wl-set-hash-val 'elmo-set-hash-val)

(defsubst wl-set-string-width (width string)
  (wl-set-work-buf
   (elmo-set-buffer-multibyte default-enable-multibyte-characters)
   (insert string)
   (if (> (current-column) width)
       (if (> (move-to-column width) width)
	   (progn
	     (condition-case nil ; ignore error 
		 (backward-char 1)
	       (error))
	     (concat (buffer-substring (point-min) (point)) " "))
	 (buffer-substring (point-min) (point)))
     (if (= (current-column) width)
	 string
       (concat string
	       (format (format "%%%ds" 
			       (- width (current-column)))
		       " "))))))

(defun wl-display-bytes (num)
  (let (result remain)
    (cond
     ((> (setq result (/ num 1000000)) 0)
      (setq remain (% num 1000000))
      (if (> remain 400000)
	  (setq result (+ 1 result)))
      (format "%dM" result))
     ((> (setq result (/ num 1000)) 0)
      (setq remain (% num 1000))
      (if (> remain 400)
	  (setq result (+ 1 result)))
      (format "%dK" result))
     (t (format "%dB" result)))))

(defun wl-generate-user-agent-string ()
  "A candidate of wl-generate-mailer-string-func. 
Insert User-Agent field instead of X-Mailer field."
  (let ((mime-user-agent (and (boundp 'mime-edit-insert-user-agent-field)
			      mime-edit-insert-user-agent-field
			      mime-edit-user-agent-value)))
    (if mime-user-agent
	(concat "User-Agent: "
		wl-appname "/" wl-version
		" (" wl-codename ") "
		mime-user-agent)
      (if (and (boundp 'mime-editor/version)
	       mime-editor/version)
	  (concat "User-Agent: "
		  wl-appname "/" wl-version
		  " (" wl-codename ") "
		  "tm/" mime-editor/version
		  (if (and (boundp 'mime-editor/codename)
			   mime-editor/codename)
		      (concat " (" mime-editor/codename ")"))
		  " " (wl-extended-emacs-version3 "/" t))
	(concat "User-Agent: " wl-appname "/" wl-version " (" wl-codename ") "
		(wl-extended-emacs-version3 "/" t))))))

(defun wl-make-modeline ()
  "Create new modeline format for Wanderlust"
  (let* ((duplicated (copy-sequence mode-line-format))
	 (cur-entry duplicated)
	 return-modeline)
    (if (memq 'wl-plug-state-indicator mode-line-format)
	duplicated
      (catch 'done
	(while cur-entry
	  (if (or (and (symbolp (car cur-entry))
		       (eq 'mode-line-buffer-identification 
			      (car cur-entry)))
		  (and (consp (car cur-entry))
		       (or 
			(eq 'modeline-buffer-identification 
			       (car (car cur-entry)))
			(eq 'modeline-buffer-identification 
			       (cdr (car cur-entry))))))
	      (progn
		(setq return-modeline (append return-modeline
					      (list 'wl-plug-state-indicator)
					      cur-entry))
		(throw 'done return-modeline))
	    (setq return-modeline (append return-modeline
					  (list (car cur-entry)))))
	  (setq cur-entry (cdr cur-entry)))))))

(if (fboundp 'display-error)
    (defalias 'wl-display-error 'display-error)
  (defun wl-display-error (error-object stream)
    "a tiny function to display error-object to the stream."
    (let ((first t)
	  (errobj error-object)
	  err-mes)
      (while errobj
	(setq err-mes (concat err-mes (format 
				       (if (stringp (car errobj))
					   "%s"
					 (if wl-on-nemacs
					     "%s"
					   "%S")) (car errobj))))
	(setq errobj (cdr errobj))
	(if errobj (setq err-mes (concat err-mes (if first ": " ", "))))
	(setq first nil))
      (princ err-mes stream))))

(defun wl-get-assoc-list-value (assoc-list folder)
  (catch 'found
    (let ((alist assoc-list))
      (while alist
	(let ((pair (car alist)))
	  (if (string-match (car pair) folder)
	      (throw 'found (cdr pair))
	    ))
	(setq alist (cdr alist)))
      )))

(defun wl-folder-persistent-p (folder)
  (or (catch 'found
	(let ((li wl-save-folder-list))
	  (while li
	    (if (string-match (car li) folder)
		(throw 'found t))
	    (setq li (cdr li)))))
      (not (catch 'found
	     (let ((li wl-no-save-folder-list))
	       (while li
		 (if (string-match (car li) folder)
		     (throw 'found t))
		 (setq li (cdr li))))))))

(defmacro wl-match-string (pos string)
  "Substring POSth matched string."
  (` (substring (, string) (match-beginning (, pos)) (match-end (, pos)))))

(defmacro wl-match-buffer (pos)
  "Substring POSth matched from the current buffer."
  (` (buffer-substring-no-properties
      (match-beginning (, pos)) (match-end (, pos)))))

;; open-network-stream-as-binary
(or (fboundp 'open-network-stream-as-binary)
    (if wl-on-mule
	(defun open-network-stream-as-binary (name buffer host service)
	  "Like `open-network-stream', q.v., but don't do code conversion."
	  (let ((process (open-network-stream name buffer host service)))
	    (set-process-coding-system process *noconv* *noconv*)
	    process))
      (if wl-on-nemacs
	  (defun open-network-stream-as-binary (name buffer host service)
	    "Like `open-network-stream', q.v., but don't do code conversion."
	    (let ((process (open-network-stream name buffer host service)))
	      (set-process-kanji-code process 0)
	      process))
	(if wl-on-mule3
	    (defun open-network-stream-as-binary (name buffer host service)
	      "Like `open-network-stream', q.v., but don't code conversion."
	      (let ((coding-system-for-read 'binary)
		    (coding-system-for-write 'binary))
		(open-network-stream name buffer host service)))))))

(defmacro wl-string (string)
  "String without text property"
  (` (format "%s" (, string))))

(defun wl-parse-newsgroups (string)
  (let* ((nglist (wl-parse string "[ \t\f\r\n,]*\\([^ \t\f\r\n,]+\\)"))
	 spec ret-val)
    (while nglist
      (if (intern-soft (car nglist) wl-folder-newsgroups-hashtb)
	  (wl-append ret-val (list (car nglist))))
      (setq nglist (cdr nglist)))
    ret-val))

;; Check if active region exists or not.
(if (boundp 'mark-active)
    (defmacro wl-region-exists-p ()
      'mark-active)
  (if (fboundp 'region-exists-p)
      (defmacro wl-region-exists-p ()
	(list 'region-exists-p))))
  

(if (not (fboundp 'overlays-in))
    (defun overlays-in (beg end)
      "Return a list of the overlays that overlap the region BEG ... END.
Overlap means that at least one character is contained within the overlay
and also contained within the specified region.
Empty overlays are included in the result if they are located at BEG
or between BEG and END."
      (let ((ovls (overlay-lists))
	    tmp retval)
	(if (< end beg)
	    (setq tmp end
		  end beg
		  beg tmp))
	(setq ovls (nconc (car ovls) (cdr ovls)))
	(while ovls
	  (setq tmp (car ovls)
		ovls (cdr ovls))
	  (if (or (and (<= (overlay-start tmp) end)
		       (>= (overlay-start tmp) beg))
		  (and (<= (overlay-end tmp) end)
		       (>= (overlay-end tmp) beg)))
	      (setq retval (cons tmp retval))))
	retval)))

(defsubst wl-repeat-string (str times)
  (let ((loop times)
	ret-val)
    (while (> loop 0)
      (setq ret-val (concat ret-val str))
      (setq loop (- loop 1)))
    ret-val))

(defun wl-list-diff (list1 list2)
  "Return a list of elements of LIST1 that do not appear in LIST2."
  (let ((list1 (copy-sequence list1)))
    (while list2
      (setq list1 (delq (car list2) list1))
      (setq list2 (cdr list2)))
    list1))

(defun wl-append-assoc-list (item value alist)
  "make assoc list '((item1 value1-1 value1-2 ...)) (item2 value2-1 ...)))"
  (let ((entry (assoc item alist)))
    (if entry
	(progn
	  (when (not (member value (cdr entry)))
	    (nconc entry (list value)))
	  alist)
      (append alist
	      (list (list item value))))))

