;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <1999-03-03 17:52:52 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'wl-vars)
(require 'wl-util)
(provide 'wl-refile)

(defvar wl-refile-alist nil)
(defvar wl-refile-alist-file-name "refile-alist")

(defun wl-refile-alist-load ()
  (save-excursion
    (let ((filename (expand-file-name wl-refile-alist-file-name
				      elmo-msgdb-dir))
	  (tmp-buffer (get-buffer-create " *wl-refile-alist-tmp*"))
	  insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
	  insert-file-contents-post-hook 
	  ret-val)
      (if (not (file-readable-p filename))
	  ()
	(set-buffer tmp-buffer)
	(insert-file-contents filename)
	(setq ret-val
	      (condition-case nil
		  (read (current-buffer)) 
		(error nil nil))))
      (kill-buffer tmp-buffer)
      ret-val)))

(defun wl-refile-alist-save ()
  (save-excursion
    (let ((filename (expand-file-name wl-refile-alist-file-name
				      elmo-msgdb-dir))
	  (tmp-buffer (get-buffer-create " *wl-refile-alist-tmp*")))
      (set-buffer tmp-buffer)
      (erase-buffer)
      (prin1 wl-refile-alist tmp-buffer)
      (princ "\n" tmp-buffer)
      (if (file-writable-p filename)
	  (write-region (point-min) (point-max) 
			filename nil 'no-msg)
	(message (format "%s is not writable." filename)))
      (kill-buffer tmp-buffer))))

(defun wl-refile-learn (entity dst)
  (let ((myself (wl-address-header-extract-address wl-from))
	tocc-list from key hit)
    (setq tocc-list 
	  (mapcar 'downcase
		  (mapcar '(lambda (entity) 
			     (wl-address-header-extract-address
			      entity))
			  (wl-parse-addresses 
			   (concat
			    (elmo-msgdb-overview-entity-get-to entity) ","
			    (elmo-msgdb-overview-entity-get-cc entity))))))
    (or (setq key ; subscribed entity!!
	      (catch 'found
		(while tocc-list
		  (if (wl-string-member 
		       (car tocc-list) 
		       (mapcar 'downcase wl-subscribed-mailing-list))
		      (throw 'found (car tocc-list))
		    (setq tocc-list (cdr tocc-list))))))
	(and (not (string= (downcase myself)
			   (downcase 
			    (setq from 
				  (wl-address-header-extract-address
				   (elmo-msgdb-overview-entity-get-from 
				    entity))))))
	     (setq key from)))
    (if (not wl-refile-alist)
	(setq wl-refile-alist (wl-refile-alist-load)))
    (when key
      (if (setq hit (assoc key wl-refile-alist))
	  (progn
	    (setcdr hit dst)
	    wl-refile-alist)
	(setq wl-refile-alist (append wl-refile-alist
				      (list (cons key dst)))))
      (if (not hit) (wl-refile-alist-save)))))

;;
;; refile guess
;;
(defun wl-refile-guess (entity)
  (or (and wl-refile-rule-alist
	   (wl-refile-guess-by-rule entity))
      (wl-refile-guess-by-history entity)))

(defun wl-refile-guess-by-rule (entity)
  (let ((rules wl-refile-rule-alist)
	(rule-set) (field) (field-cont))
    (catch 'found
      (while rules
	(setq rule-set (cdr (car rules))
	      field (car (car rules)))
	(cond ((string-match field "From")
	       (setq field-cont
		     (elmo-msgdb-overview-entity-get-from entity)))
	      ((string-match field "Subject")
	       (setq field-cont
		     (elmo-msgdb-overview-entity-get-subject entity)))
	      ((string-match field "To")
	       (setq field-cont
		     (elmo-msgdb-overview-entity-get-to entity)))
	      ((string-match field "Cc")
	       (setq field-cont
		     (elmo-msgdb-overview-entity-get-cc entity)))
	      (t
	       (setq field-cont
		     (elmo-msgdb-overview-entity-get-extra-field
		      entity (downcase field)))))
	(if field-cont
	    (while rule-set
	      (if (string-match (car (car rule-set)) field-cont)
		  (throw 'found (cdr (car rule-set)))
		(setq rule-set (cdr rule-set)))))
	(setq rules (cdr rules))))))

(defun wl-refile-guess-by-history (entity)
  (let ((tocc-list 
	 (mapcar 'downcase
		 (mapcar '(lambda (entity) 
			  (wl-address-header-extract-address
			   entity))
			 (wl-parse-addresses 
			  (concat
			   (elmo-msgdb-overview-entity-get-to entity) ","
			   (elmo-msgdb-overview-entity-get-cc entity))))))
	ret-val
	)
    (setq tocc-list (delete (wl-address-header-extract-address
			     wl-from) tocc-list))
    (if (not wl-refile-alist)
	(setq wl-refile-alist
	      (wl-refile-alist-load)))
    (catch 'found
      (while tocc-list
	(if (setq ret-val (cdr (assoc (car tocc-list) wl-refile-alist)))
	    (throw 'found nil))
	(setq tocc-list (cdr tocc-list))
	))
    (or ret-val
	(wl-refile-guess-by-from entity))))

(defun wl-refile-get-account-part-from-address (address)
  (if (string-match "\\([^@]+\\)@[^@]+" address)
      (wl-match-string 1 address)
    address))
		 
(defun wl-refile-guess-by-from (entity)
  (let ((from 
	 (wl-address-header-extract-address
	  (elmo-msgdb-overview-entity-get-from entity))))
    (if (not wl-refile-alist)
	(setq wl-refile-alist
	      (wl-refile-alist-load)))
    ;; search from alist
    (or (cdr (assoc from wl-refile-alist))
	(format "%s/%s" wl-refile-default-from-folder 
		(wl-refile-get-account-part-from-address from)))))
  
