;;;
;;; elmo-util.el -- Utilities for Elmo.
;;;   Copyright 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'elmo-vars)

;; For NEmacs.
(or (fboundp 'eval-and-compile)
    (progn
      (put 'eval-and-compile 'lisp-indent-hook 0)
      (defmacro eval-and-compile (&rest body)
	(cons 'progn body))
      ))
(or (fboundp 'eval-when-compile)
    (progn
      (put 'eval-when-compile 'lisp-indent-hook 0)
      (defmacro eval-when-compile (&rest body)
	(cons 'progn body))
      ))

(require 'elmo-date)
(require 'std11)
(require 'cl)

(defvar elmo-work-buf-name " *elmo work*")
(defvar elmo-temp-buf-name " *elmo temp*")

(eval-when-compile
  (mapcar
   (function
    (lambda (symbol)
      (unless (boundp symbol)
	(make-local-variable symbol)
	(eval (list 'setq symbol nil)))))
   '(message-log-max))
  ;;; From x-face-mule
  ;;; For tm.
  (unless (fboundp 'save-current-buffer)
    (defmacro save-current-buffer (&rest body)
      (` (let ((orig-buffer (current-buffer)))
	   (unwind-protect
	       (progn (,@ body))
	     (set-buffer orig-buffer))))))
  (unless (fboundp 'with-current-buffer)
    (defmacro with-current-buffer (buffer &rest body)
      (` (save-current-buffer
	   (set-buffer (, buffer))
	   (,@ body)))))
  (unless (fboundp 'with-temp-buffer)
    (defmacro with-temp-buffer (&rest forms)
      (let ((temp-buffer (make-symbol "temp-buffer")))
	(` (let (((, temp-buffer)
		  ;(get-buffer-create (generate-new-buffer-name " *temp*"))))
		  (generate-new-buffer " *temp*")))
	     (unwind-protect
		 (with-current-buffer
		     (, temp-buffer)
		   (,@ forms))
	       (and (buffer-name (, temp-buffer))
		    (kill-buffer (, temp-buffer))))))))))

(or (boundp 'default-enable-multibyte-characters)
    (defvar default-enable-multibyte-characters (featurep 'mule)
      "The mock variable except for Emacs 20."))

;; base64 encoding/decoding
(require 'mel)
(eval-and-compile
  (if (fboundp 'mel-find-function) 
      (progn
	(fset 'elmo-base64-encode-string 
	      (mel-find-function 'mime-encode-string "base64"))
	(fset 'elmo-base64-decode-string
	      (mel-find-function 'mime-decode-string "base64")))
    (require 'mel-b)
    (defun mel-find-function (a b)) ; silence byte-compiler
    (fset 'elmo-base64-encode-string 'base64-encode-string)
    (fset 'elmo-base64-decode-string 'base64-decode-string)))

(eval-and-compile
  ;; encoded word decoding
  (if (module-installed-p 'eword-decode)
      (progn
	(autoload 'eword-decode-string "eword-decode" nil t)
	(defalias 'elmo-eword-decode-string 'eword-decode-string))
    (if (module-installed-p 'tm-ew-d) 
	(progn
	  (autoload 'mime-eword/decode-string "tm-ew-d" nil t)
	  (defalias 'elmo-eword-decode-string 'mime-eword/decode-string))
      (error "no encoding module is installed."))))

;; Any Emacsen may have add-name-to-file(), because loadup.el requires it. :-p
;; Check make-symbolic-link() instead.  -- 981002 by Fuji
(if (fboundp 'make-symbolic-link)  ;; xxx
    (defalias 'elmo-add-name-to-file 'add-name-to-file)
  (defun elmo-add-name-to-file 
    (filename newname &optional ok-if-already-exists)
    (copy-file filename newname ok-if-already-exists t)
    ))

;;;; for OS dependent safe filename string.
(if (fboundp 'convert-standard-filename)
    (defalias 'elmo-convert-standard-filename 'convert-standard-filename)
  (if (fboundp 'mule-for-win32-version) ;; Mule for Win32
      (defun elmo-convert-standard-filename (filename)
	(elmo-replace-in-string filename "*" "!"))
    (defalias 'elmo-convert-standard-filename 'identity)))

(defsubst elmo-call-func (folder func-name &rest args)
  (let* ((spec (if (stringp folder)
		   (elmo-folder-get-spec folder)
		 folder))
	 (type (symbol-name (car spec)))
	 (backend-str (concat "elmo-" type))
	 (backend-sym (intern backend-str)))
    (unless (featurep backend-sym)
      (require backend-sym))
    (apply (intern (format "%s-%s" backend-str func-name))
	   spec
	   args)))

(defmacro elmo-set-work-buf (&rest body)
  "Execute BODY on work buffer. Work buffer remains."
  (` (save-excursion
       (set-buffer (get-buffer-create elmo-work-buf-name))
       (erase-buffer)
       (,@ body))))

(defmacro elmo-match-string (pos string)
  "Substring POSth matched string."
  (` (substring (, string) (match-beginning (, pos)) (match-end (, pos)))))

(defmacro elmo-match-buffer (pos)
  "Substring POSth matched from the current buffer."
  (` (buffer-substring-no-properties
      (match-beginning (, pos)) (match-end (, pos)))))

(defmacro elmo-bind-directory (dir &rest body)
  "Set current directory DIR and execute BODY."
  (` (let ((default-directory (file-name-as-directory (, dir))))
       (,@ body))))

(defmacro elmo-folder-get-type (folder)
  "Get type of FOLDER."
  (` (and (stringp (, folder))
	  (cdr (assoc (string-to-char (, folder)) elmo-spec-alist)))))

(defun elmo-object-load (path &optional set-nil-if-failed)
  "Load OBJECT from PATH.
If optional argument SET-NIL-IF-FAILED is non-nil, 
content of PATH is set as nil."
  (let (ret-val)
    (if (not (file-readable-p path))
	()
      (elmo-set-work-buf
       (as-binary-input-file
	(insert-file-contents path))
       (elmo-set-buffer-multibyte default-enable-multibyte-characters)
       (setq ret-val
	     (condition-case nil
		 (read (current-buffer)) 
	       (error (message "Warning: Loading object from %s failed."
			       path)
		      (if set-nil-if-failed
			  (elmo-object-save path nil))
		      nil))))
      ret-val)))

(defun elmo-object-save (path object)
  "Save OBJECT in PATH."
  (let ((dir (directory-file-name (file-name-directory path)))
	(tmp-buf (get-buffer-create elmo-work-buf-name)))
    (elmo-set-work-buf
     (prin1 object tmp-buf)
     (princ "\n" tmp-buf)
     (if (file-directory-p dir)
	 (); ok.
       (if (file-exists-p dir)
	   (error "File %s already exists." dir)
	 (elmo-make-directory dir)))
     (if (file-writable-p path)
	 (write-region (point-min) (point-max) path nil 'no-msg)
       (message (format "%s is not writable." path))))
    object))

(defun elmo-imap4-get-spec (folder)
  (let (fld-name
	user auth server port ssl
	(default-server elmo-default-imap4-server)
	(default-user
	  (or (getenv "USER") (getenv "LOGNAME") (user-login-name)))
	(default-auth  elmo-default-imap4-authenticate-type)
	)
    (when (string-match "\\(.*\\)@\\(.*\\)" default-server)
      ;; case: default-imap4-server is specified like "hoge%imap.server@gateway".
      (setq default-user (elmo-match-string 1 default-server))
      (setq default-server (elmo-match-string 2 default-server)))
    (if (string-match
	 "^\\(%\\)\\([^:@!]*\\):?\\([^/@!]*\\)/?\\([^/:@!]*\\)@?\\([^:/!]*\\):?\\([0-9]*\\)\\(!\\)?.*$"
	 folder)
	(progn
	  (if (eq (length (setq fld-name
				(elmo-match-string 2 folder))) 0)
	      (setq fld-name ""))
	  (if (eq (length (setq user
				(elmo-match-string 3 folder))) 0)
	      (setq user default-user))
	  (if (eq (length (setq auth
				(elmo-match-string 4 folder))) 0)
	      (setq auth default-auth))
	  (if (eq (length (setq server
				(elmo-match-string 5 folder))) 0)
	      (setq server default-server))
	  (if (= (setq port
		       (string-to-int
			(elmo-match-string 6 folder))) 0)
	      (setq port elmo-default-imap4-port))
	  (if (match-beginning 7)
	      (setq ssl (string= (elmo-match-string 7 folder) "!"))
	    (setq ssl elmo-default-imap4-ssl))
	  (list 'imap4 fld-name server user auth port ssl)))))

(defun elmo-nntp-get-spec (folder)
  (let (fld-name server port user ssl)
    (if (string-match
	 "^\\(-\\)\\([^:@!]*\\):?\\([^/@!]*\\)/?\\([^@:!]*\\)@?\\([^:/!]*\\):?\\([0-9]*\\)\\(!\\)?.*$"
	 folder)
	(progn
	  (if (eq (length (setq fld-name
				(elmo-match-string 2 folder))) 0)
	      (setq fld-name nil))
	  (if (eq (length (setq user
				(elmo-match-string 3 folder))) 0)
	      (setq user nil))
	  (if (eq (length (setq server
				(elmo-match-string 5 folder))) 0)
	      (setq server elmo-default-nntp-server))
	  (if (= (setq port
		       (string-to-int
			(elmo-match-string 6 folder))) 0)
	      (setq port elmo-default-nntp-port))
	  (if (match-beginning 7)
	      (setq ssl (string= (elmo-match-string 7 folder) "!"))
	    (setq ssl elmo-default-nntp-ssl))	  
	  (list 'nntp fld-name server user port ssl)))))

(defun elmo-localdir-get-spec (folder)
  (let (fld-name path)
    (when (string-match
	   "^\\(\\+\\)\\(.*\\)$"
	   folder)
      (if (eq (length (setq fld-name
			    (elmo-match-string 2 folder))) 0)
	  (setq fld-name "")
	)
      (if (string-match "^[/~].*\\|^[A-Z]:" fld-name)
	  (setq path (expand-file-name fld-name))
	(setq path fld-name))
	;(setq path (expand-file-name fld-name
	;elmo-localdir-folder-path)))
      (list (if (elmo-folder-maildir-p folder)
		'maildir
	      'localdir) path))))

(defun elmo-localnews-get-spec (folder)
  (let (fld-name)
    (when (string-match
	 "^\\(=\\)\\(.*\\)$"
	 folder)
      (if (eq (length (setq fld-name
			    (elmo-match-string 2 folder))) 0)
	  (setq fld-name "")
	)
      (list 'localnews 
	    (elmo-replace-in-string fld-name "\\." "/")))))

;; Archive interface by OKUNISHI Fujikazu <fuji0924@mbox.kyoto-inet.or.jp>
(defun elmo-archive-get-spec (folder)
  (require 'elmo-archive)
  (let (fld-name type prefix)
    (when (string-match
	   "^\\(\\$\\)\\([^;]*\\);?\\([^;]*\\);?\\([^;]*\\)$"
	   folder)
      ;; Drive letter is OK!
      (if (eq (length (setq fld-name
			    (elmo-match-string 2 folder))) 0)
	  (setq fld-name "")
	)
      (if (eq (length (setq type
			    (elmo-match-string 3 folder))) 0)
	  (setq type (symbol-name elmo-archive-default-type)))
      (if (eq (length (setq prefix
			    (elmo-match-string 4 folder))) 0)
	  (setq prefix ""))
      (list 'archive fld-name (intern-soft type) prefix))))

(defun elmo-pop3-get-spec (folder)
  (require 'elmo-pop3)
  (let (user server port auth ssl)
    (when (string-match
	   "\\(&\\)\\([^/@]*\\)/?\\([^@]*\\)@?\\([^:]*\\):?\\([0-9]*\\)\\(!\\)?$"
	   folder)
      (if (eq (length (setq user
			    (elmo-match-string 2 folder))) 0)
	  (setq user 
		(or (getenv "USER") (getenv "LOGNAME") (user-login-name))))
      (if (eq (length (setq server
			    (elmo-match-string 4 folder))) 0)
	  (setq server elmo-default-pop3-server))
      (if (= (setq port
		   (string-to-int
		    (elmo-match-string 5 folder))) 0)
	  (setq port elmo-default-pop3-port))
      (if (eq (length (setq auth
			    (elmo-match-string 3 folder))) 0)
	  (setq auth elmo-default-pop3-authenticate-type))
      (if (match-beginning 6)
	  (setq ssl (string= (elmo-match-string 6 folder) "!"))
	(setq ssl elmo-default-pop3-ssl))
      (list 'pop3 user server port auth ssl))))

(defun elmo-internal-get-spec (folder)
  (if (string-match "\\('\\)\\([^=]*\\)=?\\(.*\\)$"
		    folder)
      (list 'internal
	    (intern (elmo-match-string 2 folder))
	    (elmo-match-string 3 folder))))

(defun elmo-multi-get-spec (folder)
  (save-match-data
    (when (string-match
	   "^\\(\\*\\)\\(.*\\)$"
	   folder)
      (append (list 'multi)
	      (elmo-tokenize-string 
	       (elmo-match-string 2 folder)
	       ",")))))

(defun elmo-filter-get-spec (folder)
  (save-match-data
    (when (string-match
	   "^\\(/\\)\\(.*\\)$"
	   folder)
      (let ((spec (elmo-match-string 2 folder))
	    by key value
	    filter)
	(when (string-match "\\([^/]+\\)/" spec)
	  (setq filter (elmo-match-string 1 spec))
	  (setq spec (substring spec (match-end 0))))
	(cond
	 ((string-match "^\\([a-zA-Z\\-]+\\)=\\(.*\\)$" filter) ; match
	  (setq by 'match)
	  (setq key   (elmo-match-string 1 filter))
	  (setq value (elmo-match-string 2 filter)))
	 ((string-match "^\\([a-zA-Z\\-]+\\)!=\\(.*\\)$" filter) ; unmatch
	  (setq by 'unmatch)
	  (setq key   (elmo-match-string 1 filter))
	  (setq value (elmo-match-string 2 filter)))
	 ((string-match "^\\([a-zA-Z\\-]+\\):\\([0-9]+\\)$" filter) ; partial
	  (setq key   (elmo-match-string 1 filter))
	  (setq value (string-to-int
		       (elmo-match-string 2 filter)))
	  (setq by 'partial))
	 (t
	  (error "syntax error in folder spec \"%s\"." folder)))
	(list 'filter by key value spec)))))

(defun elmo-tokenize-string (string delimiter)
  (read (concat "(\"" (elmo-replace-in-string 
		       string
		       delimiter "\" \"") "\")")))

(defun elmo-folder-get-spec (folder)
  "return spec of folder"
  (let ((type (elmo-folder-get-type folder)))
    (if type
	(funcall (intern (concat "elmo-" (symbol-name type) "-get-spec"))
		 folder)
      (error "%s is not supported folder type." folder)
      )))

(defun elmo-folder-number-get-type (folder number)
  (car (elmo-folder-number-get-spec folder number)))

(defun elmo-folder-number-get-spec (folder number)
  (let ((type (elmo-folder-get-type folder)))
    (cond
     ((eq type 'multi)
      (elmo-multi-folder-number-get-spec folder number))
     ((eq type 'filter)
      (elmo-folder-number-get-spec
       (nth 4 (elmo-folder-get-spec folder)) number))
     (t
      (elmo-folder-get-spec folder)
      ))))

(defun elmo-folder-maildir-p (folder)
  (catch 'found
    (let ((li elmo-maildir-list))
      (while li
	(if (string-match (car li) folder)
	    (throw 'found t))
	(setq li (cdr li))))))

(defun elmo-multi-p (folder)
  (let ((type (elmo-folder-get-type folder)))
    (cond
     ((eq type 'multi)
      t)
     ((eq type 'filter)
      (elmo-multi-p
       (nth 4 (elmo-folder-get-spec folder))))
     (t
      nil
      ))))

(defun elmo-get-real-folder-number (folder number)
  (let ((type (elmo-folder-get-type folder)))
    (cond
     ((eq type 'multi)
      (elmo-multi-get-real-folder-number folder number))
     ((eq type 'filter)
      (elmo-get-real-folder-number
       (nth 4 (elmo-folder-get-spec folder)) number))
     (t
      (cons folder number)
      ))))

(defun elmo-folder-get-primitive-spec-list (folder &optional spec-list)
  (let* ((type (elmo-folder-get-type folder))
	 specs)
    (cond
     ((eq type 'multi)
      (let ((flds (cdr (elmo-folder-get-spec folder)))
	    spec)
	(while flds
	  (setq spec (elmo-folder-get-primitive-spec-list (car flds)))
	  (if (not (memq (car spec) specs))
	      (setq specs (append specs spec)))
	  (setq flds (cdr flds)))))
     ((eq type 'filter)
      (setq specs
	    (elmo-folder-get-primitive-spec-list
	     (nth 4 (elmo-folder-get-spec folder)))))
     (t
      (setq specs (list (elmo-folder-get-spec folder)))
      ))
    specs))

(defun elmo-folder-get-primitive-folder-list (folder)
  (let* ((type (elmo-folder-get-type folder)))
    (cond
     ((eq type 'multi)
      (let ((flds (cdr (elmo-folder-get-spec folder)))
	    ret-val)
	(while flds
	  (setq ret-val (append ret-val
				(elmo-folder-get-primitive-folder-list
				 (car flds))))
	  (setq flds (cdr flds)))
	ret-val))
     ((eq type 'filter)
      (elmo-folder-get-primitive-folder-list
       (nth 4 (elmo-folder-get-spec folder))))
     (t
      (list folder)
      ))))

(defun elmo-multi-folder-number-get-spec (folder number)
  (let* ((spec (elmo-folder-get-spec folder))
	 (flds (cdr spec))
	 (fld (nth (- (/ number elmo-multi-divide-number) 1) flds)))
    (elmo-folder-number-get-spec fld number)))

(defun elmo-multi-get-real-folder-number (folder number)
  (let* ((spec (elmo-folder-get-spec folder))
	 (flds (cdr spec))
	 (num number)
	 (fld (nth (- (/ num elmo-multi-divide-number) 1) flds)))
    (cons fld (% num elmo-multi-divide-number))))

(defsubst elmo-delete-char (char string)
  (save-match-data
    (elmo-set-work-buf
     (let ((coding-system-for-read 'no-conversion)
	   (coding-system-for-write 'no-conversion))
       (insert string)
       (goto-char (point-min))
       (while (search-forward (char-to-string char) nil t)
	 (replace-match ""))
       (buffer-string)))))

(defun elmo-delete-cr-get-content-type (outbuf &optional string)
  (save-excursion
    (set-buffer outbuf)
    (when string
      (erase-buffer)
      (insert string))
    (goto-char (point-min))
    (while (search-forward "\r\n" nil t)
      (replace-match "\n"))
    (goto-char (point-min))
    (let (ret-val)
      (if (setq ret-val (std11-field-body "content-type"))
	  ret-val
	t))))

(defun elmo-delete-cr (string)
  (save-match-data
    (elmo-set-work-buf
     (insert string)
     (goto-char (point-min))
     (while (search-forward "\r\n" nil t)
       (replace-match "\n"))
     (buffer-string))))

(defun elmo-string-partial-p (string)
  (and (stringp string) (string-match "message/partial" string)))

(defun elmo-get-file-string (filename &optional remove-final-newline)
  (elmo-set-work-buf
   (let (insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
	 insert-file-contents-post-hook)
     (when (file-exists-p filename)
       (if filename
	   (as-binary-input-file (insert-file-contents filename)))
       (when (and remove-final-newline
		  (> (buffer-size) 0)
		  (= (char-after (1- (point-max))) ?\n))
	 (goto-char (point-max))
	 (delete-backward-char 1))
       (buffer-string)))))

(defun elmo-save-string (string filename)
  (if string
      (elmo-set-work-buf
       (as-binary-output-file
	(insert string)
	(write-region (point-min) (point-max) 
		      filename nil 'no-msg))
       )))

(defun elmo-max-of-list (nlist)
  (let ((l nlist) 
	(max-num 0))
    (while l
      (if (< max-num (car l))
	  (setq max-num (car l)))
      (setq l (cdr l)))
    max-num))

(defun elmo-concat-path (path filename)
  (if (not (string= path ""))
      (if (string= elmo-path-sep (substring path (- (length path) 1)))
	  (concat path filename)
	(concat path elmo-path-sep filename))
    filename))

(defvar elmo-passwd-alist nil)

(defun elmo-passwd-alist-load ()
  (save-excursion
    (let ((filename (expand-file-name elmo-passwd-alist-file-name
                                      elmo-msgdb-dir))
          (tmp-buffer (get-buffer-create " *elmo-passwd-alist-tmp*"))
	  insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
          insert-file-contents-post-hook 
          ret-val)
      (if (not (file-readable-p filename))
          ()
        (set-buffer tmp-buffer)
        (insert-file-contents filename)
        (setq ret-val
              (condition-case nil
                  (read (current-buffer)) 
                (error nil nil))))
      (kill-buffer tmp-buffer)
      ret-val)))

(defun elmo-passwd-alist-save ()
  "Save password into file."
  (interactive)
  (save-excursion
    (let ((filename (expand-file-name elmo-passwd-alist-file-name
                                      elmo-msgdb-dir))
          (tmp-buffer (get-buffer-create " *elmo-passwd-alist-tmp*")))
      (set-buffer tmp-buffer)
      (erase-buffer)
      (prin1 elmo-passwd-alist tmp-buffer)
      (princ "\n" tmp-buffer)
;      (if (and (file-exists-p filename)
;             (not (equal 384 (file-modes filename))))
;        (error "%s is not safe.chmod 600 %s!" filename filename))
      (if (file-writable-p filename)
         (progn
           (write-region (point-min) (point-max) 
                         filename nil 'no-msg)
           (set-file-modes filename 384))
        (message (format "%s is not writable." filename)))
      (kill-buffer tmp-buffer))))

(defun elmo-get-passwd (user-at-host)
  "Get password from password pool."
  (let (data pass)
    (if (not elmo-passwd-alist)
	(setq elmo-passwd-alist (elmo-passwd-alist-load)))
    (setq data (assoc user-at-host elmo-passwd-alist))
    (if data
	(elmo-base64-decode-string (cdr data))
      (setq pass (elmo-read-passwd (format "Password for %s: " 
					   user-at-host) t))
      (setq elmo-passwd-alist
	    (append elmo-passwd-alist
		    (list (cons user-at-host 
				(elmo-base64-encode-string pass)))))
      pass)))

(defun elmo-remove-passwd (user-at-host)
  "Remove password from password pool (for failure)."
  (setq elmo-passwd-alist
	(delete (assoc user-at-host elmo-passwd-alist)
		elmo-passwd-alist
		)))

(defmacro elmo-read-char-exclusive ()
  (cond ((featurep 'xemacs)
         '(let ((table (quote ((backspace . ?\C-h) (delete . ?\C-?)
                               (left . ?\C-h))))
                event key)
            (while (not
                    (and
                     (key-press-event-p (setq event (next-command-event)))
                     (setq key (or (event-to-character event)
                                   (cdr (assq (event-key event) table)))))))
            key))
        ((fboundp 'read-char-exclusive)
         '(read-char-exclusive))
        (t
         '(read-char))))

(defun elmo-read-passwd (prompt &optional stars)
  "Read a single line of text from user without echoing, and return it."
  (let ((ans "")
	(c 0)
	(echo-keystrokes 0)
	(cursor-in-echo-area t)
	(log-message-max-size 0)
	message-log-max	done msg truncate)
    (while (not done)
      (if (or (not stars) (string= "" ans))
	  (setq msg prompt)
	(setq msg (concat prompt (make-string (length ans) ?.)))
	(setq truncate
	      (1+ (- (length msg) (window-width (minibuffer-window)))))
	(and (> truncate 0)
	     (setq msg (concat "$" (substring msg (1+ truncate))))))
      (message "%s" msg)
      (setq c (elmo-read-char-exclusive))
      (cond ((= c ?\C-g)
	     (setq quit-flag t
		   done t))
	    ((or (= c ?\r) (= c ?\n) (= c ?\e))
	     (setq done t))
	    ((= c ?\C-u)
	     (setq ans ""))
	    ((and (/= c ?\b) (/= c ?\177))
	     (setq ans (concat ans (char-to-string c))))
	    ((> (length ans) 0)
	     (setq ans (substring ans 0 -1)))))
    (if quit-flag
	(prog1
	    (setq quit-flag nil)
	  (message "Quit")
	  (beep t))
      (message "")
      ans)))

;; from subr.el
(defun elmo-replace-in-string (str regexp newtext &optional literal)
  "Replaces all matches in STR for REGEXP with NEWTEXT string,
 and returns the new string.
Optional LITERAL non-nil means do a literal replacement.
Otherwise treat \\ in NEWTEXT string as special:
  \\& means substitute original matched text,
  \\N means substitute match for \(...\) number N,
  \\\\ means insert one \\."
  (let ((rtn-str "")
	(start 0)
	(special)
	match prev-start)
    (while (setq match (string-match regexp str start))
      (setq prev-start start
	    start (match-end 0)
	    rtn-str
	    (concat
	      rtn-str
	      (substring str prev-start match)
	      (cond (literal newtext)
		    (t (mapconcat
			(function
			 (lambda (c)
			   (if special
			       (progn
				 (setq special nil)
				 (cond ((eq c ?\\) "\\")
				       ((eq c ?&)
					(elmo-match-string 0 str))
				       ((and (>= c ?0) (<= c ?9))
					(if (> c (+ ?0 (length
							(match-data))))
					; Invalid match num
					    (error "Invalid match num: %c" c)
					  (setq c (- c ?0))
					  (elmo-match-string c str)))
				       (t (char-to-string c))))
			     (if (eq c ?\\) (progn (setq special t) nil)
			       (char-to-string c)))))
			newtext ""))))))
    (concat rtn-str (substring str start))))

;; Just for utility
(defun elmo-add-to-cont-list (cont-list msg)
  (let ((elist cont-list)
	(ret-val cont-list)
	entity found)
    (while (and elist (not found))
      (setq entity (car elist))
      (cond 
       ((and (consp entity)
	     (eq (+ 1 (cdr entity)) msg))
	(setcdr entity msg)
	(setq found t))
       ((and (integerp entity)
	     (eq (+ 1 entity) msg))
	(setcar elist (cons entity msg))
	(setq found t))
       ((or (and (integerp entity) (eq entity msg))
	    (and (consp entity) 
		 (<= (car entity) msg)
		 (<= msg (cdr entity)))) ; $B$U$/$^$l$k(B
	(setq found t))); noop
      (setq elist (cdr elist)))
    (if (not found)
	(setq ret-val (append cont-list (list msg))))
    ret-val))

(defun elmo-string-to-list (string)
  (elmo-set-work-buf
   (let ((coding-system-for-read 'binary)
	 (coding-system-for-write 'binary))
     (insert string)
     (goto-char (point-min))
     (insert "(")
     (goto-char (point-max))
     (insert ")")
     (goto-char (point-min))
     (read (current-buffer)))))

(defmacro elmo-set-buffer-multibyte (flag)
  "Set the multibyte flag of the current buffer to FLAG."
  (cond ((boundp 'MULE)
         (list 'setq 'mc-flag flag))
        ((featurep 'xemacs)
         flag)
        ((and (boundp 'emacs-major-version) (>= emacs-major-version 20))
         (list 'set-buffer-multibyte flag))
        (t
         flag)))

(defun elmo-check-plugged ()
  (if (not elmo-plugged)
      (error "Not plugged.")))

(defun elmo-disk-usage (path)
  "Get disk usage (bytes) in PATH."
  (let ((file-attr 
	 (condition-case () (file-attributes path) (error nil))))
    (if file-attr
	(if (nth 0 file-attr) ; directory
	    (let ((files (condition-case () 
			     (directory-files path t "^[^\\.]")
			   (error nil)))
		  (result 0.0))
	      ;; (result (nth 7 file-attr))) ... directory size
	      (while files
		(setq result (+ result (or (elmo-disk-usage (car files)) 0)))
		(setq files (cdr files)))
	      result)
	  (float (nth 7 file-attr))))))

(defun elmo-get-last-accessed-time (path &optional dir)
  "Returns last accessed time."
  (let ((last-accessed (nth 4 (file-attributes (or (and dir
							(expand-file-name
							 path dir))
						   path)))))
    (if last-accessed
	(setq last-accessed (+ (* (nth 0 last-accessed)
				  (float 65536)) (nth 1 last-accessed)))
      0)))

(defun elmo-get-last-modification-time (path &optional dir)
  "Returns last accessed time."
  (let ((last-modified (nth 5 (file-attributes (or (and dir
							(expand-file-name
							 path dir))
						   path)))))
    (setq last-modified (+ (* (nth 0 last-modified)
			      (float 65536)) (nth 1 last-modified)))))

(defun elmo-make-directory (path)
  "create directory recursively."
  (let ((parent (directory-file-name (file-name-directory path))))
    (if (null (file-directory-p parent))
	(elmo-make-directory parent))
    (make-directory path)
    (if (string= path (expand-file-name elmo-msgdb-dir))
	(set-file-modes path 448) ; 700
      )
    ))

(defun elmo-list-filter (l1 l2)
  "L1 is filter."
  (if (eq l1 t)
      ;; t means filter all.
      nil
    (if l1
	(delete-if (lambda (x) (not (memq x l1))) l2)
      ;; filter is nil
      l2)))

(defun elmo-folder-writable-p (folder)
  (let ((type (elmo-folder-get-type folder)))
    (or (eq type 'imap4)
	(eq type 'localdir)
	(eq type 'archive))))

(defun elmo-folder-contains-type (folder type)
  (let ((spec (elmo-folder-get-spec folder)))
    (cond 
     ((eq (car spec) 'filter)
      (elmo-folder-contains-type (nth 4 spec) type))
     ((eq (car spec) 'multi)
      (let ((folders (cdr spec)))
	(catch 'done
	  (while folders
	    (if (elmo-folder-contains-type (car folders) type)
		(throw 'done t))
	    (setq folders (cdr folders))))))
     ((eq (car spec) type)
      t)
     (t nil))))

(defun elmo-folder-contains-multi (folder)
  (let ((cur-spec (elmo-folder-get-spec folder)))
    (catch 'done
      (while cur-spec
	(cond 
	 ((eq (car cur-spec) 'filter)
	  (setq cur-spec (elmo-folder-get-spec (nth 4 cur-spec))))
	 ((eq (car cur-spec) 'multi)
	  (throw 'done nil))
	 (t (setq cur-spec nil)))))
    cur-spec))

(defun elmo-multi-get-intlist-list (numlist &optional as-is)
  (let ((numbers (sort numlist '<))
	(cur-number 0)
	one-list int-list-list)
    (while numbers
      (setq cur-number (+ cur-number 1))
      (setq one-list nil)
      (while (and numbers 
		  (eq 0
		      (/ (- (car numbers)
			    (* elmo-multi-divide-number cur-number))
			 elmo-multi-divide-number)))
	(setq one-list (nconc
			one-list 
			(list 
			 (if as-is
			     (car numbers)
			   (% (car numbers)
			      (* elmo-multi-divide-number cur-number))))))
	(setq numbers (cdr numbers)))
      (setq int-list-list (nconc int-list-list (list one-list))))
    int-list-list))

(defsubst elmo-list-delete-if-smaller (list number)
  (let ((ret-val (copy-sequence list)))
    (while list
      (if (< (car list) number)
	  (setq ret-val (delq (car list) ret-val)))
      (setq list (cdr list)))
    ret-val))

(defun elmo-list-diff (list1 list2 &optional mes)
  (if mes 
      (message mes))
  (let ((clist1 (copy-sequence list1))
	(clist2 (copy-sequence list2))
	(max-of-l2 (or (nth (max 0 (1- (length list2))) list2) 0)))
    (while list2
      (setq clist1 (delq (car list2) clist1))
      (setq list2 (cdr list2)))
    (while list1
      (setq clist2 (delq (car list1) clist2))
      (setq list1 (cdr list1)))
    (if mes
	(message (concat mes "done.")))
    (list clist1 clist2)))

(defun elmo-list-bigger-diff (list1 list2 &optional mes)
  "Returns a list (- +). + is bigger than max of LIST1, in LIST2"
  (if (null list2)
      (cons list1  nil)
    (let* ((l1 list1)
	   (l2 list2)
	   (max-of-l2 (or (nth (max 0 (1- (length l2))) l2) 0))
	   diff1 num i percent
	   )
      (setq i 0)
      (setq num (+ (length l1)))
      (while l1
	(if (memq (car l1) l2)
	    (if (eq (car l1) (car l2))
		(setq l2 (cdr l2))
	      (delq (car l1) l2))
	  (if (> (car l1) max-of-l2)
	      (setq diff1 (nconc diff1 (list (car l1))))))
	(if mes
	    (progn
	      (setq i (+ i 1))
	      (setq percent (/ (* i 100) num))
	      (if (eq (% percent 5) 0)
		  (message "%s%d%%" mes percent))))
	(setq l1 (cdr l1)))
      (cons diff1 (list l2)))))

(defun elmo-multi-list-bigger-diff (list1 list2 &optional mes)
  (let ((list1-list (elmo-multi-get-intlist-list list1 t))
	(list2-list (elmo-multi-get-intlist-list list2 t))
	result
	dels news)
    (while (or list1-list list2-list)
      (setq result (elmo-list-bigger-diff (car list1-list) (car list2-list) 
					  mes))
      (setq dels (append dels (car result)))
      (setq news (append news (cadr result)))
      (setq list1-list (cdr list1-list))
      (setq list2-list (cdr list2-list)))
    (cons dels (list news))))

(defvar elmo-imap4-name-space-regexp-list nil)
(defun elmo-imap4-identical-name-space-p (fld1 fld2)
  ;; only on UW?
  (if (or (eq (string-to-char fld1) ?#)
	  (eq (string-to-char fld2) ?#))
      (string= (car (elmo-tokenize-string fld1 "/"))
	       (car (elmo-tokenize-string fld2 "/")))
    t))

(defun elmo-folder-identical-system-p (folder1 folder2)
  "folder1 and folder2 should be real folder (not virtual)."
  (let ((type1 (elmo-folder-get-type folder1))
	(type2 (elmo-folder-get-type folder2)))
    (cond ((eq type1 'imap4)
	   (let ((spec1 (elmo-folder-get-spec folder1))
		 (spec2 (elmo-folder-get-spec folder2)))
	     (and (elmo-imap4-identical-name-space-p
		   (nth 1 spec1) (nth 1 spec2))
		  (string= (nth 2 spec1) (nth 2 spec2))    ; hostname
		  (string= (nth 3 spec1) (nth 3 spec2))))) ; username
	  ((eq type1 'localdir)
	   (or (eq type2 'localdir)
	       (eq type2 'archive)))
	  (t nil))))

(defsubst elmo-file-field-string-match (file string field)
  (elmo-set-work-buf
    (let (ret-val fval)
      (as-binary-input-file
       (insert-file-contents file))
      (elmo-set-buffer-multibyte default-enable-multibyte-characters)
      (decode-mime-charset-region (point-min)(point-max) elmo-mime-charset)
      (goto-char (point-min))
      (cond 
       ((and elmo-date-match
	     (string-match "since" field))
	(let ((date (elmo-date-get-datevec string)))
	  (setq ret-val 
		(string<
		 (timezone-make-sortable-date (aref date 0) 
					      (aref date 1)
					      (aref date 2)
					      (timezone-make-time-string
					       (aref date 3) 
					       (aref date 4) (aref date 5)))
		 (timezone-make-date-sortable (std11-field-body "date"))))))
       ((string-match "before" field)
	(let ((date (elmo-date-get-datevec string)))
	  (setq ret-val 
		(string<
		 (timezone-make-date-sortable (std11-field-body "date"))
		 (timezone-make-sortable-date (aref date 0) 
					      (aref date 1)
					      (aref date 2)
					      (timezone-make-time-string
					       (aref date 3) 
					       (aref date 4) (aref date 5)))))))
       ((string-match "body" field)
	(re-search-forward "^$" nil t)	   ; goto body
	(setq ret-val (search-forward string nil t)))
       (t
	(if (setq fval (std11-field-body field))
	    (setq ret-val (string-match string fval)))))
      ret-val)))

(defun elmo-cross-device-link-error-p (err)
  (let ((errobj err)
	cur)
    (catch 'done
      (while errobj
	(if (and (stringp (setq cur (car errobj)))
		 (string-match "cross-device" cur))
	    (throw 'done t))
	(setq errobj (cdr errobj)))
      nil)))

(defmacro elmo-get-hash-val (string hashtable)
  (list 'symbol-value (list 'intern-soft string hashtable)))

(defmacro elmo-set-hash-val (string value hashtable)
  (list 'set (list 'intern string hashtable) value))

;; Make a hash table (default and minimum size is 128).
(defun elmo-make-hash (&optional hashsize)
  (make-vector (if hashsize (max (elmo-create-hash-size hashsize) 128) 128) 0))

(defun elmo-create-hash-size (min)
  (let ((i 1))
    (while (< i min)
      (setq i (* 2 i)))
    i))

(defun elmo-safe-filename (folder)
  (elmo-replace-in-string
   (elmo-replace-in-string folder "/" " ")
   ":" ">")
  )

(defvar elmo-msgid-replace-chars nil)

(defsubst elmo-replace-msgid-as-filename (msgid)
  "Replace message-id string as filename." 
  (setq msgid (elmo-replace-in-string msgid " " "  "))
  (if (null elmo-msgid-replace-chars)
      (setq elmo-msgid-replace-chars 
	    (regexp-quote (mapconcat 
			   'car elmo-msgid-replace-string-alist ""))))
  (while (string-match (concat "[" elmo-msgid-replace-chars "]")
		       msgid)
    (setq msgid (concat 
		 (substring msgid 0 (match-beginning 0))
		 (cdr (assoc 
		       (substring msgid 
				  (match-beginning 0) (match-end 0))
		       elmo-msgid-replace-string-alist))
		 (substring msgid (match-end 0)))))
  msgid)

(defsubst elmo-recover-msgid-from-filename (filename)
  "Recover Message-ID from filename."
  (let (tmp result next-point)
    (while (string-match " " filename)
      (setq tmp (substring filename 
			   (match-beginning 0)
			   (+ (match-end 0) 1)))
      (if (string= tmp "  ")
	  (setq tmp " ")
	(setq tmp (car (rassoc tmp 
			       elmo-msgid-replace-string-alist))))
      (setq result
	    (concat result 
		    (substring filename 0 (match-beginning 0))
		    tmp))
      (setq filename (substring filename (+ (match-end 0) 1))))
    (concat result filename)))

;; modules for Maildir.
(defun elmo-maildir-number-to-filename (dir number loc-alist 
					    &optional candidates)
  (let* ((candidates (or candidates
			 (directory-files 
			  (expand-file-name 
			   "cur"
			   dir) t)))
	 (location (cdr (assq number loc-alist))))
    (if location 
	(elmo-maildir-get-filename 
	 location candidates))))

(defun elmo-maildir-get-filename (location candidates)
  (catch 'done
    (while candidates
      (if (string-match (symbol-name location) (car candidates))
	  (throw 'done (car candidates)))
      (setq candidates (cdr candidates)))))

(defsubst elmo-copy-file (src dst)
  (condition-case err
      (elmo-add-name-to-file src dst t)
    (error (if (elmo-cross-device-link-error-p err)
	       (copy-file src dst t)
	     (error "copy file failed.")))))

(provide 'elmo-util)

