;;;
;;; cmail-folders.el
;;;
;;; Author: IWAMURO Motonori <iwa@mmp.fujitsu.co.jp>
;;;
(eval-when-compile
  (require 'cmail))

;;;
;;; $BJQ?t(B/$BDj?t(B
;;;

;; $B%-!<%^%C%W$N@_Dj(B
(defvar cmail-folders-mode-map nil)
(cond
 ((null cmail-folders-mode-map)
  (setq cmail-folders-mode-map (make-sparse-keymap))
  (define-key cmail-folders-mode-map "P" 'cmail-folders-previous-unread-folder)
  (define-key cmail-folders-mode-map "N" 'cmail-folders-next-unread-folder)
  (define-key cmail-folders-mode-map "p" 'cmail-folders-previous-line)
  (define-key cmail-folders-mode-map "n" 'cmail-folders-next-line)
  (define-key cmail-folders-mode-map "k" 'cmail-folders-previous-line)
  (define-key cmail-folders-mode-map "j" 'cmail-folders-next-line)
  (define-key cmail-folders-mode-map " " 'cmail-folders-visit-folder)
  (define-key cmail-folders-mode-map "\C-m" 'cmail-folders-visit-folder)
  (define-key cmail-folders-mode-map "\M- " 'cmail-folders-visit-folder-without-thread)
  (define-key cmail-folders-mode-map "\M-\C-m" 'cmail-folders-visit-folder-without-thread)
  (define-key cmail-folders-mode-map "\M-\C-t" 'cmail-toggle-thread-ignore-limit)
  (define-key cmail-folders-mode-map "=" 'cmail-folders-expand-folder-window)
  (define-key cmail-folders-mode-map "l" 'cmail-folders-redisplay)
  (define-key cmail-folders-mode-map "L" 'cmail-folders-redisplay-all)
  (define-key cmail-folders-mode-map "c" 'cmail-folders-catch-up-all)
  (define-key cmail-folders-mode-map "u" 'cmail-folders-visit-parent-dir)
  (define-key cmail-folders-mode-map "d" 'cmail-folders-visit-top-dir)
  (define-key cmail-folders-mode-map "x" 'cmail-folders-toggle-unfold)
  (define-key cmail-folders-mode-map "v" 'cmail-folders-visit-folder-directly)
  (define-key cmail-folders-mode-map "V" 'cmail-folders-visit-folder-directly-without-thread)
  (define-key cmail-folders-mode-map "m" 'cmail-folders-mail)
  (define-key cmail-folders-mode-map "g" 'cmail-folders-get-newmail)
  (define-key cmail-folders-mode-map "q" 'cmail-folders-quit)
  (define-key cmail-folders-mode-map "Q" 'cmail-folders-close-all-folders)
  (define-key cmail-folders-mode-map "w" 'cmail-folders-update)
  (define-key cmail-folders-mode-map "\M-w" 'cmail-folders-update)
  (define-key cmail-folders-mode-map "\M-y" 'cmail-folders-copy-folder)
  (define-key cmail-folders-mode-map "\M-c" 'cmail-folders-copy-folder)
  (define-key cmail-folders-mode-map "\M-n" 'cmail-folders-rename-folder)
  (define-key cmail-folders-mode-map "\M-k" 'cmail-folders-kill-folder)
  (define-key cmail-folders-mode-map "C" 'cmail-folders-copy-folder)
  (define-key cmail-folders-mode-map "R" 'cmail-folders-rename-folder)
  (define-key cmail-folders-mode-map "D" 'cmail-folders-kill-folder)
  (define-key cmail-folders-mode-map "h" 'describe-mode)
  (define-key cmail-folders-mode-map "?" 'describe-mode)
  (define-key cmail-folders-mode-map "\C-i" 'cmail-folders-to-summary)))

;; $B%a%K%e!<%P!<(B/$B%^%&%9$N@_Dj(B
(cond
 (window-system
  (defun cmail-folders-mouse-select (event)
    (interactive "e")
    (mouse-set-point event)
    (cmail-folders-visit-folder))
  (cond
   ((or (featurep 'cmail-e19) (featurep 'cmail-e20))
    ;; $B%a%K%e!<%P!<(B
    (define-key cmail-folders-mode-map [menu-bar] (make-sparse-keymap))
    (define-key cmail-folders-mode-map [menu-bar cmail]
      (cons "Cmail" (make-sparse-keymap "Cmail")))
    (define-key cmail-folders-mode-map [menu-bar cmail folders-quit]
      '("Folders Quit" . cmail-folders-quit))
    (define-key cmail-folders-mode-map [menu-bar cmail separator-quit]
      '("---"))
    (define-key cmail-folders-mode-map [menu-bar cmail update]
      '("Save All Folders" . cmail-folders-update))
    (define-key cmail-folders-mode-map [menu-bar cmail rename]
      '("Rename Folder" . cmail-folders-rename-folder))
    (define-key cmail-folders-mode-map [menu-bar cmail copy]
      '("Copy Folder" . cmail-folders-copy-folder))
    (define-key cmail-folders-mode-map [menu-bar cmail get]
      '("New Mail" . cmail-folders-get-newmail))
    (define-key cmail-folders-mode-map [menu-bar cmail mail]
      '("Mail Send" . cmail-folders-mail))
    (define-key cmail-folders-mode-map [menu-bar cmail unfold]
      '("Toggle Unfolding" . cmail-folders-toggle-unfold))
    (define-key cmail-folders-mode-map [menu-bar cmail top-dir]
      '("Top Directory" . cmail-folders-visit-top-dir))
    (define-key cmail-folders-mode-map [menu-bar cmail up-to-dir]
      '("Up to Directory" . cmail-folders-visit-parent-dir))
    (define-key cmail-folders-mode-map [menu-bar cmail visit]
      '("Visit Folder" . cmail-folders-visit-folder-directly))
    (if (fboundp 'customize-group)
	(progn
	  (define-key cmail-folders-mode-map [menu-bar cmail separator-folder]
	    '("--"))
	  (define-key cmail-folders-mode-map [menu-bar cmail customize]
	    '("Cmail Customize" . cmail-customize))))
    ;; $B%^%&%9%$%Y%s%H(B
    (define-key cmail-folders-mode-map [mouse-2] 'cmail-folders-mouse-select))
   ((featurep 'cmail-x20)
    ;; $B%^%&%9%$%Y%s%H(B
    (define-key cmail-folders-mode-map 'button2 'cmail-folders-mouse-select)))
  ))

;; $B%-%c%C%7%e%U%!%$%kL>(B
(defvar *cmail-folders-cache
  (expand-file-name ".cmail-folders-cache.el" cmail-path))

;; $B$=$NB>FbItJQ?t(B
(defvar *cmail-folders-current-dir "")
(defvar *cmail-folders-current-dir-length
  (length *cmail-folders-current-dir))
(defvar *cmail-folders-current-dir-pattern
  (concat "^\\(" (regexp-quote *cmail-folders-current-dir) "\\)."))
(defvar *cmail-folders-beginning-point nil)
(defvar *cmail-folders-updated-folders-list '())
(defvar *cmail-folders-removed-folders-list '())
(defvar *cmail-folders-cache-modified nil)
(defvar *cmail-folders-system-folders-list
  (list cmail-incoming-mails-folder *cmail-reorder-folder))
(defvar *cmail-folders-set-overlay 'ignore)
(defvar cmail-folders-mail-count-column 6) ;;; $B8e$G(Bcmail-vars.el$B$"$?$j$K0\F0(B
(defvar *cmail-folders-folder-format
  (concat "%s %" (int-to-string cmail-folders-mail-count-column) "d%s %s"))
(defvar *cmail-folders-dir-format
  (if (< cmail-folders-mail-count-column 3)
      (format "  %s: "
	      (make-string cmail-folders-mail-count-column ?*))
    (format "  %s***: "
	    (make-string (- cmail-folders-mail-count-column 3) 32))))
(defvar *cmail-folders-num-part-length
  (1- (length *cmail-folders-dir-format)))
(defvar *cmail-folders-sparate-line
  (concat (make-string cmail-folders-mail-count-column ?-) "--+---\n"))

;; $BDj?t(B
(defconst *cmail-folders-buffer "**cmail-folders**")
(defconst *cmail-folders-pattern "^. *[0-9*]+[-+:] ")
(defconst *cmail-folders-pattern2 (concat *cmail-folders-pattern "\\(.*\\)$"))
(defconst *cmail-folders-tmp-buffer " *cmail-folders-tmp*")
(defconst *cmail-folders-cache-buffer " *cmail-folders-cache*")

;;;
;;; $B%^%/%m(B/$B%$%s%i%$%s4X?tDj5A(B
;;;

;;; $B%-%c%C%7%e%(%s%H%j%j%9%H$N%/%j%"(B
(defsubst cmail-folders-init-cache-alist ()
  (setq *cmail-folders-cache-alist (list (list ""))))

;;; $B%-%c%C%7%e%(%s%H%j%j%9%H$N<B8zIt(B ($B@hF,$O%@%_!<(B)
(defsubst cmail-folders-cache-alist ()
  (cdr *cmail-folders-cache-alist))

;;; $B%-%c%C%7%e%(%s%H%j$N%G%#%l%/%H%j?<$5$rJV$9(B
(defsubst cmail-folders-get-ce-depth (entry)
  (nth 1 entry))

;;; $B%-%c%C%7%e%(%s%H%j$N%9%F!<%?%9CM$rJV$9(B
(defsubst cmail-folders-get-ce-status (entry)
  (nth 2 entry))

;;; $B%-%c%C%7%e%(%s%H%j$N%9%F!<%?%9CM$r@_Dj$9$k(B
(defsubst cmail-folders-set-ce-status (entry status)
  (setcar (nthcdr 2 entry) status))

;;; $B%-%c%C%7%e%(%s%H%j$N%a!<%k?t5Z$SL$FI%a!<%k?t$r<hF@$9$k(B
(defsubst cmail-folders-get-ce-mail-nums (entry)
  (nthcdr 3 entry))

;;; $B%-%c%C%7%e%(%s%H%j$N%a!<%k?t5Z$SL$FI%a!<%k?t$r@_Dj$9$k(B
(defsubst cmail-folders-set-ce-mail-nums (entry nums)
  (setcdr (nthcdr 2 entry) nums))

;;; $B%-%c%C%7%e%(%s%H%jCf$N%a!<%k?t$r<hF@$9$k(B ($B%U%)%k%@(B)
(defsubst cmail-folders-get-ce-mail-num (entry)
  (nth 3 entry))

;;; $B%-%c%C%7%e%(%s%H%jCf$N%a!<%k?t$r@_Dj$9$k(B ($B%U%)%k%@(B)
(defsubst cmail-folders-set-ce-mail-num (entry num)
  (setcar (nthcdr 3 entry) num))

;;; $B%-%c%C%7%e%(%s%H%jCf$N%U%)%k%@?t$r<hF@$9$k(B ($B%G%#%l%/%H%j(B)
(defsubst cmail-folders-get-ce-folder-num (entry)
  (nth 3 entry))

;;; $B%-%c%C%7%e%(%s%H%jCf$N%U%)%k%@?t$r@_Dj$9$k(B ($B%G%#%l%/%H%j(B)
(defsubst cmail-folders-set-ce-folder-num (entry num)
  (setcar (nthcdr 3 entry) num))

;;; $B%-%c%C%7%e%(%s%H%jCf$NL$FI%a!<%k?t$r<hF@$9$k(B
(defsubst cmail-folders-get-ce-unread-mail-num (entry)
  (nth 4 entry))

;;; $B%-%c%C%7%e%(%s%H%jCf$NL$FI%a!<%k?t$r@_Dj$9$k(B
(defsubst cmail-folders-set-ce-unread-mail-num (entry unum)
  (setcar (nthcdr 4 entry) unum))

;;; $B%-%c%C%7%e%(%s%H%j$N%(%s%H%j;HMQ%U%i%0$r<hF@$9$k(B
(defsubst cmail-folders-get-ce-update-flag (entry)
  (nthcdr 5 entry))

;;; $B%-%c%C%7%e%(%s%H%j$N%(%s%H%j;HMQ%U%i%0$r@_Dj$9$k(B
(defsubst cmail-folders-set-ce-update-flag (entry flag)
  (setcdr (nthcdr 4 entry) flag))

;;; $B2>A[%U%)%k%@$+(B?
(defsubst cmail-folders-is-virtual-folder (folder)
  (= (elt folder 0) ?/))

;;; $B%U%)%k%@$N%G%#%l%/%H%jItA4BN$rJV$9(B ($B%H%C%W3,AX0J30$O=*C<$K(B`/'$B$,IU$/(B)
;;; folder$B$N=*C<$,(B`/'$B$J$i$P(Bfolder$BA4BN$rJV$9(B
(defsubst cmail-folders-get-dir-head (folder)
  (substring folder 0 (string-match "[^/]+$" folder)))

;;; $B%U%)%k%@$N%G%#%l%/%H%jItA4BN$rJV$9(B ($B%H%C%W3,AX0J30$O=*C<$K(B`/'$B$,IU$/(B)
;;; folder$B$N=*C<$,(B`/'$B$J$i$P:G8e$N%G%#%l%/%H%jIt$O=|$+$l$k(B
(defsubst cmail-folders-get-dir-head2 (folder)
  (substring folder 0 (string-match "[^/]+/?$" folder)))

;;; $B%+%l%s%H%G%#%l%/%H%j$K$"$k%U%)%k%@$+(B?
(defsubst cmail-folders-is-in-dir (folder)
  (and (> (length folder) *cmail-folders-current-dir-length)
       (string-match *cmail-folders-current-dir-pattern folder)))

;;; $B%U%)%k%@Cf!"%+%l%s%H%G%#%l%/%H%j$+$i8+$F0l$DL\$N%G%#%l%/%H%jIt$rJV(B
;;; $B$9(B($B=*C<$K(B`/'$B$,IU$/(B)
;;; $B%G%#%l%/%H%jIt$,L5$1$l$P(Bnil
(defsubst cmail-folders-get-sub-dir (folder)
  (let ((sep (string-match "/" folder *cmail-folders-current-dir-length)))
    (if sep (substring folder *cmail-folders-current-dir-length (1+ sep)))))

;;; $B%U%)%k%@L>$+$i%+%l%s%H%G%#%l%/%H%j$NItJ,$r=|$$$?J8;zNs$rJV$9(B
(defsubst cmail-folders-get-sub-folder (folder)
  (substring folder *cmail-folders-current-dir-length))

;;; $B;XDj$N%U%)%k%@$K%^%C%A$9$k%Q%?!<%s$rJV$9(B
(defsubst cmail-folders-get-pattern (folder)
  (concat *cmail-folders-pattern
	  "\\(" (regexp-quote (cmail-folders-get-sub-folder folder)) "\\)$"))

;;; $B%+%l%s%H%G%#%l%/%H%j$NJQ99(B
(defsubst cmail-folders-chdir (dir)
  (setq *cmail-folders-current-dir dir
	*cmail-folders-current-dir-length (length dir)
	*cmail-folders-current-dir-pattern
	(concat "^\\(" (regexp-quote dir) "\\).")))

;;; $B%G%#%l%/%H%jL>(B
(defsubst cmail-folders-dir-name ()
  (if (zerop *cmail-folders-current-dir-length)
      "TOP" *cmail-folders-current-dir))

;;; $B2hLL%/%j%"$J$7$G(Brecenter$B$r$+$1$k(B
(defsubst cmail-folders-recenter ()
  (let ((height (window-height)))
    (if (> (count-lines (point-min) (1- (point-max))) height)
	(recenter (/ height 2)))))

;;; $B99?7$5$l$?%U%)%k%@$r%W%C%7%e$9$k(B
(defsubst cmail-folders-put-updated-folder (folder)
  (if (not (member folder *cmail-folders-updated-folders-list))
      (push folder *cmail-folders-updated-folders-list)))

;;; $B:o=|$5$l$?%U%)%k%@$r%W%C%7%e$9$k(B
(defsubst cmail-folders-put-removed-folder (folder)
  (setq *cmail-folders-updated-folders-list
	(delete folder *cmail-folders-updated-folders-list))
  (if (not (member folder *cmail-folders-removed-folders-list))
      (push folder *cmail-folders-removed-folders-list)))

;;; $B99?7$5$l$?%U%)%k%@$r%W%C%7%e$9$k(B
(defun cmail-folders-put-updated-folder2 (folder)
  (mapcar 'cmail-folders-put-updated-folder
	  (cons folder (cmail-folders-get-path-list folder))))

;;; $B:o=|$5$l$?%U%)%k%@$r%W%C%7%e$9$k(B
(defun cmail-folders-put-removed-folder2 (folder)
  (cmail-folders-put-removed-folder folder)
  (mapcar 'cmail-folders-put-updated-folder
	  (cmail-folders-get-path-list folder)))

;;; $B;XDj$N%U%)%k%@$N99?7>uBV$r<hF@$9$k!#(B
;;; save-excursion$B$7$F$$$J$$$N$GMWCm0U!#(B
;;; $B99?7$5$l$?%U%)%k%@$N>l9g!"(Bupdated-folders-list$B$KDI2C$5$l$k(B
(defsubst cmail-folders-is-updated-folder (folder)
  (set-buffer (cmail-find-buffer folder))
  (when (or *cmail-folders-modified
	    (cmail-modified-p folder))
    (push folder updated-folders-list)
    (setq *cmail-folders-cache-modified t)))

;;; $BDL>o$N%U%)%k%@$+(B?
(defsubst cmail-folders-is-normal-folder (folder)
  (not (member folder *cmail-folders-system-folders-list)))

;;; $B%G%#%l%/%H%j$N?<$5(B(=`/'$B$N?t(B)$B$r<hF@$9$k(B
(defun cmail-folders-get-depth (folder)
  (let ((index (1- (length folder)))
	(depth 0))
    (while (natnump index)
      (if (= (elt folder index) ?/)
	  (setq depth (1+ depth)))
      (setq index (1- index)))
    depth))

;;; $B%G%#%l%/%H%j%(%s%H%j$N@8@.(B
(defsubst cmail-folders-create-ce-directory (dir)
  (list dir (cmail-folders-get-depth dir) 'directory 0 0))

;;;
;;; $B4X?tDj5A(B
;;;

;;; $B%b!<%I@_Dj(B
(defun cmail-folders-mode ()
  "cmail$B$N%U%)%k%@$rA*Br$9$k(B.

$B%-!<%P%$%s%I$O0J2<$NDL$j(B,
\\{cmail-folders-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (use-local-map cmail-folders-mode-map)
  (setq mode-name "Folders")
  (setq major-mode 'cmail-folders-mode)
  (setq truncate-lines t)
  (setq case-fold-search nil)
  (setq buffer-read-only t)
  (run-hooks 'cmail-folders-mode-hook))

;;; $B%U%)%k%@0lMw$r>o;~I=<($5$;$k$+$I$&$+$r%H%0%k$9$k(B
(defun cmail-toggle-always-display-folders ()
  (interactive)
  (setq cmail-always-display-folders (not cmail-always-display-folders)))

;;; $B2<5-$N$h$&$J%j%9%H$rJV$9(B
;;; "DIR1/DIR2/"       -> ("DIR1/")
;;; "DIR1/DIR2/FOLDER" -> ("DIR1/" "DIR1/DIR2/")
;;; folder$B$,2>A[%U%)%k%@$J$i(Bnil$B$rJV$9(B
(defun cmail-folders-get-path-list (folder)
  (if (cmail-folders-is-virtual-folder folder)
      nil
    (let ((index (- (length folder) 2)) path-list)
      (while (natnump index)
	(if (= (elt folder index) ?/)
	    (push (substring folder 0 (1+ index)) path-list))
	(setq index (1- index)))
      path-list)))

;;; $B%U%!%$%k$N%5%$%:5Z$S=$@5;~9o$rJV$9(B
(defun cmail-folders-get-file-status (filename)
  (let ((status (file-attributes filename)))
    (if status
	(cons (nth 7 status) (nth 5 status)))))

;;; $B%U%)%k%@Cf$N%a!<%k?t(B/$BL$FI%a!<%k?t$r?t$($k!#(B
;;; filename$B$,M?$($i$l$?$H$-$O;XDj$N%U%!%$%k$rD4$Y$k(B
(defun cmail-folders-count-mails (folder &optional filename)
  (save-excursion
    (cmail-message-resource1 'folders-count-mails-1 folder)
    (if (null filename)
	(setq filename (expand-file-name folder cmail-path)))
    (let (num unum)
      (set-buffer (get-buffer-create *cmail-folders-tmp-buffer))
      (as-binary-input-file
       (insert-file-contents filename nil nil nil t))
      (goto-char (point-min))
      (cond
       ((re-search-forward *cmail-re-bdr nil t)
	(setq num (count-lines (point-min) (match-beginning 0)))
	(setq unum 0)
	(while (re-search-backward "^[0-9]+ U" nil t)
	  (setq unum (1+ unum)))))
      (erase-buffer)
      (cmail-message-resource1 'folders-count-mails-2 folder)
      (list num unum))))

;;; $B%P%C%U%!$r85$K%-%c%C%7%e$N%(%s%H%j$rJV$9(B
(defun cmail-folders-get-cache-entry-from-buffer (folder)
  (save-excursion
    (let ((unum 0))
      (set-buffer (cmail-find-buffer folder))
      (setq *cmail-folders-modified nil)
      (set-buffer (cmail-header-buffer folder))
      (save-restriction
	(widen)
	(goto-char (point-min))
	(while (re-search-forward "^[0-9]+ U" nil t)
	  (setq unum (1+ unum)))
	(list folder (cmail-folders-get-depth folder)
	      nil (count-lines (point-min) (point-max)) unum)))))

;;; $B%U%!%$%k$r85$K%-%c%C%7%e$N%(%s%H%j$rJV$9(B
(defun cmail-folders-get-cache-entry-from-file (folder)
  (let ((filename (expand-file-name folder cmail-path)))
    (nconc
     (list folder (cmail-folders-get-depth folder)
	   (cmail-folders-get-file-status filename))
     (cmail-folders-count-mails folder filename))))

;;; $B%-%c%C%7%e%(%s%H%j72(Bnew-entries$B$r(B*cmail-folders-cache-alist$B$KA^F~$9(B
;;; $B$k!#(Bnew-entries$B$NMWAG$O%=!<%H:Q$G$J$1$l$P$J$i$J$$!#$^$?%(%s%H%j$K$O(B
;;; *cmail-folders-cache-alist$B$HF1$8%-!<$,$"$C$F$O$J$i$J$$!#(B
(defun cmail-folders-add-new-entries (new-entries)
  (let ((alist *cmail-folders-cache-alist) alist-next)
    (mapcar
     '(lambda (new-entry)
	(while (and (setq alist-next (cdr alist))
		    (string< (car (car alist-next)) (car new-entry)))
	  (setq alist alist-next))
	(setcdr alist (cons new-entry alist-next)))
     (sort new-entries '(lambda (a b) (string< (car a) (car b)))))))

;;; entry$B$r(Bnew-entries$B$K%W%C%7%e!#%G%#%l%/%H%jIt$,%-%c%C%7%e>e$K$J$1$l(B
;;; $B$P!"%G%#%l%/%H%jIt$b(Bnew-entries$B$KDI2C$9$k(B ($B9_=g(B)
(defun cmail-folders-push-new-entry (entry)
  (push entry new-entries)
  (mapcar
   '(lambda (dir)
      (if (null (assoc dir (cmail-folders-cache-alist)))
	  (push (cmail-folders-create-ce-directory dir) new-entries)))
   (cmail-folders-get-path-list (car entry))))

;;; $B%P%C%U%!$KJ];}$5$l$?%U%)%k%@$N%-%c%C%7%e%(%s%H%j$r99?7$9$k!#(B
;;; all-opened-folders$B$,(Bt$B$J$i(Bopen$B$5$l$F$$$k%U%)%k%@A4$F$K4X$9$k%-%c%C(B
;;; $B%7%e%(%s%H%j$r99?7$9$k!#(Ball-opened-folders$B$,(Bnil$B$J$i99?7$5$l$?%U%)(B
;;; $B%k%@$K4X$9$k%(%s%H%j$r99?7$7!"99?7$5$l$?%U%)%k%@$N%j%9%H$rJV$9(B
(defun cmail-folders-update-cache-entries-on-buffer (&optional
						     all-opened-folders)
  (save-excursion
    (let (new-entries updated-folders-list)
      (mapcar
       '(lambda (folder)
	  (and (cmail-folders-is-normal-folder folder)
	       (or (cmail-folders-is-updated-folder folder)
		   all-opened-folders)
	       (let ((entry (assoc folder (cmail-folders-cache-alist)))
		     (new-entry
		      (cmail-folders-get-cache-entry-from-buffer folder)))
		 (if entry
		     (if (listp (cmail-folders-get-ce-status entry))
			 (setcdr entry (cdr new-entry))
		       (cmail-folders-set-ce-mail-nums
			entry (cmail-folders-get-ce-mail-nums new-entry)))
		   (cmail-folders-push-new-entry new-entry)))))
       *cmail-opened-folders-list)
      (cmail-folders-add-new-entries new-entries)
      updated-folders-list)))

;;; $B%G%#%l%/%H%j$r:F5"E*$KAv::$7$F(B*cmail-folders-cache-alist$B$r99?7$9$k!#(B
;;; $B!&F~NO(B
;;;   dir: $BAv::$9$k%G%#%l%/%H%j(B (cmail-path$B$+$i$NAjBP%Q%9(B)
;;;   status-flag: non nil$B$N$H$-%P%C%U%!>e$KJ];}$5$l$F$$$k%U%)%k%@$G$"$C(B
;;;                $B$F$b%U%!%$%k%9%F!<%?%9$r<h$j9~$`(B
;;; $B!&=PNO(B
;;;   new-entries: *cmail-folders-cache-alist$B$K%(%s%H%j$N$J$$%U%)%k%@(B
;;;                $B$KBP1~$9$k%(%s%H%j$N%j%9%H!#7k2LE*$K9_=g$G%=!<%H$5(B
;;;                $B$l$k!#(B
(defun cmail-folders-scan-folders-recursively (dir)
  (let ((dir-list (directory-files (expand-file-name dir cmail-path)
				   nil "^[^\.#%~][^~]+$")))
    (while dir-list
      (let* ((sub-folder (car dir-list))
	     (folder (concat dir sub-folder))
	     (filename (expand-file-name folder cmail-path))
	     status entry)
	(cond
	 ;; $B%G%#%l%/%H%j(B
	 ((file-directory-p filename)
	  (setq folder (concat folder "/"))
	  (setq entry (assoc folder (cmail-folders-cache-alist)))
	  (if entry (nconc entry t)
	    (push (cmail-folders-create-ce-directory folder) new-entries))
	  (cmail-folders-scan-folders-recursively folder))
	 ;; $B%U%)%k%@%U%!%$%k(B
	 ((setq status (cmail-folders-get-file-status filename))
	  (setq entry (assoc folder (cmail-folders-cache-alist)))
	  (if entry
	      (let ((e-status (cmail-folders-get-ce-status entry)))
		(cond
		 ((null e-status)
		  (if status-flag
		      (cmail-folders-set-ce-status entry status)))
		 ((not (equal e-status status))
		  (setcdr (cdr entry)
			  (cons status
				(cmail-folders-count-mails folder filename)))))
		(nconc entry t))
	    (push (nconc (list folder (cmail-folders-get-depth folder) status)
			 (cmail-folders-count-mails folder filename))
		  new-entries)))))
      (setq dir-list (cdr dir-list)))))

;;; *cmail-opened-folders-list$B$N%-%c%C%7%e%(%s%H%j!"5Z$SKvHx$N(Bcdr$B$,(Bt$B$G(B
;;; $B$"$k%-%c%C%7%e%(%s%H%j$N>l9g!"(Bcdr$B$r(Bnil$B$K$9$k!#$=$l0J30$N>l9g!"%-%c%C(B
;;; $B%7%e%(%s%H%j$r(B*cmail-folder-cache-alist$B$+$i:o=|$9$k(B
(defun cmail-folders-remove-unused-entries ()
  (let ((alist (copy-sequence (cmail-folders-cache-alist))))
    (while alist
      (let ((entry (car alist)))
	(if (or (null (cmail-folders-get-ce-status entry))
		(cmail-folders-get-ce-update-flag entry))
	    (cmail-folders-set-ce-update-flag entry nil)
	  (delq entry *cmail-folders-cache-alist)))
      (setq alist (cdr alist)))))

;;; $B%U%)%k%@%G%#%l%/%H%j$r%9%-%c%s$7$FA4%-%c%C%7%e%(%s%H%j$r99?7$9$k(B
(defun cmail-folders-scan-all-folders (&optional status-flag)
  (let (new-entries)
    (mapcar 'cmail-folders-put-updated-folder2
	    (cmail-folders-update-cache-entries-on-buffer t))
    (cmail-folders-scan-folders-recursively "")
    (cmail-folders-remove-unused-entries)
    (cmail-folders-add-new-entries new-entries)))

;;; $B%(%s%H%jCf$N%U%)%k%@?t$rA}2C$5$;$k(B
(defsubst cmail-folders-add-ce-folder-num (dir-entry num)
  (cmail-folders-set-ce-folder-num
   dir-entry (+ (cmail-folders-get-ce-folder-num dir-entry) num)))

;;; $B%(%s%H%jCf$NL$FI?t$rA}2C$5$;$k(B
(defsubst cmail-folders-add-ce-unread-mail-num (dir-entry num)
  (cmail-folders-set-ce-unread-mail-num
   dir-entry (+ (cmail-folders-get-ce-unread-mail-num dir-entry) num)))

;;; dir-entry-list$B$N@hF,$N%(%s%H%j$N%G%#%l%/%H%j$N?<$5$,!"(Bdepth$B0J2<$K(B
;;; $B$J$k$^$G(Bdir-entry-list$B$r%]%C%W$9$k!#$^$?!"%]%C%W$5$l$?%(%s%H%jCf$N(B
;;; $BL$FI%a!<%k?t$r>e0L$N%(%s%H%j$KB-$79~$`(B
;;; $B!&F~NO(B
;;;   depth: $B%G%#%l%/%H%j$N?<$5(B
;;;   cur-dir-depth: dir-entry-list$B$N@hF,$K$"$k%G%#%l%/%H%j$N?<$5(B
;;;   dir-entry-list: $B%G%#%l%/%H%j%(%s%H%j%j%9%H(B
;;; $B!&=PNO(B
;;;   $BJV$jCM(B: dir-entry-list$B$NCM(B
;;;   cur-dir-depth: dir-entry-list$B$N@hF,$K$"$k%G%#%l%/%H%j$N?<$5(B
;;;   dir-entry-list: $B%G%#%l%/%H%j%(%s%H%j%j%9%H(B
(defun cmail-folders-pop-dir-entry-list (depth)
  (let (dir-entry parent-dir-entry)
    (while (< depth cur-dir-depth)
      (setq dir-entry (pop dir-entry-list))
      (cond
       (dir-entry-list
	(setq parent-dir-entry (car dir-entry-list))
	(setq cur-dir-depth (cmail-folders-get-ce-depth parent-dir-entry))
	(cmail-folders-add-ce-unread-mail-num
	 parent-dir-entry
	 (cmail-folders-get-ce-unread-mail-num dir-entry)))
       (t
	(setq cur-dir-depth 0))))
    dir-entry-list))

;;; $B%G%#%l%/%H%j2<$N%U%)%k%@?t$r%+%&%s%H$9$k!#7k2L$O%-%c%C%7%e%(%s%H%j(B
;;; $B$K3JG<$5$l$k(B
(defun cmail-folders-count-folder-num ()
  (let ((alist (cmail-folders-cache-alist))
	dir-entry-list
	(cur-dir-depth 0))
    (while alist
      (let* ((entry (car alist))
	     (folder (car entry))
	     (depth (cmail-folders-get-ce-depth entry))
	     (unum (cmail-folders-get-ce-unread-mail-num entry)))
	(cond
	 ;; $BDL>o$N%U%!%$%k5Z$S2>A[%U%)%k%@$OL5;k(B
	 ((or (null unum) (cmail-folders-is-virtual-folder folder))
	  nil)
	 ;; $B%G%#%l%/%H%j(B
	 ((eq (cmail-folders-get-ce-status entry) 'directory)
	  (cmail-folders-set-ce-folder-num entry 0)
	  (cmail-folders-set-ce-unread-mail-num entry 0)
	  (if (<= depth cur-dir-depth)
	      ;; $B3,AX$,JQ$o$i$J$$(B/$B@u$/$J$k>l9g(B
	      (cmail-folders-pop-dir-entry-list (1- depth)))
	  (if dir-entry-list
	      (cmail-folders-add-ce-folder-num (car dir-entry-list) 1))
	  (push entry dir-entry-list)
	  (setq cur-dir-depth depth))
	 ;; $B%U%)%k%@(B ($B%H%C%W%G%#%l%/%H%j$N%U%)%k%@$O<+F0E*$KL5;k$5$l$k(B)
	 ((cmail-folders-pop-dir-entry-list depth)
	  (cmail-folders-add-ce-folder-num (car dir-entry-list) 1)
	  (cmail-folders-add-ce-unread-mail-num (car dir-entry-list) unum))))
      (setq alist (cdr alist)))
    (cmail-folders-pop-dir-entry-list 0)))

;;; $B5l%P!<%8%g%s$N%-%c%C%7%e%G!<%?$r%3%s%P!<%H$9$k(B
(defun cmail-folders-convert-old-cache ()
  (when (= cmail-folders-cache-format-version 1)
    (mapcar
     '(lambda (entry)
	(setcdr entry (cons (cmail-folders-get-depth (car entry))
			    (cdr entry))))
     (cmail-folders-cache-alist)))
  (when (<= cmail-folders-cache-format-version 2)
    (setq *cmail-folders-cache-alist
	  (sort *cmail-folders-cache-alist
		'(lambda (a b) (string< (car a) (car b)))))))

;;; $B%-%c%C%7%e%U%!%$%k$r%m!<%I$9$k(B
(defun cmail-folders-load-cache ()
  (save-excursion
    (cmail-message-resource 'folders-load-cache-1)
    (set-buffer (find-file-noselect *cmail-folders-cache))
    (auto-save-mode nil)
    (rename-buffer *cmail-folders-cache-buffer)
    (setq buffer-read-only t)
    (setq inhibit-read-only t)
    (cond
     ((and (boundp 'cmail-folders-cache-format-version)
	   (<= cmail-folders-cache-format-version 3))
      (eval-current-buffer)
      (cmail-folders-convert-old-cache))
     (t
      (erase-buffer)
      (set-buffer-modified-p nil)
      (cmail-folders-init-cache-alist)))
    (cmail-folders-scan-all-folders)
    (setq inhibit-read-only nil)
    (cmail-message-resource 'folders-load-cache-2)))

;;; $B%-%c%C%7%e%U%!%$%k$r%;!<%V$9$k!#(B
(defun cmail-folders-save-cache ()
  (save-excursion
    (let ((buf (get-buffer-create *cmail-folders-cache-buffer)))
      (set-buffer buf)
      (if buffer-file-name
	  nil
	(set-visited-file-name *cmail-folders-cache t)
	(auto-save-mode nil)
	(setq buffer-read-only t))
      (let ((inhibit-read-only t))
	(erase-buffer)
	(insert "(setq *cmail-folders-cache-alist '((\"\")\n")
	(mapcar
	 '(lambda (entry)
	    (when (cmail-folders-get-ce-status entry)
	      (prin1 entry buf)
	      (terpri buf)))
       (cmail-folders-cache-alist))
	(insert "))\n"
		"; Local Variables:\n"
		"; mode: Fundamental\n"
		"; cmail-folders-cache-format-version: 3\n"
		"; End:\n"))
      (save-buffer)
      (rename-buffer *cmail-folders-cache-buffer) ;keep buffer-name
      )))

;;; $B%U%)%k%@(B/$B%G%#%l%/%H%j$KBP1~$9$k=PNOJ8;zNs$rJV$9(B
(defun cmail-folders-format (entry)
  (format *cmail-folders-folder-format
	  (if (zerop (cmail-folders-get-ce-unread-mail-num entry)) " " "*")
	  (if cmail-folders-display-unread-mails
	      (cmail-folders-get-ce-unread-mail-num entry)
	    (cmail-folders-get-ce-mail-num entry))
	  (cond
	   ((listp (cmail-folders-get-ce-status entry)) ":")
	   ((member (car entry) cmail-folders-unfold-dirs) "-")
	   (t "+"))
	  (cmail-folders-get-sub-folder (car entry))))

;;; $BI=<($7$F$b$h$$%U%)%k%@$+(B?
(defun cmail-folders-output-p (folder)
  (and folder
       (string-match *cmail-folders-current-dir-pattern folder)
       (or (string-match "^[^/]+/?$" (substring folder (match-end 1)))
	   (member (cmail-folders-get-dir-head2 folder)
		   cmail-folders-unfold-dirs)
	   (member folder cmail-folders-unfold-dirs)
	   (and (cmail-folders-is-virtual-folder folder)
		(zerop *cmail-folders-current-dir-length)))))

;;; $B%(%s%H%j$KBP1~$9$kI=<(MQJ8;zNs$rJV$9!#(B
;;; $B=PNO$7$J$$%(%s%H%j$KBP$7$F$O(Bnil$B$rJV$9(B
(defun cmail-folders-folder-line (entry)
  (and (cmail-folders-get-ce-mail-num entry)
       (or cmail-folders-display-all-folders
	   (if cmail-always-display-folders
	       (string= (car entry) cmail-inbox-folder))
	   (> (cmail-folders-get-ce-unread-mail-num entry) 0))
       (cmail-folders-output-p (car entry))
       (cmail-folders-format entry)))

(defsubst cmail-folders-output-line (entry)
  (let ((line (cmail-folders-folder-line entry)))
    (if line
	(let ((pos (point)))
	  (insert line "\n")
	  (funcall *cmail-folders-set-overlay pos)))))

(defun cmail-folders-output-all-lines ()
  (let ((cache-alist (cmail-folders-cache-alist))
	(skip-folders-list (append cmail-folders-list-top
				   cmail-folders-list-bottom nil))
	cursor)
    (setq cursor cmail-folders-list-top)
    (while cursor
      (let ((entry (assoc (car cursor) cache-alist)))
	(if entry (cmail-folders-output-line entry))
	(setq cursor (cdr cursor))))
    (setq cursor cache-alist)
    (while cursor
      (let* ((entry (car cursor))
	     (folder (car entry)))
	(if (member folder skip-folders-list)
	    (setq skip-folders-list (delete folder skip-folders-list))
	  (cmail-folders-output-line entry)))
	(setq cursor (cdr cursor)))
    (setq cursor cmail-folders-list-bottom)
    (while cursor
      (let ((entry (assoc (car cursor) cache-alist)))
	(if entry (cmail-folders-output-line entry))
	(setq cursor (cdr cursor))))))

;;; $B%P%C%U%!$N:n@.(B
(defun cmail-folders-create-buffer ()
  (let ((buf (get-buffer-create *cmail-folders-buffer)) pos)
    (set-buffer buf)
    (if (not (eq major-mode 'cmail-folders-mode))
	(cmail-folders-mode))
    (setq mode-line-buffer-identification
	  (concat "cmail:   [" (cmail-folders-dir-name) "]"))
    (setq inhibit-read-only t)
    (erase-buffer)
    (insert "Directory = [" (cmail-folders-dir-name) "]\n"
	    *cmail-folders-sparate-line)
    (setq *cmail-folders-beginning-point (point))
    (if (zerop *cmail-folders-current-dir-length)
	nil
      (insert *cmail-folders-dir-format "../\n")
      (funcall *cmail-folders-set-overlay *cmail-folders-beginning-point))
    (cmail-folders-update-cache-entries-on-buffer)
    (cmail-folders-count-folder-num)
    (cmail-folders-output-all-lines)
    (setq *cmail-folders-updated-folders-list '())
    (setq *cmail-folders-removed-folders-list '())
    (set-buffer-modified-p nil)
    (setq inhibit-read-only nil)
    buf))

;;; $B%U%)%k%@%P%C%U%!$N99?7(B

;;; list$B>e$N=g=x$,(B e1 < e2 $B$+H]$+(B
(defun cmail-folders-<-on-list (e1 e2 list)
  (let ((l1 (member e1 list))
	(l2 (member e2 list)))
    (if (and l1 l2)
	(> (length l1) (length l2))
      'error)))

;;; $BA^F~0LCV$N8!:w(B
(defun cmail-folders-goto-inserting-point (sub-folder type)
  (while (and (re-search-forward *cmail-folders-pattern2 nil 0)
	      (cond
	       ((eq type 'middle)
		(let ((match-folder (match-string 1)))
		  (cond ((member match-folder sub-list-top) t)
			((member match-folder sub-list-bottom) nil)
			((string< match-folder sub-folder) t))))
	       ((eq type 'top)
		(eq (cmail-folders-<-on-list (match-string 1) sub-folder
					     sub-list-top) t))
	       (t ;;; (eq type 'bottom)
		(cmail-folders-<-on-list (match-string 1) sub-folder
					 sub-list-bottom)))))
  (beginning-of-line))

;;; $B;XDj$N%U%)%k%@$KBP1~$9$k9T$r99?7(B/$BDI2C$9$k(B
(defun cmail-folders-update-line (folder type)
  (if (cmail-folders-output-p folder)
      (let* ((sub-folder (cmail-folders-get-sub-folder folder))
	     (pattern (cmail-folders-get-pattern folder))
	     (entry (assoc folder (cmail-folders-cache-alist)))
	     (line (cmail-folders-format entry))
	     pos overlay)
	(cond
	 ((re-search-forward pattern nil t) ;; $B0lCW9T$"$j(B
	  (setq overlay t)
	  (setq pos (match-beginning 0))
	  (replace-match line t t))
	 ((or cmail-folders-display-all-folders;; $B0lCW9T$J$7(B
	      (> (cmail-folders-get-ce-unread-mail-num entry) 0))
	  (setq overlay t)
	  (cmail-folders-goto-inserting-point sub-folder type)
	  (backward-char)
	  (insert "\n")
	  (setq pos (point))
	  (insert line)))
	(and overlay
	     (funcall *cmail-folders-set-overlay pos)))))

;;; $B;XDj$N%U%)%k%@$KBP1~$9$k9T$r:o=|$9$k(B
(defun cmail-folders-remove-line (folder type)
  (and (cmail-folders-output-p folder)
       (re-search-forward (cmail-folders-get-pattern folder) nil t)
       (delete-region (match-beginning 0) (1+ (match-end 0)))))

;;; match-list$B$H(Blist$B$H$N6&DLItJ,$r(Bmatch-list$B$N=g$GJV$9!#(B
;;; list$B$K$*$1$k(Bmatch-list$B$H$N6&DLItJ,$O(Bnil$B$KCV49$5$l$k(B
(defun cmail-folders-pickup-folders (match-list list)
  (delq nil (mapcar
	     '(lambda (folder)
		(let ((mem (member folder list)))
		  (when mem
		    (setcar mem nil)
		    folder))) match-list)))

(defun cmail-folders-get-sub-folders-list (list)
  (delq nil
	(mapcar
	 '(lambda (folder)
	    (if (> (length folder) *cmail-folders-current-dir-length)
		(cmail-folders-get-sub-folder folder)
	      nil))
	 list)))
      
;;; $B%U%)%k%@%P%C%U%!$N3F9T$r99?7(B/$BDI2C(B/$B:o=|$9$k(B
(defun cmail-folders-modify-lines (func list)
  (let ((top-list (cmail-folders-pickup-folders 
		   cmail-folders-list-top list))
	(bottom-list (cmail-folders-pickup-folders
		      cmail-folders-list-bottom list))
	(sub-list-top (cmail-folders-get-sub-folders-list 
		       cmail-folders-list-top))
	(sub-list-bottom (cmail-folders-get-sub-folders-list
			  cmail-folders-list-bottom))
	folder-list type)
    (setq folder-list (nconc (list 'top)
			     top-list
			     (list 'middle)
			     (sort (delq nil list) 'string<)
			     (list 'bottom)
			     bottom-list))
    (while folder-list
      (let ((folder (car folder-list)))
	(if (symbolp folder)
	    (setq type folder)
	  (funcall func folder type))
	(setq folder-list (cdr folder-list))))))

;;; $B%U%)%k%@%P%C%U%!$N99?7(B
(defun cmail-folders-update-buffer ()
  (mapcar 'cmail-folders-put-updated-folder2
	  (cmail-folders-update-cache-entries-on-buffer))
  (cmail-folders-count-folder-num)
  (let ((buf (set-buffer *cmail-folders-buffer))
	(inhibit-read-only t))
    (goto-char *cmail-folders-beginning-point)
    (if (not (zerop *cmail-folders-current-dir-length))
	(beginning-of-line 2))
    (cmail-folders-modify-lines 'cmail-folders-update-line
				*cmail-folders-updated-folders-list)
    (goto-char *cmail-folders-beginning-point)
    (cmail-folders-modify-lines 'cmail-folders-remove-line
				*cmail-folders-removed-folders-list)
    (setq *cmail-folders-updated-folders-list '())
    (setq *cmail-folders-removed-folders-list '())
    (set-buffer-modified-p nil)
    buf))

;;; $B;XDj$5$l$?%U%)%k%@$N$"$k9T$X0\F0(B
(defun cmail-folders-goto-folder-line (folder)
  (if (cmail-folders-is-in-dir folder)
      (let ((pattern (cmail-folders-get-pattern folder)) window)
	(beginning-of-line)
	(when (or (re-search-forward pattern nil t)
		  (re-search-backward pattern nil t))
	  (goto-char (1- (match-beginning 1)))
	  (when (setq window (get-buffer-window (current-buffer)))
	    (select-window window)
	    (cmail-folders-recenter))
	  t))))

(defun cmail-folders-select-buffer ()
  (switch-to-buffer *cmail-folders-buffer)
  (if cmail-use-full-window
      (progn
	(delete-other-windows)
	(and cmail-always-display-folders
	     (cmail-folders-split-window t)))
    (set-window-configuration *cmail-window-conf)
    (let ((gbw (get-buffer-window *cmail-folders-buffer)))
      (if gbw
	  (select-window gbw)
	(switch-to-buffer *cmail-folders-buffer)))))

;;; cmail-folders$B%P%C%U%!$NI=<((B
(defun cmail-folders-buffer (&optional rescan recreate)
  (if rescan
      (cmail-folders-scan-all-folders))
  (let ((prev-dir *cmail-folders-current-dir)
	(buf (get-buffer *cmail-folders-buffer)))
    ;; cmail-current-folder$B$NCM$K1~$8$F%+%l%s%H%G%#%l%/%H%j$r@_Dj(B
    ;;   cmail-current-folder$B$,%U%)%k%@%b!<%I$,J];}$9$k%+%l%s%H%G%#%l(B
    ;;   $B%/%H%j$KB8:_$7$J$$>l9g!"0lC6%H%C%W%G%#%l%/%H%j$K0\$C$F$+$iI=(B
    ;;   $B<($NM-L5$r3NG'$7!"I=<($5$l$J$$$J$i(Bcmail-current-folder$B$NB8:_(B
    ;;   $B$9$k%G%#%l%/%H%j$K0\$k!#%G%#%l%/%H%jE83+5!G=$KBP1~$9$k$?$a$3(B
    ;;   $B$N$h$&$J;EMM$H$J$C$F$$$k!#Nc$($P!"%U%)%k%@(BDIR-A/FOLDER-1,
    ;;   DIR-B/FOLDER-2$B$,$"$k$H$-!"(BDIR-B$B$rE83+$7$F%H%C%W%G%#%l%/%H%j$+(B
    ;;   $B$iD>@\%"%/%;%9$G$-$k$h$&$K$7$F$$$k$H$9$k!#$3$3$G!"0lC6(B
    ;;   DIR-A/FOLDER-1$B$KF~$C$F$+$i%U%)%k%@%b!<%I$r7PM3$;$:$K(B
    ;;   DIR-B/FOLDER-2$B$KF~$C$?$H$9$k!#$3$N8e%U%)%k%@%b!<%I$K0\9T$9$k(B
    ;;   $B$H!"(BDIR-B$B$G$O$J$/%H%C%W%G%#%l%/%H%j$K0\F0$9$k$3$H$K$J$k(B
    (unless (cmail-folders-is-in-dir cmail-current-folder)
      (cmail-folders-chdir ""))
    (unless (cmail-folders-output-p cmail-current-folder)
      (cmail-folders-chdir
       (cmail-folders-get-dir-head cmail-current-folder)))
    (if (or rescan
	    recreate
	    (null buf)
	    (not (string= prev-dir *cmail-folders-current-dir)))
	(cmail-folders-create-buffer) ; $B%P%C%U%!$r?75,:n@.(B
      (cmail-folders-update-buffer))  ; $B%P%C%U%!$r99?7(B
    ))

(defun cmail-folders-fixcp ()
  (goto-char *cmail-folders-beginning-point)
  (if (or cmail-folders-display-unread-mails
	  cmail-always-display-folders
	  (not (re-search-forward "^\\*[ ]*[0-9]+[:\\+] " nil t)))
      (if (or (null cmail-current-folder)
	      (not (cmail-folders-goto-folder-line cmail-current-folder)))
	  (progn
	    (goto-char *cmail-folders-beginning-point)
	    (move-to-column *cmail-folders-num-part-length)))))

(defun cmail-folders-fixcp2 (folder)
  (if (and folder
	   (not (string= folder ""))
	   (>= (length folder) *cmail-folders-current-dir-length)
	   (cmail-folders-goto-folder-line folder))
      nil
    (goto-char *cmail-folders-beginning-point)
    (move-to-column *cmail-folders-num-part-length)))

(defun cmail-folders-push-curpos ()
  (let ((buf (get-buffer *cmail-folders-buffer)))
    (if buf
	(save-excursion
	  (set-buffer buf)
	  (cmail-folders-get-current-folder))
      cmail-current-folder)))

(defun cmail-folders-pop-curpos (pos)
  (let ((cmail-current-folder pos))
    (cmail-folders-fixcp)))

(defun cmail-folders (&optional rescan recreate confirm)
  "$B%U%)%k%@0lMw%b!<%I$KF~$k(B.
\\[universal-argument]$B$,M?$($i$l$?>l9g(B, $B%a!<%k%G%#%l%/%H%j$r:FAv::$9$k(B."
  (interactive "P")
  ;; cmail$B$,5/F0$5$l$F$$$J$1$l$P(Bcmail-start$B$r<B9T(B
  (if (null (get-buffer *cmail-summary-buffer))
      (save-excursion (cmail-start (if confirm cmail-inbox-folder))))
  (let ((pos (cmail-folders-push-curpos)))
    (cmail-folders-buffer rescan recreate)
    (cmail-folders-select-buffer)
    (cmail-folders-pop-curpos pos)))

(defun cmail-folders-expand-folder-window ()
  "$B%5%^%j%&%#%s%I%&$H%a%$%k%&%#%s%I%&$r>C5n$9$k(B"
  (interactive)
  (if cmail-use-full-window
      (delete-other-windows)
    (let ((swin (get-buffer-window *cmail-summary-buffer))
	  (mwin (get-buffer-window *cmail-mail-buffer)))
      (and swin (delete-window swin))
      (and mwin (delete-window mwin))))
  (switch-to-buffer *cmail-folders-buffer)
  (recenter))

(defun cmail-folders-redisplay ()
  (interactive)
  (let ((pos (cmail-folders-push-curpos)))
    (cmail-folders-create-buffer)
    (cmail-folders-pop-curpos pos)))

(defun cmail-folders-redisplay-all ()
  (interactive)
  (let ((pos (cmail-folders-push-curpos))
	(cmail-folders-display-all-folders t))
    (cmail-folders-create-buffer)
    (cmail-folders-pop-curpos pos)))

(defun cmail-folders-split-window (&optional called-from-folders-mode)
  "Summary$B%P%C%U%!$,I=<($5$l$F$$$?$i$=$A$i$KHt$V(B.
$BI=<($5$l$F$$$J$$>l9g$d(B, $B@8@.$5$l$F$$$J$$>l9g$K$O!$(B
$B$=$N%&%#%s%I%&$r@8@.$9$k(B."
  (let ((fol (car cmail-window-horizontal-size))
	(smr (cdr cmail-window-horizontal-size))
	wwi adf)
    (setq wwi (window-width))
    (and cmail-always-display-folders
	 (get-buffer *cmail-folders-buffer)
	 (if (and (eq cmail-always-display-folders 'auto)
		  (< wwi cmail-always-display-folders-threshold))
	     (setq adf nil)
	   (setq adf t)))
    (if adf
	(progn
	  (if fol
	      (let ((total (+ fol smr)))
		(setq fol (max window-min-width (/ (* wwi fol) total)))
		(setq smr (max window-min-width (/ (* wwi smr) total)))
		(if (< wwi smr)
		    (enlarge-window (- smr wwi))))
	    (if (< (- wwi window-min-width) smr)
		(setq smr (- wwi window-min-width)))
	    (setq fol (- wwi smr)))
	  (save-excursion
	    (split-window nil fol t)
	    (if called-from-folders-mode
		(switch-to-buffer-other-window *cmail-summary-buffer)
	      (switch-to-buffer *cmail-folders-buffer)
	      (save-excursion
		(cmail-folders-buffer)))
	    (other-window 1))))))

;;; cmail$BK\BN$H$NF14|=hM}(B

;;; folder$B$KBP1~$9$k%-%c%C%7%e%(%s%H%j$r99?7$9$k(B
(defun cmail-folders-update-cache-entry (folder)
  (let ((entry (assoc folder (cmail-folders-cache-alist))) new-entries)
    (if entry
	(setcdr entry
		(cdr (if (member folder *cmail-opened-folders-list)
			 (cmail-folders-get-cache-entry-from-buffer folder)
		       (cmail-folders-get-cache-entry-from-file folder))))
      (cmail-folders-push-new-entry
       (cmail-folders-get-cache-entry-from-file folder))
      (cmail-folders-add-new-entries new-entries))))

;;; folder$B$KBP1~$9$k%-%c%C%7%e%(%s%H%j$r:o=|$9$k(B
(defun cmail-folders-remove-cache-entry (folder)
  (let ((entry (assoc folder (cmail-folders-cache-alist))))
    (if entry
	(delq entry *cmail-folders-cache-alist))))

;;; $B%*!<%W%s:Q%U%)%k%@$K$*$1$k%-%c%C%7%e%(%s%H%j$r99?7$9$k(B
(defun cmail-folders-scan-opened-folders ()
  (mapcar 'cmail-folders-put-updated-folder2
	  (cmail-folders-update-cache-entries-on-buffer))
  (mapcar
   '(lambda (folder)
      (let ((entry (assoc folder (cmail-folders-cache-alist))))
	(and entry
	     (cmail-folders-is-normal-folder folder)
	     (not (cmail-folders-is-virtual-folder folder))
	     (cmail-folders-set-ce-status
	      entry
	      (cmail-folders-get-file-status
	       (expand-file-name folder cmail-path))))))
   *cmail-opened-folders-list))

;;; $B%*!<%W%s:Q%U%)%k%@$K$*$1$k%-%c%C%7%e%(%s%H%j$N%U%!%$%k%9%F!<%?%9It(B
;;; $B$r%/%j%"$9$k!#3F4X?t$G$=$N%(%s%H%j$,%*!<%W%s:Q$_%U%!%$%k$G$"$k$HG'(B
;;; $B<1$5$l$k!#(B
(defun cmail-folders-clear-opened-folder-status ()
  (mapcar
   '(lambda (folder)
      (if (cmail-folders-is-normal-folder folder)
	  (let ((entry (assoc folder (cmail-folders-cache-alist))))
	    (if entry
		(cmail-folders-set-ce-status entry nil)))))
   *cmail-opened-folders-list))

;;; cmail-copy-folder$B<B9T;~$K8F$S=P$5$l$k(B
(defun cmail-after-copy-folder-function (folder1 folder2)
  (when *cmail-folders-cache-alist
    (when (cmail-folders-is-normal-folder folder1)
      (cmail-folders-update-cache-entry folder1)
      (cmail-folders-put-updated-folder2 folder1))
    (when (cmail-folders-is-normal-folder folder2)
      (cmail-folders-update-cache-entry folder2)
      (cmail-folders-put-updated-folder2 folder2))
    (setq *cmail-folders-cache-modified t)))

;;; cmail-rename-folder$B<B9T;~$K8F$S=P$5$l$k(B
(defun cmail-after-rename-folder-function (folder1 folder2)
  (when *cmail-folders-cache-alist
    (when (cmail-folders-is-normal-folder folder1)
      (cmail-folders-remove-cache-entry folder1)
      (cmail-folders-put-removed-folder2 folder1))
    (when (cmail-folders-is-normal-folder folder2)
      (cmail-folders-update-cache-entry folder2)
      (cmail-folders-put-updated-folder2 folder2))
    (setq *cmail-folders-cache-modified t)))

;;; cmail-kill-folder$B<B9T;~$K8F$S=P$5$l$k(B
(defun cmail-after-kill-folder-function (folder)
  (when (and *cmail-folders-cache-alist
	     (cmail-folders-is-normal-folder folder))
    (cmail-folders-remove-cache-entry folder)
    (cmail-folders-put-removed-folder2 folder)
    (setq *cmail-folders-cache-modified t)))

;;; cmail-save-all-folders$B<B9T;~$K8F$S=P$5$l$k(B
(defun cmail-after-save-all-folders-function (mod)
  (when (and (or mod
		 *cmail-folders-cache-modified)
	     *cmail-folders-cache-alist)
    (cmail-message-resource 'after-save-all-folders-function-1)
    (cmail-folders-scan-opened-folders)
    (cmail-folders-save-cache)
    (cmail-folders-clear-opened-folder-status)
    (setq *cmail-folders-cache-modified nil)))

;;; cmail-quit$B<B9T;~$K8F$S=P$5$l$k(B
(defun cmail-after-quit-function ()
  (when (and *cmail-folders-cache-modified
	     *cmail-folders-cache-alist)
    (cmail-message-resource 'after-save-all-folders-function-1)
    (cmail-folders-scan-all-folders t)
    (cmail-folders-save-cache)
    (cmail-message-resource 'after-quit-function-2))
  (let (buf)
    (and (setq buf (get-buffer *cmail-folders-buffer))
	 (kill-buffer buf))
    (and (setq buf (find-buffer-visiting *cmail-folders-cache))
	 (kill-buffer buf))
    (and (setq buf (get-buffer *cmail-folders-tmp-buffer))
	 (kill-buffer buf)))
  (setq *cmail-folders-cache-modified nil)
  (setq *cmail-folders-cache-alist nil)
  (setq *cmail-folders-updated-folders-list nil)
  (setq *cmail-folders-removed-folders-list nil))

;;; cmail-dismiss-folder$B<B9T;~$K8F$S=P$5$l$k(B
(defun cmail-before-dismiss-folder-function (folder)
  (when (and *cmail-folders-cache-alist
	     (cmail-folders-is-normal-folder folder))
    (let ((entry (assoc folder (cmail-folders-cache-alist))))
      (when entry
	(if (cmail-folders-is-virtual-folder folder)
	    (progn
	      (delq entry *cmail-folders-cache-alist)
	      (cmail-folders-put-removed-folder folder))
	  (setcdr entry
		  (cdr (cmail-folders-get-cache-entry-from-buffer folder)))
	  (cmail-folders-set-ce-status entry
				       (cmail-folders-get-file-status
					(expand-file-name folder cmail-path)))
	  (cmail-folders-put-updated-folder2 folder))))))

(defun cmail-folders-previous-unread-folder ()
  "$B>e$NL$FI%a%$%k$N$"$k%U%)%k%@$X0\F0$9$k(B."
  (interactive)
  (let ((pos (point)))
    (beginning-of-line)
    (if (re-search-backward "^\\*[ ]*[0-9]+[:\\+]" nil t)
	(progn
	  (forward-char *cmail-folders-num-part-length)
	  (cmail-folders-visit-folder) t)
      (goto-char pos)
      (cmail-message-resource 'folders-previous-unread-folder-1)
      nil)))

(defun cmail-folders-next-unread-folder ()
  "$B2<$NL$FI%a%$%k$N$"$k%U%)%k%@$X0\F0$9$k(B."
  (interactive)
  (let ((found (re-search-forward "^\\*[ ]*[0-9]+[:\\+]" nil t)))
    (if found
	(cmail-folders-visit-folder)
      (cmail-message-resource 'folders-previous-unread-folder-1))
    found))

(defun cmail-folders-previous-line ()
  "$B>e$N%U%)%k%@$X0\F0$9$k(B."
  (interactive)
  (beginning-of-line 0)
  (if (>= (point) *cmail-folders-beginning-point)
      (move-to-column *cmail-folders-num-part-length)))

(defun cmail-folders-next-line ()
  "$B2<$N%U%)%k%@$X0\F0$9$k(B."
  (interactive)
  (beginning-of-line 2)
  (if (>= (point) *cmail-folders-beginning-point)
      (move-to-column *cmail-folders-num-part-length)))

(defun cmail-folders-to-summary ()
  "$B%U%)%k%@%P%C%U%!$+$i%5%^%j%P%C%U%!$K%+!<%=%k$r0\F0$5$;$k(B."
  (interactive)
  (cmail-select-buffer *cmail-summary-buffer))

(defun cmail-summary-to-folders ()
  "$B%5%^%j%P%C%U%!$+$i%U%)%k%@%P%C%U%!$K%+!<%=%k$r0\F0$5$;$k(B."
  (interactive)
  (cmail-select-buffer *cmail-folders-buffer)
  (beginning-of-line)
  (forward-char *cmail-folders-num-part-length))

;;; $B?F%G%#%l%/%H%j$r<hF@$9$k(B
(defun cmail-folders-get-parent-dir ()
  (let ((sep (string-match "/[^/]+/$" *cmail-folders-current-dir)))
    (if sep (substring *cmail-folders-current-dir 0 (1+ sep)) "")))

;;; $B8=:_9T$N%U%)%k%@L>(B
(defun cmail-folders-get-current-folder ()
  (save-excursion
    (when (>= (point) *cmail-folders-beginning-point)
      (beginning-of-line)
      (if (re-search-forward *cmail-folders-pattern2 nil t)
	  (let ((sub-folder (match-string 1)))
	    (goto-char (match-beginning 1))
	    (if (string= sub-folder "../")
		(cmail-folders-get-parent-dir)
	      (concat *cmail-folders-current-dir sub-folder)))))))

;;; $B%G%#%l%/%H%j$+H]$+(B
(defun cmail-folders-is-dir (folder)
  (or (zerop (length folder))
      (eq (elt folder (1- (length folder))) ?/)))

;;; $B%G%#%l%/%H%j$K(Bvisit$B$9$k(B
(defun cmail-folders-visit-dir (dir)
  (let ((prev-dir *cmail-folders-current-dir))
    (cmail-folders-chdir dir)
    (cmail-folders-create-buffer)
    (cmail-folders-select-buffer)
    (if (cmail-folders-is-in-dir prev-dir)
	(or (cmail-folders-goto-folder-line
	     (substring
	      prev-dir
	      0 (1+ (string-match "/" prev-dir
				  *cmail-folders-current-dir-length))))
	    (let ((pattern (cmail-folders-get-pattern prev-dir)))
	      (goto-char *cmail-folders-beginning-point)
	      (if (re-search-forward
		   (substring pattern 0 (1- (length pattern))) nil t)
		  (goto-char (match-beginning 1))
		(move-to-column *cmail-folders-num-part-length))))
      (goto-char *cmail-folders-beginning-point)
      (move-to-column *cmail-folders-num-part-length))))

;;; $B%U%)%k%@$K(Bvisit$B$9$k(B
(defun cmail-folders-visit-sub-folder (folder)
  (let (*cmail-prev-folder)
    (set-buffer *cmail-summary-buffer)
    (cmail-visit-folder folder t)))

(defun cmail-folders-visit-folder-without-thread ()
  "FOLDER$B$KF~$k(B. thread$BI=<(BP>]$N(BFOLDER$B$G$b(Bthread$BI=<($r9T$o$J$$(B."
  (interactive)
  (let ((folder (cmail-folders-get-current-folder)))
    (if folder
	(if (cmail-folders-is-dir folder)
	    (cmail-folders-visit-dir folder)
	  (cmail-get-folder folder)
	  (setq *cmail-disp-thread nil)
	  (setq cmail-current-folder nil)
	  (cmail-folders-visit-sub-folder folder)))))

(defun cmail-folders-visit-folder ()
  "FOLDER$B$KF~$k(B."
  (interactive)
  (let ((folder (cmail-folders-get-current-folder)))
    (if folder
	(if (cmail-folders-is-dir folder)
	    (cmail-folders-visit-dir folder)
	  (cmail-folders-visit-sub-folder folder)))))

(defun cmail-folders-visit-parent-dir ()
  "$B0l$D>e$N3,AX$K0\$k(B."
  (interactive)
  (cmail-folders-visit-dir (cmail-folders-get-parent-dir)))

(defun cmail-folders-visit-top-dir ()
  "$B%H%C%W$N3,AX$K0\$k(B."
  (interactive)
  (cmail-folders-visit-dir ""))

;;; $B%G%#%l%/%H%j$rE83+$9$k(B
(defun cmail-folders-unfold-dir (dir)
  (let ((pos (point)))
    (push dir cmail-folders-unfold-dirs)
    (cmail-folders-create-buffer)
    (goto-char pos)
    (cmail-folders-recenter)))

;;; $B%G%#%l%/%H%j$r@^$j>v$`(B
(defun cmail-folders-fold-dir (dir)
  (setq cmail-folders-unfold-dirs (delete dir cmail-folders-unfold-dirs))
  (cmail-folders-create-buffer)
  (let ((path-list (cons dir (nreverse (cmail-folders-get-path-list dir)))))
    (while (cond
	    ((cmail-folders-goto-folder-line (car path-list)) nil)
	    ((setq path-list (cdr path-list)) t)
	    (t (goto-char *cmail-folders-beginning-point)
	       (move-to-column *cmail-folders-num-part-length) nil)))))

(defun cmail-folders-toggle-unfold ()
  "$B%G%#%l%/%H%j$NE83+>uBV$r@Z$j49$($k(B."
  (interactive)
  (let ((dir (cmail-folders-get-dir-head (cmail-folders-get-current-folder))))
    (cond
     ((or (string= dir "/")
	  (<= (length dir) *cmail-folders-current-dir-length))
      nil)
     ((member dir cmail-folders-unfold-dirs)
      (cmail-folders-fold-dir dir))
     (t
      (cmail-folders-unfold-dir dir)))))

(defun cmail-folders-visit-folder-directly-without-thread (folder)
  "$BD>@\(BFOLDER$B$KF~$k(B. thread$BI=<(BP>]$N(BFOLDER$B$G$b(Bthread$BI=<($r9T$o$J$$(B."
  (interactive (list (cmail-complete-foldername (cmail-get-resource 'visit-folder-1))))
  (cmail-get-folder folder)
  (setq *cmail-disp-thread nil)
  (setq cmail-current-folder nil)
  (cmail-folders-visit-sub-folder folder))

(defun cmail-folders-visit-folder-directly (folder)
  "$BD>@\(BFOLDER$B$KF~$k(B."
  (interactive (list (cmail-complete-foldername (cmail-get-resource 'visit-folder-1))))
  (cmail-folders-visit-sub-folder folder))

(defun cmail-folders-mail ()
  "$B%a%$%k$r=P$9(B."
  (interactive)
  (let ((gbw (get-buffer-window *cmail-summary-buffer)))
    (if gbw
	(select-window gbw))
    (cmail-mail2 t)))

(defun cmail-folders-update (&optional rescan)
  "$BA4$F$N%U%)%k%@$r%;!<%V$7$F%U%)%k%@0lMw2hLL$r99?7$9$k(B.
\\[universal-argument]$B$,M?$($i$l$?>l9g(B, $B%a!<%k%G%#%l%/%H%j$r:FAv::$9$k(B."
  (interactive "P")
  (if (null rescan)
      (save-excursion
	(cmail-message-resource 'folders-update-1)
	(save-window-excursion
	  (cmail-save-all-folders)))
    (save-excursion
      (save-window-excursion
	(set-buffer *cmail-summary-buffer)
	(cmail-flush-folders)))
    (cmail-folders-save-cache)
    (cmail-message-resource 'folders-update-2)
    (cmail-folders-scan-all-folders))
  (let ((folder (cmail-folders-get-current-folder)))
    (cmail-folders-create-buffer)
    (cmail-folders-fixcp2 folder))
  (when rescan
    (cmail-message-resource 'folders-update-3)))

(defun cmail-folders-get-newmail ()
  "$B?7Ce%a!<%k$r<hF@$9$k(B."
  (interactive)
  (cond
   ((cmail-get-spooled-mail)
    (setq cmail-current-folder cmail-inbox-folder)
    (cond
     (cmail-archive-on-get
      (cmail-make-summary cmail-inbox-folder)
      (cmail-auto-archive)
      (cmail-folders nil t))
     (cmail-use-folders-mode-first
      (cmail-folders nil t)
      (cmail-make-summary cmail-inbox-folder)
      (cmail-folders-visit-sub-folder cmail-inbox-folder))
     (t
      (cmail-folders nil t))))
   (t
    (or cmail-folders-display-all-folders
	(cmail-folders nil t))
    (cmail-message-resource 'get-spooled-mail-9))))

;;; interactive$B$N$?$a$N%Q%i%a!<%?$rJV$9(B
(defun cmail-folders-get-interactive-param (prompt emes)
  (let ((folder (cmail-folders-get-current-folder)))
    (if (and folder (not (cmail-folders-is-dir folder)))
	(list folder
	      (cmail-complete-foldername
	       (cmail-format-resource1 prompt folder) nil nil *cmail-folders-current-dir))
      (cmail-error-resource emes)
      (list nil nil))))

;;; cmail-folders$B%P%C%U%!Cf$N0LCV$r%^!<%/$9$k(B
(defun cmail-folders-keep-position ()
  (beginning-of-line)
  (point-marker))

;;; cmail-folders$B%P%C%U%!Cf$G%^!<%/$7$?0LCV$r2sI|$9$k(B
(defun cmail-folders-recover-position (m)
  (goto-char m)
  (beginning-of-line)
  (if (or (re-search-forward *cmail-folders-pattern nil t)
	  (re-search-backward *cmail-folders-pattern nil t))
      (goto-char (1- (match-end 0))))
  (set-marker m nil))

(defun cmail-folders-copy-folder (folder1 folder2)
  "$B%+%l%s%H9T$N(BFOLDER$B$r(BFOLDER2$B$K%3%T!<$9$k(B.
$BCm0U;v9`$O(Bcmail-copy-folder$B$HF1$8(B."
  (interactive (cmail-folders-get-interactive-param
		'folders-copy-folder-1 'folders-copy-folder-2))
  (when folder1
    (save-window-excursion
      (cmail-copy-folder folder1 folder2))
    (let ((m (cmail-folders-keep-position)))
      (cmail-folders-update-buffer)
      (cmail-folders-recover-position m))))

(defun cmail-folders-rename-folder (folder1 folder2)
  "$B%+%l%s%H9T$N(BFOLDER$B$NL>A0$r(BFOLDER2$B$K$9$k(B.
$BCm0U;v9`$O(Bcmail-rename-folder$B$HF1$8(B."
  (interactive (cmail-folders-get-interactive-param
		'folders-rename-folder-1 'folders-rename-folder-2))
  (when folder1
    (save-window-excursion
      (cmail-rename-folder folder1 folder2))
    (let ((m (cmail-folders-keep-position)))
      (cmail-folders-update-buffer)
      (cmail-folders-recover-position m))))

;;; $B%+%l%s%H9T$N%G%#%l%/%H%j$r:o=|$9$k(B
(defun cmail-folders-kill-dir (dir)
  (when (yes-or-no-p (cmail-format-resource1 'folders-kill-dir-1 dir))
    (delete-directory (expand-file-name dir cmail-path))
    (cmail-folders-remove-cache-entry dir)
    (cmail-folders-put-removed-folder dir)))

(defun cmail-folders-kill-folder ()
  "$B%+%l%s%H9T$N(BFOLDER$B$r:o=|$9$k(B."
  (interactive)
  (let ((folder (cmail-folders-get-current-folder))
	m *cmail-prev-folder)
    (cond
     ((null folder)
      (cmail-error-resource 'folders-kill-folder-1))
     ((cmail-folders-is-dir folder)
      (if (<= (length folder) *cmail-folders-current-dir-length)
	  (cmail-error-resource 'folders-kill-folder-2))
      (cmail-folders-kill-dir folder))
     (t
      (save-window-excursion
	(condition-case nil
	    (cmail-kill-folder folder)
	  (error nil)))))
    (setq m (cmail-folders-keep-position))
    (cmail-folders-update-buffer)
    (cmail-folders-recover-position m)))

(defun cmail-folders-quit-internal ()
  (cond
   ((y-or-n-p (cmail-get-resource 'folders-quit-internal-1))
    (cmail-select-buffer *cmail-summary-buffer)
    (if (not cmail-archive-on-quit)
	(cmail-quit t)
      (if (not (string= cmail-current-folder cmail-inbox-folder))
	(cmail-folders-visit-sub-folder cmail-inbox-folder))
      (cmail-quit)))
   (t
    (message nil))))

(defun cmail-folders-quit ()
  "cmail$B$r=*N;$9$k(B."
  (interactive)
  (if (and cmail-use-visiting-parent-dir-as-quit
	   (not (zerop *cmail-folders-current-dir-length)))
      (cmail-folders-visit-parent-dir)
    (if cmail-use-folders-mode-first
	(cmail-folders-quit-internal)
      (cmail-folders-visit-sub-folder cmail-current-folder))))

(defun cmail-folders-catch-up-all ()
  (interactive)
  (cmail-folders-visit-folder)
  (cmail-catch-up-all nil))

;;; $B%U%)%k%@0lMw$rJV$9(B
(defun cmail-folders-get-folders-list (&optional with-dir)
  (mapcar 'cmail-folders-put-updated-folder2
	  (cmail-folders-update-cache-entries-on-buffer))
  (let (folders-list folder)
    (mapcar
     '(lambda (entry)
	(and (cmail-folders-get-ce-mail-num entry)
	     (or (listp (cmail-folders-get-ce-status entry)) with-dir)
	     (progn
	       (setq folder (car entry))
	       (and (eq cmail-compressed-folders 'jam-zcat)
		    (string-match "\\.gz$\\|\\.Z$" folder)
		    (setq folder (substring folder 0 (match-beginning 0))))
	       (push (list folder) folders-list))))
     (cmail-folders-cache-alist))
    (nreverse folders-list)))

(defun cmail-folders-close-all-folders ()
  "INBOX$B$r=|$/A4%U%)%k%@$r%/%m!<%:$9$k(B."
  (interactive)
  (save-excursion
    (save-window-excursion
      (cmail-save-all-folders)
      (mapcar
       '(lambda (folder)
	  (if (equal folder cmail-inbox-folder)
	      (cmail-save-folder folder)
	    (cmail-dismiss-folder folder)
	    (cmail-message-resource1 'folders-close-all-folders-1 folder)))
       *cmail-opened-folders-list))
    (cmail-message-resource 'folders-close-all-folders-2)))

(provide 'cmail-folders)
