#
# 1.0 (Feb 1995)
#
# $Id: Enforcer.w,v 1.1 1996-09-25 09:23:42+02 mho Exp $

@class XfwfEnforcer (XfwfBoard) @file=Enforcer

@ The Enforcer widget can be used to apply location resources to a
widget that is not a subclass of XfwfBoard. The Widget accepts a
single child and forces that child to the same size as itself (minus
the frame).

It can also be used to put a frame around some widget and to add a label.

@EXPORTS

@ The symbolic constants specify which keys are used to for keyboard
traversal. Any keys not specified in |traversalKeys| will be propagated
to |propagateTarget|.

	@def XfwfTraverseKeyTab		=   1
	@def XfwfTraverseKeyHome	=   2
	@def XfwfTraverseKeyUp		=   4
	@def XfwfTraverseKeyDown	=   8
	@def XfwfTraverseKeyLeft	=  16
	@def XfwfTraverseKeyRight	=  32
	@def XfwfTraverseKeyKPEnter	=  64
	@def XfwfTraverseKeyAll		= 127

@PUBLIC

@ There are two ways the geometry of the child may be handled. First
the geometry may be forced to fit to the Enforcer, or second the Enforcer
may shrink to fit around the child.

	@var Boolean shrinkToFit = FALSE

@ The default width and height are changed to 10 x 10.

	@var float rel_width = <String> "0.0"
	@var float rel_height = <String> "0.0"
	@var Position abs_width = 10
	@var Position abs_height = 10

@ The label must be a single line. It is displayed superimposed on the
frame, in the upper lefthand corner. Currently, it must be simple
string in a single font.

	@var String label = NULL

@ If the highlightThickness is set to 0 it is sometimes nice to have a
little offset for the label.

	@var Dimension labelOffset = 0

@ The font for the label is set with |font|.

	@var <FontStruct> XFontStruct *font = <String> XtDefaultFont

@ The foreground color is the color used to draw the text.

	@var Pixel foreground = <String> XtDefaultForeground

@ The label can be aligned in the widget the following ways:
|XfwfTop|, |XfwfTopLeft| (i.e. left label aligned at top), and
|XfwfLeft| (i.e. left label aligned at center).

	@var Alignment alignment = XfwfTop

@ The |propagateTarget| will receive the key events that occure by
traversal.

	@var <PropagateTarget> Widget propagateTarget = 0

@ Which keys should be allowed for keyboard traversal

	@var int traversalKeys = XfwfTraverseKeyAll

@ Should it be allowed to quote the |TAB| character using |Ctrl-Q|?

	@var Boolean allowQuotedInsert = TRUE

@ For some widgets it is the user may desire to change the focus, when
the pointer enters the window (e.g. text windows). By default it is switched
off.

	@var Boolean focusOnEnter = FALSE

@PRIVATE

@ Keep last |Ctrl-Q|.

	@var Boolean quotedInsert


@TRANSLATIONS

	@trans <Enter>:		focusOnEnter()
        @trans <FocusIn>:	focusIn()
	@trans <FocusOut>:	focusOut()
	@trans <Key>:		propagateKey()

@UTILITIES

@ Check if the current key is selected by traversalKeys.

@def CHECK(key) =
    if ( (key & $traversalKeys) && ($traversalOn) )
	do_propagate = FALSE;
    else
	break;

@ACTIONS

@ Change focus, when pointer enters the window.

@proc focusOnEnter
{
    if ($focusOnEnter) {
	Time time = event->xcrossing.time;
	XtCallAcceptFocus($, &time);
    }
}

@ The |propagateKey| sends the key event to a |propagateTarget|.

@proc propagateKey
{
    Bool do_propagate = TRUE;
    /* check if key is used for keyboard traversal */
    KeySym keysym; (void)XLookupString(&(event->xkey), NULL, 0, &keysym, NULL);
    if ($allowQuotedInsert && $quotedInsert) {
	$quotedInsert = FALSE;
    }
    else if ((event->xkey.state & ControlMask)
	 &&  (keysym == 'Q' || keysym == 'q')
         &&  (XfwfTraverseKeyTab & $traversalKeys)
	 &&  ($traversalOn)
	 &&  ($allowQuotedInsert))
    {
	$quotedInsert = TRUE;
    }
    else if (!event->xkey.state) 
	switch (keysym) {
	case XK_Tab: CHECK(XfwfTraverseKeyTab)
	    $traverse($, TraverseNext, $, &event->xkey.time);
	    break;
	case XK_Home: CHECK(XfwfTraverseKeyHome)
	    $traverse($, TraverseHome, $, &event->xkey.time);
	    break;
	case XK_Up: CHECK(XfwfTraverseKeyUp)
	    $traverse($, TraverseUp, $, &event->xkey.time);
	    break;
	case XK_Down: CHECK(XfwfTraverseKeyDown)
	    $traverse($, TraverseDown, $, &event->xkey.time);
	    break;
	case XK_Left: CHECK(XfwfTraverseKeyLeft)
	    $traverse($, TraverseLeft, $, &event->xkey.time);
	    break;
	case XK_Right: CHECK(XfwfTraverseKeyRight)
	    $traverse($, TraverseRight, $, &event->xkey.time);
	    break;
	case XK_KP_Enter: CHECK(XfwfTraverseKeyKPEnter)
	    $traverse($, TraverseNextTop, $, &event->xkey.time);
	    break;
	}
    else if ((event->xkey.state & ShiftMask)
	 &&  (keysym == XK_Tab)
         &&  (XfwfTraverseKeyTab & $traversalKeys)
	 &&  ($traversalOn))
    {
	$traverse($, TraversePrev, $, &event->xkey.time);
	do_propagate = FALSE;
    }


    /* propaget key to child if specified and key not used for traversal */
    if ($propagateTarget && do_propagate) {
	event->xkey.display	= XtDisplay($propagateTarget);
	event->xkey.send_event	= True;
	event->xkey.window	= XtWindow($propagateTarget);
#if XlibSpecificationRelease < 6
	XSendEvent(XtDisplay($propagateTarget), XtWindow($propagateTarget),
		   FALSE, KeyPressMask, event);
#else
	XtDispatchEventToWidget($propagateTarget, event);
#endif
    }
}

@PRIVATE

@ |labelWidth| and |labelHeight| are stored for faster access.

	@var Dimension labelWidth

	@var Dimension labelHeight

@ The GC is used for the text.

	@var GC textgc

@METHODS

@ The |initialize| method initializes the private variables.

@proc initialize
{
    Dimension tmp;

    $textgc = NULL;
    make_textgc($);
    $quotedInsert = FALSE;

    /* allocate string */
    if ($label) $label = XtNewString($label);
    /* compute initial size */
    compute_label_size($);
    if ($label) switch ($alignment) {
    case XfwfTop: default:
	$alignment = XfwfTop; /* secure valid alignment */
	tmp        = $height + $labelHeight;
	$height    = max($height, tmp);
	break;
    case XfwfLeft: case XfwfTopLeft:
	tmp    = $width + $labelWidth;
	$width = max($width, tmp);
	break;
    }
}

@ Free private data.

@proc destroy
{
    if ($textgc) XtReleaseGC($, $textgc); $textgc = NULL;
    if ($label)  XtFree($label); $label = NULL;
}

@ The |set_values| method has to deal with changes in |label|, |font|
or |foreground|.

@proc set_values
{
    Boolean need_redraw = False;

    if ($old$label != $label) {
	XtFree($old$label);
	$label = XtNewString($label);
	need_redraw = True;
    }
    if ($font             != $old$font
    ||  $foreground       != $old$foreground
    ||  $background_pixel != $old$background_pixel) {
	make_textgc($);
	if ($label != NULL)
	    need_redraw = True;
    }
    if ($label              != $old$label
    ||  $font               != $old$font
    ||  $alignment          != $old$alignment
    ||  $labelOffset        != $old$labelOffset
    ||  $highlightThickness != $old$highlightThickness) {
	compute_label_size($);
	need_redraw = True;
    }

    /* adjust board abs variables
    if ($width != $old$width)
	$abs_width = $width;
    if ($height != $old$height)
	$abs_height = $height;
    */

    return need_redraw;
}

@ The |expose| method first calls the |expose| method of its
superclass -- which basically just draws the frame -- and then adds
the label to it.

@proc _expose
{
    Position x, y; Dimension w, h;

    if (! XtIsRealized($)) return;

    $compute_inside($, &x, &y, &w, &h);
    if (region != NULL) {
	XSetRegion(XtDisplay($), $bordergc, region);
        XSetRegion(XtDisplay($), $lightgc, region);
        XSetRegion(XtDisplay($), $darkgc, region);
        XSetRegion(XtDisplay($), $backgroundgc, region);
        XSetRegion(XtDisplay($), $sunkengc, region);
        XSetRegion(XtDisplay($), $textgc, region);
    }
    /* draw highlight border */
    if ($traversal_focus)
	$highlight_border($);
    /* draw frame */
    XfwfDrawFrame($, x - $frameWidth - $innerHOffset,     y - $frameWidth - $innerVOffset,
		     w + 2*$innerHOffset + 2*$frameWidth, h + 2*$innerVOffset + 2*$frameWidth,
		     $frameType, $frameWidth, $lightgc, $darkgc);
    /* draw label */
    if ($label) {
	switch ($alignment) {
	case XfwfTop:     x += $labelOffset; y = $font->ascent; break;
	case XfwfTopLeft: x = 0; y += $font->ascent; break;
	case XfwfLeft:    x = 0; y += (h-$labelHeight) / 2 + $font->ascent;
	}
	XDrawImageString(XtDisplay($), XtWindow($), $textgc, x, y, $label, strlen($label));
    }
    if (region != NULL) {
	XSetClipMask(XtDisplay($), $bordergc, None);
        XSetClipMask(XtDisplay($), $lightgc, None);
        XSetClipMask(XtDisplay($), $darkgc, None);
        XSetClipMask(XtDisplay($), $backgroundgc, None);
        XSetClipMask(XtDisplay($), $sunkengc, None);
        XSetClipMask(XtDisplay($), $textgc, None);
    }
}

@ The |resize| method passes on the resize message to its child, after
decreasing the area by the amount needed for the frame.

@proc resize
{
    Position  x, y;
    Dimension w, h;
    Widget child;

    if ($num_children == 0) return;
    $compute_inside($, &x, &y, &w, &h);
    child = $children[0];

    /* compute own needs */
    w -= 2 * child->core.border_width;
    h -= 2 * child->core.border_width;
    XtConfigureWidget(child, x, y, max(w,1), max(h,1), child->core.border_width);
}

@ The |insert_child| method is called, when a child is inserted
in the |children| list. If |shrinkToFit| is true the enforcer
widget has to be resized to fit around the frame.

@proc insert_child
{
    #insert_child(child);
    if (child == $children[0] && $shrinkToFit) {
	Position  x, y;
	Dimension w, h;
	/* compute own needs */
	$compute_inside($, &x, &y, &w, &h);
	w = $width  - w;
	h = $height - h;
	/* add child needs */
	w += ($label && $alignment == XfwfTop ?
		max($labelWidth, child->core.width) :
		child->core.width)
	     + 2*child->core.border_width;
	h += child->core.height + 2*child->core.border_width;
	/* set size */
	XtVaSetValues($, XtNwidth, w, XtNheight, h, NULL);
    }
}

@ The |change_managed| method is called when a child becomes managed
or unmanaged. The task of the routine is enforcing the layout policy,
which in this case consists of resizing the child to fit inside the
frame, or the frame around the child.

@proc change_managed
{
    Widget child;
    Position x, y; Dimension w, h;

    if ($num_children == 0) return;
    $compute_inside($, &x, &y, &w, &h);
    child = $children[0];

    if ($shrinkToFit) {
	Dimension selfw, selfh;
	/* compute own needs */
	selfw = $width  - w;
	selfh = $height - h;
	/* add child needs */
	selfw += ($label && $alignment == XfwfTop ?
		   max($labelWidth, child->core.width) :
		   child->core.width)
		 + 2*child->core.border_width;
	selfh += child->core.height + 2*child->core.border_width;
	/* set size */
	XtVaSetValues($, XtNwidth, selfw, XtNheight, selfh, NULL);
	$compute_inside($, &x, &y, &w, &h);
    }
    w -= 2 * child->core.border_width;
    h -= 2 * child->core.border_width;
    XtConfigureWidget(child, x, y, max(w,1), max(h,1), child->core.border_width);
}

@ If a child requests to be resized, the request is always ignored, or if 
|shrinkToFit| is TRUE, the enforcer resizes to fit.

@proc geometry_manager
{
    Position  x, y;
    Dimension w, h, extraw, extrah;

    if ($shrinkToFit) {
	$compute_inside($, &x, &y, &w, &h);
	/* compute new size of Enforcer widget */
	extraw = $width  - w;
	extrah = $height - h;
	w      = (request->request_mode & CWWidth)  ? request->width  + extraw : $width;
	h      = (request->request_mode & CWHeight) ? request->height + extrah : $height;
	/* query parent to resize this widget */
	XtMakeResizeRequest($, w, h, NULL, NULL);
	/* resize child */
	$compute_inside($, &x, &y, &w, &h);
	w -= 2 * child->core.border_width;
	h -= 2 * child->core.border_width;
	XtConfigureWidget(child, x, y, max(w,1), max(h,1), child->core.border_width);
	/* done */
	return XtGeometryDone;
    }
    return XtGeometryNo;
}

@ The method |compute_inside| is re-defined. The method now leaves
place for the label.

@proc compute_inside
{
    #compute_inside($, x, y, w, h);
    /* change sizes to have enough space for the label */
    if ($label) {
	switch ($alignment) {
	case XfwfTop:
	    *y += $labelHeight;
	    *h -= $labelHeight;
	    break;
	case XfwfLeft:
	case XfwfTopLeft:
	    *x += $labelWidth;
	    *w -= $labelWidth;
	    break;
	}
    }
}

@ The highlight and unhighlight methods have to be overriden to skip the
label.

@proc highlight_border
{
    XRectangle  rect[4];
    Position    x, y;
    Dimension   w, h;

    if ($highlightThickness == 0) return;

    x = $outerOffset + (($label && $alignment != XfwfTop) ? $labelWidth  : 0);
    y = $outerOffset + (($label && $alignment == XfwfTop) ? $labelHeight : 0);
    w = $width  - x - $outerOffset;
    h = $height - y - $outerOffset;

    rect[0].x = x;
    rect[0].y = y;
    rect[0].width = w;
    rect[0].height = $highlightThickness;

    rect[1].x = x;
    rect[1].y = y;
    rect[1].width = $highlightThickness;
    rect[1].height = h;

    rect[2].x = $width - $highlightThickness - $outerOffset;
    rect[2].y = y;
    rect[2].width = $highlightThickness;
    rect[2].height = h;

    rect[3].x = x;
    rect[3].y = $height - $highlightThickness - $outerOffset;
    rect[3].width = w;
    rect[3].height = $highlightThickness;

    XFillRectangles(XtDisplay($), XtWindow($), $bordergc, &rect[0], 4);
}

@proc unhighlight_border
{
    Position   x, y;
    Dimension  w, h;

    if ($highlightThickness == 0) return;

    x = $outerOffset + (($label && $alignment != XfwfTop) ? $labelWidth  : 0);
    y = $outerOffset + (($label && $alignment == XfwfTop) ? $labelHeight : 0);
    w = $width  - x - $outerOffset;
    h = $height - y - $outerOffset;

    XClearArea(XtDisplay($), XtWindow($), 
               x, y, w, $highlightThickness, False);
    XClearArea(XtDisplay($), XtWindow($),
               x, y, $highlightThickness, h, False);
    XClearArea(XtDisplay($), XtWindow($),
               $width - $highlightThickness - $outerOffset, y, 
               $highlightThickness, h, False);
    XClearArea(XtDisplay($), XtWindow($),
               x, $height - $highlightThickness - $outerOffset,
               w, $highlightThickness, False);
}

@ A Enforcer widget (and most subclasses) return |True| for
|would_accept_focus|, if the |sensitive|, |visible| and |traversalOn|
resources are set and it has no children. If it has a child the value
of the child is returned.

@proc would_accept_focus
{
    if (! XtIsRealized($) || ! $sensitive || ! $visible || ! $traversalOn)
        return False;
    else if ($num_children > 0) {
	$ = $children[0];
	if (XtIsSubclass($, xfwfCommonWidgetClass)) {
	    if (! XtIsRealized($) || ! $sensitive || ! $visible || ! $traversalOn)
		/* the enforcer will do the traversal job */
		return True;
	    else 
		/* ask if the child wants to do the traversal job */
		return $would_accept_focus($);
	}
    }
    return True;
}

@utilities

@ The |compute_label_size| routine computes width and height of label.

@proc compute_label_size($)
{
    int direction, ascent, descent;
    XCharStruct overall;

    if ($alignment != XfwfLeft && $alignment != XfwfTopLeft)
	$alignment = XfwfTop; /* secure valid alignment */

    if ($label) {
	XTextExtents($font, $label, strlen($label),
		     &direction, &ascent, &descent, &overall);
	if ($alignment == XfwfTop) {
	    $labelWidth  = overall.width + $labelOffset;
	    $labelHeight = ascent + descent + $labelOffset + $highlightThickness;
	} else {
	    $labelWidth  = overall.width + $labelOffset + $highlightThickness;
	    $labelHeight = ascent + descent + $labelOffset;
	}
    } else {
	$labelWidth = $labelHeight = 0;
    }
}

@ The |make_textgc| routine creates the GC for the text. 

@proc make_textgc($)
{
    XGCValues values;

    if ($textgc != NULL) XtReleaseGC($, $textgc);
    values.background = $background_pixel;
    values.foreground = $foreground;
    values.font       = $font->fid;
    $textgc = XtGetGC($, GCFont | GCBackground | GCForeground, &values);
}

@IMPORTS

@incl <stdio.h>
@incl <string.h>
@incl <X11/keysym.h>
