# File "display.tcl":
# Main module to draw malaga analysis trees, variables and results.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de 
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# global variables ------------------------------------------------------------

# $char_set - used character set; is "hangul" or "latin1"
# $font_list -- list of available font heights in pixels
# $font($size) -- font description for each $size in $font_list
# $menu_font -- menu font

#------------------------------------------------------------------------------

proc create_font_menu {kind} {

  global font font_list menu_font font_size
  
  # Font size selection
  menubutton .$kind.menu.font \
  -text " Font size " \
  -font $menu_font \
  -menu .$kind.menu.font.radio
  
  menu .$kind.menu.font.radio
  
  foreach size $font_list {
    .$kind.menu.font.radio add radio \
    -label " $size point " \
    -font $menu_font \
    -variable font_size($kind) \
    -value $size \
    -command "set font($kind) $font($size); ${kind}_redraw new_fontsize"
  }
  pack .$kind.menu.font -side left
}

#------------------------------------------------------------------------------

proc create_window_menu {kind} {
  
  global menu_font
  
  # Window operations:
  menubutton .$kind.menu.window \
  -text " Window " \
  -font $menu_font \
  -menu .$kind.menu.window.m
  menu .$kind.menu.window.m

  # Export tree as postscript file
  .$kind.menu.window.m add command \
  -label " Export Postscript... " \
  -font $menu_font \
  -command "export_postscript .$kind.frame.canvas $kind.ps" \
  -accelerator "Control+e "
  bind .$kind <Control-e> "export_postscript .$kind.frame.canvas $kind.ps"

  # Close:
  .$kind.menu.window.m add command \
  -label " Close " \
  -font $menu_font \
  -command "${kind}_close_window" \
  -accelerator "Control+c "
  bind .$kind <Control-c> "${kind}_close_window"

  pack .$kind.menu.window -side left
}

#------------------------------------------------------------------------------

proc create_result_menu {kind} {
  
  global menu_font
  
  set menu .$kind.menu
  menubutton $menu.result \
  -text " Result " \
  -font $menu_font \
  -menu $menu.result.m
  menu $menu.result.m
  
  foreach command_entry {
    {"path_display_first_result" " First result " f}
    {"path_display_previous_result" " Previous result " p}
    {"path_display_next_result" " Next result " n}
    {"path_display_last_result" " Last result " l} 
  } {
    $menu.result.m add command \
    -label [lindex $command_entry 1] \
    -font $menu_font \
    -command [lindex $command_entry 0] \
    -accelerator Control+[lindex $command_entry 2]
  }

  bind .$kind <Control-f> {path_display_first_result}
  bind .$kind <Control-l> {path_display_last_result}
  bind .$kind <Control-n> {path_display_next_result}
  bind .$kind <Control-p> {path_display_previous_result}

  pack .$kind.menu.result -side left
}

#------------------------------------------------------------------------------

proc create_scroll_frame {kind} {

  set frame .$kind.frame
  
  # Define frame
  frame $frame -relief raised -bd 2
  
  # Define canvas
  canvas $frame.canvas -scrollregion {0c 0c 10c 20c} \
  -xscrollcommand "$frame.hscroll set" \
  -yscrollcommand "$frame.vscroll set"
  
  # Define scrollbars
  scrollbar $frame.vscroll -relief sunken -command "$frame.canvas yview"
  scrollbar $frame.hscroll \
  -orient horiz \
  -relief sunken \
  -command "$frame.canvas xview"
  
  pack $frame.vscroll -side right -fill y
  pack $frame.hscroll -side bottom -fill x
  pack $frame.canvas -expand yes -fill both
  pack $frame -side top -expand yes -fill both
  
  # Define scroll possibility by pressing of the arrow keys:
  bind $frame.canvas <Up> "$frame.canvas yview scroll -1 unit"
  bind $frame.canvas <Down> "$frame.canvas yview scroll 1 unit"
  bind $frame.canvas <Left> "$frame.canvas xview scroll -1 unit"
  bind $frame.canvas <Right> "$frame.canvas xview scroll 1 unit"
  bind $frame.canvas <Shift-Up> "$frame.canvas yview scroll -1 page"
  bind $frame.canvas <Shift-Down> "$frame.canvas yview scroll 1 page"
  bind $frame.canvas <Shift-Left> "$frame.canvas xview scroll -1 page"
  bind $frame.canvas <Shift-Right> "$frame.canvas xview scroll 1 page"
}

#------------------------------------------------------------------------------

proc display_read {} {
  switch [read_line stdin] {
    "tree" {tree_redraw reload}
    "result" {result_redraw reload}
    "variables" {variables_redraw reload}
    "allomorph" {allomorph_redraw reload}
  }
}

#------------------------------------------------------------------------------

proc display_init {} {

  global font_list font menu_font
  
  set font(8) -*-helvetica-medium-r-normal-*-8-80-*-*-*-*-iso8859-1
  set font(10) -*-helvetica-medium-r-normal-*-10-100-*-*-*-*-iso8859-1
  set font(12) -*-helvetica-medium-r-normal-*-12-120-*-*-*-*-iso8859-1
  set font(14) -*-helvetica-medium-r-normal-*-14-140-*-*-*-*-iso8859-1
  set font(18) -*-helvetica-medium-r-normal-*-18-180-*-*-*-*-iso8859-1
  set font_list {8 10 12 14 18}
  set menu_font -*-helvetica-medium-r-normal-*-17-*-*-*-*-*-iso8859-1
}

#------------------------------------------------------------------------------

# Load matrix package.
load "$env(MALAGA)/tcl/libmatrix[info sharedlibextension]" 

lappend auto_path "$env(MALAGA)/tcl"
set char_set [read_line stdin]
display_init
tree_init
variables_init
result_init
allomorph_init
wm withdraw .
fileevent stdin readable {display_read}

#------------------------------------------------------------------------------
