/* File "pools.h":
 * A new data type, "pool_t", for growing vectors of items 
 * of an arbitrary type. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* the abstract data type <pool_t> */
typedef struct POOL_T *pool_t;

/*---------------------------------------------------------------------------*/

extern pool_t new_pool (long_t item_size);
/* Create a new pool that records items of size <item_size>. */

extern void free_pool (pool_t pool);
/* Free all memory used by <pool>. */

extern void clear_pool (pool_t pool);
/* Clear <pool> (do not free any memory used by the pool). */

extern void *pool_to_vector (pool_t pool);
/* Return <pool> as a C vector (contiguous memory).
 * The vector can be freed with "free" after use. */

extern void write_pool (pool_t pool, FILE *stream, string_t file_name);
/* Write <pool> to <stream> (<file_name> is needed for error messages). */

extern void *get_pool_space (pool_t pool, long_t num_items, long_t *index);
/* Get space for <num_items> contiguous items in pool <pool>,
 * return its address as the function's result and the index in *<index>,
 * if <index> != NULL. */

extern long_t pool_items (pool_t pool);
/* Return the number of the items in <pool>. */

extern void *pool_item (pool_t pool, long_t index);
/* Return the address of item with <index> in pool <pool>,
 * or NULL if there is no such item. */

extern void *copy_to_pool (pool_t pool, 
			   void *ptr, 
			   long_t num_items, 
			   long_t *index);
/* Copy the vector *<ptr> of length <num_items> into <pool>.
 * The items of *<ptr> must be of same size as the items in <pool>.
 * Return the address of the copy as the function's result and the
 * index in *<index>, if <index> != NULL. */

extern string_t copy_string_to_pool (pool_t pool, 
				     string_t string, 
				     long_t *index);
/* Copy the string <string> into <pool>, which must be a string pool.
 * Return the copy of the string as the function's result as well as
 * its index in *<index>, if <index> != NULL. */
