/* File "malaga_lib.c":
 * Options and functions for starting and terminating malaga. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <setjmp.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "input.h"
#include "commands.h"
#include "options.h"
#include "malaga_lib.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rules.h"
#include "files.h"
#include "analysis.h"
#include "cache.h"
#include "symbols.h"
#include "lex_type.h"
#include "lexicon.h"
#include "transmit_process.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL
#include "malaga_lib.h"

/*---------------------------------------------------------------------------*/

LOCAL void do_pruning (string_t arguments)
/* Enable/disable pruning_rule. */
{
  if (*arguments == EOS) 
    printf ("pruning: %s\n", 
	    get_analysis_option (PRUNING_OPTION) ? "yes" : "no");
  else
    set_analysis_option (PRUNING_OPTION, parse_yes_no (&arguments));

  parse_end (arguments);
}

LOCAL command_t pruning_option =
{
  "pruning p", do_pruning,
  "Arguments:\n"
  "  yes -- turn pruning_rule on\n"
  "  no -- turn pruning_rule off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_robust (string_t arguments)
/* Enable/disable robust parsing. */
{
  if (*arguments == EOS)
    printf ("robust: %s\n", 
	    get_analysis_option (ROBUST_OPTION) ? "yes" : "no");
  else
    set_analysis_option (ROBUST_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}

LOCAL command_t robust_option =
{
  "robust", do_robust,
  "Arguments:\n"
  "  yes -- turn robust_rule on\n"
  "  no -- turn robust_rule off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_mor_filter (string_t arguments)
/* Enable/disable morphology output filter. */
{
  if (*arguments == EOS)
    printf ("mor-filter: %s\n", 
	    get_analysis_option (MOR_FILTER_OPTION) ? "yes" : "no");
  else
    set_analysis_option (MOR_FILTER_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}

LOCAL command_t mor_filter_option =
{
  "mor-filter mfil", do_mor_filter,
  "Arguments:\n"
  "  yes -- turn morphology output filter on\n"
  "  no -- turn morphology output filter off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_syn_filter (string_t arguments)
/* Enable/disable syntax output filter. */
{
  if (*arguments == EOS)
    printf ("syn-filter: %s\n", 
	    get_analysis_option (SYN_FILTER_OPTION) ? "yes" : "no");
  else
    set_analysis_option (SYN_FILTER_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}

LOCAL command_t syn_filter_option =
{
  "syn-filter sfil", do_syn_filter,
  "Arguments:\n"
  "  yes -- turn syntax output filter on\n"
  "  no -- turn syntax output filter off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_input_filter (string_t arguments)
/* Enable/disable syntax input filter. */
{
  if (*arguments == EOS)
    printf ("input-filter: %s\n", 
	    get_analysis_option (INPUT_FILTER_OPTION) ? "yes" : "no");
  else
    set_analysis_option (INPUT_FILTER_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}

LOCAL command_t input_filter_option =
{
  "input-filter ifil", do_input_filter,
  "Arguments:\n"
  "  yes -- turn syntax input filter on\n"
  "  no -- turn syntax input filter off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_cache_size (string_t arguments)
/* Change the cache size. */
{
  if (*arguments == EOS)
    printf ("cache-size: %ld (%ld used)\n", 
	    get_cache_size (TRUE), get_cache_size (FALSE));
  else
  {
    long_t size = parse_integer (&arguments);

    if (size < 0)
      error ("don't use a negative cache size");

    set_cache_size (size);
    set_analysis_option (CACHE_OPTION, (size > 0));
  }
  parse_end (arguments);
}

LOCAL command_t cache_size_option =
{
  "cache-size", do_cache_size,
  "Arguments: <size> -- the new cache size\n"
  "If <size> == 0, cache is switched off.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_tree_option (string_t arguments)
/* Determine if tree is shown after analysis. */
{
  if (*arguments == EOS)
    printf ("tree: %s\n", show_tree ? "yes" : "no");
  else
    show_tree = parse_yes_no (&arguments);
  parse_end (arguments);
}

LOCAL command_t tree_option =
{
  "tree", do_tree_option,
  "Arguments:\n"
  "  yes -- show tree after analysis\n"
  "  no -- don't show tree after analysis\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_output_format (string_t arguments)
/* Change analysis output format to "arguments". */
{
  if (*arguments == EOS)
  {
    string_t format = new_string_readable (output_format, NULL);
    printf ("output-format: \"%s\"\n", format);
    free (format);
  }
  else
  {
    string_t new_format = parse_word (&arguments);
    free (output_format);
    output_format = new_format;
  }

  parse_end (arguments);
}

LOCAL command_t output_format_option =
{
  "output-format", do_output_format,
  "Describe the format in which analysis results will be printed.\n"
  "Arguments: <output_format_string>\n"
  "The output format string may contain the following special sequences:\n"
  "  %s -- surface\n"
  "  %l -- input line number\n"
  "  %c -- result category\n"
  "  %n -- number of analysis states\n"
  "  %r -- the ambiguity index\n"
};

/*---------------------------------------------------------------------------*/

/* Change unknown analysis format to "arguments". */

LOCAL void do_unknown_format (string_t arguments)
{
  if (*arguments == EOS)
  {
    string_t format = new_string_readable (unknown_format, NULL);
    printf ("unknown-format: \"%s\"\n", format);
    free (format);
  }
  else
  {
    string_t new_format = parse_word (&arguments);
    free (unknown_format);
    unknown_format = new_format;
  }
  
  parse_end (arguments);
}

LOCAL command_t unknown_format_option =
{
  "unknown-format", do_unknown_format,
  "Describe the format in which unknown forms will be printed.\n"
  "Arguments: <unknown_format_string>\n"
  "The unknown format string may contain the following special sequences:\n"
  "  %s -- surface\n"
  "  %l -- input line number\n"
  "  %n -- number of analysis states\n"
};

/*===========================================================================*/

LOCAL string_t morphology_file, syntax_file, lexicon_file;
LOCAL string_t symbol_file, extended_symbol_file;

/*---------------------------------------------------------------------------*/

/* Read the project file. */

LOCAL void read_project_file (string_t project_file)
{
  FILE *project_stream;
  
  project_stream = fopen_save (project_file, "r");
  while (! feof (project_stream)) 
  {
    char project_line[INPUT_LINE_LENGTH];
    string_t project_line_ptr, argument, extension;
    string_t *file_name;
    
    read_line (project_stream, project_line, INPUT_LINE_LENGTH);
    project_line_ptr = project_line;
    
    if (*project_line_ptr == EOS)
      continue;
    
    argument = parse_word (&project_line_ptr);
    
    extension = NULL;
    file_name = NULL;
    if (strcmp_no_case (argument, "sym:") == 0) 
    {
      file_name = &symbol_file;
      extension = "sym_c";
    } 
    if (strcmp_no_case (argument, "esym:") == 0) 
    {
      file_name = &extended_symbol_file;
      extension = "esym_c";
    } 
    else if (strcmp_no_case (argument, "lex:") == 0) 
    {
      file_name = &lexicon_file;
      extension = "lex_c";
    } 
    else if (strcmp_no_case (argument, "mor:") == 0) 
    {
      file_name = &morphology_file;
      extension = "mor_c";
    }
    else if (strcmp_no_case (argument, "syn:") == 0)
    {
      file_name = &syntax_file;
      extension = "syn_c";
    }
    else if (strcmp_no_case (argument, "include:") == 0)
    {
      string_t include_file = parse_word (&project_line_ptr);
      string_t include_path = new_string (absolute_path (include_file,
							 project_file));
      
      parse_end (project_line_ptr);
      read_project_file (include_path);
      free (include_path);
      free (include_file);
    }
    else if (strcmp_no_case (argument, "morinfo:") == 0
	     || strcmp_no_case (argument, "syninfo:") == 0)
    {
      grammar_t grammar = ((strcmp_no_case (argument, "morinfo:") == 0) ?
			   MORPHOLOGY : SYNTAX);
      
      if (info[grammar] == NULL)
	info[grammar] = concat_strings (project_line_ptr, "\n", NULL);
      else
      {
	string_t old_info = info[grammar];
	
	info[grammar] = concat_strings (info[grammar], project_line_ptr, "\n",
					NULL);
	free (old_info);
      }
    }
    
    free (argument);
    
    if (file_name != NULL && *file_name == NULL && *project_line_ptr != EOS) 
    {
      argument = parse_word (&project_line_ptr);
      set_file_name (file_name, absolute_path (argument, project_file), 
		     extension);
      free (argument);
    }
  }
  fclose_save (project_stream, project_file);
}

/*---------------------------------------------------------------------------*/

LOCAL command_t *malaga_options[] = 
/* the commands that can be called on startup, in alphabetical order */
{
  &alias_option, &cache_size_option,
#ifdef HANGUL
  &hangul_option,
#endif
  &heap_size_option, &hidden_option, &input_filter_option, &mor_filter_option,
  &output_option, &output_format_option, &pruning_option, &result_option,
  &robust_option, &sort_records_option, &switch_option, &syn_filter_option,
  &tree_option, &unknown_format_option, &variables_option,
  NULL
};

/*---------------------------------------------------------------------------*/

GLOBAL void init_malaga (string_t project_file)
/* Initialise this module. */
{
#ifdef HANGUL
  init_hangul ();
#endif

  if (! has_extension (project_file, "pro"))
    error ("project file \"%s\" must have extension \"pro\"", project_file);

  read_project_file (project_file);

  if (morphology_file == NULL)
    error ("missing morphology rules");

  if (lexicon_file == NULL)
    error ("missing lexicon");

  if (symbol_file == NULL)
    error ("missing symbol file");

  if (extended_symbol_file != NULL)
    read_symbol_file (extended_symbol_file);
  else
    read_symbol_file (symbol_file);

  read_lexicon_file (lexicon_file);

  transmit_function = transmit;

  init_analysis (morphology_file, syntax_file);

  output_format = new_string ("%l: \"%s\": %c");
  unknown_format = new_string ("%l: \"%s\": unknown");
  show_output = TRUE;
  show_tree = show_result = FALSE;

  options = malaga_options;
  execute_set_commands (project_file, "malaga:");
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_malaga (void)
/* Terminate this module. */
{
  free (unknown_format);
  free (output_format);

  terminate_analysis ();

  free (syntax_file);
  syntax_file = NULL;

  free (morphology_file);
  morphology_file = NULL;

  stop_transmit_process ();

  free_lexicon ();
  free (lexicon_file);
  lexicon_file = NULL;

  free_symbol_table ();

  free (extended_symbol_file);
  extended_symbol_file = NULL;

  free (symbol_file);
  symbol_file = NULL;

  free_value_heap ();

  free (info[MORPHOLOGY]);
  info[MORPHOLOGY] = NULL;

  free (info[SYNTAX]);
  info[SYNTAX] = NULL;

#ifdef HANGUL
  terminate_hangul ();
#endif
}
