///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////
#include "config.h"

#include "heap.h"
#include "processheap.h"
#include "superblock.h"

static const char version[] = "The Hoard memory allocator, version 1.5.2 (http://www.hoard.org). Copyright (C) 1998, 1999, 2000 The University of Texas at Austin. $Id: heap.cpp,v 1.39 2000/02/24 03:13:44 emery Exp $";

// NB: Use maketable.cpp to update this
//     if SIZE_CLASSES, ALIGNMENT, SIZE_CLASS_BASE, MAX_EMPTY_SUPERBLOCKS,
//     or SUPERBLOCK_SIZE changes.

size_t hoardHeap::_sizeTable[hoardHeap::SIZE_CLASSES] = {8UL, 16UL, 24UL, 32UL, 40UL, 48UL, 56UL, 72UL, 80UL, 96UL, 120UL, 144UL, 168UL, 200UL, 240UL, 288UL, 344UL, 416UL, 496UL, 592UL, 712UL, 856UL, 1024UL, 1232UL, 1472UL, 1768UL, 2120UL, 2544UL, 3048UL, 3664UL, 4392UL, 5272UL, 6320UL, 7584UL, 9104UL, 10928UL, 13112UL, 15728UL, 18872UL, 22648UL, 27176UL, 32616UL, 39136UL, 46960UL, 56352UL, 67624UL, 81144UL, 97376UL, 116848UL, 140216UL, 168256UL, 201904UL, 242288UL, 290744UL, 348896UL, 418672UL, 502408UL, 602888UL, 723464UL, 868152UL, 1041784UL, 1250136UL, 1500160UL, 1800192UL, 2160232UL, 2592280UL, 3110736UL, 3732880UL, 4479456UL, 5375344UL, 6450408UL, 7740496UL, 9288592UL, 11146312UL, 13375568UL, 16050680UL, 19260816UL, 23112984UL, 27735576UL, 33282688UL, 39939224UL, 47927072UL, 57512488UL, 69014984UL, 82817976UL, 99381576UL, 119257888UL, 143109472UL, 171731360UL, 206077632UL, 247293152UL, 296751776UL, 356102144UL, 427322560UL, 512787072UL, 615344512UL, 738413376UL, 886096064UL, 1063315264UL}; 

size_t hoardHeap::_threshold[hoardHeap::SIZE_CLASSES] = {2048UL, 1024UL, 682UL, 512UL, 408UL, 340UL, 292UL, 226UL, 204UL, 170UL, 136UL, 112UL, 96UL, 80UL, 68UL, 56UL, 46UL, 38UL, 32UL, 26UL, 22UL, 18UL, 16UL, 12UL, 10UL, 8UL, 6UL, 6UL, 4UL, 4UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL, 2UL}; 


hoardHeap::hoardHeap (void)
  : _index (0)
#if HEAP_DEBUG
  , _magic (HEAP_MAGIC)
#endif
{
  // Initialize the per-heap lock.
  hoardLockInit (_lock);
  for (int i = 0; i < SUPERBLOCK_FULLNESS_GROUP; i++) {
    for (int j = 0; j < SIZE_CLASSES; j++) {
      // Initialize all superblocks lists to empty.
      _superblocks[i][j] = NULL;
    }
  }
  for (int k = 0; k < SIZE_CLASSES; k++) {
    _leastEmptyBin[k] = 0;
  }
}


void hoardHeap::insertSuperblock (int sizeclass,
				  superblock * sb)
{
  assert (sb->isValid());
  assert ((sizeclass == 0)
	  || (sb->getBlockSizeClass() == sizeclass));
  assert (_magic == HEAP_MAGIC);

  // Insert the superblock at the head of the list for this sizeclass.

  sb->setOwner (this);

  // How full is this superblock?
  int fullness = sb->getFullness();

  sb->insertBefore (_superblocks[fullness][sizeclass]);
  _superblocks[fullness][sizeclass] = sb;
  assert (_superblocks[fullness][sizeclass]->isValid());

  // Reset the least-empty bin counter.
  _leastEmptyBin[sizeclass] = SUPERBLOCK_FULLNESS_GROUP-2;
}


superblock * hoardHeap::removeMaxSuperblock (int sizeclass)
{
  assert (_magic == HEAP_MAGIC);

  // Instead of finding the superblock with the most available space
  // (something that would either involve a linear scan through the
  // superblocks or maintaining the superblocks in sorted order), we
  // just pick one that is no more than
  // 1/(SUPERBLOCK_FULLNESS_GROUP-1) more full than the superblock
  // with the most available space.  We start with the emptiest group.

  int i = 0;
  superblock * head = NULL;

  // Note: the last group (SUPERBLOCK_FULLNESS_GROUP - 1) is full, so
  // we never need to check it. But for robustness, we leave it in.
  while (i < SUPERBLOCK_FULLNESS_GROUP) {
    head = _superblocks[i][sizeclass];
    if (head) {
      break;
    }
    i++;
  }

  if (!head) {
    return NULL;
  }

  removeSuperblock (head, sizeclass);

  assert (head->isValid());
  return head;
}


void hoardHeap::removeSuperblock (superblock * sb,
				  int sizeclass)
{
  assert (_magic == HEAP_MAGIC);

  assert (sb->isValid());
  assert (sb->getOwner() == this);
  assert ((sizeclass == 0)
	  || (sb->getBlockSizeClass() == sizeclass));

  for (int i = 0; i < SUPERBLOCK_FULLNESS_GROUP; i++) {
    if (sb == _superblocks[i][sizeclass]) {
      _superblocks[i][sizeclass] = sb->getNext();
      if (_superblocks[i][sizeclass] != NULL) {
	assert (_superblocks[i][sizeclass]->isValid());
      }
      break;
    }
  }

  sb->remove();
  sb->setOwner (NULL);
}


void hoardHeap::moveSuperblock (superblock * sb,
				int sizeclass,
				int fromBin,
				int toBin)
{
  assert (_magic == HEAP_MAGIC);
  assert (sb->isValid());
  assert (sb->getOwner() == this);
  assert ((sizeclass == 0)
	  || (sb->getBlockSizeClass() == sizeclass));
  assert (sb->getFullness() == toBin);

  // Remove the superblock from the old bin.

  superblock *& oldHead = _superblocks[fromBin][sizeclass];
  if (sb == oldHead) {
    oldHead = sb->getNext();
    if (oldHead != NULL) {
      assert (oldHead->isValid());
    }
  }

  sb->remove();

  // Insert the superblock into the new bin.

  superblock *& newHead = _superblocks[toBin][sizeclass];
  sb->insertBefore (newHead);
  newHead = sb;
  assert (newHead->isValid());

  // Reset the least-empty bin counter.
  _leastEmptyBin[sizeclass] = SUPERBLOCK_FULLNESS_GROUP-2;
}



void hoardHeap::freeBlock (block *& b,
			   superblock *& sb,
			   int sizeclass,
			   processHeap * processHeap)
{
  assert (this == sb->getOwner());

  const int oldFullness = sb->getFullness();
  sb->putBlock (b);
  const int newFullness = sb->getFullness();
  
  // If the fullness value has changed, move the superblock.
  if (newFullness != oldFullness) {
    moveSuperblock (sb, sizeclass, oldFullness, newFullness);
  }

  // If this is the process heap, see if we can recycle this superblock.
  // Then we're done.
  if (this == (hoardHeap *) processHeap) {
    // First, do a quick check to see if this superblock is in the
    // emptiest group.
    if (newFullness == 0) {
      // Now, check to see if this really is a candidate for
      // recycling.
      if ((sb->getNumBlocks() > 1) &&
	  (sb->getNumAvailable() == sb->getNumBlocks())) {
	// It's empty and recyclable.
	// Move it to the recycle bin.
	removeSuperblock (sb, sizeclass);
	processHeap->recycle (sb);
      }
    }
    return;
  }
  
  //
  // Release a superblock, if necessary.
  //

  // We ALWAYS release "big" superblocks.
  // A big superblock is one that contains only one block.
  if (sb->getNumBlocks() == 1) {

    // Remove this superblock's stats from the owner.
    decStats (sizeclass, 1, 1);

    removeSuperblock (sb, sizeclass);

#if defined(__sgi) || defined(__BEOS__)
    // Give the superblock back to the process heap.
    processHeap->lock ();
    processHeap->release (sb);
    processHeap->unlock ();
#else
    hoardUnsbrk (sb, hoardHeap::align (sizeof(superblock) + hoardHeap::align (sizeof(block) + hoardHeap::sizeFromClass (sb->getBlockSizeClass()))));
#endif
    return;
  }

  // Decrement U.
  decUStats (sizeclass);

  //
  // Check to see if the amount free exceeds the release threshold
  // (two superblocks worth of blocks for a given sizeclass) and if
  // the heap is sufficiently empty.

  int inUse, allocated;
  getStats (sizeclass, inUse, allocated);

  if ((inUse < (allocated - allocated / EMPTY_FRACTION))
      && (allocated > inUse + getReleaseThreshold(sizeclass))) {
    
    // We've crossed the magical threshold. Find the superblock with
    // the most free blocks and give it to the process heap.
    superblock * const maxSb = removeMaxSuperblock (sizeclass);
    assert (maxSb);

    // Update the statistics.
    
    int numBlocks = maxSb->getNumBlocks();
    assert (numBlocks >= maxSb->getNumAvailable());
    decStats (sizeclass,
		     numBlocks - maxSb->getNumAvailable(),
		     numBlocks);
    
    // Give the superblock back to the process heap.
    processHeap->lock ();
    processHeap->release (maxSb);
    processHeap->unlock ();
  }
}
