/********************************************************************************
*                                                                               *
*                            T a b l e   W i d g e t                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 1999 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* Contributions from: Pierre Cyr <pcyr@po-box.mcgill.ca>                        *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXTable.cpp,v 1.26 1999/10/29 13:41:59 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollWindow.h"
#include "FXTable.h"


/*
  Notes:

  We lust for the features below; basically we want to do everything Xbae
  can do, and then some...

  - Table looks like:
  
    +--------+--------+--------+--------+
    |        | ColHdr | ColHdr | ColHdr |
    +--------+--------+--------+--------+
    | RowHdr |    3.14|        |Pi      |
    +--------+--------+--------+--------+
    | RowHdr |        |        |        |
    +--------+--------+--------+--------+
    | RowHdr |        |        |        |
    +--------+--------+--------+--------+
    
  - Grid lines horizontal should be optional.
  - Grid lines vertictal should be optional also, independent of horizontal ones.
  - Grid line have different styles [Similar to frame styles]; normally dotted lines or light grey.
  - Cells have margins around the text.
  - Column headers are optional.
  - Row headers are optional.
  - Headers stay in place, i.e. column headers stay on top, row headers stay on right.
  - Cells can contain string or icon or number.
  - Justification and formatting [for numbers]:
  
      - Format same for whole table
      - Format same for column
      - Format same for row
      - Format different for each cell

  - Resizing columns [same for rows]:
    
      - Off, no resizing allowed.
      - Column bounds.
      - Adjustment of subsequent columns (proportional to old sizes).
      - Adjustment of prior columns (proportional to old sizes).
      - Adjustment of all columns (proportional to old sizes).
      - Adjustment of first/last column.
      - Uniform column width, or per-column defined width.
      
  - Selection:
  
      - Disabled.
      - Select rows only.
      - Select columns only.
      - Select rows and columns.
      - Select cells only.
      
  - Selection ranges:
  
      - Single entity (i.e. single row, column, etc.)
      - Range of entities (multiple rows, multiple columns, blocks)
      
  - Reordering:
  
      - Disabled.
      - Reordering of columns allowed.
      - Reordering of rows allowed.
      - Both.
      
  - Alternating colors:
      
      - All the same color
      - Alternate background/foreground every other row
      - Alternate background/foreground every other column
      
  - Header buttons:
  
      - Column header button to select whole column.
      - Row header button to select whole row.
      
  - Fixed columns or rows:
  
      - First n columns and last m columns. (e.g. to show totals).
      - First n rows and last m rows.
    
  - Grid line width should NOT be part of cell width/cell height.
  
  - Virtual storage capability for HUGE arrays:
  
    o When exposing virtual cells, we ask to supply content 
      for the exposed cells.
      
    o Keep track of part of table which is visible; this is
      the actual table.
      
    o The actual table keeps REAL cells for those virtual cells
      which are visible [so you can manipulate them, and for
      quick repainting].
      
    o When scrolling, we roll over the cells as follows:
    
      +---------+      +---------+      +---------+
      |XXXXXXXBB|      |BBXXXXXXX|      |DDCCCCCCC|
      |XXXXXXXBB|      |BBXXXXXXX|      |BBXXXXXXX|
      |XXXXXXXBB|  ->  |BBXXXXXXX|  ->  |BBXXXXXXX|
      |XXXXXXXBB|      |BBXXXXXXX|      |BBXXXXXXX|
      |CCCCCCCDD|      |DDCCCCCCC|      |BBXXXXXXX|
      +---------+      +---------+      +---------+
    
      Then of course we ask to refill the cells marked B, D, 
      and C.
      
    o When resizing, we resize the actual table, and ask to
      refill the cells on the bottom/right [or top/left, if
      we're at the maximum scrolled position and new cells are
      introduced at the top!]
      
    o Virtual cell from actual one:
    
       vr = ar+firstrow (0<=ar<visiblerows)
       vc = ac+firstcol (0<=ac<visiblecols)
       
    o Actual cell from virtual one:
    
       ar = vr-firstrow (firstrow<=vr<firstrow+visiblerows)
       ac = vc-firstcol (colstart<=vr<firstcol+visiblecols)
       
      In virtual mode, virtual cells outside the actual table should probably
      return NULL.
      
      Perhaps we can do it as follows:
      
      ar = (vr-firstrow)%visiblerows
      ac = (vc-firstcol)%visiblecols
      
      and just update nrows and ncols during scrolling.
    
    - Need cells which could span multiple rows/columns
*/


#define DEFAULTCOLUMNWIDTH  100     // Initial value for defColumnWidth
#define DEFAULTROWHEIGHT    20      // Initial value for defRowHeight
#define FUDGE               1

#define MODE_NONE           0       // Nothing going on
#define MODE_COLUMN         1       // Resizing column
#define MODE_ROW            2       // Resizing row

#define TABLE_MASK          (TABLE_HOR_GRIDLINES|TABLE_VER_GRIDLINES)


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXTableItem,FXObject,NULL,0)



// Draw dotted rectangle for focus
void FXTableItem::drawFocus(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setFillStyle(FILL_OPAQUESTIPPLED);
  dc.setStipple(STIPPLE_GRAY);
  dc.setForeground(table->getTextColor());
  dc.setBackground(table->getBackColor());
  dc.drawRectangle(x+1,y+1,w-3,h-3);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }


// Draw raised
void FXTableItem::drawRaised(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setForeground(table->getHiliteColor());
  dc.drawLine(x,y,x+w-2,y);
  dc.drawLine(x,y,x,y+h-2);
  dc.setForeground(table->getBaseColor());
  dc.drawLine(x+1,y+1,x+w-3,y+1);
  dc.drawLine(x+1,y+1,x+1,y+h-3);
  dc.setForeground(table->getShadowColor());
  dc.drawLine(x+1,y+h-2,x+w-2,y+h-2);
  dc.drawLine(x+w-2,y+h-2,x+w-2,y+1);
  dc.setForeground(table->getBorderColor());
  dc.drawLine(x,y+h-1,x+w-1,y+h-1);
  dc.drawLine(x+w-1,y,x+w-1,y+h-1);
  }


// Draw sunken
void FXTableItem::drawSunken(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setForeground(table->getShadowColor());
  dc.drawLine(x,y,x+w-1,y);
  dc.drawLine(x,y,x,y+h-1);
  dc.setForeground(table->getBorderColor());
  dc.drawLine(x+1,y+1,x+w-2,y+1);
  dc.drawLine(x+1,y+1,x+1,y+h-2);
  dc.setForeground(table->getHiliteColor());
  dc.drawLine(x+1,y+h-1,x+w-1,y+h-1);
  dc.drawLine(x+w-1,y+h-1,x+w-1,y+1);
  dc.setForeground(table->getBaseColor());
  dc.drawLine(x+2,y+h-2,x+w-2,y+h-2);
  dc.drawLine(x+w-2,y+2,x+w-2,y+h-2);
  }


// Draw item
void FXTableItem::draw(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  FXFont *font=table->getFont();
  FXint len=label.length();
  FXint hh=font->getFontHeight();
  FXint ww=font->getTextWidth(label.text(),len);
  FXint xx,yy;

  // Vertical placement
  if(state&FXTableItem::TOP) yy=y+table->getMarginTop();
  else if(state&FXTableItem::BOTTOM) yy=y+w-table->getMarginBottom()-hh;
  else yy=y+table->getMarginTop()+(h-table->getMarginBottom()-table->getMarginTop()-hh)/2;

  // Horizontal placement
  if(state&FXTableItem::LEFT) xx=x+table->getMarginLeft();
  else if(state&FXTableItem::RIGHT) xx=x+w-table->getMarginRight()-ww;
  else xx=x+table->getMarginLeft()+(w-table->getMarginLeft()-table->getMarginRight()-ww)/2;

  // Button style
  if(state&FXTableItem::BUTTON){
    dc.setForeground(table->getBaseColor());
    dc.fillRectangle(x+2,y+2,w-4,h-4);
    if(state&FXTableItem::PRESSED){
      drawSunken(table,dc,x,y,w,h);
      dc.setForeground(table->getTextColor());
      dc.drawText(xx+1,yy+font->getFontAscent()+1,label.text(),len);
      }
    else{
      drawRaised(table,dc,x,y,w,h);
      dc.setForeground(table->getTextColor());
      dc.drawText(xx,yy+font->getFontAscent(),label.text(),len);
      }
    }

  // Normal style
  else{
    if(state&FXTableItem::SELECTED){
      dc.setForeground(table->getSelBackColor());
      dc.fillRectangle(x,y,w,h);
      dc.setForeground(table->getSelTextColor());
      dc.drawText(xx,yy+font->getFontAscent(),label.text(),len);
      }
    else{
      dc.setForeground(table->getTextColor());
      dc.drawText(xx,yy+font->getFontAscent(),label.text(),len);
      }
    }

  // Got focus
  if(hasFocus()){
    drawFocus(table,dc,x,y,w,h);
    }
  }


// // Draw cell text range
// void FXTableItem::drawTextRange(FXString& string,FXint x,FXint y,FXint w,FXint h,FXuint mode,FXuint fm,FXuint to,FXuint si,FXuint ei,FXint scroll){
//   FXint sx,ex,xx,yy,cw,hh,ww;
//   FXDC *dc=DC();
//   if(to<=fm) return;
// 
//   // Height
//   hh=font->getFontHeight();
//   ww=font->getTextWidth(string.text(),string.length());
// 
//   // Clipped against cell interior
//   dc.setClipRectangle(x,y,w,h);
//   
//   // Set text color
//   dc.setForeground(textColor);
//   
//   // Vertical placement
//   if(mode&FXTableItem::TOP) yy=y+margintop;
//   else if(mode&FXTableItem::BOTTOM) yy=y+h-marginbottom-hh;
//   else yy=y+margintop+(h-marginbottom-margintop-hh)/2;
// 
//   // Horizontal placement
//   if(mode&FXTableItem::LEFT) xx=x+scroll+marginleft;
//   else xx=x+w+scroll-marginright-ww;
// 
//   // Nothing selected
//   if(to<=si || ei<=fm){
//     drawTextFragment(string,xx,yy,fm,to);
//     }
// 
//   // Stuff selected
//   else{
//     if(fm<si) drawTextFragment(string,xx,yy,fm,si); else si=fm;
//     if(ei<to) drawTextFragment(string,xx,yy,ei,to); else ei=to;
//     if(si<ei){
//       sx=xx+font->getTextWidth(string.text(),si);
//       ex=xx+font->getTextWidth(string.text(),ei);
//       if(hasFocus()){
//         dc.setForeground(selbackColor);
//         dc.fillRectangle(sx,y+margintop,ex-sx,h-marginbottom-margintop);
//         dc.setForeground(seltextColor);
//         drawTextFragment(string,xx,yy,si,ei);
//         }
//       else{
//         dc.setForeground(baseColor);
//         dc.fillRectangle(sx,y+margintop,ex-sx,h-marginbottom-margintop);
//         dc.setForeground(textColor);
//         drawTextFragment(string,xx,yy,si,ei);
//         }
//       }
//     }
//   }

// // Draw text fragment
// void FXTable::drawTextFragment(FXDCWindow& dc,FXString& string,FXint x,FXint y,FXuint fm,FXuint to){
//   x+=font->getTextWidth(string.text(),fm);
//   y+=font->getFontAscent();
//   dc.drawText(x,y,&string[fm],to-fm);
//   }



// Create icon
void FXTableItem::create(){ if(icon) icon->create(); }


// Destroy icon
void FXTableItem::destroy(){ if(icon) icon->destroy(); }


// Detach from icon resource
void FXTableItem::detach(){ if(icon) icon->detach(); }


// Get width of item
FXint FXTableItem::getWidth(const FXTable*) const {
  return 1;
  }


// Get height of item
FXint FXTableItem::getHeight(const FXTable*) const {
  return 1;
  }


// Save data
void FXTableItem::save(FXStream& store) const {
  FXObject::save(store);
  store << label;
  store << icon;
  store << state;
  }


// Load data
void FXTableItem::load(FXStream& store){ 
  FXObject::load(store);
  store >> label;
  store >> icon;
  store >> state;
  }


/*******************************************************************************/

// Map
FXDEFMAP(FXTable) FXTableMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXTable::onPaint),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_CARETBLINK,FXTable::onBlink),
  FXMAPFUNC(SEL_MOTION,0,FXTable::onMotion),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXTable::onAutoScroll),
  FXMAPFUNC(SEL_UNGRABBED,0,FXTable::onUngrabbed),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXTable::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXTable::onLeftBtnRelease),
  FXMAPFUNC(SEL_KEYPRESS,0,FXTable::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXTable::onKeyRelease),
  FXMAPFUNC(SEL_FOCUSIN,0,FXTable::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXTable::onFocusOut),
  FXMAPFUNC(SEL_CHANGED,0,FXTable::onChanged),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXTable::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXTable::onSelectionGained),
  FXMAPFUNC(SEL_ACTIVATE,0,FXTable::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXTable::onDeactivate),
  FXMAPFUNC(SEL_SELECTED,0,FXTable::onSelected),
  FXMAPFUNC(SEL_DESELECTED,0,FXTable::onDeselected),
  FXMAPFUNC(SEL_INSERTED,0,FXTable::onInserted),
  FXMAPFUNC(SEL_DELETED,0,FXTable::onDeleted),
  FXMAPFUNC(SEL_CLICKED,0,FXTable::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXTable::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXTable::onTripleClicked),
  FXMAPFUNC(SEL_COMMAND,0,FXTable::onCommand),
  FXMAPFUNC(SEL_UPDATE,FXTable::ID_HORZ_GRID,FXTable::onUpdHorzGrid),
  FXMAPFUNC(SEL_UPDATE,FXTable::ID_VERT_GRID,FXTable::onUpdVertGrid),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_HORZ_GRID,FXTable::onCmdHorzGrid),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_VERT_GRID,FXTable::onCmdVertGrid),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_DELETE_COLUMN,FXTable::onCmdDeleteColumn),
  FXMAPFUNC(SEL_UPDATE,FXTable::ID_DELETE_COLUMN,FXTable::onUpdDeleteColumn),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_DELETE_ROW,FXTable::onCmdDeleteRow),
  FXMAPFUNC(SEL_UPDATE,FXTable::ID_DELETE_ROW,FXTable::onUpdDeleteRow),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_INSERT_COLUMN,FXTable::onCmdInsertColumn),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_INSERT_ROW,FXTable::onCmdInsertRow),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_MOVE_RIGHT,FXTable::onCmdMoveRight),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_MOVE_LEFT,FXTable::onCmdMoveLeft),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_MOVE_UP,FXTable::onCmdMoveUp),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_MOVE_DOWN,FXTable::onCmdMoveDown),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_SELECT_COLUMN,FXTable::onCmdSelectColumn),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_SELECT_ROW,FXTable::onCmdSelectRow),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_SELECT_CELL,FXTable::onCmdSelectCell),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_SELECT_ALL,FXTable::onCmdSelectAll),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_DESELECT_ALL,FXTable::onCmdDeselectAll),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_MARK,FXTable::onCmdMark),
  FXMAPFUNC(SEL_COMMAND,FXTable::ID_EXTEND,FXTable::onCmdExtend),
  };


// Object implementation
FXIMPLEMENT(FXTable,FXScrollArea,FXTableMap,ARRAYNUMBER(FXTableMap))

  
// Serialization
FXTable::FXTable(){
  flags|=FLAG_ENABLED;
  cells=(FXTableItem**)-1;
  col_x=(FXint*)-1;
  row_y=(FXint*)-1;
  font=(FXFont*)-1;
  visiblerows=0;
  visiblecols=0;
  textColor=0;
  cellcursor=0;
  cellanchor=0;
  cellscroll=0;
  mode=MODE_NONE;
  blinker=NULL;
  }

  
// Build table
FXTable::FXTable(FXComposite *p,FXint nr,FXint nc,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXScrollArea(p,opts,x,y,w,h){
  register FXint i,j,t;
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  FXCALLOC(&cells,FXTableItem*,1);
  FXCALLOC(&col_x,FXint,1);
  FXCALLOC(&row_y,FXint,1);
  font=getApp()->normalFont;
  nrows=0;
  ncols=0;
  visiblerows=FXMAX(nr,0);
  visiblecols=FXMAX(nc,0);
  margintop=pt;
  marginbottom=pb;
  marginleft=pl;
  marginright=pr;
  textColor=getApp()->foreColor;
  baseColor=getApp()->baseColor;
  hiliteColor=getApp()->hiliteColor;
  shadowColor=getApp()->shadowColor;
  borderColor=getApp()->borderColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  cellBackColor[0][0]=FXRGB(255,255,255);     // Even row, even column
  cellBackColor[0][1]=FXRGB(255,255,255);     // Even row, odd column
  cellBackColor[1][0]=FXRGB(255,240,240);     // Odd row, even column
  cellBackColor[1][1]=FXRGB(255,240,240);     // Odd row, odd column
  gridColor=getApp()->baseColor;
  defColumnWidth=DEFAULTCOLUMNWIDTH;
  defRowHeight=DEFAULTROWHEIGHT;
  leading_rows=0;
  leading_cols=0;
  trailing_rows=0;
  trailing_cols=0;
  scrollable_left=0;
  scrollable_right=0;
  scrollable_top=0;
  scrollable_bottom=0;
  table_left=0;
  table_right=0;
  table_top=0;
  table_bottom=0;
  current.row=0;
  current.col=0;
  anchor.row=0;
  anchor.col=0;
  extent.row=0;
  extent.col=0;
  cellcursor=0;
  cellanchor=0;
  cellscroll=0;
  mode=MODE_NONE;
  blinker=NULL;
/*
  // Make cells
  FXCALLOC(&cells,FXTableItem*,nrows*ncols+1);
  if(!cells){ fxerror("FXTable::FXTable: out of memory\n"); }

  // Column widths
  FXMALLOC(&col_x,FXint,ncols+1);
  if(!col_x){ fxerror("FXTable::FXTable: out of memory\n"); }
  for(i=t=0; i<=ncols; i++){ col_x[i]=t; t+=defColumnWidth; }

  // Custom row heights
  FXMALLOC(&row_y,FXint,nrows+1);
  if(!row_y){ fxerror("FXTable::FXTable: out of memory\n"); }
  for(i=t=0; i<=nrows; i++){ row_y[i]=t; t+=defRowHeight; }

////// Begin test
  static const FXchar *months[]={"January","February","March","April","May","June","July","August","September","October","November","December"};
  FXTableItem *item;
  for(i=0; i<nrows; i++){
    for(j=0; j<ncols; j++){
#ifndef WIN32
      item=createItem(FXStringVal(1000.0*drand48()),NULL,NULL);
#else
      item=createItem(FXStringVal(1000.0*rand()/(double)RAND_MAX),NULL,NULL);
#endif
      cells[i*ncols+j]=item;
      if(i==0 || j==0 || i==nrows-1 || j==ncols-1) item->state=FXTableItem::BUTTON;
      if(i==0 && j==0) item->label="";
      if(i==0 && j>0) item->label=months[(j-1)%12];
      if(j==0 && i>0) item->label=FXStringVal(i);
      }
    }
////// End test
*/
  }


// Create window
void FXTable::create(){
  register FXint i;
  FXScrollArea::create();
  for(i=0; i<nrows*ncols; i++) cells[i]->create(); 
  font->create();
  }


// Detach window
void FXTable::detach(){
  register FXint i;
  FXScrollArea::detach();
  for(i=0; i<nrows*ncols; i++) cells[i]->detach(); 
  font->detach();
  }


// Can have focus
FXbool FXTable::canFocus() const { return TRUE; }


// Get default width
FXint FXTable::getDefaultWidth(){
  if(visiblecols){
    return visiblecols*defColumnWidth;
    }
  return FXScrollArea::getDefaultWidth();
  }


// Get default height
FXint FXTable::getDefaultHeight(){
  if(visiblerows){
    return visiblerows*defRowHeight;
    }
  return FXScrollArea::getDefaultHeight();
  }


// Create item
FXTableItem* FXTable::createItem(const FXString& text,FXIcon* icon,void* ptr){ 
  return new FXTableItem(text,icon,ptr); 
  }


// Propagate size change
void FXTable::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  }


// Find value in array within [l,h]
FXint FXTable::bsearch(FXint *array,FXint l,FXint h,FXint value){
  register FXint m;
  do{
    m=(h+l)>>1;
    if(array[m+1]<=value) l=m+1;
    else if(value<array[m]) h=m-1; 
    else break;
    }
  while(h>=l);
  return m;
  }


// Move content
void FXTable::moveContents(FXint x,FXint y){
  FXint dx=x-pos_x;
  FXint dy=y-pos_y;
  
  // Hide cursor
  drawCursor(0);
  
  // Update position
  pos_x=x;
  pos_y=y;
  
  // Scroll leading fixed rows
  if(leading_rows){ 
    scroll(scrollable_left,table_top,scrollable_right-scrollable_left,scrollable_top-table_top,dx,0);
    }
  
  // Scroll leading fixed columns
  if(leading_cols){ 
    scroll(table_left,scrollable_top,scrollable_left-table_left,scrollable_bottom-scrollable_top,0,dy);
    }
  
  // Scroll trailing fixed rows
  if(trailing_rows){ 
    scroll(scrollable_left,scrollable_bottom,scrollable_right-scrollable_left,table_bottom-scrollable_bottom,dx,0);
    }
  
  // Scroll trailing fixed columns
  if(trailing_cols){ 
    scroll(scrollable_right,scrollable_top,table_right-scrollable_right,scrollable_bottom-scrollable_top,0,dy);
    }

  // Scroll table
  scroll(scrollable_left,scrollable_top,scrollable_right-scrollable_left,scrollable_bottom-scrollable_top,dx,dy);
  }


// Force position to become fully visible
void FXTable::makePositionVisible(FXint r,FXint c){
  register FXint xlo,xhi,ylo,yhi,px,py;
  if(xid){
    px=pos_x;
    py=pos_y;
    if(leading_cols<=c && c<ncols-trailing_cols){
      xlo=col_x[c]; 
      xhi=col_x[c+1]; 
      if(px+xhi >= scrollable_right) px=scrollable_right-xhi;
      if(px+xlo <= scrollable_left) px=scrollable_left-xlo;
      }
    if(leading_rows<=r && r<nrows-trailing_rows){
      ylo=row_y[r]; 
      yhi=row_y[r+1]; 
      if(py+yhi >= scrollable_bottom) py=scrollable_bottom-yhi;
      if(py+ylo <= scrollable_top) py=scrollable_top-ylo;
      }
    setPosition(px,py);
    }
  }
  

// Get row containing y
FXint FXTable::rowAtY(FXint y) const {
  if(y<table_top || table_bottom<=y) return -1;
  if(y<scrollable_top) return bsearch(row_y,0,leading_rows-1,y);
  if(scrollable_bottom<=y) return bsearch(row_y,nrows-trailing_rows,nrows-1,y-scrollable_bottom+row_y[nrows-trailing_rows]);
  return bsearch(row_y,leading_rows,nrows-trailing_rows-1,y-pos_y);
  }


// Get column containing x, -1 if outside table
FXint FXTable::colAtX(FXint x) const {
  if(x<table_left || table_right<=x) return -1;
  if(x<scrollable_left) return bsearch(col_x,0,leading_cols-1,x);
  if(scrollable_right<=x) return bsearch(col_x,ncols-trailing_cols,ncols-1,x-scrollable_right+col_x[ncols-trailing_cols]);
  return bsearch(col_x,leading_cols,ncols-trailing_cols-1,x-pos_x);
  }


// Repaint
void FXTable::updateItem(FXint r,FXint c){
  FXint xl,xr,yt,yb;
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::updateItem: indices out of range.\n",getClassName()); } 
  if(xid){
    if(c<leading_cols){
      xl=col_x[c];
      xr=col_x[c+1];
      }
    else if(ncols-trailing_cols<=c){
      xl=scrollable_right+col_x[c]-col_x[ncols-trailing_cols];
      xr=scrollable_right+col_x[c+1]-col_x[ncols-trailing_cols];
      }
    else{
      xl=pos_x+col_x[c];
      xr=pos_x+col_x[c+1];
      }
    if(r<leading_rows){
      yt=row_y[r];
      yb=row_y[r+1];
      }
    else if(nrows-trailing_rows<=r){
      yt=scrollable_bottom+row_y[r]-row_y[nrows-trailing_rows];
      yb=scrollable_bottom+row_y[r+1]-row_y[nrows-trailing_rows];
      }
    else{
      yt=pos_y+row_y[r];
      yb=pos_y+row_y[r+1];
      }
    update(xl,yt,xr-xl+1,yb-yt+1);
    }
  }


// Change item text
void FXTable::setItemText(FXint r,FXint c,const FXString& text){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::setItemText: indices out of range.\n",getClassName()); } 
  cells[r*ncols+c]->label=text;
  updateItem(r,c);
  }


// Get item text
FXString FXTable::getItemText(FXint r,FXint c) const {
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::getItemText: indices out of range.\n",getClassName()); } 
  return cells[r*ncols+c]->label;
  }


// Set item icon
void FXTable::setItemIcon(FXint r,FXint c,FXIcon* icon){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::setItemIcon: indices out of range.\n",getClassName()); } 
  cells[r*ncols+c]->icon=icon;
  updateItem(r,c);
  }

// Get item icon
FXIcon* FXTable::getItemIcon(FXint r,FXint c) const {
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::getItemIcon: indices out of range.\n",getClassName()); } 
  return cells[r*ncols+c]->icon;
  }


// Set item data
void FXTable::setItemData(FXint r,FXint c,void* ptr){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::setItemData: indices out of range.\n",getClassName()); } 
  cells[r*ncols+c]->data=ptr;
  }


// Get item data
void* FXTable::getItemData(FXint r,FXint c) const {
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::getItemData: indices out of range.\n",getClassName()); } 
  return cells[r*ncols+c]->data;
  }


// True if item is selected
FXbool FXTable::isItemSelected(FXint r,FXint c) const {
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::isItemSelected: indices out of range.\n",getClassName()); } 
  return (cells[r*ncols+c]->state&FXTableItem::SELECTED)!=0; 
  }


// True if item is current
FXbool FXTable::isItemCurrent(FXint r,FXint c) const {
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::isItemCurrent: indices out of range.\n",getClassName()); } 
  return current.row==r && current.col==c;
  }


// True if item (partially) visible
FXbool FXTable::isItemVisible(FXint r,FXint c) const {
  FXbool vis=TRUE;
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::isItemVisible: indices out of range.\n",getClassName()); } 
  if(c<leading_cols){
    if(table_right<=col_x[c]) vis=FALSE;
    }
  else if(ncols-trailing_cols<=c){
    if(scrollable_right+col_x[c+1]-col_x[ncols-trailing_cols]<table_left) vis=FALSE;
    }
  else{
    if(pos_x+col_x[c+1]<scrollable_left || scrollable_right<=pos_x+col_x[c]) vis=FALSE;
    }
  if(r<leading_rows){
    if(table_bottom<=row_y[r]) vis=FALSE;
    }
  else if(nrows-trailing_rows<=r){
    if(scrollable_bottom+row_y[r+1]-row_y[nrows-trailing_rows]<table_top) vis=FALSE;
    }
  else{
    if(pos_y+row_y[r+1]<scrollable_top || scrollable_bottom<=pos_y+row_y[r]) vis=FALSE;
    }
  return vis;
  }


// True if item is enabled
FXbool FXTable::isItemEnabled(FXint r,FXint c) const {
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::isItemEnabled: indices out of range.\n",getClassName()); } 
  return (cells[r*ncols+c]->state&FXTableItem::DISABLED)==0; 
  }


// Enable one item
FXbool FXTable::enableItem(FXint r,FXint c){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::enableItem: indices out of range.\n",getClassName()); } 
  FXTableItem *item=cells[r*ncols+c];
  if(item->state&FXTableItem::DISABLED){
    item->state&=~FXTableItem::DISABLED;
    updateItem(r,c);
    return TRUE;
    }
  return FALSE;
  }


// Disable one item
FXbool FXTable::disableItem(FXint r,FXint c){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::disableItem: indices out of range.\n",getClassName()); } 
  FXTableItem *item=cells[r*ncols+c];
  if(!(item->state&FXTableItem::DISABLED)){
    item->state|=FXTableItem::DISABLED;
    updateItem(r,c);
    return TRUE;
    }
  return FALSE;
  }


// Select one item
FXbool FXTable::selectItem(FXint r,FXint c){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::selectItem: indices out of range.\n",getClassName()); } 
  FXTableItem *item=cells[r*ncols+c];
  if(!(item->state&FXTableItem::SELECTED)){
    item->state|=FXTableItem::SELECTED;
    updateItem(r,c);
    return TRUE;
    }
  return FALSE;
  }


// Deselect one item
FXbool FXTable::deselectItem(FXint r,FXint c){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::deselectItem: indices out of range.\n",getClassName()); } 
  FXTableItem *item=cells[r*ncols+c];
  if(item->state&FXTableItem::SELECTED){
    item->state&=~FXTableItem::SELECTED;
    updateItem(r,c);
    return TRUE;
    }
  return FALSE;
  }


// Toggle one item
FXbool FXTable::toggleItem(FXint r,FXint c){
  if(r<0 || nrows<=r || c<0 || ncols<=c){ fxerror("%s::toggleItem: indices out of range.\n",getClassName()); } 
  FXTableItem *item=cells[r*ncols+c];
  item->state^=FXTableItem::SELECTED;
  updateItem(r,c);
  return TRUE;
  }


// Set current item
void FXTable::setCurrentItem(FXint r,FXint c){
  if(r<0) r=0;
  if(c<0) c=0;
  if(r>=nrows) r=nrows-1;
  if(c>=ncols) c=ncols-1;
  if(r!=current.row || c!=current.col){
    
    // Deactivate old item
    if(0<=current.row && 0<=current.col){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        FXASSERT(current.row<nrows);
        FXASSERT(current.col<ncols);
        cells[current.row*ncols+current.col]->state&=~FXTableItem::FOCUS;
        updateItem(current.row,current.col);
        }
      }

    current.row=r;
    current.col=c;
    
    // Activate new item
    if(0<=current.row && 0<=current.col){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        FXASSERT(current.row<nrows);
        FXASSERT(current.col<ncols);
        cells[current.row*ncols+current.col]->state|=FXTableItem::FOCUS;
        updateItem(current.row,current.col);
        }
      }
    }
  }


// Set anchor item
void FXTable::setAnchorItem(FXint r,FXint c){
  if(r<0) r=0;
  if(c<0) c=0;
  if(r>=nrows) r=nrows-1;
  if(c>=ncols) c=ncols-1;
  anchor.row=r;
  anchor.col=c;
  extent.row=r;
  extent.col=c;
  }


// Mark and select
FXbool FXTable::mark(FXint rlo,FXint rhi,FXint clo,FXint chi,FXuint sel){
  register FXbool changes=FALSE;
  register FXTableItem *item;
  register FXint r,c;
  for(r=rlo; r<=rhi; r++){
    for(c=clo; c<=chi; c++){
      item=cells[r*ncols+c];
      if(sel){
        if(!(item->state&FXTableItem::SELECTED)){
          item->state&=~FXTableItem::MARKED;
          if(!(item->state&FXTableItem::DISABLED)){
            item->state|=FXTableItem::SELECTED;
            updateItem(r,c);
            changes=TRUE;
            }
          }
        else{
          item->state|=FXTableItem::MARKED;
          }
        }
      else{
        if(item->state&FXTableItem::SELECTED){
          item->state|=FXTableItem::MARKED;
          if(!(item->state&FXTableItem::DISABLED)){
            item->state&=~FXTableItem::SELECTED;
            updateItem(r,c);
            changes=TRUE;
            }
          }
        else{
          item->state&=~FXTableItem::MARKED;
          }
        }
      }
    }
  return changes;
  }


// Restore to mark
FXbool FXTable::restore(FXint rlo,FXint rhi,FXint clo,FXint chi){
  register FXbool changes=FALSE;
  register FXTableItem *item;
  register FXint r,c;
  for(r=rlo; r<=rhi; r++){
    for(c=clo; c<=chi; c++){
      item=cells[r*ncols+c];
      if(item->state&FXTableItem::MARKED){
        if(!(item->state&FXTableItem::SELECTED)){
          item->state|=FXTableItem::SELECTED;
          updateItem(r,c);
          changes=TRUE;
          }
        }
      else{
        if(item->state&FXTableItem::SELECTED){
          item->state&=~FXTableItem::SELECTED;
          updateItem(r,c);
          changes=TRUE;
          }
        }
      }
    }
  return changes;
  }


// Extend selection
FXbool FXTable::extendSelection(FXint r,FXint c){
  register FXbool changes=FALSE;
  register FXuint sel;
  if(0<=r && 0<=c && 0<=anchor.row && 0<=anchor.col){
    sel=(cells[anchor.row*ncols+anchor.col]->state&FXTableItem::SELECTED);
    changes|=restore(FXMIN(extent.row,anchor.row),FXMAX(extent.row,anchor.row),FXMIN(extent.col,anchor.col),FXMAX(extent.col,anchor.col));
    changes|=mark(FXMIN(r,anchor.row),FXMAX(r,anchor.row),FXMIN(c,anchor.col),FXMAX(c,anchor.col),sel);
    extent.row=r;
    extent.col=c;
    }
  return changes;
  }


// Kill selection
FXbool FXTable::killSelection(){
  register FXbool changes=FALSE;
  register FXint r,c;
  FXTableItem *item;
  for(r=0; r<nrows; r++){
    for(c=0; c<ncols; c++){
      item=cells[r*ncols+c];
      if(item->state&FXTableItem::SELECTED){
        item->state&=~FXTableItem::SELECTED;
        updateItem(r,c);
        changes=TRUE;
        }
      }
    }
  extent=anchor;
  return changes;
  }


// Determine scrollable content width
FXint FXTable::getContentWidth(){
  return col_x[ncols];
  }


// Determine scrollable content height
FXint FXTable::getContentHeight(){
  return row_y[nrows];
  }


// Recalculate layout determines item locations and sizes
void FXTable::layout(){
    
  // Calculate contents 
  FXScrollArea::layout();
  
  // Whole table placement
  table_left=col_x[0];
  table_top=row_y[0];
  table_right=col_x[ncols];
  table_bottom=row_y[nrows];
  
  // Adjust right and bottom
  if(table_right>viewport_w) table_right=viewport_w;
  if(table_bottom>viewport_h) table_bottom=viewport_h;
  
  // Scrollable part of table
  scrollable_left=col_x[leading_cols];
  scrollable_top=row_y[leading_rows];
  scrollable_right=table_right-col_x[ncols]+col_x[ncols-trailing_cols];
  scrollable_bottom=table_bottom-row_y[nrows]+row_y[nrows-trailing_rows];
  
  FXTRACE((250,"     table_left=%d      table_right=%d      table_top=%d table_bottom=%d\n",table_left,table_right,table_top,table_bottom));
  FXTRACE((250,"scrollable_left=%d scrollable_right=%d scrollable_top=%d table_bottom=%d\n",scrollable_left,scrollable_right,scrollable_top,scrollable_bottom));

  // Determine line size for scroll bars
  vertical->setLine(defRowHeight);
  horizontal->setLine(defColumnWidth);
  
  // Force repaint
  update();
  
  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Gained focus
long FXTable::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(!blinker) blinker=getApp()->addTimeout(getApp()->blinkSpeed,this,FXWindow::ID_CARETBLINK);
  drawCursor(FLAG_CARET);
  if(0<=current.row && 0<=current.col){
    FXASSERT(current.row<nrows);
    FXASSERT(current.col<ncols);
    FXTableItem *item=cells[current.row*ncols+current.col];
    if(item){
      item->state|=FXTableItem::FOCUS;
      updateItem(current.row,current.col);
      }
    }
  return 1;
  }


// Lost focus
long FXTable::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(blinker){getApp()->removeTimeout(blinker);blinker=NULL;}
  drawCursor(0);
  if(0<=current.row && 0<=current.col){
    FXASSERT(current.row<nrows);
    FXASSERT(current.col<ncols);
    FXTableItem *item=cells[current.row*ncols+current.col];
    if(item){
      item->state&=~FXTableItem::FOCUS;
      updateItem(current.row,current.col);
      }
    }
  return 1;
  }


// We have the selection
long FXTable::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionGained(sender,sel,ptr);
  ////
  return 1;
  }


// We lost the selection
long FXTable::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionLost(sender,sel,ptr);
  ////
  return 1;
  }


// Blink the cursor
long FXTable::onBlink(FXObject*,FXSelector,void*){
  drawCursor(flags^FLAG_CARET);
  blinker=getApp()->addTimeout(getApp()->blinkSpeed,this,FXWindow::ID_CARETBLINK);
  return 0;
  }


// Draw the cursor
void FXTable::drawCursor(FXuint state){
  //register FXint xx,yt,yb,cl;
  if((state^flags)&FLAG_CARET){
    FXDCWindow dc(this);
//     xx=pos_x+marginleft+cursorx;
//     yt=pos_y+margintop+cursory;
//     yb=yt+font->getFontHeight()-1;
//     dc.begin(this);
//     dc.setClipRectangle(marginleft,margintop,viewport_w-marginleft-marginright,viewport_h-margintop-marginbottom);
//     if(flags&FLAG_CARET){
//       if(lines[0]<=cursorpos && cursorpos<=lines[nvislines]){
//         
//         // Cursor may be in the selection
//         if(hasSelection() && selstartpos<=cursorpos && cursorpos<selendpos){
//           dc.setForeground(selbackColor);
//           }
//         else{
//           dc.setForeground(backColor);
//           }
//         
//         // Repaint cursor in background to erase it
//         dc.drawLine(xx,yt,xx,yb);
//         dc.drawLine(xx+1,yt,xx+1,yb);
//         dc.drawLine(xx-2,yt,xx+3,yt);
//         dc.drawLine(xx-2,yb,xx+3,yb);
//         
//         // Restore text
//         dc.setTextFont(font);
//         cl=posToLine(cursorpos);
//         drawTextLine(cl,xx-3,xx+3,0,10000);
//         }
//       flags&=~FLAG_CARET;
//       }
//     else{
//       if(lines[0]<=cursorpos && cursorpos<=lines[nvislines]){
//         dc.setForeground(cursorColor);
//         dc.drawLine(xx,yt,xx,yb);
//         dc.drawLine(xx+1,yt,xx+1,yb);
//         dc.drawLine(xx-2,yt,xx+3,yt);
//         dc.drawLine(xx-2,yb,xx+3,yb);
//         flags|=FLAG_CARET;
//         }
//       }
//     dc.clearClipRectangle();
//     dc.end();
    }
  }


// Draw table fragment [rlo:rhi, clo:chi] clipping against box
void FXTable::drawTableRange(FXDC& dc,FXint xlo,FXint xhi,FXint ylo,FXint yhi,FXint xoff,FXint yoff,FXint rlo,FXint rhi,FXint clo,FXint chi){
  FXint r,c,xl,xr,yt,yb,fr,lr,fc,lc;
  FXTableItem *item;
  if(xlo<xhi && ylo<yhi){
    
    // Find dirty part of table
    fc=bsearch(col_x,clo,chi,xlo-xoff);
    lc=bsearch(col_x,clo,chi,xhi-xoff-1);
    fr=bsearch(row_y,rlo,rhi,ylo-yoff);
    lr=bsearch(row_y,rlo,rhi,yhi-yoff-1);
    FXASSERT(0<=fc && lc<ncols);
    FXASSERT(0<=fr && lr<nrows);
    
    FXTRACE((250,"fr=%d lr=%d fc=%d lc=%d\n",fr,lr,fc,lc));
    
    // Draw cell contents
    for(r=fr; r<=lr; r++){
      yt=yoff+row_y[r];
      yb=yoff+row_y[r+1];
      if(options&TABLE_HOR_GRIDLINES) yb--;
      for(c=fc; c<=lc; c++){
        xl=xoff+col_x[c];
        xr=xoff+col_x[c+1];
        if(options&TABLE_VER_GRIDLINES) xr--;
        item=cells[r*ncols+c];
        dc.setClipRectangle(FXMAX(xlo,xl),FXMAX(ylo,yt),FXMIN(xhi,xr)-FXMAX(xlo,xl),FXMIN(yhi,yb)-FXMAX(ylo,yt));
        dc.setForeground(cellBackColor[r&1][c&1]);
        dc.fillRectangle(xl,yt,xr-xl,yb-yt);
        if(item) item->draw(this,dc,xl,yt,xr-xl,yb-yt);
        }
      }

    // Draw horizontal grid lines
    if(options&TABLE_HOR_GRIDLINES){
      dc.setClipRectangle(xlo,ylo,xhi-xlo,yhi-ylo);
      dc.setForeground(gridColor);
      for(r=fr; r<=lr+1; r++){
        yt=yoff+row_y[r]-1;
        dc.drawLine(xoff+col_x[fc],yt,xoff+col_x[lc+1],yt);
        }
      }

    // Draw vertical grid lines
    if(options&TABLE_VER_GRIDLINES){
      dc.setClipRectangle(xlo,ylo,xhi-xlo,yhi-ylo);
      dc.setForeground(gridColor);
      for(c=fc; c<=lc+1; c++){
        xl=xoff+col_x[c]-1;
        dc.drawLine(xl,yoff+row_y[fr],xl,yoff+row_y[lr+1]);
        }
      }
    }
  }


// Draw exposed part of table
long FXTable::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXDCWindow dc(this,event);
  FXint xlo,xhi,ylo,yhi;
  FXint llx,lhx,mlx,mhx,rlx,rhx;
  FXint tly,thy,mly,mhy,bly,bhy;

  dc.setTextFont(font);
  
//dc.setForeground(FXRGB(255,0,0));
//dc.fillRectangle(event->rect.x,event->rect.y,event->rect.w,event->rect.h);

  // Fill background right of the table
  dc.setForeground(backColor);
  if(width>col_x[ncols]){
    dc.fillRectangle(col_x[ncols],0,width-col_x[ncols],height);
    }
  
  // Fill background below the table
  if(height>row_y[nrows]){
    dc.fillRectangle(0,row_y[nrows],col_x[ncols],height-row_y[nrows]);
    }
  
  // Exposed area
  xlo=event->rect.x; 
  xhi=event->rect.x+event->rect.w; 
  ylo=event->rect.y; 
  yhi=event->rect.y+event->rect.h; 
    
  // Overlap with columns
  llx=FXMAX(xlo,table_left);
  lhx=FXMIN(xhi,scrollable_left);
  mlx=FXMAX(xlo,scrollable_left);
  mhx=FXMIN(xhi,scrollable_right);
  rlx=FXMAX(xlo,scrollable_right);
  rhx=FXMIN(xhi,table_right);
  
  
  // Overlap with rows
  tly=FXMAX(ylo,table_top);
  thy=FXMIN(yhi,scrollable_top);
  mly=FXMAX(ylo,scrollable_top);
  mhy=FXMIN(yhi,scrollable_bottom);
  bly=FXMAX(ylo,scrollable_bottom);
  bhy=FXMIN(yhi,table_bottom);
  
  FXTRACE((250,"llx=%d lhx=%d mlx=%d mhx=%d rlx=%d rhx=%d\n",llx,lhx,mlx,mhx,rlx,rhx));
  FXTRACE((250,"tly=%d thy=%d mly=%d mhy=%d bly=%d bhy=%d\n",tly,thy,mly,mhy,bly,bhy));
  
  // Make sure invariants are met
  FXASSERT(leading_rows+trailing_rows<=nrows);
  FXASSERT(leading_cols+trailing_cols<=ncols);
  
  // Draw unscrolled pieces 
  drawTableRange(dc,llx,lhx,tly,thy,0,0,0,leading_rows-1,0,leading_cols-1);
  drawTableRange(dc,rlx,rhx,tly,thy,scrollable_right-col_x[ncols-trailing_cols],0,0,leading_rows-1,ncols-trailing_cols,ncols-1);
  drawTableRange(dc,llx,lhx,bly,bhy,0,scrollable_bottom-row_y[nrows-trailing_rows],nrows-trailing_rows,nrows-1,0,leading_cols-1);
  drawTableRange(dc,rlx,rhx,bly,bhy,scrollable_right-col_x[ncols-trailing_cols],scrollable_bottom-row_y[nrows-trailing_rows],nrows-trailing_rows,nrows-1,ncols-trailing_cols,ncols-1);
  
  // Draw horizontal-only scrolled pieces
  drawTableRange(dc,mlx,mhx,tly,thy,pos_x,0,0,leading_rows-1,leading_cols,ncols-trailing_cols-1);
  drawTableRange(dc,mlx,mhx,bly,bhy,pos_x,scrollable_bottom-row_y[nrows-trailing_rows],nrows-trailing_rows,nrows-1,leading_cols,ncols-trailing_cols-1);

  // Draw vertical-only scrolled pieces
  drawTableRange(dc,llx,lhx,mly,mhy,0,pos_y,leading_rows,nrows-trailing_rows-1,0,leading_cols-1);
  drawTableRange(dc,rlx,rhx,mly,mhy,scrollable_right-col_x[ncols-trailing_cols],pos_y,leading_rows,nrows-trailing_rows-1,ncols-trailing_cols,ncols-1);
  
  // Draw main table piece which is scrolled both ways
  drawTableRange(dc,mlx,mhx,mly,mhy,pos_x,pos_y,leading_rows,nrows-trailing_rows-1,leading_cols,ncols-trailing_cols-1);
  
  return 1;
  }


// Key Press
long FXTable::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTableItem* item;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Home:
    case KEY_End:
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Up:
    case KEY_KP_Up:
      if(!(event->state&SHIFTMASK)){
        handle(this,MKUINT(ID_DESELECT_ALL,SEL_COMMAND),NULL);
        }
      handle(this,MKUINT(ID_MOVE_UP,SEL_COMMAND),NULL);
      if(event->state&SHIFTMASK){
        handle(this,MKUINT(ID_EXTEND,SEL_COMMAND),NULL);
        }
      else{
        handle(this,MKUINT(ID_MARK,SEL_COMMAND),NULL);
        }
      return 1;
    case KEY_Down:
    case KEY_KP_Down:
      if(!(event->state&SHIFTMASK)){
        handle(this,MKUINT(ID_DESELECT_ALL,SEL_COMMAND),NULL);
        }
      handle(this,MKUINT(ID_MOVE_DOWN,SEL_COMMAND),NULL);
      if(event->state&SHIFTMASK){
        handle(this,MKUINT(ID_EXTEND,SEL_COMMAND),NULL);
        }
      else{
        handle(this,MKUINT(ID_MARK,SEL_COMMAND),NULL);
        }
      return 1;
    case KEY_Left:
    case KEY_KP_Left:
      if(!(event->state&SHIFTMASK)){
        handle(this,MKUINT(ID_DESELECT_ALL,SEL_COMMAND),NULL);
        }
      handle(this,MKUINT(ID_MOVE_LEFT,SEL_COMMAND),NULL);
      if(event->state&SHIFTMASK){
        handle(this,MKUINT(ID_EXTEND,SEL_COMMAND),NULL);
        }
      else{
        handle(this,MKUINT(ID_MARK,SEL_COMMAND),NULL);
        }
      return 1;
    case KEY_Right:
    case KEY_KP_Right:
      if(!(event->state&SHIFTMASK)){
        handle(this,MKUINT(ID_DESELECT_ALL,SEL_COMMAND),NULL);
        }
      handle(this,MKUINT(ID_MOVE_RIGHT,SEL_COMMAND),NULL);
      if(event->state&SHIFTMASK){
        handle(this,MKUINT(ID_EXTEND,SEL_COMMAND),NULL);
        }
      else{
        handle(this,MKUINT(ID_MARK,SEL_COMMAND),NULL);
        }
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      return 1;
    }
  return 0;
  }


// Key Release 
long FXTable::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  flags|=FLAG_UPDATE;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Page_Up:
    case KEY_Page_Down:
    case KEY_Left:
    case KEY_Right:
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      flags|=FLAG_UPDATE;
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      return 1;
    }
  return 0;
  }


// Automatic scroll
long FXTable::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  
  // Scroll window
  FXScrollArea::onAutoScroll(sender,sel,ptr);
  
  return 1;
  }


// Mouse moved
long FXTable::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint r,c,i,xx,yy;
//// TEST
  //FXTRACE((100,"x=%d y=%d col=%d row=%d\n",event->win_x,event->win_y,colAtX(event->win_x),rowAtY(event->win_y)));
  if(flags&FLAG_PRESSED){
    
    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;
    
    return 1;
    }
  
//   // Resize column cursor
//   c=vertGridLine(1,ncols,event->win_x);
//   if(0<=c){
//     setDefaultCursor(getApp()->hsplitCursor);
//     setDragCursor(getApp()->hsplitCursor);
//     return 0;
//     }
//   
//   // Resize row cursor
//   r=horzGridLine(1,nrows,event->win_y);
//   if(0<=r){
//     setDefaultCursor(getApp()->vsplitCursor);
//     setDragCursor(getApp()->vsplitCursor);
//     return 0;
//     }
//   
//   // Normal cursor
//   setDefaultCursor(getApp()->arrowCursor);
//   setDragCursor(getApp()->arrowCursor);
  return 0;
  }


// Y coordinate of row r
FXint FXTable::yOfRow(FXint r) const {
  FXint y=row_y[r];
  if(nrows-trailing_rows<=r) 
    y=y+scrollable_bottom-row_y[nrows-trailing_rows];
  else if(leading_rows<=r)
    y=y+pos_y;
  return y;
  }


// X coordinate of column c
FXint FXTable::xOfCol(FXint c) const {
  FXint x=col_x[c];
  if(ncols-trailing_cols<=c) 
    x=x+scrollable_right-col_x[ncols-trailing_cols];
  else if(leading_cols<=c) 
    x=x+pos_x;
  return x;
  }

 
  /*
// Find horizontal grid line close to y
FXint FXTable::horzGridLine(FXint rlo,FXint rhi,FXint y) const {
  if(scrollable_bottom<=y) 
    y=y-scrollable_bottom+row_y[nrows-trailing_rows];
  else if(scrollable_top<=y) 
    y=y-pos_y;
  for(FXint r=rlo; r<=rhi; r++){
    if(row_y[r]-FUDGE<=y && y<row_y[r]+FUDGE-1) return r;
    }
  return -1;
  }


// Find vertical grid line close to x
FXint FXTable::vertGridLine(FXint clo,FXint chi,FXint x) const {
  if(scrollable_right<=x) 
    x=x-scrollable_right+col_x[ncols-trailing_cols];
  else if(scrollable_left<=x) 
    x=x-pos_x;
  for(FXint c=clo; c<=chi; c++){
    if(col_x[c]-FUDGE<=x && x<col_x[c]+FUDGE-1) return c;
    }
  return -1;
  }
*/


// Pressed button
long FXTable::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTablePos tablepos;
  FXint xx,yy;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    tablepos.row=rowAtY(event->win_y);
    tablepos.col=colAtX(event->win_x);
    FXTRACE((100,"press row=%d col=%d\n",tablepos.row,tablepos.col));
    xx=xOfCol(tablepos.col+1);
    yy=yOfRow(tablepos.row+1);
    if(xx-FUDGE<=event->win_x && event->win_x<xx+FUDGE-1){
      FXTRACE((100,"v drag\n"));
      }
    else if(yy-FUDGE<=event->win_y && event->win_y<yy+FUDGE-1){
      FXTRACE((100,"h drag\n"));
      }
    else{
      FXTRACE((100,"cell\n"));
      }
    makePositionVisible(tablepos.row,tablepos.col);
    handle(this,MKUINT(0,SEL_CHANGED),(void*)&tablepos);
    handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    flags&=~FLAG_UPDATE;
    flags|=FLAG_PRESSED;
    return 1;
    }
  return 0;
  }


// Released button
long FXTable::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    flags&=~FLAG_PRESSED;
    flags|=FLAG_UPDATE;
    stopAutoScroll();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
    makePositionVisible(current.row,current.col);
    mode=MODE_NONE;
    return 1;
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXTable::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags&=~FLAG_CHANGED;
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }


// Button or Key activate
long FXTable::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTableItem *item;
  if(0<=current.row && 0<=current.col){
    item=cells[current.row*ncols+current.col];
    if(item && item->isEnabled()){
      if(event->state&SHIFTMASK){
        if(0<=anchor.row && 0<=anchor.col){
          selectItem(anchor.row,anchor.col);
          extendSelection(current.row,current.col);
          }
        else{
          selectItem(current.row,current.col);
          setAnchorItem(current.row,current.col);
          }
        }
      else if(event->state&CONTROLMASK){
        toggleItem(current.row,current.col);
        setAnchorItem(current.row,current.col);
        }
      else{
        killSelection();
        selectItem(current.row,current.col);
        setAnchorItem(current.row,current.col);
        }
      }
    }
  else{
    if(!(event->state&(SHIFTMASK|CONTROLMASK))) killSelection();
    }
  return 1;
  }


// Button or Key deactivate
long FXTable::onDeactivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTableItem *item;
  setAnchorItem(current.row,current.col);
  if(event->click_count==1){
    handle(this,MKUINT(0,SEL_CLICKED),(void*)&current);
    }
  else if(event->click_count==2){
    handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)&current);
    }
  else if(event->click_count==3){
    handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)&current);
    }
  if(0<=current.row && 0<=current.col){
    item=cells[current.row*ncols+current.col];
    if(item && item->isEnabled()){
      handle(this,MKUINT(0,SEL_COMMAND),(void*)&current);
      }
    }
  return 1;
  }


// Clicked in list
long FXTable::onClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_CLICKED),ptr);
  }


// Double clicked in list; ptr may or may not point to an item
long FXTable::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr);
  }


// Triple clicked in list; ptr may or may not point to an item
long FXTable::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Current item changed
long FXTable::onChanged(FXObject*,FXSelector,void* ptr){
  FXTablePos *tablepos=(FXTablePos*)ptr;
  setCurrentItem(tablepos->row,tablepos->col);
  if(target) target->handle(this,MKUINT(message,SEL_CHANGED),ptr);
  return 1;
  }


// Command message
long FXTable::onCommand(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
  }


// Selected items
long FXTable::onSelected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_SELECTED),ptr);
  }


// Deselected items
long FXTable::onDeselected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DESELECTED),ptr);
  }


// Inserted cells
long FXTable::onInserted(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_INSERTED),ptr);
  }


// Deleted cells
long FXTable::onDeleted(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DELETED),ptr);
  }


// Toggle horizontal grid lines
long FXTable::onCmdHorzGrid(FXObject*,FXSelector,void*){
  options^=TABLE_HOR_GRIDLINES;
  recalc();
  return 1;
  }

long FXTable::onUpdHorzGrid(FXObject* sender,FXSelector,void*){
  FXuint msg=(options&TABLE_HOR_GRIDLINES) ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Toggle vertical grid lines
long FXTable::onCmdVertGrid(FXObject*,FXSelector,void*){
  options^=TABLE_VER_GRIDLINES;
  recalc();
  return 1;
  }

long FXTable::onUpdVertGrid(FXObject* sender,FXSelector,void*){
  FXuint msg=(options&TABLE_VER_GRIDLINES) ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Delete current column
long FXTable::onCmdDeleteColumn(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  if(current.col<0) return 1;
  tablerange.fm.row=0;
  tablerange.fm.col=current.col;
  tablerange.to.row=nrows-1;
  tablerange.to.col=current.col;
  handle(this,MKUINT(0,SEL_DELETED),(void*)&tablerange);
  deleteColumns(current.col,1);
  killSelection();
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&current);
  return 1;
  }


// Update delete current column
long FXTable::onUpdDeleteColumn(FXObject* sender,FXSelector,void*){
  if(0<=current.col && current.col<ncols && 0<ncols)
    sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(ID_DISABLE,SEL_COMMAND),NULL);
  return 1;
  }


// Delete current row
long FXTable::onCmdDeleteRow(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  if(current.row<0) return 1;
  tablerange.fm.row=current.row;
  tablerange.fm.col=0;
  tablerange.to.row=current.row;
  tablerange.to.col=ncols-1;
  handle(this,MKUINT(0,SEL_DELETED),(void*)&tablerange);
  deleteRows(current.row,1);
  killSelection();
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&current);
  return 1;
  }


// Update delete current row
long FXTable::onUpdDeleteRow(FXObject* sender,FXSelector,void*){
  if(0<=current.row && current.row<nrows && 0<nrows)
    sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(ID_DISABLE,SEL_COMMAND),NULL);
  return 1;
  }


// Insert new column at current
long FXTable::onCmdInsertColumn(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  if(current.col<0){
    insertColumns(ncols,1);
    }
  else{
    insertColumns(current.col,1);
    }
  tablerange.fm.row=0;
  tablerange.fm.col=current.col;
  tablerange.to.row=nrows-1;
  tablerange.to.col=current.col;
  handle(this,MKUINT(0,SEL_INSERTED),(void*)&tablerange);
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&current);
  return 1;
  }


// Insert new row at current
long FXTable::onCmdInsertRow(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  if(current.row<0){
    insertRows(nrows,1);
    }
  else{
    insertRows(current.row,1);
    }
  tablerange.fm.row=current.row;
  tablerange.fm.col=0;
  tablerange.to.row=current.row;
  tablerange.to.col=ncols-1;
  handle(this,MKUINT(0,SEL_INSERTED),(void*)&tablerange);
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&current);
  return 1;
  }


// Move to next column
long FXTable::onCmdMoveRight(FXObject*,FXSelector,void*){
  FXTablePos tablepos;
  if(current.col>ncols-2) return 1;
  tablepos.row=current.row;
  tablepos.col=current.col+1;
  makePositionVisible(tablepos.row,tablepos.col);
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&tablepos);
  return 1;
  }


// Move to previous column
long FXTable::onCmdMoveLeft(FXObject*,FXSelector,void*){
  FXTablePos tablepos;
  if(current.col<1) return 1;
  tablepos.row=current.row;
  tablepos.col=current.col-1;
  makePositionVisible(tablepos.row,tablepos.col);
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&tablepos);
  return 1;
  }


// Move to previous row
long FXTable::onCmdMoveUp(FXObject*,FXSelector,void*){
  FXTablePos tablepos;
  if(current.row<1) return 1;
  tablepos.row=current.row-1;
  tablepos.col=current.col;
  makePositionVisible(tablepos.row,tablepos.col);
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&tablepos);
  return 1;
  }


// Move to next row
long FXTable::onCmdMoveDown(FXObject*,FXSelector,void*){
  FXTablePos tablepos;
  if(current.row>nrows-2) return 1;
  tablepos.row=current.row+1;
  tablepos.col=current.col;
  makePositionVisible(tablepos.row,tablepos.col);
  handle(this,MKUINT(0,SEL_CHANGED),(void*)&tablepos);
  return 1;
  }


// Select cell
long FXTable::onCmdSelectCell(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  tablerange.fm.row=tablerange.to.row=current.row;
  tablerange.fm.col=tablerange.to.col=current.col;
  setAnchorItem(tablerange.fm.row,tablerange.fm.col);
  extendSelection(tablerange.to.row,tablerange.to.col);
  handle(this,MKUINT(0,SEL_SELECTED),(void*)&tablerange);
  return 1;
  }


// Select row
long FXTable::onCmdSelectRow(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  tablerange.fm.row=current.row;
  tablerange.fm.col=0;
  tablerange.to.row=current.row;
  tablerange.to.col=ncols-1;
  setAnchorItem(tablerange.fm.row,tablerange.fm.col);
  extendSelection(tablerange.to.row,tablerange.to.col);
  handle(this,MKUINT(0,SEL_SELECTED),(void*)&tablerange);
  return 1;
  }


// Select column
long FXTable::onCmdSelectColumn(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  tablerange.fm.row=0;
  tablerange.fm.col=current.col;
  tablerange.to.row=nrows-1;
  tablerange.to.col=current.col;
  setAnchorItem(tablerange.fm.row,tablerange.fm.col);
  extendSelection(tablerange.to.row,tablerange.to.col);
  handle(this,MKUINT(0,SEL_SELECTED),(void*)&tablerange);
  return 1;
  }


// Select all cells
long FXTable::onCmdSelectAll(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  tablerange.fm.row=0;
  tablerange.fm.col=0;
  tablerange.to.row=nrows-1;
  tablerange.to.col=ncols-1;
  setAnchorItem(tablerange.fm.row,tablerange.fm.col);
  extendSelection(tablerange.to.row,tablerange.to.col);
  handle(this,MKUINT(0,SEL_SELECTED),(void*)&tablerange);
  return 1;
  }


// Deselect all cells
long FXTable::onCmdDeselectAll(FXObject*,FXSelector,void*){
  FXTableRange tablerange;
  tablerange.fm.row=0;
  tablerange.fm.col=0;
  tablerange.to.row=nrows-1;
  tablerange.to.col=ncols-1;
  handle(this,MKUINT(0,SEL_DESELECTED),(void*)&tablerange);
  killSelection();
  return 1;
  }


// Mark
long FXTable::onCmdMark(FXObject*,FXSelector,void*){
  setAnchorItem(current.row,current.col);
  return 1;
  }


// Extend
long FXTable::onCmdExtend(FXObject*,FXSelector,void*){
  extendSelection(current.row,current.col);
  return 1;
  }


// Change the font
void FXTable::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Change top margin
void FXTable::setMarginTop(FXint mt){
  if(margintop!=mt){
    margintop=mt;
    recalc();
    update();
    }
  }


// Change bottom margin
void FXTable::setMarginBottom(FXint mb){
  if(marginbottom!=mb){
    marginbottom=mb;
    recalc();
    update();
    }
  }


// Change left margin
void FXTable::setMarginLeft(FXint ml){
  if(marginleft!=ml){
    marginleft=ml;
    recalc();
    update();
    }
  }


// Change right margin
void FXTable::setMarginRight(FXint mr){
  if(marginright!=mr){
    marginright=mr;
    recalc();
    update();
    }
  }


// Set text color
void FXTable::setTextColor(FXColor clr){
  if(clr!=textColor){
    textColor=clr;
    update();
    }
  }


// Set base color
void FXTable::setBaseColor(FXColor clr){
  if(clr!=baseColor){
    baseColor=clr;
    update();
    }
  }


// Set highlight color
void FXTable::setHiliteColor(FXColor clr){
  if(clr!=hiliteColor){
    hiliteColor=clr;
    update();
    }
  }


// Set shadow color
void FXTable::setShadowColor(FXColor clr){
  if(clr!=shadowColor){
    shadowColor=clr;
    update();
    }
  }


// Set border color
void FXTable::setBorderColor(FXColor clr){
  if(clr!=borderColor){
    borderColor=clr;
    update();
    }
  }


// Set select background color
void FXTable::setSelBackColor(FXColor clr){
  if(clr!=selbackColor){
    selbackColor=clr;
    update();
    }
  }


// Set selected text color
void FXTable::setSelTextColor(FXColor clr){
  if(clr!=seltextColor){
    seltextColor=clr;
    update();
    }
  }


// Change grid color
void FXTable::setGridColor(FXColor clr){
  if(clr!=gridColor){
    gridColor=clr;
    update();
    }
  }


// Set cell color
void FXTable::setCellColor(FXint r,FXint c,FXColor clr){
  if(clr!=cellBackColor[r&1][c&1]){
    cellBackColor[r&1][c&1]=clr;
    update();
    }
  }


// Get cell color
FXColor FXTable::getCellColor(FXint r,FXint c) const {
  return cellBackColor[r&1][c&1];
  }


// Change list style
void FXTable::setTableStyle(FXuint style){
  options=(options&~TABLE_MASK) | (style&TABLE_MASK);
  }


// Get list style
FXuint FXTable::getTableStyle() const { 
  return (options&TABLE_MASK); 
  }


// Change table size to nr x nc
void FXTable::setTableSize(FXint nr,FXint nc){
  register FXint r,c,p,q;
  
  // Any change?
  if(nrows!=nr || ncols!=nc){
    
    // Grow it now
    if((nr*nc)>(nrows*ncols)){ 
      if(!FXRESIZE(&cells,FXTableItem*,nr*nc+1)){ 
        fxerror("%s::setTableSize: out of memory\n",getClassName()); 
        } 
      }
 
    FXTRACE((10,"nr=%d nrows=%d\n",nr,nrows));
    FXTRACE((10,"nc=%d ncols=%d\n",nc,ncols));

    // Shrinking number of columns
    if(nc<ncols){
      
      p=0;
      q=0;
      
      // Shrinking number of rows
      if(nr<nrows){
        for(r=0; r<nr; r++){
          for(c=0; c<nc; c++){
            cells[p++]=cells[q++];
            }
          for(c=nc; c<ncols; c++){
            delete cells[q++];
            }
          }
        for(r=nr; r<nrows; r++){
          for(c=0; c<ncols; c++){
            delete cells[q++];
            }
          }
        FXASSERT(p==nr*nc);
        FXASSERT(q==nrows*ncols);
        }
      
      // Expanding number of rows
      else{
        for(r=0; r<nrows; r++){
          for(c=0; c<nc; c++){
            cells[p++]=cells[q++];
            }
          for(c=nc; c<ncols; c++){
            delete cells[q++];
            }
          }
        for(r=nrows; r<nr; r++){
          for(c=0; c<nc; c++){
            cells[p++]=createItem("",NULL,NULL);
            }
          }
        FXASSERT(p==nr*nc);
        FXASSERT(q==nrows*ncols);
        }
      }
    
    // Expanding number of columns
    else{
      
      p=nr*nc;
      q=nrows*ncols;
      
      // Shrinking number of rows
      if(nr<nrows){
        for(r=nr; r<nrows; r++){
          for(c=0; c<ncols; c++){
            delete cells[--q];
            }
          }
        for(r=0; r<nr; r++){
          for(c=ncols; c<nc; c++){
            cells[--p]=createItem("",NULL,NULL);
            }
          for(c=0; c<ncols; c++){
            cells[--p]=cells[--q];
            }
          }
        FXASSERT(p==0);
        FXASSERT(q==0);
        }
      
      // Expanding number of rows
      else{
        for(r=nrows; r<nr; r++){
          for(c=0; c<nc; c++){
            cells[--p]=createItem("",NULL,NULL);
            }
          }
        for(r=0; r<nrows; r++){
          for(c=ncols; c<nc; c++){
            cells[--p]=createItem("",NULL,NULL);
            }
          for(c=0; c<ncols; c++){
            cells[--p]=cells[--q];
            }
          }
        FXASSERT(p==0);
        FXASSERT(q==0);
        }
      }
              
    // Shrink table 
    if((nr*nc)<(nrows*ncols)){
      if(!FXRESIZE(&cells,FXTableItem*,nr*nc+1)){ 
        fxerror("%s::setTableSize: out of memory\n",getClassName()); 
        }
      }
    
    // Resize row heights
    if(!FXRESIZE(&row_y,FXint,nr+1)){ 
      fxerror("%s::setTableSize: out of memory\n",getClassName()); 
      }
      
    // Resize column widths
    if(!FXRESIZE(&col_x,FXint,nc+1)){ 
      fxerror("%s::setTableSize: out of memory\n",getClassName()); 
      }
    
    // Got new rows
    for(r=nrows; r<nr; r++){
      row_y[r+1]=row_y[r]+defRowHeight;
      }
    
    // Got new columns
    for(c=ncols; c<nc; c++){
      col_x[c+1]=col_x[c]+defColumnWidth;
      }
 
    // Try preserve leading/trailing rows
    if(nr<leading_rows){ leading_rows=nr; trailing_rows=0; }
    else if(nr<leading_rows+trailing_rows){ trailing_rows=nr-leading_rows; }
  
    // Try preserve leading/trailing columns
    if(nc<leading_cols){ leading_cols=nc; trailing_cols=0; }
    else if(nc<leading_cols+trailing_cols){ trailing_cols=nc-leading_cols; }
  
    FXASSERT(leading_rows+trailing_rows<=nr);
    FXASSERT(leading_cols+trailing_cols<=nc);
    
    // Fix up anchor and current
    if(anchor.col>=nc) anchor.col=-1;
    if(anchor.row>=nr) anchor.row=-1;
    if(current.col>=nc) current.col=-1;
    if(current.row>=nr) current.row=-1;
    
    // Update new size
    nrows=nr;
    ncols=nc;
    recalc();
    }
  }


// Insert a row
void FXTable::insertRows(FXint row,FXint nr){
  register FXint r,c,p,q,s;
  if(nr<1) return;
  if(row<0 || row>nrows){ fxerror("%s::insertRows: row out of range\n",getClassName()); }

  // Resize row heights
  if(!FXRESIZE(&row_y,FXint,nrows+nr+1)){ 
    fxerror("%s::insertRows: out of memory\n",getClassName()); 
    }
  
  // Space for nr new rows
  s=nr*defRowHeight;
  
  // Initial size of added rows is the default row height
  for(r=nrows; r>row; r--){ row_y[r+nr]=row_y[r]+s; }
  for(r=row; r<row+nr; r++){ row_y[r+1]=row_y[r]+defRowHeight; }

  // Expand table array
  if(!FXRESIZE(&cells,FXTableItem*,(nrows+nr)*ncols+1)){ 
    fxerror("%s::insertRows: out of memory\n",getClassName()); 
    } 

  // Adjust table
  p=(nrows+nr)*ncols;
  q=nrows*ncols;
  for(r=row; r<nrows; r++){
    for(c=0; c<ncols; c++){
      cells[--p]=cells[--q];
      }
    }
  for(r=0; r<nr; r++){
    for(c=0; c<ncols; c++){
      cells[--p]=createItem("",NULL,NULL);
      }
    }
  
  // Fix up anchor and current
  if(anchor.row>=row) anchor.row+=nr;
  if(current.row>=row) current.row+=nr; 
  
  nrows+=nr;
  
  recalc();
  }


// Insert a column
void FXTable::insertColumns(FXint col,FXint nc){
  register FXint r,c,p,q,s;
  if(nc<1) return;
  if(col<0 || col>ncols){ fxerror("%s::insertColumns: column out of range\n",getClassName()); }

  // Resize column widths
  if(!FXRESIZE(&col_x,FXint,ncols+nc+1)){ 
    fxerror("%s::insertColumns: out of memory\n",getClassName()); 
    }
  
  // Space for nr new rows
  s=nc*defColumnWidth;
  
  // Initial size of added rows is the default row height
  for(c=ncols; c>col; c--){ col_x[c+nc]=col_x[c]+s; }
  for(c=col; c<col+nc; c++){ col_x[c+1]=col_x[c]+defColumnWidth; }


  // Expand table array
  if(!FXRESIZE(&cells,FXTableItem*,nrows*(ncols+nc)+1)){ 
    fxerror("%s::insertColumns: out of memory\n",getClassName()); 
    }

  // Adjust table
  p=nrows*(ncols+nc);
  q=nrows*ncols;
  for(r=0; r<nrows; r++){
    for(c=col; c<ncols; c++){
      cells[--p]=cells[--q];
      }
    for(c=0; c<nc; c++){
      cells[--p]=createItem("",NULL,NULL);
      }
    for(c=0; c<col; c++){
      cells[--p]=cells[--q];
      }
    }
      
  // Fix up anchor and current
  if(anchor.col>=col) anchor.col+=nc;
  if(current.col>=col) current.col+=nc;
  
  ncols+=nc;
  
  recalc();
  }


// Delete a row
void FXTable::deleteRows(FXint row,FXint nr){
  register FXint r,c,s,p,q;
  if(nr<1) return;
  if(row<0 || row+nr>nrows){ fxerror("%s::deleteRows: row out of range\n",getClassName()); }
  
  // Remove row
  s=row_y[row+nr]-row_y[row];
  for(r=row+nr+1; r<nrows; r++){ row_y[r-nr]=row_y[r]-s; }

  // The array is one longer than nrows
  FXRESIZE(&row_y,FXint,nrows-nr+1);               
  
  // Adjust table
  p=row*ncols;
  q=p;
  for(r=0; r<nr; r++){
    for(c=0; c<ncols; c++){
      delete cells[q++];
      }
    }
  for(r=row+nr; r<nrows; r++){
    for(c=0; c<ncols; c++){
      cells[p++]=cells[q++];
      }
    }
  FXASSERT(p==(nrows-nr)*ncols);
  FXASSERT(q==nrows*ncols);
  
  // Shrink table array
  FXRESIZE(&cells,FXTableItem*,(nrows-nr)*ncols+1);
  
  // Fix up anchor and current
  if(anchor.row>=row+nr)  anchor.row-=nr;  else if(anchor.row>=nrows-nr)  anchor.row=nrows-nr-1;
  if(current.row>=row+nr) current.row-=nr; else if(current.row>=nrows-nr) current.row=nrows-nr-1;
  
  // Was a fixed row?
  FXASSERT(0<=row && row+nr<=nrows);
  if(row+nr<=leading_rows) leading_rows-=nr; else if(row<leading_rows) leading_rows=row;
  if(nrows-trailing_rows<=row) trailing_rows-=nr; else if(nrows-trailing_rows<=row+nr) trailing_rows=nrows-row-nr;

  nrows-=nr;
  
  FXASSERT(leading_rows+trailing_rows<=nrows);

  recalc();
  }


// Delete a column
void FXTable::deleteColumns(FXint col,FXint nc){
  register FXint r,c,p,q,s;
  if(nc<1) return;
  if(col<0 || col+nc>ncols){ fxerror("%s::deleteColumns: column out of range\n",getClassName()); }

  // Remove column
  s=col_x[col+nc]-col_x[col];
  for(c=col+nc+1; c<ncols; c++){ col_x[c-nc]=col_x[c]-s; }

  // The array is one longer than ncols
  FXRESIZE(&col_x,FXint,ncols-nc+1);
  
  // Adjust table
  p=0;
  q=0;
  for(r=0; r<nrows; r++){
    for(c=0; c<col; c++){
      cells[p++]=cells[q++];
      }
    for(c=0; c<nc; c++){
      delete cells[q++];
      }
    for(c=col+nc; c<ncols; c++){
      cells[p++]=cells[q++];
      }
    }
  FXASSERT(p==nrows*(ncols-nc));
  FXASSERT(q==nrows*ncols);
    
  // Shrink table array
  FXRESIZE(&cells,FXTableItem*,nrows*(ncols-nc)+1);
  
  // Fix up anchor and current
  if(anchor.col>=col+nc)  anchor.col-=nc;  else if(anchor.col==ncols-nc)  anchor.col=ncols-nc-1;
  if(current.col>=col+nc) current.col-=nc; else if(current.col==ncols-nc) current.col=ncols-nc-1;
  
  // Was a fixed column?
  FXASSERT(0<=col && col+nc<=ncols);
  if(col+nc<=leading_cols) leading_cols-=nc; else if(col<leading_cols) leading_cols=col;
  if(ncols-trailing_cols<=col) trailing_cols-=nc; else if(ncols-trailing_cols<=col+nc) trailing_cols=ncols-col-nc;

  ncols-=nc;
  
  FXASSERT(leading_cols+trailing_cols<=ncols);
  
  recalc();
  }



// Change width of custom column
void FXTable::setColumnWidth(FXint col,FXint cwidth){
  register FXint i,d;
  if(col<0 || col>=ncols){ fxerror("%s::setColumnWidth: column out of range\n",getClassName()); }
  d=cwidth-col_x[col+1]+col_x[col];
  if(d){
    for(i=col+1; i<=ncols; i++) col_x[i]+=d;
    recalc();
    }
  }


// Get width of custom column
FXint FXTable::getColumnWidth(FXint col) const {
  if(col<0 || col>=ncols){ fxerror("%s::getColumnWidth: column out of range\n",getClassName()); }
  return col_x[col+1]-col_x[col];
  }



// Change default column width 
void FXTable::setDefColumnWidth(FXint cwidth){
  if(defColumnWidth!=cwidth){
    defColumnWidth=cwidth;
    recalc();
    }
  }


// Change height of custom row
void FXTable::setRowHeight(FXint row,FXint rheight){
  register FXint i,d;
  if(row<0 || row>=nrows){ fxerror("%s::setRowHeight: row out of range\n",getClassName()); }
  d=rheight-row_y[row+1]+row_y[row];
  if(d){
    for(i=row+1; i<=nrows; i++) row_y[i]+=d;
    recalc();
    }
  }


// Get height of custom row
FXint FXTable::getRowHeight(FXint row) const {
  if(row<0 || row>=nrows){ fxerror("%s::getRowHeight: row out of range\n",getClassName()); }
  return row_y[row+1]-row_y[row];
  }


// Change default row height 
void FXTable::setDefRowHeight(FXint rheight){
  if(defRowHeight!=rheight){
    defRowHeight=rheight;
    recalc();
    }
  }


// Set leading fixed rows
void FXTable::setLeadingRows(FXint leadrows){
  if(leadrows<0 || leadrows>nrows-trailing_rows){ fxerror("%s::setLeadingRows: number out of range\n",getClassName()); }
  if(leadrows!=leading_rows){
    leading_rows=leadrows;
    recalc();
    }
  }


// Set leading fixed columns
void FXTable::setLeadingCols(FXint leadcols){
  if(leadcols<0 || leadcols>ncols-trailing_cols){ fxerror("%s::setLeadingCols: number out of range\n",getClassName()); }
  if(leadcols!=leading_cols){
    leading_cols=leadcols;
    recalc();
    }
  }


// Set trailing fixed rows
void FXTable::setTrailingRows(FXint trailrows){
  if(trailrows<0 || trailrows>nrows-leading_rows){ fxerror("%s::setTrailingRows: number out of range\n",getClassName()); }
  if(trailrows!=trailing_rows){
    trailing_rows=trailrows;
    recalc();
    }
  }


// Set trailing fixed columns
void FXTable::setTrailingCols(FXint trailcols){
  if(trailcols<0 || trailcols>ncols-leading_cols){ fxerror("%s::setTrailingCols: number out of range\n",getClassName()); }
  if(trailcols!=trailing_cols){
    trailing_cols=trailcols;
    recalc();
    }
  }


// Change visible rows
void FXTable::setVisibleRows(FXint nvrows){
  if(nvrows<0) nvrows=0;
  if(visiblerows!=nvrows){
    visiblerows=nvrows;
    recalc();
    }
  }


// Change visible columns
void FXTable::setVisibleCols(FXint nvcols){
  if(nvcols<0) nvcols=0;
  if(visiblecols!=nvcols){
    visiblecols=nvcols;
    recalc();
    }
  }


// Change help text
void FXTable::setHelpText(const FXString& text){
  help=text;
  }


// Save data
void FXTable::save(FXStream& store) const {
  FXScrollArea::save(store);
  store << nrows;
  store << ncols;
  for(int i=0; i<nrows*ncols; i++) store << cells[i];
  store.save(col_x,ncols+1);
  store.save(row_y,nrows+1);
  store << visiblerows;
  store << visiblecols;
  store << margintop;
  store << marginbottom;
  store << marginleft;
  store << marginright;
  store << textColor;
  store << baseColor;
  store << hiliteColor;
  store << shadowColor;
  store << borderColor;
  store << selbackColor;
  store << seltextColor;
  store << gridColor;
  store << cellBackColor[0][0];
  store << cellBackColor[0][1];
  store << cellBackColor[1][0];
  store << cellBackColor[1][1];
  store << font;
  store << help;
  }


// Load data
void FXTable::load(FXStream& store){ 
  FXScrollArea::load(store);
  store >> nrows;
  store >> ncols;
  FXMALLOC(&cells,FXTableItem*,nrows*ncols);
  FXMALLOC(&col_x,FXint,ncols+1);
  FXMALLOC(&row_y,FXint,nrows+1);
  for(int i=0; i<nrows*ncols; i++) store >> cells[i];
  store.load(col_x,ncols+1);
  store.load(row_y,nrows+1);
  store >> visiblerows;
  store >> visiblecols;
  store >> margintop;
  store >> marginbottom;
  store >> marginleft;
  store >> marginright;
  store >> textColor;
  store >> baseColor;
  store >> hiliteColor;
  store >> shadowColor;
  store >> borderColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> gridColor;
  store >> cellBackColor[0][0];
  store >> cellBackColor[0][1];
  store >> cellBackColor[1][0];
  store >> cellBackColor[1][1];
  store >> font;
  store >> help;
  }


// Clean up
FXTable::~FXTable(){
  register FXint i,j;
  if(blinker) getApp()->removeTimeout(blinker);
  for(i=0; i<nrows; i++){
    for(j=0; j<ncols; j++){
      delete cells[i*ncols+j];
      }
    }
  FXFREE(&cells);
  FXFREE(&col_x);
  FXFREE(&row_y);
  cells=(FXTableItem**)-1;
  col_x=(FXint*)-1;
  row_y=(FXint*)-1;
  font=(FXFont*)-1;
  blinker=(FXTimer*)-1;
  }


