/********************************************************************************
*                                                                               *
*                            L i s t   O b j e c t                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXList.cpp,v 1.6 1999/10/26 15:35:53 jeroen Exp $                        *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXObjectList.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollWindow.h"
#include "FXList.h"



/*

  Notes:
  - Draw stuff differently when disabled.
  - Move active item by means of focus navigation, or first letters typed.
  - PageUp/PageDn should also change currentitem.
  - Should support borders in ScrollWindow & derivatives.
  - Should issue callbacks from keyboard also.
  - Need distinguish various callbacks better:
     - Selection changes (all selected/unselected items or just changes???)
     - Changes of currentitem
     - Clicks on currentitem
  - Return key simulates double click.
  - Autoselect mode selects whatever is under the cursor, and gives callback
    on mouse release.
  - Sortfunc's will be hard to serialize, and hard to write w/o secretly #including
    the FXListItem header!
  - Should replace hard-wired SIDE_SPACING etc by flexible padding
  - When in single/browse mode, current item always the one that's selected
*/


#define ICON_SPACING             4    // Spacing between icon and label
#define SIDE_SPACING             6    // Left or right spacing between items
#define LINE_SPACING             4    // Line spacing between items

#define SELECT_MASK (LIST_SINGLESELECT|LIST_BROWSESELECT)
#define LIST_MASK   (LIST_SINGLESELECT|LIST_BROWSESELECT|LIST_AUTOSELECT)



/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXListItem,FXObject,NULL,0)


// Draw item
void FXListItem::draw(const FXList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h){
  FXint ih=0,th=0;
  if(icon) ih=icon->getHeight();
  if(!label.empty()) th=list->getFont()->getFontHeight();
  if(isSelected())
    dc.setForeground(list->getSelBackColor());
  else
    dc.setForeground(list->getBackColor());
  dc.fillRectangle(x,y,w,h);
  if(hasFocus()){
    drawFocus(list,dc,x,y,w,h);
    }
  x+=SIDE_SPACING/2;
  if(icon){
    dc.drawIcon(icon,x,y+(h-ih)/2);
    x+=ICON_SPACING+icon->getWidth();
    }
  if(!label.empty()){
    dc.setTextFont(list->getFont());
    if(!isEnabled())
      dc.setForeground(makeShadowColor(list->getBackColor()));
    else if(state&SELECTED)
      dc.setForeground(list->getSelTextColor());
    else
      dc.setForeground(list->getTextColor());
    dc.drawText(x,y+(h-th)/2+list->getFont()->getFontAscent(),label.text(),label.length());
    }
  }

  
// Draw dotted rectangle for focus
void FXListItem::drawFocus(const FXList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setFillStyle(FILL_OPAQUESTIPPLED);
  dc.setStipple(STIPPLE_GRAY);
  dc.setForeground(list->getTextColor());
  dc.setBackground(list->getBackColor());
  dc.drawRectangle(x+1,y+1,w-3,h-3);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }


// Create icon
void FXListItem::create(){ if(icon) icon->create(); }


// Destroy icon
void FXListItem::destroy(){ if(icon) icon->destroy(); }


// Detach from icon resource
void FXListItem::detach(){ if(icon) icon->detach(); }


// Get width of item
FXint FXListItem::getWidth(const FXList* list) const {
  FXint w=0;
  if(!label.empty()) w+=list->getFont()->getTextWidth(label.text(),label.length());
  if(icon) w+=icon->getWidth();
  if(!label.empty() && icon) w+=ICON_SPACING;
  return SIDE_SPACING+w;
  }


// Get height of item
FXint FXListItem::getHeight(const FXList* list) const {
  FXint th,ih;
  th=ih=0;
  if(!label.empty()) th=list->getFont()->getFontHeight();
  if(icon) ih=icon->getHeight();
  return LINE_SPACING+FXMAX(th,ih);
  }


// Save data
void FXListItem::save(FXStream& store) const {
  FXObject::save(store);
  store << label;
  store << icon;
  store << state;
  }


// Load data
void FXListItem::load(FXStream& store){ 
  FXObject::load(store);
  store >> label;
  store >> icon;
  store >> state;
  }


/*******************************************************************************/


// Map
FXDEFMAP(FXList) FXListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXList::onMotion),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXList::onAutoScroll),
  FXMAPFUNC(SEL_UNGRABBED,0,FXList::onUngrabbed),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXList::onLeftBtnRelease),
  FXMAPFUNC(SEL_KEYPRESS,0,FXList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXList::onKeyRelease),
  FXMAPFUNC(SEL_ACTIVATE,0,FXList::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXList::onDeactivate),
  FXMAPFUNC(SEL_FOCUSIN,0,FXList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXList::onFocusOut),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXList::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXList::onSelectionGained),
  FXMAPFUNC(SEL_CHANGED,0,FXList::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXList::onTripleClicked),
  FXMAPFUNC(SEL_COMMAND,0,FXList::onCommand),
  };


// Object implementation
FXIMPLEMENT(FXList,FXScrollArea,FXListMap,ARRAYNUMBER(FXListMap))

  
// List
FXList::FXList(){
  flags|=FLAG_ENABLED;
  items=NULL;
  nitems=0;
  anchor=-1;
  current=-1;
  extent=-1;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  itemWidth=1;
  itemHeight=1;
  visible=0;
  font=(FXFont*)-1;
  }

  
// List
FXList::FXList(FXComposite *p,FXint nvis,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  items=NULL;
  nitems=0;
  anchor=-1;
  current=-1;
  extent=-1;
  font=getApp()->normalFont;
  sortfunc=NULL;
  textColor=getApp()->foreColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  itemWidth=1;
  itemHeight=1;
  visible=FXMAX(nvis,0);
  }


// Create window
void FXList::create(){
  register FXint i;
  FXScrollArea::create();
  for(i=0; i<nitems; i++){items[i]->create();}
  font->create();
  }


// Detach window
void FXList::detach(){
  register FXint i;
  FXScrollArea::detach();
  for(i=0; i<nitems; i++){items[i]->detach();}
  font->detach();
  }


// Can have focus
FXbool FXList::canFocus() const { return TRUE; }


// Get default width
FXint FXList::getDefaultWidth(){
  return FXScrollArea::getDefaultWidth();
  }


// Get default height
FXint FXList::getDefaultHeight(){
  if(visible){
    if(flags&FLAG_RECALC) recompute();
    return visible*itemHeight;
    }
  return FXScrollArea::getDefaultHeight();
  }


// Propagate size change
void FXList::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  }


// List is multiple of nitems
void FXList::setNumVisible(FXint nvis){
  if(nvis<0) nvis=0;
  if(visible!=nvis){
    visible=nvis;
    recalc();
    }
  }


// Recompute interior 
void FXList::recompute(){
  register FXint w,h,i;
  itemWidth=1;
  itemHeight=1;
  for(i=0; i<nitems; i++){
    w=items[i]->getWidth(this);
    h=items[i]->getHeight(this);
    if(w>itemWidth) itemWidth=w;
    if(h>itemHeight) itemHeight=h;
    }
  flags&=~FLAG_RECALC;
  }


// Determine content width of list
FXint FXList::getContentWidth(){ 
  if(flags&FLAG_RECALC) recompute();
  return itemWidth;
  }


// Determine content height of list
FXint FXList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return nitems*itemHeight; 
  }


// Recalculate layout determines item locations and sizes
void FXList::layout(){
  
  // Force repaint if content changed
  //if(flags&FLAG_RECALC) update();
  
  // Calculate contents 
  FXScrollArea::layout();
  
  // Determine line size for scroll bars
  vertical->setLine(itemHeight);
  horizontal->setLine(1);
  
  update();
  
  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Change item text
void FXList::setItemText(FXint index,const FXString& text){
  if(index<0 || nitems<=index){ fxerror("%s::setItemText: index out of range.\n",getClassName()); } 
  items[index]->label=text;
  recalc();
  }


// Get item text
FXString FXList::getItemText(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemText: index out of range.\n",getClassName()); } 
  return items[index]->label;
  }


// Set item icon
void FXList::setItemIcon(FXint index,FXIcon* icon){
  if(index<0 || nitems<=index){ fxerror("%s::setItemIcon: index out of range.\n",getClassName()); } 
  items[index]->icon=icon;
  recalc();
  }


// Get item icon
FXIcon* FXList::getItemIcon(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemIcon: index out of range.\n",getClassName()); } 
  return items[index]->icon;
  }


// Set item data
void FXList::setItemData(FXint index,void* ptr){
  if(index<0 || nitems<=index){ fxerror("%s::setItemData: index out of range.\n",getClassName()); } 
  items[index]->data=ptr;
  }


// Get item data
void* FXList::getItemData(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemData: index out of range.\n",getClassName()); } 
  return items[index]->data;
  }


// True if item is selected
FXbool FXList::isItemSelected(FXint index) const { 
  if(index<0 || nitems<=index){ fxerror("%s::isItemSelected: index out of range.\n",getClassName()); } 
  return (items[index]->state&FXListItem::SELECTED)!=0; 
  }


// True if item is current
FXbool FXList::isItemCurrent(FXint index) const { 
  if(index<0 || nitems<=index){ fxerror("%s::isItemCurrent: index out of range.\n",getClassName()); } 
  return index==current; 
  }


// True if item is enabled
FXbool FXList::isItemEnabled(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::isItemEnabled: index out of range.\n",getClassName()); } 
  return (items[index]->state&FXListItem::DISABLED)==0; 
  }


// True if item (partially) visible
FXbool FXList::isItemVisible(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::isItemVisible: index out of range.\n",getClassName()); } 
  return (0 < (pos_y+index*itemHeight+itemHeight)) && ((pos_y+index*itemHeight) < viewport_h); 
  }


// Make item fully visible
void FXList::makeItemVisible(FXint index){ 
  if(0<=index && index<nitems){
    if((pos_y+index*itemHeight) < 0){
      setPosition(pos_x,-index*itemHeight);
      }
    else if(viewport_h<=(pos_y+index*itemHeight+itemHeight)){
      setPosition(pos_x,viewport_h-index*itemHeight-itemHeight);
      }
    }
  }


// Get item at position x,y
FXint FXList::getItemAt(FXint,FXint y) const {
  FXint index=(y-pos_y)/itemHeight;
  if(index<0 || index>=nitems) index=-1;
  return index;
  }


// Get item at position x,y
FXint FXList::findItem(const FXString& text,FXuint len) const {
  FXint index=nitems-1;
  for(index=nitems-1; 0<=index; index--){
    if(equal(text,items[index]->label,len)) return index;
    }
  return index;
  }


// Repaint
void FXList::updateItem(FXint index){
  if(0<=index && index<nitems){
    update(0,pos_y+index*itemHeight,viewport_w,itemHeight);
    }
  }


// Enable one item
FXbool FXList::enableItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::enableItem: index out of range.\n",getClassName()); } 
  if(items[index]->state&FXListItem::DISABLED){
    items[index]->state&=~FXListItem::DISABLED;
    updateItem(index);
    return TRUE;
    }
  return FALSE;
  }


// Disable one item
FXbool FXList::disableItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::disableItem: index out of range.\n",getClassName()); } 
  if(!(items[index]->state&FXListItem::DISABLED)){
    items[index]->state|=FXListItem::DISABLED;
    updateItem(index);
    return TRUE;
    }
  return FALSE;
  }


// Select one item
FXbool FXList::selectItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::selectItem: index out of range.\n",getClassName()); } 
  if(!(items[index]->state&FXListItem::SELECTED)){
    switch(options&SELECT_MASK){
      case LIST_SINGLESELECT:
      case LIST_BROWSESELECT:
        killSelection();
        items[index]->state|=FXListItem::SELECTED;
        updateItem(index);
        break;
      case LIST_EXTENDEDSELECT:
      case LIST_MULTIPLESELECT:
        items[index]->state|=FXListItem::SELECTED;
        updateItem(index);
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// Deselect one item
FXbool FXList::deselectItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::deselectItem: index out of range.\n",getClassName()); } 
  if(items[index]->state&FXListItem::SELECTED){
    switch(options&SELECT_MASK){
      case LIST_EXTENDEDSELECT:
      case LIST_MULTIPLESELECT:
      case LIST_SINGLESELECT:
        items[index]->state&=~FXListItem::SELECTED;
        updateItem(index);
      case LIST_BROWSESELECT:
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// Toggle one item
FXbool FXList::toggleItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::toggleItem: index out of range.\n",getClassName()); } 
  switch(options&SELECT_MASK){
    case LIST_BROWSESELECT:
      if(!(items[index]->state&FXListItem::SELECTED)){
        killSelection();
        items[index]->state|=FXListItem::SELECTED;
        updateItem(index);
        }
      break;
    case LIST_SINGLESELECT:
      if(!(items[index]->state&FXListItem::SELECTED)){
        killSelection();
        }
      items[index]->state^=FXListItem::SELECTED;
      updateItem(index);
      break;
    case LIST_EXTENDEDSELECT:
    case LIST_MULTIPLESELECT:
      items[index]->state^=FXListItem::SELECTED;
      updateItem(index);
      break;
    }
  return TRUE;
  }


// Gained focus
long FXList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(0<=current){
    FXASSERT(current<nitems);
    items[current]->state|=FXListItem::FOCUS;
    updateItem(current);
    }
  return 1;
  }


// Lost focus
long FXList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(0<=current){
    FXASSERT(current<nitems);
    items[current]->state&=~FXListItem::FOCUS;
    updateItem(current);
    }
  return 1;
  }


// We have the selection
long FXList::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  if(FXScrollArea::onSelectionGained(sender,sel,ptr)) return 1;
  return 1;
  }


// We lost the selection
long FXList::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  if(FXScrollArea::onSelectionLost(sender,sel,ptr)) return 1;
  killSelection();
  return 1;
  }


// Draw item list
long FXList::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXDCWindow dc(this,event);
  FXint indexlo=(event->rect.y-pos_y)/itemHeight;
  FXint indexhi=(event->rect.y+event->rect.h-pos_y)/itemHeight;
  FXint i,y;
  if(indexlo<0) indexlo=0;
  if(indexhi>=nitems) indexhi=nitems-1;
  y=pos_y+indexlo*itemHeight;
  for(i=indexlo; i<=indexhi; i++){
    items[i]->draw(this,dc,pos_x,y,content_w,itemHeight);
    y+=itemHeight;
    }
  if(y<event->rect.y+event->rect.h){
    dc.setForeground(backColor);
    dc.fillRectangle(event->rect.x,y,event->rect.w,event->rect.y+event->rect.h-y);
    }
  return 1;
  }


// Key Press
long FXList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint index;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Shift_L:
    case KEY_Shift_R:
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      index=current;
      switch(event->code){
        case KEY_Up:
          index--;
          break;
        case KEY_Down: 
          index++;
          break;
        case KEY_Home: 
          index=0; 
          break;
        case KEY_End:  
          index=nitems-1; 
          break;
        }
      if(index<0) index=0;
      if(index>=nitems) index=nitems-1;
      makeItemVisible(index);
      handle(this,MKUINT(0,SEL_CHANGED),(void*)index);
      if(0<=index && index!=current){
        if(event->state&(SHIFTMASK|CONTROLMASK)){
          if((options&SELECT_MASK)==LIST_EXTENDEDSELECT){
            extendSelection(index);
            }
          }
        }
      if(event->click_count==1){
        if((options&SELECT_MASK)==LIST_BROWSESELECT){
          handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
          }
        }
      flags&=~FLAG_UPDATE;
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      return 1;
    }
  return 0;
  }


// Key Release 
long FXList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Page_Up:
    case KEY_Page_Down:
      return 1;
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      if(event->click_count==1){
        if((options&SELECT_MASK)==LIST_BROWSESELECT){
          handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
          }
        }
      flags|=FLAG_UPDATE;
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      flags|=FLAG_UPDATE;
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      return 1;
    }
  return 0;
  }


// Automatic scroll
long FXList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint index;
  FXint xx,yy;
    
  // Scroll the window
  FXScrollArea::onAutoScroll(sender,sel,ptr);
  
  // Validated position
  xx=event->win_x; if(xx<0) xx=0; else if(xx>=viewport_w) xx=viewport_w-1;
  yy=event->win_y; if(yy<0) yy=0; else if(yy>=viewport_h) yy=viewport_h-1;
  
  // Find item
  index=getItemAt(xx,yy);
  
  // Got item and different from last time
  if(0<=index && index!=current){
    
    // Make it the current item
    handle(this,MKUINT(0,SEL_CHANGED),(void*)index);
 
    // Extend the selection
    if((options&SELECT_MASK)==LIST_EXTENDEDSELECT){
      extendSelection(index);
      }
    return 1;
    }

  return 0;
  }


// Mouse moved
long FXList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint index;
  if((flags&FLAG_PRESSED) || (options&LIST_AUTOSELECT)){
    
    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;
    
    // Find item
    index=getItemAt(event->win_x,event->win_y);
    
    // Got an item different from before
    if(0<=index && index!=current){
      
      // Make it the current item
      handle(this,MKUINT(0,SEL_CHANGED),(void*)index);
      
      // Extend the selection
      if((options&SELECT_MASK)==LIST_EXTENDEDSELECT){
        extendSelection(index);
        }
      return 1;
      }
    }
  return 0;
  }
 

// Pressed button
long FXList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint index;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(options&LIST_AUTOSELECT) return 1;
    index=getItemAt(event->win_x,event->win_y);
    makeItemVisible(index);
    handle(this,MKUINT(0,SEL_CHANGED),(void*)index);
    handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    flags|=FLAG_PRESSED;
    flags&=~FLAG_UPDATE;
    return 1;
    }
  return 0;
  }


// Released button
long FXList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    stopAutoScroll();
    flags&=~FLAG_PRESSED;
    flags|=FLAG_UPDATE;
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
    makeItemVisible(current);
    return 1;
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags&=~FLAG_CHANGED;
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }


// Current item changed
long FXList::onChanged(FXObject*,FXSelector,void* ptr){
  setCurrentItem((FXint)(long)ptr);
  if(target) target->handle(this,MKUINT(message,SEL_CHANGED),ptr);
  return 1;
  }


// Command message
long FXList::onCommand(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
  }


// Clicked in list
long FXList::onClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_CLICKED),ptr);
  }


// Double clicked in list; ptr may or may not point to an item
long FXList::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr);
  }


// Triple clicked in list; ptr may or may not point to an item
long FXList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Button or Key activate
long FXList::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(0<=current && items[current]->isEnabled()){
    switch(options&SELECT_MASK){
      case LIST_EXTENDEDSELECT:
        if(event->state&SHIFTMASK){
          if(0<=anchor){
            selectItem(anchor);
            extendSelection(current);
            }
          else{
            selectItem(current);
            setAnchorItem(current);
            }
          }
        else if(event->state&CONTROLMASK){
          toggleItem(current);
          setAnchorItem(current);
          }
        else{
          killSelection();
          selectItem(current);
          setAnchorItem(current);
          }
        break;
      case LIST_SINGLESELECT:
      case LIST_MULTIPLESELECT:
        toggleItem(current);
        setAnchorItem(current);
        break;
      case LIST_BROWSESELECT:
        break;
      }
    }
  else{
    switch(options&SELECT_MASK){
      case LIST_EXTENDEDSELECT:
      case LIST_SINGLESELECT:
        if(!(event->state&(SHIFTMASK|CONTROLMASK))) killSelection();
        break;
      case LIST_MULTIPLESELECT:
      case LIST_BROWSESELECT:
        break;
      }
    }
  return 1;
  }


// Button or Key deactivate
long FXList::onDeactivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  setAnchorItem(current);
  if(event->click_count==1){
    handle(this,MKUINT(0,SEL_CLICKED),(void*)current);
    }
  else if(event->click_count==2){
    handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)current);
    }
  else if(event->click_count==3){
    handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)current);
    }
  if(0<=current && items[current]->isEnabled()){
    handle(this,MKUINT(0,SEL_COMMAND),(void*)current);
    }
  return 1;
  }


// Mark and select
FXbool FXList::mark(FXint fm,FXint to,FXuint sel){
  register FXbool changes=FALSE;
  register FXint i;
  if(sel){
    for(i=fm; i<=to; i++){
      if(!(items[i]->state&FXListItem::SELECTED)){
        items[i]->state&=~FXListItem::MARKED;
        if(!(items[i]->state&FXListItem::DISABLED)){
          items[i]->state|=FXListItem::SELECTED;
          updateItem(i);
          changes=TRUE;
          }
        }
      else{
        items[i]->state|=FXListItem::MARKED;
        }
      }
    }
  else{
    for(i=fm; i<=to; i++){
      if(items[i]->state&FXListItem::SELECTED){
        items[i]->state|=FXListItem::MARKED;
        if(!(items[i]->state&FXListItem::DISABLED)){
          items[i]->state&=~FXListItem::SELECTED;
          updateItem(i);
          changes=TRUE;
          }
        }
      else{
        items[i]->state&=~FXListItem::MARKED;
        }
      }
    }
  return changes;
  }


// Restore to mark
FXbool FXList::restore(FXint fm,FXint to){
  register FXbool changes=FALSE;
  register FXint i;
  for(i=fm; i<=to; i++){
    if(items[i]->state&FXListItem::MARKED){
      if(!(items[i]->state&FXListItem::SELECTED)){
        items[i]->state|=FXListItem::SELECTED;
        updateItem(i);
        changes=TRUE;
        }
      }
    else{
      if(items[i]->state&FXListItem::SELECTED){
        items[i]->state&=~FXListItem::SELECTED;
        updateItem(i);
        changes=TRUE;
        }
      }
    }
  return changes;
  }


// Extend selection
FXbool FXList::extendSelection(FXint index){
  register FXbool changes=FALSE;
  register FXuint sel;
  if(0<=index && 0<=anchor){
    sel=items[anchor]->state&FXListItem::SELECTED;
    if(extent<anchor){
      if(index<extent){                     // index < extent < anchor
        changes|=mark(index,extent-1,sel);
        }
      else if(anchor<index){                // extent < anchor < index
        changes|=restore(extent,anchor-1);
        changes|=mark(anchor+1,index,sel);
        }
      else{                                 // extent <= index <= anchor
        changes|=restore(extent,index-1);
        }
      }
    else{
      if(extent<index){                     // anchor <= extent < index
        changes|=mark(extent+1,index,sel);
        }
      else if(index<anchor){                // index < anchor <= extent
        changes|=restore(anchor+1,extent);
        changes|=mark(index,anchor-1,sel);
        }
      else{                                 // anchor <= index <= extent
        changes|=restore(index+1,extent);
        }
      }
    extent=index;
    }
  return changes;
  }


// Kill selection
FXbool FXList::killSelection(){
  register FXbool changes=FALSE;
  register FXint i;
  for(i=0; i<nitems; i++){
    if(items[i]->state&FXListItem::SELECTED){
      items[i]->state&=~FXListItem::SELECTED;
      updateItem(i);
      changes=TRUE;
      }
    }
  extent=anchor;
  return changes;
  }


// Sort the items based on the sort function
void FXList::sortItems(){
  register FXListItem *v;
  register FXint i,j,h;
  register FXbool exch=FALSE;
  if(sortfunc){
    for(h=1; h<=nitems/9; h=3*h+1);
    if(hasFocus() && 0<=current) items[current]->state&=~FXListItem::FOCUS;
    for(; h>0; h/=3){
      for(i=h+1;i<=nitems;i++){
        v=items[i-1]; 
        j=i;
        while(j>h && sortfunc(items[j-h-1],v)){
          items[j-1]=items[j-h-1];
          exch=TRUE;
          j-=h;
          }
        items[j-1]=v;
        }
      }
    if(hasFocus() && 0<=current) items[current]->state|=FXListItem::FOCUS;
    if(exch) update();
    }
  }


// Set current item
void FXList::setCurrentItem(FXint index){
  if(index<-1 || nitems<=index){ fxerror("%s::setCurrentItem: index out of range.\n",getClassName()); } 
  if(index!=current){
    
    // Deactivate old item
    if(0<=current){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        items[current]->state&=~FXListItem::FOCUS;
        updateItem(current);
        }
      }

    current=index;
    
    // Activate new item
    if(0<=current){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        items[current]->state|=FXListItem::FOCUS;
        updateItem(current);
        }
      }
    }

  // In browse select mode, select this item
  if((options&SELECT_MASK)==LIST_BROWSESELECT){
    if(0<=current && !isItemSelected(current)){
      killSelection();
      selectItem(current);
      }
    }
  }


// Set anchor item
void FXList::setAnchorItem(FXint index){
  if(index<-1 || nitems<=index){ fxerror("%s::setAnchorItem: index out of range.\n",getClassName()); } 
  anchor=index;
  extent=index;
  }


// Create custom item
FXListItem *FXList::createItem(const FXString& text,FXIcon* icon,void* ptr){
  return new FXListItem(text,icon,ptr);
  }


// Retrieve item
FXListItem *FXList::retrieveItem(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::retrieveItem: index out of range.\n",getClassName()); } 
  return items[index];
  }


// Insert item
FXint FXList::insertItem(FXint index,FXListItem* item){
  if(!item){ fxerror("%s::insertItem: item is NULL.\n",getClassName()); } 
  if(index<0 || nitems<index){ fxerror("%s::insertItem: index out of range.\n",getClassName()); } 
  FXRESIZE(&items,FXListItem*,nitems+1);
  memmove(&items[index+1],&items[index],sizeof(FXListItem*)*(nitems-index));
  items[index]=item;
  if(anchor>=index)  anchor++;
  if(extent>=index)  extent++;
  if(current>=index) current++;
  nitems++;
  recalc();
  return index;
  }


// Insert item
FXint FXList::insertItem(FXint index,const FXString& text,FXIcon *icon,void* ptr){
  return insertItem(index,createItem(text,icon,ptr));
  }


// Replace item with another
FXint FXList::replaceItem(FXint index,FXListItem* item){
  if(!item){ fxerror("%s::replaceItem: item is NULL.\n",getClassName()); } 
  if(index<0 || nitems<=index){ fxerror("%s::replaceItem: index out of range.\n",getClassName()); } 
  item->state=items[index]->state;    // Copy the state over
  delete items[index];
  items[index]=item;
  recalc();
  return index;
  }


// Replace item with another
FXint FXList::replaceItem(FXint index,const FXString& text,FXIcon *icon,void* ptr){
  return replaceItem(index,createItem(text,icon,ptr));
  }


// Append item at end
FXint FXList::appendItem(FXListItem* item){
  if(!item){ fxerror("%s::appendItem: item is NULL.\n",getClassName()); } 
  FXRESIZE(&items,FXListItem*,nitems+1);
  items[nitems]=item;
  nitems++;
  recalc();
  return nitems-1;
  }


// Append item at end
FXint FXList::appendItem(const FXString& text,FXIcon *icon,void* ptr){
  return appendItem(createItem(text,icon,ptr));
  }


// Remove node from list
void FXList::removeItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::removeItem: index out of range.\n",getClassName()); } 
  nitems--;
  delete items[index];
  memmove(&items[index],&items[index+1],sizeof(FXListItem*)*(nitems-index));
  if(anchor>index)  anchor--;  else if(anchor==index)  anchor=-1;
  if(extent>index)  extent--;  else if(extent==index)  extent=-1;
  if(current>index) current--; else if(current==index) current=-1;
  recalc();
  }


// Remove all items
void FXList::clearItems(){
  FXint i;
  for(i=0; i<nitems; i++){delete items[i];}
  FXFREE(&items);
  nitems=0;
  current=-1;
  anchor=-1;
  extent=-1;
  recalc();
  }


// Change the font
void FXList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Set text color
void FXList::setTextColor(FXColor clr){
  textColor=clr;
  update();
  }


// Set select background color
void FXList::setSelBackColor(FXColor clr){
  selbackColor=clr;
  update();
  }


// Set selected text color
void FXList::setSelTextColor(FXColor clr){
  seltextColor=clr;
  update();
  }


// Change list style
void FXList::setListStyle(FXuint style){
  options=(options&~LIST_MASK) | (style&LIST_MASK);
  }


// Get list style
FXuint FXList::getListStyle() const { 
  return (options&LIST_MASK); 
  }


// Change help text
void FXList::setHelpText(const FXString& text){
  help=text;
  }


// Save data
void FXList::save(FXStream& store) const {
  register FXint i;
  FXScrollArea::save(store);
  store << nitems;
  for(i=0; i<nitems; i++){store<<items[i];}
  store << anchor;
  store << current;
  store << extent;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << itemWidth;
  store << itemHeight;
  store << visible;
  store << font;
  store << help;
  }


// Load data
void FXList::load(FXStream& store){ 
  register FXint i;
  FXScrollArea::load(store);
  store >> nitems;
  FXRESIZE(&items,FXListItem*,nitems);
  for(i=0; i<nitems; i++){store>>items[i];}
  store >> anchor;
  store >> current;
  store >> extent;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> itemWidth;
  store >> itemHeight;
  store >> visible;
  store >> font;
  store >> help;
  }


// Clean up
FXList::~FXList(){
  clearItems();
  items=(FXListItem**)-1;
  font=(FXFont*)-1;
  }


