/********************************************************************************
*                                                                               *
*      D e v i c e   C o n t e x t   D r a w i n g   o n   a   W i n d o w      *
*                                                                               *
*********************************************************************************
* Copyright (C) 1999 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXDCWindow.cpp,v 1.6 1999/11/04 08:22:00 jeroen Exp $                    *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXFont.h"
#include "FXCursor.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXBitmap.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXRootWindow.h"
#include "FXShell.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXProfiler.h"
  
/*
  Notes:
  
  - Associate a DC with a surface before you begin using it:
  
     long SomeWidget::onPaint(FXObject*,FXSelector,void* ptr){
       FXDCWindow dc(this,ptr);
       dc.drawLine(...);
       ... jadajadajada ...
       return 1;
       }

    The association is automatically broken when you go out of scope; the
    destructor of the FXDCWindow does this for you.

  - Optimizations: only perform style/attribute changes just before an actual
    drawing command takes place:- X-Windows apparently already does this;
    MS-Windows also?
      
  - We assume the following initial state:
      
    BLIT Function:        BLT_SRC
    Foreground:           black (0)
    Background:           white (1)
    Line Width:           0 (meaning thinnest/fastest, no guaranteed pixelation)
    Cap Style:            CAP_BUTT
    Join Style:           JOIN_MITER
    Line Style:           LINE_SOLID
    Fill Style:           FILL_SOLID
    Fill Rule:            RULE_EVEN_ODD
    Font:                 None
    Other Paremeters:     To Be Determined
    
  - Under X-Windows, end() will restore the GC to the state above; flags 
    keeps track of which changes have been made to minimize the necessary 
    updating.
    
  - Under X, graphics_exposures should be OFF:- at least some SGI IRIX machines
    have broken implementations of graphics_exposures.
*/


/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXDCWindow,FXDC,NULL,0)


/********************************************************************************
*                                    X-Windows                                  *
********************************************************************************/

#ifndef FX_NATIVE_WIN32


// Construct for expose event painting
FXDCWindow::FXDCWindow(FXDrawable* drawable,FXEvent* event):FXDC(drawable->getApp()){
  begin(drawable);
  if(event){
    clip=event->rect;
    XSetClipRectangles(getApp()->display,gc,0,0,(XRectangle*)&clip,1,Unsorted);
    flags|=GCClipMask;
    }
  else{
    clip.x=0;
    clip.y=0;
    clip.w=drawable->getWidth();
    clip.h=drawable->getHeight();
    }
  }


// Construct for normal painting
FXDCWindow::FXDCWindow(FXDrawable* drawable):FXDC(drawable->getApp()){
  begin(drawable);
  clip.x=0;
  clip.y=0;
  clip.w=drawable->getWidth();
  clip.h=drawable->getHeight();
  }

  
// Destruct
FXDCWindow::~FXDCWindow(){
  end();
  }


// Begin locks in a drawable surface
void FXDCWindow::begin(FXDrawable *drawable){
  if(surface){ fxerror("FXDCWindow::begin: DC already connected to drawable.\n"); }
  if(!drawable){ fxerror("FXDCWindow::begin: NULL drawable.\n"); }
  if(!drawable->id()){ fxerror("FXDCWindow::begin: drawable not created yet.\n"); }
  surface=drawable;
  visual=drawable->getVisual();
  flags=0;
  gc=visual->gc;
  }


// End unlock the drawable surface; restore it
void FXDCWindow::end(){
  XGCValues gcv;
  if(!surface){ fxerror("FXDCWindow::end: DC not connected to drawable.\n"); }
  if(flags){
    if(flags&GCFunction) gcv.function=BLT_SRC;
    if(flags&GCForeground) gcv.foreground=BlackPixel(getApp()->display,DefaultScreen(getApp()->display));
    if(flags&GCBackground) gcv.background=WhitePixel(getApp()->display,DefaultScreen(getApp()->display));
    if(flags&GCLineWidth) gcv.line_width=0;
    if(flags&GCCapStyle) gcv.cap_style=CAP_BUTT;
    if(flags&GCJoinStyle) gcv.join_style=JOIN_MITER;
    if(flags&GCLineStyle) gcv.line_style=LINE_SOLID;
    if(flags&GCFillStyle) gcv.fill_style=FILL_SOLID;
    if(flags&GCStipple) gcv.stipple=getApp()->stipples[STIPPLE_WHITE];    // Needed for IRIX6.4 bug workaround!
    if(flags&GCFillRule) gcv.fill_rule=RULE_EVEN_ODD;
    if(flags&GCFont) gcv.font=getApp()->normalFont->id();
    if(flags&GCClipMask) gcv.clip_mask=None;
    if(flags&GCClipXOrigin) gcv.clip_x_origin=0;
    if(flags&GCClipYOrigin) gcv.clip_y_origin=0;
    if(flags&GCDashOffset) gcv.dash_offset=0;
    if(flags&GCDashList) gcv.dashes=4;
    if(flags&GCTileStipXOrigin) gcv.ts_x_origin=0;
    if(flags&GCTileStipYOrigin) gcv.ts_y_origin=0;
    if(flags&GCGraphicsExposures) gcv.graphics_exposures=True;
    if(flags&GCTile) gcv.tile=None;
    if(flags&GCSubwindowMode) gcv.subwindow_mode=ClipByChildren;
    XChangeGC(getApp()->display,gc,flags,&gcv);
    flags=0;
    }
  surface=NULL;
  }


void FXDCWindow::drawPoint(FXint x,FXint y){ 
  if(!surface){ fxerror("FXDCWindow::drawPoint: DC not connected to drawable.\n"); }
  XDrawPoint(getApp()->display,surface->id(),gc,x,y); 
  }


void FXDCWindow::drawPoints(const FXPoint* points,FXuint npoints){ 
  if(!surface){ fxerror("FXDCWindow::drawPoints: DC not connected to drawable.\n"); }
  XDrawPoints(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,CoordModeOrigin); 
  }


void FXDCWindow::drawPointsRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::drawPointsRel: DC not connected to drawable.\n"); }
  XDrawPoints(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,CoordModePrevious); 
  }


void FXDCWindow::drawLine(FXint x1,FXint y1,FXint x2,FXint y2){
  if(!surface){ fxerror("FXDCWindow::drawLine: DC not connected to drawable.\n"); }
  XDrawLine(getApp()->display,surface->id(),gc,x1,y1,x2,y2);
  }


void FXDCWindow::drawLines(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::drawLines: DC not connected to drawable.\n"); }
  XDrawLines(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,CoordModeOrigin); 
  }


void FXDCWindow::drawLinesRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::drawLinesRel: DC not connected to drawable.\n"); }
  XDrawLines(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,CoordModePrevious); 
  }


void FXDCWindow::drawLineSegments(const FXSegment* segments,FXuint nsegments){
  if(!surface){ fxerror("FXDCWindow::drawLineSegments: DC not connected to drawable.\n"); }
  XDrawSegments(getApp()->display,surface->id(),gc,(XSegment*)segments,nsegments); 
  }
 

void FXDCWindow::drawRectangle(FXint x,FXint y,FXint w,FXint h){
  if(!surface){ fxerror("FXDCWindow::drawRectangle: DC not connected to drawable.\n"); }
  XDrawRectangle(getApp()->display,surface->id(),gc,x,y,w,h);
  }


void FXDCWindow::drawRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  if(!surface){ fxerror("FXDCWindow::drawRectangles: DC not connected to drawable.\n"); }
  XDrawRectangles(getApp()->display,surface->id(),gc,(XRectangle*)rectangles,nrectangles);
  }


void FXDCWindow::drawArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  if(!surface){ fxerror("FXDCWindow::drawArc: DC not connected to drawable.\n"); }
  XDrawArc(getApp()->display,surface->id(),gc,x,y,w,h,ang1,ang2);
  }


void FXDCWindow::drawArcs(const FXArc* arcs,FXuint narcs){
  if(!surface){ fxerror("FXDCWindow::drawArcs: DC not connected to drawable.\n"); }
  XDrawArcs(getApp()->display,surface->id(),gc,(XArc*)arcs,narcs);
  }


void FXDCWindow::fillRectangle(FXint x,FXint y,FXint w,FXint h){
  if(!surface){ fxerror("FXDCWindow::fillRectangle: DC not connected to drawable.\n"); }
  XFillRectangle(getApp()->display,surface->id(),gc,x,y,w,h);
  }


void FXDCWindow::fillRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  if(!surface){ fxerror("FXDCWindow::fillRectangles: DC not connected to drawable.\n"); }
  XFillRectangles(getApp()->display,surface->id(),gc,(XRectangle*)rectangles,nrectangles);
  }


void FXDCWindow::fillArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  if(!surface){ fxerror("FXDCWindow::fillArc: DC not connected to drawable.\n"); }
  XFillArc(getApp()->display,surface->id(),gc,x,y,w,h,ang1,ang2);
  }


void FXDCWindow::fillArcs(const FXArc* arcs,FXuint narcs){
  if(!surface){ fxerror("FXDCWindow::fillArcs: DC not connected to drawable.\n"); }
  XFillArcs(getApp()->display,surface->id(),gc,(XArc*)arcs,narcs);
  }


void FXDCWindow::fillPolygon(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillArcs: DC not connected to drawable.\n"); }
  XFillPolygon(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,Convex,CoordModeOrigin);
  }


void FXDCWindow::fillConcavePolygon(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygon: DC not connected to drawable.\n"); }
  XFillPolygon(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,Nonconvex,CoordModeOrigin);
  }


void FXDCWindow::fillComplexPolygon(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygon: DC not connected to drawable.\n"); }
  XFillPolygon(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,Complex,CoordModeOrigin);
  }


void FXDCWindow::fillPolygonRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillPolygonRel: DC not connected to drawable.\n"); }
  XFillPolygon(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,Convex,CoordModePrevious);
  }


void FXDCWindow::fillConcavePolygonRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygonRel: DC not connected to drawable.\n"); }
  XFillPolygon(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,Nonconvex,CoordModePrevious);
  }


void FXDCWindow::fillComplexPolygonRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygonRel: DC not connected to drawable.\n"); }
  XFillPolygon(getApp()->display,surface->id(),gc,(XPoint*)points,npoints,Complex,CoordModePrevious);
  }


void FXDCWindow::drawText(FXint x,FXint y,const FXchar* string,FXuint length){
  if(!surface){ fxerror("FXDCWindow::drawText: DC not connected to drawable.\n"); }
  XDrawString(getApp()->display,surface->id(),gc,x,y,(char*)string,length);
  }


void FXDCWindow::drawImageText(FXint x,FXint y,const FXchar* string,FXuint length){
  if(!surface){ fxerror("FXDCWindow::drawImageText: DC not connected to drawable.\n"); }
  XDrawImageString(getApp()->display,surface->id(),gc,x,y,(char*)string,length);
  }


void FXDCWindow::drawArea(const FXDrawable* source,FXint sx,FXint sy,FXint sw,FXint sh,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawArea: DC not connected to drawable.\n"); }
  if(!source || !source->id()){ fxerror("FXDCWindow::drawArea: illegal source specified.\n"); }
  XCopyArea(getApp()->display,source->id(),surface->id(),gc,sx,sy,sw,sh,dx,dy);
  }


void FXDCWindow::drawImage(const FXImage* image,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawImage: DC not connected to drawable.\n"); }
  if(!image || !image->id()){ fxerror("FXDCWindow::drawImage: illegal image specified.\n"); }
  XCopyArea(getApp()->display,image->id(),surface->id(),gc,0,0,image->width,image->height,dx,dy);
  }


// Draw bitmap (Contributed by Michal Furmanczyk <mf@cfdrc.com>)
void FXDCWindow::drawBitmap(const FXBitmap* bitmap,FXint dx,FXint dy) {
  if(!surface) fxerror("FXDCWindow::drawBitmap: DC not connected to drawable.\n");
  if(!bitmap || !bitmap->id()) fxerror("FXDCWindow::drawBitmap: illegal bitmap specified.\n");
  XCopyPlane(getApp()->display,bitmap->id(),surface->id(),gc,0,0,bitmap->width,bitmap->height,dx,dy,1);
  }


void FXDCWindow::drawIcon(const FXIcon* icon,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawIcon: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIcon: illegal icon specified.\n"); }
  XGCValues gcv;  
  gcv.clip_mask=icon->shape;
  gcv.clip_x_origin=dx;
  gcv.clip_y_origin=dy;
  XChangeGC(getApp()->display,gc,GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
  XCopyArea(getApp()->display,icon->id(),surface->id(),gc,0,0,icon->width,icon->height,dx,dy);
  gcv.clip_mask=None;
  XChangeGC(getApp()->display,gc,GCClipMask,&gcv);
  }


void FXDCWindow::drawIconShaded(const FXIcon* icon,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawIconShaded: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIconShaded: illegal icon specified.\n"); }
  XGCValues gcv;  
  gcv.clip_mask=icon->shape;
  gcv.clip_x_origin=dx;
  gcv.clip_y_origin=dy;
  XChangeGC(getApp()->display,gc,GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
  XCopyArea(getApp()->display,icon->id(),surface->id(),gc,0,0,icon->width,icon->height,dx,dy);
  gcv.function=BLT_SRC;
  gcv.stipple=getApp()->stipples[STIPPLE_GRAY];
  gcv.fill_style=FILL_STIPPLED;
  gcv.foreground=visual->getPixel(getApp()->baseColor);
  XChangeGC(getApp()->display,gc,GCForeground|GCFunction|GCStipple|GCFillStyle,&gcv);
  XFillRectangle(getApp()->display,surface->id(),gc,dx,dy,icon->width,icon->height);
  gcv.clip_mask=None;
  gcv.function=BLT_SRC;
  gcv.fill_style=FILL_SOLID;
  XChangeGC(getApp()->display,gc,GCClipMask|GCFunction|GCFillStyle,&gcv);
  }


void FXDCWindow::drawHashBox(FXint x,FXint y,FXint w,FXint h,FXint b){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::drawHashBox: DC not connected to drawable.\n"); }
  gcv.stipple=getApp()->stipples[STIPPLE_GRAY];
  gcv.fill_style=FILL_STIPPLED;
  XChangeGC(getApp()->display,gc,GCStipple|GCFillStyle,&gcv);
  XFillRectangle(getApp()->display,surface->id(),gc,x,y,w-b,b);
  XFillRectangle(getApp()->display,surface->id(),gc,x+w-b,y,b,h-b);
  XFillRectangle(getApp()->display,surface->id(),gc,x+b,y+h-b,w-b,b);
  XFillRectangle(getApp()->display,surface->id(),gc,x,y+b,b,h-b);
  gcv.stipple=getApp()->stipples[STIPPLE_WHITE];    // Needed for IRIX6.4 bug workaround!
  gcv.fill_style=FILL_SOLID;
  XChangeGC(getApp()->display,gc,GCStipple|GCFillStyle,&gcv);
  }


void FXDCWindow::setForeground(FXColor clr){
  if(!surface){ fxerror("FXDCWindow::setForeground: DC not connected to drawable.\n"); }
  XSetForeground(getApp()->display,gc,visual->getPixel(clr));
  flags|=GCForeground;
  fg=clr;
  }


void FXDCWindow::setBackground(FXColor clr){
  if(!surface){ fxerror("FXDCWindow::setBackground: DC not connected to drawable.\n"); }
  XSetBackground(getApp()->display,gc,visual->getPixel(clr));
  flags|=GCBackground;
  bg=clr;
  }


void FXDCWindow::setDashes(FXuint dashoffset,const FXchar *dashlist,FXuint n){
  if(!surface){ fxerror("FXDCWindow::setDashes: DC not connected to drawable.\n"); }
  XSetDashes(getApp()->display,gc,dashoffset,(char*)dashlist,n);
  flags|=(GCDashList|GCDashOffset);
  }


void FXDCWindow::setLineWidth(FXuint linewidth){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineWidth: DC not connected to drawable.\n"); }
  gcv.line_width=linewidth;
  XChangeGC(getApp()->display,gc,GCLineWidth,&gcv);
  flags|=GCLineWidth;
  width=linewidth;
  }


void FXDCWindow::setLineCap(FXCapStyle capstyle){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineCap: DC not connected to drawable.\n"); }
  gcv.cap_style=capstyle;
  XChangeGC(getApp()->display,gc,GCCapStyle,&gcv);
  flags|=GCCapStyle;
  cap=capstyle;
  }


void FXDCWindow::setLineJoin(FXJoinStyle joinstyle){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineJoin: DC not connected to drawable.\n"); }
  gcv.join_style=joinstyle;
  XChangeGC(getApp()->display,gc,GCJoinStyle,&gcv);
  flags|=GCJoinStyle;
  join=joinstyle;
  }


void FXDCWindow::setLineStyle(FXLineStyle linestyle){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineStyle: DC not connected to drawable.\n"); }
  gcv.line_style=linestyle;
  XChangeGC(getApp()->display,gc,GCLineStyle,&gcv);
  flags|=GCLineStyle;
  style=linestyle;
  }


void FXDCWindow::setFillStyle(FXFillStyle fillstyle){
  if(!surface){ fxerror("FXDCWindow::setFillStyle: DC not connected to drawable.\n"); }
  XSetFillStyle(getApp()->display,gc,fillstyle);
  flags|=GCFillStyle;
  fill=fillstyle;
  }
  
  
void FXDCWindow::setFillRule(FXFillRule fillrule){
  if(!surface){ fxerror("FXDCWindow::setFillRule: DC not connected to drawable.\n"); }
  XSetFillRule(getApp()->display,gc,fillrule);
  flags|=GCFillRule;
  rule=fillrule;
  }


void FXDCWindow::setFunction(FXFunction func){
  if(!surface){ fxerror("FXDCWindow::setFunction: DC not connected to drawable.\n"); }
  XSetFunction(getApp()->display,gc,func);
  flags|=GCFunction;
  rop=func;
  }


void FXDCWindow::setTile(FXImage* image,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setTile: DC not connected to drawable.\n"); }
  if(!image || !image->id()){ fxerror("FXDCWindow::setTile: illegal image specified.\n"); }
  gcv.tile=image->id();
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  XChangeGC(getApp()->display,gc,GCTileStipXOrigin|GCTileStipYOrigin|GCTile,&gcv);
  if(dx) flags|=GCTileStipXOrigin;
  if(dy) flags|=GCTileStipYOrigin;
  flags|=GCTile;
  tile=image;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setStipple(FXBitmap* bitmap,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  if(!bitmap || !bitmap->id()){ fxerror("FXDCWindow::setStipple: illegal image specified.\n"); }
  gcv.stipple=bitmap->id();
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  XChangeGC(getApp()->display,gc,GCTileStipXOrigin|GCTileStipYOrigin|GCStipple,&gcv);
  if(dx) flags|=GCTileStipXOrigin;
  if(dy) flags|=GCTileStipYOrigin;
  flags|=GCStipple;
  stipple=bitmap;
  pattern=STIPPLE_NONE;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setStipple(FXStipplePattern pat,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  if(pat>STIPPLE_CROSSDIAG) pat=STIPPLE_CROSSDIAG;
  FXASSERT(getApp()->stipples[pat]);
  gcv.stipple=getApp()->stipples[pat];
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  XChangeGC(getApp()->display,gc,GCTileStipXOrigin|GCTileStipYOrigin|GCStipple,&gcv);
  if(dx) flags|=GCTileStipXOrigin;
  if(dy) flags|=GCTileStipYOrigin;
  stipple=NULL;
  pattern=pat;
  flags|=GCStipple;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setClipRectangle(FXint x,FXint y,FXint w,FXint h){
  XRectangle c;
  if(!surface){ fxerror("FXDCWindow::setClipRectangle: DC not connected to drawable.\n"); }
  c.x=x; c.y=y; c.width=w; c.height=h;
  XSetClipRectangles(getApp()->display,gc,0,0,&c,1,Unsorted);
  flags|=GCClipMask;
  }
 

void FXDCWindow::clearClipRectangle(){
  if(!surface){ fxerror("FXDCWindow::clearClipRectangle: DC not connected to drawable.\n"); }
  XSetClipMask(getApp()->display,gc,None);
  flags&=~GCClipMask;
  }


void FXDCWindow::setClipMask(FXBitmap* bitmap,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setClipMask: DC not connected to drawable.\n"); }
  if(!bitmap || !bitmap->id()){ fxerror("FXDCWindow::setClipMask: illegal mask specified.\n"); }
  gcv.clip_mask=bitmap->id();
  gcv.clip_x_origin=dx;
  gcv.clip_y_origin=dy;
  XChangeGC(getApp()->display,gc,GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
  if(dx) flags|=GCClipXOrigin;
  if(dy) flags|=GCClipYOrigin;
  flags|=GCClipMask;
  mask=bitmap;
  cx=dx;
  cy=dy;
  }


void FXDCWindow::clearClipMask(){
  if(!surface){ fxerror("FXDCWindow::clearClipMask: DC not connected to drawable.\n"); }
  XSetClipMask(getApp()->display,gc,None);
  flags&=~GCClipMask;
  mask=NULL;
  cx=0;
  cy=0;
  }


void FXDCWindow::setTextFont(FXFont *fnt){
  if(!surface){ fxerror("FXDCWindow::setTextFont: DC not connected to drawable.\n"); }
  if(!fnt || !fnt->id()){ fxerror("FXDCWindow::setTextFont: illegal or NULL font specified.\n"); }
  XSetFont(getApp()->display,gc,fnt->id());
  flags|=GCFont;
  font=fnt;
  }


void FXDCWindow::clipChildren(FXbool yes){ 
  if(!surface){ fxerror("FXDCWindow::clipChildren: window has not yet been created.\n"); }
  if(yes){
    XSetSubwindowMode(getApp()->display,gc,ClipByChildren);
    flags&=~GCSubwindowMode;
    }
  else{
    XSetSubwindowMode(getApp()->display,gc,IncludeInferiors);
    flags|=GCSubwindowMode;
    }
  }


/********************************************************************************
*                                   MS-Windows                                  *
********************************************************************************/

#else 

// This one is not defined in the Cygwin header files
#ifndef PS_JOIN_MASK
#define PS_JOIN_MASK 0x0000F000
#endif

// Construct for expose event painting
FXDCWindow::FXDCWindow(FXDrawable* drawable,FXEvent* event):FXDC(drawable->getApp()){
  FXPROFILE("FXDCWindow::FXDCWindow()");
  tile=NULL;
  stipple=NULL;
  pattern=STIPPLE_NONE;
  if(event){
    clip=event->rect;
    }
  else{
    clip.x=0;
    clip.y=0;
    clip.w=drawable->getWidth();
    clip.h=drawable->getHeight();
    }
  needsNewBrush=FALSE;
  FXMALLOC(&brush,LOGBRUSH,1);
  needsNewPen=FALSE;
  FXMALLOC(&pen,EXTLOGPEN,1);
  needsPath=FALSE;
  begin(drawable);
  }


// Construct for normal painting
FXDCWindow::FXDCWindow(FXDrawable* drawable):FXDC(drawable->getApp()){
  FXPROFILE("FXDCWindow::FXDCWindow()");
  clip.x=0;
  clip.y=0;
  clip.w=drawable->getWidth();
  clip.h=drawable->getHeight();
  needsNewBrush=FALSE;
  FXMALLOC(&brush,LOGBRUSH,1);
  needsNewPen=FALSE;
  FXMALLOC(&pen,EXTLOGPEN,1);
  needsPath=FALSE;
  begin(drawable);
  }

  
// Destruct
FXDCWindow::~FXDCWindow(){
  FXPROFILE("FXDCWindow::~FXDCWindow()");
  FXFREE(&brush);
  FXFREE(&pen);
  end();
  }


// Begin locks in a drawable surface
void FXDCWindow::begin(FXDrawable *drawable){
  FXPROFILE("FXDCWindow::begin()");
  if(surface){ fxerror("FXDCWindow::begin: DC already connected to drawable.\n"); }
  if(!drawable){ fxerror("FXDCWindow::begin: NULL drawable.\n"); }
  if(!drawable->id()){ fxerror("FXDCWindow::begin: drawable not created yet.\n"); }

  surface=drawable;// Careful:- surface->id() can be HWND or HBITMAP depending on drawable
  visual=drawable->getVisual();
  dc=drawable->GetDC();

  setClipRectangle(clip.x,clip.y,clip.w,clip.h);

  // Select and realize palette, if necessary
  if(visual->hPalette){
    SelectPalette(dc,visual->hPalette,FALSE);
    RealizePalette(dc);
    }

  // Create our default pen (black, solid, one pixel wide)
  pen->elpPenStyle=PS_GEOMETRIC|PS_SOLID|PS_ENDCAP_ROUND|PS_JOIN_BEVEL;
  pen->elpWidth=1;
  pen->elpBrushStyle=BS_SOLID;
  pen->elpColor=PALETTERGB(0,0,0);
  pen->elpHatch=0;
  LOGBRUSH lb;
  lb.lbStyle=pen->elpBrushStyle;
  lb.lbColor=pen->elpColor;
  lb.lbHatch=pen->elpHatch;
  SelectObject(dc,ExtCreatePen(pen->elpPenStyle,pen->elpWidth,&lb,0,NULL));

  // Create our default brush (solid white, for fills)
  brush->lbStyle=BS_SOLID;
  brush->lbColor=PALETTERGB(255,255,255);
  brush->lbHatch=0;
  SelectObject(dc,CreateBrushIndirect(brush));

  // Text alignment
  SetTextAlign(dc,TA_BASELINE|TA_LEFT);

  // Polygon fill mode
  SetPolyFillMode(dc,ALTERNATE);

  // Default text font
  setTextFont(FXApp::instance()->normalFont);
  }


// End unlocks the drawable surface 
void FXDCWindow::end(){
  FXPROFILE("FXDCWindow::end()");
  if(!surface){ fxerror("FXDCWindow::end: DC not connected to drawable.\n"); }
  DeleteObject(SelectObject(dc,GetStockObject(BLACK_PEN)));
  DeleteObject(SelectObject(dc,GetStockObject(WHITE_BRUSH)));
  surface->ReleaseDC(dc);
  DWORD dwFlags=GetWindowLong((HWND)surface->id(),GWL_STYLE);//// WRONG
  SetWindowLong((HWND)surface->id(),GWL_STYLE,dwFlags|WS_CLIPCHILDREN);//// WRONG!!
  surface=NULL;
  dc=0;
  }


// Draw pixel in current foreground color
void FXDCWindow::drawPoint(FXint x,FXint y){ 
  FXPROFILE("FXDCWindow::drawPoint()");
  if(!surface){ fxerror("FXDCWindow::drawPoint: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  SetPixel(dc,x,y,pen->elpColor);
  }


void FXDCWindow::drawPoints(const FXPoint* points,FXuint npoints){ 
  FXPROFILE("FXDCWindow::drawPoints()");
  if(!surface){ fxerror("FXDCWindow::drawPoints: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<npoints; i++) drawPoint(points[i].x,points[i].y);
  }


void FXDCWindow::drawPointsRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::drawPointsRel()");
  if(!surface){ fxerror("FXDCWindow::drawPointsRel: DC not connected to drawable.\n"); }
  int x=0,y=0;
  for(FXuint i=0; i < npoints; i++){
    x+=points[i].x;
    y+=points[i].y;
    drawPoint(x,y);
    }
  }


void FXDCWindow::drawLine(FXint x1,FXint y1,FXint x2,FXint y2){
  FXPROFILE("FXDCWindow::drawLine()");
  if(!surface){ fxerror("FXDCWindow::drawLine: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  MoveToEx(dc,x1,y1,NULL);
  if(needsPath) BeginPath(dc);
  LineTo(dc,x2,y2);
  if(needsPath){
    EndPath(dc);
    StrokePath(dc);
    }
  }


void FXDCWindow::drawLines(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::drawLines()");
  if(!surface){ fxerror("FXDCWindow::drawLines: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  MoveToEx(dc,points[0].x,points[0].y,NULL);
  if(needsPath) BeginPath(dc);
  for(FXuint i=1; i<npoints; i++) LineTo(dc,points[i].x,points[i].y);
  if(needsPath){
    EndPath(dc);
    StrokePath(dc);
    }
  }


void FXDCWindow::drawLinesRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::drawLinesRel()");
  if(!surface){ fxerror("FXDCWindow::drawLinesRel: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  int x=points[0].x,y=points[0].y;
  MoveToEx(dc,x,y,NULL);
  if(needsPath) BeginPath(dc);
  for(FXuint i=1; i < npoints; i++){
    x+=points[i].x;
    y+=points[i].y;
    LineTo(dc,x,y);
    }
  if(needsPath){
    EndPath(dc);
    StrokePath(dc);
    }
  }


void FXDCWindow::drawLineSegments(const FXSegment* segments,FXuint nsegments){
  FXPROFILE("FXDCWindow::drawLineSegments()");
  if(!surface){ fxerror("FXDCWindow::drawLineSegments: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  if(needsPath) BeginPath(dc);
  for(FXuint i=0; i<nsegments; i++){
    MoveToEx(dc,segments[i].x1,segments[i].y1,NULL);
    LineTo(dc,segments[i].x2,segments[i].y2);
    }
  if(needsPath){
    EndPath(dc);
    StrokePath(dc);
    }
  }


// Unfilled rectangle
void FXDCWindow::drawRectangle(FXint x,FXint y,FXint w,FXint h){
  FXPROFILE("FXDCWindow::drawRectangle()");
  if(!surface){ fxerror("FXDCWindow::drawRectangle: DC not connected to drawable.\n"); }
  if(fill==FILL_STIPPLED || fill==FILL_OPAQUESTIPPLED){
    // Assume correct brush is already selected
    if(needsNewBrush) updateBrush();
    PatBlt(dc,x,y,w,1,PATCOPY);
    PatBlt(dc,x,y,1,h,PATCOPY);
    PatBlt(dc,x,y+h,w,1,PATCOPY);
    PatBlt(dc,x+w,y,1,h,PATCOPY);
    }
  else{
    if(needsNewPen) updatePen();
    SaveDC(dc);
    SelectObject(dc,GetStockObject(NULL_BRUSH));
    if(needsPath) BeginPath(dc);
    Rectangle(dc,x,y,x+w+1,y+h+1);
    if(needsPath){
      EndPath(dc);
      StrokePath(dc);
      }
    RestoreDC(dc,-1);
    }
  }


void FXDCWindow::drawRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  FXPROFILE("FXDCWindow::drawRectangles()");
  if(!surface){ fxerror("FXDCWindow::drawRectangles: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  SaveDC(dc);
  SelectObject(dc,GetStockObject(NULL_BRUSH));
  if(needsPath) BeginPath(dc);
  for(FXuint i=0; i < nrectangles; i++) Rectangle(dc,rectangles[i].x,rectangles[i].y,rectangles[i].x+rectangles[i].w+1,rectangles[i].y+rectangles[i].h+1);
  if(needsPath){
    EndPath(dc);
    StrokePath(dc);
    }
  RestoreDC(dc,-1);
  }


// Draw arc; angles in degrees*64, ang2 relative to ang1
void FXDCWindow::drawArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  FXPROFILE("FXDCWindow::drawArc()");
  if(!surface){ fxerror("FXDCWindow::drawArc: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  ang2+=ang1;
  int xStart=int(x+0.5*w+w*cos(ang1*PI/(180.0*64.0)));
  int yStart=int(y+0.5*h-h*sin(ang1*PI/(180.0*64.0)));
  int xEnd=int(x+0.5*w+w*cos(ang2*PI/(180.0*64.0)));
  int yEnd=int(y+0.5*h-h*sin(ang2*PI/(180.0*64.0)));
  if(needsPath) BeginPath(dc);
  Arc(dc,x,y,x+w,y+h,xStart,yStart,xEnd,yEnd);
  if(needsPath){
    EndPath(dc);
    StrokePath(dc);
    }
  }


void FXDCWindow::drawArcs(const FXArc* arcs,FXuint narcs){
  FXPROFILE("FXDCWindow::drawArcs()");
  if(!surface){ fxerror("FXDCWindow::drawArcs: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<narcs; i++){
    drawArc(arcs[i].x,arcs[i].y,arcs[i].w,arcs[i].h,arcs[i].a,arcs[i].b);
    }
  }


// Fill using currently selected ROP code
void FXDCWindow::fillRectangle(FXint x,FXint y,FXint w,FXint h){
  FXPROFILE("FXDCWindow::fillRectangle()");
  if(!surface){ fxerror("FXDCWindow::fillRectangle: DC not connected to drawable.\n"); }
  if(needsNewBrush) updateBrush();
  HPEN hPen=(HPEN)SelectObject(dc,GetStockObject(NULL_PEN));
  Rectangle(dc,x,y,x+w+1,y+h+1);
  SelectObject(dc,hPen);
  }


// Fill using currently selected ROP code
void FXDCWindow::fillRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  FXPROFILE("FXDCWindow::fillRectangles()");
  if(!surface){ fxerror("FXDCWindow::fillRectangles: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<nrectangles; i++){
    fillRectangle(rectangles[i].x,rectangles[i].y,rectangles[i].w,rectangles[i].h);
    }
  }


// Draw filled arc; angles are in degrees*64; ang2 is relative from ang1
void FXDCWindow::fillArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  FXPROFILE("FXDCWindow::fillArc()");
  if(!surface){ fxerror("FXDCWindow::fillArc: DC not connected to drawable.\n"); }
  if(needsNewBrush) updateBrush();
  ang2+=ang1;
  int xStart=int(x+0.5*w+w*cos(ang1*PI/(180.0*64.0)));
  int yStart=int(y+0.5*h-h*sin(ang1*PI/(180.0*64.0)));
  int xEnd=int(x+0.5*w+w*cos(ang2*PI/(180.0*64.0)));
  int yEnd=int(y+0.5*h-h*sin(ang2*PI/(180.0*64.0)));
  HPEN hPen=(HPEN)SelectObject(dc,GetStockObject(NULL_PEN));
  Pie(dc,x,y,x+w,y+h,xStart,yStart,xEnd,yEnd);
  SelectObject(dc,hPen);
  }


void FXDCWindow::fillArcs(const FXArc* arcs,FXuint narcs){
  FXPROFILE("FXDCWindow::fillArcs()");
  if(!surface){ fxerror("FXDCWindow::fillArcs: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<narcs; i++){
    fillArc(arcs[i].x,arcs[i].y,arcs[i].w,arcs[i].h,arcs[i].a,arcs[i].b);
    }
  }


// Filled simple polygon
void FXDCWindow::fillPolygon(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillPolgon()");
  if(!surface){ fxerror("FXDCWindow::fillPolygon: DC not connected to drawable.\n"); }
  if(needsNewBrush) updateBrush();
  POINT *pts=new POINT[npoints];
  for(FXuint i=0; i<npoints; i++){
    pts[i].x=points[i].x;
    pts[i].y=points[i].y;
    }
  HPEN hPen=(HPEN)SelectObject(dc,GetStockObject(NULL_PEN));
  Polygon(dc,pts,npoints);
  SelectObject(dc,hPen);
  delete [] pts;
  }


void FXDCWindow::fillConcavePolygon(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillConcavePolygon()");
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygon: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


void FXDCWindow::fillComplexPolygon(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillComplexPolygon()");
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygon: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


// Filled simple polygon with relative points
void FXDCWindow::fillPolygonRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillPolygonRel()");
  if(!surface){ fxerror("FXDCWindow::fillPolygonRel: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


void FXDCWindow::fillConcavePolygonRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillConcavePolygonRel()");
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygonRel: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


void FXDCWindow::fillComplexPolygonRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillComplexPolygonRel()");
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygonRel: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


// Draw string (only foreground bits)
void FXDCWindow::drawText(FXint x,FXint y,const FXchar* string,FXuint length){
  FXPROFILE("FXDCWindow::drawText()");
  if(!surface){ fxerror("FXDCWindow::drawText: DC not connected to drawable.\n"); }
  SaveDC(dc);
  SetBkMode(dc,TRANSPARENT);
  TextOut(dc,x,y,string,length);
  RestoreDC(dc,-1);
  }


// Draw string (both foreground and background bits)
void FXDCWindow::drawImageText(FXint x,FXint y,const FXchar* string,FXuint length){
  FXPROFILE("FXDCWindow::drawImageText()");
  if(!surface){ fxerror("FXDCWindow::drawImageText: DC not connected to drawable.\n"); }
  SaveDC(dc);
  SetBkMode(dc,OPAQUE);
  TextOut(dc,x,y,string,length);
  RestoreDC(dc,-1);
  }


// Draw area from source
// Some of these ROP codes do not have names; the full list can be found in the MSDN docs
// at Platform SDK/Reference/Appendixes/Win32 Appendixes/Raster Operation Codes/Ternary Raster Operations
void FXDCWindow::drawArea(const FXDrawable* source,FXint sx,FXint sy,FXint sw,FXint sh,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawArea()");
  if(!surface){ fxerror("FXDCWindow::drawArea: DC not connected to drawable.\n"); }
  if(!source || !source->id()){ fxerror("FXDCWindow::drawArea: illegal source specified.\n"); }
  if(needsNewBrush) updateBrush();
  HDC shdc=source->GetDC();
  switch(rop){
    case BLT_CLR:                     // D := 0
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,BLACKNESS); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,SRCAND); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,SRCERASE); 
      break;
    case BLT_SRC:                     // D := S
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,SRCCOPY); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,0x220326);
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,SRCINVERT); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,SRCPAINT); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,NOTSRCERASE); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,0x990066); // Not sure about this one
      break;
    case BLT_NOT_DST:                 // D := ~D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,DSTINVERT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,0xDD0228); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,NOTSRCCOPY); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,MERGEPAINT); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,0x7700E6); 
      break;
    case BLT_SET:                     // D := 1
      BitBlt(dc,dx,dy,sw,sh,shdc,sx,sy,WHITENESS); 
    default: 
      break;
    }
  source->ReleaseDC(shdc); // For symmetry
  }


// Draw image (Fixed to obey function)
void FXDCWindow::drawImage(const FXImage* image,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawImage()");
  if(!surface){ fxerror("FXDCWindow::drawImage: DC not connected to drawable.\n"); }
  if(!image || !image->id()){ fxerror("FXDCWindow::drawImage: illegal image specified.\n"); }
  if(needsNewBrush) updateBrush();
  HDC dcMem=image->GetDC();
  POINT ptSize;
  POINT ptOrg;
  SetMapMode(dcMem,GetMapMode(dc));
  ptSize.x=image->width;
  ptSize.y=image->height;
  DPtoLP(dc,&ptSize,1);
  ptOrg.x=0; 
  ptOrg.y=0;
  DPtoLP(dc,&ptOrg,1);
  switch(rop){
    case BLT_CLR:                     // D := 0
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,BLACKNESS); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCAND); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCERASE); 
      break;
    case BLT_SRC:                     // D := S
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCCOPY); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0x220326);
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCINVERT); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCPAINT); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,NOTSRCERASE); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0x990066); // Not sure about this one
      break;
    case BLT_NOT_DST:                 // D := ~D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,DSTINVERT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0xDD0228); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,NOTSRCCOPY); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,MERGEPAINT); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0x7700E6); 
      break;
    case BLT_SET:                     // D := 1
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,WHITENESS); 
    default: 
      break;
    }
  image->ReleaseDC(dcMem); // For symmetry
  }


// Draw bitmap (Contributed by Michal Furmanczyk <mf@cfdrc.com>)
void FXDCWindow::drawBitmap(const FXBitmap* bitmap,FXint dx,FXint dy) {
  FXPROFILE("FXDCWindow::drawBitmap()");
  if(!surface) fxerror("FXDCWindow::drawBitmap: DC not connected to drawable.\n");
  if(!bitmap || !bitmap->id()) fxerror("FXDCWindow::drawBitmap: illegal bitmap specified.\n");
  if(needsNewBrush) updateBrush();
  HDC dcMem=bitmap->GetDC();
  POINT ptSize;
  POINT ptOrg;
  SetMapMode(dcMem,GetMapMode(dc));
  ptSize.x=bitmap->width;
  ptSize.y=bitmap->height;
  DPtoLP(dc,&ptSize,1);
  ptOrg.x=0; 
  ptOrg.y=0;
  DPtoLP(dc,&ptOrg,1);
  switch(rop){
    case BLT_CLR:                     // D := 0
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,BLACKNESS); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCAND); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCERASE); 
      break;
    case BLT_SRC:                     // D := S
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCCOPY); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0x220326);
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCINVERT); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,SRCPAINT); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,NOTSRCERASE); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0x990066); // Not sure about this one
      break;
    case BLT_NOT_DST:                 // D := ~D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,DSTINVERT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0xDD0228); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,NOTSRCCOPY); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,MERGEPAINT); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,0x7700E6); 
      break;
    case BLT_SET:                     // D := 1
      BitBlt(dc,dx,dy,ptSize.x,ptSize.y,dcMem,ptOrg.x,ptOrg.y,WHITENESS); 
    default: 
      break;
    }
  bitmap->ReleaseDC(dcMem); // For symmetry
  }


// Draw icon
void FXDCWindow::drawIcon(const FXIcon* icon,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawIcon()");
  if(!surface){ fxerror("FXDCWindow::drawIcon: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIcon: illegal icon specified.\n"); }
  HDC hdcSrc=icon->GetDC();
  HDC hdcMsk=CreateCompatibleDC(dc);
  SelectObject(hdcMsk,icon->shape);
  COLORREF crOldBack=SetBkColor(dc,RGB(255,255,255));
  COLORREF crOldText=SetTextColor(dc,RGB(0,0,0));
  BitBlt(dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCINVERT);
  BitBlt(dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcMsk,0,0,SRCAND);
  BitBlt(dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCINVERT);
  DeleteDC(hdcMsk);
  SetBkColor(dc,crOldBack);
  SetTextColor(dc,crOldText);
  icon->ReleaseDC(hdcSrc); // For symmetry
  }


void FXDCWindow::drawIconShaded(const FXIcon* icon,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawIconShaded()");
  if(!surface){ fxerror("FXDCWindow::drawIconShaded: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIconShaded: illegal icon specified.\n"); }
  HDC hdcSrc=icon->GetDC();
  HDC hdcMask=CreateCompatibleDC(dc);
  SelectObject(hdcMask,icon->shape);
  HDC hdcShadedMask=CreateCompatibleDC(dc);
  HBITMAP hbmpShadedMask=CreateBitmap(icon->getWidth(),icon->getHeight(),1,1,NULL);
  SelectObject(hdcShadedMask,hbmpShadedMask);
  BitBlt(hdcShadedMask,0,0,icon->getWidth(),icon->getHeight(),hdcMask,0,0,SRCCOPY);
  DeleteDC(hdcMask);
  HBRUSH hBrush=CreatePatternBrush(getApp()->stipples[STIPPLE_GRAY]);
  SelectObject(hdcShadedMask,hBrush);
  PatBlt(hdcShadedMask,0,0,icon->getWidth(),icon->getHeight(),0xFA0089);// Preserve background based on shape AND stipple
  DeleteObject(hBrush);
  COLORREF crOldBack=SetBkColor(dc,RGB(255,255,255));
  COLORREF crOldText=SetTextColor(dc,RGB(0,0,0));
  BitBlt(dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCINVERT);
  BitBlt(dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcShadedMask,0,0,SRCAND);
  BitBlt(dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCINVERT);
  DeleteDC(hdcShadedMask);
  DeleteObject(hbmpShadedMask);
  SetBkColor(dc,crOldBack);
  SetTextColor(dc,crOldText);
  icon->ReleaseDC(hdcSrc);
  }


void FXDCWindow::drawHashBox(FXint x,FXint y,FXint w,FXint h,FXint b){
  FXPROFILE("FXDCWindow::drawHashBox()");
  if(!surface){ fxerror("FXDCWindow::drawHashBox: DC not connected to drawable.\n"); }
  HBRUSH hBrush=(HBRUSH)SelectObject(dc,CreatePatternBrush(getApp()->stipples[STIPPLE_GRAY]));
  HPEN hPen=(HPEN)SelectObject(dc,GetStockObject(NULL_PEN));
  PatBlt(dc,x,y,w-b,b,PATINVERT);
  PatBlt(dc,x+w-b,y,b,h-b,PATINVERT);
  PatBlt(dc,x+b,y+h-b,w-b,b,PATINVERT);
  PatBlt(dc,x,y+b,b,h-b,PATINVERT);
  SelectObject(dc,hPen);
  DeleteObject(SelectObject(dc,hBrush));
  }


void FXDCWindow::updatePen(){
  FXPROFILE("FXDCWindow::updatePen()");
  LOGBRUSH lb;
  lb.lbStyle=pen->elpBrushStyle;
  lb.lbColor=pen->elpColor;
  lb.lbHatch=pen->elpHatch;
  DeleteObject(SelectObject(dc,ExtCreatePen(pen->elpPenStyle,pen->elpWidth,&lb,0,NULL)));
  needsPath = (pen->elpWidth>1) ? TRUE : FALSE;
  needsNewPen=FALSE;
  }


void FXDCWindow::setForeground(FXColor clr){
  FXPROFILE("FXDCWindow::setForeground()");
  if(!surface){ fxerror("FXDCWindow::setForeground: DC not connected to drawable.\n"); }
  fg=clr;

  // Create a new pen with same settings and new color
  pen->elpColor=visual->getPixel(fg);
  needsNewPen=TRUE;

  // Reset the brush
  brush->lbColor=visual->getPixel(fg);
  needsNewBrush=TRUE;

  // And the text color
  SetTextColor(dc,visual->getPixel(fg));
  }


void FXDCWindow::setBackground(FXColor clr){
  FXPROFILE("FXDCWindow::setBackground()");
  if(!surface){ fxerror("FXDCWindow::setBackground: DC not connected to drawable.\n"); }
  bg=clr;
  SetBkColor(dc,visual->getPixel(bg));
  }


// Set dash pattern (for the LINE_ONOFF_DASH line style)
void FXDCWindow::setDashes(FXuint dashoffset,const FXchar *dashlist,FXuint n){
  FXPROFILE("FXDCWindow::setDashes()");
  if(!surface){ fxerror("FXDCWindow::setDashes: DC not connected to drawable.\n"); }
  }


void FXDCWindow::setLineWidth(FXuint linewidth){ 
  FXPROFILE("FXDCWindow::setLineWidth()");
  if(!surface){ fxerror("FXDCWindow::setLineWidth: DC not connected to drawable.\n"); }
  // Create a new pen with this line width
  pen->elpWidth=linewidth;
  width=linewidth;
  needsNewPen=TRUE;
  }


void FXDCWindow::setLineCap(FXCapStyle capstyle){
  FXPROFILE("FXDCWindow::setLineCap()");
  if(!surface){ fxerror("FXDCWindow::setLineCap: DC not connected to drawable.\n"); }

  // Modify its cap style
  DWORD dwPenType=pen->elpPenStyle&PS_TYPE_MASK;
  DWORD dwPenStyle=pen->elpPenStyle&PS_STYLE_MASK;
  DWORD dwEndCapAttributes=pen->elpPenStyle&PS_ENDCAP_MASK;
  DWORD dwJoinAttributes=pen->elpPenStyle&PS_JOIN_MASK;
  switch(capstyle){
    case CAP_NOT_LAST:
      dwPenType=PS_GEOMETRIC;
      dwEndCapAttributes=PS_ENDCAP_SQUARE;
      break;
    case CAP_BUTT:
      dwPenType=PS_GEOMETRIC;
      dwEndCapAttributes=PS_ENDCAP_FLAT;
      break;
    case CAP_ROUND:
      dwPenType=PS_GEOMETRIC;
      dwEndCapAttributes=PS_ENDCAP_ROUND;
      break;
    case CAP_PROJECTING:
      FXASSERT(FALSE);
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create a new pen with this style
  pen->elpPenStyle=dwPenType|dwPenStyle|dwEndCapAttributes|dwJoinAttributes;
  cap=capstyle;
  needsNewPen=TRUE;
  }


void FXDCWindow::setLineJoin(FXJoinStyle joinstyle){
  FXPROFILE("FXDCWindow::setLineJoin()");
  if(!surface){ fxerror("FXDCWindow::setLineJoin: DC not connected to drawable.\n"); }

  // Now modify the style flags
  DWORD dwPenType=pen->elpPenStyle&PS_TYPE_MASK;
  DWORD dwPenStyle=pen->elpPenStyle&PS_STYLE_MASK;
  DWORD dwEndCapAttributes=pen->elpPenStyle&PS_ENDCAP_MASK;
  DWORD dwJoinAttributes=pen->elpPenStyle&PS_JOIN_MASK;
  switch(joinstyle){
    case JOIN_MITER:
      dwPenType=PS_GEOMETRIC;
      dwJoinAttributes=PS_JOIN_MITER;
      break;
    case JOIN_ROUND:
      dwPenType=PS_GEOMETRIC;
      dwJoinAttributes=PS_JOIN_ROUND;
      break;
    case JOIN_BEVEL:
      dwPenType=PS_GEOMETRIC;
      dwJoinAttributes=PS_JOIN_BEVEL;
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create a new pen with this style
  pen->elpPenStyle=dwPenType|dwPenStyle|dwEndCapAttributes|dwJoinAttributes;
  join=joinstyle;
  needsNewPen=TRUE;
  }


void FXDCWindow::setLineStyle(FXLineStyle linestyle){
  FXPROFILE("FXDCWindow::setLineStyle()");
  if(!surface){ fxerror("FXDCWindow::setLineStyle: DC not connected to drawable.\n"); }

  // Modify style bits
  DWORD dwPenType=pen->elpPenStyle&PS_TYPE_MASK;
  DWORD dwPenStyle=pen->elpPenStyle&PS_STYLE_MASK;
  DWORD dwEndCapAttributes=pen->elpPenStyle&PS_ENDCAP_MASK;
  DWORD dwJoinAttributes=pen->elpPenStyle&PS_JOIN_MASK;
  switch(linestyle){
    case LINE_SOLID:
      dwPenType=PS_GEOMETRIC;
      dwPenStyle=PS_SOLID;
      break;
    case LINE_ONOFF_DASH:
      dwPenType=PS_COSMETIC;
      dwPenStyle=PS_DOT; // PS_ALTERNATE looks better, but is only supported for NT
      break;
    case LINE_DOUBLE_DASH:
      FXASSERT(FALSE);
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create a new pen with this style
  pen->elpPenStyle=dwPenType|dwPenStyle|dwEndCapAttributes|dwJoinAttributes;
  style=linestyle;
  needsNewPen=TRUE;
  }


static DWORD FXStipplePattern2Hatch(FXStipplePattern pat){
  FXPROFILE("FXStipplePattern2Hatch()");
  switch(pat){
    case STIPPLE_HORZ:
      return HS_HORIZONTAL;
    case STIPPLE_VERT:
      return HS_VERTICAL;
    case STIPPLE_CROSS:
      return HS_CROSS;
    case STIPPLE_DIAG:
      return HS_BDIAGONAL;
    case STIPPLE_REVDIAG:
      return HS_FDIAGONAL;
    case STIPPLE_CROSSDIAG:
      return HS_DIAGCROSS;
    default:
      FXASSERT(FALSE);
      return 0;
    }
  }


void FXDCWindow::updateBrush(){
  FXPROFILE("FXDCWindow::updateBrush()");
  // Get previous brush info
  DeleteObject(SelectObject(dc,GetStockObject(NULL_BRUSH)));
  switch(fill){
    case FILL_SOLID:
      brush->lbStyle=BS_SOLID;
      SetBkMode(dc,OPAQUE);
      break;
    case FILL_TILED:
      FXASSERT(FALSE);
      break;
    case FILL_STIPPLED:
      SetBkMode(dc,TRANSPARENT);
      if(stipple){
      	brush->lbStyle=BS_PATTERN;
      	brush->lbHatch=(LONG)stipple->id();    // This should be a HBITMAP
      	}
      else{
	if(pattern>=STIPPLE_0 && pattern<=STIPPLE_16){
	  brush->lbStyle=BS_PATTERN;
	  brush->lbHatch=(LONG)getApp()->stipples[pattern];
	  }
      	else{
	  brush->lbStyle=BS_HATCHED;
	  brush->lbHatch=FXStipplePattern2Hatch(pattern);
	  }
	}
      break;
    case FILL_OPAQUESTIPPLED:
      SetBkMode(dc,OPAQUE);
      if(stipple){
      	brush->lbStyle=BS_PATTERN;
      	brush->lbHatch=(LONG)stipple->id();    // This should be a HBITMAP
      	}
      else{
      	if(pattern>=STIPPLE_0 && pattern<=STIPPLE_16){
	  brush->lbStyle=BS_PATTERN;
	  brush->lbHatch=(LONG)getApp()->stipples[pattern];
	  }
      	else{
	  brush->lbStyle=BS_HATCHED;
	  brush->lbHatch=FXStipplePattern2Hatch(pattern);
	  }
       	}
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create new brush and select it into the DC
  SelectObject(dc,CreateBrushIndirect(brush));
  needsNewBrush=FALSE;
  }


void FXDCWindow::setFillStyle(FXFillStyle fillstyle){
  FXPROFILE("FXDCWindow::setFillStyle()");
  if(!surface){ fxerror("FXDCWindow::setFillStyle: DC not connected to drawable.\n"); }
  fill=fillstyle;
  needsNewBrush=TRUE;
  }
  
  
// Set fill rule
void FXDCWindow::setFillRule(FXFillRule fillrule){
  FXPROFILE("FXDCWindow::setFillRule()");
  if(!surface){ fxerror("FXDCWindow::setFillRule: DC not connected to drawable.\n"); }
  if(fillrule==RULE_EVEN_ODD)
    SetPolyFillMode(dc,ALTERNATE);
  else
    SetPolyFillMode(dc,WINDING);
  rule=fillrule;
  }


// Set blit function
void FXDCWindow::setFunction(FXFunction func){
  FXPROFILE("FXDCWindow::setFunction()");
  if(!surface){ fxerror("FXDCWindow::setFunction: DC not connected to drawable.\n"); }
  rop=func;

  // Also set ROP2 code for lines
  switch(rop){
    case BLT_CLR:                     // D := 0
      SetROP2(dc,R2_BLACK); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      SetROP2(dc,R2_MASKPEN); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      SetROP2(dc,R2_MASKPENNOT); 
      break;
    case BLT_SRC:                     // D := S
      SetROP2(dc,R2_COPYPEN); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      SetROP2(dc,R2_MASKNOTPEN); 
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      SetROP2(dc,R2_XORPEN); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      SetROP2(dc,R2_MERGEPEN); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      SetROP2(dc,R2_NOTMERGEPEN); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      SetROP2(dc,R2_NOTXORPEN); // Is this the right one?
      break;
    case BLT_NOT_DST:                 // D := ~D
      SetROP2(dc,R2_NOT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      SetROP2(dc,R2_MERGEPENNOT); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      SetROP2(dc,R2_NOTCOPYPEN); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      SetROP2(dc,R2_MERGENOTPEN); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      SetROP2(dc,R2_NOTMASKPEN); 
      break;
    case BLT_SET:                     // D := 1
      SetROP2(dc,R2_WHITE); 
      break;
    default:
      FXASSERT(FALSE);
    }
  }


// Set tile image
void FXDCWindow::setTile(FXImage* image,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setTile()");
  if(!surface){ fxerror("FXDCWindow::setTile: DC not connected to drawable.\n"); }
  tile=image;
  tx=dx;
  ty=dy;
  }

////////////// FIXME: likely not a good idea to hang on to the FXImage; the FOX object may go
//////////////        away even though the handle may hang around...

// Set stipple pattern
void FXDCWindow::setStipple(FXBitmap* bitmap,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setStipple()");
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  FXASSERT(dx==0); // We ignore this for the Windows version
  FXASSERT(dy==0); // We ignore this for the Windows version
  stipple=bitmap;
  pattern=STIPPLE_NONE;
  needsNewBrush=TRUE;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setStipple(FXStipplePattern pat,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setStipple()");
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  FXASSERT(dx==0); // We ignore this for the Windows version
  FXASSERT(dy==0); // We ignore this for the Windows version
  stipple=NULL;
  pattern=pat;
  needsNewBrush=TRUE;
  tx=dx;
  ty=dy;
  }

////////////// FIXME: likely not a good idea to hang on to the FXBitmap; the FOX object may go
//////////////        away even though the handle may hang around...

// Set clip rectangle
void FXDCWindow::setClipRectangle(FXint x,FXint y,FXint w,FXint h){
  FXPROFILE("FXDCWindow::setClipRectangle()");
  if(!surface){ fxerror("FXDCWindow::setClipRectangle: DC not connected to drawable.\n"); }
  HRGN hClipRgn=CreateRectRgn(x,y,x+w,y+h);
  FXASSERT(hClipRgn!=NULL);
  int s=ExtSelectClipRgn(dc,hClipRgn,RGN_COPY);
  FXASSERT(s!=ERROR);
  DeleteObject(hClipRgn);
  }
 

// Clear clipping
void FXDCWindow::clearClipRectangle(){
  FXPROFILE("FXDCWindow::clearClipRectangle()");
  if(!surface){ fxerror("FXDCWindow::clearClipRectangle: DC not connected to drawable.\n"); }
  int s=ExtSelectClipRgn(dc,NULL,RGN_COPY);
  FXASSERT(s!=ERROR);
  }


// Set clip mask
void FXDCWindow::setClipMask(FXBitmap* bitmap,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setClipMask()");
  if(!surface){ fxerror("FXDCWindow::setClipMask: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  mask=bitmap;
  cx=dx;
  cy=dy;
  }


// Clear clip mask
void FXDCWindow::clearClipMask(){
  FXPROFILE("FXDCWindow::clearClipMask()");
  if(!surface){ fxerror("FXDCWindow::clearClipMask: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  mask=NULL;
  cx=0;
  cy=0;
  }

  
// Set font to draw text with
void FXDCWindow::setTextFont(FXFont *fnt){
  FXPROFILE("FXDCWindow::setTextFont()");
  if(!surface){ fxerror("FXDCWindow::setTextFont: DC not connected to drawable.\n"); }
  if(!fnt || !fnt->id()){ fxerror("FXDCWindow::setTextFont: illegal or NULL font specified.\n"); }
  SelectObject(dc,fnt->id());
  font=fnt;
  }


// Save the current state of the DC
void FXDCWindow::saveDC(HPEN& hPen,HBRUSH& hBrush,HFONT& hFont,RECT& cr,FXuint& tc,FXuint& bc,FXint& fm){
  FXPROFILE("FXDCWindow::saveDC()");
  hPen=(HPEN)SelectObject(dc,GetStockObject(NULL_PEN));
  hBrush=(HBRUSH)SelectObject(dc,GetStockObject(NULL_BRUSH));
  hFont=(HFONT)SelectObject(dc,GetStockObject(SYSTEM_FONT));
  //GetClipBox(dc,&cr);
  tc=GetTextColor(dc);
  bc=GetBkColor(dc);
  fm=GetPolyFillMode(dc);
  }


// Restore the state
void FXDCWindow::restoreDC(const HPEN& hPen,const HBRUSH& hBrush,const HFONT& hFont,const RECT& cr,FXuint tc,FXuint bc,FXint fm){
  FXPROFILE("FXDCWindow::restoreDC()");
  SelectObject(dc,hFont);
  SelectObject(dc,hPen);
  SelectObject(dc,hBrush);
  /*
  if(cr.left!=0 || cr.top!=0 || cr.right!=0 || cr.bottom!=0){
    HRGN hClipRgn=CreateRectRgn(cr.left,cr.top,cr.right,cr.bottom);
    ExtSelectClipRgn(dc,hClipRgn,RGN_COPY);
    DeleteObject(hClipRgn);
    }
  */
  if(visual->hPalette){
    SelectPalette(dc,visual->hPalette,FALSE);
    RealizePalette(dc);
    }
  SetTextAlign(dc,TA_BASELINE|TA_LEFT);
  SetTextColor(dc,tc);
  SetBkColor(dc,bc);
  SetPolyFillMode(dc,fm);
  }


// Window will clip against child windows
void FXDCWindow::clipChildren(FXbool yes){ 
  FXPROFILE("FXDCWindow::clipChildren()");
  if(!surface){ fxerror("FXDCWindow::clipChildren: window has not yet been created.\n"); }
  HPEN hPen;
  HBRUSH hBrush;
  HFONT hFont;
  RECT cliprect;
  FXuint textcolor,backcolor;
  FXint fillmode;
  DWORD dwFlags=GetWindowLong((HWND)surface->id(),GWL_STYLE);
  if(yes){
    if((dwFlags&WS_CLIPCHILDREN)==0){
      saveDC(hPen,hBrush,hFont,cliprect,textcolor,backcolor,fillmode);
      ReleaseDC((HWND)surface->id(),dc);
      SetWindowLong((HWND)surface->id(),GWL_STYLE,dwFlags|WS_CLIPCHILDREN);
      dc=GetDC((HWND)surface->id());
      restoreDC(hPen,hBrush,hFont,cliprect,textcolor,backcolor,fillmode);
      }
    }
  else{
    if(dwFlags&WS_CLIPCHILDREN){
      saveDC(hPen,hBrush,hFont,cliprect,textcolor,backcolor,fillmode);
      ReleaseDC((HWND)surface->id(),dc);
      SetWindowLong((HWND)surface->id(),GWL_STYLE,dwFlags & ~WS_CLIPCHILDREN);
      dc=GetDC((HWND)surface->id());
      restoreDC(hPen,hBrush,hFont,cliprect,textcolor,backcolor,fillmode);
      }
    }
  }

#endif
