/********************************************************************************
*                                                                               *
*                          I c o n L i s t   O b j e c t                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXOldIconList.cpp,v 1.2 1999/09/21 18:02:58 jeroen Exp $                 *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollWindow.h"
#include "FXOldHeader.h"
#include "FXOldIconList.h"



/*
  To do:
  - Item size should depend on mode.
  - Update Items should be optimized.
  - Scrolling stuff should be cleaned up.
  - When switching to detail mode, there is interaction between sb's and header
  - Need to handle mouse actions properly.
  - Need to fix click/double click etc. stuff.
  - drawIconShaded should be made to work correctly for colormapped displays.
  - Need to use selection history here too! (e.g. selectionrectangle)
  - Slight overlap between update rectangles?
  - Need restore item stuff back?
  - In detail-mode, some items should be left, some right, and some centered in the field.
  - Slight optimization:- only recompute() when needed.
  - Text layout not good if bigger texts.
  - Typing first few letters of an item should make that item current.
  - Maybe need item-by-name access?
  - Return key simulates double click.
  - Need method to set header columns.
  - Sortfunc's will be hard to serialize, and hard to write w/o secretly #including
    the FXTreeItem header!
*/



#define SIDE_SPACING             4    // Left or right spacing between items

#define DETAIL_LINE_SPACING      0    // Line spacing in detail mode
#define DETAIL_TEXT_SPACING      4    // Spacing between text and icon in detail icon mode

#define MINI_LINE_SPACING        1    // Line spacing in mini icon mode
#define MINI_TEXT_SPACING        4    // Spacing between text and icon in mini icon mode

#define BIG_LINE_SPACING         6    // Line spacing in big icon mode
#define BIG_TEXT_SPACING         4    // Spacing between text and icon in big icon mode

#define MINI_ICON_WIDTH         18    // Default mini icon width
#define MINI_ICON_HEIGHT        18    // Default mini icon height

#define BIG_ICON_WIDTH          32    // Default big icon width
#define BIG_ICON_HEIGHT         32    // Default big icon height

#define ITEM_SPACE             128    // Default space for item

#define OLDICONLIST_MASK (OLDICONLIST_MINI_ICONS|OLDICONLIST_BIG_ICONS|OLDICONLIST_COLUMNS|OLDICONLIST_RANDOM|OLDICONLIST_WANTSELECTION)



/*******************************************************************************/

// Map
FXDEFMAP(FXOldIconList) FXOldIconListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXOldIconList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXOldIconList::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXOldIconList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXOldIconList::onLeftBtnRelease),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXOldIconList::onAutoScroll),
  FXMAPFUNC(SEL_TIMEOUT,FXOldIconList::ID_TIPTIMER,FXOldIconList::onTipTimer),
  FXMAPFUNC(SEL_UNGRABBED,0,FXOldIconList::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXOldIconList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXOldIconList::onKeyRelease),
  FXMAPFUNC(SEL_ENTER,0,FXOldIconList::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXOldIconList::onLeave),
  FXMAPFUNC(SEL_FOCUSIN,0,FXOldIconList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXOldIconList::onFocusOut),
  FXMAPFUNC(SEL_ACTIVATE,0,FXOldIconList::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXOldIconList::onDeactivate),
  FXMAPFUNC(SEL_CHANGED,0,FXOldIconList::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXOldIconList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXOldIconList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXOldIconList::onTripleClicked),
  FXMAPFUNC(SEL_SELECTED,0,FXOldIconList::onSelected),
  FXMAPFUNC(SEL_DESELECTED,0,FXOldIconList::onDeselected),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXOldIconList::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXOldIconList::onSelectionGained),
  FXMAPFUNC(SEL_CHANGED,FXOldIconList::ID_HEADER_CHANGE,FXOldIconList::onHeaderChanged),
  FXMAPFUNC(SEL_UPDATE,FXOldIconList::ID_SHOW_DETAILS,FXOldIconList::onUpdShowDetails),
  FXMAPFUNC(SEL_UPDATE,FXOldIconList::ID_SHOW_MINI_ICONS,FXOldIconList::onUpdShowMiniIcons),
  FXMAPFUNC(SEL_UPDATE,FXOldIconList::ID_SHOW_BIG_ICONS,FXOldIconList::onUpdShowBigIcons),
  FXMAPFUNC(SEL_UPDATE,FXOldIconList::ID_ARRANGE_BY_ROWS,FXOldIconList::onUpdArrangeByRows),
  FXMAPFUNC(SEL_UPDATE,FXOldIconList::ID_ARRANGE_BY_COLUMNS,FXOldIconList::onUpdArrangeByColumns),
  FXMAPFUNC(SEL_UPDATE,FXOldIconList::ID_ARRANGE_AT_RANDOM,FXOldIconList::onUpdArrangeAtRandom),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXOldIconList::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXOldIconList::onQueryHelp),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_SHOW_DETAILS,FXOldIconList::onCmdShowDetails),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_SHOW_MINI_ICONS,FXOldIconList::onCmdShowMiniIcons),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_SHOW_BIG_ICONS,FXOldIconList::onCmdShowBigIcons),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_ARRANGE_BY_ROWS,FXOldIconList::onCmdArrangeByRows),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_ARRANGE_BY_COLUMNS,FXOldIconList::onCmdArrangeByColumns),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_ARRANGE_AT_RANDOM,FXOldIconList::onCmdArrangeAtRandom),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_SELECT_ALL,FXOldIconList::onCmdSelectAll),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_DESELECT_ALL,FXOldIconList::onCmdDeselectAll),
  FXMAPFUNC(SEL_COMMAND,FXOldIconList::ID_SELECT_INVERSE,FXOldIconList::onCmdSelectInverse),
  };


// Object implementation
FXIMPLEMENT(FXOldIconList,FXScrollArea,FXOldIconListMap,ARRAYNUMBER(FXOldIconListMap))


/*******************************************************************************/


// Serialization
FXOldIconList::FXOldIconList(){
  flags|=FLAG_ENABLED;
  header=(FXOldHeader*)-1;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  iconWidth=BIG_ICON_WIDTH;
  iconHeight=BIG_ICON_HEIGHT;
  miniIconWidth=MINI_ICON_WIDTH;
  miniIconHeight=MINI_ICON_HEIGHT;
  itemSpace=ITEM_SPACE;
  itemWidth=0;
  itemHeight=0;
  totalWidth=0;
  totalHeight=0;
  anchorx=0;
  anchory=0;
  currentx=0;
  currenty=0;
  timer=NULL;
  }


// Icon List
FXOldIconList::FXOldIconList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  header=new FXOldHeader(this,this,FXOldIconList::ID_HEADER_CHANGE,FRAME_RAISED|FRAME_THICK);
  target=tgt;
  message=sel;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  font=getApp()->normalFont;
  sortfunc=NULL;
  textColor=getApp()->foreColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  iconWidth=BIG_ICON_WIDTH;
  iconHeight=BIG_ICON_HEIGHT;
  miniIconWidth=MINI_ICON_WIDTH;
  miniIconHeight=MINI_ICON_HEIGHT;
  itemSpace=ITEM_SPACE;
  itemWidth=0;
  itemHeight=0;
  totalWidth=0;
  totalHeight=0;
  anchorx=0;
  anchory=0;
  currentx=0;
  currenty=0;
  timer=NULL;
  }


// Create window
void FXOldIconList::create(){
  FXOldIconItem *item;
  FXScrollArea::create();
  font->create();
  if(!deleteType){deleteType=getApp()->registerDragType(deleteTypeName);}
  if(!textType){textType=getApp()->registerDragType(textTypeName);}
  for(item=firstitem; item; item=item->next){
    if(item->icon){item->icon->create();}
    if(item->miniIcon){item->miniIcon->create();}
    }
  dropEnable();
  }


// Detach window
void FXOldIconList::detach(){
  FXOldIconItem *item;
  FXScrollArea::detach();
  font->create();
  deleteType=0;
  textType=0;
  for(item=firstitem; item; item=item->next){
    if(item->icon){item->icon->detach();}
    if(item->miniIcon){item->miniIcon->detach();}
    }
  }


// If window can have focus
FXbool FXOldIconList::canFocus() const { return 1; }


// Create item
FXOldIconItem* FXOldIconList::createItem(){ 
  return new FXOldIconItem; 
  }


// Destroy item
void FXOldIconList::deleteItem(FXOldIconItem* item){ 
  delete item; 
  }


// Get number of items
FXint FXOldIconList::getNumItems() const {
  register FXOldIconItem *item=firstitem;
  register FXint n=0;
  while(item){
    item=item->next;
    n++;
    }
  return n;
  }


// Get number of selected items
FXint FXOldIconList::getNumSelectedItems() const {
  register FXOldIconItem *item=firstitem;
  register FXint n=0;
  while(item){
    if(item->state&OLDICONITEM_SELECTED) n++;
    item=item->next;
    }
  return n;
  }


// Get list of selected ite,s
FXOldIconItem** FXOldIconList::getSelectedItems() const {
  FXOldIconItem *item=firstitem;
  FXOldIconItem **itemlist;
  FXint i=0;
  FXMALLOC(&itemlist,FXOldIconItem*,getNumSelectedItems()+1);
  while(item){
    if(item->state&OLDICONITEM_SELECTED) itemlist[i++]=item;
    item=item->next;
    }
  itemlist[i]=NULL;
  return itemlist;
  }
 

// Helper function
static FXint itemnamelen(const FXchar* name){
  register FXint len;
  for(len=0; name[len] && name[len]!='\t'; len++);
  return len;
  }


// Get item X
FXint FXOldIconList::getItemX(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::getItemX: item is NULL.\n",getClassName()); } 
  return item->x; 
  }


// Get item Y
FXint FXOldIconList::getItemY(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::getItemY: item is NULL.\n",getClassName()); } 
  return item->y; 
  }


// Get item width
FXint FXOldIconList::getItemWidth(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::getItemWidth: item is NULL.\n",getClassName()); } 
  return itemWidth; 
  }


// Get Item height
FXint FXOldIconList::getItemHeight(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::getItemHeight: item is NULL.\n",getClassName()); } 
  return itemHeight; 
  }


// True if item is selected
FXbool FXOldIconList::isItemSelected(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::isItemSelected: item is NULL.\n",getClassName()); } 
  return (item->state&OLDICONITEM_SELECTED)!=0; 
  }


// True if item is current
FXbool FXOldIconList::isItemCurrent(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::isItemCurrent: item is NULL.\n",getClassName()); } 
  return (item->state&OLDICONITEM_CURRENT)!=0; 
  }


// True if item visible
FXbool FXOldIconList::isItemVisible(const FXOldIconItem* item) const {
  if(!item){ fxerror("%s::isItemVisible: item is NULL.\n",getClassName()); } 
  return 0<(pos_x+getItemX(item)+getItemWidth(item)) && (pos_x+getItemX(item))<viewport_w && 0<(pos_y+getItemY(item)+getItemHeight(item)) && (pos_y+getItemY(item))<viewport_h;
  }  


// Make item fully visible
void FXOldIconList::makeItemVisible(FXOldIconItem* item){
  register FXint x,y,w,h,hh,px,py;
  if(!xid) return;
  if(item){
    px=pos_x;
    py=pos_y;
    if(options&(OLDICONLIST_BIG_ICONS|OLDICONLIST_MINI_ICONS)){
      x=getItemX(item);
      y=getItemY(item);
      w=getItemWidth(item);
      h=getItemHeight(item);
      if(px+x+w >= viewport_w) px=viewport_w-x-w;
      if(px+x <= 0) px=-x;
      if(py+y+h >= viewport_h) py=viewport_h-y-h;
      if(py+y <= 0) py=-y;
      }
    else{
      y=getItemY(item);
      h=getItemHeight(item);
      hh=header->getDefaultHeight();
      if(py+y+h >= viewport_h+hh) py=hh+viewport_h-y-h;
      if(py+y <= hh) py=hh-y;
      }
    setPosition(px,py);
    }
  }


// Get item at position x,y (NULL if none)
FXOldIconItem* FXOldIconList::getItemAt(FXint x,FXint y) const {
  register FXint off,to,tw;
  register FXOldIconItem* n;
  y-=pos_y;
  x-=pos_x;
  if(options&OLDICONLIST_BIG_ICONS){
    off=(itemSpace-iconWidth)/2;
    for(n=firstitem; n; n=n->next){
      if(y<n->y || n->y+itemHeight<=y || x<n->x || n->x+itemWidth<=x) continue;
      if(y<n->y+iconHeight && (x<n->x+off || n->x+off+iconWidth<=x)) continue;
      tw=font->getTextWidth(n->label.text(),itemnamelen(n->label.text()));
      if(tw>itemWidth) tw=itemWidth;
      to=(itemWidth-tw)/2;
      if(x<n->x+to || n->x+to+tw<=x) continue;
      return n;
      }
    }
  else if(options&OLDICONLIST_MINI_ICONS){
    for(n=firstitem; n; n=n->next){
      if(x<n->x || n->x+itemWidth<=x || y<n->y || n->y+itemHeight<=y) continue;
      if(x<n->x+miniIconWidth+4+font->getTextWidth(n->label.text(),itemnamelen(n->label.text()))) return n;
      }
    }
  else{
    for(n=firstitem; n; n=n->next){
      if(n->y<=y && y<n->y+itemHeight) return n;
      }
    }
  return NULL;
  }


// Did we hit the item icon
FXbool FXOldIconList::hitItemIcon(const FXOldIconItem* item,FXint x,FXint y) const {
  register FXint off;
  if(!item){ fxerror("%s::hitItemIcon: item is NULL.\n",getClassName()); }
  x=x-pos_x;
  y=y-pos_y;
  if(options&OLDICONLIST_BIG_ICONS){
    off=(itemSpace-iconWidth)/2;
    if(item->y<=y && y<item->y+iconHeight && item->x+off<=x && x<item->x+off+iconWidth) return TRUE;
    }
  else if(options&OLDICONLIST_MINI_ICONS){
    if(item->y<=y && y<item->y+miniIconHeight && item->x<=x && x<item->x+miniIconWidth) return TRUE;
    }
  else{
    if(item->y<=y && y<item->y+itemHeight && item->x<=x && x<item->x+miniIconWidth) return TRUE;
    }
  return FALSE;
  }


// Did we hit the item
FXbool FXOldIconList::hitItem(const FXOldIconItem* item,FXint x,FXint y) const {
  register FXint off,to,tw;
  if(!item){ fxerror("%s::hitItem: item is NULL.\n",getClassName()); } 
  x=x-pos_x;
  y=y-pos_y;
  if(options&OLDICONLIST_BIG_ICONS){
    off=(itemSpace-iconWidth)/2;
    if(item->y<=y && y<item->y+itemHeight && item->x<=x && x<item->x+itemWidth){
      if(y<item->y+iconHeight){
        if(item->x+off<=x && x<item->x+off+iconWidth) return TRUE;
        }
      else{
        tw=font->getTextWidth(item->label.text(),itemnamelen(item->label.text()));
        if(tw>itemWidth) tw=itemWidth;
        to=(itemWidth-tw)/2;
        if(item->x+to<=x && x<item->x+to+tw) return TRUE;
        }
      }
    }
  else if(options&OLDICONLIST_MINI_ICONS){
    if(item->y<=y && y<item->y+itemHeight && item->x<=x && x<item->x+miniIconWidth+4+font->getTextWidth(item->label.text(),itemnamelen(item->label.text()))) return TRUE;
    }
  else{
    if(item->y<=y && y<item->y+itemHeight) return TRUE;
    }
  return FALSE;
  }


// Get next item
FXOldIconItem* FXOldIconList::getNextItem(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::getNextItem: item is NULL.\n",getClassName()); } 
  return item->next; 
  }


// Get previous item
FXOldIconItem* FXOldIconList::getPrevItem(const FXOldIconItem* item) const { 
  if(!item){ fxerror("%s::getPrevItem: item is NULL.\n",getClassName()); } 
  return item->prev; 
  }


// Get index of item
FXint FXOldIconList::indexOfItem(const FXOldIconItem* item) const {
  register FXint index=-1;
  while(item){item=item->prev;index++;}
  return index;
  }


// Get item at index
FXOldIconItem* FXOldIconList::itemAtIndex(FXint index) const {
  register FXOldIconItem* item=firstitem;
  if(index<0) return NULL;
  while(index && item){item=item->next;index--;}
  return item;
  }


// Sort items
void FXOldIconList::sort(FXOldIconItem*& f1,FXOldIconItem*& t1,FXOldIconItem*& f2,FXOldIconItem*& t2,int n){
  FXOldIconItem *ff1,*tt1,*ff2,*tt2,*q;
  FXint m;
  if(f2==NULL){ 
    f1=NULL; 
    t1=NULL; 
    return; 
    }
  if(n>1){
    m=n/2;
    n=n-m;
    sort(ff1,tt1,f2,t2,n);  // 1 or more
    sort(ff2,tt2,f2,t2,m);  // 0 or more
    FXASSERT(ff1);
    if(ff2 && sortfunc(ff1,ff2)){
      f1=ff2;
      ff2->prev=NULL;
      ff2=ff2->next;
      }
    else{
      f1=ff1;
      ff1->prev=NULL;
      ff1=ff1->next;
      }
    t1=f1;
    t1->next=NULL;
    while(ff1 || ff2){
      if(ff1==NULL){ t1->next=ff2; ff2->prev=t1; t1=tt2; break; }
      if(ff2==NULL){ t1->next=ff1; ff1->prev=t1; t1=tt1; break; }
      if(sortfunc(ff1,ff2)){
        t1->next=ff2;
        ff2->prev=t1;
        t1=ff2;
        ff2=ff2->next;
        }
      else{
        t1->next=ff1;
        ff1->prev=t1;
        t1=ff1;
        ff1=ff1->next;
        }
      t1->next=NULL;
      }
    return;
    }
  FXASSERT(f2);
  f1=f2;
  t1=f2;
  f2=f2->next;
  while(f2){
    f2->prev=NULL;
    if(sortfunc(f2,t1)){
      t1->next=f2;
      f2->prev=t1;
      t1=f2;
      f2=f2->next;
      continue;
      }
    if(sortfunc(f1,f2)){
      q=f2;
      f2=f2->next;
      q->next=f1;
      f1->prev=q;
      f1=q;
      continue;
      }
    break;
    }
  FXASSERT(f1);
  FXASSERT(t1);
  f1->prev=NULL;
  t1->next=NULL;
  }


// Sort the items based on the sort function
void FXOldIconList::sortItems(){
  if(sortfunc){
    FXOldIconItem *f=firstitem;
    FXOldIconItem *l=lastitem;
    sort(firstitem,lastitem,f,l,getNumItems());
    recalc();
    }
  }


// Repaint
void FXOldIconList::updateItem(FXOldIconItem* item){
  if(xid){update(pos_x+getItemX(item),pos_y+getItemY(item),getItemWidth(item),getItemHeight(item));}
  }


// All turned on
FXbool FXOldIconList::selectItemRange(FXOldIconItem* beg,FXOldIconItem* end){
  register FXbool selected=FALSE;
  if(beg && end){
    register FXOldIconItem *item=beg;
    do{
      selected|=selectItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return selected;
  }


// All turned off
FXbool FXOldIconList::deselectItemRange(FXOldIconItem* beg,FXOldIconItem* end){
  register FXbool deselected=FALSE;
  if(beg && end){
    register FXOldIconItem *item=beg;
    do{
      deselected|=deselectItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return deselected;
  }


// Toggle items
FXbool FXOldIconList::toggleItemRange(FXOldIconItem* beg,FXOldIconItem* end){
  register FXbool toggled=FALSE;
  if(beg && end){
    register FXOldIconItem *item=beg;
    do{
      toggled|=toggleItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return toggled;
  }


// Restore items to previous state
FXbool FXOldIconList::restoreItemRange(FXOldIconItem* beg,FXOldIconItem* end){
  register FXbool restored=FALSE;
  if(beg && end){
    register FXOldIconItem *item=beg;
    do{
      restored|=restoreItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return restored;
  }


// Select one item
FXbool FXOldIconList::selectItem(FXOldIconItem* item){
  if(item){
    if(!(item->state&OLDICONITEM_SELECTED)){
      item->state|=OLDICONITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// Deselect one item
FXbool FXOldIconList::deselectItem(FXOldIconItem* item){
  if(item){
    if(item->state&OLDICONITEM_SELECTED){
      item->state&=~OLDICONITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// toggle one item
FXbool FXOldIconList::toggleItem(FXOldIconItem* item){
  if(item){
    item->state^=OLDICONITEM_SELECTED;
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Restore items to previous state
FXbool FXOldIconList::restoreItem(FXOldIconItem* item){
  if(item){
    if((item->state&OLDICONITEM_MARK) && !(item->state&OLDICONITEM_SELECTED)){
      item->state|=OLDICONITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    if(!(item->state&OLDICONITEM_MARK) && (item->state&OLDICONITEM_SELECTED)){
      item->state&=~OLDICONITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// Mark items
void FXOldIconList::markItems(){
  register FXOldIconItem* item;
  for(item=firstitem; item; item=item->next){
    (item->state&OLDICONITEM_SELECTED)?item->state|=OLDICONITEM_MARK:item->state&=~OLDICONITEM_MARK;
    }
  }


// Little helper function; 1=select, 0=deselect, 2=toggle, 3=restore, 4=nop
FXbool FXOldIconList::performSelectionChange(FXOldIconItem* item,FXuint code){
  switch(code){
    case 0: return deselectItem(item);
    case 1: return selectItem(item);
    case 2: return toggleItem(item);
    case 3: return restoreItem(item);
    }
  return FALSE;
  }


// Select, deselect, or toggle items in rectangle
FXbool FXOldIconList::performSelectionRectangle(FXint lx,FXint ty,FXint rx,FXint by,FXuint in,FXuint out){
  register FXbool changed=FALSE;
  register FXint ix,iy,off;
  register FXOldIconItem* n;
  FXDragType types[2];
  if(lx<rx && ty<by){
    if(options&OLDICONLIST_BIG_ICONS){
      off=(itemSpace-iconWidth)/2;
      for(n=firstitem; n; n=n->next){
        ix=n->x;
        iy=n->y;
        if(ix+off<=rx && iy<=by && lx<=ix+off+iconWidth && ty<=iy+iconHeight){
          changed|=performSelectionChange(n,in);
          }
        else{
          changed|=performSelectionChange(n,out);
          }
        }
      }
    else if(options&OLDICONLIST_MINI_ICONS){
      for(n=firstitem; n; n=n->next){
        ix=n->x;
        iy=n->y;
        if(ix<=rx && iy<=by && lx<=ix+miniIconWidth && ty<=iy+miniIconHeight){
          changed|=performSelectionChange(n,in);
          }
        else{
          changed|=performSelectionChange(n,out);
          }
        }
      }
    else{
      for(n=firstitem; n; n=n->next){
        ix=n->x;
        iy=n->y;
        if(ix<=rx && iy<=by && lx<=ix+itemWidth && ty<=iy+itemHeight){
          changed|=performSelectionChange(n,in);
          }
        else{
          changed|=performSelectionChange(n,out);
          }
        }
      }
    
    // If we want selection, see wether to acquire or release it
    if(options&OLDICONLIST_WANTSELECTION){
      if(getNumSelectedItems()){
        types[0]=stringType;
        types[1]=textType;
        acquireSelection(types,2);
        }
      else{
        releaseSelection();
        }
      }
    }
  return changed;
  }


// Select items in rectangle
FXbool FXOldIconList::selectInRectangle(FXint x,FXint y,FXint w,FXint h){
  return performSelectionRectangle(x,y,x+w-1,y+h-1,1,4);
  }


// Deselect items in rectangle
FXbool FXOldIconList::deselectInRectangle(FXint x,FXint y,FXint w,FXint h){
  return performSelectionRectangle(x,y,x+w-1,y+h-1,0,4);
  }


// Toggle items in rectangle
FXbool FXOldIconList::toggleInRectangle(FXint x,FXint y,FXint w,FXint h){
  return performSelectionRectangle(x,y,x+w-1,y+h-1,2,4);
  }


// Toggle items in rectangle
FXbool FXOldIconList::restoreInRectangle(FXint x,FXint y,FXint w,FXint h){
  return performSelectionRectangle(x,y,x+w-1,y+h-1,3,4);
  }


// Move content
void FXOldIconList::moveContents(FXint x,FXint y){
  FXint dx=x-pos_x;
  FXint dy=y-pos_y;
  FXint top=0;
  pos_x=x;
  pos_y=y;
  if(!(options&(OLDICONLIST_MINI_ICONS|OLDICONLIST_BIG_ICONS))){
    top=header->getDefaultHeight();
    header->move(x,0);          // Perhaps just draw those headers??
    }
  scroll(0,top,viewport_w,viewport_h,dx,dy);
  }


// Size of a possible column caption
FXint FXOldIconList::getViewportHeight(){
  return (options&(OLDICONLIST_MINI_ICONS|OLDICONLIST_BIG_ICONS)) ? height : height-header->getDefaultHeight();
  }


// Propagate size change
void FXOldIconList::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  }


// Recompute interior 
void FXOldIconList::recompute(){
  register FXint y,nh,nv,r,k,th,num;
  register FXOldIconItem* n;

  totalWidth=0;
  totalHeight=0;
  th=1+font->getFontHeight();
  num=getNumItems();
  
  // Big Icons
  if(options&OLDICONLIST_BIG_ICONS){
    itemHeight=iconHeight+th+BIG_LINE_SPACING+BIG_TEXT_SPACING;
    itemWidth=FXMAX(iconWidth,itemSpace)+SIDE_SPACING;
    }
 
  // Mini Icons
  else if(options&OLDICONLIST_MINI_ICONS){
    itemHeight=FXMAX(miniIconHeight,th)+MINI_LINE_SPACING;
    itemWidth=miniIconWidth+itemSpace+MINI_TEXT_SPACING+SIDE_SPACING;
    }
 
  // Detail list
  else{
    itemHeight=FXMAX(miniIconHeight,th)+DETAIL_LINE_SPACING;
    itemWidth=header->getDefaultWidth();
    }
  
  // For icon modes
  if(options&(OLDICONLIST_MINI_ICONS|OLDICONLIST_BIG_ICONS)){

    // Arrangement at random
    if(options&OLDICONLIST_RANDOM){
      for(n=firstitem; n; n=n->next){
        if(n->x+itemWidth>totalWidth) totalWidth=n->x+itemWidth;
        if(n->y+itemHeight>totalHeight) totalHeight=n->y+itemHeight;
        }
      }

    // Arrangement by columns
    else if(options&OLDICONLIST_COLUMNS){
      nh=width/itemWidth;
      if(nh<1) nh=1;
      nv=(num+nh-1)/nh;
      if(nv*itemHeight > height){
        nh=(width-vertical->getDefaultWidth())/itemWidth;
        if(nh<1) nh=1;
        nv=(num+nh-1)/nh;
        }
      for(n=firstitem,r=0,k=0; n; n=n->next){
        n->x=k*itemWidth;
        n->y=r*itemHeight;
        if(++k==nh){k=0;r++;}
        }
      totalWidth=itemWidth*nh;
      totalHeight=itemHeight*nv;
      }

    // Arrangement by rows
    else{
      nv=height/itemHeight;
      if(nv<1) nv=1;
      nh=(num+nv-1)/nv;
      if(nh*itemWidth > width){
        nv=(height-horizontal->getDefaultHeight())/itemHeight;
        if(nv<1) nv=1;
        nh=(num+nv-1)/nv;
        }
      for(n=firstitem,r=0,k=0; n; n=n->next){
        n->x=k*itemWidth;
        n->y=r*itemHeight;
        if(++r==nv){r=0;k++;}
        }
      totalWidth=itemWidth*nh;
      totalHeight=itemHeight*nv;
      }
    }

  // For detail mode
  else{
    for(n=firstitem,y=header->getDefaultHeight(); n; y+=itemHeight,n=n->next){
      n->x=0;
      n->y=y;
      }
    totalWidth=itemWidth;
    totalHeight=num*itemHeight;//+header->getDefaultHeight();
    }
  
  // No more recalc 
  flags&=~FLAG_RECALC;
  }


// Determine content width of tree list
FXint FXOldIconList::getContentWidth(){
  if(flags&FLAG_RECALC) recompute();
  return totalWidth;
  }


// Determine content height of tree list
FXint FXOldIconList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return totalHeight;  
  }


// Recalculate layout
void FXOldIconList::layout(){
  FXint ww;
  
  // Should be set ONLY when in row/col mode & rows/cols have changed
  flags|=FLAG_RECALC;
  
  // Update scroll bars
  FXScrollArea::layout();

  // In detail mode
  if(!(options&(OLDICONLIST_MINI_ICONS|OLDICONLIST_BIG_ICONS))){ 
    ww=header->getDefaultWidth();
    if(ww<width) ww=width;
    header->position(pos_x,0,ww,header->getDefaultHeight());
    header->show();
    if(itemWidth<viewport_w) itemWidth=viewport_w;
    }
  else{
    header->hide();
    }

  // Set line size
  vertical->setLine(itemHeight);
  horizontal->setLine(1);
  
  // Force repaint
  update();
  
  flags&=~FLAG_DIRTY;
  }


// Start motion timer while in this window
long FXOldIconList::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onEnter(sender,sel,ptr);
  if(!timer){timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);}
  return 1;
  }


// Stop motion timer when leaving window
long FXOldIconList::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onLeave(sender,sel,ptr);
  if(timer){getApp()->removeTimeout(timer);timer=NULL;}
  return 1;
  }


// Gained focus
long FXOldIconList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(currentitem) updateItem(currentitem);
  return 1;
  }


// Lost focus
long FXOldIconList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(currentitem) updateItem(currentitem);
  return 1;
  }


// We have the selection
long FXOldIconList::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionGained(sender,sel,ptr);
  update();
  return 1;
  }


// We lost the selection
long FXOldIconList::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionLost(sender,sel,ptr);
  deselectItemRange(firstitem,lastitem);
  return 1;
  }


// Draw dashed focus rectangle
void FXOldIconList::drawFocusRectangle(FXDCWindow& dc,FXint x,FXint y,FXint w,FXint h){
//   static const char onoff[]={1,2};
//   dc.setForeground(textColor);
//   dc.setDashes(0,onoff,2);
//   dc.setLineStyle(LINE_ONOFF_DASH);
//   dc.drawLine(x,y,x+w-1,y);
//   dc.drawLine(x,y,x,y+h-1);
//   dc.drawLine(x,y+h-1,x+w-1,y+h-1);
//   dc.drawLine(x+w-1,y,x+w-1,y+h-1);
//   dc.setLineStyle(LINE_SOLID);
  dc.setFillStyle(FILL_STIPPLED);
  dc.setStipple(STIPPLE_GRAY);
  dc.setForeground(textColor);
  dc.drawRectangle(x,y,w-1,h-1);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }


// Draw Lasso rectangle
void FXOldIconList::drawLasso(FXint x0,FXint y0,FXint x1,FXint y1){
  FXDCWindow dc(this);
  dc.setFunction(BLT_NOT_DST);
  x0+=pos_x;
  x1+=pos_x;
  y0+=pos_y;
  y1+=pos_y;
  dc.drawLine(x0,y0,x1,y0);
  dc.drawLine(x1,y0,x1,y1);
  dc.drawLine(x1,y1,x0,y1);
  dc.drawLine(x0,y1,x0,y0);
  }


// Draw big icon
void FXOldIconList::drawBigIcon(FXDCWindow& dc,FXint x,FXint y,FXOldIconItem* item){
  register FXint iconoffx,iconoffy,textoffx,textoffy,tw,th,dw,tlen,tdrw,s;
  register const FXchar* text;
  if(item->icon){
    iconoffx=x+((itemSpace-item->icon->getWidth())/2)+(SIDE_SPACING/2);
    iconoffy=y+(BIG_LINE_SPACING/2);
    if(item->state&OLDICONITEM_SELECTED){
      dc.drawIconShaded(item->icon,iconoffx,iconoffy);
      }
    else{
      dc.drawIcon(item->icon,iconoffx,iconoffy);
      }
    }
  text=item->label.text();
  if(text){
    for(tlen=0; text[tlen] && text[tlen]!='\t'; tlen++);
    if(tlen>0){
      th=font->getFontHeight();
      tw=font->getTextWidth(text,tlen);
      tdrw=tlen;
      dw=0;
      if(tw>itemSpace){
        dw=font->getTextWidth("...",3);
        s=itemSpace-dw;
        while((tw=font->getTextWidth(text,tdrw))>s && tdrw>1) --tdrw;
        if(tw>s) dw=0;
        }
      if(tw<=itemSpace){
        textoffy=y+iconHeight+BIG_TEXT_SPACING;
        textoffx=x+((itemSpace-tw-dw)/2)+(SIDE_SPACING/2);
        if(item->state&OLDICONITEM_SELECTED){
          if(hasFocus() && (item->state&OLDICONITEM_CURRENT)){
            dc.setForeground(selbackColor);
            dc.fillRectangle(textoffx+1,textoffy+1,tw+dw,th-1);
            drawFocusRectangle(dc,textoffx,textoffy,tw+dw+2,th+1);
            }
          else{
            dc.setForeground(selbackColor);
            dc.fillRectangle(textoffx,textoffy,tw+dw+2,th+1);
            }
          dc.setForeground(seltextColor);
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),text,tdrw);
          if(dw) dc.drawText(textoffx+tw+1,textoffy+font->getFontAscent(),"...",3);
          }
        else{
          if(hasFocus() && (item->state&OLDICONITEM_CURRENT)){
            drawFocusRectangle(dc,textoffx,textoffy,tw+dw+2,th+1);
            }
          dc.setForeground(textColor);
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),text,tdrw);
          if(dw) dc.drawText(textoffx+tw+1,textoffy+font->getFontAscent(),"...",3);
          }
        }
      }
    }
  }


// Draw mini icon
void FXOldIconList::drawMiniIcon(FXDCWindow& dc,FXint x,FXint y,FXOldIconItem* item){
  register FXint iconoffx,iconoffy,textoffx,textoffy,tw,th,dw,tlen,tdrw,s;
  register const FXchar* text;
  if(item->miniIcon){
    iconoffx=x+(SIDE_SPACING/2);
    iconoffy=y+(MINI_LINE_SPACING/2)+(itemHeight-item->miniIcon->getHeight())/2;
    if(item->state&OLDICONITEM_SELECTED){
      dc.drawIconShaded(item->miniIcon,iconoffx,iconoffy);
      }
    else{
      dc.drawIcon(item->miniIcon,iconoffx,iconoffy);
      }
    }
  text=item->label.text();
  if(text){
    for(tlen=0; text[tlen] && text[tlen]!='\t'; tlen++);
    if(tlen>0){
      th=font->getFontHeight();
      tw=font->getTextWidth(text,tlen);
      tdrw=tlen;
      dw=0;
      if(tw>itemSpace){
        dw=font->getTextWidth("...",3);
        s=itemSpace-dw;
        while((tw=font->getTextWidth(text,tdrw))>s && tdrw>1) --tdrw;
        if(tw>s) dw=0;
        }
      if(tw<=itemSpace){
        textoffx=x+miniIconWidth+(SIDE_SPACING/2)+MINI_TEXT_SPACING;
        textoffy=y+(MINI_LINE_SPACING/2)+(itemHeight-th)/2;
        if(item->state&OLDICONITEM_SELECTED){
          if(hasFocus() && (item->state&OLDICONITEM_CURRENT)){
            dc.setForeground(selbackColor);
            dc.fillRectangle(textoffx+1,textoffy+1,tw+dw,th-1);
            drawFocusRectangle(dc,textoffx,textoffy,tw+dw+2,th+1);
            }
          else{
            dc.setForeground(selbackColor);
            dc.fillRectangle(textoffx,textoffy,tw+dw+2,th+1);
            }
          dc.setForeground(seltextColor);
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),text,tdrw);
          if(dw) dc.drawText(textoffx+tw+1,textoffy+font->getFontAscent(),"...",3);
          }
        else{
          if(hasFocus() && (item->state&OLDICONITEM_CURRENT)){
            drawFocusRectangle(dc,textoffx,textoffy,tw+dw+2,th+1);
            }
          dc.setForeground(textColor);
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),text,tdrw);
          if(dw) dc.drawText(textoffx+tw+1,textoffy+font->getFontAscent(),"...",3);
          }
        }
      }
    }
  }


// Draw detail
void FXOldIconList::drawDetails(FXDCWindow& dc,FXint x,FXint y,FXOldIconItem* item){
  register FXint iconoffx,iconoffy,textoffx,textoffy,tw,th,dw,ddw,tlen,tdrw,space,s;
  register const FXchar* text=item->label.text();
  FXOldHeaderItem* hi=header->getFirstItem();
  if(!hi) return;
  if(item->miniIcon){
    iconoffx=x+(SIDE_SPACING/2);
    iconoffy=y+(DETAIL_LINE_SPACING/2)+(itemHeight-item->miniIcon->getHeight())/2;
    if(item->state&OLDICONITEM_SELECTED){
      dc.drawIconShaded(item->miniIcon,iconoffx,iconoffy);
      }
    else{
      dc.drawIcon(item->miniIcon,iconoffx,iconoffy);
      }
    }
  if(text){
    th=font->getFontHeight();
    textoffx=x+miniIconWidth+(SIDE_SPACING/2)+DETAIL_TEXT_SPACING;
    textoffy=y+(DETAIL_LINE_SPACING/2)+(itemHeight-th)/2;
    if(item->state&OLDICONITEM_SELECTED){
      if(hasFocus() && (item->state&OLDICONITEM_CURRENT)){
        dc.setForeground(selbackColor);
        dc.fillRectangle(textoffx+1,textoffy+1,itemWidth-2,th-1);
        drawFocusRectangle(dc,textoffx,textoffy,itemWidth,th+1);
        }
      else{
        dc.setForeground(selbackColor);
        dc.fillRectangle(textoffx,textoffy,itemWidth,th+1);
        }
      }
    else{
      if(hasFocus() && (item->state&OLDICONITEM_CURRENT)){
        drawFocusRectangle(dc,textoffx,textoffy,itemWidth,th+1);
        }
      }
    dc.setForeground((item->state&OLDICONITEM_SELECTED) ? seltextColor : textColor);
    ddw=font->getTextWidth("...",3);
    space=header->getItemWidth(hi)-miniIconWidth-DETAIL_TEXT_SPACING;
    while(1){
      for(tlen=0; text[tlen] && text[tlen]!='\t'; tlen++);
      if(tlen>0){
        tw=font->getTextWidth(text,tlen);
        tdrw=tlen;
        dw=0;
        if(tw>(space-10)){
          dw=ddw;
          s=space-10-dw;
          while((tw=font->getTextWidth(text,tdrw))>s && tdrw>1) --tdrw;
          if(tw>(space-10)) dw=0;
          }
        if(tw<=(space-10)){
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),text,tdrw);
          if(dw) dc.drawText(textoffx+tw+1,textoffy+font->getFontAscent(),"...",3);
          }
        }
      if(!text[tlen]) break;
      hi=header->getNextItem(hi);
      if(!hi) break;
      textoffx+=space;
      space=header->getItemWidth(hi);
      text=text+tlen+1;
      }
    }
  }


// Draw item list
long FXOldIconList::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  register FXint xmin,xmax,ymin,ymax;
  register FXOldIconItem* n;
  register FXint x,y;
  FXDCWindow dc(this,event);
  
  
  dc.setForeground(backColor);
  dc.fillRectangle(event->rect.x,event->rect.y,event->rect.w,event->rect.h);
  dc.setTextFont(font);
  
  // Possible overlap
  xmin=event->rect.x-itemWidth;
  ymin=event->rect.y-itemHeight;
  xmax=event->rect.x+event->rect.w;
  ymax=event->rect.y+event->rect.h;
 
  // Big icons
  if(options&OLDICONLIST_BIG_ICONS){
    for(n=firstitem; n; n=n->next){
      x=pos_x+n->x;
      y=pos_y+n->y;
      if(x<=xmax && y<=ymax && xmin<=x && ymin<=y) drawBigIcon(dc,x,y,n);
      }
    }
 
  // Mini icons
  else if(options&OLDICONLIST_MINI_ICONS){
    for(n=firstitem; n; n=n->next){
      x=pos_x+n->x;
      y=pos_y+n->y;
      if(x<=xmax && y<=ymax && xmin<=x && ymin<=y) drawMiniIcon(dc,x,y,n);
      }
    }
 
  // Detail mode
  else{
    for(n=firstitem; n; n=n->next){
      x=pos_x+n->x;
      y=pos_y+n->y;
      if(y<=ymax && ymin<=y) drawDetails(dc,x,y,n);
      }
    }
  return 1;
  }


// We were asked about tip text
long FXOldIconList::onQueryTip(FXObject* sender,FXSelector,void*){
  FXint x,y; FXuint state;
  if(flags&FLAG_TIP){
    getCursorPosition(x,y,state);
    FXTRACE((250,"%s::onQueryTip %08x (%d,%d)\n",getClassName(),this,x,y));
    FXOldIconItem *item=getItemAt(x,y);
    if(item){
      FXString string=item->label.extract(0,'\t');
      sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&string);
      return 1;
      }
    }
  return 0;
  }


// We were asked about status text
long FXOldIconList::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(help.text() && (flags&FLAG_HELP)){
    FXTRACE((250,"%s::onQueryHelp %08x\n",getClassName(),this));
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&help);
    return 1;
    }
  return 0;
  }


// Arrange by rows
long FXOldIconList::onCmdArrangeByRows(FXObject*,FXSelector,void*){
  options&=~OLDICONLIST_COLUMNS;
  options&=~OLDICONLIST_RANDOM;
  recalc();
  return 1;
  }


// Update sender
long FXOldIconList::onUpdArrangeByRows(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&(OLDICONLIST_COLUMNS|OLDICONLIST_RANDOM))?ID_UNCHECK:ID_CHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Arrange by columns
long FXOldIconList::onCmdArrangeByColumns(FXObject*,FXSelector,void*){
  options&=~OLDICONLIST_RANDOM;
  options|=OLDICONLIST_COLUMNS;
  recalc();
  return 1;
  }


// Update sender
long FXOldIconList::onUpdArrangeByColumns(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&OLDICONLIST_COLUMNS)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Arrange randomly
long FXOldIconList::onCmdArrangeAtRandom(FXObject*,FXSelector,void*){
  options&=~OLDICONLIST_COLUMNS;
  options|=OLDICONLIST_RANDOM;
  recalc();
  return 1;
  }


// Update sender
long FXOldIconList::onUpdArrangeAtRandom(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&OLDICONLIST_RANDOM)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Show detailed list
long FXOldIconList::onCmdShowDetails(FXObject*,FXSelector,void*){
  options&=~OLDICONLIST_MINI_ICONS;
  options&=~OLDICONLIST_BIG_ICONS;
  options&=~OLDICONLIST_RANDOM;
  recalc();
  return 1;
  }


// Update sender
long FXOldIconList::onUpdShowDetails(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&(OLDICONLIST_MINI_ICONS|OLDICONLIST_BIG_ICONS))?ID_UNCHECK:ID_CHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Show big icons
long FXOldIconList::onCmdShowBigIcons(FXObject*,FXSelector,void*){
  options&=~OLDICONLIST_MINI_ICONS;
  options|=OLDICONLIST_BIG_ICONS;
  options&=~OLDICONLIST_RANDOM;
  recalc();
  return 1;
  }


// Update sender
long FXOldIconList::onUpdShowBigIcons(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&OLDICONLIST_BIG_ICONS)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Show small icons
long FXOldIconList::onCmdShowMiniIcons(FXObject*,FXSelector,void*){
  options|=OLDICONLIST_MINI_ICONS;
  options&=~OLDICONLIST_BIG_ICONS;
  options&=~OLDICONLIST_RANDOM;
  recalc();
  return 1;
  }


// Update sender
long FXOldIconList::onUpdShowMiniIcons(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=(options&OLDICONLIST_MINI_ICONS)?ID_CHECK:ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Select all items
long FXOldIconList::onCmdSelectAll(FXObject*,FXSelector,void*){
  selectItemRange(firstitem,lastitem);
  return 1;
  }


// Deselect all items
long FXOldIconList::onCmdDeselectAll(FXObject*,FXSelector,void*){
  deselectItemRange(firstitem,lastitem);
  return 1;
  }


// Select inverse of current selection
long FXOldIconList::onCmdSelectInverse(FXObject*,FXSelector,void*){
  toggleItemRange(firstitem,lastitem);
  return 1;
  }


// Header subdivision has changed
long FXOldIconList::onHeaderChanged(FXObject*,FXSelector,void*){
  recalc();
  return 1;
  }


// Add item as first one under parent p
FXOldIconItem* FXOldIconList::addItemFirst(const char* text,FXIcon* ic,FXIcon* mi,void* ptr){
  FXOldIconItem *item=createItem();
  item->prev=NULL;
  item->next=firstitem;
  if(item->next) item->next->prev=item; else lastitem=item;
  firstitem=item;
  item->label=text;
  item->icon=ic;
  item->miniIcon=mi;
  item->state=OLDICONITEM_DRAGGABLE;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Add item as last one under parent p
FXOldIconItem* FXOldIconList::addItemLast(const char* text,FXIcon* ic,FXIcon* mi,void* ptr){
  FXOldIconItem *item=createItem();
  item->prev=lastitem;
  item->next=NULL;
  if(item->prev) item->prev->next=item; else firstitem=item;
  lastitem=item;
  item->label=text;
  item->icon=ic;
  item->miniIcon=mi;
  item->state=OLDICONITEM_DRAGGABLE;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Link item after other
FXOldIconItem* FXOldIconList::addItemAfter(FXOldIconItem* other,const char* text,FXIcon* ic,FXIcon* mi,void* ptr){
  if(other==NULL){ fxerror("%s::addItemAfter: other item is NULL.\n",getClassName()); }
  FXOldIconItem *item=createItem();
  item->prev=other;
  item->next=other->next;
  other->next=item;
  if(item->next) item->next->prev=item; else lastitem=item; 
  item->label=text;
  item->icon=ic;
  item->miniIcon=mi;
  item->state=OLDICONITEM_DRAGGABLE;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Link item before other 
FXOldIconItem* FXOldIconList::addItemBefore(FXOldIconItem* other,const char* text,FXIcon* ic,FXIcon* mi,void* ptr){
  if(other==NULL){ fxerror("%s::addItemBefore: other item is NULL.\n",getClassName()); }
  FXOldIconItem *item=createItem();
  item->next=other;
  item->prev=other->prev;
  other->prev=item;
  if(item->prev) item->prev->next=item; else firstitem=item;
  item->label=text;
  item->icon=ic;
  item->miniIcon=mi;
  item->state=OLDICONITEM_DRAGGABLE;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Remove node from list
void FXOldIconList::removeItem(FXOldIconItem* item){
  if(item){
    if(item->prev) item->prev->next=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else lastitem=item->prev;
    if(currentitem==item) currentitem=NULL;
    if(anchoritem==item) anchoritem=NULL;
    deleteItem(item);
    recalc();
    }
  }


// Remove all siblings from [fm,to]
void FXOldIconList::removeItems(FXOldIconItem* fm,FXOldIconItem* to){
  if(fm && to){
    register FXOldIconItem *item;
    if(fm->prev) fm->prev->next=to->next; else firstitem=to->next;
    if(to->next) to->next->prev=fm->prev; else lastitem=fm->prev;
    do{
      item=fm;
      if(currentitem==item) currentitem=NULL;
      if(anchoritem==item) anchoritem=NULL;
      fm=fm->next;
      deleteItem(item);
      }
    while(item!=to);
    recalc();
    }
  }



// Remove all items
void FXOldIconList::removeAllItems(){
  register FXOldIconItem *item;
  while(firstitem){
    item=firstitem;
    firstitem=firstitem->next;
    deleteItem(item);
    }
  firstitem=NULL;
  lastitem=NULL;
  currentitem=NULL;
  anchoritem=NULL;
  recalc();
  }


// Set current item
void FXOldIconList::setCurrentItem(FXOldIconItem* item){
  if(item!=currentitem){ 
    if(currentitem){
      currentitem->state&=~OLDICONITEM_CURRENT;
      updateItem(currentitem);
      }
    currentitem=item;
    if(currentitem){
      currentitem->state|=OLDICONITEM_CURRENT;
      updateItem(currentitem);
      }
    }
  }


// Set anchor item
void FXOldIconList::setAnchorItem(FXOldIconItem* item){
  anchoritem=item;
  }


void FXOldIconList::setItemText(FXOldIconItem* item,const FXString& text){
  if(!item){ fxerror("%s::setItemText: NULL item argument.\n",getClassName()); } 
  item->label=text;
  recalc();
  }


FXString FXOldIconList::getItemText(const FXOldIconItem* item) const {
  if(!item){ fxerror("%s::getItemText: NULL item argument.\n",getClassName()); } 
  return item->label;
  }


void FXOldIconList::setItemMiniIcon(FXOldIconItem* item,FXIcon* icon){
  if(!item){ fxerror("%s::setItemMiniIcon: NULL item argument.\n",getClassName()); } 
  item->miniIcon=icon;
  updateItem(item);
  }


FXIcon* FXOldIconList::getItemMiniIcon(const FXOldIconItem* item) const {
  if(!item){ fxerror("%s::getItemMiniIcon: NULL item argument.\n",getClassName()); } 
  return item->miniIcon;
  }


void FXOldIconList::setItemIcon(FXOldIconItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemIcon: NULL item argument.\n",getClassName()); }
  item->icon=icon;
  updateItem(item);
  }


FXIcon* FXOldIconList::getItemIcon(const FXOldIconItem* item) const {
  if(item==NULL){ fxerror("%s::getItemIcon: NULL item argument.\n",getClassName()); }
  return item->icon;
  }


void FXOldIconList::setItemData(FXOldIconItem* item,void* ptr) const {
  if(item==NULL){ fxerror("%s::setItemData: NULL item argument.\n",getClassName()); }
  item->data=ptr;
  }


void* FXOldIconList::getItemData(const FXOldIconItem* item) const {
  if(item==NULL){ fxerror("%s::getItemData: NULL item argument.\n",getClassName()); }
  return item->data;
  }


// Find item by prefix of length len
FXOldIconItem* FXOldIconList::findItem(const FXString& text,FXuint len) const {
  register FXOldIconItem* item;
  for(item=firstitem; item; item=item->next){
    if(equal(text,item->label,len)) return item;
    }
  return NULL;
  }


// Test if a is before b
FXbool FXOldIconList::before(FXOldIconItem* a,FXOldIconItem* b){
  while(a && a->next!=b) a=a->next;
  return a!=NULL;
  }


// Extend selection
FXbool FXOldIconList::extendSelection(FXOldIconItem* item){
  FXbool changes=FALSE;
  FXDragType types[2];
  if(item && anchoritem){

    // Browse select mode
    if((options&OLDICONLIST_BROWSESELECT) && !(options&OLDICONLIST_SINGLESELECT)){
      changes|=deselectItemRange(firstitem,item->prev);
      changes|=deselectItemRange(item->next,lastitem);
      changes|=selectItem(item);
      }

    // Extended number selected
    else if(!(options&(OLDICONLIST_SINGLESELECT|OLDICONLIST_BROWSESELECT))){
      if(item==anchoritem){
        changes|=restoreItemRange(firstitem,anchoritem->prev);
        changes|=restoreItemRange(anchoritem->next,lastitem);
        }
      else if(before(item,anchoritem)){
        changes|=restoreItemRange(firstitem,item->prev);
        changes|=restoreItemRange(anchoritem->next,lastitem);
        if(anchoritem->state&OLDICONITEM_SELECTED){
          changes|=selectItemRange(item,anchoritem->prev);
          }
        else{
          changes|=deselectItemRange(item,anchoritem->prev);
          }
        }
      else{
        changes|=restoreItemRange(firstitem,anchoritem->prev);
        changes|=restoreItemRange(item->next,lastitem);
        if(anchoritem->state&OLDICONITEM_SELECTED){
          changes|=selectItemRange(anchoritem->next,item);
          }
        else{
          changes|=deselectItemRange(anchoritem->next,item);
          }
        }
      }
    
    // If we want selection, see wether to acquire or release it
    if(options&OLDICONLIST_WANTSELECTION){
      if(getNumSelectedItems()){
        types[0]=stringType;
        types[1]=textType;
        acquireSelection(types,2);
        }
      else{
        releaseSelection();
        }
      }
    }
  return changes;
  }


// Key Press
long FXOldIconList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldIconItem *item;
  FXDragAction action;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Right:
    case KEY_Left:
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      item=currentitem;
      switch(event->code){
        case KEY_Right:
          if(!item) item=firstitem; 
          break;
        case KEY_Left:
          if(!item) item=lastitem; 
          break;
        case KEY_Up: 
          if(item && item->prev) item=item->prev; 
          if(!item) item=lastitem; 
          break;
        case KEY_Down:
          if(item && item->next) item=item->next; 
          if(!item) item=firstitem; 
          break;
        case KEY_Home: 
          item=firstitem; 
          break;
        case KEY_End:  
          item=lastitem; 
          break;
        }
      if(item && item!=currentitem){
        handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
        if((options&OLDICONLIST_BROWSESELECT) && !(options&OLDICONLIST_SINGLESELECT)){    // Browse select
          handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
          }
        flags&=~FLAG_UPDATE;
        flags|=FLAG_KEY;
        }
      return 1;
    case KEY_space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      flags|=FLAG_KEY;
      return 1;
    case KEY_KP_Enter:
    case KEY_Return:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      flags|=FLAG_KEY;
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Control_L:
    case KEY_Control_R:
      flags|=FLAG_KEY;
      if(isDragging()){
        action=DRAG_COPY;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndCopyCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
      flags|=FLAG_KEY;
      if(isDragging()){
        action=DRAG_MOVE;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndMoveCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(isDragging()){
        action=DRAG_LINK;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndLinkCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    }
  return 0;
  }


// Key Release 
long FXOldIconList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXDragAction action;
  if(!isEnabled()) return 0;
  flags|=FLAG_UPDATE;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Right:
    case KEY_Left:
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      if(flags&FLAG_KEY){
        if((options&OLDICONLIST_BROWSESELECT) && !(options&OLDICONLIST_SINGLESELECT)){    // Browse select
          handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
          handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
          }
        flags&=~FLAG_KEY;
        }
      return 1;
    case KEY_space:
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
      flags&=~FLAG_KEY;
      return 1;
    case KEY_KP_Enter:
    case KEY_Return:
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)currentitem);
      flags&=~FLAG_KEY;
      return 1;
    case KEY_Page_Up:
    case KEY_Page_Down:
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
      flags&=~FLAG_KEY;
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(isDragging()){
        action=DRAG_COPY;
        if(isDropTarget()) action=DRAG_MOVE;
        if(didAccept()!=DRAG_REJECT){
          if(action==DRAG_MOVE)
            setDragCursor(getApp()->dndMoveCursor);
          else
            setDragCursor(getApp()->dndCopyCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    }
  return 0;
  }


// We timed out, i.e. the user didn't move for a while
long FXOldIconList::onTipTimer(FXObject*,FXSelector,void*){
  FXTRACE((250,"%s::onTipTimer %08x\n",getClassName(),this));
  timer=NULL;
  flags|=FLAG_TIP;
  return 1;
  }


// Timeout
long FXOldIconList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint lx,rx,ty,by;
  
  // Lasso mode
  if(flags&FLAG_LASSO){
    
    // Hide the lasso before scrolling
    drawLasso(anchorx,anchory,currentx,currenty);
    
    // Scroll the content
    FXScrollArea::onAutoScroll(sender,sel,ptr);

    currentx=event->win_x-pos_x;
    currenty=event->win_y-pos_y;
    FXMINMAX(lx,rx,anchorx,currentx);
    FXMINMAX(ty,by,anchory,currenty);

    // Perform selection
    if(event->state&SHIFTMASK){
      performSelectionRectangle(lx,ty,rx,by,1,3);
      }
    else if(event->state&CONTROLMASK){
      performSelectionRectangle(lx,ty,rx,by,0,3);
      }
    else{
      performSelectionRectangle(lx,ty,rx,by,1,3);
      }

    getApp()->flush();

    // Show lasso again
    drawLasso(anchorx,anchory,currentx,currenty);
    
    return 1;
    }
  
  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    
    // Scroll the content
    FXScrollArea::onAutoScroll(sender,sel,ptr);
    
    // Content scrolled, so perhaps something else under cursor
    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
  
    return 1;
    }

  return 1;
  }


// Mouse moved
long FXOldIconList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXint lx,rx,ty,by;
  
  // Lasso mode
  if(flags&FLAG_LASSO){
    flags&=~FLAG_TIP;

    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;

    // Hide lasso
    drawLasso(anchorx,anchory,currentx,currenty);
    currentx=event->win_x-pos_x;
    currenty=event->win_y-pos_y;
    FXMINMAX(lx,rx,anchorx,currentx);
    FXMINMAX(ty,by,anchory,currenty);

    // Perform selection
    if(event->state&SHIFTMASK){
      performSelectionRectangle(lx,ty,rx,by,1,3);
      }
    else if(event->state&CONTROLMASK){
      performSelectionRectangle(lx,ty,rx,by,0,3);
      }
    else{
      performSelectionRectangle(lx,ty,rx,by,1,3);
      }

    getApp()->flush();

    // Show lasso again
    drawLasso(anchorx,anchory,currentx,currenty);
    return 1;
    }

  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    flags&=~FLAG_TIP;

    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,TRUE)) return 1;

    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
    return 1;
    }
  
  // Tentative drag and drop
  if(flags&FLAG_TRYDRAG){
    flags&=~FLAG_TIP;
    if(event->moved){
      flags&=~FLAG_TRYDRAG;
      if(handle(this,MKUINT(0,SEL_BEGINDRAG),ptr)){
        flags|=FLAG_DODRAG;
        }
      }
    return 1;
    }

  // Reset tip timer
  if(timer) getApp()->removeTimeout(timer);
  timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);

  // Movement kills tip
  if(flags&FLAG_TIP){
    flags&=~FLAG_TIP;
    return 1;
    }
  
  // No change
  return 0; 
  }


// Pressed a button
long FXOldIconList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldIconItem *item;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(event->click_count==1){
      item=getItemAt(event->win_x,event->win_y);
      if(item!=currentitem){
        handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
        }
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      if(item){
        if(item->state&OLDICONITEM_DRAGGABLE){
          flags|=(FLAG_PRESSED|FLAG_TRYDRAG);
          }
        }
      else{
        if(!(options&(OLDICONLIST_SINGLESELECT|OLDICONLIST_BROWSESELECT))){
          anchorx=currentx=event->win_x-pos_x;
          anchory=currenty=event->win_y-pos_y;
          drawLasso(anchorx,anchory,currentx,currenty);
          markItems();
          flags|=(FLAG_PRESSED|FLAG_LASSO);
          }
        }
      }
    flags&=~FLAG_UPDATE;
    return 1;
    }
  return 0;
  }


// Released button
long FXOldIconList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(isEnabled()){
    ungrab();
    flags&=~(FLAG_PRESSED|FLAG_TRYDRAG);
    flags|=FLAG_UPDATE;
    stopAutoScroll();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(event->click_count==1){
      if(flags&FLAG_LASSO){
        drawLasso(anchorx,anchory,currentx,currenty);
        handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
        handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
        flags&=~FLAG_LASSO;
        }
      else if(flags&FLAG_DODRAG){
        handle(this,MKUINT(0,SEL_ENDDRAG),ptr);
        handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
        flags&=~FLAG_DODRAG;
        }
      else{
        handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
        handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
        }
      }
    else if(event->click_count==2){
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)currentitem);
      }
    else if(event->click_count==3){
      handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)currentitem);
      }
    return 1; 
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXOldIconList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~(FLAG_DODRAG|FLAG_LASSO|FLAG_TRYDRAG|FLAG_PRESSED|FLAG_CHANGED);
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }



// Button or Key activate
long FXOldIconList::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  register FXOldIconItem* item;
  FXDragType types[2];
  
  // Remember previous state
  for(item=firstitem; item; item=item->next){
    if(item->state&OLDICONITEM_SELECTED)
      item->state|=OLDICONITEM_HISTORY;
    else
      item->state&=~OLDICONITEM_HISTORY;
    }
  
  // Have currentitem
  if(currentitem){
    
    // Multiple selection mode
    if((options&OLDICONLIST_SINGLESELECT)&&(options&OLDICONLIST_BROWSESELECT)){
      if(isItemSelected(currentitem)){
        deselectItem(currentitem);
        }
      else{
        selectItem(currentitem);
        }
      //markItems();
      setAnchorItem(currentitem);
      }

    // Browse select mode
    else if(options&OLDICONLIST_BROWSESELECT){
      deselectItemRange(firstitem,currentitem->prev);
      deselectItemRange(currentitem->next,lastitem);
      selectItem(currentitem);
      setAnchorItem(currentitem);
      }

    // Single selection mode
    else if(options&OLDICONLIST_SINGLESELECT){
      if(isItemSelected(currentitem)){
        deselectItemRange(firstitem,lastitem);
        }
      else{
        deselectItemRange(firstitem,currentitem->prev);
        deselectItemRange(currentitem->next,lastitem);
        selectItem(currentitem);
        }
      setAnchorItem(currentitem);
      }

    // Add selection mode
    else if(event->state&SHIFTMASK){
      if(anchoritem){
        selectItem(anchoritem);
        //markItems();
        extendSelection(currentitem);
        }
      else{
        setAnchorItem(currentitem);
        selectItem(currentitem);
        //markItems();
        }
      }
    
    // Extended selection mode
    else if(!(event->state&CONTROLMASK)){
      if(!isItemSelected(currentitem)){
        deselectItemRange(firstitem,currentitem->prev);
        deselectItemRange(currentitem->next,lastitem);
        selectItem(currentitem);
        }
      //markItems();
      setAnchorItem(currentitem);
      }
    }
  
  // No currentitem & not multiple or browse select mode
  else if(!(options&OLDICONLIST_BROWSESELECT)){
    
    // If not add or toggle mode, turn them off
    if(!(event->state&(SHIFTMASK|CONTROLMASK))){
      deselectItemRange(firstitem,lastitem);
      }
    markItems();
    setAnchorItem(currentitem);
    }
  
  // If we want selection, see wether to acquire or release it
  if(options&OLDICONLIST_WANTSELECTION){
    if(getNumSelectedItems()){
      types[0]=stringType;
      types[1]=textType;
      acquireSelection(types,2);
      }
    else{
      releaseSelection();
      }
    }
  return 1;
  }
  

// Button or Key deactivate
long FXOldIconList::onDeactivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  setAnchorItem(currentitem);
  
  // Toggle selection mode
  if(!event->moved && (event->state&CONTROLMASK) && currentitem){
    if(!(options&(OLDICONLIST_SINGLESELECT|OLDICONLIST_BROWSESELECT))){
      toggleItem(currentitem);
      }
    }
  return 1;
  }


// Current item changed
long FXOldIconList::onChanged(FXObject*,FXSelector,void* ptr){
  makeItemVisible((FXOldIconItem*)ptr);
  setCurrentItem((FXOldIconItem*)ptr);
  if(target) target->handle(this,MKUINT(message,SEL_CHANGED),ptr);
  return 1;
  }


// Selected items
long FXOldIconList::onSelected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_SELECTED),ptr);
  }


// Deselected items
long FXOldIconList::onDeselected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DESELECTED),ptr);
  }


// Clicked in list
long FXOldIconList::onClicked(FXObject*,FXSelector,void* ptr){
  FXOldIconItem **selectedlist,**deselectedlist,*item;
  FXint numselected,numdeselected,i,j;
  
  // Notify target of the click
  if(target){
    
    // Clicked message indicates a click anywhere in the widget
    if(target->handle(this,MKUINT(message,SEL_CLICKED),ptr)) return 1;
    
    // Command message indicates click on an item
    if(ptr && target->handle(this,MKUINT(message,SEL_COMMAND),ptr)) return 1;
    }
    
  // Find out number of items whose selection state changed
  for(item=firstitem,numselected=numdeselected=0; item; item=item->next){
    if((item->state&OLDICONITEM_SELECTED) && !(item->state&OLDICONITEM_HISTORY)) numselected++;
    if(!(item->state&OLDICONITEM_SELECTED) && (item->state&OLDICONITEM_HISTORY)) numdeselected++;
    }
 
  // Make some room
  FXMALLOC(&selectedlist,FXOldIconItem*,numselected+1);
  FXMALLOC(&deselectedlist,FXOldIconItem*,numdeselected+1);
 
  // Add items to the proper lists
  for(item=firstitem,i=j=0; item; item=item->next){
    if((item->state&OLDICONITEM_SELECTED) && !(item->state&OLDICONITEM_HISTORY)) selectedlist[i++]=item;
    if(!(item->state&OLDICONITEM_SELECTED) && (item->state&OLDICONITEM_HISTORY)) deselectedlist[j++]=item;
    }
 
  // Close off lists
  selectedlist[i]=NULL;
  deselectedlist[j]=NULL;
 
  // Tell the target about the newly selected items
  handle(this,MKUINT(0,SEL_SELECTED),selectedlist);
 
  // Tell the target about the newly deselected items
  handle(this,MKUINT(0,SEL_DESELECTED),deselectedlist);
 
  // Free the lists
  FXFREE(&selectedlist);
  FXFREE(&deselectedlist);
  return 1;
  }


// Double Clicked in list; ptr may or may not point to an item
long FXOldIconList::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr);
  }


// Triple Clicked in list; ptr may or may not point to an item
long FXOldIconList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }



// Change the font
void FXOldIconList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  font=fnt;
  recalc();
  update();
  }


// Set text color
void FXOldIconList::setTextColor(FXColor clr){
  if(clr!=textColor){
    textColor=clr;
    update();
    }
  }


// Set select background color
void FXOldIconList::setSelBackColor(FXColor clr){
  if(clr!=selbackColor){
    selbackColor=clr;
    update();
    }
  }


// Set selected text color
void FXOldIconList::setSelTextColor(FXColor clr){
  if(clr!=seltextColor){
    seltextColor=clr;
    update();
    }
  }


// Set text width
void FXOldIconList::setItemSpace(FXint s){
  if(itemSpace!=s){
    itemSpace=s;
    recalc();
    }
  }


// Set BIG icon width
void FXOldIconList::setIconWidth(FXint w){
  if(iconWidth!=w){
    iconWidth=w;
    recalc();
    }
  }


// Set BIG icon height
void FXOldIconList::setIconHeight(FXint h){
  if(iconHeight!=h){
    iconHeight=h;
    recalc();
    }
  }


// Set MINI icon width
void FXOldIconList::setMiniIconWidth(FXint w){
  if(miniIconWidth!=w){
    miniIconWidth=w;
    recalc();
    }
  }


// Set MINI icon height
void FXOldIconList::setMiniIconHeight(FXint h){
  if(miniIconHeight!=h){
    miniIconHeight=h;
    recalc();
    }
  }


// Change list style
void FXOldIconList::setListStyle(FXuint style){
  FXuint opts=(options&~OLDICONLIST_MASK) | (style&OLDICONLIST_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Get list style
FXuint FXOldIconList::getListStyle() const { 
  return (options&OLDICONLIST_MASK); 
  }


// Change help text
void FXOldIconList::setHelpText(const FXString& text){
  help=text;
  }


// Save data
void FXOldIconList::save(FXStream& store) const {
  FXScrollArea::save(store);
  store << header;
  store << font;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << iconWidth;
  store << iconHeight;
  store << miniIconWidth;
  store << miniIconHeight;
  store << itemSpace;
  store << itemWidth;
  store << itemHeight;
  store << totalWidth;
  store << totalHeight;
  store << help;
  }


// Load data
void FXOldIconList::load(FXStream& store){ 
  FXScrollArea::load(store);
  store >> header;
  store >> font;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> iconWidth;
  store >> iconHeight;
  store >> miniIconWidth;
  store >> miniIconHeight;
  store >> itemSpace;
  store >> itemWidth;
  store >> itemHeight;
  store >> totalWidth;
  store >> totalHeight;
  store >> help;
  }


// Cleanup
FXOldIconList::~FXOldIconList(){
  removeAllItems();
  if(timer) getApp()->removeTimeout(timer);
  header=(FXOldHeader*)-1;
  firstitem=(FXOldIconItem*)-1;
  lastitem=(FXOldIconItem*)-1;
  anchoritem=(FXOldIconItem*)-1;
  currentitem=(FXOldIconItem*)-1;
  font=(FXFont*)-1;
  timer=(FXTimer*)-1;
  }


