/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1995  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.14 of 01-November-95
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	sunsite.unc.edu:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	ftp.cygnus.com:/pub/forth/pfe-VERSION.tar.gz
 *
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * types.h ---        Data structures for pfe.
 * (duz 09Jul93)
 */

#ifndef __TYPES_H
#define __TYPES_H

#ifndef	_SETJMP_H
#include <setjmp.h>
#endif

#ifndef FILE
#include <stdio.h>
#endif

#if defined HAVE_SYS_TYPES_H
# include <sys/types.h>		/* size_t, time_t and friends */
#endif


typedef unsigned char Byte;	/* an 8-bit unsigned number */
typedef unsigned short WORD;	/* a 16-bit unsigned number (probably) */

typedef struct Head Head;	/* pointer set comprising a definition */
typedef struct Wordl Wordl;	/* body of a word list */
typedef struct File File;	/* neccessary information about open files */
typedef struct Dict Dict;	/* a dictionary specification */
typedef struct Input Input;	/* an input source specification */
typedef struct Iframe Iframe;	/* a frame to save an input specification */
typedef struct Except Except;	/* an exception frame */

struct Head			/* set of pointers comprising a header */
  {				/* when preceeded by a name */
    char *link;			/* link back to previous word in vocabulary */
    pCode aux;			/* execution semantics or pointer for DOES> */
    pCode cfa;			/* compilation/interpretation semantics */
  };

struct Wordl			/* a word list */
  {				/* a word list identifier is a (Wordl *) */
    char *thread[THREADS];
    Wordl *prev;		/* word lists are chained */
  };

struct File			/* describes a file */
  {
    FILE *f;			/* associated ANSI-C file structure */
    char mdstr[4];		/* mode string for fopen() */
    char mode;			/* mode code for open_file() */
    signed char last_op;	/* -1 write, 0 none, 1 read */
    WORD len;			/* if stream: length of input line */
    uCell size;			/* if block file: size of file in blocks */
    long pos;			/* a saved position, e.g. beginning of line */
    uCell n;			/* block in buffer or source line */
    Cell updated;		/* if block file: block updated? */
    char name[PATH_LENGTH];	/* file name */
    char buffer[1024];		/* buffer for block or input line */
  };

struct Input			/* an input source specification */
  {
    Cell source_id;		/* SOURCE-ID */
    File *block_file;		/* which file is active? */
    uCell blk;			/* currently loaded block */
    long pos;			/* stream file: beginning of line in file */
    char *tib;			/* points to TIB or to EVALUATED string */
    uCell number_tib;		/* #TIB, length of string in TIB */
    uCell to_in;		/* input parsing position */
  };

struct Iframe			/* a frame to save an input specification */
  {
    Cell magic;
    Input input;
    Iframe *prev;
  };

struct Except
  {
    Cell magic;
    Xt *ip;
    Cell *sp;
    Cell *lp;
    double *fp;
    Iframe *iframe;
    jmp_buf jmp;
    Except *prev;
  };

struct memory			/* contains memory-map, one of lo/hi */
  {				/* each start/end of: */
    Byte *dict;			/*  dictionary */
    Cell *stack;		/*  data stack */
    double *fstack;		/*  floating point stack */
    Xt **rstack;		/*  return stack */
    char (*pocket)[POCKET_SIZE];	/*  POCKET for interactive S" */
    char *tib;			/*  TIB */
    char *history;		/*  command line history buffer */
    File *files;		/*  files */
  };

struct memsiz			/* size of allocated memory areas */
  {				/* in items of type stored there */
    uCell dict;
    uCell stack;
    uCell fstack;
    uCell rstack;
    uCell pocket;
    uCell tib;
    uCell history;
    uCell files;
  };

struct Dict			/* all system variables needed to describe */
  {				/* a dictionary */
    Byte *dp;			/* actual top of the dictionary */
    Byte *fence;		/* can't forget below that address */
    char *last;			/* NFA of most recently CREATEd header */
    Wordl *voc_link;		/* link to chained word lists */
    Wordl *context[ORDER_LEN];	/* dictionary search order */
    Wordl *only;		/* ONLY is always searched */
    Wordl *current;		/* points to vocabulary in extension */
    Wordl *forth;		/* points to body of FORTH vocabulary */
    Wordl *dforder[ORDER_LEN];	/* default dictionary search order */
    Xt application;		/* word to run initially or 0 */
    char *hld;			/* auxiliary pointer for number output */
    Cell dpl;			/* position of input decimal point */
  };

#if defined __IBMC__ && defined __OS2__
enum dictOffs			/* normally these are determined with */
  {				/* OFFSET_OF, C Set/2 can't do that. */
    do_dp		= 0,
    do_fence		= do_dp + 4,
    do_last		= do_fence + 4,
    do_voc_link		= do_last + 4,
    do_context		= do_voc_link + 4,
    do_only		= 16 + 4 * ORDER_LEN,
    do_current		= do_only + 4,
    do_forth		= do_current + 4,
    do_dforder		= do_forth + 4,
    do_application	= do_dforder + 4 * ORDER_LEN,
    do_hld		= do_application + 4,
    do_dpl		= do_hld + 4
  };
#endif

struct sysvar			/* all FORTH system variables */
  {
    Cell *s0;			/* base of stack pointer */
    double *f0;			/* base of floating point stack pointer */
#if defined AIX3 && !defined __GNUC__
    /* Workaround for xlc compiler bug */
    void **r0;			/* base of return stack pointer */
#else
    Xt **r0;			/* base of return stack pointer */
#endif
    Dict *dict;			/* the actually used dictionary */
    Input input;		/* active input source, full specification */
    Input input_err;		/* input specification when error occurred */
    Iframe *saved_input;	/* links to chain of saved input specs */
    Except *cAtch;		/* links to chain of CATCHed words */
    uCell span;			/* char count of last EXPECT */
    uCell scr;			/* latest LISTed block number */
    Cell out;			/* current output column on screen */
    Cell state;			/* interpreting (0) or compiling (-1) */
    Cell *locals;		/* number of locals in current def. */
    char (*local)[32];		/* names of locals in current def. */
    Cell *csp;			/* compiler security, saves sp here */
    uCell base;			/* of number i/o conversion */
    Cell precision;		/* floating point output precision */

#if defined(AIX3) && !defined(__GNUC__)
    /* avoid "internal compiler error" from AIX 3.2 cc compiler */
    void *key;			/* executed by KEY */
    void *emit;			/* executed by EMIT */
    void *expect;		/* executed by EXPECT */
    void *type;			/* executed by TYPE */
#else
    Xt key;			/* executed by KEY */
    Xt emit;			/* executed by EMIT */
    Xt expect;			/* executed by EXPECT */
    Xt type;			/* executed by TYPE */
#endif
    Cell lower_case;		/* do toupper() before dictionary search */
    Cell lower_case_fn;		/* do tolower() on file names */
    Cell redefined_msg;		/* no `"xxx" is redefined' msg if false */
    Cell float_input;		/* don't try floating pt input when false */
    Cell reset_order;		/* if true: reset search order on ABORT */
    Cell pocket;		/* which pocket to use next */

    File *stdIn;		/* C-library standard files */
    File *stdOut;		/* mapped to Forth-files */
    File *stdErr;
    uCell more;			/* for a more-like effect */
    uCell lines;
  };

/* file modes */

enum
  {
    FMODE_RO = 1, FMODE_WO, FMODE_RW,
    FMODE_ROB, FMODE_WOB, FMODE_RWB
  };

#define FMODE_BIN (FMODE_ROB - FMODE_RO)

/* Shortcuts for easy access of some system variables: */

#define DP		(sys.dict->dp)
#define FENCE		(sys.dict->fence)
#define LAST		(sys.dict->last)
#define VOC_LINK	(sys.dict->voc_link)
#define CONTEXT		(sys.dict->context)
#define DEFAULT_ORDER	(sys.dict->dforder)
#define ONLY		(sys.dict->only)
#define CURRENT		(sys.dict->current)
#define FORTH		(sys.dict->forth)
#define APPLICATION	(sys.dict->application)
#define HLD		(sys.dict->hld)
#define DPL		(sys.dict->dpl)

#define SOURCE_ID	(sys.input.source_id)
#define SOURCE_FILE	((File *)SOURCE_ID)
#define BLOCK_FILE	(sys.input.block_file)
#define BLK		(sys.input.blk)
#define TIB		(sys.input.tib)
#define NUMBER_TIB	(sys.input.number_tib)
#define TO_IN		(sys.input.to_in)

#define SPAN		(sys.span)
#define SCR		(sys.scr)
#define OUT		(sys.out)
#define STATE		(sys.state)
#define CSP		(sys.csp)
#define BASE		(sys.base)
#define PRECISION	(sys.precision)
#define LOWER_CASE	(sys.lower_case)
#define LOWER_CASE_FN	(sys.lower_case_fn)
#define REDEFINED_MSG	(sys.redefined_msg)
#define FLOAT_INPUT	(sys.float_input)
#define RESET_ORDER	(sys.reset_order)

#endif
