/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1995  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.14 of 01-November-95
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	sunsite.unc.edu:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	ftp.cygnus.com:/pub/forth/pfe-VERSION.tar.gz
 *
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * term-wat.c ---	Terminal driver for Watcom C, uses functions from
 *			conio.h and ANSI escape sequences for attributes.
 * (duz 24Feb94)
 */

#include "forth.h"
#include "term.h"
#include <conio.h>

#if defined WC_OS2V2		/* __OS2__ should do here but doesn't */
# define INCL_KBD 1
# define INCL_VIO 1
# include <string.h>
# include <os2.h>
#elif defined WC_DOS4G
# include <dos.h>
#elif defined WC_WINDOWS
# include <stdio.h>
#endif


char *
rawkey_string [NO_OF_KEYS] =	/* what function keys send */
{
  "\377;", "\377<", "\377=", "\377>", "\377?",
  "\377@", "\377A", "\377B", "\377C", "\377D",
  "\377K", "\377M", "\377H", "\377P",
  "\377G", "\377O", "\377Q", "\377I",
  NULL,	   "\377S", NULL,    "\377R",
  NULL,    NULL,    NULL,    NULL,    /*"\r"*/
};

int tty_interrupt_key (char ch)		{ return 0; }
void interactive_terminal (void)	{}
void system_terminal (void)		{}
void query_winsize (void)		{}

#if defined WC_DOS4G
static int page;		/* display screen, 0-7 */
#endif

int
prepare_terminal (void)
{
#if defined WC_OS2V2

  VIOMODEINFO viomi;
  VioGetMode (&viomi, 0);
  rows = viomi.row;
  cols = viomi.col;
  VioSetAnsi (1, 0);

#elif defined WC_DOS4G

  union REGS regs;
  regs.h.ah = 15;		/* int 10 subfunction 15, get video mode */
  int386 (0x10, &regs, &regs);
  switch (regs.h.al)
    {
    default:
      page = regs.h.bh;
      rows = 25;
      cols = regs.h.ah;
    }

#endif

  return 1;
}

int
c_keypressed (void)
{
#if 0
  KBDKEYINFO kbci;
  KbdPeek (&kbci, 0);
  return kbci.fbStatus >> 6 & 1;
#else
  return kbhit ();
#endif
}

int				/* return '\377' instead of DOS' '\0' */
c_getkey (void)			/* for function keys. */
{
  int c;
#if 0
  KBDKEYINFO kbci;
  KbdCharIn (&kbci, 0, 0);
  c = kbci.chChar;
#else
  c = getch ();
#endif
  return (c &0xFF) == 0 ? '\377' : c;
}

#if defined WC_OS2V2 && 0
  /* the following gives an internal compiler error */
# define CPUTC(C) VioWrtTTY (&(C), 1, 0)
# define CPUTS(S) VioWrtTTY (S, (strlen (S)), 0)
#elif defined WC_WINDOWS
# define CPUTC(C) putchar (C)
# define CPUTS(S) fputs (S, stdout)
#else
# define CPUTC(C) putch (C)
# define CPUTS(S) cputs (S)
#endif

void
c_putc_noflush (char c)
{
  switch (c)
    {
    case '\n':
      CPUTS ("\r\n");
      break;
    default:
      CPUTC (c);
      break;
    }
}

void
c_flush (void)
{
#if defined WC_WINDOWS
  fflush (stdout);
#endif
}

void
c_putc (char c)
{
  c_putc_noflush (c);
  c_flush ();
}

void
c_puts (const char *s)
{
  while (*s)
    c_putc_noflush (*s++);
  c_flush ();
}

void
c_gotoxy (int x, int y)
{
#if defined WC_OS2V2

  VioSetCurPos (y, x, 0);

#elif defined WC_DOS4G

  union REGS regs;
  regs.h.ah = 2;		/* int 10 subfunction 2, set cursor pos */
  regs.h.bh = page;		/* logical screen */
  regs.h.dl = x;
  regs.h.dh = y;
  int386 (0x10, &regs, &regs);

#else

  cprintf ("\033[%d;%dH", y, x);

#endif
}

void
c_wherexy (int *x, int *y)
{
#if defined WC_OS2V2

  USHORT row, col;
  VioGetCurPos (&row, &col, 0);
  *x = col;
  *y = row;

#elif defined WC_DOS4G

  union REGS regs;
  regs.h.ah = 3;		/* int 10 subfunction 3, get cursor pos */
  regs.h.bh = page;		/* logical screen */
  int386 (0x10, &regs, &regs);
  *x = regs.h.dl;
  *y = regs.h.dh;

#else

  *x = *y = 0;			/* uargh! */

#endif
}

static void			/* move cursor in x and y */
addxy (int dx, int dy)
{
  int x, y;
  c_wherexy (&x, &y);
  x += dx;
  y += dy;
  c_gotoxy (x, y);
  OUT = x;
}

void c_goleft (void)		{ addxy (-1,  0); }
void c_goright (void)		{ addxy ( 1,  0); }
void c_goup (void)		{ addxy ( 0, -1); }
void c_godown (void)		{ addxy ( 0,  1); }

void c_home (void)		{ c_gotoxy (0, 0); }
void c_clrscr (void)		{ c_home (); c_clrdown (); }
void c_bell (void)		{ CPUTS ("\a"); }

void
c_clreol (void)
{
#if defined WC_OS2V2

  static char cell [2] = { ' ', 0x07 };
  USHORT row, col;
  VioGetCurPos (&row, &col, 0);
  VioWrtNCell (cell, cols - col, row, col, 0);

#else

  CPUTS ("\033[K");

#endif
}

void
c_clrdown (void)
{
#if defined WC_OS2V2

  static char cell [2] = { ' ', 0x07 };
  int r;
  USHORT row, col;

  VioGetCurPos (&row, &col, 0);
  VioWrtNCell (cell, cols - col, row, col, 0);
  for (r = row + 1; r < rows; r++)
    VioWrtNCell (cell, cols, r, 0, 0);

#else /*if defined WC_DOS4G */

  int x, y, i;

  c_clreol ();
  c_wherexy (&x, &y);
  for (i = y + 1; i < rows; i++)
    {
      c_gotoxy (i, 0);
      c_clreol ();
    }
  c_gotoxy (x, y);

#endif
}

enum {
  none, bold, faint, italic, blink = 5,
  rapid_blink, reverse_video, concealed
};

static int attrib;

static void
setattr (int attr)
{
  if (attr == none)
    {
      attrib = 0;
      CPUTS ("\033[0m");
    }
  else
    {
      attrib |= 1 << attr;
      cprintf ("\033[%dm", attr);
    }
}

static void
clrattr (int attr)
{
  int i;

  attrib &= ~(1 << attr);
  CPUTS ("\033[0");
  for (i = bold; i <= concealed; i++)
    if (attrib >> i & 1)
      cprintf (";%d", i);
  CPUTS ("m");
}

void c_standout_on (void)	{ setattr (bold); }
void c_standout_off (void)	{ clrattr (bold); }
void c_reverse (void)		{ setattr (reverse_video); }
void c_bright (void)		{ setattr (bold); }
void c_blinking (void)		{ setattr (blink); }
void c_normal (void)		{ setattr (none); }
void c_underline_on (void)	{ setattr (italic); }
void c_underline_off (void)	{ clrattr (italic); }
