/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1995  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.14 of 01-November-95
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	sunsite.unc.edu:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	ftp.cygnus.com:/pub/forth/pfe-VERSION.tar.gz
 *
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * curses.c ---	Experimental terminal driver for UNIX-like systems
 *		using the curses library.
 *
 *		This driver works with Linux and the ncurses library.
 *		It doesn't work any better than the termcap.c driver.
 *		To use it change makefile as follows:
 *
 *			OPTS=	-I/usr/include/ncurses
 *			LIBS=	-lncurses
 *			TERM_O=	curses.o
 * (duz 17May94)
 */

#include "forth.h"
#include "term.h"

#undef TRUE
#undef FALSE

#include <limits.h>
#include <sys/ioctl.h>		/* ioctl(), TIOCGWINSZ */

#ifdef Linux
# include <ncurses.h>
#else
# include <curses.h>
#endif

#include "missing.h"

#define SEQ(X)	{ '\033', (Byte)((X) >> CHAR_BIT), (Byte)(X), 0 }

static Byte
rks [][4] =			/* what function keys send */
{
  SEQ (KEY_F(1)),	SEQ (KEY_F(2)),
  SEQ (KEY_F(3)),	SEQ (KEY_F(4)),
  SEQ (KEY_F(5)),	SEQ (KEY_F(6)),
  SEQ (KEY_F(7)),	SEQ (KEY_F(8)),
  SEQ (KEY_F(9)),	SEQ (KEY_F(10)),
  SEQ (KEY_LEFT),	SEQ (KEY_RIGHT),
  SEQ (KEY_UP),		SEQ (KEY_DOWN),
  SEQ (KEY_HOME),	SEQ (KEY_END),
  SEQ (KEY_PPAGE),	SEQ (KEY_NPAGE),
  SEQ (KEY_BACKSPACE),	SEQ (KEY_DC),
  SEQ (KEY_IC),		SEQ (KEY_IC),
  SEQ (KEY_IL),		SEQ (KEY_EOL),
  SEQ (KEY_DL),		SEQ (KEY_CLEAR)
};

Byte *rawkey_string [] =
{
  rks [ 0], rks [ 1], rks [ 2], rks [ 3], rks [ 4],
  rks [ 5], rks [ 6], rks [ 7], rks [ 8], rks [ 9],
  rks [10], rks [11], rks [12], rks [13], rks [14],
  rks [15], rks [16], rks [17], rks [18], rks [19],
  rks [20], rks [21], rks [22], rks [23], rks [24],
  rks [25]
};

void show_control_strings() {}
void show_rawkey_strings () {}

int tty_interrupt_key (char ch)		{ return 0; }
void interactive_terminal (void)	{ fixterm (); }
void system_terminal (void)		{ resetterm (); }

int
prepare_terminal (void)
{
  initscr ();
  cbreak ();
  nonl ();
  noecho ();
  idlok (stdscr, TRUE);
  scrollok (stdscr, TRUE);
  meta (stdscr, TRUE);
  keypad (stdscr, TRUE);
  nodelay (stdscr, FALSE);
  saveterm ();
  refresh ();
  return 1;
}

#ifdef TIOCGWINSZ
void
query_winsize (void)
{
  struct winsize size;

  if (ioctl (1, TIOCGWINSZ, (char *) &size) >= 0)
    {
      rows = size.ws_row;
      cols = size.ws_col;
      xmax = size.ws_xpixel;
      ymax = size.ws_ypixel;
    }
}

#else
void query_winsize (void) {} 
#endif

static Byte nextch[3];
static int nnext = 0;

static int
key (void)
{
  int c;

  c = getch ();
  if (c == -1)
    return 0;
  if (c < 0x100)
    {
      nextch[nnext++] = c;
    }
  else
    {
      nextch[nnext++] = c;
      nextch[nnext++] = c >> CHAR_BIT;
      nextch[nnext++] = '\033';
    }
  return 1;
}

int
c_keypressed (void)
{
  refresh ();
  if (nnext)
    return 1;
  nodelay (stdscr, TRUE);
  key ();
  nodelay (stdscr, FALSE);
  return nnext != 0;
}

int
c_getkey (void)
{
  refresh ();
  if (nnext || key ())
    return nextch[--nnext];
  else
    return -1;
}

void 
c_putc_noflush (char c)
{
  addch (c);
}
void c_flush ()			{ refresh (); }
void c_putc (char c)		{ addch (c); refresh (); }
void c_puts (const char *s)	{ addstr ((char *)s); refresh (); }

void c_gotoxy (int x, int y)	{ move (y, x); }
void c_wherexy (int *x, int *y)	{ getyx (stdscr, *y, *x); }

static void
addxy (int x, int y)
{
  int col, row;
  getyx (stdscr, row, col);
  move (row + y, col + x);
}

void c_goleft (void)		{ addxy (-1,  0); }
void c_goright (void)		{ addxy ( 1,  0); }
void c_goup (void)		{ addxy ( 0, -1); }
void c_godown (void)		{ addxy ( 0,  1); }

void c_clrscr (void)		{ clear (); refresh (); }
void c_home (void)		{ move (0, 0); }
void c_clreol (void)		{ clrtoeol (); }
void c_clrdown (void)		{ clrtobot (); }
void c_bell (void)		{ beep (); }

void c_standout_on (void)		{ standout (); }
void c_standout_off (void)	{ standend (); }
void c_bright (void)		{ attron (A_BOLD); }
void c_reverse (void)		{ attron (A_REVERSE); }
void c_blinking (void)		{ attron (A_BLINK); }
void c_normal (void)		{ attrset (A_NORMAL); }
void c_underline_on (void)	{ attron (A_UNDERLINE); }
void c_underline_off (void)	{ attroff (A_UNDERLINE); }
