// ---------------------------------------------------------------------------
// - List.cpp                                                                -
// - standard object library - doubly linked list class implementation       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "List.hpp"
#include "Method.hpp"
#include "Integer.hpp"
#include "Exception.hpp"

namespace aleph {

  // the private list structure
  struct s_list {
    // the object to store
    Object* p_object;
    // the previous element
    s_list* p_prev;
    // the next element;
    s_list* p_next;
    // simple constructor
    s_list (void) {
      p_object = nilp;
      p_prev   = nilp;
      p_next   = nilp;
    }
    // simple destructor
    ~s_list (void) {
      Object::dref (p_object);
      delete p_next;
    }
  };

  // -------------------------------------------------------------------------
  // - list class section                                                    -
  // -------------------------------------------------------------------------

  // create an empty list

  List::List (void) {
    p_root = nilp;
    p_last = nilp;
  }

  // copy constructor for this list

  List::List (const List& that) {
    p_root = nilp;
    p_last = nilp;
    s_list* node = that.p_root;
    while (node != nilp) {
      append (node->p_object);
      node = node->p_next;
    }
  }
    
  // destroy this list

  List::~List (void) {
    delete p_root;
  }

  // return the class name
  
  String List::repr (void) const {
    return "List";
  }

  // assign a list to this one

  List& List::operator = (const List& that) {
    if (this == &that) return *this;
    delete p_root;
    p_root = nilp;
    p_last = nilp;
    s_list* node = that.p_root;
    while (node != nilp) {
      append (node->p_object);
      node = node->p_next;
    }
    return *this;
  }

  // insert an object at the beginning of the list

  void List::insert (Object* object) {
    s_list* node   = new s_list;
    node->p_object = Object::iref (object);
    node->p_next   = p_root;
    if (p_root == nilp) {
      p_root = node;
      p_last = node;
      return;
    }
    p_root->p_prev = node;
    p_root = node;
  }

  // append an object to the end of the list

  void List::append (Object* object) {
    s_list* node   = new s_list;
    node->p_object = Object::iref (object);
    if (p_root == nilp) {
      p_root = node;
      p_last = node;
      return;
    }
    p_last->p_next = node;
    node->p_prev = p_last;
    p_last = node;
  }
  
  // return the number of elements in the list

  long List::length (void) const {
    s_list* node = p_root;
    long result  = 0;
    while (node != nilp) {
      result++;
      node = node->p_next;
    }
    return result;
  }

  // return an object by index

  Object* List::get (const long index) const {
    long count   = 0;
    s_list* node = p_root;
    if (index < 0) throw Exception ("index-error",
				    "invalid negative index in list get");
    // loop in the list
    while (node != nilp) {
      if (count == index) return node->p_object;
      count++;
      node = node->p_next;
    }
    throw Exception ("index-error", "invalid index in list get method");
  }

  // return a new iterator for this list
  
  Iterator* List::makeit (void) {
    return new ListIterator (this);
  }

  // create a new list in a generic way

  Object* List::mknew (Vector* argv) {
    long len = 0;
    if ((argv == nilp) || ((len = argv->length ()) == 0)) return new List;
    // build the list
    List* result = new List;
    for (long i = 0; i < len; i++)
      result->append (argv->get (i));
    return result;
  }


  // evaluate this list with a member name

  Object* List::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply a list method by name

  Object* List::apply (Interp* interp, Nameset* nset, const String& name,
		       Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "length") && (argc == 0)) {
      delete argv;
      return new Integer (length ());
    }
    if ((name == "get-iterator") && (argc == 0)) {
      delete argv;
      return makeit ();
    }

    // dispatch 1 argument
    if ((name == "append") && (argc == 1)) {
      Object* result = argv->get (0);
      append (result);
      delete argv;
      return result;
    }
    if ((name == "insert") && (argc == 1)) {
      Object* result = argv->get (0);
      insert (result);
      delete argv;
      return result;
    }
    if ((name == "get") && (argc == 1)) {
      long val = argv->getint (0);
      Object* result = get (val);
      delete argv;
      return result;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }

  // -------------------------------------------------------------------------
  // - list iterator class section                                           -
  // -------------------------------------------------------------------------

  // create a new list iterator

  ListIterator::ListIterator (List* lst) {
    p_list = lst;
    Object::iref (lst);
    p_node = nilp;
    begin ();
  }

  // destroy this list iterator

  ListIterator::~ListIterator (void) {
    Object::dref (p_list);
  }

  // return the class name

  String ListIterator::repr (void) const {
    return "ListIterator";
  }

  // reset the iterator to the begining

  void ListIterator::begin (void) {
    p_node = p_list->p_root;
  }

  // reset the iterator to the end

  void ListIterator::end (void) {
    p_node = p_list->p_last;
  }

  // go to the next object

  void ListIterator::next (void) {
    if (p_node == nilp) return;
    p_node = p_node->p_next;
  }

  // go to the previous object
  void ListIterator::prev (void) {
    if (p_node == nilp) return;
    if (p_node->p_prev == nilp) return;
    p_node = p_node->p_prev;
  }

  // get the object at the current position

  Object* ListIterator::getobj (void) {
    if (p_node == nilp) return nilp;
    return p_node->p_object;
  }

  // return true if the iterator is at the end
  
  bool ListIterator::isend (void) {
    if (p_node == nilp) return true;
    return false;
  }
}
