// ---------------------------------------------------------------------------
// - Boolean.hpp                                                             -
// - standard object library - boolean class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_BOOLEAN_HPP
#define  ALEPH_BOOLEAN_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

#ifndef  ALEPH_VECTOR_HPP
#include "Vector.hpp"
#endif

namespace aleph {

  /// The Boolean class is a builtin object for the boolean native type.
  /// The Boolean class accepts only two values (true or false) and is mostly
  /// used for binary testing. There is no automatic conversion from or to the
  /// boolean type - except for string representation.
  /// @author amaury darsch

  class Boolean : public Literal {
  private:
    /// the native boolean representation
    bool d_value;

  public:
    /// Create a new default boolean - by default it is false
    Boolean (void);

    /// Create a new boolean from a native boolean
    /// @param value the value to create
    Boolean (const bool value);

    /// Create a new boolean from a string
    /// @param value the value to convert
    /// @exception illegal-literal
    Boolean (const String& value);

    /// Copy constructor for this boolean
    /// @param that the boolean class to copy
    Boolean (const Boolean& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this boolean
    String toLiteral (void) const;

    /// @return a string representation of this boolean
    String toString (void) const;

    /// @return the boolean value of this boolean
    inline bool toBoolean (void) const {
      return d_value;
    }

    /// assign a boolean with a native value
    /// @param value the value to assign
    Boolean& operator = (const bool value);

    /// assign a boolean with a boolean
    /// @param value the value to assign
    Boolean& operator = (const Boolean& value);

    /// compare this boolean with a native value
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const bool value) const;

    /// compare this boolean with a native value
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const bool value) const;

    /// compare two booleans
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Boolean& value) const;

    /// compare two boolean
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const Boolean& value) const;

    /// operate this boolean with another object
    /// @param type the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// generate a new boolean
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set an object to this boolean
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate a boolean member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this boolean with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);
  };
}

#endif
