// ---------------------------------------------------------------------------
// - Operator.cpp                                                            -
// - aleph engine - operator builtin functions implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Real.hpp"
#include "Interp.hpp"
#include "Builtin.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace aleph {

  // this procedure extract and evaluate the calling object for an operator
  static inline Object* getobj (Interp* interp, Nameset* nset, Cons* args, 
				const char* opname) {
    Object* car = nilp;
    Object* obj = nilp;
    if ((args == nilp) || (args->length () != 2))
      throw Exception ("argument-error",
		       "missing or too many arguments with operator", opname);
    car = args->getcar ();
    obj = (car == nilp) ? nilp : car->eval (interp,nset);
    if (obj == nilp) 
      throw Exception ("type-error", "invalid nil object with operator",
		       opname);
    return obj;
  }

  // this procedure evaluates the cadr and return it
  static inline Object* getarg (Interp* interp, Nameset* nset, Cons* args) {
    Object* car = (args == nilp) ? nilp : args->getcadr ();
    return (car == nilp) ? nilp : car->eval (interp, nset);
  }

  // operator + 

  Object* builtin_add (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "+");
    Object* arg = getarg (interp, nset, args);
    Object* result = obj->oper (Object::ADD, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator -

  Object* builtin_sub (Interp* interp, Nameset* nset, Cons* args) {
    long argc = 0;
    if ((args == nilp) || (((argc = args->length ()) != 1) && (argc != 2)))
      throw Exception ("argument-error",
		       "missing or too many arguments with operator -");
    Object* car = args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp,nset);
    if (obj == nilp) 
      throw Exception ("type-error", "invalid nil object with operator -");
    if (argc == 1) {
      Object* result = obj->oper (Object::MINUS, (Object*) nilp);
      Object::cref (obj);
      return result;
    }
    Object* arg = getarg (interp, nset, args);
    Object* result = obj->oper (Object::SUB, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator * 

  Object* builtin_mul (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    Object* result = obj->oper (Object::MUL, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator /

  Object* builtin_div (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    Object* result = obj->oper (Object::DIV, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator ==

  Object* builtin_eql (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    return obj->oper (Object::EQL, arg);
  }

  // operator !=

  Object* builtin_neq (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    return obj->oper (Object::NEQ, arg);
  }
  
  // operator >=

  Object* builtin_geq (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    return obj->oper (Object::GEQ, arg);
  }
  
  // operator >

  Object* builtin_gth (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    return obj->oper (Object::GTH, arg);
  }
  
  // operator <=

  Object* builtin_leq (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    return obj->oper (Object::LEQ, arg);
  }

  // operator <

  Object* builtin_lth (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = getobj (interp, nset, args, "==");
    Object* arg = getarg (interp, nset, args);
    return obj->oper (Object::LTH, arg);
  }
  

  // process an assert command - this does nothing if the assert flag is false

  Object* builtin_assert (Interp* interp, Nameset* nset, Cons* args) {
    // do nothing if assert flag is false
    if (interp->getassert () == false) return nilp;

    // check for nil first
    Object* car1 = (args == nilp) ? nilp : args->getcar ();
    Object* obj1 = (car1 == nilp) ? nilp : car1->eval (interp, nset);
    Object* car2 = (args == nilp) ? nilp : args->getcadr ();
    Object* obj2 = (car2 == nilp) ? nilp : car2->eval (interp, nset);
    if ((obj1 == nilp) && (obj2 == nilp)) return nilp;
    
    // normal compare
    bool status = false;
    if (obj1 != nilp) {
      Boolean* bobj = dynamic_cast <Boolean*> (obj1->oper (Object::EQL, obj2));
      status = bobj->toBoolean ();
      Object::cref (bobj);
    }
    Object::cref (obj1);
    Object::cref (obj2);
    if (status == true) return nilp;
    Exception e ("assert-error");
    e.setabf (true);
    throw e;  
  }
}
