// ---------------------------------------------------------------------------
// - Localset.hpp                                                            -
// - aleph engine - local set class definition                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_LOCALSET_HPP
#define  ALEPH_LOCALSET_HPP

#ifndef  ALEPH_NAMESET_HPP
#include "Nameset.hpp"
#endif

#ifndef  ALEPH_NAMETABLE_HPP
#include "NameTable.hpp"
#endif

namespace aleph {

  /// The Localset class is a nameset designed to handle a small set of
  /// objects. Each object is associated with a symbol. Creating by name
  /// automatically creates the symbol object. The evaluation automatically
  /// retreive the symbol by name and return the object.
  /// @author amaury darsch

  class Localset : public Nameset {
  private:
    /// the primary name table
    NameTable* p_ptbl;
    /// the secondary name table
    NameTable* p_stbl;

  public:
    /// create a new local set
    Localset (void);
    
    /// create a new local set with the primary table only
    /// @param lset the local nameset
    Localset (Localset* lset);

    /// destroy this local set
    ~Localset (void);

    /// @return the class name
    String repr (void) const;

    /// add a new object by name
    /// @param name the object name
    /// @param object the object to bind
    void bind (const String& name, Object* object);

    /// @return true if the name exists in this nameset
    bool exists (const String& name) const;

    /// @return an object by name but do not evaluate
    Object* find (const String& name) const;

    /// remove an object in this localset
    /// @param name the binding to remove
    void remove (const String& name);

    /// set an object as a const object by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to bind this constant
    /// @param object the object to set
    Object* cdef (Interp* interp, Nameset* nset, const String& name, 
		  Object* object);

    /// set an object to this object by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to bind this constant
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, const String& name, 
		  Object* object);

    /// evaluate an object in the current nameset by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @return the evaluated object
    Object* eval (Interp* interp, Nameset* nset, const String& name);

  private:
    // make the copy constructor private
    Localset (const Localset&);
    // make the assignment operator private
    Localset& operator = (const Localset&);
  };
}

#endif
