// ---------------------------------------------------------------------------
// - Class.cpp                                                               -
// - aleph engine - class class implementation                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Class.hpp"
#include "Lexical.hpp"
#include "Boolean.hpp"
#include "Builtin.hpp"
#include "Instance.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new class object

  Class::Class (void) {
    p_cset  = new Localset;
    Object::iref (p_cset);
  }

  // destroy this class object

  Class::~Class (void) {
    Object::dref (p_cset);
  }

  // return the class name

  String Class::repr (void) const {
    return "Class";
  }

  // add a new data member name
  
  void Class::madd (const String& name) {
    if (d_mdata.exists (name) == true)
      throw Exception ("duplicate-error", "duplicate data member name", name);
    d_mdata.add (name);
  }

  // operate this class with another object

  Object* Class::oper (t_oper type, Object* object) {
    Class* cobj = dynamic_cast <Class*> (object);
    switch (type) {
    case Object::EQL:
      if (cobj != nilp) return new Boolean (this == cobj);
      break;
    case Object::NEQ:
      if (cobj != nilp) return new Boolean (this != cobj);
      break;
    default:
      throw Exception ("operator-error", "unsupported class operator");
    }
    throw Exception ("type-error", "invalid operand with class",
		     Object::repr (object));
  }

  // create a new static const symbol

  Object* Class::cdef (Interp* interp, Nameset* nset, const String& name, 
		       Object* object) {
    return p_cset->cdef (interp, nset, name, object);
  }

  // create or set a new static symbol

  Object* Class::vdef (Interp* interp, Nameset* nset, const String& name, 
		       Object* object) {
    return p_cset->vdef (interp, nset, name, object);
  }

  // evaluate a static member by name

  Object* Class::eval (Interp* interp, Nameset* nset, const String& name) {
    // look in the local set
    Object* obj = p_cset->find (name);
    if (obj != nilp) return obj->eval (interp, nset);
    // go to the object
    return Object::eval (interp, nset, name);
  }

  // create a new instance with a set of arguments

  Object* Class::apply (Interp* interp, Nameset* nset, Cons* args) {
    return new Instance (interp, nset, args, this);
  }

  // apply a member function by name with a set of arguments

  Object* Class::apply (Interp* interp, Nameset* nset, const String& name,
			Cons* args) {
    Object* obj = eval (interp, nset, name);
    if (obj == nilp) throw Exception ("nil-error",
				      "cannot apply nil object as", name);
    return obj->apply (interp, nset, args);
  }
  
  // create a new class object - this is the builtin function

  Object* builtin_class (Interp* interp, Nameset* nset, Cons* args) {
    long len  = (args == nilp) ? 0 : (args->length ());
    // process 0 arguments
    if (len == 0) return new Class;
    // process data member list
    if (len == 1) {
      Class* cls = new Class;
      Cons* cons = dynamic_cast <Cons*> (args->getcar ());
      if (cons == nilp) throw Exception ("argument-error",
					 "only data member list with class");
      while (cons != nilp) {
	Lexical* lex = dynamic_cast <Lexical*> (cons->getcar ());
	if (lex == nilp) 
	  throw Exception ("argument-error",
			   "only lexical name with class data memeber list");
	cls->madd (lex->toLiteral ());
	cons = cons->getcdr ();
      }
      return cls;
    }
    throw Exception ("argument-error",
		     "too many arguments with class definition");
  }
}
