/***************************************************************************
                          rsettings.cpp  -  description
                             -------------------
    begin                : Sun Sep 12 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "rsettings.h"

#include "rconfig.h"
#include "rprgdef.h"

#include <stdio.h>

#include <qfile.h>
#include <qregexp.h>
#include <qstring.h>
#include <qtextstream.h>

#include "rfileparser.h"


/**
 * \class RSettings
 * Class RSettings is a base class for different kind of settings
 *   (identifying strings and values). RSettings provides a method for reading
 *   settings from a file if they're stored in this way:<br>
 *   [section] { $setting = "value" }<br>
 *   or<br>
 *   [section] { <setting> {list of values} }<br>
 */


/**
 * Constructor for settings stored like:<br>
 *   [section] { $name="value" $name2="value2" }<br>
 *   (4 separators)<br>
 *   Gets parsed into a (name|value) - list
 *
 * \param sectionName  Name of the section where these settings are stored (e.g.: "[variables]").
 * \param nameStarter  Character at the start of the identifying name.
 * \param nameStopper  Character at the end of the identifying name.
 * \param valueStarter Character at the start of the value.
 * \param valueStopper Character at the end of the value.
 */
RSettings::RSettings(QString _sectionName,
                   QChar _nameStarter, QChar _nameStopper,
                   QChar _valueStarter, QChar _valueStopper)
{
  sectionName  = _sectionName;
  nameStarter  = _nameStarter;
  nameStopper  = _nameStopper;
  valueStarter = _valueStarter;
  valueStopper = _valueStopper;
  settingList.setAutoDelete(true);
}


/**
 * Constructor for settings stored like:<br>
 *   [section] { <name param="value" param2="value2"> <name2 param="value" param2="value2"> }<br>
 *   (4 separators)<br>
 *   Gets parsed into a (name|param="value" param="value") - list
 *
 * \param sectionName  Name of the section where these settings are stored (e.g.: "[variables]").
 * \param nameStarter  Character at the start of the identifying name.
 * \param separator    Character between identifying name and value.
 * \param valueStopper Character at the end of the value.
 */
RSettings::RSettings(QString _sectionName,
                   QChar _nameStarter,
                   QChar _separator,
                   QChar _valueStopper)
{
  sectionName  = _sectionName;
  nameStarter  = _nameStarter;
  nameStopper  = _separator;
  valueStopper = _valueStopper;
  valueStarter=0;
  settingList.setAutoDelete(true);
}


/**
 * Constructor for settings stored like:<br>
 *   param="value" param2="value2"<br>
 *   (3 separators)<br>
 *   Gets parsed into a (param|value) - list
 *
 * \param nameStopper  Character at the start of the identifying name.
 * \param valueStarter Character between identifying name and value.
 * \param valueStopper Character at the end of the value.
 */
RSettings::RSettings(QChar _nameStopper, QChar _valueStarter,
                   QChar _valueStopper)
{
  sectionName = "";
  nameStarter = 0;
  nameStopper  = _nameStopper;
  valueStarter = _valueStarter;
  valueStopper = _valueStopper;
  settingList.setAutoDelete(true);
}


/**
 * Destructor.
 */
RSettings::~RSettings()
{
}


/**
 * Clears all settings.
 */
void
RSettings::clear()
{
  settingList.clear();
}


/**
 * Adds a setting. If a setting with the given name is already defined it
 *   gets overwritten.
 *
 * \param name Setting name (unique).
 * \param value Value of this setting as a string separated by ';'.
 */
void
RSettings::addSetting(QString _name, QString _value, bool _autoValue)
{
  // Remove all settings with the same name:
  //
  removeSettings(_name.simplifyWhiteSpace());

  // Add new setting:
  //
  RSetting* newSetting = new RSetting(_name.simplifyWhiteSpace(),
                                            _value.simplifyWhiteSpace(),
                                            _autoValue);
  settingList.append(newSetting);
}


/**
 * Adds an int setting. If a setting with the given name is already defined it
 *   gets overwritten.
 *
 * \param name Setting name (unique).
 * \param value Value of this setting as an int.
 */
void
RSettings::addSetting(QString _name, int _value, bool _autoValue)
{
  QString strVal;
  strVal.setNum(_value);

  addSetting(_name, strVal, _autoValue);
}



/**
 * Adds a float setting. If a setting with the given name is already defined it
 *   gets overwritten.
 *
 * \param name Setting name (unique).
 * \param value Value of this setting as a float.
 */
void
RSettings::addSetting(QString _name, float _value, bool _autoValue)
{
  QString strVal;
  strVal.setNum(_value);

  addSetting(_name, strVal, _autoValue);
}


/**
 * Removes all setting with the given name (should be unique anyway).
 *
 * \param name Setting name (unique).
 * \return true: if a setting was removed.
 *         false: otherwise.
 */
bool
RSettings::removeSettings(QString name)
{
  RSetting* s;
  bool ret=false;

  for(s=settingList.first(); s!=0;) {
    if(s->getName()==name) {
      settingList.remove(s);
      s=settingList.current();
      ret=true;
    }
    else {
      s=settingList.next();
    }
  }

  return ret;
}


/**
 * Removes all setting which are not automatically generated.
 */
void
RSettings::removeNonAutoSettings()
{
  RSetting* s;

  for(s=settingList.first(); s!=0;) {
    if(!s->isAutoValue()) {
      settingList.remove(s);
      s=settingList.current();
    }
    else {
      s=settingList.next();
    }
  }
}


/**
 * Gets the setting value of the given setting.
 *
 * \param name Setting name (unique).
 */
QString
RSettings::getSetting(QString name)
{
  RSetting* s;

  for(s=settingList.first(); s!=0; s=settingList.next()) {
    if(s->getName()==name) return s->getValue();
  }

  return "";
}


/**
 * Gets the setting value of the given setting as an integer.
 *
 * \param name Setting name (unique).
 * \param ok   Pointer to flag, which tells us if the conversion from string to int was successful.
 * \param base Base for conversion.
 */
int
RSettings::getSettingInt(QString name, bool* ok, int base)
{
  return getSetting(name).toInt(ok, base);
}


/**
 * Gets the setting value of the given setting as a double.
 *
 * \param name Setting name (unique).
 * \param ok   Pointer to flag, which tells us if the conversion from string to int was successful.
 */
double
RSettings::getSettingDouble(QString name, bool* ok)
{
  return getSetting(name).toDouble(ok);
}


/**
 * Reads settings from a file. Settings already stored won't get lost except
 *   they've the same name. Only settings in the section stored in 'sectionName'
 *   are read.
 *
 * \param fileName Name of the file to read.
 * \return 'true' if we've read some settings successfully.
 *         'false' if the file doesn't exist or there was no section [settings].
 */
bool
RSettings::readFromFile(QString fileName)
{
  QString s;                    // Buffer for the whole section

  s = RFileParser::getSection(RFileParser::getContents(fileName), sectionName);   // The whole section in a string
  s = RFileParser::removeComments(s);

  if(s.length()==0) {
    return false;
  }

  return readFromString(s);
}


/**
 * Reads the settings from a string. The string should NOT contain sections.
 *
 * \param s String to read from.
 */
bool
RSettings::readFromString(QString s)
{
  bool done=false;              // Have we finished
  QString settingName;          // Name of current setting
  QString settingValue;         // Value of current setting
  int i=0;                      // Current index

  while(!done) {

    // No name starter given:
    //
    if(nameStarter==0) {
      settingName=RFileParser::getNextStringUntil(s, i, nameStopper);
    }

    // Name starter given:
    //
    else {
      settingName=RFileParser::getNextStringBetween(s, i, nameStarter, nameStopper);
    }

    if(!settingName.isEmpty()) {

      i+=settingName.length();

      // No value starter given -> use nameStopper as separator!
      //
      if(valueStarter==0) {
        settingValue=RFileParser::getNextStringUntil(s, i, valueStopper);
      }

      // Value starter given:
      //
      else {
        settingValue=RFileParser::getNextStringBetween(s, i, valueStarter, valueStopper);
      }

      i+=settingValue.length();
      addSetting(settingName, settingValue, false);
    }
    else {
      done=true;
    }
  }

  return true;
}



/**
 * Replaces variables in the given string by their values.
 *
 * \param str The original string.
 * \return The new string.
 */
QString
RSettings::replace(QString str)
{
  if(!str.contains('$')) return str;

  RSetting* t;
  QString result=str.copy();
  int i;
  bool found;

  // Replace until we have no occurences of variables or recursive counter>16:
  //
  do {
    found=false;
    for(t=settingList.first(); t!=0; t=settingList.next()) {
      if(result.contains('$')) {
        i=0;
        do {
          i = result.find(QRegExp("\\$" + t->getName() + "[^a-zA-z0-9_]"), i);
          if(i!=-1) {
            result.replace(i, (int)t->getName().length()+1, t->getValue());
            found=true;
          }
        }while(i!=-1);
      }
      else goto end;
    }
  } while(found);

  end:

  return result;
}



// EOF












