#include <config.h>
#ifdef __GNUG__
#pragma implementation "gmjts.hh"
#endif
#include "gmjts.hh"
#include "tile.hh"
#include "swgeneral.hh"
#include <string.h>

#define NPICTURES	(Tileset::IVORY_NPICTURES)
#define NIMAGES		(2*NPICTURES + 1)

#define NACROSS		21
#define NDOWN		4


GnomeMjTileset::GnomeMjTileset(Gif_Stream *gfs, Gif_XContext *gifx)
  : Tileset("ivory"), _gfs(gfs),
    _gifx(gifx), _colormap(0), _image_error(ieNone)
{
  initialize_images();
  if (check()) {
    initialize();
    _gfs->refcount++;
  } else
    _gfs = 0;
}


GnomeMjTileset::~GnomeMjTileset()
{
  Gif_DeleteColormap(_colormap);
  
  Display *display = _gifx->display;
  if (_gfs && _images.count())
    for (int i = 0; i < NIMAGES; i++) {
      if (_images[i]) XFreePixmap(display, _images[i]);
      if (_masks[i]) XFreePixmap(display, _masks[i]);
    }
  
  Gif_DeleteStream(_gfs);
}


void
GnomeMjTileset::initialize_images()
{
  _images.assign(NIMAGES, None);
  _masks.assign(NIMAGES, None);
  check_images();
}


void
GnomeMjTileset::check_images()
{
  if (_gfs == 0 || Gif_ImageCount(_gfs) != 1) {
    _image_error = ieBadGif;
    return;
  }

  Gif_Image *gfi = Gif_GetImage(_gfs, 0);
  if (!_gfs->global && !gfi->local) {
    _image_error = ieBadGif;
    return;
  }

  // Check the size
  if (gfi->width % NACROSS != 0 || gfi->height % NDOWN != 0) {
    _image_error = ieBadSize;
    return;
  }

  // The size minus the border (4) must be even
  int width = gfi->width / NACROSS;
  int height = gfi->height / NDOWN;
  if ((width - 4) % 2 != 0 || (height - 4) % 2 != 0) {
    _image_error = ieBadSize;
    return;
  }
  
  _image_error = ieNone;
}


bool
GnomeMjTileset::check() const
{
  return _image_error == ieNone;
}


void
GnomeMjTileset::initialize()
{
  Gif_Image *gfi = Gif_GetImage(_gfs, 0);
  _colormap = Gif_CopyColormap(gfi->local ? gfi->local : _gfs->global);

  _width = gfi->width / NACROSS;
  _height = gfi->height / NDOWN;
  _border = 4;
  
  _halfwidth = (_width - _border) / 2;
  _halfheight = (_height - _border) / 2;
}


static int mapping[] = {
  33, 34, 35, 36,			/* 0-3: seasons */
  38, 39, 40, 41,			/* 4-7: plants; switch w/seasons?? */
  14, 37, 13,				/* 8-10: dragons */
  9, 11, 12, 10,			/* 11-14: winds */
  0, 1, 2, 3, 4, 5, 6, 7, 8,		/* 15-23: circles */
  24, 25, 26, 27, 28, 29, 30, 31, 32,	/* 24-32: bamboo */
  15, 16, 17, 18, 19, 20, 21, 22, 23,	/* 33-41: characters */
};

void
GnomeMjTileset::draw(SwDrawable *drawable, short x, short y, short which)
{
  if (!_images[which]) {
    Gif_Image *gfi = Gif_GetImage(_gfs, 0);
    int pos;
    if (which == 2*NPICTURES)
      pos = 37; // white dragon
    else if (which >= NPICTURES)
      pos = mapping[which - NPICTURES] + 42;
    else
      pos = mapping[which];
    _images[which] = Gif_XSubImageColormap
      (_gifx, _gfs, _colormap, gfi,
       (pos%NACROSS)*_width, (pos/NACROSS)*_height, _width, _height);
    _masks[which] = Gif_XSubMask
      (_gifx, _gfs, gfi,
       (pos%NACROSS)*_width, (pos/NACROSS)*_height, _width, _height);
  }
  
  drawable->draw_image(_images[which], _masks[which], _width, _height, x, y);
}


void
GnomeMjTileset::draw_normal(const Tile *t, SwDrawable *drawable, short x, short y)
{
  int which = picture(t->number());
  assert(which >= 0 && which < NPICTURES);
  draw(drawable, x, y, which);
}

void
GnomeMjTileset::draw_lit(const Tile *t, SwDrawable *drawable, short x, short y)
{
  int which = picture(t->number());
  assert(which >= 0 && which < NPICTURES);
  draw(drawable, x, y, which + NPICTURES);
}

void
GnomeMjTileset::draw_obscured(const Tile *t, SwDrawable *drawable,
			      short x, short y)
{
  int which = picture(t->number());
  assert(which >= 0 && which < NPICTURES);
  draw(drawable, x, y, 2*NPICTURES);
}
