## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys

# PySol imports
if sys.modules.has_key("pysoltk"):                                  #bundle#
    from gamedb import registerGame, GameInfo, GI                   #bundle#
    from util import *                                              #bundle#
    from stack import *                                             #bundle#
    from game import Game                                           #bundle#
    from layout import Layout                                       #bundle#
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint #bundle#
    from pysoltk import MfxCanvasText                               #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class FortyThieves_Hint(CautiousDefaultHint):
    # FIXME: demo is not too clever in this game

    def _preferHighRankMoves(self):
        return 1


# /***********************************************************************
# // Forty Thieves
# ************************************************************************/

class FortyThieves(Game):
    Foundation_Class = SS_FoundationStack
    RowStack_Class = SS_RowStack
    Hint_Class = FortyThieves_Hint

    #
    # game layout
    #

    def createGame(self, max_rounds=1, num_deal=1, rows=10, XCARDS=64, XOFFSET=10):
        # create layout
        XM = (10, 4)[rows > 10]
        l, s = Layout(self, XM=XM, XOFFSET=XOFFSET, YBOTTOM=16), self.s

        # set window
        # (compute best XOFFSET - up to 64/72 cards can be in the Waste)
        maxrows = max(rows, 10)
        w1, w2 = maxrows*l.XS+l.XM, 2*l.XS
        if w2 + XCARDS * l.XOFFSET > w1:
            l.XOFFSET = int((w1 - w2) / XCARDS)
        # (piles up to 12 cards are playable without overlap in default window size)
        h = max(2*l.YS, l.YS+(12-1)*l.YOFFSET)
        self.setSize(w1, l.YM + l.YS + h + l.YS + l.YBOTTOM)

        # create stacks
        x = l.XM + (maxrows - 8) * l.XS / 2
        y = l.YM
        for i in range(8):
            s.foundations.append(self.Foundation_Class(x, y, self, i / 2))
            x = x + l.XS
        x = l.XM + (maxrows - rows) * l.XS / 2
        y = l.YM + l.YS
        for i in range(rows):
            s.rows.append(self.RowStack_Class(x, y, self, max_move=1))
            x = x + l.XS
        x = self.width - l.XS
        y = self.height - l.YS - l.YBOTTOM
        ty = self.height - int((l.YM + l.YBOTTOM) / 2)
        s.talon = WasteTalonStack(x, y, self, max_rounds=max_rounds, num_deal=num_deal)
        l.createText(s.talon, "s")
        if max_rounds > 1:
            s.talon.texts.rounds = MfxCanvasText(self.canvas,
                                       x + l.CW / 2, y - l.YM, anchor="s")
        x = x - l.XS
        s.waste = WasteStack(x, y, self)
        s.waste.CARD_XOFFSET = -l.XOFFSET
        l.createText(s.waste, "s")

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    DEAL = (0, 4)
    FILL_EMPTY_ROWS = 0

    def startGame(self):
        for i in range(self.DEAL[0]):
            self.s.talon.dealRow(flip=0, frames=0)
        for i in range(self.DEAL[1] - 1):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        self.s.talon.dealCards()          # deal first card to WasteStack

    def fillStack(self, stack):
        if self.FILL_EMPTY_ROWS and stack in self.s.rows and not stack.cards:
            ## FIXME
            pass

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.suit == card2.suit and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))


# /***********************************************************************
# // Busy Aces
# // Waning Moon
# ************************************************************************/

class BusyAces(FortyThieves):
    DEAL = (0, 1)

    def createGame(self):
        FortyThieves.createGame(self, rows=12)


class WaningMoon(FortyThieves):
    def createGame(self):
        FortyThieves.createGame(self, rows=13)


# /***********************************************************************
# // Deuces
# // Courtyard
# ************************************************************************/

class Deuces(FortyThieves):
    ## FIXME: deal Twos to Foundations
    FILL_EMPTY_ROWS = 1
    DEAL = (0, 1)


class Courtyard(BusyAces):
    ## FIXME
    FILL_EMPTY_ROWS = 1


# /***********************************************************************
# // Forty and Eight
# ************************************************************************/

class FortyAndEight(FortyThieves):
    def createGame(self):
        FortyThieves.createGame(self, max_rounds=2, rows=8, XCARDS=72, XOFFSET=8)


# /***********************************************************************
# // Little Forty
# ************************************************************************/

class LittleForty(FortyThieves):
    RowStack_Class = FullStackWrapper(Spider_SS_RowStack)

    def createGame(self):
        FortyThieves.createGame(self, max_rounds=4, num_deal=3)


# /***********************************************************************
# // Streets
# // Maria
# // Number Ten
# // Rank and File
# //   build down by alternate color
# ************************************************************************/

class Streets(FortyThieves):
    RowStack_Class = AC_RowStack

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.color != card2.color and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))


class Maria(Streets):
    def createGame(self):
        FortyThieves.createGame(self, rows=9)


class NumberTen(Streets):
    RowStack_Class = FullStackWrapper(AC_RowStack)
    DEAL = (2, 2)


class RankAndFile(NumberTen):
    DEAL = (3, 1)


# /***********************************************************************
# // Red and Black
# //   build down by alternate color, foundations up by alternate color
# ************************************************************************/

class RedAndBlack(Streets):
    Foundation_Class = AC_FoundationStack
    RowStack_Class = FullStackWrapper(AC_RowStack)
    DEAL = (0, 1)

    def createGame(self):
        FortyThieves.createGame(self, rows=8)


# /***********************************************************************
# // Indian
# // Midshipman
# //   build down by any suit but own
# ************************************************************************/

class Indian_RowStack(SequenceRowStack):
    def _isSequence(self, cards):
        return isAnySuitButOwnSequence(cards, self.cap.mod, self.cap.dir)


class Indian(FortyThieves):
    RowStack_Class = Indian_RowStack
    DEAL = (1, 2)

    def createGame(self):
        FortyThieves.createGame(self, XCARDS=74, XOFFSET=8)

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.suit != card2.suit and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))


class Midshipman(Indian):
    DEAL = (2, 2)

    def createGame(self):
        FortyThieves.createGame(self, rows=9)


# register the game
registerGame(GameInfo(13, FortyThieves, "Forty Thieves",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(80, BusyAces, "Busy Aces",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(79, WaningMoon, "Waning Moon",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(23, FortyAndEight, "Forty and Eight",
                      GI.GT_FORTY_THIEVES, 2, 1))
registerGame(GameInfo(72, LittleForty, "Little Forty",
                      GI.GT_FORTY_THIEVES, 2, 3))
registerGame(GameInfo(76, Streets, "Streets",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(73, Maria, "Maria",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(70, NumberTen, "Number Ten",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(71, RankAndFile, "Rank and File",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(75, RedAndBlack, "Red and Black",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(69, Indian, "Indian",
                      GI.GT_FORTY_THIEVES, 2, 0))
registerGame(GameInfo(74, Midshipman, "Midshipman",
                      GI.GT_FORTY_THIEVES, 2, 0))

