// Copyright (c) 1996-1999 The University of Cincinnati.
// All rights reserved. 

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Author: Radharamanan Radhakrishnan          ramanan@ece.uc.edu

//---------------------------------------------------------------------------
//
// $Id: library_manager.cc,v 1.10 1999/10/08 12:58:40 dmartin Exp $
//
//---------------------------------------------------------------------------

#include "library_manager.hh"
#include "scram.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "IIR_DesignFile.hh"
#include "IIR_Identifier.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_ConfigurationDeclaration.hh"
#include "StandardPackage.hh"
#include "IIR_SelectedName.hh"
#include "IIR_SimpleName.hh"
#include "IIR_IndexedName.hh"
#include "IIR_Identifier.hh"
#include "error_func.hh"
#include "file_manager.hh"
#include "symbol_lookup.hh"

#include <string.h>
#include <errno.h>

// These should be replaced with something like an "os_interface_manager"
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <strstream.h>

extern bool verbose_output;

const char *
library_manager::library_suffix = "._savant_lib";

const char *
library_manager::entity_suffix = "_entity.vhdl";

const char *
library_manager::configuration_suffix = "_config.vhdl";

const char *
library_manager::package_suffix = "_package.vhdl";

const char *
library_manager::package_body_suffix = "_package_body.vhdl";

const char *
library_manager::secondary_units_suffix = "_secondary_units";

library_manager::library_manager( char *logical_work_directory_name ){
  libraries = new set<IIR_LibraryDeclaration>();
  my_symbol_table = new symbol_lookup();

  set_work_dir_name( logical_work_directory_name );
  my_standard_package = StandardPackage::std_standard_decl;
  _lib_out.set_file_output( true );
  fileName = NULL;


  // This stuff sets up the standard library
  IIR_LibraryDeclaration *std_library = StandardPackage::std_decl;
  ASSERT( std_library != NULL );
  savant_root_dir = NULL;
  char *std_dir = find_library_directory( std_library->get_declarator() );
  ASSERT( std_dir != NULL );
  std_library->_set_path_to_directory( std_dir );
  libraries->add( std_library );
}

library_manager::~library_manager(){ 
}

void
library_manager::reset(){
  _lib_out.reset();
}

// calls the switch file's flush
void
library_manager::flush(){
  _lib_out.flush();
  _lib_out.close();
}

// retrieves the work directory's actual name
char*
library_manager::get_work_dir_name(){
  return get_work_library()->_get_path_to_directory();
}

// sets the work directory's actual name
void 
library_manager::set_work_dir_name(char *name){
  // So, build the path to the physical work directory here.
  ostrstream stream;
  stream << name << library_suffix << ends;  
  char *work_dir_name = stream.str();
  
  // Now, we'll instantiate an empty "work" library.  This library needs
  // to be named work, because elements in it will be referenced with
  // "work.foo" and "work.bar", even if we're parsing into library "std"
  // or something.
  my_work_library = new IIR_LibraryDeclaration();
  IIR_Identifier *lib_name = IIR_Identifier::get( "work", strlen("work") );
  // Logical name made "work" for this parse.
  my_work_library->set_declarator( lib_name );
  // Here is the path to this work library.  (Might not be work._savant_lib
  // - could be std._savant_lib or something, for instance.)
  my_work_library->_set_path_to_directory( work_dir_name );
  // The code generator needs to know the "real" name of this library, so we'll
  // store it in the library declaration.
  my_work_library->_set_logical_name( IIR_Identifier::get( name, strlen(name) ) );

  libraries->add( my_work_library );
}

bool
library_manager::check_root_dir( char *root_dir, bool complain ){
  char *file_name = "textio_package.vhdl";
  char *full_path_to_file;
  const char *sep = file_manager::get_directory_separator();
  ostrstream err;
  ostrstream full_path_stream;

  bool error = false;

  if( file_manager::check_file_status( root_dir, 
				       file_manager::DIRECTORY ) != file_manager::OK){
    if( complain == true ){
      err << "|" << root_dir << "| isn't a valid directory.";
    }
    error = true;
  }
  
  if( error == false ){
    full_path_stream << root_dir << sep << "lib" << sep << "std" << library_suffix  
		     << sep << file_name << ends;
    
    full_path_to_file = full_path_stream.str();
    if( file_manager::check_file_status( full_path_to_file, 
					 file_manager::REGULAR_FILE ) != file_manager::OK ){      
      if( complain == true ){
	err << "Directory |" << root_dir << "| doesn't appear to contain the file |" 
	    << full_path_to_file  << "| which should be in the Savant library directory."
	    << ends;
      }
      error = true;
    }
    // full_path_stream.freeze(0);
    delete [] full_path_to_file;
  }
  
  if( error == true && complain == true ){
    report_error( err, -1, NULL, FATAL );
  }

  return !error;
}


char *
library_manager::check_default_directories( bool complain ){
  char *retval = NULL;
  const char *sep = file_manager::get_directory_separator();

  int i = 0;
  const char **std_lib_dirs = file_manager::get_library_directories();
  const char *current = std_lib_dirs[i];  

  while( current != NULL ){
    ostrstream full_path_stream;
    full_path_stream << current << sep << "savant" << sep << ends;
    char *full_path = full_path_stream.str();
    if( check_root_dir( full_path, complain ) == true ){
      retval = strdup( full_path );
      break;
    }
    i++;
    current = std_lib_dirs[i];
    // full_path_stream.freeze(0);
    delete [] full_path;
  }

  return retval;
}

char *
library_manager::get_savant_root_directory(){
  char *retval = NULL;
  
  if( savant_root_dir != NULL ){
    retval = savant_root_dir;
  }
  else{
    retval = check_environment_variable( false );
    if( retval == NULL ){
      retval = check_default_directories( false );
    }

    if( retval == NULL ){
      ostrstream err1, err2;
      err1 << "Unable to locate savant library directory.  Currently, scram will:\n"
	   << "1) Check the SAVANTROOT environment variable.  Problems with this method"
	   << " included:" << ends;
      report_error( err1, -1, NULL, FATAL );
      check_environment_variable( true );
      err2 << "2) Looking in a list of standard directories. Problems with this method were:";
      report_error( err2, -1, NULL, FATAL );
      check_default_directories( true );

      exit( -1 );
    }
    else{
      savant_root_dir = retval;
    }
  }

  return retval;
}


char *
library_manager::check_environment_variable( bool complain ){
  char *retval = NULL;
  char *varname = "SAVANTROOT";
  const char *sep = file_manager::get_directory_separator();
  ostrstream err;

  bool error = false;

  char *savant_root_var = getenv( varname );
  if( savant_root_var == NULL && complain == true ){
    err << "|" << varname << "| isn't defined.";
    error = true;
  }
  else{
    if( check_root_dir( savant_root_var, false ) == true ){
      retval = savant_root_var;
    }
  }

  
  if( error == true && complain == true ){
    // Then we're going to need to print out a rather verbose error message
    // explaining to the user how they should have set the environment variable.
    int uid = getuid();
    struct passwd *pwd_entry;
    pwd_entry = getpwuid( uid );
    char *name, *dir;
    if( pwd_entry == NULL ){
      name = "johndoe";
      name = "/home/johndoe";
    }
    else{
      name = pwd_entry->pw_name; 
      dir = pwd_entry->pw_dir;
    }
    
    err << "  If you installed Savant in a \"non-standard\" place, please set SAVANTROOT"
	<< " to the top directory of the savant hierarchy. "
	<< "For example, if you have installed savant in your home "
	<< "directory, you might issue one of the following commands to "
	<< "to set this variable:\n"
	<< "setenv SAVANTROOT " << dir << sep << "savant\n"
	<< "or\n"
	<< "SAVANTROOT=" << dir << sep << "savant; export SAVANTROOT\n"
	<< "See the man page or other documentation for your shell for details.";
      
    endpwent();
        
    report_error( err, -1, NULL, FATAL );
  }
  
  return retval;
}

int
library_manager::create_work_dir(){
  register int errorFlag = 0;

  if((errorFlag = 
      file_manager::make_directory( get_work_dir_name(), 0755)) == -1 && errno != EEXIST){
    cerr << "Error : Unable to open " << get_work_dir_name() << endl;
  }

  return errorFlag;
}

// create the design library as specified by the lib name and change
// directories to this new directory
void
library_manager::create_design_library(char *libName){
  ostrstream stream;
  stream << libName << "._savant_lib" << ends;
  char *library_name = stream.str();

  file_manager::make_directory( library_name );
  set_work_dir_name(library_name);
}

// As createDir and createWorkDir change the current directory, this method
// steps back one directory level. Returns the error condition if any.
int 
library_manager::step_back(){
  cerr << "Changing to ../" << endl;
  register int errorFlag = 0;
  if( ( errorFlag = file_manager::change_directory("../") ) == -1){
    cerr << "Error : Unable to step back one directory" << endl;
  }
  return errorFlag;
}

// change directory as specified by the library name
void
library_manager::change_dir(char *libName){
  cerr << "Changing to libName" << endl;
  register int errorFlag = 0;
  if((errorFlag = chdir(libName)) == -1){
    cerr << "Library Manager Error : unable to change directory " << endl;
  }
}

// this methods gets a library unit and depending on the type does the
// appropriate name mangling and returns a handle to a ostream.
ostream& 
library_manager::set_output_file( IIR_LibraryUnit *lib_unit ){
  
  char *file_name = NULL;
  char *library_dir = get_work_dir_name();

  switch( lib_unit->get_kind() ){
  case IIR_ENTITY_DECLARATION:{
    file_name = build_entity_file_name( lib_unit->get_declarator(), library_dir );
    break;
  }

 case IIR_CONFIGURATION_DECLARATION:{
    file_name = build_configuration_file_name( lib_unit->get_declarator(), library_dir );
    break;   
  }

  case IIR_PACKAGE_DECLARATION:{
    file_name = build_package_file_name( lib_unit->get_declarator(), library_dir );
    break;
  }

  case IIR_ARCHITECTURE_DECLARATION:{
    IIR_ArchitectureDeclaration *as_arch = (IIR_ArchitectureDeclaration*)lib_unit;
    IIR_EntityDeclaration *entity_decl = as_arch->get_entity();
    ASSERT( entity_decl != NULL );

    file_name = build_arch_file_name( entity_decl->get_declarator(), as_arch->get_declarator(),
				      library_dir );
    
    break;
  }

  case IIR_PACKAGE_BODY_DECLARATION:{
    file_name = build_package_body_file_name( lib_unit->get_declarator(), library_dir );
    break;
  }

  default:{
    ostrstream err;
    err << "Internal error in library_manager::setOutputFile - don't know what to do"
	<< " with kind " << lib_unit->get_kind_text();
    abort();
    break;
  }
  }
  _lib_out.set_file_output( true );
  _lib_out.set_file( file_name );
  delete [] file_name;

  return _lib_out;
}

// calls publish_vhdl_decl so that analyzed design units can be stored in
// the work directory
void
library_manager::parse_into_work_dir(IIR_LibraryUnit *lib_unit){
  my_symbol_table->lookup_add( lib_unit );

  create_directories( lib_unit );

  lib_unit->_publish_vhdl_decl( set_output_file( lib_unit ) );
  flush();
}
			 
IIR_LibraryUnit *
library_manager::get_primary_unit( IIR_LibraryDeclaration *lib_decl,
				   char *path_to_file ){
  
  // So, the path sent to us is a relative path and doesn't include the savant lib
  // directory.  If we don't find the file, we'll need to look in the SAVANT root
  // directory.  (Eventually we'll need to look in a user defined list of other 
  // possibilities.)
  IIR_LibraryUnit *retval = NULL;
  char *current_path = path_to_file;
  if( file_manager::check_file_status( current_path,
				       file_manager::REGULAR_FILE ) != file_manager::OK ){
    ostrstream file_stream;
    file_stream << get_savant_root_directory() << "/lib/" << path_to_file << ends;
    current_path = file_stream.str();
    if( file_manager::check_file_status( current_path,
					 file_manager::REGULAR_FILE ) != file_manager::OK ){
      delete [] current_path;
      current_path = NULL;
    }
  }
  
  if( current_path != NULL ){
    scram *tmp_scram = new scram();
    IIR_DesignFile *use_file = tmp_scram->parse_file( current_path, lib_decl );
    //    delete tmp_scram;

    ASSERT( use_file != NULL );
    ASSERT( use_file->library_units.num_elements() == 1 );
    retval =  use_file->library_units.first();
  
    if( retval != NULL ){
      my_symbol_table->lookup_add( retval );
    }
  
    if( current_path != path_to_file ){
      delete [] current_path;
    }
  }
  
  return retval;
}

IIR_LibraryUnit *
library_manager::get_primary_unit( IIR_LibraryDeclaration *lib_decl,
				   IIR_Name *name,
				   IIR_Kind unitType,
				   IIR_Boolean complain_on_error ){
  IIR_LibraryUnit *retval = NULL;
  
  if( name->get_kind() == IIR_SIMPLE_NAME ){
    IIR_SimpleName *as_simple_name = (IIR_SimpleName *)name;
    ASSERT( as_simple_name->get_prefix() != NULL );
    ASSERT( as_simple_name->get_prefix()->_is_iir_text_literal() == TRUE );
    char *path_to_file = build_entity_file_name( as_simple_name->_get_prefix_string(),
						 get_work_dir_name() );

    retval = get_primary_unit( lib_decl, path_to_file );
  }
  else if( name->get_kind() == IIR_SELECTED_NAME ){
    IIR_SelectedName *as_selected_name = (IIR_SelectedName *)name;
    ASSERT( as_selected_name->get_prefix()->get_kind() == IIR_SIMPLE_NAME );
    ASSERT( as_selected_name->get_suffix()->get_kind() == IIR_SIMPLE_NAME );
    
    IIR_TextLiteral *logical_lib_name = as_selected_name->_get_prefix_string();

    // Find the library declaration - it has the mapping to the library directory
    // oin it.
    IIR_LibraryDeclaration *lib_decl = find_library( logical_lib_name );
    if( lib_decl != NULL ){
      char *lib_dir_name = lib_decl->_get_path_to_directory();
      
      char *entity_file_name = build_entity_file_name( ((IIR_SimpleName *)as_selected_name->get_suffix())->_get_prefix_string(), lib_dir_name );

      name->_get_symbol_table()->open_scope( lib_decl );
      retval = get_primary_unit( lib_decl, entity_file_name );
      name->_get_symbol_table()->close_scope( lib_decl );
      if( retval != NULL ){
	lib_decl->primary_units.append( retval );
      }
    }
  }
  else{
    ostrstream err;
    err << "Internal error in library_manager::get_primary_unit(symbol_table *sym_tab,"
	<< "IIR *name, IIR_Kind unitType) - don't know how to look up name " << *name
	<< " - only know how to look up simple and selected names!" << ends;
    report_error( name, err );
  }

  return retval;
}

// goes and retrieves a particular primary unit given a secondary unit
IIR_LibraryUnit *
library_manager::get_primary_unit( IIR_LibraryDeclaration *lib_decl,
				   IIR_TextLiteral *primary_unit_name,
				   IIR_Kind unitType,
				   IIR_Boolean complain_on_error ){
  IIR_LibraryUnit *retval = 0;
  ostrstream file_name_stream;
  char *file_name = NULL;
  char *unit_name_as_string = primary_unit_name->_convert_to_c_string();
  file_name_stream << lib_decl->_get_path_to_directory() << "/" << unit_name_as_string;
  switch(unitType){
  case IIR_ENTITY_DECLARATION:
    file_name_stream << entity_suffix << ends;
    break;
  case IIR_PACKAGE_DECLARATION:
    file_name_stream << package_suffix << ends;
    break;
  case IIR_CONFIGURATION_DECLARATION:
    file_name_stream << configuration_suffix << ends;
    break;    
  default:
    ostrstream err;
    err << "Library Manager : get_primary_unit(" << unitType << ") not implemented yet " << ends;
    report_error( primary_unit_name, err );
    break;
  }

  file_name = file_name_stream.str();
  retval = get_primary_unit( lib_decl, file_name );

  delete [] unit_name_as_string;
  delete [] file_name;

  return retval;
}

IIR_LibraryUnit *
library_manager::get_primary_unit( IIR_LibraryDeclaration *lib_decl,
				   IIR_Name *unit_name,
				   IIR_Boolean complain_on_error ){
  IIR_LibraryUnit *retval = NULL;
  // So, this method needs to find the primary unit specificed in "unit_name".
  // It's possible that unit name is some sort of complex name, so we'll get 
  // the suffix and search for it in the declarations we find.
  // If we're looking for "work.foo" and there is a configuration foo, and
  // an entity foo, and a package foo, obviously then this method needs to
  // complain.
  
  IIR_EntityDeclaration *entity_decl;
  IIR_PackageDeclaration *package_decl;
  IIR_ConfigurationDeclaration *configuration_decl;

  IIR_TextLiteral *unit_declarator = unit_name->_get_prefix_string();

  entity_decl = 
    (IIR_EntityDeclaration *)get_primary_unit( lib_decl,
					       unit_declarator, 
					       IIR_ENTITY_DECLARATION );

  package_decl = 
    (IIR_PackageDeclaration *)get_primary_unit( lib_decl,
						unit_declarator,
						IIR_PACKAGE_DECLARATION);
  
  configuration_decl =
    (IIR_ConfigurationDeclaration*)get_primary_unit( lib_decl, 
						     unit_declarator,
						     IIR_CONFIGURATION_DECLARATION );
    
  if( unit_name->_get_suffix() == NULL || 
      unit_name->_get_suffix()->get_kind() == IIR_DESIGNATOR_BY_ALL ){
    // If there was more than one potential primary unit, then we need to
    // complain.
    set<IIR_Declaration> decls;
    if( entity_decl != NULL ){
      decls.add( entity_decl );
    }
    if( package_decl != NULL ){
      decls.add( package_decl );
    }
    if( configuration_decl != NULL ){
      decls.add( configuration_decl );
    }
    switch( decls.num_elements() ){
    case 0:{
      if( complain_on_error == TRUE ){
	report_undefined_symbol( lib_decl, unit_name );
      }
      break;
    }
    case 1:{
      retval = (IIR_LibraryUnit *)decls.get_element();
      break;
    }
    default:{
      report_ambiguous_error( lib_decl, unit_name, &decls );
      break;
    }
    }
  }

  return retval;
}


char *
library_manager::build_package_body_file_name( IIR_TextLiteral *package_body_name,
					       char *library_directory ){

  char *converted_package_name = package_body_name->_convert_to_c_string();
  
  // This is the stream that we'll build the file name in.
  ostrstream package_stream;
  package_stream << library_directory << "/" << converted_package_name 
		 << secondary_units_suffix << library_suffix
		 << "/" << converted_package_name  << package_body_suffix << ends;
  
  delete [] converted_package_name;

  return package_stream.str();     
}

char *
library_manager::build_arch_file_name( IIR_TextLiteral *entity_name,
				       IIR_TextLiteral *arch_name,
				       char *library_directory ){
  char *converted_entity_name = entity_name->_convert_to_c_string();
  char *converted_arch_name = arch_name->_convert_to_c_string();
  
  // This is the stream that we'll build the file name in.
  ostrstream arch_stream;
  arch_stream << library_directory << "/" << converted_entity_name 
	      <<  "_secondary_units._savant_lib/" << converted_entity_name  
	      << "_" << converted_arch_name << "_arch.vhdl" << ends;
  
  delete [] converted_entity_name;
  delete [] converted_arch_name;
  

  return arch_stream.str();  
}

char *
library_manager::build_configuration_file_name( IIR_TextLiteral *configuration_name,
						char *library_directory ){
  ostrstream name_stream;
  name_stream << library_directory << "/" << configuration_name->_convert_to_c_string() 
	      << configuration_suffix << ends;
  return name_stream.str();  

}

char *
library_manager::build_package_file_name( IIR_TextLiteral *package_name,
					  char *library_directory ){
  ostrstream name_stream;
  name_stream << library_directory << "/" << package_name->_convert_to_c_string() 
	      << package_suffix << ends;
  return name_stream.str();  

}

char *
library_manager::build_entity_file_name( IIR_TextLiteral *entity_name, char *library_directory ){
  ostrstream name_stream;
  name_stream << library_directory << "/" << entity_name->_convert_to_c_string() 
	      << entity_suffix << ends;
  return name_stream.str();  
}

char *
library_manager::build_library_directory_name( IIR_TextLiteral *library_name ){
  ostrstream name_stream;

  char *as_c_string = library_name->_convert_to_c_string();
  name_stream << as_c_string << library_suffix << ends;
  delete [] as_c_string;

  return name_stream.str();
}

char *
library_manager::build_directory_name_for_secondary_unit( IIR_LibraryUnit *library_unit ){
  ostrstream name_stream;
  
  ASSERT( library_unit->_is_secondary_unit() == TRUE );
  name_stream << get_work_dir_name() << "/";

  if( library_unit->get_kind() == IIR_PACKAGE_BODY_DECLARATION ){
    name_stream << library_unit->_get_declarator()->_convert_to_c_string();
  }
  else{
    ASSERT( library_unit->get_kind() == IIR_ARCHITECTURE_DECLARATION );

    IIR_ArchitectureDeclaration *as_arch;
    as_arch = (IIR_ArchitectureDeclaration *)library_unit;
    ASSERT( as_arch->get_entity() != NULL );

    name_stream << as_arch->get_entity()->_get_declarator()->_convert_to_c_string();
  }
      
  name_stream << secondary_units_suffix << library_suffix << ends;

  return name_stream.str();
}

IIR_ArchitectureDeclaration *
library_manager::parse_architecture( IIR_EntityDeclaration *entity_decl, 
				     IIR_Name *architecture_name ){

  ASSERT( architecture_name != NULL );
  ASSERT( architecture_name->get_kind() == IIR_SIMPLE_NAME );  

  IIR_SimpleName *as_simple_name = (IIR_SimpleName *)architecture_name;
  ASSERT( as_simple_name->get_prefix() != NULL );
  ASSERT( as_simple_name->get_prefix()->_is_iir_text_literal() == TRUE );
  IIR_TextLiteral *arch_as_text_literal = (IIR_TextLiteral *)as_simple_name->get_prefix();

  ASSERT( entity_decl->_get_declarative_region() != NULL );
  ASSERT( entity_decl->_get_declarative_region()->get_kind() == IIR_LIBRARY_DECLARATION  );

  return parse_architecture( (IIR_LibraryDeclaration *)entity_decl->_get_declarative_region(),
			     entity_decl->get_declarator(), arch_as_text_literal );
}


// goes out and retrieves secondary units
IIR_ArchitectureDeclaration *
library_manager::parse_architecture( IIR_LibraryDeclaration *lib_decl,
				     IIR_TextLiteral *entity_name,
				     IIR_TextLiteral *architecture_name ){

  IIR_ArchitectureDeclaration *retval = NULL;
  char *arch_file_name = build_arch_file_name( entity_name, 
					       architecture_name,
					       lib_decl->_get_path_to_directory() );
  
  scram *tmp_scram = new scram();
  IIR_DesignFile *use_file = tmp_scram->parse_file( arch_file_name, lib_decl );
  
  if( use_file != NULL ){
    IIR_Declaration *decl = use_file->library_units.first();
    
    ASSERT( use_file->library_units.num_elements() == 1 );
    if( decl->get_kind() == IIR_ARCHITECTURE_DECLARATION ){
      IIR_ArchitectureDeclaration *arch_decl = (IIR_ArchitectureDeclaration *)decl;
      retval = arch_decl;
      my_symbol_table->lookup_add( retval );
    }
    else{
      ostrstream err;
      err << "Internal error in library_manager::parse_architecture - library file"
	  << arch_file_name << " didn't contain an architecture declaration - "
	  << " it contained a " << decl->get_kind_text() << " and I don't know what to do!"
	  << ends;

      report_error( architecture_name, err );

      abort();
    }
  }
  else{
    ostrstream err;
    err << "Library manager could not find library corresonding to architecture |"
	<< *architecture_name << "| for entity |" << *entity_name << "| - expected it "
	<< "to be in file |" << arch_file_name << "|.  Perhaps you need to parse the"
	<< " vhdl file containing this architecture?" << ends;
    report_error( architecture_name, err );
  }

  //  delete tmp_scram;
  delete [] arch_file_name;

  return retval;
}

void 
library_manager::create_directories( IIR_LibraryUnit *lib_unit ){
  IIR_Boolean need_to_delete_name = FALSE;
  char *directory_name;
  if( lib_unit->_is_primary_unit() == TRUE ){
    directory_name = get_work_dir_name();
  }
  else{
    ASSERT( lib_unit->_is_secondary_unit() == TRUE );
    directory_name = build_directory_name_for_secondary_unit( lib_unit );
  }

  file_manager::make_directory( directory_name );

  if( need_to_delete_name == TRUE ){
    delete [] directory_name;
  }
}

IIR_LibraryDeclaration *
library_manager::find_library( char *library_name ){
  IIR_LibraryDeclaration *retval;

  IIR_Identifier *temp_id = IIR_Identifier::get( library_name, strlen( library_name ) );
  retval = find_library( temp_id );
  // This is safe - find library method clones the name if it needs too.
  delete temp_id;

  return retval;
}

IIR_LibraryDeclaration *
library_manager::find_library( IIR_TextLiteral *library_name ){
  IIR_LibraryDeclaration *retval = NULL;  

  IIR_LibraryDeclaration *current = libraries->get_element();
  while( current != NULL ){
    if( IIR_TextLiteral::_cmp( library_name, current->get_declarator() ) == 0 ){
      retval = current;
      break;
    }
    current = libraries->get_next_element( );
  }
  
  if( retval == NULL ){
    // OK, then we need to find the library directory for this logical library,
    // complain if it doesn't exist, and create a library declaration if it does.
    char *lib_directory = find_library_directory( library_name );
    if( lib_directory != NULL ){
      retval = new IIR_LibraryDeclaration();
      
      retval->set_declarator( (IIR_TextLiteral *)library_name->_clone() );
      retval->_set_path_to_directory( lib_directory );
      libraries->add( retval );
    }
  }

  return retval;
}

char *
library_manager::find_library_directory( IIR_TextLiteral *logical_library_name ){
  char *retval = NULL;

  // Build the subdir name that would be associated with this library.
  char *physical_lib_name = build_library_directory_name( logical_library_name );

  // Now we're going to walk through the list of directories that libraries
  // could potentially be stored in, and try to find the library in one of
  // them.  Once we find it, we'll quit.
  char *full_path_string = 0;
  file_manager::file_status stat;
  bool done = false;
  const char **search_dirs = get_search_dirs();

  if( verbose_output == true ){
    cerr << "Search for directory for library |" << *logical_library_name 
	 << "|.  Looking in:\n";
  }

  int i;
  const char *current_dir;
  for( i = 0, current_dir = search_dirs[i]; current_dir != 0; 
       i++, current_dir = search_dirs[i] ){
    ostrstream full_path;
    full_path << current_dir << file_manager::get_directory_separator()
	      << physical_lib_name << ends;
    full_path_string = full_path.str();

    if( verbose_output == true ){
      cerr << "  " << full_path_string << " - ";
    }

    stat = file_manager::check_file_status( full_path_string, file_manager::DIRECTORY );
    if( stat == file_manager::OK ){
      if( verbose_output == true ){
	cerr << "FOUND" << endl;
      }
      break;
    }
    else{
      if( verbose_output == true ){
	cerr << "NOT FOUND" << endl;
      }
      full_path.freeze( 0 );
    }
  }
  if( stat == file_manager::OK ){
    retval = full_path_string;
  }
  else{
    //    full_path.freeze( 0 );
    ostrstream err;
    err << "Cannot find directory corresponding to logical library |" << *logical_library_name
	<< "|.  Run with \"-v\" flag to see search path." << ends;
    report_error( logical_library_name, err );
  }

  delete [] physical_lib_name;

  return retval;
}

IIR_LibraryUnit *
library_manager::_lookup_unit( IIR_Boolean complain_on_error,
			       IIR *unit_name,
			       IIR_LibraryDeclaration *library_declaration,
			       IIR_Kind unit_kind ){
  
  ASSERT( unit_name != NULL );
  IIR_LibraryUnit *retval = NULL;

  IIR_TextLiteral *unit_declarator = NULL;
  IIR_TextLiteral *library_declarator = NULL;
  if( unit_name->_is_iir_name() == TRUE ){
    unit_declarator = get_unit_name( (IIR_Name *)unit_name );
    library_declarator = get_library_name( (IIR_Name *)unit_name );
    if( library_declarator != NULL ){
      ASSERT( library_declaration == NULL ||
	      IIR_TextLiteral::_cmp( library_declaration->get_declarator(),
				     library_declarator ) == 0 );
    }
    else{
      if( library_declaration != NULL ){
	library_declarator = library_declaration->get_declarator();
      }
    }
  }
  else if(  unit_name->_is_iir_text_literal() == TRUE ){
    unit_declarator = (IIR_TextLiteral *)unit_name;
  }
  else{
    ostrstream err;
    err << "Internal error in library_manager::_lookup_unit - don't know"
	<< "what to do with unit name kind |" << unit_name->get_kind_text() << "|." << ends;
    report_error( unit_name, err );
    abort();
  }

  if( unit_declarator == NULL ){
    return NULL;
  }

  IIR_Boolean (IIR::*constraint_function)() = 0;
  switch( unit_kind ){
  case IIR_ENTITY_DECLARATION:{
    constraint_function = &IIR::_is_iir_entity_declaration;
    break;
  }
  case IIR_PACKAGE_DECLARATION:{
    constraint_function = &IIR::_is_iir_package_declaration;
    break;
  }
  case IIR_CONFIGURATION_DECLARATION:{
    constraint_function = &IIR::_is_iir_configuration_declaration;
    break;
  }
  default:{
    ostrstream err;
    err << "Internal error in library_manager::_lookup_unit - don't know"
	<< "what to do with kind |" << unit_kind << "|." << ends;
    report_error( unit_name, err );
    abort();
  }
  }

  set<IIR_Declaration> *decls = 
    new set<IIR_Declaration>(*my_symbol_table->find_set( unit_declarator ));
  
  if( decls != NULL ){
    decls->reduce_set( constraint_function );

    IIR_Declaration *current = decls->get_element();
    while( current != NULL ){
      ASSERT( (current->*constraint_function)() == TRUE );
      ASSERT( current->_get_declarative_region() != NULL );
      ASSERT( current->_get_declarative_region()->get_kind() == IIR_LIBRARY_DECLARATION );
      IIR_LibraryDeclaration *its_lib;
      its_lib = (IIR_LibraryDeclaration *)current->_get_declarative_region();
      if( library_declarator == NULL ){
	if( IIR_TextLiteral::_cmp( my_work_library->get_declarator(),
				   its_lib->get_declarator() ) == 0 ){
	  retval = (IIR_LibraryUnit *)current;
	}
      }
      else{
	if( IIR_TextLiteral::_cmp( library_declarator, its_lib->get_declarator() ) == 0 ){
	  retval = (IIR_LibraryUnit *)current;
	}
      }
      current = decls->get_next_element();
    }
  }
  
  delete decls;

  if( retval == NULL ){
    IIR_LibraryUnit *temp_retval;
    
    IIR_LibraryDeclaration *lib_decl;
    if( library_declarator != NULL ){
      lib_decl = find_library( library_declarator );
    }
    else{
      lib_decl = my_work_library;
    }
    
    temp_retval = get_primary_unit( lib_decl, unit_declarator, unit_kind, complain_on_error );
    ASSERT( temp_retval == NULL || temp_retval->get_kind() == unit_kind );
    retval = temp_retval;

    if( retval == NULL && complain_on_error == TRUE ){
      ostrstream err;
      err << "No primary unit |" << *unit_name << "| found." << ends;
      report_error( unit_name, err );
    }
  }

  return retval;
}

IIR_ArchitectureDeclaration *
library_manager::_lookup_architecture( IIR_Boolean complain_on_error, 
				       IIR_EntityDeclaration *entity,
				       IIR_Name *architecture_name ){
  IIR_ArchitectureDeclaration *retval = NULL;

  ASSERT( complain_on_error == TRUE ||  complain_on_error == FALSE );
  ASSERT( entity != NULL );
  
  IIR_TextLiteral *architecture_declarator = get_unit_name( architecture_name );
  IIR_TextLiteral *library_declarator =  get_library_name( architecture_name );

  set<IIR_Declaration> *decls = 
    new set<IIR_Declaration>(*my_symbol_table->find_set( architecture_declarator ));
  if( decls != NULL && decls->num_elements() > 0 ){
    decls->reduce_set( &IIR::_is_iir_architecture_declaration );
  
    IIR_Declaration *current_decl = decls->get_element();
    while( current_decl != NULL ){
      ASSERT( current_decl->_is_iir_architecture_declaration() == TRUE );
      IIR_ArchitectureDeclaration *current_arch_decl = (IIR_ArchitectureDeclaration *)current_decl;
      if( current_arch_decl->get_entity() != entity ){
	decls->remove( current_decl );
      }
      current_decl = decls->get_next_element();
    }
    
    int num_elements = decls->num_elements();
    int i;
    IIR_Declaration *current = decls->get_element();
    for( i = 0; i < num_elements - 1; i++ ){
      current = decls->get_next_element();
      ASSERT( current != NULL );
    }
    delete decls;
    retval = (IIR_ArchitectureDeclaration *)current;    
  }

  if( retval == NULL ){
    // Then we need to go to the library manager and try to find the
    // architecture there.
    if( library_declarator == NULL ){
      retval = parse_architecture( entity, architecture_name );
    }
    else{
      IIR_LibraryDeclaration *lib = find_library( library_declarator );
      if( lib != NULL ){
	retval = parse_architecture( lib, entity->get_declarator(), architecture_declarator );
      }
    }
    
    if( retval == NULL && complain_on_error == TRUE ){
      report_undefined_symbol( (IIR *)this );
    }
  }
  
    
  if(  retval == NULL && complain_on_error == TRUE ){
    ostrstream err;
    err << "No architecture |" << *architecture_name << "| declared for entity |" << 
      *entity->get_declarator() << "|." << ends;
    report_error( architecture_name, err );
  }
  
  return retval;
}

IIR_PackageDeclaration *
library_manager::_lookup_package( IIR_Boolean complain_on_error,
				  IIR *package_name,
				  IIR_LibraryDeclaration *lib_decl ){
  IIR_PackageDeclaration *retval = NULL;

  retval = (IIR_PackageDeclaration *)_lookup_unit( complain_on_error,
						   package_name,
						   lib_decl,
						   IIR_PACKAGE_DECLARATION );

  return retval;
}

IIR_EntityDeclaration *
library_manager::_lookup_entity( IIR_Boolean complain_on_error,
				 IIR_Name *entity_name,
				 IIR_LibraryDeclaration *lib_decl ){
  IIR_EntityDeclaration *retval = NULL;

  retval = (IIR_EntityDeclaration *)_lookup_unit( complain_on_error,
						  entity_name,
						  lib_decl,
						  IIR_ENTITY_DECLARATION );

  return retval;  
}

IIR_ConfigurationDeclaration *
library_manager::_lookup_configuration( IIR_Boolean complain_on_error,
					IIR_Name *configuration_name,
					IIR_LibraryDeclaration *lib_decl ){

  IIR_ConfigurationDeclaration *retval = NULL;
  retval = (IIR_ConfigurationDeclaration *)_lookup_unit( complain_on_error,
							 configuration_name,
							 lib_decl,
							 IIR_CONFIGURATION_DECLARATION );

  return retval;  
}

IIR_TextLiteral *
library_manager::get_library_name( IIR_Name *name ){
  IIR_TextLiteral *retval = NULL;

  switch( name->get_kind() ){
  case IIR_SIMPLE_NAME:{
    break;
  }
  case IIR_SELECTED_NAME:{
    if( name->get_prefix()->get_kind() == IIR_SIMPLE_NAME &&
	name->_get_suffix()->get_kind() == IIR_SIMPLE_NAME ){
      retval = name->get_prefix()->_get_prefix_string();
    }
    break;
  }
  default:{
    
  }
  }
  return retval;
}

IIR_TextLiteral *
library_manager::get_unit_name( IIR_Name *name ){
  IIR_TextLiteral *retval = NULL;

  switch( name->get_kind() ){
  case IIR_SIMPLE_NAME:{
    retval = name->_get_prefix_string();
    break;
  }
  case IIR_SELECTED_NAME:{
    if( name->get_prefix()->get_kind() == IIR_SIMPLE_NAME &&
	name->_get_suffix()->get_kind() == IIR_SIMPLE_NAME ){
      retval = name->_get_suffix()->_get_prefix_string();
    }
    break;
  }
  default:{
    
  }
  }
  
  return retval;
}

void
library_manager::add_declaration( IIR_LibraryUnit *to_add ){
  ASSERT( my_symbol_table != NULL );
  ASSERT( to_add != NULL );
  my_symbol_table->lookup_add( to_add );
}

char *
library_manager::savant_root_dir = 0;

const char **
library_manager::get_search_dirs(){
  // These static entries only initialize once by definition.
  static char **retval;
  static bool needToInit = true;
  if( needToInit == true ){
    // First, count how many there will be.  We'll start counting at 1
    // because we're going to put "." in the list, and we need a blank
    // entry to signal the end of the list.
    int count = 2;
    if( get_savant_root_directory() != 0 ){
      count++;
    }
    count += parse_vhdl_library_path( 0 );
    count += get_num_file_manager_directories();
    
    // Allocate the return array.
    retval = new char*[count];
    int i;
    for( i = 0; i < count; i++ ){
      retval[i] = 0;
    }

    // Order matters here.  We'll do user specified places first,
    // and then look in the standard places.
    retval[0] = new char[2];
    retval[0][0] = '.';
    retval[0][1] = (char)0;
    
    if( parse_vhdl_library_path( 0 ) > 0 ){
      parse_vhdl_library_path( retval );
    }
    if( get_savant_root_directory() != 0 ){
	ostrstream path;
	// We have to append "/lib" onto the specified path.
	path << get_savant_root_directory() << file_manager::get_directory_separator() 
	     << "lib" << ends;
	retval[ get_next_index( (const char **)retval ) ] = path.str();
    }
    const char *current;
    if( get_num_file_manager_directories() > 0 ){
      int i = 0;
      const char **dirs = file_manager::get_library_directories();
      current = dirs[i];
      while( current != 0 ){
	ostrstream path;
	// We append "savant/lib" onto whatever lib dirs the file manager
	// has.  If the user put their libs somewhere else, they should
	// have specified in VHDL_LIBRARY_PATH.
	path << current << file_manager::get_directory_separator() 
	     << "savant" << file_manager::get_directory_separator() 
	     << "lib" << ends;
	retval[ get_next_index( (const char **)retval ) ] = path.str();
	i++;
	current = dirs[i];
      }
    }
    // if -v, print out the search directories.

    if( verbose_output == true ){
      cerr << "Searching these subdirectories for libraries:\n";
      for( i = 0, current = retval[i]; current != 0; i++, current = retval[i] ){
	cout << "  "  << current << "\n";
      }
      cerr << endl;
    }
    
    needToInit = false;
    ASSERT( get_next_index( (const char **)retval ) == count - 1 );
  }
  return (const char **)retval;
}

unsigned int 
library_manager::parse_vhdl_library_path( char **fill_in ){
  unsigned int retval = 0;
  // So, if the argument == 0 then we're just counting for the moment.
  char *vhdl_library_path_string = getenv("VHDL_LIBRARY_PATH");
  if( vhdl_library_path_string  != 0 ){
    char *current = vhdl_library_path_string;
    char *begin = current;
    bool done = false;
    while( done == false ){
      while( *current != ':' && *current != 0 ){
	current++;
      }
      retval++;
      if( fill_in != 0 ){
	int index = get_next_index( (const char **)fill_in );
	unsigned int strleng = current - begin;
	fill_in[index] = new char[ strleng ];
	memcpy( fill_in[index], begin, strleng );
	fill_in[index][strleng] = 0;
	begin = current + 1;
      }
      if( *current == 0 ){
	done = true;
      }
      else{
	// Advance past the colon
	current++;
      }
    }
  }

  return retval;
}

// Returns the number of entries in the file manager's list.
unsigned int 
library_manager::get_num_file_manager_directories(){
  return get_next_index( file_manager::get_library_directories() );
}
  
// Returns the next available index for the array of directories.
unsigned int 
library_manager::get_next_index( const char **to_check ){
  unsigned int retval = 0;
  const char **dirs = to_check;
  while( dirs[retval] != 0 ){
    retval++;
  }

  return retval;
}
