// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ececs.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_Statement.cc,v 1.4 1999/10/26 16:44:45 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_Statement.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_CaseStatementAlternative.hh"
#include "IIR_CaseStatementAlternativeList.hh"
#include "IIR_Identifier.hh"
#include "IIR_SimpleName.hh"
#include "IIR_Label.hh"
#include "IIR_WaveformElement.hh"
#include "IIR_WaveformList.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "StandardPackage.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_PhysicalSubtypeDefinition.hh"
#include "IIR_ArrayTypeDefinition.hh"

IIRScram_Statement::~IIRScram_Statement() {
  delete _mangled_label;
}

void 
IIRScram_Statement::_type_check( ) {
  _report_undefined_scram_fn("_type_check()");
}


IIR_Boolean
IIRScram_Statement::_has_statement_list( ) {
  return FALSE;
}


set<IIR_Declaration> *
IIRScram_Statement::_find_declarations( IIR_Name *name ){
  _report_undefined_scram_fn("IIRScram_Statement::_find_declarations( IIR_Name *name )");

  return NULL;
}


void 
IIRScram_Statement::_get_signal_source_info(set<IIR_Declaration>* siginfo) {
  _report_undefined_scram_fn("_get_signal_source_info()");
}

IIR *
IIRScram_Statement::_type_check_and_resolve_boolean_condition( IIR *condition ){
  ASSERT( condition != NULL );
  
  if( condition->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
    IIR *tmp = ((IIR_AssociationElementByExpression *)condition)->get_actual();
    ASSERT( ((IIR_AssociationElementByExpression *)condition)->get_formal() != NULL );
    condition = tmp;
  }

  set<IIR_TypeDefinition> *bool_rvals = new set<IIR_TypeDefinition>(StandardPackage::boolean_type);

  set<IIR_TypeDefinition> *condition_rvals = condition->_get_rval_set();
  if( condition_rvals == NULL ){
    report_undefined_symbol( condition );
    return condition;
  }

  condition = condition->_semantic_transform( bool_rvals );
  condition->_type_check( bool_rvals );
  
  reconcile_sets( bool_rvals, condition_rvals );
  
  switch( condition_rvals->num_elements() ){
  case 0:{
    // Then there are no boolean rvals for condition.
    ostrstream err;
    err <<"|" << *condition << "| does not return a boolean, which makes "
	<< "it invalid as the condition in this context." << ends;
    report_error( condition, err );
    break;
    }
  case 1:{
    condition = condition->_rval_to_decl( condition_rvals->get_element() );
    if( condition->_is_readable() == FALSE ){
      ostrstream err;
      err << "Expression |" << *condition << "| is not readable, and not a valid "
	  << "condition in this context." << ends;
      report_error( condition, err );
    }
    break;
  }
  
  default:{
    report_ambiguous_error( condition, condition_rvals );
  }
  } // default:
  
  delete condition_rvals;
  delete bool_rvals;

  return condition;
}

void
IIRScram_Statement::_type_check_case_statement_expression( IIR_CaseStatementAlternativeList *alternatives ){
  IIR_TypeDefinition *correct_rval = NULL;

  ASSERT( _get_case_statement_expression() != NULL );
  set<IIR_TypeDefinition> *expression_rvals;
  
  expression_rvals = _get_case_statement_expression()->_get_rval_set();
  if( expression_rvals == NULL ){
    report_undefined_symbol( _get_case_statement_expression() );
    goto finish;
  }

  // Eliminate all of the non-discrete types.  According to the LRM,
  // a discrete type is an enumeration, a range type, or a one
  // dimensional array...
  expression_rvals->reduce_set( &IIR::_is_discrete_type );

  if( expression_rvals->num_elements() == 0 ){
    ostrstream err;
    err << "|" << *_get_case_statement_expression() << "| is an invalid expression for a "
        << "case statement (or selected signal assignment)." << ends;
    report_error( _get_case_statement_expression(), err );
    goto finish;
  }

  IIR_CaseStatementAlternative *current_alternative;
  current_alternative = alternatives->first();
  if( current_alternative->get_kind() != IIR_CASE_STATEMENT_ALTERNATIVE_BY_OTHERS ){
    set<IIR_TypeDefinition> *first_alt_rvals = current_alternative->_get_rval_set();
    if( first_alt_rvals == NULL ){
      report_undefined_symbol( current_alternative );
      goto finish;
    }
    reconcile_sets( expression_rvals, first_alt_rvals );
    delete first_alt_rvals;
  }
  
  _set_case_statement_expression( _get_case_statement_expression()->_semantic_transform( expression_rvals ) );

  switch( expression_rvals->num_elements() ){
  case 0:{
    ostrstream err;
    err << "|" << *_get_case_statement_expression() << "| and |" << *current_alternative 
	<< "| are " << "incompatible." << ends;
    report_error( _get_case_statement_expression(), err );
    goto finish;
  }
  case 1:{
    correct_rval = expression_rvals->get_element();
    _set_case_statement_expression( _get_case_statement_expression()->_semantic_transform( correct_rval ) );
    _get_case_statement_expression()->_type_check( correct_rval );
    _set_case_statement_expression( _get_case_statement_expression()->_rval_to_decl( correct_rval ) );
    break;
  }
  default:{
    report_ambiguous_error( _get_case_statement_expression(), expression_rvals );
    goto finish;
  }
  }

  current_alternative = alternatives->first();
  while( current_alternative != NULL ){
    if( current_alternative->get_kind() != IIR_CASE_STATEMENT_ALTERNATIVE_BY_OTHERS ){
      IIR *temp_alternative = current_alternative->_semantic_transform( correct_rval );
      temp_alternative->_type_check( correct_rval );
      temp_alternative = temp_alternative->_rval_to_decl( correct_rval );

      alternatives->_replace( current_alternative, temp_alternative );
    }
    // else do nothing.
    current_alternative = 
      alternatives->successor( current_alternative );
  }

 finish:
  // The semantic transform and type check might have narrowed the rvals
  // of our expression...
  delete expression_rvals;  
}

IIR *
IIRScram_Statement::_get_case_statement_expression(){
  _report_undefined_scram_fn("_get_case_statement_expression()");
  return NULL;
}

void 
IIRScram_Statement::_set_case_statement_expression( IIR * ){
  _report_undefined_scram_fn("_set_case_statement_expression( IIR * )");
}

void 
IIRScram_Statement::_type_check_target_and_waveform( IIR_WaveformList *waveform_list ){
  set<IIR_TypeDefinition> *waveform_rvals = waveform_list->first()->_get_rval_set();
  if( waveform_rvals == NULL ){
    report_undefined_symbol( waveform_list->first() );
    return;
  }
  
  _set_target( _get_target()->_semantic_transform( waveform_rvals ) );
  _get_target()->_type_check( waveform_rvals );

  set<IIR_TypeDefinition> *target_lvals = _get_target()->_get_rval_set(&IIR::_is_signal);

  if( target_lvals == NULL ){
    report_undefined_symbol( _get_target() );
    return;
  }

  IIR_TypeDefinition *target_lval = NULL;

  reconcile_sets( target_lvals, waveform_rvals );  
  switch( target_lvals->num_elements() ){
  case 0:{
    ostrstream err;
    err << "Incompatible types in assignment: " << *_get_target() <<
      " <= " << *waveform_list->first() << ends;
    report_error( _get_target(), err );
    return;
    break;
  }
  case 1:{
    target_lval = target_lvals->get_element();
    _set_target( _get_target()->_rval_to_decl( target_lval ) );
    if( _get_target()->_is_signal() == FALSE ){
      ostrstream err;
      err << "|" << *_get_target() << "| is not signal valued, and therefore may not be"
	  << " the target of a signal assignment statement." << ends;
      report_error( _get_target(), err );
      return;
    }
    if( _get_target()->_is_writable() == FALSE ){
      ostrstream err;
      err << "|" << *_get_target() << "| is not writable, and therefore may not be"
	  << " the target of a signal assignment statement." << ends;
      report_error( _get_target(), err );
      delete target_lvals;
      return;
    }
    break;
  }
  default:{
    report_ambiguous_error( _get_target(), target_lvals );
    return;
  }
  }
 
  delete target_lvals;

  // We have a whole list of waveforms that we need to check against...
  IIR_WaveformElement *current_waveform = waveform_list->first();
  IIR_WaveformElement *temp_waveform = current_waveform;
  while( current_waveform != NULL ){
    temp_waveform =
      (IIR_WaveformElement *)current_waveform->IIR::_semantic_transform( target_lval );

    temp_waveform->IIR::_type_check( target_lval );
    
    waveform_list->_replace( current_waveform, temp_waveform );
    current_waveform = temp_waveform;
    current_waveform = (IIR_WaveformElement *)current_waveform->_rval_to_decl( target_lval );
    ASSERT( current_waveform == temp_waveform );
    //    waveform_list->_replace( current_waveform, temp_waveform );

    if( current_waveform->_is_readable() == FALSE ){
      ostrstream err;
      err << "|" << *current_waveform << "| is not readable, and therefore may not be"
	  << " an expression of a signal assignment statement." << ends;
      report_error( _get_target(), err );
      return;      
    }

    current_waveform = waveform_list->successor( current_waveform );
  }

}

void 
IIRScram_Statement::_type_check_mechanism_and_time( IIR_WaveformList *waveform ){
  
  if( _get_delay_mechanism() == IIR_INERTIAL_DELAY ){
    // Either a reject time was specified, or not...
    if( _get_reject_time_expression() == NULL ){
      IIR_WaveformElement *first_waveform = waveform->first();
      ASSERT( first_waveform != NULL );
      IIR *time_expression = first_waveform->get_time();
      if( time_expression == NULL ){
	// Then the statement is something like x <= inertial y;
	// We should checnge it to transport...
	_set_delay_mechanism( IIR_TRANSPORT_DELAY );
      }
      else{
	ASSERT( time_expression->_is_resolved() == TRUE );
	_set_reject_time_expression( time_expression );
      }
    }
    else{
      IIR_TypeDefinition *time_rval = StandardPackage::time_type;
      _set_reject_time_expression(_get_reject_time_expression()->_semantic_transform( time_rval ));
      _get_reject_time_expression()->_type_check( time_rval );
      _set_reject_time_expression( _get_reject_time_expression()->_rval_to_decl( time_rval ) );
    }
  }  
}

IIR *
IIRScram_Statement::_get_target(){
  _report_undefined_scram_fn("_get_target()");
  return NULL;
}

void 
IIRScram_Statement::_set_target( IIR * ){
  _report_undefined_scram_fn("_set_target( IIR * )");
}

IIR_DelayMechanism
IIRScram_Statement::_get_delay_mechanism(){
  _report_undefined_scram_fn("_get_delay_mechanism");
  return IIR_TRANSPORT_DELAY;
}

void 
IIRScram_Statement::_set_delay_mechanism( IIR_DelayMechanism ){
  _report_undefined_scram_fn("_set_delay_mechanism( IIR_DelayMechanism )");
}

IIR *
IIRScram_Statement::_get_reject_time_expression(){
  _report_undefined_scram_fn("_get_reject_time_expression");
  return NULL;
}

void 
IIRScram_Statement::_set_reject_time_expression( IIR * ){
  _report_undefined_scram_fn("");
}

void
IIRScram_Statement::_mangle_label()
{
  IIR_Identifier *new_identifier;
  ostrstream identStream;
  char *identStr;
  
  if (get_label() != NULL) {
    identStream << *get_label();
    if ((get_label()->_get_declarative_region() != NULL) &&
	(_get_mangling_flag() == FALSE)) {
      identStream << *(get_label()->_get_declarative_region()->_get_declarator());
    }
  }
  else {
    identStream << "ALBL_" << this;
    if (_get_mangling_flag() == FALSE) {
      if (_current_architecture_name != NULL) {
	identStream << "A" << _current_architecture_name;
      }
    }
  }
    
  identStream << ends;

  identStr       = identStream.str();
  new_identifier = IIR_Identifier::get(identStr, strlen(identStr));

  _set_label(new IIR_Label());
  copy_location( this, _get_label() );
  _get_label()->set_declarator(new_identifier);

  delete [] identStr;
}
      
void
IIRScram_Statement::_set_label(IIR_Label* new_label){
  _mangled_label = new_label;
}

IIR_Label*
IIRScram_Statement::_get_label(){
  if (_mangled_label == NULL) {
    _mangle_label();
  }

  return _mangled_label;
}

// This and associated functions (_get_label(), _mangle_label(), _set_label())
// were duplicated in process, concurrent generate for & if, block, for loop
// statements earlier. Hence removed the redundant code from other places and
// moved it down here.

IIR_TextLiteral*
IIRScram_Statement::_get_declarator(){
  ASSERT (_get_label() != NULL);

  return _get_label()->_get_declarator();
}

IIR *
IIRScram_Statement::_get_assertion_condition(){
  _report_undefined_scram_fn("_get_assertion_condition()");
  return NULL;
}

void 
IIRScram_Statement::_set_assertion_condition( IIR * ){
  _report_undefined_scram_fn("_set_assertion_condition( IIR * )");
}

IIR *
IIRScram_Statement::_get_report_expression(){
  _report_undefined_scram_fn("_get_report_expression()");
  return NULL;
}

void 
IIRScram_Statement::_set_report_expression( IIR * ){
  _report_undefined_scram_fn(":_set_report_expression( IIR * )");
}

IIR *
IIRScram_Statement::_get_severity_expression(){
  _report_undefined_scram_fn("_get_severity_expression()");
  return NULL;
}

void 
IIRScram_Statement::_set_severity_expression( IIR * ){
  _report_undefined_scram_fn(":_set_severity_expression( IIR * )");
}

ostream &
IIRScram_Statement::_print( ostream &os ){
  if( get_label() != NULL ){
    os << *get_label();
  }
  else{
    os << "ANONYMOUS " << get_kind_text();
  }

  return os;
}

void 
IIRScram_Statement::_type_check_assertion_condition(){
  _set_assertion_condition( _type_check_and_resolve_boolean_condition( _get_assertion_condition()) );
}

void 
IIRScram_Statement::_type_check_report_expression(){
  // This code was copied too IIRScram_ConcurrentAssertionStatement.
  // Any changes here need to be made there, too.

  // Type check the expression
  set<IIR_TypeDefinition> *string_rvals = new set<IIR_TypeDefinition>( StandardPackage::string_type );
  _set_report_expression( _get_report_expression()->_semantic_transform( string_rvals ));
  _get_report_expression()->_type_check( string_rvals );
  
  set<IIR_TypeDefinition> *expression_rvals = _get_report_expression()->_get_rval_set();

  reconcile_sets( expression_rvals, string_rvals );

  switch( expression_rvals->num_elements() ){
  case 0:{
    ostrstream err;

    err << "|" << *_get_report_expression() << "| must have a string "
	<< "return value" << ends;
    report_error( this, err );

    break;
  }
  case 1:{
    IIR_TypeDefinition *my_rval = expression_rvals->get_element();
    _set_report_expression( _get_report_expression()->_rval_to_decl( my_rval ) );
    break;
  }
  default:{
    report_ambiguous_error( _get_report_expression(), expression_rvals );
  }
  }

  delete string_rvals;
  delete expression_rvals;
}


void 
IIRScram_Statement::_type_check_severity_expression(){  
  // This code was copied too IIRScram_ConcurrentAssertionStatement.
  // Any changes here need to be made there, too.

  set<IIR_TypeDefinition> *severity_rvals =
    new set<IIR_TypeDefinition>(StandardPackage::severity_level_type);
  _set_severity_expression( _get_severity_expression()->_semantic_transform( severity_rvals ));
  _get_severity_expression()->_type_check( severity_rvals );
  
  set<IIR_TypeDefinition> *expression_rvals = _get_severity_expression()->_get_rval_set();
  if( expression_rvals == NULL ){
    report_undefined_symbol( _get_severity_expression() );
    return;
  }
  
  reconcile_sets( expression_rvals, severity_rvals );
  switch( expression_rvals->num_elements() ){
  case 0:{
    ostrstream err;

    err << "|" << *_get_severity_expression() << "| must have a severity_leval "
	<< "return value" << ends;
    report_error( this, err );

    break;
  }
  case 1:{
    _set_severity_expression( _get_severity_expression()->_rval_to_decl( expression_rvals->get_element() ) );
    break;
  }
  default:{
    report_ambiguous_error( this, expression_rvals );
  }
  }

  delete severity_rvals;
  delete expression_rvals;
}

IIR_Label *
IIRScram_Statement::_find_instantiate_label( IIR_SimpleName *to_find ){
  return NULL;
}

void 
IIRScram_Statement::_make_interface_visible( symbol_table *add_to ){
  if( get_label() != NULL ){
    add_to->make_visible( get_label() );
  }
}
