//  UPrint.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "UPrint.h"
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <netinet/in.h>
#include <memory.h>
#include <iostream.h>
#include <fstream.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>

static void (*oldHandler)(int signum) = 0;
static void sigHandler (int signum);
static void sigAdd();
static void sigRemove();
static int sigGot();

int sigReceived=0;


#ifdef HAVE_MMAP
#include <fcntl.h>
#include <sys/mman.h>
#if !defined (__svr4__) && !defined (__SVR4) && defined (sun) 
extern "C" {
extern int munmap(void *, size_t);
}
#endif
#endif

#define HEADER_MARGIN  20.0
static double f2dot14 (short x);
static void  printFloat (ostream* ostr, double f);
static void  printUnic (ostream* ostr, int u);

static char* psCreator = "yudit 1.5 GNU (C) Gaspar Sinai";
static UCS2 headerStr[] = {'0', '1', '2', '3', '4', '5', '6', '7', 
   '8', '9', 'o', 'f', '.', '/', ' ', '-', ':', 0};

typedef struct _TableMap
{
	char*		name;
	UPrint::Tables	index;
}TableMap;

static TableMap tableMap[] = 
{	
	{"name",	UPrint::TB_NAME},
	{"head",	UPrint::TB_HEAD},
	{"hhea",	UPrint::TB_HHEA},
	{"post",	UPrint::TB_POST},
	{"glyf",	UPrint::TB_GLYPH},
	{"cmap",	UPrint::TB_CMAP},
	{"kern",	UPrint::TB_KERN},
	{"maxp",	UPrint::TB_MAXP},
	{"hmtx",	UPrint::TB_HTMX},
	{"loca",	UPrint::TB_LOCA}
};

static unsigned char base64chars[] =
	{"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"};

static char* base64names[] =
	{"A","B","C","D","E","F","G","H",
	 "I","J","K","L","M","N","O","P",
	 "Q","R","S","T","U","V","W","X",
	 "Y","Z","a","b","c","d","e","f",
	 "g","h","i","j","k","l","m","n",
	 "o","p","q","r","s","t","u","v",
	 "w","x","y","z","zero","one","two","three",
	 "four","five","six","seven","eight","nine","plus","slash"};

static char* reverse64=0;

//
// The following definitions, and basically the whole program  was derived from
// Andrew Weeks's excellent font converter program. Thanks!
//

#define BYTE unsigned char
#define CHAR signed char
#define USHORT unsigned short
#define SHORT signed short
#define ULONG unsigned int
#define LONG signed int
#define FWORD SHORT
#define UFWORD USHORT

#define ONOROFF 0x01
#define XSHORT  0x02
#define YSHORT  0x04
#define REPEAT  0x08
#define XSAME   0x10
#define YSAME   0x20
#define ARG_1_AND_2_ARE_WORDS           0x0001
#define ARGS_ARE_XY_VALUES                      0x0002
#define XY_BOUND_TO_GRID                        0x0004
#define WE_HAVE_A_SCALE                         0x0008
#define MORE_COMPONENTS                         0x0020
#define WE_HAVE_AN_X_AND_Y_SCALE        0x0040
#define WE_HAVE_A_TWO_BY_TWO            0x0080
#define WE_HAVE_INSTRUCTIONS            0x0100
#define USE_MY_METRICS                          0x0200

typedef struct short_2 {
        SHORT   upper;
        USHORT  lower;
} FIXED ;

typedef struct longhormetric {
        UFWORD  advanceWidth;
        FWORD   lsb;
} LONGHORMETRIC;

typedef struct ttf_hhea {
        BYTE    version[4];
        SHORT   ascender, descender, lineGap;
        USHORT  advnaceWidthMax;
        SHORT   minLSB, minRSB, xMaxExtent;
        SHORT   caretSlopeRise, caretSlopeRun;
        SHORT   reserved[5];
        SHORT   metricDataFormat;
        USHORT  numberOfHMetrics;
} TTF_HHEA;

typedef struct ttf_dir_entry {
        char    tag[4];
        ULONG   checksum;
        ULONG   offset;
        ULONG   length;
} TTF_DIR_ENTRY ;

typedef struct ttf_directory {
        ULONG                   sfntVersion;
        USHORT                  numTables;
        USHORT                  searchRange;
        USHORT                  entrySelector;
        USHORT                  rangeShift;
        TTF_DIR_ENTRY   list;
} TTF_DIRECTORY ;

typedef struct ttf_name_rec {
        USHORT  platformID;
        USHORT  encodingID;
        USHORT  languageID;
        USHORT  nameID;
        USHORT  stringLength;
        USHORT  stringOffset;
} TTF_NAME_REC;

typedef struct ttf_name {
        USHORT                  format;
        USHORT                  numberOfNameRecords;
        USHORT                  offset;
        TTF_NAME_REC    nameRecords;
} TTF_NAME ;

typedef struct ttf_head {
        ULONG   version;
        ULONG   fontRevision;
        ULONG   checksumAdjust;
        ULONG   magicNo;
        USHORT  flags;
        USHORT  unitsPerEm;
        BYTE    created[8];
        BYTE    modified[8];
        FWORD   xMin, yMin, xMax, yMax;
        USHORT  macStyle, lowestRecPPEM;
        SHORT   fontDirection, indexToLocFormat, glyphDataFormat;
} TTF_HEAD ;
typedef struct ttf_kern {
        USHORT  version, nTables;
} TTF_KERN ;

typedef struct ttf_kern_sub {
        USHORT version, length, coverage;
        USHORT nPairs, searchRange, entrySelector, rangeShift;
} TTF_KERN_SUB;

typedef struct ttf_kern_entry {
        USHORT  left, right;
        FWORD   value;
} TTF_KERN_ENTRY;

typedef struct ttf_cmap_fmt0 {
        USHORT  format;
        USHORT  length;
        USHORT  version;
        BYTE    glyphIdArray[256];
} TTF_CMAP_FMT0;

typedef struct ttf_cmap_fmt4 {
        USHORT  format;
        USHORT  length;
        USHORT  version;
        USHORT  segCountX2;
        USHORT  searchRange;
        USHORT  entrySelector;
        USHORT  rangeShift;
} TTF_CMAP_FMT4;

typedef struct ttf_cmap_entry {
        USHORT  platformID;
        USHORT  encodingID;
        ULONG   offset;
} TTF_CMAP_ENTRY;

typedef struct ttf_cmap {
        USHORT                  version;
        USHORT                  numberOfEncodingTables;
        TTF_CMAP_ENTRY  encodingTable[1];
} TTF_CMAP ;

typedef struct ttf_glyf {
        SHORT   numberOfContours;
        FWORD   xMin, yMin, xMax, yMax;
} TTF_GLYF ;

typedef struct ttf_maxp {
        ULONG   version;
        USHORT  numGlyphs, maxPoints, maxContours;
        USHORT  maxCompositePoints, maxCompositeContours;
        USHORT  maxZones, maxTwilightPoints, maxStorage;
        USHORT  maxFunctionDefs, maxInstructionsDefs;
        USHORT  maxSizeOfInstructions, maxComponentElements;
        USHORT  maxComponentDepth;
} TTF_MAXP ;

typedef struct ttf_post_head {
        ULONG   formatType;
        FIXED   italicAngle;
        FWORD   underlinePosition;
        FWORD   underlineThickness;
        ULONG   isFixedPitch;
        ULONG   minMemType42;
        ULONG   maxMemType42;
        ULONG   minMemType1;
        ULONG   maxMemType1;
        USHORT  numGlyphs;
        USHORT  glyphNameIndex;
} TTF_POST_HEAD ;


UPrintMap::UPrintMap()
{
	glyphIndex = -1;
}

UPrintMap::~UPrintMap()
{
}

UPrint::UPrint (const char* fontIn, double fontSizeIn, int tabStopsIn, 
  double headerHeightIn)
{
	unsigned int	i;
	memset (printMap, 0, 256 * sizeof (UPrintMap**));
	fontFile = (const char*) fontIn;
	fileBuffer=0;
	if (reverse64==0)
	{
		reverse64 = new char[256];
		CHECKNULL (reverse64);
		memset (reverse64, 0x7f, 256);
		for (i=0; i<sizeof (base64chars)-1; i++)
		{
			reverse64[base64chars[i]] = i;
		}
	}
	for (i=0; i<(int)TN_MAX; i++)
	{
		name_fields[i] = 0;
	}
	documentTitle = "Untitled";
	mediaName = "A4";
	mediaWidth = 595.0;
	mediaHeight = 842.0;
	marginWidth = 64.0;
	marginHeight = 48.0;
	headerFontSize = (headerHeightIn > 48.0) ? 48.0 : headerHeightIn;
	headerHeight =  (headerFontSize==0.0) ? 0.0 
	    : headerFontSize + headerFontSize/3 + HEADER_MARGIN;

	fontSize =  fontSizeIn;
	if (fontSize<1.0) fontSize = 1.0;
	tabChars = tabStopsIn;
	if (tabChars<1) tabChars = 8;
	fd = -1;
}

UPrint::~UPrint ()
{
}

UPrint::UStatus
UPrint::print (ostream* printerIn, const UCS2* const linesIn[], const int lineSizes[], int countIn, const UCS2* headerIn, const int headerL)
{
	int		i;
	int		j;
	unsigned short	indexR;
	unsigned short	indexC;
	struct stat	statbuf;
	TTF_DIRECTORY*	directory;
	TTF_DIR_ENTRY*	dir_entry;
	int		long_offsets;

#ifndef HAVE_MMAP
	ifstream* 	fontStream;
#endif
	TTF_HEAD*	head_table;
	UPrintMap*	xPrintMap;
	TTF_HHEA*	hhea_table;
	double		lastWidth;
	int		newPage;
	
	errors = (const char*) 0;
	printer = printerIn;
	pageCount = 0;
	// get the character map.
	clear ();
	headerString = headerIn;
	headerLength = headerL;
	if (stat ((const char*) fontFile, &statbuf) == -1)
	{
		errors = "error: can not access font '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}

	fileBufferLen = statbuf.st_size;

#ifdef HAVE_MMAP
	fd = open ((const char*) fontFile, O_RDONLY); 
	if (fd ==-1)
	{
		errors = "error: can not access font '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	fileBuffer = (char*) mmap (0, fileBufferLen,  PROT_READ, MAP_PRIVATE, fd, 0);
	if (fileBuffer==0)
	{
		errors = "error: can not mmap font  '";
		errors += (const char*) fontFile;
		errors += "'.";
		close (fd);
		fd = -1;
		return ERROR;
	}
#else 
	fileBuffer = new char[fileBufferLen];
	if (fileBuffer==0)
	{
		errors = "error: not enough memory to read '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	fontStream = new ifstream ((const char*) fontFile);
	CHECKNULL (fontStream);
	if (!fontStream->is_open())
	{
		errors = "error: can not read '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	if ( !fontStream->read (fileBuffer, fileBufferLen))
	{
		errors = "error: can not read whole file '";
		errors += (const char*) fontFile;
		errors += "'.";
		delete fontStream;
		return ERROR;
	}
	delete fontStream;
#endif
	directory = (TTF_DIRECTORY *) fileBuffer;
	if (ntohl(directory->sfntVersion) != 0x00010000)
	{
		errors = "error: '";
		errors += (const char*) fontFile;
		errors += "' is not a TrueType font file.";
		return ERROR;
	}

	
	dir_entry = &(directory->list);
	memset (tables, 0, TB_MAX * sizeof (void*));
	for (i=0; i < ntohs(directory->numTables); i++)
	{
		for (j=0; j< int (TB_MAX); j++)
		{
			if (memcmp(dir_entry->tag, tableMap[j].name, 4)==0)
			{
				tables[tableMap[j].index]
				  = fileBuffer + ntohl(dir_entry->offset);
			}
			if (memcmp(dir_entry->tag, "EBDT", 4)==0
			 || memcmp(dir_entry->tag, "EBLC", 4)==0
			 || memcmp(dir_entry->tag, "EBSC", 4)==0)
			{
				cerr << "warn: font contains bitmaps.\n";
			}
		}
		dir_entry++;
	}

	if (handleName()!=OK)
	{
		clear();
		return ERROR;
	}
	if (checkTables()!=OK)
	{
		clear();
		return ERROR;
	}
	// handle_head
	head_table = (TTF_HEAD*) tables[TB_HEAD];
	hhea_table = (TTF_HHEA*) tables[TB_HHEA];


	long_offsets = ntohs(head_table->indexToLocFormat);
	if (long_offsets != 0 && long_offsets != 1)
	{
		errors = "error: '";
		errors += (const char*) fontFile;
		errors += "' indexToLocFormat wrong.";
		clear();
		return ERROR;
	}
	// glyph list
	// postscript table grossly ignored :)

	// Find out what fonts are needed
	for (i=0; i<countIn; i++)
	{
		addMap (linesIn[i], lineSizes[i]);
	}
	// Add header characters if necessary
	if (headerHeight > 0.0)
	{
		addMap (headerStr, sizeof (headerStr) / sizeof (UCS2));
		if (headerIn!=0) addMap (headerIn, headerL);
	}

	if (buildCmap()==ERROR)
	{
		clear();
		return ERROR;
	}
	if (buildHtmx()==ERROR)
	{
		clear();
		return ERROR;
	}

	sigAdd ();
	if (printFont()==ERROR || sigGot())
	{
		clear();
		return ERROR;
	}


	*printer << "%%BeginResource: procset for setting fonts\n";
	for (i=0; i<segmentCount; i++)
	{
		*printer << "/SF" << i << "{\n";
		*printer << "/" << (const char*) name_fields[TN_FONTNAME] 
			<< "_" <<i;
		*printer << " findfont " << fontSize << " scalefont setfont\n";
		*printer << "} bind def\n";
		if (headerHeight>0.0)
		{
			*printer << "/SH" << i << "{\n";
			*printer << "/" 
			   << (const char*) name_fields[TN_FONTNAME] << "_" <<i;
			*printer << " findfont " << headerFontSize 
			   << " scalefont setfont\n";
			*printer << "} bind def\n";
		} 
		if (sigGot())
		{
			clear();
			return ERROR;
		}
	}
	*printer << "%%EndResource: fonts\n";
	*printer << "%%EndProlog\n";

	// Print Body Here


	scaleFactor = fontSize / (double)((short)ntohs(head_table->unitsPerEm));
	lineHeight = scaleFactor * (double)
			((short)htons (hhea_table->ascender) - 
			(short)htons (hhea_table->descender) + 
			(short)htons (hhea_table->lineGap)) + 1.0;

	charWidth =  scaleFactor * (double)
		          ((short) ntohs(head_table->xMax) 
			   + (short) ntohs(head_table->xMin)) + 1.0;

	charDescent = scaleFactor * (double)
			-((short)htons (hhea_table->descender)) + 1.0;

	// count the pages.
	pageCount = 0;
	newPage = 1;
	fontSwitch = "SF";
	for (i=0; i<countIn; i++)
	{
		if (newPage!=0)
		{
			printChar (0, 0, NEWPAGETEST);
			pageCount++;
			newPage = 0;
		}
		for (j=0; j < lineSizes [i]; j++)
		{
			if (newPage!=0)
			{
				printChar (0, 0, NEWPAGETEST);
				pageCount++;
				newPage = 0;
			}
			indexR = linesIn[i][j] >> 8;
			indexC = linesIn[i][j] & 0xff;

			// Sanity
			if (printMap[indexR][indexC] == 0)
			{
				continue;
			}
			xPrintMap = printMap[indexR][indexC];
			lastWidth = printChar (linesIn[i][j],
			    xPrintMap, TESTONLY);
			// linefeed char
			if (lastWidth == 0.0)
			{
				printChar (0, 0, NEWPAGETEST);
				pageCount++;
				continue;
			}
			if (lastWidth>0.0) continue;
			// Print before new line
			if (cursorX == marginWidth)
			{
				// print here.
				cursorX += lastWidth;

				if (j+1 < lineSizes[i])
				{
					cursorX = marginWidth;
					cursorY-=lineHeight;
					if (cursorY<marginHeight)
					{
						newPage = 1;
					}
				}
				continue;
			}
			// Print after new line
			cursorX = marginWidth;
			cursorY-=lineHeight;
			if (cursorY<marginHeight)
			{
				printChar (0, 0, NEWPAGETEST);
				pageCount++;
			}
		}
		cursorY-=lineHeight;
		cursorX = marginWidth;
		if (cursorY<marginHeight) newPage=1;
		if (sigGot())
		{
			clear();
			return ERROR;
		}
	}

	// print the actual pages here 
	pageIndex = 0;
	newPage = 1;
	printChar (0, 0, START);
	wrappedCount = 0;
	for (i=0; i<countIn; i++)
	{
		if (newPage!=0)
		{
			printChar (0, 0, NEWPAGE);
			newPage = 0;
		}
		for (j=0; j < lineSizes[i]; j++)
		{
			if (newPage!=0)
			{
				printChar (0, 0, NEWPAGE);
				newPage = 0;
			}
			indexR = linesIn[i][j] >> 8;
			indexC = linesIn[i][j] & 0xff;

			// Sanity
			if (printMap[indexR][indexC] == 0)
			{
				continue;
			}
			xPrintMap = printMap[indexR][indexC];
			lastWidth = printChar (linesIn[i][j], xPrintMap, DRAW);

			// linefeed 
			if (lastWidth == 0.0)
			{
				printChar (0, 0, NEWPAGE);
				continue;
			}
			if (lastWidth>0) continue;
			// Print before new line
			if (cursorX == marginWidth)
			{
				// print anyway.
				lastWidth = printChar(linesIn[i][j], xPrintMap,
				     FORCE);

				if (j+1 < lineSizes[i])
				{
					wrappedCount++;
					cursorX = marginWidth;
					cursorY-=lineHeight;
					if (cursorY<marginHeight)
					{
						newPage = 1;
					}
				}
				continue;
			}
			// Print after new line
			printChar (0, 0, FINISH);
			cursorX = marginWidth;
			cursorY-=lineHeight;
			if (cursorY<marginHeight)
			{
				printChar (0, 0, NEWPAGE);
			}
			else
			{
				printChar (0, 0, MOVE);
			}
			lastWidth = printChar(linesIn[i][j], xPrintMap, FORCE);
			wrappedCount++;
		}
		cursorY -= lineHeight;
		cursorX = marginWidth;
		if (cursorY<marginHeight)
		{
			newPage = 1;
		}
		else
		{
			printChar (0, 0, FINISH);
			printChar (0, 0, MOVE);
		}
		if (sigGot())
		{
			clear();
			return ERROR;
		}
	}
	printChar (0, 0, FINISH);
	if (pageIndex > 0)
	{
		*printer << "showpage restore\n\n";
	}
	*printer << "%%DocumentFonts: Times-Roman\n";
	*printer << "%%Pages: ";
	*printer << pageCount;
	*printer << "\n";
	*printer << "%%Trailer\n";
	*printer << "%%EOF\n";
	printer->flush();
	clear();
	return OK;
}

void
UPrint::addMap (const UCS2* lineIn, const int _size)
{
	int		i;
	unsigned short	indexR;
	unsigned short	indexC;
	for (i=0; i<_size; i++)
	{
		indexR = lineIn[i] >> 8;
		indexC = lineIn[i] & 0xff;
		if (printMap[indexR]==0)
		{
			printMap[indexR] = new UPrintMap*[256];
			CHECKNULL (printMap[indexR]);
			memset (printMap[indexR], 0, sizeof(UPrintMap*) * 256);
		}
		if (printMap[indexR][indexC] != 0) continue;
		printMap[indexR][indexC] = new UPrintMap();
	}
	return;
}

void
UPrint::doHeader ()
{
	int		i;
	char		pageStr[64]; 
	double   	length;
	double		headerScale;
	double 		baseLine;
	double 		headerLine;
	int		dateLen;
	int		pageLen;
	
	UCS2		ucs2;
	time_t		now;
	struct tm*	localTime;

	// Believe or not 0.0 can be expressed
	if (headerHeight==0.0) return;


	// Do the header numbering.
	sprintf (pageStr, "%d of %d", pageIndex, pageCount);
	length = 0.0;
	dateLen = 0;
	pageLen = 0;
	for (i=0; pageStr[i] != 0; i++)
	{
		length += printMap[0][pageStr[i]]->width;
	}
	// lets reuse this 
	headerScale = scaleFactor * headerFontSize/fontSize;
	length = length * headerScale;
	// print it out
	printChar (0, 0, FINISH);
	fontSwitch = "SH";
	cursorX = mediaWidth - length - marginWidth;
	headerLine = mediaHeight - marginHeight  + HEADER_MARGIN/2
	   - headerHeight + 1.0;
	baseLine =  headerLine +  headerFontSize/3;
	cursorY = baseLine; 
	printChar (0, 0, MOVE);
	for (i=0; pageStr[i] != 0; i++)
	{
		pageLen += printMap[0][pageStr[i]]->width;
		printChar (pageStr[i], printMap[0][pageStr[i]], FORCE);
	}
	pageLen += printMap[0][' ']->width;
	printChar (0, 0, FINISH);

	time(&now);
	localTime = localtime (&now);

	if (localTime!=0)
	{
		memset (pageStr, 0, sizeof (pageStr));
		// Let's stick to international format
		//strftime (pageStr, sizeof (pageStr)-1, "%m/%d/%Y", localTime);
		strftime (pageStr, sizeof (pageStr) - 1, "%Y-%m-%d %H:%M:%S", localTime);
		cursorX = marginWidth;
		cursorY = baseLine; 
		printChar (0, 0, MOVE);
		for (i=0; pageStr[i] != 0 && i < sizeof (pageStr); i++)
		{
			dateLen += printMap[0][pageStr[i]]->width;
			printChar (pageStr[i], printMap[0][pageStr[i]], FORCE);
		}
		dateLen += printMap[0][' ']->width;
		printChar (0, 0, FINISH);
	}
	// Draw line
	*printer << "1.0 setlinewidth newpath " << marginWidth << " " 
	  << headerLine 
	  << " moveto " << mediaWidth - marginWidth << " "
	  << headerLine 
	  << " lineto stroke\n";
	
	if (headerString == 0)
	{
		fontSwitch = "SF";
		return;
	}
	length =  2 * printMap[0]['.']->width;
	for (i=headerLength-1; i >= 0; i--)
	{
		ucs2 = headerString[i];
		length += printMap[ucs2>>8][ucs2&0xff]->width;
		//if (length*headerScale >= mediaWidth/2) break;
		if ((length+dateLen+2*pageLen)*headerScale >= mediaWidth - 2*marginWidth) break;
	}
	// lets reuse this 
	length = length * headerScale;
	// print it out
	printChar (0, 0, FINISH);
	
	if (dateLen*headerScale + marginWidth < mediaWidth/2 - length/2)
	{
		cursorX = mediaWidth/2 - length/2;
	}
	else
	{
		cursorX = dateLen*headerScale + marginWidth;
	}

	cursorY = baseLine; 
	printChar (0, 0, MOVE);
	if (i>0)
	{
		printChar ('.', printMap[0]['.'], FORCE);
		printChar ('.', printMap[0]['.'], FORCE);
	}
	else
	{
		i=0;
	}
	while (i<headerLength)
	{
		ucs2 = headerString[i];
		printChar (ucs2, printMap[ucs2>>8][ucs2&0xff], FORCE);
		i++;
	}
	printChar (0, 0, FINISH);
	fontSwitch = "SF";
}

// return 0 on pagefeed, negative if can not fit.
double
UPrint::printChar (int ucs2, UPrintMap* xPrintMap, DrawType drawType)
{
	double		tabTo;
	double		deltaX;
	int		counter;
	int		denom;
	
	switch (drawType)
	{
	case START:
		fontNow = -64;
		return 0.0;
	case FINISH:
		if (fontNow >= 0) *printer << ") show\n";
		*printer << "\n";
		fontNow = -64;
		return 0.0;
	case NEWPAGETEST:
		cursorY=mediaHeight-marginHeight -headerHeight-lineHeight;
		cursorX = marginWidth;
		return 0.0;
	case NEWPAGE:
		printChar (0, 0, FINISH);
		if (pageIndex > 0)
		{
			*printer << "showpage restore\n\n";
		}
		pageIndex++;
				
		*printer << "%%Page: " << pageIndex << " " 
			<< pageCount << "\n"; *printer << "save\n";
		doHeader ();
		cursorY=mediaHeight-marginHeight -headerHeight-lineHeight;
		cursorX = marginWidth;
		*printer << cursorX << " " << cursorY << " moveto ";
		return 0.0;
	case MOVE:
		*printer << cursorX << " " << cursorY << " moveto ";
		return 0.0;
	default:
		break;
	}

	// Handle control characters.
	if (ucs2<' ') switch (ucs2)
	{
	case '\f': return 0.0;
	case '\t':
		tabTo = charWidth * tabChars * 100.0;
		if (tabTo<1.01) tabTo = 1.01;
		counter = (int) (100*(cursorX-marginWidth));
		denom = (int) tabTo;
		deltaX = tabTo + tabTo * (counter/denom);
		deltaX = deltaX/100.0 - (cursorX-marginWidth);
		cursorX += deltaX;
		if (drawType != TESTONLY)
		{
			if (fontNow>=0) *printer << ") show\n";
			*printer << cursorX << " " << cursorY << " moveto ";
		}
		fontNow = -64;
		if (deltaX<=0.0) deltaX = 0.1;
		if (cursorX <= mediaWidth)
		{
			return deltaX;
		}
		return -deltaX;
			
	}
	if (ucs2<' ' || xPrintMap->psFont==-1)
	{
		// It it too long to print ?
		deltaX = charWidth;
		if (mediaWidth-marginWidth < deltaX+cursorX)
		{
			if (drawType != FORCE)
			{
				if (deltaX<=0.0) deltaX = 0.1;
				return -deltaX;
			}
		}
		cursorX += deltaX;
		if (drawType==TESTONLY) return deltaX;
		if (fontNow<0)
		{
			// SGC *printer << "SF0 (";
			*printer << fontSwitch << "0 (";
			fontNow=0;
		}
		// Most fonts dont have wide space.
		if (ucs2==0x3000) 
		{
			*printer << " ";
		}
		// Most fonts dont have paragraph separator
		else if (ucs2==0x2029)
		{
			*printer << " ";
		}
		// Most fonts dont have line separator
		else if (ucs2==0x2028)
		{
			*printer << " ";
		}
		else
		{
			*printer << "?";
		}
		return deltaX;
	}

	// It is a printable character...
	deltaX = scaleFactor * (double) xPrintMap->width;
	if (deltaX==0.0) deltaX = -0.1;
	if (mediaWidth-marginWidth< deltaX+cursorX)
	{
		if (drawType==TESTONLY) return -deltaX;
		if (drawType!=FORCE) return -deltaX;
	}

	if (xPrintMap->psFont/64!=fontNow/64)
	{
		if (drawType!=TESTONLY && fontNow>=0)
		{
			*printer << ") show\n";
		}
		fontNow=xPrintMap->psFont;
		if (drawType!=TESTONLY)
		{
			*printer << fontSwitch << fontNow/64 << " "; 
			*printer << "(";
		}
	}
	cursorX += deltaX;
	if (drawType!=TESTONLY)
	{
		*printer << base64chars[xPrintMap->psFont%64];
	}
	return deltaX;
}

void
UPrint::printProlog()
{
	time_t		now;

	time(&now);
	*printer << "%!PS-Adobe-3.0\n";
	*printer << "%%BoundingBox: ";
	*printer << marginWidth << " " << marginHeight << " ";
	*printer << mediaWidth - marginWidth << " ";
	*printer << mediaHeight - marginHeight << "\n";
	*printer << "%%Title: ";
	*printer << (const char*) documentTitle << "\n";
	*printer << "%%Creator: ";
	*printer << psCreator << "\n";
	*printer << "%%CreationDate: " << (char*) ctime(&now);
	*printer << "%%Orientation: Portrait\n";
	*printer << "%%DocumentMedia: " << (const char*) mediaName << " ";
	*printer << mediaWidth << " "  << mediaHeight << "  0 () ()\n";
	*printer << "%%Pages: (atend)\n";
	*printer << "%%EndComments\n\n";
	*printer << "%%BeginProlog\n\n";
}

void
UPrint::clear()
{
	int		i;
	int		j;

	for (i=0; i<256; i++)
	{
		if (printMap[i] !=0)
		{
			for (j=0; j<256; j++)
			{
				if (printMap[i][j] != 0) delete printMap[i][j];
			}
			delete  printMap[i];
			printMap[i]=0;
		}
	}
#ifdef HAVE_MMAP
	if (fileBuffer!=0) munmap (fileBuffer, fileBufferLen);
	if (fd != -1) close (fd);
#else 
	if (fileBuffer!=0) delete fileBuffer;
#endif
	fd = -1;
	fileBuffer=0;
	fileBufferLen=0;
	for (i=0; i<(int) TN_MAX; i++)
	{
		if (name_fields[i]!=0) delete name_fields[i];
		name_fields[i] = 0;
	}
	sigRemove();
	if (sigGot())
	{
		errors = "Error: could not print.\n";
		errors += "The stream was broken.\n";
		errors += "Possible reason: print command is not found.";
	}
	sigReceived=0;
}

UPrint::UStatus
UPrint::checkTables ()
{
	TTF_POST_HEAD*  post_table;

	if (tables[TB_HEAD] == 0)
	{
		errors = "error: can not find font head table in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	if (tables[TB_HHEA] == 0)
	{
		errors = "error: can not find font hhea table in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	if (tables[TB_GLYPH] == 0)
	{
		errors = "error: can not find font glyf table in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	if (tables[TB_CMAP] == 0)
	{
		errors = "error: can not find font cmap table in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	if (tables[TB_HTMX] == 0)
	{
		errors = "error: can not find font cmap table in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	if (tables[TB_LOCA] == 0)
	{
		errors = "error: can not find font loca table in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}
	post_table = (TTF_POST_HEAD*) tables[TB_POST];
	if (post_table == 0)
	{
		cerr << "warn: can not find font postscript table in '";
		cerr << (const char*) fontFile;
		cerr << "'. using sane defaults\n";
		italicAngle = 0;
		underlineThickness = 100;
		underlinePosition = 0;
		isFixedPitch = 0;
	}
	else
	{
		italicAngle = (ntohs(post_table->italicAngle.upper)) +
			(ntohs(post_table->italicAngle.lower) / 65536.0);
		underlineThickness = (short)ntohs(post_table->underlineThickness);
		underlinePosition = (short)ntohs(post_table->underlinePosition);
		isFixedPitch = (ntohl(post_table->isFixedPitch))? 1 : 0;
	}
	
	return OK;
}

UPrint::UStatus
UPrint::handleName ()
{
	int		i;
	short		lang;
	short		len;
	short		platform;
	short		nameId;
	short		strOffset;

	int 		found;
	TTF_NAME*	name_table = (TTF_NAME*) tables[TB_NAME];
	TTF_NAME_REC*	name_record;
	char*		string_area;

	string_area = (char *)name_table + ntohs(name_table->offset);
	name_record = &(name_table->nameRecords);
    
	found=0;
	for (i=0; i < ntohs(name_table->numberOfNameRecords); i++) 
	{
        	platform = ntohs(name_record->platformID);
        	if (platform == 3)
		{
			found = 1;
			lang = ntohs(name_record->languageID) & 0xff;
			len = ntohs(name_record->stringLength);
			strOffset = ntohs(name_record->stringOffset);
			if (lang == 0 || lang == 9)
			{
				nameId = ntohs(name_record->nameID);
 				if (nameId < (int)TN_MAX)
				{
					getName ((Names) nameId,
						&string_area[strOffset],
						len);
				}
			}
		}
        	name_record++;
	}

	name_record = &(name_table->nameRecords);
	if (!found) for (i=0; 
		i < ntohs(name_table->numberOfNameRecords); i++) 
	{
        	platform = ntohs(name_record->platformID);
        	if (platform ==1)
		{
			found = 1;
			lang = ntohs(name_record->languageID) & 0xff;
			len = ntohs(name_record->stringLength);
			strOffset = ntohs(name_record->stringOffset);
			if (lang == 0 || lang == 9)
			{
				nameId = ntohs(name_record->nameID);
 				if (nameId < (int)TN_MAX)
				{
					getName ((Names) nameId,
						&string_area[strOffset],
						len);
				}
			}
		}
        	name_record++;
	}

	if (!found)
	{
		errors = "error: cannot decode font name fields in '";
		errors += (const char*) fontFile;
		errors += "'.";
		return ERROR;
	}

	if (name_fields[TN_FONTNAME] == NULL)
	{
		getName (TN_FONTNAME, name_fields[TN_FULLNAME], 
			strlen(name_fields[TN_FULLNAME]));
	}

	return OK;
}


void
UPrint::getName(Names id, char* str, int len)
{
	int	i;
	int	j;

	if (name_fields[id]!=0) delete name_fields[id];
	name_fields[id] = new char[len+1];
	CHECKNULL (name_fields[id]);
	for (i=0, j=0; i<len; i++)
	{
		if (str[i] == 0) continue;
		if (id==TN_FONTNAME)
		{
			if (isalnum(str[i]))
			{
				name_fields[id][j] = str[i];
			}
			else
			{
				name_fields[id][j] = '_';
			}
		}
		else switch (str[i])
		{
		case '(':
			name_fields[id][j] = '[';
			break;
		case ')':
			name_fields[id][j] = ']';
			break;
		default:
			name_fields[id][j] = str[i];
		}
		j++;
	}
	name_fields[id][j] = 0;
}

UPrint::UStatus
UPrint::buildCmap ()
{
    int num_tables;
    BYTE *ptr;
    int i, j, k, format, offset, seg_c2, found;
    int platform, encoding_id;
    TTF_CMAP_ENTRY *table_entry;
    TTF_CMAP_FMT4 *encoding4;

    USHORT start, end, ro;
    unsigned short n;
    short delta;
    TTF_CMAP *cmap_table;
    short cmap_n_segs;
    USHORT *cmap_seg_start, *cmap_seg_end;
    short *cmap_idDelta, *cmap_idRangeOffset;

    cmap_table = (TTF_CMAP*) tables[TB_CMAP];
    num_tables = ntohs(cmap_table->numberOfEncodingTables);
    found = 0;

    for (i=0; i < num_tables && !found; i++) {
        table_entry = &(cmap_table->encodingTable[i]);
        offset = ntohl(table_entry->offset);
        encoding4 = (TTF_CMAP_FMT4 *) ((BYTE *)cmap_table + offset);
        format = ntohs(encoding4->format);
        platform = ntohs(table_entry->platformID);
        encoding_id = ntohs(table_entry->encodingID);

        if (platform == 3 && format == 4 && encoding_id==1) 
        {
            found = 1;
            seg_c2 = ntohs(encoding4->segCountX2);
            cmap_n_segs = seg_c2 >> 1;
            ptr = (BYTE *)encoding4 + 14;
            cmap_seg_end = (USHORT *) ptr;
            cmap_seg_start = (USHORT *) (ptr + seg_c2 + 2);
            cmap_idDelta = (short *) (ptr + (seg_c2 * 2 )+ 2);
            cmap_idRangeOffset = (short *) (ptr + (seg_c2 * 3) + 2);

            for (j=0; j < cmap_n_segs-1; j++) {
                start = ntohs(cmap_seg_start[j]);
                end   = ntohs(cmap_seg_end[j]);
                delta = ntohs(cmap_idDelta[j]);
                ro    = ntohs(cmap_idRangeOffset[j]);

                for (k = start; k <= end; k++)
                {
                    if (delta != 0) {
                        n = k + delta;
                    } else {
                        n = (unsigned short) ntohs(*( (ro >> 1) + (k - start) +
                                    &(cmap_idRangeOffset[j])));
                    }
                    // k is the unicode index and n is the glyph index.
                    // set the glyph indeces
                    if (printMap[k>>8] !=0 && printMap[k>>8][k&0xff]!=0) {
                          printMap[k>>8][k&0xff]->glyphIndex = n;
                    }
                }
            }
        }
    }

    if (!found)
    {
	errors = "error: '";
	errors += (const char*) fontFile;
	errors += "' does not have unicode encoding.";
	return ERROR;
    }

    return OK;
}

UPrint::UStatus
UPrint::buildHtmx()
{
	TTF_HHEA*	hhea_table;
	LONGHORMETRIC*	hmtx_entry;
	int		i;
	FWORD		*lsblist;
	UPrintMap*	pPrintMap;
	int		n_hmetrics;

	hhea_table = (TTF_HHEA*) tables[TB_HHEA];
	hmtx_entry = (LONGHORMETRIC*) tables[TB_HTMX];

	n_hmetrics = ntohs(hhea_table->numberOfHMetrics);
	lsblist = (FWORD *) &hmtx_entry[n_hmetrics];

	// Gather info for those that will be printed
	for (i = 1; i <= 0xffff; i++)
	{
		if (printMap[i>>8] !=0 
			&& printMap[i>>8][i&0xff]!=0
			&& printMap[i>>8][i&0xff]->glyphIndex>=0)
		{
			pPrintMap=printMap[i>>8][i&0xff];
			if (pPrintMap->glyphIndex >= n_hmetrics)
			{
				pPrintMap->width = ntohs(hmtx_entry[n_hmetrics-1].advanceWidth);
				pPrintMap->lsb = ntohs(lsblist[pPrintMap->glyphIndex-n_hmetrics]);
				continue;
			}
			pPrintMap->width = ntohs(hmtx_entry[pPrintMap->glyphIndex].advanceWidth);
			pPrintMap->lsb = ntohs(hmtx_entry[pPrintMap->glyphIndex].lsb);
		}
	}
	return OK;
}

UPrint::UStatus
UPrint::printFont()
{
	int		i;
	int		j;
	int		index;
	TTF_HEAD*	head_table;
	TTF_HHEA*	hhea_table;
	double		scale_factor;
	int		nextChar;
	int		base64Index;
	int		blockSize;
	short		xBox[2];
	short		yBox[2];

	index = 0;
	head_table = (TTF_HEAD*) tables[TB_HEAD];
	hhea_table = (TTF_HHEA*) tables[TB_HHEA];
	
	for (i=0; i<256; i++)
	{
		if (printMap[i]==0) continue;
		for (j=0; j<256; j++)
		{
			if (printMap[i][j]==0) continue;
			if ((i<< 8) + j < ' ')
			{
				printMap[i][j]->psFont = -1;
				continue;
			}
			if (printMap[i][j]->glyphIndex==-1)
			{
				// Most fonts dont have wide space
				if (((i<<8)|j) != 0x3000)
				{
					cerr << "warn: font does not have '";
					printUnic (&cerr, (i<<8)|j);
					cerr << "'.\n";
				}
				printMap[i][j]->psFont = -1;
				continue;
			}
			printMap[i][j]->psFont = index;
			index++;
		}
	}
	if (index==0)
	{
		errors = "error: nothing left to print.";
		return ERROR;
	}
	printProlog();
	// print the postscript header and fonts.
	*printer << "%%BeginResource: fonts\n";
	*printer << "% Converted from TrueType font " << name_fields[TN_FONTNAME] << "\n";
	segmentCount = (index+63)/64;
	*printer << "% Font Segments " << segmentCount << "\n";
	nextChar = -1;
	for (i=0; i<segmentCount; i++)
	{
		blockSize = (index-i*64 >= 64) ? 64 : index % 64;
		*printer << "20 dict begin\n/FontType 3 def\n";
		*printer << "/FontName (" 
		    << (const char*) name_fields[TN_FONTNAME] 
		    << "_" << i << ") def\n";
		*printer << "/FontInfo 9 dict dup begin\n";
		*printer << "/FullName ("
		    << (const char*) name_fields[TN_FULLNAME]
		    << "_" << segmentCount << ") def\n";
		*printer << "/Notice ("
		    << (const char*) name_fields[TN_NOTICE] << ") def\n";
		*printer << "/FamilyName ("
		    << (const char*) name_fields[TN_FAMILY] << ") def\n";
		*printer << "/Weight ("
		    << (const char*) name_fields[TN_WEIGHT] << ") def\n";
		*printer << "/version ("
		    << (const char*) name_fields[TN_VERSION] << ") def\n";
		*printer << "/italicAngle " ;
		printFloat (printer, italicAngle);
		*printer  << " def\n";

		scale_factor = 1000.0 / ntohs(head_table->unitsPerEm);
		*printer << "/underlineThickness " << underlineThickness << " def\n";
		*printer << "/underlinePosition " << underlinePosition << " def\n";
		*printer << "/isFixedPitch " 
		   << (char*) ((isFixedPitch)?"true":"false")<< " def end def\n";
		*printer << "/FontMatrix [";
		printFloat (printer, scale_factor/1000.0);
		*printer << " 0 0 ";
		printFloat (printer, scale_factor/1000.0);
		*printer << " 0 0] def\n";
		*printer << "/FontBBox ["
		    << (short)ntohs(head_table->xMin) << " "
		    << (short)ntohs(head_table->yMin) << " "
		    << (short)ntohs(head_table->xMax) << " "
		    << (short)ntohs(head_table->yMax) << "] def\n";
		*printer << "/Encoding [";

		// Do I need to have all 256 elements?
		for (j=0; j<256; j++)
		{
			if (j%8==0) *printer << "\n";
			base64Index = reverse64[j];
			// wide space is usually not present in font.
			if (j== ' ')
			{
				*printer << "/space ";
			}
			else if (base64Index>=blockSize)
			{
				*printer << "/.notdef ";
			}
			else
			{
				*printer << "/" << base64names[base64Index] << " ";
			}
		}
		*printer << "] def\n/CharProcs " << blockSize + 1 ;
		*printer << " dict def CharProcs begin\n";

		// Wide space
		*printer << "/space { % Wide space\n";
    		*printer << ((short) ntohs(head_table->xMax) 
			+ (short) ntohs(head_table->xMin)) << " 0 ";

		*printer << (short) ntohs(head_table->xMin) << " "
		    << (short) ntohs(head_table->yMin) << " "
		    << (short) ntohs(head_table->xMax) << " "
		    << (short) ntohs(head_table->yMax) << " setcachedevice\n";

		*printer << "fill } bind def\n";

		// Empty glyph
		*printer << "/.notdef { % Empty glyph\n";
    		*printer << ((short) ntohs(head_table->xMax) + (short) ntohs(head_table->xMin))
		    << " 0 ";
		*printer << (short) ntohs(head_table->xMin) << " "
		    << (short) ntohs(head_table->yMin) << " "
		    << (short) ntohs(head_table->xMax) << " "
		    << (short) ntohs(head_table->yMax) << " setcachedevice\n";
		//*printer << underlineThickness);

		//*printer << " setlinewidth\n";
		xBox[0] = (short) underlineThickness;
		xBox[1] = (short) ntohs(head_table->xMax) 
			+ (short) ntohs(head_table->xMin);
/*
		xBox[1] = (short) ntohs(hhea_table->advnaceWidthMax) 
			- 2 * underlineThickness;
*/
		yBox[0] = underlinePosition;
		yBox[1] = (short) ntohs(head_table->yMax)
			- 2 * underlineThickness;
		
		*printer << xBox[0] << " " << yBox[0] << " moveto ";
		*printer << xBox[0] << " " << yBox[1] << " lineto stroke closepath\n";
		*printer << "newpath ";
		*printer << xBox[0] << " " << yBox[1] << " moveto ";
		*printer << xBox[1] << " " << yBox[1] << " lineto stroke closepath\n";
		*printer << "newpath ";
		*printer << xBox[1] << " " << yBox[1] << " moveto ";
		*printer << xBox[1] << " " << yBox[0] << " lineto stroke closepath\n";
		*printer << "newpath ";
		*printer << xBox[1] << " " << yBox[0] << " moveto ";
		*printer << xBox[0] << " " << yBox[0] << " lineto stroke closepath\n";
		
		*printer << "} bind def\n";
		
		for (j=0; j<blockSize; j++)
		{
			nextChar++;
			while (printMap[nextChar>>8]==0 
		   	    || printMap[nextChar>>8][nextChar&0xff]==0
		   	    || printMap[nextChar>>8][nextChar&0xff]->psFont < 0)
			{
				nextChar++;
				if (nextChar > 65535) break;
			}
			if(nextChar>65535)
			{
				cerr << "error: internal error in printFont.\n";
				return ERROR;
			}
			doGlyph (nextChar, j);
		}
		*printer << "end\n/BuildGlyph {\n";
		*printer << "exch /CharProcs get exch\n";
		*printer << "2 copy known not {pop /.notdef} if ";
		*printer << "get exec } bind def\n";
		*printer << "/BuildChar { 1 index /Encoding get exch get\n";
		*printer << "1 index /Encoding get exec } bind def\n";
		*printer << "currentdict end /" 
		    << (const char*) name_fields[TN_FONTNAME] 
		    << "_" << i << " exch definefont pop\n";
	}
	*printer << "%%EndFont\n";
	*printer << "%%EndResource: fonts\n";

	return OK;
}
void
UPrint::doGlyph(int unicode, int base64Code)
{
    int len;
    short ncontours;
    USHORT flagbyte, glyphindex;
    SHORT arg1, arg2;
    BYTE *ptr;
    char *bptr;
    SHORT *sptr;
    double matrix[4];
    ULONG *long_loca_table;
    USHORT *short_loca_table;
    TTF_HEAD *head_table;
    BYTE *glyf_start;
    TTF_GLYF* glyf_table;
    TTF_GLYF* unaligned_glyf_table;
    TTF_GLYF* aligned_glyf_table;
    unsigned long  glp;
    int long_offsets;
    UPrintMap*	pmap;
    int glyphno;
    int ncurves;

    long_loca_table = (ULONG *) tables[TB_LOCA];
    short_loca_table = (USHORT *) tables[TB_LOCA];
    head_table = (TTF_HEAD*) tables[TB_HEAD];
    long_offsets = ntohs(head_table->indexToLocFormat);
    glyf_start = (BYTE *) tables[TB_GLYPH];
    pmap = printMap[unicode>>8][unicode&0xff];
    glyphno = pmap->glyphIndex;

    ncurves = 0;
    

    if (long_offsets) {
        unaligned_glyf_table = (TTF_GLYF *) (glyf_start + ntohl(long_loca_table[glyphno]));
        len = ntohl(long_loca_table[glyphno+1]) - ntohl(long_loca_table[glyphno]);
    } else {
        unaligned_glyf_table = (TTF_GLYF *) (glyf_start + (ntohs(short_loca_table[glyphno]) << 1));
        len = (ntohs(short_loca_table[glyphno+1]) - ntohs(short_loca_table[glyphno])) << 1;
    }

    glp = (unsigned long) unaligned_glyf_table;
    glyf_table = unaligned_glyf_table;
    aligned_glyf_table = 0;
    if ((glp&1)!=0)
    {
	cerr << "warn: fixing broken font (glyf_table) at " << glyphno << ".\n";
	aligned_glyf_table = new TTF_GLYF[len]; 
	memcpy (aligned_glyf_table, glyf_table, len * sizeof (TTF_GLYF));
	glyf_table = aligned_glyf_table;
    }
    *printer << "/" << base64names[base64Code]<<" { % " << base64Code ;
    printUnic (printer, unicode);
    *printer << "\n";

    // These guys get initialized here
    pmap->xMin = ntohs(glyf_table->xMin);
    pmap->yMin = ntohs(glyf_table->yMin);
    pmap->xMax = ntohs(glyf_table->xMax);
    pmap->yMax = ntohs(glyf_table->yMax);

    *printer <<  pmap->width << " 0 "
        << pmap->xMin << " "
        << pmap->yMin << " "
        << pmap->xMax << " "
        << pmap->yMax << " setcachedevice\n";

    if (len != 0) {
        ncontours = ntohs(glyf_table->numberOfContours);

        if (ncontours <= 0) {
            ptr = ((BYTE *)glyf_table + sizeof(TTF_GLYF));
            sptr = (SHORT *) ptr;
            do {
                flagbyte = ntohs(*sptr); sptr ++;
                glyphindex = ntohs(*sptr); sptr ++;

		*printer << "% flags " << flagbyte << " glyph " << 
			base64names[base64Code] << "\n";

                if (flagbyte & ARG_1_AND_2_ARE_WORDS) {
                    arg1 = ntohs(*sptr); sptr++;
                    arg2 = ntohs(*sptr); sptr++;
                } else {
                    bptr = (char *)sptr;
                    arg1 = (signed char)bptr[0];
                    arg2 = (signed char)bptr[1];
                    sptr ++;
                }
                matrix[1] = matrix[2] = 0.0;

                if (flagbyte & WE_HAVE_A_SCALE) {
                    matrix[0] = matrix[3] = f2dot14(*sptr);
                    sptr ++;
                }
                else if (flagbyte & WE_HAVE_AN_X_AND_Y_SCALE) {
                    matrix[0] = f2dot14(*sptr); sptr ++;
                    matrix[3] = f2dot14(*sptr); sptr ++;
                }
                else if (flagbyte & WE_HAVE_A_TWO_BY_TWO) {
                    matrix[0] = f2dot14(*sptr); sptr ++;
                    matrix[1] = f2dot14(*sptr); sptr ++;
                    matrix[2] = f2dot14(*sptr); sptr ++;
                    matrix[3] = f2dot14(*sptr); sptr ++;
                } else {
                    matrix[0] = matrix[3] = 1.0;
                }

		*printer << "matrix currentmatrix\n[ ";
		printFloat (printer, matrix[0]); *printer << " ";
		printFloat (printer, matrix[1]); *printer << " ";
		printFloat (printer, matrix[2]); *printer << " ";
		printFloat (printer, matrix[3]); *printer << " ";
		*printer << arg1 << " ";
		*printer << arg2 << "] concat\n";

                ncurves += drawGlyph(glyphindex, glyphno, long_offsets);

		*printer << "setmatrix\n";

            } while (flagbyte & MORE_COMPONENTS);
        } else {
            ncurves += drawGlyph(glyphno, glyphno, long_offsets);
        }
        if (ncurves > 100)
        {
		*printer << "% This glyph is too long, may have to be removed\n";
        }
    }
    *printer << "fill } bind def\n";
    if (aligned_glyf_table!=0) delete aligned_glyf_table;
}

int
UPrint::drawGlyph(int glyphno, int parent, int long_offsets)
{
    ULONG *long_loca_table;
    USHORT *short_loca_table;
    int i, j, k, k1, len, first, cs, ce;
    int finished, nguide, contour_start, contour_end;
    short ncontours, n_inst, last_point;
    USHORT *contour_end_pt;
    BYTE *ptr;
    short xcoord[2000], ycoord[2000], xrel[2000], yrel[2000];
    BYTE flags[2000];
    TTF_GLYF* glyf_table;
    char pfa_file[256];
    int ncurves;
    BYTE *glyf_start;
    TTF_GLYF* unaligned_glyf_table;
    TTF_GLYF* aligned_glyf_table;
    unsigned long  glp;

    long_loca_table = (ULONG *) tables[TB_LOCA];
    short_loca_table = (USHORT *) tables[TB_LOCA];
    glyf_start = (BYTE *) tables[TB_GLYPH];
    ncurves =0;

    if (long_offsets) {
        unaligned_glyf_table = (TTF_GLYF *) (glyf_start + ntohl(long_loca_table[glyphno]));
        len = ntohl(long_loca_table[glyphno+1]) - ntohl(long_loca_table[glyphno]);
    } else {
        unaligned_glyf_table = (TTF_GLYF *) (glyf_start + (ntohs(short_loca_table[glyphno]) << 1));
        len = (ntohs(short_loca_table[glyphno+1]) - ntohs(short_loca_table[glyphno])) << 1;
    }

    contour_start = 0;
    ce = contour_start;

    if (len <= 0) {
	*printer <<"%**** Composite glyph %s refers to non-existent glyph";
	*printer << parent << " " << glyphno << "\n";
        return 0;
    }

    glp = (unsigned long) unaligned_glyf_table;
    glyf_table = unaligned_glyf_table;
    aligned_glyf_table = 0;
    if ((glp&1)!=0)
    {
	// cerr << "warn: fixing broken font (glyf_table) at " << glyphno << ".\n";
	aligned_glyf_table = new TTF_GLYF[len]; 
	memcpy (aligned_glyf_table, glyf_table, len * sizeof (TTF_GLYF));
	glyf_table = aligned_glyf_table;
    }

    ncontours = ntohs(glyf_table->numberOfContours);
    if (ncontours <= 0) {
	*printer <<"%**** Composite glyph %s refers to non-existent glyph";
	*printer << parent << " " << glyphno << "\n";
	if (aligned_glyf_table!=0) delete aligned_glyf_table;
        return 0;
    }

    contour_end_pt = (USHORT *) ((char *)glyf_table + sizeof(TTF_GLYF));

    last_point = ntohs(contour_end_pt[ncontours-1]);
    n_inst = ntohs(contour_end_pt[ncontours]);

    ptr = ((BYTE *)contour_end_pt) + (ncontours << 1) + n_inst + 2;
    j = k = 0;
    while (k <= last_point) {
        flags[k] = ptr[j];

        if (ptr[j] & REPEAT) {
            for (k1=0; k1 < ptr[j+1]; k1++) {
                k++;
                flags[k] = ptr[j];
            }
            j++;
        }
        j++; k++;
    }

    for (k=0; k <= last_point; k++) {
        if (flags[k] & XSHORT) {
            if (flags[k] & XSAME) {
                xrel[k] = ptr[j];
            } else {
                xrel[k] = - ptr[j];
            }
            j++;
        } else if (flags[k] & XSAME) {
            xrel[k] = 0;
        } else {
            xrel[k] = ptr[j] * 256 + ptr[j+1];
            j += 2;
        }
        if (k==0) {
            xcoord[k] = xrel[k];
        } else {
            xcoord[k] = xrel[k] + xcoord[k-1];
        }
    }

    for (k=0; k <= last_point; k++) {
        if (flags[k] & YSHORT) {
            if (flags[k] & YSAME) {
                yrel[k] = ptr[j];
            } else {
                yrel[k] = - ptr[j];
            }
            j++;
        } else if (flags[k] & YSAME) {
            yrel[k] = 0;
        } else {
            yrel[k] = ptr[j] * 256 + ptr[j+1];
            j += 2;
        }
        if (k==0) {
            ycoord[k] = yrel[k];
        } else {
            ycoord[k] = yrel[k] + ycoord[k-1];
        }
    }

    i = j = 0;
    first = 1;

    while (i <= ntohs(contour_end_pt[ncontours-1])) {
        contour_end = ntohs(contour_end_pt[j]);

        if (first) {
            sprintf(pfa_file, "%d %d moveto\n", xcoord[i], ycoord[i]);
	    *printer << pfa_file;
            ncurves ++;
            contour_start = i;
            first = 0;
        } else if (flags[i] & ONOROFF) {
            sprintf(pfa_file, "%d %d lineto\n", xcoord[i], ycoord[i]);
	    *printer << pfa_file;
            ncurves ++;
        } else {
            cs = i-1;
            finished = nguide = 0;
            while (!finished) {
                if (i == contour_end+1) {
                    ce = contour_start;
                    finished = 1;
                } else if (flags[i] & ONOROFF) {
                    ce = i;
                    finished = 1;
                } else {
                    i++;
                    nguide++;
                }
            }

            switch (nguide) {
                case 0: 
			sprintf( pfa_file,"%d %d lineto\n",
				xcoord[ce], ycoord[ce]);
	    		*printer << pfa_file;
                        ncurves ++;
                        break;

                case 1: sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (xcoord[cs]+2*xcoord[cs+1])/3,
                                (ycoord[cs]+2*ycoord[cs+1])/3,
                                (2*xcoord[cs+1]+xcoord[ce])/3,
                                (2*ycoord[cs+1]+ycoord[ce])/3,
                                xcoord[ce], ycoord[ce]);
			*printer << pfa_file;
                        ncurves ++;
                        break;

                case 2: sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (-xcoord[cs]+4*xcoord[cs+1])/3,
                                (-ycoord[cs]+4*ycoord[cs+1])/3,
                                (4*xcoord[cs+2]-xcoord[ce])/3,
                                (4*ycoord[cs+2]-ycoord[ce])/3,
                                xcoord[ce], ycoord[ce]);
			*printer << pfa_file;
                        ncurves ++;
                        break;

                case 3: sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (xcoord[cs]+2*xcoord[cs+1])/3,
                                (ycoord[cs]+2*ycoord[cs+1])/3,
                                (5*xcoord[cs+1]+xcoord[cs+2])/6,
                                (5*ycoord[cs+1]+ycoord[cs+2])/6,
                                (xcoord[cs+1]+xcoord[cs+2])/2,
                                (ycoord[cs+1]+ycoord[cs+2])/2);
			*printer << pfa_file;
                        sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (xcoord[cs+1]+5*xcoord[cs+2])/6,
                                (ycoord[cs+1]+5*ycoord[cs+2])/6,
                                (5*xcoord[cs+2]+xcoord[cs+3])/6,
                                (5*ycoord[cs+2]+ycoord[cs+3])/6,
                                (xcoord[cs+3]+xcoord[cs+2])/2,
                                (ycoord[cs+3]+ycoord[cs+2])/2);
			*printer << pfa_file;
                        sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (xcoord[cs+2]+5*xcoord[cs+3])/6,
                                (ycoord[cs+2]+5*ycoord[cs+3])/6,
                                (2*xcoord[cs+3]+xcoord[ce])/3,
                                (2*ycoord[cs+3]+ycoord[ce])/3,
                                xcoord[ce], ycoord[ce]);
			*printer << pfa_file;
                        ncurves += 3;
                        break;

                default:k1 = cs + nguide;

                        sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (xcoord[cs]+2*xcoord[cs+1])/3,
                                (ycoord[cs]+2*ycoord[cs+1])/3,
                                (5*xcoord[cs+1]+xcoord[cs+2])/6,
                                (5*ycoord[cs+1]+ycoord[cs+2])/6,
                                (xcoord[cs+1]+xcoord[cs+2])/2,
                                (ycoord[cs+1]+ycoord[cs+2])/2);
			*printer << pfa_file;
                        for (k = cs+2; k <= k1-1; k++) {
                            sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                    (xcoord[k-1]+5*xcoord[k])/6,
                                    (ycoord[k-1]+5*ycoord[k])/6,
                                    (5*xcoord[k]+xcoord[k+1])/6,
                                    (5*ycoord[k]+ycoord[k+1])/6,
                                    (xcoord[k]+xcoord[k+1])/2,
                                    (ycoord[k]+ycoord[k+1])/2);
				*printer << pfa_file;
                        }
                        sprintf( pfa_file, "%d %d %d %d %d %d curveto\n",
                                (xcoord[k1-1]+5*xcoord[k1])/6,
                                (ycoord[k1-1]+5*ycoord[k1])/6,
                                (2*xcoord[k1]+xcoord[ce])/3,
                                (2*ycoord[k1]+ycoord[ce])/3,
                                xcoord[ce], ycoord[ce]);
			*printer << pfa_file;
                        ncurves += nguide;
                        break;
            }
        }
        if (i >= contour_end) {
	    *printer << " closepath ";
            first = 1;
            i = contour_end + 1;
            j++;
        } else {
            i++;
        }
    }
    if (aligned_glyf_table!=0) delete aligned_glyf_table;
    return ncurves;
}

static
double f2dot14 (short x)
{
	short y = ntohs(x);
	return (y >> 14) + ((y & 0x3fff) / 16384.0);
}

static void
printFloat (ostream* ostr, double f)
{
	char buffer[32];
	sprintf (buffer, "%9.7lf", f);
	*ostr << buffer;
	return;
}
static void
printUnic (ostream* ostr, int u)
{
	char buffer[32];
	sprintf (buffer, " - U+%4.4x", u);
	*ostr << buffer;
	return;
}

static void
sigAdd()
{
	void (*isOldHandler)(int signum) = 0;
	isOldHandler = signal (SIGPIPE, sigHandler);
	if (isOldHandler!=sigHandler)
	{
		oldHandler = isOldHandler;
	}
	sigReceived=0;
	return;
}

static void
sigRemove()
{
	if (oldHandler==0)
	{
		signal (SIGPIPE, SIG_IGN);
	}
	else
	{
		signal (SIGPIPE, oldHandler);
	}
	return;
}

static int
sigGot()
{
	return sigReceived;
}

static void
sigHandler (int signum)
{
	sigReceived=signum;
	return;
}

