;;; x-symbol-sgml.el --- token language "SGML entity" for package x-symbol

;; Copyright (C) 1996-1999 Free Software Foundation, Inc.
;;
;; Author: Christoph Wedler <wedler@fmi.uni-passau.de>
;; Maintainer: (Please use `M-x x-symbol-package-bug' to contact the maintainer)
;; Version: $Id: x-symbol-sgml.el,v 3.3 1999/01/18 14:16:58 wedler Exp wedler $
;; Keywords: WYSIWYG, HTML, wp, math, internationalization
;; X-URL: http://www.fmi.uni-passau.de/~wedler/x-symbol/

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; If you want to use package x-symbol, please visit the URL (use
;; \\[x-symbol-package-web]) and read the info (use \\[x-symbol-package-info]).

;; Token language sgml is registered in x-symbol-hooks.

;;; Code:

(provide 'x-symbol-sgml)
(require 'x-symbol-vars)


;;;===========================================================================
;;;  Miscellaneous variables
;;;===========================================================================

(defface x-symbol-sgml-non-l1-face
  '((((class color) (background light))
     (:foreground "orange4")))
  "*Face, normally used for tokens only allowed in TeX's math mode.
Used in `x-symbol-tex-class-face-alist'."
  :group 'x-symbol-sgml
  :group 'x-symbol-info-general)

(defcustom x-symbol-sgml-modeline-name "sgml"
  "*String naming the language SGML in the modeline."
  :group 'x-symbol-sgml
  :type 'string)

(defcustom x-symbol-sgml-header-groups-alist
  '(("Operator" bigop operator)
    ("Relation" relation)
    ("Arrow, Punctuation" arrow triangle shape
     white line dots punctuation quote parenthesis)
    ("Symbol" symbol currency mathletter setsymbol)
    ("Greek Letter" greek greek1)
    ("Misc. Letter" letter slash)
    ("Cedilla, Ogonek" cedilla ogonek)
    ("Dotaccent, Ring" dotaccent ring)
    ("Tilde, Breve" tilde breve)
    ("Circumflex, Caron" circumflex caron)
    ("Diaeresis, Umlaut" diaeresis hungarumlaut)
    ("Acute, Grave" acute grave))
  "*If non-nil, used in SGML specific grid/menu.
See `x-symbol-header-groups-alist'."
  :group 'x-symbol-sgml
  :group 'x-symbol-input-init
  :type 'x-symbol-headers)

(defcustom x-symbol-sgml-class-alist
  '((non-l1)
    (VALID "SGML entity" (x-symbol-info-face))
    (INVALID "no SGML entity" (red x-symbol-info-face)))
  "Alist for SGML's token classes displayed by info in echo area.
See `x-symbol-language-access-alist' for details."
  :group 'x-symbol-sgml
  :group 'x-symbol-info-strings
  :set 'x-symbol-set-cache-variable
  :type 'x-symbol-class-info)

(defcustom x-symbol-sgml-class-face-alist
  '((non-l1 x-symbol-sgml-non-l1-face (x-symbol-sgml-non-l1-face)))
  "Alist for SGML's color scheme in SGML's grid and info.
See `x-symbol-language-access-alist' for details."
  :group 'x-symbol-sgml
  :group 'x-symbol-input-init
  :group 'x-symbol-info-general
  :set 'x-symbol-set-cache-variable
  :type 'x-symbol-class-faces)

(defcustom x-symbol-sgml-electric-ignore nil
  "*Additional SGML version of `x-symbol-electric-ignore'."
  :group 'x-symbol-sgml
  :group 'x-symbol-input-control
  :type 'x-symbol-function-or-regexp)

(defvar x-symbol-sgml-required-fonts nil
  "List of features providing fonts for language `sgml'.")
(defvar x-symbol-sgml-case-insensitive nil
  "If non-nil, SGML entities are case insensitive.  They are not.")
(defvar x-symbol-sgml-token-shape nil
  "Shape of SGML entities.  See `x-symbol-language-access-alist'.")
(defvar x-symbol-sgml-exec-specs
  '(nil ("\\`&[#0-9A-Za-z]*;\\'" . "&[#0-9A-Za-z]*;"))
  "Specification for SGML's executables.
See `x-symbol-exec-create' and `x-symbol-language-access-alist'.")

(defvar x-symbol-sgml-token-list 'x-symbol-sgml-token-list-netscape
  "Function returning a list of SGML entities from table specification.
The TOKEN-SPEC in sgml table look like (NUMBER STRING...) where NUMBER
is the Unicode value.  See `x-symbol-init-language',
`x-symbol-sgml-token-list-name', `x-symbol-sgml-token-list-code' and
`x-symbol-sgml-token-list-netscape'.")

(defvar x-symbol-sgml-input-token-ignore nil
  "Regexp matching SGML entities not being used for input method TOKEN.
See `x-symbol-init-language'.")

(defvar x-symbol-sgml-user-table nil
  "User table defining SGML entities, used in `x-symbol-sgml-table'.")


;;;===========================================================================
;;;  Image support
;;;===========================================================================

(defcustom x-symbol-sgml-master-directory 'ignore
  "Function returning the directory of the master file or nil.
See `x-symbol-image-parse-buffer'."
  :group 'x-symbol-sgml
  :group 'x-symbol-image-language
  :type 'function)

(defcustom x-symbol-sgml-image-searchpath '("./")
  "Search path used for implicitly relative image file names.
See `x-symbol-image-use-remote'."
  :group 'x-symbol-sgml
  :group 'x-symbol-image-language
  :type '(repeat directory))

(defcustom x-symbol-sgml-image-cached-dirs '("images/" "pictures/")
  "Directory parts of image file names stored in the memory cache.
See `x-symbol-image-use-remote'."
  :group 'x-symbol-sgml
  :group 'x-symbol-image-language
  :type '(repeat string))

(defcustom x-symbol-sgml-image-file-truename-alist
  '(("\\`file:" . "")
    ("\\`[A-Za-z]+:"))
  "Alist used to determine the file name of an image URL.
Each element looks like
  (REGEXP) or
  (REGEXP . NEWTEXT) or
  (REGEXP FUNCTION ARG...)
If the the image file name is matched by REGEXP, the corresponding
element is processed, if no REGEXP matches, the image file name is used
as it is.  With the first form, the image command will not be
highlighted.  With the second form, replace text matched by REGEXP with
NEWTEXT, see `replace-match' for details.  With the third form,
FUNCTION, call FUNCTION with the image file name and the remaining
arguments ARGs to get the true file name.

E.g., I add the following element to this variable:
  (\"\\\\`http://www\\\\.fmi\\\\.uni-passau\\\\.de/~wedler/\" . \"~/public_html/\")"
  :group 'x-symbol-sgml
  :group 'x-symbol-image-language
  :type '(repeat (cons :format "%v"
		       :value ("" . "")	; doesn't work (custom bug?)
		       regexp
		       (choice ;;:value ""
			       (const :tag "Not highlighted" nil)
			       (string :tag "Replace match with")
			       (cons :tag "Call"
				     function
				     (repeat :tag "With arguments" sexp))))))

(defcustom x-symbol-sgml-image-keywords
  '("\\.\\(gif\\|png\\|jpe?g\\)\\'"
    ("<img[ \t][^\n>]*src=\"\\([^\n\"]+\\)\"[^\n>]*>"
     x-symbol-sgml-image-file-truename 1))
  "Keywords used to find image insertion commands.
See `x-symbol-image-parse-buffer'."
  :group 'x-symbol-sgml
  :group 'x-symbol-image-language
  :type 'x-symbol-image-keywords)

(defun x-symbol-sgml-image-file-truename (num)
  "Return true image file name for last match.
Return text matched by the NUMth regexp group of the corresponding
keyword regexp, after being processed according to
`x-symbol-sgml-image-file-truename-alist'."
  (x-symbol-match-in-alist (setq num (match-string num))
			   x-symbol-sgml-image-file-truename-alist
			   num t))


;;;===========================================================================
;;;  Super- and Subscripts
;;;===========================================================================

(defvar x-symbol-sgml-font-lock-keywords
  '((x-symbol-sgml-match-subscript
     (1 x-symbol-invisible-face t)
     (2 (if (eq (char-after (+ 3 (match-beginning 1))) ?b)
	    'x-symbol-sub-face 'x-symbol-sup-face)
	prepend)
     (3 x-symbol-invisible-face t)))
  "SGML's font-lock keywords for super- and subscripts.")

(defcustom x-symbol-sgml-font-lock-regexp "<su[bp]>"
  "Regexp matching the start tag of SGML's super- and subscripts.
See also `x-symbol-sgml-font-lock-alist'."
  :group 'x-symbol-sgml
  :type 'regexp)

(defcustom x-symbol-sgml-font-lock-limit-regexp "\n\\|</?su[bp]>"
  "Regexp matching the end tag of SGML's super- and subscripts.
This regexp should match the end of line and the closing tags in
`x-symbol-sgml-font-lock-alist'."
  :group 'x-symbol-sgml
  :type 'regexp)

(defcustom x-symbol-sgml-font-lock-contents-regexp "[^ \t\n\240]"
  "*Regexp matching the super- and subscript contents.
This regexp should match the text between the opening and closing super-
or subscript tag."
  :group 'x-symbol-sgml
  :type 'regexp)

(defcustom x-symbol-sgml-font-lock-alist
  '(("<sub>" . "</sub>") ("<sup>" . "</sup>"))
  "Alist for correct tag pairs for SGML's super- and subscripts.
Each element looks like (OPEN . CLOSE).  All keys OPEN in this alist
should be matched by `x-symbol-sgml-font-lock-regexp', all CLOSEs should
be matched by `x-symbol-sgml-font-lock-limit-regexp'."
  :group 'x-symbol-sgml
  :type '(repeat (cons :format "%v"
		       (string :tag "Open tag")
		       (string :tag "Close tag"))))


;;;===========================================================================
;;;  The tables
;;;===========================================================================

(defun x-symbol-sgml-token-list-name (tokens)
  "Return list of tokens from token specification TOKENS.
TOKENS = (UNICODE NAME-ENTITY...).  Return (NAME-ENTITY... &#UNICODE;)."
  (if (car tokens)
      (append (cdr tokens) (list (format "&#%d;" (car tokens))))
    (cdr tokens)))

(defun x-symbol-sgml-token-list-code (tokens)
  "Return list of tokens from token specification TOKENS.
TOKENS = (UNICODE NAME-ENTITY...).  Return (&#UNICODE; NAME-ENTITY...)."
  (if (car tokens)
      (cons (format "&#%d;" (car tokens)) (cdr tokens))
    (cdr tokens)))

(defun x-symbol-sgml-token-list-netscape (tokens)
  "Return list of tokens from token specification TOKENS.
TOKENS = (UNICODE NAME-ENTITY...).  Return (NAME-ENTITY... &#UNICODE;)
for Latin-1 characters, (&#UNICODE; NAME-ENTITY...) otherwise."
  (if (car tokens)
      (if (< (car tokens) 256)
	  (append (cdr tokens) (list (format "&#%d;" (car tokens))))
	(cons (format "&#%d;" (car tokens)) (cdr tokens)))
    (cdr tokens)))

;; http://www.w3.org/TR/REC-html40/sgml/entities.html
;; (query-replace-regexp "<!ENTITY[ \t]*\\([A-Za-z][A-Za-z0-9]*\\)[ \t]*CDATA[ \t]*\"&#\\([0-9]+\\);\"[ \t]*--\\(.+\\) -->[ \t]*.*$" " (\\1 () \\2 \"&\\1;\") ; \\3")

(defvar x-symbol-sgml-latin1-table
  '((nobreakspace () 160 "&nbsp;")
    (exclamdown () 161 "&iexcl;")
    (cent () 162 "&cent;")
    (sterling () 163 "&pound;")
    (currency () 164 "&curren;")
    (yen () 165 "&yen;")
    (brokenbar () 166 "&brvbar;" "&brkbar;")
    (section () 167 "&sect;")
    (diaeresis () 168 "&uml;" "&die;")
    (copyright () 169 "&copy;")
    (ordfeminine () 170 "&ordf;")
    (guillemotleft () 171 "&laquo;")
    (notsign () 172 "&not;")
    (hyphen () 173 "&shy;")
    (registered () 174 "&reg;")
    (macron () 175 "&macr;" "&hibar;")
    (degree () 176 "&deg;")
    (plusminus () 177 "&plusmn;")
    (twosuperior () 178 "&sup2;")
    (threesuperior () 179 "&sup3;")
    (acute () 180 "&acute;")
    (mu1 () 181 "&micro;")
    (paragraph () 182 "&para;")
    (periodcentered () 183 "&middot;")
    (cedilla () 184 "&cedil;")
    (onesuperior () 185 "&sup1;")
    (masculine () 186 "&ordm;")
    (guillemotright () 187 "&raquo;")
    (onequarter () 188 "&frac14;")
    (onehalf () 189 "&frac12;")
    (threequarters () 190 "&frac34;")
    (questiondown () 191 "&iquest;")
    (Agrave () 192 "&Agrave;")
    (Aacute () 193 "&Aacute;")
    (Acircumflex () 194 "&Acirc;")
    (Atilde () 195 "&Atilde;")
    (Adiaeresis () 196 "&Auml;")
    (Aring () 197 "&Aring;")
    (AE () 198 "&AElig;")
    (Ccedilla () 199 "&Ccedil;")
    (Egrave () 200 "&Egrave;")
    (Eacute () 201 "&Eacute;")
    (Ecircumflex () 202 "&Ecirc;")
    (Ediaeresis () 203 "&Euml;")
    (Igrave () 204 "&Igrave;")
    (Iacute () 205 "&Iacute;")
    (Icircumflex () 206 "&Icirc;")
    (Idiaeresis () 207 "&Iuml;")
    (ETH () 208 "&ETH;") ; "&Dstrok;" for Dbar (U0110) = latin2#208?
    (Ntilde () 209 "&Ntilde;")
    (Ograve () 210 "&Ograve;")
    (Oacute () 211 "&Oacute;")
    (Ocircumflex () 212 "&Ocirc;")
    (Otilde () 213 "&Otilde;")
    (Odiaeresis () 214 "&Ouml;")
    (multiply () 215 "&times;")
    (Ooblique () 216 "&Oslash;")
    (Ugrave () 217 "&Ugrave;")
    (Uacute () 218 "&Uacute;")
    (Ucircumflex () 219 "&Ucirc;")
    (Udiaeresis () 220 "&Uuml;")
    (Yacute () 221 "&Yacute;")
    (THORN () 222 "&THORN;")
    (ssharp () 223 "&szlig;")
    (agrave () 224 "&agrave;")
    (aacute () 225 "&aacute;")
    (acircumflex () 226 "&acirc;")
    (atilde () 227 "&atilde;")
    (adiaeresis () 228 "&auml;")
    (aring () 229 "&aring;")
    (ae () 230 "&aelig;")
    (ccedilla () 231 "&ccedil;")
    (egrave () 232 "&egrave;")
    (eacute () 233 "&eacute;")
    (ecircumflex () 234 "&ecirc;")
    (ediaeresis () 235 "&euml;")
    (igrave () 236 "&igrave;")
    (iacute () 237 "&iacute;")
    (icircumflex () 238 "&icirc;")
    (idiaeresis () 239 "&iuml;")
    (eth () 240 "&eth;")
    (ntilde () 241 "&ntilde;")
    (ograve () 242 "&ograve;")
    (oacute () 243 "&oacute;")
    (ocircumflex () 244 "&ocirc;")
    (otilde () 245 "&otilde;")
    (odiaeresis () 246 "&ouml;")
    (division () 247 "&divide;")
    (oslash () 248 "&oslash;")
    (ugrave () 249 "&ugrave;")
    (uacute () 250 "&uacute;")
    (ucircumflex () 251 "&ucirc;")
    (udiaeresis () 252 "&uuml;")
    (yacute () 253 "&yacute;")
    (thorn () 254 "&thorn;")
    (ydiaeresis () 255 "&yuml;"))
  "Table defining SGML entities, see `x-symbol-sgml-table'.")

(defvar x-symbol-sgml-latinN-table
  '((Aogonek (non-l1) 260)
    (breve (non-l1) 728)
    (Lslash (non-l1) 321)
    (Lcaron (non-l1) 317)
    (Sacute (non-l1) 346)
    (Scaron (non-l1) 352 "&Scaron;")
    (Scedilla (non-l1) 350)
    (Tcaron (non-l1) 356)
    (Zacute (non-l1) 377)
    (Zcaron (non-l1) 381)
    (Zdotaccent (non-l1) 379)
    (aogonek (non-l1) 261)
    (ogonek (non-l1) 731)
    (lslash (non-l1) 322)
    (lcaron (non-l1) 318)
    (sacute (non-l1) 347)
    (caron (non-l1) 711)
    (scaron (non-l1) 353 "&scaron;")
    (scedilla (non-l1) 351)
    (tcaron (non-l1) 357)
    (zacute (non-l1) 378)
    (hungarumlaut (non-l1) 733)
    (zcaron (non-l1) 382)
    (zdotaccent (non-l1) 380)
    (Racute (non-l1) 340)
    (Abreve (non-l1) 258)
    (Lacute (non-l1) 313)
    (Cacute (non-l1) 262)
    (Ccaron (non-l1) 268)
    (Eogonek (non-l1) 280)
    (Ecaron (non-l1) 282)
    (Dcaron (non-l1) 270)
    (Dbar (non-l1) 272)
    (Nacute (non-l1) 323)
    (Ncaron (non-l1) 327)
    (Ohungarumlaut (non-l1) 336)
    (Rcaron (non-l1) 344)
    (Uring (non-l1) 366)
    (Uhungarumlaut (non-l1) 368)
    (Tcedilla (non-l1) 354)
    (racute (non-l1) 341)
    (abreve (non-l1) 259)
    (lacute (non-l1) 314)
    (cacute (non-l1) 263)
    (ccaron (non-l1) 269)
    (eogonek (non-l1) 281)
    (ecaron (non-l1) 283)
    (dcaron (non-l1) 271)
    (dbar (non-l1) 273)
    (nacute (non-l1) 324)
    (ncaron (non-l1) 328)
    (ohungarumlaut (non-l1) 337)
    (rcaron (non-l1) 345)
    (uring (non-l1) 367)
    (uhungarumlaut (non-l1) 369)
    (tcedilla (non-l1) 355)
    (dotaccent (non-l1) 729)
    (Hbar (non-l1) 294)
    (Hcircumflex (non-l1) 292)
    (Idotaccent (non-l1) 304)
    (Gbreve (non-l1) 286)
    (Jcircumflex (non-l1) 308)
    (hbar (non-l1) 295)
    (hcircumflex (non-l1) 293)
    (dotlessi (non-l1) 305)
    (gbreve (non-l1) 287)
    (jcircumflex (non-l1) 309)
    (Cdotaccent (non-l1) 266)
    (Ccircumflex (non-l1) 264)
    (Gdotaccent (non-l1) 288)
    (Gcircumflex (non-l1) 284)
    (Ubreve (non-l1) 364)
    (Scircumflex (non-l1) 348)
    (cdotaccent (non-l1) 267)
    (ccircumflex (non-l1) 265)
    (gdotaccent (non-l1) 289)
    (gcircumflex (non-l1) 285)
    (ubreve (non-l1) 365)
    (scircumflex (non-l1) 349))
  "Table defining SGML entities, see `x-symbol-sgml-table'.")

(defvar x-symbol-sgml-xsymb0-table
  '((Delta (non-l1) 916 "&Delta;")
    (Phi (non-l1) 934 "&Phi;")
    (Gamma (non-l1) 915 "&Gamma;")
    (theta1 (non-l1) 977 "&thetasym;")
    (Lambda (non-l1) 923 "&Lambda;")
    (Pi (non-l1) 928 "&Pi;")
    (Theta (non-l1) 920 "&Theta;")
    (Sigma (non-l1) 931 "&Sigma;")
    (sigma1 (non-l1) 962 "&sigmaf;")
    (Omega (non-l1) 937 "&Omega;")
    (Xi (non-l1) 926 "&Xi;")
    (Psi (non-l1) 936 "&Psi;")
    (alpha (non-l1) 945 "&alpha;")
    (beta (non-l1) 946 "&beta;")
    (chi (non-l1) 967 "&chi;")
    (delta (non-l1) 948 "&delta;")
    (epsilon (non-l1) 949 "&epsilon;")
    (phi (non-l1) 966 "&phi;")
    (gamma (non-l1) 947 "&gamma;")
    (eta (non-l1) 951 "&eta;")
    (iota (non-l1) 953 "&iota;")
    (kappa (non-l1) 954 "&kappa;")
    (lambda (non-l1) 955 "&lambda;")
    (mu (non-l1) 956 "&mu;")
    (nu (non-l1) 957 "&nu;")
    (pi (non-l1) 960 "&pi;")
    (theta (non-l1) 952 "&theta;")
    (rho (non-l1) 961 "&rho;")
    (sigma (non-l1) 963 "&sigma;")
    (tau (non-l1) 964 "&tau;")
    (upsilon (non-l1) 965 "&upsilon;")
    (omega1 (non-l1) 982 "&piv;")
    (omega (non-l1) 969 "&omega;")
    (xi (non-l1) 958 "&xi;")
    (psi (non-l1) 968 "&psi;")
    (zeta (non-l1) 950 "&zeta;")
    (Upsilon1 (non-l1) 978 "&upsih;")

    (florin (non-l1) 402 "&fnof;")
    (bullet (non-l1) 8226 "&bull;")
    (ellipsis (non-l1) 8230 "&hellip;")
    (minute (non-l1) 8242 "&prime;")
    (second (non-l1) 8243 "&Prime;")
    (radicalex (non-l1) 8254 "&oline;")
    (fraction (non-l1) 8260 "&frasl;")
    (weierstrass (non-l1) 8472 "&weierp;")
    (Ifraktur (non-l1) 8465 "&image;")
    (Rfraktur (non-l1) 8476 "&real;")
    (trademark (non-l1) 8482 "&trade;")
    (aleph (non-l1) 8501 "&alefsym;")
    (arrowleft (non-l1) 8592 "&larr;")
    (arrowup (non-l1) 8593 "&uarr;")
    (arrowright (non-l1) 8594 "&rarr;")
    (arrowdown (non-l1) 8595 "&darr;")
    (arrowboth (non-l1) 8596 "&harr;")
    (carriagereturn (non-l1) 8629 "&crarr;")
    (arrowdblleft (non-l1) 8656 "&lArr;")
    (arrowdblup (non-l1) 8657 "&uArr;")
    (arrowdblright (non-l1) 8658 "&rArr;")
    (arrowdbldown (non-l1) 8659 "&dArr;")
    (arrowdblboth (non-l1) 8660 "&hArr;")

    (partialdiff (non-l1) 8706 "&part;")
    (emptyset (non-l1) 8709 "&empty;")
    (gradient (non-l1) 8711 "&nabla;")
    (element (non-l1) 8712 "&isin;")
    (notelement (non-l1) 8713 "&notin;")
    (suchthat (non-l1) 8715 "&ni;")
    (product (non-l1) 8719 "&prod;")
    (summation (non-l1) 8721 "&sum;")
    (minus1 (non-l1) 8722 "&minus;")
    (asterisk1 (non-l1) 8727 "&lowast;")
    (radical (non-l1) 8730 "&radic;")
    (proportional (non-l1) 8733 "&prop;")
    (infinity (non-l1) 8734 "&infin;")
    (angle (non-l1) 8736 "&ang;")
    (logicaland (non-l1) 8743 "&and;")
    (logicalor (non-l1) 8744 "&or;")
    (intersection (non-l1) 8745 "&cap;")
    (union (non-l1) 8746 "&cup;")
    (integral (non-l1) 8747 "&int;")
    (similar (non-l1) 8764 "&sim;")
    (congruent (non-l1) 8773 "&cong;")
    (asym (non-l1) 8776 "&asymp;")
    (notequal (non-l1) 8800 "&ne;")
    (equivalence (non-l1) 8801 "&equiv;")
    (lessequal (non-l1) 8804 "&le;")
    (greaterequal (non-l1) 8805 "&ge;")
    (propersubset (non-l1) 8834 "&sub;")
    (propersuperset (non-l1) 8835 "&sup;")
    (notsubset (non-l1) 8836 "&nsub;")
    (reflexsubset (non-l1) 8838 "&sube;")
    (reflexsuperset (non-l1) 8839 "&supe;")
    (circleplus (non-l1) 8853 "&oplus;")
    (circlemultiply (non-l1) 8855 "&otimes;")
    (perpendicular (non-l1) 8869 "&perp;")
    (periodcentered1 (non-l1) 8901 "&sdot;")
    (ceilingleft (non-l1) 8968 "&lceil;")
    (ceilingright (non-l1) 8969 "&rceil;")
    (floorleft (non-l1) 8970 "&lfloor;")
    (floorright (non-l1) 8971 "&rfloor;")
    (angleleft (non-l1) 9001 "&lang;")
    (angleright (non-l1) 9002 "&rang;")
    (lozenge (non-l1) 9674 "&loz;")
    (spade (non-l1) 9824 "&spades;")
    (club (non-l1) 9827 "&clubs;")
    (heart (non-l1) 9829 "&hearts;")
    (diamond (non-l1) 9830 "&diams;"))
  "Table defining SGML entities, see `x-symbol-sgml-table'.")

(defvar x-symbol-sgml-xsymb1-table
  '((ampersand2 () 38 "&amp;")
    (quotedbl1 () 34 "&quot;")
    (less2 () 60 "&lt;")
    (greater2 () 62 "&gt;")
    (universal1 (non-l1) 8704 "&forall;")
    (existential1 (non-l1) 8707 "&exist;")
    (OE (non-l1) 338 "&OElig;")
    (oe (non-l1) 339 "&oelig;")
    (Ydiaeresis (non-l1) 376 "&Yuml;")
    (circumflex (non-l1) 710 "&circ;")
    (tilde (non-l1) 732 "&tilde;")
    ;;(ensp (non-l1) 8194 "&ensp;") ;  en space, U+2002 ISOpub
    ;;(emsp (non-l1) 8195 "&emsp;") ;  em space, U+2003 ISOpub
    ;;(thinsp (non-l1) 8201 "&thinsp;") ;  thin space, U+2009 ISOpub
    ;;(zwnj (non-l1) 8204 "&zwnj;") ;  zero width non-joiner, U+200C NEW RFC 2070
    ;;(zwj (non-l1) 8205 "&zwj;") ;  zero width joiner, U+200D NEW RFC 2070
    ;;(lrm (non-l1) 8206 "&lrm;") ;  left-to-right mark, U+200E NEW RFC 2070
    ;;(rlm (non-l1) 8207 "&rlm;") ;  right-to-left mark, U+200F NEW RFC 2070
    (endash (non-l1) 8211 "&ndash;")
    (emdash (non-l1) 8212 "&mdash;")
    ;;(lsquo (non-l1) 8216 "&lsquo;") ;  left single quotation mark, U+2018 ISOnum
    ;;(rsquo (non-l1) 8217 "&rsquo;") ;  right single quotation mark, U+2019 ISOnum
    ;;(sbquo (non-l1) 8218 "&sbquo;") ;  single low-9 quotation mark, U+201A NEW
    ;;(ldquo (non-l1) 8220 "&ldquo;") ;  left double quotation mark, U+201C ISOnum
    ;;(rdquo (non-l1) 8221 "&rdquo;") ;  right double quotation mark, U+201D ISOnum
    ;;(bdquo (non-l1) 8222 "&bdquo;") ;  double low-9 quotation mark, U+201E NEW
    (dagger (non-l1) 8224 "&dagger;")
    (daggerdbl (non-l1) 8225 "&Dagger;")
    (perthousand (non-l1) 8240 "&permil;")
    (guilsinglleft (non-l1) 8249 "&lsaquo;")
    (guilsinglright (non-l1) 8250 "&rsaquo;")
    (euro (non-l1) 8364 "&euro;")
    (therefore1 (non-l1) 8756 "&there4;")
    )
  "Table defining SGML entities, see `x-symbol-sgml-table'.")

;; Should I add symbols from http://www.bbsinc.com/iso8859.html ?
(defvar x-symbol-sgml-table
  (append x-symbol-sgml-user-table
	  '(nil)
	  x-symbol-sgml-latin1-table
	  x-symbol-sgml-latinN-table
	  x-symbol-sgml-xsymb0-table
	  x-symbol-sgml-xsymb1-table)
  "Complete table defining SGML entities.
Use `x-symbol-sgml-user-table' to define/shadow private entities.
See `x-symbol-init-language' and `x-symbol-sgml-token-list'.")


;;;===========================================================================
;;;  Internal
;;;===========================================================================

(defvar x-symbol-sgml-menu-alist nil
  "Internal.  Alist used for SGML specific menu.")
(defvar x-symbol-sgml-grid-alist nil
  "Internal.  Alist used for SGML specific grid.")

(defvar x-symbol-sgml-decode-atree nil
  "Internal.  Atree used by `x-symbol-token-input'.")
(defvar x-symbol-sgml-decode-alist nil
  "Internal.  Alist used for decoding of SGML entities.")
(defvar x-symbol-sgml-encode-alist nil
  "Internal.  Alist used for encoding to SGML entities.")

(defvar x-symbol-sgml-nomule-decode-exec nil
  "Internal.  Alist during decoding of SGML entities..")
(defvar x-symbol-sgml-nomule-encode-exec nil
  "Internal.  Alist during encoding of SGML entities..")


;;;===========================================================================
;;;  Subscript functions
;;;===========================================================================

(defun x-symbol-sgml-match-subscript (limit)
  ;; checkdoc-params: (limit)
  "Match and skip over super- and subscripts.
Return nil if `x-symbol-mode' or `x-symbol-subscripts' is nil.  Uses
`x-symbol-sgml-font-lock-regexp'."
  (and x-symbol-mode x-symbol-subscripts
       (let (open-beg open-end close-end close-beg)
	 (while
	     (and (setq open-end (re-search-forward
				  x-symbol-sgml-font-lock-regexp limit t))
		  (setq open-beg (match-beginning 0))
		  (not
		   (and (setq close-end
			      (re-search-forward
			       x-symbol-sgml-font-lock-limit-regexp
			       limit 'limit))
			(setq close-beg (match-beginning 0))
			(equal (cdr (assoc (downcase
					    (buffer-substring open-beg
							      open-end))
					   x-symbol-sgml-font-lock-alist))
			       (downcase (buffer-substring close-beg
							   close-end)))
			(save-excursion
			  (goto-char open-end)
			  (re-search-forward
			   x-symbol-sgml-font-lock-contents-regexp
			   close-beg t))))))
	 (when open-end
	   (store-match-data (list open-beg close-end
				   open-beg open-end
				   open-end close-beg
				   close-beg close-end))
	   t))))

;;; Local IspellPersDict: .ispell_xsymb
;;; x-symbol-sgml.el ends here
