/************************************************************************/
/*  A Symbol picker.							*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<debugon.h>

#   include	<X11/Xatom.h>
#   include	<X11/IntrinsicP.h>
#   include	<Xm/Text.h>
#   include	<Xm/Form.h>
#   include	<Xm/ToggleB.h>
#   include	<Xm/Frame.h>
#   include	<Xm/ScrolledW.h>
#   include	<Xm/Label.h>
#   include	<Xm/RowColumn.h>
#   include	<Xm/PanedW.h>

#   include	<Xm/MwmUtil.h>
#   include	<Xm/Protocols.h>

#   include	<appSymbolPicker.h>
#   include	<appDraw.h>

/************************************************************************/
/*									*/
/*  Resources for the Symbol Picker.					*/
/*									*/
/************************************************************************/

typedef struct AppSymbolPickerResources
    {
    char *	asprInsert;
    char *	asprFont;
    char *	asprNone;
    } AppSymbolPickerResources;

# define xx(x)	x,x

static XtResource APP_SymbolPickerResourceTable[]=
    {
    { xx("symbolPickerFont"), XtRString, sizeof(char *),
		offsetof(AppSymbolPickerResources,asprFont),
		XtRString, "Font" },
    { xx("symbolPickerNone"), XtRString, sizeof(char *),
		offsetof(AppSymbolPickerResources,asprNone),
		XtRString, "None" },
    { xx("symbolPickerInsert"), XtRString, sizeof(char *),
		offsetof(AppSymbolPickerResources,asprInsert),
		XtRString, "Insert" },
    };

/************************************************************************/
/*									*/
/*  Represents a symbol picker.						*/
/*									*/
/************************************************************************/

typedef struct AppSymbolPicker
    {
    EditApplication *		aspApplication;
    double			aspMagnifiedPixelsPerTwip;

    AppSymbolPickerResources *	aspResources;

    Widget			aspTopWidget;
    Widget			aspMainWidget;

    Widget			aspRowColumn;
    Widget			aspInsertButton;

    Widget			aspFontPulldown;
    Widget			aspFontMenu;

    AppToolDestroy		aspDestroy;
    SymbolPickerInsert		aspInsert;
    void *			aspTarget;

    AppFontFamily *		aspFontFamilies;
    int				aspFontFamilyCount;

    short			aspButtonSelected;

    int				aspFontFamilySet;
    int				aspFontFamilyChosen;

    Pixel			aspBackground;
    } AppSymbolPicker;

/************************************************************************/
/*									*/
/*  Adapt the symbol Picker to the current font family.			*/
/*									*/
/************************************************************************/

static int appSymbolAdaptToFamily(  AppSymbolPicker *	asp,
				    int			fontFamilyNumber )
    {
    WidgetList		children;
    Cardinal		childCount= 0;

    Display *		display= XtDisplay( asp->aspRowColumn );
    int			screen= DefaultScreen( display );
    Pixel		whitePixel= WhitePixel( display, screen );
    Pixel		blackPixel= BlackPixel( display, screen );

    char		scratch[120];

    unsigned int	ch;

    XFontStruct *	xfs;
    XmFontList		fontList;

    AppFontFamily *	aff;

    const int		twipsSize= 20* 10;

    XtVaGetValues( asp->aspRowColumn,
			XmNchildren,		&children,
			XmNnumChildren,		&childCount,
			NULL );

    if  ( childCount != 256 )
	{ LLDEB(childCount,256); return -1;	}

    aff= asp->aspFontFamilies+ fontFamilyNumber;

    if  ( appFontXFont( scratch, display,
		    asp->aspMagnifiedPixelsPerTwip,
		    aff, aff->affFaces, twipsSize, DOCfontREGULAR ) )
	{ SDEB(aff->affFontFamilyName); return- 1; }

    xfs= XLoadQueryFont( display, scratch );
    if  ( ! xfs )
	{ SXDEB(scratch,xfs); return -1;	}

    fontList= XmFontListCreate( xfs, XmSTRING_DEFAULT_CHARSET );

    for ( ch= 0; ch < 256; ch++ )
	{
	XmString		labelString;
	unsigned char		lab[2];

	if  ( appCharExistsInFont( xfs, ch ) )
	    {
	    lab[0]= ch; lab[1]= '\0';
	    labelString= XmStringCreateLocalized( (char *)lab );

	    XtVaSetValues( children[ch],
				XmNfontList,		fontList,
				XmNlabelString,		labelString,
				XmNset,			False,
				XmNsensitive,		True,
				XmNborderColor,		whitePixel,
				XmNbackground,		whitePixel,
				XmNforeground,		blackPixel,
#				if  XmVersion >= 2000
				XmNunselectColor,	whitePixel,
				XmNselectColor,		blackPixel,
#				endif
				NULL );
	    }
	else{
	    lab[0]= '\0';
	    labelString= XmStringCreateLocalized( (char *)lab );

	    XtVaSetValues( children[ch],
				XmNlabelString,		labelString,
				XmNset,			False,
				XmNsensitive,		False,
				XmNborderColor,		asp->aspBackground,
				XmNbackground,		asp->aspBackground,
#				if  XmVersion >= 2000
				XmNunselectColor,	asp->aspBackground,
				XmNselectColor,		asp->aspBackground,
#				endif
				NULL );
	    }

	XmStringFree( labelString );
	}

    XmFontListFree( fontList );

    asp->aspButtonSelected= -1;
    XtSetSensitive( asp->aspInsertButton, False );

    return 0;
    }

/************************************************************************/
/*  'Insert' button has been pushed.					*/
/************************************************************************/

static void appSymbolInsertPushed(	Widget		w,
					XtPointer	voidasp,
					XtPointer	voidpbcs	 )
    {
    AppSymbolPicker *	asp= (AppSymbolPicker *)voidasp;
    AppFontFamily *	aff;

    if  ( asp->aspButtonSelected < 0 || ! asp->aspInsert )
	{ LXDEB(asp->aspButtonSelected,asp->aspInsert); return;	}

    aff= asp->aspFontFamilies+ asp->aspFontFamilyChosen;
    (*asp->aspInsert)( asp->aspTarget, asp->aspButtonSelected,
			    asp->aspFontFamilyChosen, aff->affFontFamilyName );

    return;
    }

static void appSymbolChosen(	Widget		w,
				XtPointer	voidasp,
				XtPointer	voidtbcs )
    {
    AppSymbolPicker *			asp= (AppSymbolPicker *)voidasp;
    XmToggleButtonCallbackStruct *	tbcs;

#   if  XmVersion <  2000
    Display *		display= XtDisplay( asp->aspRowColumn );
    int			screen= DefaultScreen( display );
    Pixel		whitePixel= WhitePixel( display, screen );
    Pixel		blackPixel= BlackPixel( display, screen );
#   endif

    tbcs= (XmToggleButtonCallbackStruct *)voidtbcs;

    if  ( tbcs->set )
	{
	XtVaGetValues( w,	XmNpositionIndex,	&asp->aspButtonSelected,
				NULL );

	XtSetSensitive( asp->aspInsertButton, True );

#	if  XmVersion <  2000
	XtVaSetValues( w,
			XmNbackground,		blackPixel,
			XmNforeground,		whitePixel,
			NULL );
#	endif
	}
    else{
#	if  XmVersion <  2000
	XtVaSetValues( w,
			XmNbackground,		whitePixel,
			XmNforeground,		blackPixel,
			NULL );
#	endif
	}

    return;
    }

static void appSymbolClick(	Widget		w,
				XtPointer	voidasp,
				XEvent *	event,
				Boolean *	pContinue )
    {
    static Time			lastTime;
    static unsigned		clickInterval= 0;

    if  ( event->xbutton.button != Button1 )
	{ *pContinue= True; return;	}

    if  ( clickInterval == 0 )
	{
	clickInterval= XtGetMultiClickTime( XtDisplay( w ) );

	if  ( clickInterval == 0 )
	    { LDEB(clickInterval); clickInterval= 200;	}
	}

    if  ( event->xbutton.time- lastTime < clickInterval )
	{
	AppSymbolPicker *	asp= (AppSymbolPicker *)voidasp;
	short			selected;

	XtVaGetValues( w,	XmNpositionIndex,	&selected,
				NULL );

	if  ( selected == asp->aspButtonSelected )
	    {
	    AppFontFamily *	aff;

	    if  ( selected < 0 || ! asp->aspInsert )
		{ LXDEB(selected,asp->aspInsert); return;	}

	    aff= asp->aspFontFamilies+ asp->aspFontFamilyChosen;
	    (*asp->aspInsert)( asp->aspTarget, asp->aspButtonSelected,
			    asp->aspFontFamilyChosen, aff->affFontFamilyName );

	    *pContinue= False; return;
	    }
	}

    lastTime= event->xbutton.time;

    *pContinue= True; return;
    }

/************************************************************************/
/*  A font family was selected by the user.				*/
/************************************************************************/

static void appSymbolFontFamilyChosen(		Widget		w,
						XtPointer	voidasp,
						XtPointer	voidpbcs )
    {
    short		familyChosen= -1;
    AppSymbolPicker *	asp= (AppSymbolPicker *)voidasp;

    XtVaGetValues( w,
			XmNpositionIndex,	&familyChosen,
			NULL );

    if  ( asp->aspFontFamilyChosen != familyChosen )
	{
	if  ( appSymbolAdaptToFamily( asp, familyChosen ) )
	    { LDEB(familyChosen);	}

	asp->aspFontFamilyChosen= familyChosen;
	}

    return;
    }

/************************************************************************/
/*  Create the font part of the symbol picker.				*/
/************************************************************************/

static Widget appSymbolMakeFontPart(	Widget			parent,
					AppSymbolPickerResources * aspr,
					AppSymbolPicker *	asp )
    {
    Widget	frame;
    Widget	bboard;
    Widget	paned;

    appMakeVerticalFrame( &frame, &bboard, &paned, parent, aspr->asprFont );

    appMakePulldownList( &(asp->aspFontPulldown), &(asp->aspFontMenu), paned );

    XtManageChild( asp->aspFontMenu );

    XtManageChild( paned );
    XtManageChild( bboard );
    XtManageChild( frame );

    return frame;
    }

/************************************************************************/
/*  Create the symbol part of the symbol picker.			*/
/************************************************************************/

static Widget appSymbolMakeSymbolPart(	Widget			parent,
					AppSymbolPickerResources * aspr,
					AppSymbolPicker *	asp )
    {
    Widget		children[256];

    XmString		labelString;

    Arg			al[30];
    int			ac= 0;

    int			ch;

    Display *		display= XtDisplay( parent );
    int			screen= DefaultScreen( display );
    Pixel		whitePixel= WhitePixel( display, screen );
    Pixel		blackPixel= BlackPixel( display, screen );

    ac= 0;
    XtSetArg( al[ac], XmNentryAlignment,	XmALIGNMENT_BEGINNING ); ac++;
    XtSetArg( al[ac], XmNentryVerticalAlignment,
						XmALIGNMENT_BASELINE_TOP );ac++;
    XtSetArg( al[ac], XmNmarginHeight,		1 ); ac++;
    XtSetArg( al[ac], XmNmarginWidth,		1 ); ac++;
    XtSetArg( al[ac], XmNnumColumns,		8 ); ac++;
    XtSetArg( al[ac], XmNorientation,		XmHORIZONTAL ); ac++;
    XtSetArg( al[ac], XmNpacking,		XmPACK_COLUMN ); ac++;
    XtSetArg( al[ac], XmNresizeHeight,		True ); ac++;
    XtSetArg( al[ac], XmNresizeWidth,		True ); ac++;
    XtSetArg( al[ac], XmNrowColumnType,		XmWORK_AREA ); ac++;
    XtSetArg( al[ac], XmNspacing,		1 ); ac++;
    XtSetArg( al[ac], XmNentryBorder,		0 ); ac++;

    XtSetArg( al[ac], XmNradioAlwaysOne,	True ); ac++;
    XtSetArg( al[ac], XmNradioBehavior,		True ); ac++;

    XtSetArg( al[ac], XmNbackground,		blackPixel ); ac++;

    asp->aspRowColumn= XmCreateRowColumn( parent, WIDGET_NAME, al, ac );

    labelString= XmStringCreateLocalized( "W" );

    for ( ch= 0; ch < 256; ch++ )
	{
	ac= 0;
	XtSetArg( al[ac], XmNlabelString,		labelString ); ac++;
	XtSetArg( al[ac], XmNindicatorOn,		False ); ac++;
	XtSetArg( al[ac], XmNshadowThickness,		0 ); ac++;
	XtSetArg( al[ac], XmNhighlightThickness,	0 ); ac++;
	XtSetArg( al[ac], XmNmarginWidth,		5 ); ac++;
	XtSetArg( al[ac], XmNmarginHeight,		5 ); ac++;
	XtSetArg( al[ac], XmNborderWidth,		0 ); ac++;
	XtSetArg( al[ac], XmNspacing,			0 ); ac++;
	XtSetArg( al[ac], XmNbackground,		whitePixel ); ac++;
	XtSetArg( al[ac], XmNborderColor,		whitePixel ); ac++;
	XtSetArg( al[ac], XmNforeground,		blackPixel ); ac++;
#	if  XmVersion >= 2000
	XtSetArg( al[ac], XmNunselectColor,		whitePixel ); ac++;
#	endif

	XtSetArg( al[ac], XmNrecomputeSize,		False ); ac++;

	XtSetArg( al[ac], XmNfillOnSelect,		True ); ac++;
	XtSetArg( al[ac], XmNselectColor,		blackPixel ); ac++;

	children[ch]= XmCreateToggleButton( asp->aspRowColumn,
								WIDGET_NAME, al, ac );
	XtAddCallback( children[ch], XmNvalueChangedCallback,
					    appSymbolChosen, (XtPointer)asp );

	XtAddEventHandler( children[ch], ButtonPressMask, False,
					    appSymbolClick, (void *)asp );
	}

    XtManageChildren( children, 256 );
    XtManageChild( asp->aspRowColumn );

    XmStringFree( labelString );

    return asp->aspRowColumn;
    }

/************************************************************************/
/*  Make the button part of the spelling Tool.				*/
/************************************************************************/

static Widget appSymbolMakeButtonRow(	Widget			parent,
					AppSymbolPickerResources * aspr,
					AppSymbolPicker *	asp )
    {
    Widget		row;

    row= appMakeButtonRow( parent, 1 );

    asp->aspInsertButton= appMakeRowButton( row, aspr->asprInsert,
			    appSymbolInsertPushed, (void *)asp, 0, False );

    XtManageChild( row );

    return row;
    }

/************************************************************************/
/*  Fill the list of font families.					*/
/************************************************************************/

static void appSymbolFillFontMenu(	AppSymbolPickerResources *	aspr,
					AppSymbolPicker *		asp )
    {
    Dimension		width;
    int			i;
    Widget		child0= (Widget)0;
    AppFontFamily *	aff;

    XtVaGetValues( asp->aspFontMenu,
			XmNwidth,		&width,
			NULL );

    appEmptyPulldownList( asp->aspFontPulldown );

    aff= asp->aspFontFamilies;
    for ( i= 0; i < asp->aspFontFamilyCount; aff++, i++ )
	{
	Widget		fresh;
	char *		labelText;

	if  ( aff->affFontFamilyText )
	    { labelText= aff->affFontFamilyText;	}
	else{ labelText= aff->affFontFamilyName;	}

	fresh= appPulldownMakeOption( asp->aspFontPulldown,
		    labelText, width, appSymbolFontFamilyChosen, (void *)asp );

	if  ( i == 0 )
	    { child0= fresh;	}
	}

    if  ( asp->aspFontFamilyCount == 0 )
	{
	child0= appPulldownMakeOption( asp->aspFontPulldown,
			    aspr->asprNone,
			    width, appSymbolFontFamilyChosen, (void *)asp );

	XtSetSensitive( asp->aspTopWidget, 0 );
	}

    if  ( child0 )
	{
	XtVaSetValues( asp->aspFontMenu,
			    XmNmenuHistory,		child0,
			    NULL );

	asp->aspFontFamilyChosen= 0;
	}

    appPulldownSetWidth( asp->aspFontMenu, width );
    }

/************************************************************************/
/*  A spell tool must be destroyed.					*/
/************************************************************************/
static void appCloseSymbolPicker(	Widget		w,
					XtPointer	voidasp,
					XtPointer	voidlcs	 )
    {
    AppSymbolPicker *	asp= (AppSymbolPicker *)voidasp;

    if  ( asp->aspDestroy )
	{ (*asp->aspDestroy)( asp->aspTarget );	}

    XtDestroyWidget( w );

    free( asp );

    return;
    }

/************************************************************************/
/*  make a symbol picker tool.						*/
/************************************************************************/
void * appMakeSymbolPicker(	Widget			symbolOption,
				EditApplication *	ea,
				const char *		widgetName,
				Pixmap			iconPixmap,
				Atom			closeAtom,
				SymbolPickerInsert	insert,
				AppToolDestroy		destroy,
				void *			target )
    {
    AppSymbolPicker *	asp;
    
    Arg			al[20];
    int			ac= 0;

    Widget		fontPart;
    Widget		symbolPart;
    Widget		buttonForm;

    MwmHints		hints;

    Dimension		width;

    static AppSymbolPickerResources	aspr;
    static int				gotResources;

    if  ( ! gotResources )
	{
	XtGetApplicationResources( ea->eaTopWidget, (void *)&aspr,
	    APP_SymbolPickerResourceTable,
	    XtNumber(APP_SymbolPickerResourceTable),
	    NULL, 0 );

	gotResources= 1;
	}

    asp= (AppSymbolPicker *)malloc( sizeof(AppSymbolPicker) );
    if  ( ! asp )
	{ XDEB(asp); return (void *)0;	}

    asp->aspApplication= ea;
    asp->aspMagnifiedPixelsPerTwip= ea->eaMagnifiedPixelsPerTwip;

    asp->aspResources= &aspr;


    asp->aspTopWidget= (Widget)0;
    asp->aspRowColumn= (Widget)0;
    asp->aspFontPulldown= (Widget)0;
    asp->aspInsertButton= (Widget)0;

    asp->aspDestroy= destroy;
    asp->aspInsert= insert;
    asp->aspTarget= target;

    asp->aspButtonSelected= -1;
    asp->aspFontFamilySet= -1;
    asp->aspFontFamilyChosen= -1;

    hints.flags= MWM_HINTS_FUNCTIONS|MWM_HINTS_DECORATIONS;
    hints.functions=	MWM_FUNC_MOVE		|
			MWM_FUNC_MINIMIZE	|
			MWM_FUNC_CLOSE		;
    hints.decorations=	MWM_DECOR_BORDER	|
			MWM_DECOR_TITLE		|
			MWM_DECOR_MENU		|
			MWM_DECOR_MINIMIZE	;

    XtSetArg( al[ac], XmNuserData,		(void *)asp ); ac++;
    XtSetArg( al[ac], XmNdeleteResponse,	 XmDO_NOTHING ); ac++;
    XtSetArg( al[ac], XmNinput,			True ); ac++;
    XtSetArg( al[ac], XmNallowShellResize,	True );
    XtSetArg( al[ac], XmNmwmDecorations,	hints.decorations ); ac++;
    XtSetArg( al[ac], XmNmwmFunctions,		hints.functions ); ac++;

    if  ( iconPixmap )
	{ XtSetArg( al[ac], XmNiconPixmap,	iconPixmap ); ac++; }

    asp->aspTopWidget= XtAppCreateShell( ea->eaApplicationName,
					    widgetName,
					    applicationShellWidgetClass,
					    ea->eaDisplay, al, ac );

    appSetShellTitle( asp->aspTopWidget, symbolOption, ea->eaApplicationName );

    if  ( closeAtom > 0 )
	{
	XmAddWMProtocolCallback( asp->aspTopWidget, closeAtom,
				    appCloseSymbolPicker, (XtPointer)asp );
	}

    ac= 0;
    XtSetArg( al[ac], XmNuserData, 		(void *)asp ); ac++;
    XtSetArg( al[ac], XmNsashWidth,		1 ); ac++;
    XtSetArg( al[ac], XmNsashHeight,		1 ); ac++;
    XtSetArg( al[ac], XmNseparatorOn,		False ); ac++;
    XtSetArg( al[ac], XmNmarginWidth,		5 ); ac++;
    XtSetArg( al[ac], XmNmarginHeight,		5 ); ac++;
    XtSetArg( al[ac], XmNspacing,		5 ); ac++;
    asp->aspMainWidget= XmCreatePanedWindow( asp->aspTopWidget, WIDGET_NAME, al, ac );

    fontPart= appSymbolMakeFontPart( asp->aspMainWidget, &aspr, asp );
    symbolPart= appSymbolMakeSymbolPart( asp->aspMainWidget, &aspr, asp );
    buttonForm= appSymbolMakeButtonRow( asp->aspMainWidget, &aspr, asp );

    XtVaGetValues( asp->aspMainWidget,
			XmNbackground,	&asp->aspBackground,
			NULL );

    if  ( psFontCatalog( ea->eaAfmDirectory,
			    &asp->aspFontFamilies, &asp->aspFontFamilyCount ) )
	{
	SDEB(ea->eaAfmDirectory);
	asp->aspFontFamilies= (AppFontFamily *)0;
	asp->aspFontFamilyCount= 0;
	}

    appSymbolFillFontMenu( &aspr, asp );

    XtManageChild( asp->aspMainWidget );

    XtRealizeWidget( asp->aspTopWidget );
    XtMapWidget( asp->aspTopWidget );

    XtVaGetValues( asp->aspFontMenu,
			    XmNwidth,		&width,
			    NULL );

    appPulldownSetWidth( asp->aspFontMenu, width );

    return (void *)asp;
    }

/************************************************************************/
/*  Draw a spell tool to front.						*/
/************************************************************************/
void appShowSymbolPicker(	void *	voidasp	)
    {
    AppSymbolPicker *		asp= (AppSymbolPicker *)voidasp;

    XtVaSetValues( asp->aspTopWidget, XmNinitialState, NormalState, NULL );
    XtMapWidget( asp->aspTopWidget );
    XRaiseWindow( XtDisplay( asp->aspTopWidget ),
					    XtWindow( asp->aspTopWidget ) );
    }

/************************************************************************/
/*									*/
/*  Adapt the symbol Picker to the current font family.			*/
/*									*/
/************************************************************************/

int appAdaptSymbolPickerToFontFamily(	void *		voidasp,
					int		fontFamilyNumber )
    {
    AppSymbolPicker *	asp= (AppSymbolPicker *)voidasp;

    WidgetList		children;
    Cardinal		childCount= 0;

    XtVaGetValues( asp->aspFontPulldown,
		    XmNchildren,	&children,
		    XmNnumChildren,	&childCount,
		    NULL );

    if  ( fontFamilyNumber < 0			||
	  fontFamilyNumber >= (int)childCount	)
	{
	fontFamilyNumber= asp->aspFontFamilyChosen;

	if  ( fontFamilyNumber < 0			||
	      fontFamilyNumber >= (int)childCount	)
	    { LLDEB(fontFamilyNumber,childCount); return -1;	}
	}

    if  ( fontFamilyNumber != asp->aspFontFamilyChosen )
	{
	if  ( appSymbolAdaptToFamily( asp, fontFamilyNumber ) )
	    { LDEB(fontFamilyNumber);	}
	}

    if  ( fontFamilyNumber >= 0 && fontFamilyNumber < (int)childCount )
	{
	XtVaSetValues( asp->aspFontMenu,
			    XmNmenuHistory,	children[fontFamilyNumber],
			    NULL );
	}
    else{
	XtVaSetValues( asp->aspFontMenu,
			    XmNmenuHistory,	(Widget)0,
			    NULL );
	}

    asp->aspFontFamilySet= fontFamilyNumber;
    asp->aspFontFamilyChosen= fontFamilyNumber;

    return 0;
    }

void appEnableSymbolPicker(	void *	voidasp,
				int	enabled )
    {
    AppSymbolPicker *	asp= (AppSymbolPicker *)voidasp;

    XtSetSensitive( asp->aspMainWidget, enabled != 0 );

    return;
    }
