/************************************************************************/
/*  Editor functionality, indepentent of the application.		*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"
#   include	"docPs.h"

#   include	<X11/Xatom.h>

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Insert a table in the document.					*/
/*									*/
/*  0)  Also expose the table borders.					*/
/*  1)  If a range of data was selected, first discard it.		*/
/*  2)  If the selection was at the end of a paragraph, try to move to	*/
/*	the beginning of the next one.					*/
/*  3)  If the IBar is in the middle of a paragraph, split it.		*/
/*  4)  If the paragraph is not the first one in the 'cell' that	*/
/*	contains it, split the cell.					*/
/*  5)  If the 'cell' is not the first one in the 'row' that contains	*/
/*	it, split the row.						*/
/*									*/
/************************************************************************/

static int tedInsertRows(	BufferItem *		sectBi,
				const BufferItem *	refRowBi,
				BufferItem **		pNewBi,
				const RowProperties *	rp,
				TextAttribute		ta,
				BufferDocument *	bd,
				AppDrawingData *	add,
				int			pos,
				int			rows,
				Display *		display,
				DocumentRectangle *	drChanged )
    {
    int			y0= sectBi->biGroupChildren[pos]->biY0;
    int			y= y0;

    BufferItem *	newBi= (BufferItem *)0;

    int			row;
    int			col;

    newBi= (BufferItem *)0;

    for ( row= 0; row < rows; pos++, row++ )
	{
	BufferItem *		rowBi= (BufferItem *)0;

	rowBi= docInsertItem( sectBi, pos, DOClevROW );
	if  ( ! rowBi )
	    { XDEB(rowBi); return -1;	}
	rowBi->biY0= rowBi->biY1= y- 1;

	for ( col= 0; col < rp->rpCellCount; col++ )
	    {
	    BufferItem *	cellBi;
	    BufferItem *	paraBi;

	    const BufferItem *	refParaBi= (BufferItem *)0;
	    const BufferItem *	refCellBi= (BufferItem *)0;

	    if  ( refRowBi && col < refRowBi->biGroupChildCount )
		{ refCellBi= refRowBi->biGroupChildren[col];	}

	    cellBi= docInsertItem( rowBi, col, DOClevCELL );
	    if  ( ! cellBi )
		{ XDEB(cellBi); return -1;	}
	    cellBi->biY0= cellBi->biY1= y- 1;

	    if  ( refCellBi )
		{
		docCopyCellProperties( &cellBi->biCellProperties,
						&refCellBi->biCellProperties );
		}

	    paraBi= docInsertItem( cellBi, 0, DOClevPARA );
	    if  ( ! paraBi )
		{ XDEB(paraBi); return -1;	}
	    paraBi->biY0= paraBi->biY1= y- 1;

	    if  ( refCellBi && refCellBi->biGroupChildCount > 0 )
		{ refParaBi= refCellBi->biGroupChildren[0];	}

	    if  ( refParaBi )
		{
		docCopyParagraphProperties( &paraBi->biParaProperties,
						&refParaBi->biParaProperties );
		}

	    if  ( refParaBi )
		{ ta= refParaBi->biParaParticules[0].tpTextAttribute; }

	    if  ( ! docInsertTextParticule( paraBi, 0, 0, 0,
							DOCkindTEXT, ta ) )
		{ LDEB(1); return -1;	}

	    if  ( row == 0 && col == 0 )
		{ newBi= paraBi;	}

	    paraBi->biParaInTable= 1;
	    }

	if  ( docCopyRowProperties( &(rowBi->biRowProperties), rp ) )
	    { LDEB(1); return -1;	}

	if  ( drChanged->drY0 > y )
	    { drChanged->drY0=  y;	}

	tedLayoutItem( rowBi, bd, add, y, &y, &drChanged->drY1 );
	}

    *pNewBi= newBi; return 0;
    }

int tedInsertTable(		Widget			w,
				EditDocument *		ed,
				int			rows,
				int			columns )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;
    BufferItem *	bi= td->tdSelection.bsBegin.bpBi;

    BufferItem *	sectBi;
    BufferItem *	refRowBi;
    int			col;

    int			oldBackY1= add->addBackRect.drY1;
    int			scrolledX;
    int			scrolledY;
    DocumentRectangle	drChanged;

    BufferItem *	newBi;

    TextAttribute	ta;

    Display *		display= XtDisplay( w );

    int			wide;

    RowProperties	rp;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    /*  0  */
    drChanged.drY0--;

    /*  1  */
    if  ( ! tedHasIBarSelection( td ) )
	{ LDEB(1); return -1;	}

    /*  2  */
    if  ( td->tdSelection.bsBegin.bpStroff == bi->biParaStrlen )
	{
	tedNextPosition( add, &td->tdSelection.bsBegin );
	bi= td->tdSelection.bsBegin.bpBi;
	}

    /*  3  */
    if  ( td->tdSelection.bsBegin.bpStroff != 0 )
	{
	int		onNewPage= 0;

	if  ( tedSplitParaContents( &bi, &drChanged,
						ed, DOClevROW, onNewPage ) )
	    { LDEB(1); return -1;	}
	}

    /*  4  */
    if  ( bi->biNumberInParent > 0 )
	{
	if  ( docSplitGroupItem( &newBi,
					bi->biParent, bi->biNumberInParent ) )
	    { LDEB(bi->biNumberInParent); return -1;	}
	}

    /*  5  */
    if  ( bi->biParent->biNumberInParent > 0 )
	{
	if  ( docSplitGroupItem( &newBi,
		    bi->biParent->biParent, bi->biParent->biNumberInParent ) )
	    { LDEB(bi->biParent->biNumberInParent); return -1;	}
	}

    ta= bi->biParaParticules[0].tpTextAttribute;

    wide=   bd->bdGeometry.dgPaperWideTwips;
    wide -= bd->bdGeometry.dgLeftMarginTwips;
    wide -= bd->bdGeometry.dgRightMarginTwips;

    docInitRowProperties( &rp );
    rp.rpHalfGapWidthTwips= 5* ta.taFontSizeHalfPoints;
    rp.rpLeftIndentTwips=  -5* ta.taFontSizeHalfPoints;

    for ( col= 0; col < columns; col++ )
	{
	CellProperties	cp;

	docInitCellProperties( &cp );
	cp.cpRightBoundaryTwips= ( ( col+ 1 )* wide )/ columns;

	if  ( docInsertRowColumn( &rp, col, &cp ) )
	    { LDEB(col); return -1;	}

	rp.rpHasTableParagraphs= 1;
	}

    newBi= (BufferItem *)0;
    refRowBi= (BufferItem *)0;
    sectBi= bi->biParent->biParent->biParent;

    if  ( tedInsertRows( sectBi, refRowBi, &newBi,
			    &rp, ta, bd, add,
			    bi->biParent->biParent->biNumberInParent,
			    rows, display, &drChanged ) )
	{ LDEB(rows); return -1;	}

    docCleanRowProperties( &rp );

    td->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( ed, newBi, &scrolledX, &scrolledY, 0 );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    return 0;
    }

int tedInsertRowsInTable(	BufferItem *		sectBi,
				const RowProperties *	rp,
				EditDocument *		ed,
				int			row,
				int			rows )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;
    BufferItem *	refParaBi= td->tdSelection.bsBegin.bpBi;

    Widget		w= ed->edDocumentWidget;

    int			scrolledX;
    int			scrolledY;

    int			oldBackY1= add->addBackRect.drY1;
    DocumentRectangle	drChanged;

    BufferItem *	newBi;
    BufferItem *	refRowBi;

    TextAttribute	ta;

    Display *		display= XtDisplay( w );

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    ta= refParaBi->biParaParticules[0].tpTextAttribute;
    newBi= (BufferItem *)0;
    refRowBi= (BufferItem *)0;

    if  ( refParaBi->biParaInTable )
	{ refRowBi= refParaBi->biParent->biParent;	}

    if  ( tedInsertRows( sectBi, refRowBi, &newBi, rp, ta, bd, add, row,
				    rows, display, &drChanged ) )
	{ LDEB(rows); return -1;	}

    td->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( ed, newBi, &scrolledX, &scrolledY, 0 );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return 0;
    }

int tedInsertRowInTable(	Widget			w,
				EditDocument *		ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    const RowProperties *	rp;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    rp= &(sectBi->biGroupChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, rp, ed, row, /*rows*/ 1 ) )
	{ LDEB(row); return -1;	}

    return 0;
    }

int tedAppendRowToTable(	Widget			w,
				EditDocument *		ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    const RowProperties *	rp;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    rp= &(sectBi->biGroupChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, rp, ed, row+ 1, /*rows*/ 1 ) )
	{ LDEB(row); return -1;	}
    
    return 0;
    }

int tedDeleteRowsFromTable(	EditDocument *		ed,
				int			delRow0,
				int			delRow1 )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    int				y0;
    int				y1;

    BufferPosition		bpNew;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    int				scrolledX= 0;
    int				scrolledY= 0;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= sectBi->biGroupChildren[delRow0]->biY0- 1;
    drChanged.drY1= sectBi->biGroupChildren[delRow1]->biY1;

    y0= sectBi->biGroupChildren[delRow0]->biY0;
    y1= sectBi->biGroupChildren[delRow1]->biY1;

    docInitPosition( &bpNew );
    if  ( docFirstPosition( sectBi->biGroupChildren[delRow0], &bpNew )	||
	  docPrevPosition( &bpNew, /*lastOne=*/ 0 )			)
	{
	docInitPosition( &bpNew );
	if  ( docLastPosition( sectBi->biGroupChildren[delRow1], &bpNew ) ||
	      docNextPosition( &bpNew )					  )
	    { docInitPosition( &bpNew ); }
	}

    docDeleteItems( bd, sectBi, delRow0, delRow1- delRow0+ 1 );

    y0= sectBi->biY0;
    y1= sectBi->biY1;
    tedLayoutItem( sectBi, bd, add, y0, &y0, &drChanged.drY1 );

    td->tdVisibleSelectionCopied= 0;

    if  ( bpNew.bpBi )
	{
	tedEditSetIBarSelection( ed, bpNew.bpBi, &scrolledX, &scrolledY,
							    bpNew.bpStroff );
	}
    else{ XDEB(bpNew.bpBi);	}

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert cells/columns in tables.					*/
/*									*/
/************************************************************************/

static BufferItem * tedInsertCellInRow(	BufferDocument *		bd,
					const AppDrawingData *		add,
					BufferItem *			rowBi,
					int				col,
					const CellProperties *	cp,
					TextAttribute			ta,
					Display *			dis )
    {
    RowProperties *	rp;

    BufferItem *	cellBi;
    BufferItem *	paraBi;

    rp= &(rowBi->biRowProperties);

    if  ( docInsertRowColumn( rp, col, cp ) )
	{ LDEB(col); return (BufferItem *)0;	}

    cellBi= docInsertItem( rowBi, col, DOClevCELL );
    if  ( ! cellBi )
	{ LXDEB(col,cellBi); return (BufferItem *)0;	}

    paraBi= docInsertItem( cellBi, 0, DOClevPARA );
    if  ( ! paraBi )
	{ XDEB(paraBi); return (BufferItem *)0;	}

    if  ( ! docInsertTextParticule( paraBi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); return (BufferItem *)0;	}

    paraBi->biParaInTable= 1;

    return paraBi;
    }

static int tedSplitColumnInRows(	Widget		w,
					EditDocument *	ed,
					BufferItem *	sectBi,
					int		row0,
					int		row,
					int		row1,
					int		col,
					int		after )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;

    BufferItem *	newBi= (BufferItem *)0;

    BufferItem *	rowBi;

    int			scrolledX;
    int			scrolledY;
    int			y;
    int			y0;
    int			y1;

    TextAttribute	ta;

    Display *		display= XtDisplay( w );

    DocumentRectangle	drChanged;

    if  ( after )
	{ after= 1;	}

    y0= sectBi->biGroupChildren[row0]->biY0;
    y1= sectBi->biGroupChildren[row1]->biY1;
    y= y0;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= sectBi->biGroupChildren[row0]->biY0- 1;
    drChanged.drY1= sectBi->biGroupChildren[row1]->biY1;

    while( row0 <= row1 )
	{
	RowProperties *		rp;

	CellProperties *	oldCp;
	CellProperties	newCp;

	int			left;
	int			middle;

	BufferItem *		cellBi;
	BufferItem *		paraBi;

	rowBi= sectBi->biGroupChildren[row0];
	rp= &(rowBi->biRowProperties);

	oldCp= rp->rpCells+ col;

	if  ( col == 0 )
	    { left= rp->rpLeftIndentTwips;			}
	else{ left= rp->rpCells[col-1].cpRightBoundaryTwips;	}

	middle= ( rp->rpCells[col].cpRightBoundaryTwips+ left )/ 2;

	newCp= *oldCp;
	if  ( after )
	    { oldCp->cpRightBoundaryTwips= middle;	}
	else{ newCp. cpRightBoundaryTwips= middle;	}

	cellBi= rowBi->biGroupChildren[col];
	paraBi= cellBi->biGroupChildren[0];
	ta= paraBi->biParaParticules[0].tpTextAttribute;

	paraBi= tedInsertCellInRow( bd, add, rowBi, col+ after,
							&newCp, ta, display );
	if  ( row0 == row )
	    { newBi= paraBi;	}

	tedLayoutItem( rowBi, bd, add, y, &y, &drChanged.drY1 );

	row0++;
	}

    td->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( ed, newBi, &scrolledX, &scrolledY, 0 );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    return 0;
    }

int tedInsertColumnInTable(	Widget			w,
				EditDocument *		ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsBegin.bpBi;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    int			after= 0;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( w, ed, sectBi, row0, row, row1, col, after );
    }

int tedAppendColumnToTable(	Widget			w,
				EditDocument *		ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsEnd.bpBi;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    int			after= 1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( w, ed, sectBi, row0, row, row1, col, after );
    }

int tedDeleteColumnsFromRows(	EditDocument *	ed,
				int		delRow0,
				int		delRow1,
				int		delCol0,
				int		delCol1 )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsEnd.bpBi;
    BufferDocument *	bd= td->tdDocument;

    int			y;
    int			y0;
    int			y1;
    int			oldY1;

    BufferItem *	sectBi;
    BufferItem *	rowBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    int			scrolledX= 0;
    int			scrolledY= 0;

    BufferPosition	bpNew;

    DocumentRectangle	drChanged;

    int			count= delCol1- delCol0+ 1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    oldY1= sectBi->biGroupChildren[delRow1]->biY1;

    y0= sectBi->biGroupChildren[delRow0]->biY0;
    y1= sectBi->biGroupChildren[delRow1]->biY1;
    y= y0;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= sectBi->biGroupChildren[row0]->biY0;
    drChanged.drY1= sectBi->biGroupChildren[row1]->biY1;

    docInitPosition( &bpNew );
    rowBi= sectBi->biGroupChildren[delRow1];
    if  ( delCol1 >= rowBi->biGroupChildCount- 1			||
	  docLastPosition( rowBi->biGroupChildren[delCol1], &bpNew )	||
	  docNextPosition( &bpNew )					)
	{
	docInitPosition( &bpNew );
	rowBi= sectBi->biGroupChildren[delRow0];
	if  ( delCol0 <= 0						||
	      docFirstPosition( rowBi->biGroupChildren[delCol0], &bpNew ) ||
	      docPrevPosition( &bpNew, /*lastOne=*/ 0 )			)
	    { docInitPosition( &bpNew ); }
	}

    while( delRow0 <= delRow1 )
	{
	int		wide;
	RowProperties *	rp;

	rowBi= sectBi->biGroupChildren[delRow0];
	rp= &(rowBi->biRowProperties);

	if  ( delCol0 == 0 )
	    { wide= rp->rpLeftIndentTwips;				}
	else{ wide= rp->rpCells[delCol0-1].cpRightBoundaryTwips;	}
	wide= rp->rpCells[delCol1].cpRightBoundaryTwips- wide;

	rp->rpCellCount -= count;

	for ( col= delCol0; col < rp->rpCellCount; col++ )
	    {
	    docCleanCellProperties( &(rp->rpCells[col]) );
	    rp->rpCells[col]= rp->rpCells[col+ count];
	    rp->rpCells[col].cpRightBoundaryTwips -= wide;
	    }

	docDeleteItems( bd, rowBi, delCol0, count );

	tedLayoutItem( rowBi, bd, add, y, &y, &drChanged.drY1 );

	delRow0++;
	}

    td->tdVisibleSelectionCopied= 0;

    if  ( bpNew.bpBi )
	{
	tedEditSetIBarSelection( ed, bpNew.bpBi, &scrolledX, &scrolledY,
							    bpNew.bpStroff );
	}
    else{ XDEB(bpNew.bpBi);	}

    tedExposeRectangle( ed, (const DocumentRectangle *)0, 0,0 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adapt format tool to the circumstances.				*/
/*									*/
/************************************************************************/

void tedAdaptFormatToolToDocument(	void *		voidtft,
					EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferDocument *	bd= td->tdDocument;
    const DocumentGeometry *	dg= &(bd->bdGeometry);

    tedFormatToolAdaptToSelection( voidtft,
			    &(td->tdSelection), ed->edFileReadOnly, dg );

    return;
    }

/************************************************************************/
/*									*/
/*  'Set' event handler of the table tool.				*/
/*									*/
/************************************************************************/

void tedSetTableProperties(	EditApplication *	ea,
				int			col0Set,
				int			col1Set,
				int			row0Set,
				int			row1Set,
				const RowProperties *	rpFrom )
    {
    EditDocument *	ed= ea->eaCurrentDocument;
    TedDocument *	td;

    BufferItem *	sectBi;

    int			col0;
    int			col1;
    int			col11;
    int			row0;
    int			row1;
    int			row00;
    int			row11;

    RowProperties *		rpTo;
    const CellProperties *	cpFrom;
    CellProperties *		cpTo;

    int			rowHeightAlso= 0;
    Boolean		tableRectangle= False;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    if  ( ! tedGetTableRectangle( &(td->tdSelection), &sectBi,
					&col0, &col1, &col11,
					&row00, &row0, &row1, &row11 ) )
	{ tableRectangle= True;	}

    if  ( ! tableRectangle )
	{ LDEB(tableRectangle); return;	}

    if  ( col0Set >= 0 && col0Set == col1Set )
	{
	int			row;

	cpFrom= rpFrom->rpCells+ col0Set;

	for ( row= row0Set; row <= row1Set; row++ )
	    {
	    rpTo= &(sectBi->biGroupChildren[row]->biRowProperties);
	    cpTo= rpTo->rpCells+ col0Set;

	    cpTo->cpLeftBorder= cpFrom->cpLeftBorder;
	    cpTo->cpRightBorder= cpFrom->cpRightBorder;
	    }
	}

    if  ( row0Set >= 0 && row0Set == row1Set )
	{
	int			col;

	rpTo= &(sectBi->biGroupChildren[row0Set]->biRowProperties);
	cpFrom= rpFrom->rpCells;
	cpTo= rpTo->rpCells;

	for ( col= col0Set; col <= col1Set; cpFrom++, cpTo++, col++ )
	    {
	    cpTo->cpTopBorder= cpFrom->cpTopBorder;
	    cpTo->cpBottomBorder= cpFrom->cpBottomBorder;
	    }

	rowHeightAlso= 1;
	}

    tedChangeTableLayout( ed,
		td->tdSelection.bsBegin.bpBi, sectBi,
		rpFrom, rowHeightAlso, row0Set, row1Set );

    return;
    }

void tedAppSetTableSelection(	EditDocument *		ed,
				int			col0Set,
				int			col1Set,
				int			row0Set,
				int			row1Set )
    {
    TedDocument *	td;

    BufferItem *	sectBi;

    int			col0;
    int			col1;
    int			col11;
    int			row0;
    int			row1;
    int			row00;
    int			row11;

    Boolean		tableRectangle= False;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    if  ( ! tedGetTableRectangle( &(td->tdSelection), &sectBi,
					&col0, &col1, &col11,
					&row00, &row0, &row1, &row11 ) )
	{ tableRectangle= True;	}

    if  ( ! tableRectangle )
	{ LDEB(tableRectangle); return;	}

    if  ( col0Set < 0 )
	{ col0Set=  0;		}
    if  ( col1Set > col11 )
	{ col1Set=  col11;	}
    if  ( row0Set < row00 )
	{ row0Set=  row00;	}
    if  ( row1Set > row11 )
	{ row1Set=  row11;	}

    tedDocTableSelectTableRectangle( ed, td, sectBi,
					col0Set, col1Set, row0Set, row1Set );
    return;
    }

/************************************************************************/
/*									*/
/*  Update the of a table in response to a user action.			*/
/*									*/
/************************************************************************/

void tedChangeTableLayout(	EditDocument *		ed,
				BufferItem *		bi,
				BufferItem *		sectBi,
				const RowProperties *	rp,
				int			rowHeightAlso,
				int			row0,
				int			row1 )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;

    int			y;
    int			oldY1;

    BufferItem *	rowBi= sectBi->biGroupChildren[row0];

    DocumentRectangle	drChanged;

    y= rowBi->biY0;
    oldY1= sectBi->biGroupChildren[row1]->biY1;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= sectBi->biGroupChildren[row0]->biY0;
    drChanged.drY1= sectBi->biGroupChildren[row1]->biY1;

    while( row0 <= row1 )
	{
	CellProperties *	cp;
	int			col;

	rowBi= sectBi->biGroupChildren[row0];

	rowBi->biRowLeftIndentTwips= rp->rpLeftIndentTwips;

	if  ( rowHeightAlso )
	    { rowBi->biRowHeightTwips= rp->rpHeightTwips; }

	rowBi->biRowHalfGapWidthTwips= rp->rpHalfGapWidthTwips;
	rowBi->biRowLeftIndentTwips= rp->rpLeftIndentTwips;

	cp= rp->rpCells;
	for ( col= 0; col < rp->rpCellCount; cp++, col++ )
	    {
	    rowBi->biRowCells[col].cpRightBoundaryTwips=
						cp->cpRightBoundaryTwips;
	    }

	tedLayoutItem( rowBi, bd, add, y, &y, &drChanged.drY1 );

	row0++;
	}

    y--;

    tedExposeRectangle( ed, (const DocumentRectangle *)0, 0,0 );

    appDocumentChanged( ed->edApplication, ed, 1 );
    tedCalculateSelectedLines( td );
    tedSelectionCoordinates( &(td->tdSelection), add );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
					     add, &(td->tdSelection) );

    tedDocAdaptHorizontalRuler( ed, bi, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Callback for the 'DrawTableGrid' toggle.				*/
/*									*/
/************************************************************************/

void tedDocTableDrawGrid(	Widget		option,
				XtPointer	voided,
				XtPointer	voidcbs	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    XmToggleButtonCallbackStruct *	cbs;

    cbs= (XmToggleButtonCallbackStruct *)voidcbs;

    td->tdDrawTableGrid= cbs->set;

    tedExposeRectangle( ed, (const DocumentRectangle *)0, 0,0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Get a suggestion about the line height: Used for the initial value	*/
/*  for space before/after in the format tool.				*/
/*									*/
/************************************************************************/

int tedGetParaLineHeight(	int *			pLineHeight,
				EditDocument *		ed )
    {
    TedDocument *	td;
    AppDrawingData *	add;

    BufferItem *	bi;

    int			paraAscent;
    int			paraDescent;

    add= &(ed->edDrawingData);
    td= (TedDocument *)ed->edPrivateData;

    bi= td->tdSelection.bsBegin.bpBi;

    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( docPsParagraphLineExtents( &paraAscent, &paraDescent,
					&(add->addPhysicalFontList), bi ) )
	{ LDEB(1); return -1;	}

    *pLineHeight= paraAscent- paraDescent; return 0;
    }

